/* Drool.c - a little language for telling stories */
/*
	Copyright (c) 1993, by David Michael Betz
	All rights reserved
*/

#include <stdio.h>
#include "bwindow.h"
#include "Drool.h"
#include "Objects.h"
#include "ldef.h"

/* useful definitions */
#define MenuBarHeight		20
#define TitleBarHeight		20
#define ScrollBarWidth		16
#define ScreenMargin		2

/* menu id's */
#define appleID			1
#define fileID			256
#define editID			257
#define controlID		258

/* dialog item numbers */
#define ListOpenItemID		1
#define ListUserItemID		2

/* play dialog item numbers */
#define PlayTranscriptID	1
#define PlayPictureID		2
#define PlayInputID		3
#define PlayNorthID		4
#define PlaySouthID		5
#define PlayEastID		6
#define PlayWestID		7
#define PlayNorthEastID		8
#define PlayNorthWestID		9
#define PlaySouthEastID		10
#define PlaySouthWestID		11
#define PlayUpID		12
#define PlayDownID		13
#define PlayInID		14
#define PlayOutID		15

/* inspector structure */
typedef struct Inspector {
  ListDataProcPtr procPtr;
  ListHandle list;
  DialogPtr dialog;
  ObjectPtr object;
  struct Inspector **next;
} InspectorRecord,*InspectorPtr,**InspectorHandle;

/* list of current inspectors */
InspectorHandle inspectorList;

/* menu handles */
MenuHandle appleMenu;
MenuHandle fileMenu;
MenuHandle editMenu;
MenuHandle controlMenu;

/* globals */
DialogPtr playDialog;
BWindow transcriptBWindow;
BWindow inputBWindow;

/* next window offsets */
int xOffset = 4;
int yOffset = 40;

/* prototypes */
void main(void);
void SetupPlayDialog(void);
void SetupPlayDialogControl(int id,char *str);
pascal void UserItemHandler(DialogPtr d,int itemNumber);
pascal void PlayDialogUserItemHandler(DialogPtr d,int itemNumber);
pascal void InspectorListData(ListHandle h,void *data,int len,Str255 buf);
InspectorHandle MakeInspector(ObjectPtr obj);
InspectorHandle NewInspector(ObjectPtr obj);
void FreeInspector(InspectorHandle inspector);
void CloseAllInspectors(void);
void AddListEntry(ListHandle list,long data);
void OpenListEntry(ListHandle list,Cell cell);
void OpenSelectedListEntries(ListHandle list);
int IsListEvent(EventRecord *myEvent);
void PlaceWindow(WindowPtr w);
void CleanUp(void);
void SetupMenus(void);
void DoEvents(void);
void DoDialogSelect(DialogPtr dialog,int itemNumber);
void DoPlayDialogSelect(DialogPtr dialog,int itemNumber);
void DoInspectorDialogSelect(DialogPtr dialog,int itemNumber);
void DoKeyPress(EventRecord *myEvent);
void DoMouseDown(EventRecord *myEvent);
void DoDrag(EventRecord *myEvent,WindowPtr whichWindow);
void DoGoAway(EventRecord *myEvent,WindowPtr whichWindow);
void DoContent(EventRecord *myEvent,WindowPtr whichWindow);
void DoMenuClick(EventRecord *myEvent);
void DoActivate(EventRecord *myEvent);
void DoUpdate(EventRecord *myEvent);
void DoCommand(long choice);
pascal Boolean aboutfilter(DialogPtr theDialog,EventRecord *theEvent,short *itemHit);
void DoAppleMenu(int theItem);
void DoFileMenu(int theItem);
void DoLoadFile(void);
void DoSaveWorkspace(void);
void DoRestoreWorkspace(void);
void DoEditMenu(int theItem);
void DoControlMenu(int theItem);
void InsufficientMemory(void);
void Fatal(Str255 msg);

void main(void)
{
    Handle userItem;
    int type,i;
    Rect r;

    /* initialize the toolbox */
    InitGraf(&thePort);
    InitFonts();
    InitWindows();
    InitMenus();
    TEInit();
    InitDialogs(0L);
    InitCursor();
    SetupMenus();
    
    /* setup play dialog */
    SetupPlayDialog();

    InitObjectMemory(32768,1024);
    InitCompiler(4096,256);
    InitBuiltInFunctions();

    DoEvents();
}

void SetupPlayDialog(void)
{
    GrafPtr savePort;
    Handle userItem;
    short type;
    Rect r;

    /* create the play dialog */
    CheckNIL(playDialog = GetNewDialog(130,nil,(WindowPtr)-1L));
    GetPort(&savePort);
    SetPort(playDialog);
    
    /* setup transcript area */
    GetDItem(playDialog,PlayTranscriptID,&type,&userItem,&r);
    SetDItem(playDialog,PlayTranscriptID,type,(Handle)PlayDialogUserItemHandler,&r);
    InsetRect(&r,1,1);
    TextFont(monaco);
    TextSize(9);
    BWInit(&transcriptBWindow,playDialog,&r);

    /* setup picture area */
    GetDItem(playDialog,PlayPictureID,&type,&userItem,&r);
    SetDItem(playDialog,PlayPictureID,type,(Handle)PlayDialogUserItemHandler,&r);

#if 0
    /* setup input area */
    GetDItem(playDialog,PlayInputID,&type,&userItem,&r);
    SetDItem(playDialog,PlayInputID,type,(Handle)PlayDialogUserItemHandler,&r);
    InsetRect(&r,1,1);
    BWInit(&inputBWindow,playDialog,&r);
#endif

    /* restore the previous port */
    SetPort(savePort);
}

pascal void UserItemHandler(DialogPtr d,int item)
{
    Handle userItem;
    short type;
    Rect r;
    GetDItem(d,item,&type,&userItem,&r);
    r.right -= ScrollBarWidth;
    InsetRect(&r,-1,-1);
    FrameRect(&r);
    LUpdate(d->visRgn,(ListHandle)GetWRefCon(d));
}

pascal void PlayDialogUserItemHandler(DialogPtr d,int item)
{
    Handle userItem;
    short type;
    Rect r;
    GetDItem(d,item,&type,&userItem,&r);
    FrameRect(&r);
    switch (item) {
    case PlayTranscriptID:
	TextFont(monaco);
	TextSize(9);
	BWRedraw(&transcriptBWindow);
	break;
    case PlayPictureID:
	break;
    case PlayInputID:
	BWRedraw(&inputBWindow);
	break;
    }
}

void TranscriptPutC(int ch)
{
    if (ch == '\n')
	BWPutC(&transcriptBWindow,'\r');
    BWPutC(&transcriptBWindow,ch);
}

pascal void InspectorListData(ListHandle h,void *data,int len,Str255 buf)
{
   InspectorHandle inspector = (InspectorHandle)(**h).refCon;
   ObjectInspectorPrint((**inspector).object,*(long *)data,buf);
}

InspectorHandle MakeInspector(ObjectPtr obj)
{
    InspectorHandle i;
    for (i = inspectorList; i != nil; i = (**i).next)
	if ((**i).object == obj) {
	    SelectWindow((**(**i).list).port);
	    return i;
	}
    NewInspector(obj);
}
    
InspectorHandle NewInspector(ObjectPtr obj)
{
    DialogPtr dialog;
    ListHandle list;
    Handle userItem;
    InspectorHandle inspector;
    GrafPtr oldPort;
    long size,i;
    Str255 buf;
    Rect rect;
    short type;
    Rect rView,dataBounds;
    Point cSize;
    if ((size = ObjectInspectorCount(obj)) == 0)
	return nil;
    CheckNIL(dialog = GetNewDialog(128,nil,(WindowPtr)-1L));
    PlaceWindow(dialog);
    GetPort(&oldPort);
    SetPort(dialog);
    TextFont(monaco);
    TextFace(bold);
    TextSize(9);
    SetPort(oldPort);
    GetDItem(dialog,ListUserItemID,&type,&userItem,&rView);
    SetDItem(dialog,ListUserItemID,type,(Handle)UserItemHandler,&rView);
    rect = rView;
    rView.right -= ScrollBarWidth;
    SetRect(&dataBounds,0,0,1,0);
    cSize.h = 0; cSize.v = 12;
    CheckNIL(list = LNew(&rView,&dataBounds,cSize,128,dialog,FALSE,FALSE,FALSE,TRUE));
    CheckNIL(inspector = (InspectorHandle)NewHandle(sizeof(InspectorRecord)));
    (*inspector)->procPtr = InspectorListData;
    (*inspector)->list = list;
    (*inspector)->dialog = dialog;
    (*inspector)->object = obj;
    (*inspector)->next = inspectorList;
    inspectorList = inspector;
    (*list)->refCon = (long)inspector;
    SetWRefCon(dialog,(long)list);
    for (i = 0; i < size; ++i)
	AddListEntry(list,i);
    ObjectPrint(obj,buf);
    SetWTitle(dialog,buf);
    LDoDraw(TRUE,list);
    ShowWindow(dialog);
    return inspector;
}

void FreeInspector(InspectorHandle inspector)
{
    LDispose((*inspector)->list);
    DisposDialog((*inspector)->dialog);
    DisposHandle((Handle)inspector);
}
    
void CloseAllInspectors(void)
{
    InspectorHandle next;
    for (; inspectorList != nil; inspectorList = next) {
	next = (*inspectorList)->next;
	FreeInspector(inspectorList);
    }
}
	
void AddListEntry(ListHandle list,long data)
{
    Cell cell;
    cell.v = LAddRow(1,-1,list);
    cell.h = 0;
    LSetCell(&data,sizeof(long),cell,list);
}

int IsListEvent(EventRecord *myEvent)
{
    WindowPtr dialog = FrontWindow();
    ListHandle list;
    Rect rect;
    if (myEvent->what == mouseDown && (list = (ListHandle)GetWRefCon(dialog)) != nil) {
        Point p = myEvent->where;
        GrafPtr gp;
        GetPort(&gp);
        SetPort(dialog);
        GlobalToLocal(&p);
	SetPort(gp);
        rect = (*list)->rView;
        rect.right += ScrollBarWidth;
	if (PtInRect(p,&rect)) {
	    if (LClick(p,myEvent->modifiers,list))
		OpenSelectedListEntries(list);
	    return TRUE;
	}
    }
    return FALSE;
}

void OpenListEntry(ListHandle list,Cell cell)
{
    InspectorHandle inspector = (InspectorHandle)(**list).refCon;
    short len = sizeof(long);
    ObjectPtr obj;
    long data;
    LGetCell(&data,&len,cell,list);
    if ((obj = ObjectInspectorOpen((**inspector).object,data)) == nil)
	SysBeep(0);
    else if (MakeInspector(obj) == nil)
	SysBeep(0);
}

void OpenSelectedListEntries(ListHandle list)
{
    Cell cell = {0,0};
    for (; LGetSelect(TRUE,&cell,list); ++cell.v)
	OpenListEntry(list,cell);
}

void PlaceWindow(WindowPtr w)
{
    int x = w->portRect.left + xOffset;
    int y = w->portRect.top + yOffset;
    MoveWindow(w,x,y,FALSE);
    xOffset += 16;
    yOffset += 16;
}
    
void CleanUp(void)
{
    ExitToShell();
}

void SetupMenus(void)
{
    appleMenu = GetMenu(appleID);	/* setup the apple menu */
    AddResMenu(appleMenu,'DRVR');
    InsertMenu(appleMenu,0);
    fileMenu = GetMenu(fileID);		/* setup the file menu */
    InsertMenu(fileMenu,0);
    editMenu = GetMenu(editID);		/* setup the edit menu */
    InsertMenu(editMenu,0);
    controlMenu = GetMenu(controlID);	/* setup the control menu */
    InsertMenu(controlMenu,0);
    DrawMenuBar();
}

void ScanString(char *str);
void ScanString(char *str)
{
    int ch;
    BWPutC(&transcriptBWindow,'\r');
    BWPutC(&transcriptBWindow,'\n');
    while ((ch = *str++) != '\0')
	BWPutC(&transcriptBWindow,ch);
}

void DoEvents(void)
{
    short oldKind,itemNumber;
    EventRecord myEvent;
    DialogPtr dialog;
    long choice;
    
    for (;;) {
	SystemTask();
	GetNextEvent(everyEvent,&myEvent);
	if (myEvent.what == keyDown && (myEvent.modifiers & 0x100) != 0) {
	    if (choice = MenuKey((char)myEvent.message)) {
		DoCommand(choice);
		continue;
	    }
	}
	if (IsDialogEvent(&myEvent)) {
	    if (myEvent.what == keyDown
	    && (myEvent.modifiers & 0x100) == 0
	    &&  (myEvent.message & 0xFF) == '\r') {
		Str255 buf; Handle h; short type; Rect r;
		GetDItem(playDialog,PlayInputID,&type,&h,&r);
		GetIText(h,buf);
		if (buf[0]) ScanString(PtoCstr(buf));
		SetIText(h,"\P");
	    }
	    else if (IsListEvent(&myEvent))
	    	;
	    else if (DialogSelect(&myEvent,&dialog,&itemNumber))
		DoDialogSelect(dialog,itemNumber);
	}
	else
	    switch (myEvent.what) {
		case keyDown:
		case autoKey:
		    DoKeyPress(&myEvent);
		    break;
		case mouseDown:
		    DoMouseDown(&myEvent);
		    break;
		case activateEvt:
		    DoActivate(&myEvent);
		    break;
		case updateEvt:
		    DoUpdate(&myEvent);
		    break;
		}
    }
}

void DoDialogSelect(DialogPtr dialog,int itemNumber)
{
    if (dialog == playDialog)
	DoPlayDialogSelect(dialog,itemNumber);
    else
	DoInspectorDialogSelect(dialog,itemNumber);
}

void DoPlayDialogSelect(DialogPtr dialog,int itemNumber)
{
    switch (itemNumber) {
    case PlayInputID:
	break;
    case PlayNorthID:
	ScanString("North\r");
	break;
    case PlaySouthID:
	ScanString("South\r");
	break;
    case PlayEastID:
	ScanString("East\r");
	break;
    case PlayWestID:
	ScanString("West\r");
	break;
    case PlayNorthEastID:
	ScanString("NE\r");
	break;
    case PlayNorthWestID:
	ScanString("NW\r");
	break;
    case PlaySouthEastID:
	ScanString("SE\r");
	break;
    case PlaySouthWestID:
	ScanString("SW\r");
	break;
    case PlayUpID:
	ScanString("Up\r");
	break;
    case PlayDownID:
	ScanString("Down\r");
	break;
    case PlayInID:
	ScanString("In\r");
	break;
    case PlayOutID:
	ScanString("Out\r");
	break;
    }
}

void DoInspectorDialogSelect(DialogPtr dialog,int itemNumber)
{
    switch (itemNumber) {
    case ListOpenItemID:
	OpenSelectedListEntries((ListHandle)GetWRefCon(dialog));
	break;
    }
}

void DoKeyPress(EventRecord *myEvent)
{
    long choice;
    if (myEvent->modifiers & 0x100) {
	if (choice = MenuKey((char)myEvent->message))
	    DoCommand(choice);
    }
    BWKey(&inputBWindow,myEvent->message);
}

void DoMouseDown(EventRecord *myEvent)
{
    WindowPtr whichWindow;
    switch (FindWindow(myEvent->where,&whichWindow)) {
    case inMenuBar:
	DoMenuClick(myEvent);
	break;
    case inSysWindow:
	SystemClick(myEvent,whichWindow);
	break;
    case inDrag:
	DoDrag(myEvent,whichWindow);
	break;
    case inGoAway:
	DoGoAway(myEvent,whichWindow);
	break;
    case inContent:
	DoContent(myEvent,whichWindow);
	break;
    }
}

void DoDrag(EventRecord *myEvent,WindowPtr whichWindow)
{
    Rect grayRgnRect,dragRect;
    grayRgnRect = (*GetGrayRgn())->rgnBBox;
    SetRect(&dragRect,
	    ScreenMargin,
	    MenuBarHeight - ScreenMargin,
	    grayRgnRect.right - ScreenMargin,
	    grayRgnRect.bottom - TitleBarHeight - ScreenMargin);
    DragWindow(whichWindow,myEvent->where,&dragRect);
}

void DoGoAway(EventRecord *myEvent,WindowPtr whichWindow)
{
    InspectorHandle inspector,prev,this,next;
    ListHandle list;
    if (TrackGoAway(whichWindow,myEvent->where)) {
	list = (ListHandle)GetWRefCon(whichWindow);
	inspector = (InspectorHandle)(**list).refCon;
	for (prev = nil, this = inspectorList; this != nil; prev = this, this = next) {
	    next = (**this).next;
	    if (inspector == this) {
		if (prev == nil)
		    inspectorList = next;
		else
		    (**prev).next = next;
		FreeInspector(inspector);
		break;
	    }
	}
    }
}

void DoContent(EventRecord *myEvent,WindowPtr whichWindow)
{
    if (whichWindow != FrontWindow())
	SelectWindow(whichWindow);
}

void DoMenuClick(EventRecord *myEvent)
{
    long choice;
    if (choice = MenuSelect(myEvent->where))
	DoCommand(choice);
}

void DoActivate(EventRecord *myEvent)
{
    WindowPtr whichWindow;
    ListHandle list;
    int flag = (myEvent->modifiers & activeFlag) ? 0 : 0xFF;
    whichWindow = (WindowPtr)myEvent->message;
    if ((list = (ListHandle)GetWRefCon(whichWindow)) != nil)
	LActivate(flag,list);
    SetPort(whichWindow);
}

void DoUpdate(EventRecord *myEvent)
{
    WindowPtr whichWindow;
    GrafPtr savePort;
    ListHandle list;
    GetPort(&savePort);
    whichWindow = (WindowPtr)myEvent->message;
    SetPort(whichWindow);
    BeginUpdate(whichWindow);
    if ((list = (ListHandle)GetWRefCon(whichWindow)) != nil)
	LUpdate(whichWindow->visRgn,list);
    EndUpdate(whichWindow);
    SetPort(savePort);
}

void DoCommand(long choice)
{
    int theMenu,theItem;

    /* decode the menu choice */
    theMenu = HiWord(choice);
    theItem = LoWord(choice);
    
    HiliteMenu(theMenu);
    switch (theMenu) {
    case appleID:
	DoAppleMenu(theItem);
	break;
    case fileID:
	DoFileMenu(theItem);
	break;
    case editID:
	DoEditMenu(theItem);
	break;
    case controlID:
	DoControlMenu(theItem);
	break;
    }
    HiliteMenu(0);
}

pascal Boolean aboutfilter(DialogPtr theDialog,EventRecord *theEvent,short *itemHit)
{
    return (theEvent->what == mouseDown ? -1 : 0);
}

void DoAppleMenu(int theItem)
{
    DialogRecord mydialog;
    Str255 name;
    GrafPtr gp;
    short n;

    switch (theItem) {
    case 1:
	GetNewDialog(129,&mydialog,(WindowPtr)-1L);
	ModalDialog(aboutfilter,&n);
	CloseDialog((DialogPtr)&mydialog);
	break;
    default:
	GetItem(appleMenu,theItem,name);
	GetPort(&gp);
	OpenDeskAcc(name);
	SetPort(gp);
	break;
    }
}

void DoFileMenu(int theItem)
{
    switch (theItem) {
    case 1:	/* New */
	break;
    case 2:	/* Load.. */
	DoLoadFile();
	break;
    case 4:	/* Save Workspace... */
	DoSaveWorkspace();
	break;
    case 5:	/* Restore Workspace... */
	DoRestoreWorkspace();
	break;
    case 7:	/* Quit */
	CleanUp();
	break;
    }
}

#define LoadFileTypeCount 1
OSType loadFileTypes[] = { 'TEXT' };

void DoLoadFile(void)
{
    StreamHandle fh;
    SFReply reply;
    Point p;
    p.h = 100; p.v = 100;
    SFGetFile(p,"\P",nil,LoadFileTypeCount,loadFileTypes,nil,&reply);
    if (reply.good) {
	if ((fh = OpenStream(reply.fName,reply.vRefNum)) != nil) {
	    Compile(fh);
	    CloseStream(fh);
	}
    }
}

void DoSaveWorkspace(void)
{
    StreamHandle fh;
    SFReply reply;
    Point p;
    p.h = 100; p.v = 100;
    SFPutFile(p,"\PWorkspace Image File","\PDrool Workspace",0L,&reply);
    if (reply.good) {
	if ((fh = CreateStream(reply.fName,reply.vRefNum,'DROO','DRWS')) != nil) {
	    if (!SaveWorkspace(fh))
		SysBeep(1);
	    CloseStream(fh);
	}
    }
}

#define RestoreFileTypeCount 1
OSType restoreFileTypes[] = { 'DRWS' };

void DoRestoreWorkspace(void)
{
    StreamHandle fh;
    SFReply reply;
    Point p;
    p.h = 100; p.v = 100;
    SFGetFile(p,"\P",nil,RestoreFileTypeCount,restoreFileTypes,nil,&reply);
    if (reply.good) {
	if ((fh = OpenStream(reply.fName,reply.vRefNum)) != nil) {
	    CloseAllInspectors();
	    if (!RestoreWorkspace(fh))
		SysBeep(1);
	    CloseStream(fh);
	    InitCompilerVariables();
	}
    }
}

void DoEditMenu(int theItem)
{
    switch (theItem) {
    case 1:	/* undo */
    case 3:	/* cut */
    case 4:	/* copy */
    case 5:	/* paste */
    case 6:	/* clear */
	SystemEdit(theItem-1);
	break;
    }
}

void DoControlMenu(int theItem)
{
    ObjectPtr code;
    switch (theItem) {
    case 1:	/* run main */
	code = SymbolValue(InternSymbol(symbolPackage,NewStringObject((unsigned char *)"main",4)));
	if (MethodP(code))
	    Execute(code);
	break;
    case 2:	/* display symbol package */
	MakeInspector(symbolPackage);
	break;
    case 3:	/* display word package */
	MakeInspector(wordPackage);
	break;
    case 4:	/* collect garbage */
	CollectGarbage();
	break;
    }
}

void CopyRootObjects(void)
{
    InspectorHandle inspector;
    ObjectPtr obj;
    for (inspector = inspectorList; inspector != nil; inspector = (**inspector).next) {
	if ((obj = (**inspector).object) != nil)
	    (**inspector).object = CopyObject(obj);
    }
    ProtectCompilerVariables();
}

void GarbageCollectionDone(void)
{
    InspectorHandle inspector;
    Str255 buf;
    for (inspector = inspectorList; inspector != nil; inspector = (**inspector).next) {
	ListHandle list = (**inspector).list;
	LUpdate((**list).port->visRgn,list);
	ObjectPrint((**inspector).object,buf);
	SetWTitle((**list).port,buf);
    }
}

void InsufficientMemory(void)
{
    Fatal("\PInsufficient memory");
}

void Fatal(Str255 msg)
{
    ParamText(msg,"\P","\P","\P");
    StopAlert(128,nil);
    ExitToShell();
}

