/* Objects.c - object manipulation routines */
/*
	Copyright (c) 1993, by David Michael Betz
	All rights reserved
*/

#include <stdio.h>
#include <string.h>
#include "Drool.h"
#include "Objects.h"
#include "Execute.h"

/* the size of each memory space */
#define MemorySpaceSize	65536

/* round a size up to a multiple of the size of a long */
#define RoundToLong(x)	(((x) + 3) & ~3)

/* global variables */
MemorySpace *oldSpace = nil;
MemorySpace *newSpace = nil;
ObjectPtr nilObject = nil;
ObjectPtr trueObject = nil;
ObjectPtr falseObject = nil;
ObjectPtr symbolPackage = nil;
ObjectPtr wordPackage = nil;
ObjectPtr *stack,*sp,*fp,*stackTop,code;
unsigned char *cbase,*pc;

/* prototypes */
void ScanObject(ObjectPtr obj);
void CopyString(Str255 src,Str255 dst);
void PrintTagAndValue(Str255 tag,ObjectPtr obj,Str255 buf);
void PrintTagAndString(Str255 tag,char *str,Str255 buf);
void DefaultObjectPrint(ObjectPtr obj,Str255 tag,Str255 buf);
void AllocateMemorySpaces(long size,long stackSize);
void FreeMemorySpaces(void);
MemorySpace *NewMemorySpace(long size);
ObjectPtr AllocateObjectMemory(long size);
ObjectPtr CopyPropertyList(ObjectPtr plist);
int WriteLong(long n,StreamHandle fh);
int ReadLong(long *pn,StreamHandle fh);
int WritePointer(ObjectPtr p,StreamHandle fh);
int ReadPointer(ObjectPtr *pp,StreamHandle fh);

void ObjectPrint(ObjectPtr obj,Str255 buf)
{
    ObjectDispatch(obj)->print(obj,buf);
}

long ObjectSize(ObjectPtr obj)
{
    return ObjectDispatch(obj)->size(obj);
}

ObjectPtr CopyObject(ObjectPtr obj)
{
    return ObjectDispatch(obj)->copy(obj);
}

void ScanObject(ObjectPtr obj)
{
    QuickDispatch(obj)->scan(obj);
}

long ObjectInspectorCount(ObjectPtr obj)
{
    return ObjectDispatch(obj)->inspectorCount(obj);
}

void ObjectInspectorPrint(ObjectPtr obj,long data,Str255 buf)
{
    ObjectDispatch(obj)->inspectorPrint(obj,data,buf);
}

ObjectPtr ObjectInspectorOpen(ObjectPtr obj,long data)
{
    return ObjectDispatch(obj)->inspectorOpen(obj,data);
}


/* DUMMY HANDLERS */

long DumInspectorCount(ObjectPtr obj);
void DumInspectorPrint(ObjectPtr obj,long data,Str255 buf);
ObjectPtr DumInspectorOpen(ObjectPtr obj,long data);

long DumInspectorCount(ObjectPtr obj)
{
    return 0;
}

void DumInspectorPrint(ObjectPtr obj,long data,Str255 buf)
{
    NumToString(data,buf);
}

ObjectPtr DumInspectorOpen(ObjectPtr obj,long data)
{
    return nil;
}


/*** NUMBER ***/

void NumPrint(ObjectPtr obj,Str255 buf);
long NumSize(ObjectPtr obj);
ObjectPtr NumCopy(ObjectPtr obj);
void NumScan(ObjectPtr obj);
int NumWrite(ObjectPtr obj,StreamHandle fh);
int NumRead(ObjectPtr obj,StreamHandle fh);

TypeDispatch NumberDispatch = {
    NumberTypeTag,
    NumPrint,
    NumSize,
    NumCopy,
    NumScan,
    NumWrite,
    NumRead,
    DumInspectorCount,
    DumInspectorPrint,
    DumInspectorOpen
};

void NumPrint(ObjectPtr obj,Str255 buf)
{
    NumToString(UnboxNumber(obj),buf);
}

long NumSize(ObjectPtr obj)
{
    return 0;
}

ObjectPtr NumCopy(ObjectPtr obj)
{
    return obj;
}

void NumScan(ObjectPtr obj)
{
    /* should never get here! */
}

int NumWrite(ObjectPtr obj,StreamHandle fh)
{
    /* should never get here! */
}

int NumRead(ObjectPtr obj,StreamHandle fh)
{
    /* should never get here! */
}


/*** OBJECT ***/

void ObjPrint(ObjectPtr obj,Str255 buf);
long ObjSize(ObjectPtr obj);
ObjectPtr ObjCopy(ObjectPtr obj);
void ObjScan(ObjectPtr obj);
int ObjWrite(ObjectPtr obj,StreamHandle fh);
int ObjRead(ObjectPtr obj,StreamHandle fh);
long ObjInspectorCount(ObjectPtr obj);
void ObjInspectorPrint(ObjectPtr obj,long data,Str255 buf);
ObjectPtr ObjInspectorOpen(ObjectPtr obj,long data);

TypeDispatch ObjectDispatch = {
    ObjectTypeTag,
    ObjPrint,
    ObjSize,
    ObjCopy,
    ObjScan,
    ObjWrite,
    ObjRead,
    ObjInspectorCount,
    ObjInspectorPrint,
    ObjInspectorOpen
};

ObjectPtr NewObject(void)
{
    ObjectPtr new = AllocateObjectMemory(sizeof(Object));
    SetObjectDispatch(new,&ObjectDispatch);
    SetObjectClassList(new,nilObject);
    SetObjectSearchList(new,nilObject);
    SetObjectProperties(new,nilObject);
    SetObjectSharedProperties(new,nilObject);
    return new;
}

ObjectPtr CloneObject(ObjectPtr obj)
{
    cpush(obj);
    cpush(NewObject());
    SetObjectDispatch(*sp,ObjectDispatch(sp[1]));
    SetObjectClassList(*sp,ObjectClassList(sp[1]));
    SetObjectSearchList(*sp,ObjectSearchList(sp[1]));
    SetObjectProperties(*sp,CopyPropertyList(ObjectProperties(sp[1])));
    SetObjectSharedProperties(*sp,CopyPropertyList(ObjectSharedProperties(sp[1])));
    obj = pop();
    drop(1);
    return obj;
}

void ObjPrint(ObjectPtr obj,Str255 buf)
{
    DefaultObjectPrint(obj,"\PObject",buf);
}

long ObjSize(ObjectPtr obj)
{
    return sizeof(Object);
}

ObjectPtr ObjCopy(ObjectPtr obj)
{
    ObjectPtr newObj = (ObjectPtr)newSpace->free;
    long size = ObjectSize(obj);
    BlockMove(obj,newObj,size);
    newSpace->free += size;
    SetObjectDispatch(obj,&BrokenHeartDispatch);
    SetForwardingAddress(obj,newObj);
    return newObj;
}

void ObjScan(ObjectPtr obj)
{
    SetObjectClassList(obj,CopyObject(ObjectClassList(obj)));
    SetObjectSearchList(obj,CopyObject(ObjectSearchList(obj)));
    SetObjectProperties(obj,CopyObject(ObjectProperties(obj)));
    SetObjectSharedProperties(obj,CopyObject(ObjectSharedProperties(obj)));
}

int ObjWrite(ObjectPtr obj,StreamHandle fh)
{
    return WriteLong(ObjectTypeTag,fh)
    &&     WritePointer(ObjectClassList(obj),fh)
    &&     WritePointer(ObjectSearchList(obj),fh)
    &&     WritePointer(ObjectProperties(obj),fh)
    &&     WritePointer(ObjectSharedProperties(obj),fh);
}

int ObjRead(ObjectPtr obj,StreamHandle fh)
{
    ObjectPtr classList,searchList,properties,sharedProperties;
    if (!ReadPointer(&classList,fh)
    ||  !ReadPointer(&searchList,fh)
    ||  !ReadPointer(&properties,fh)
    ||  !ReadPointer(&sharedProperties,fh))
	return FALSE;
    SetObjectDispatch(obj,&ObjectDispatch);
    SetObjectClassList(obj,classList);
    SetObjectSearchList(obj,searchList);
    SetObjectProperties(obj,properties);
    SetObjectSharedProperties(obj,sharedProperties);
    return TRUE;
}

long ObjInspectorCount(ObjectPtr obj)
{
    long cCount = ListLength(ObjectClassList(obj));
    long xCount = ListLength(ObjectSearchList(obj));
    long pCount = ListLength(ObjectProperties(obj));
    long sCount = ListLength(ObjectSharedProperties(obj));
    return cCount + xCount + pCount + sCount + 4;
}

void ObjInspectorPrint(ObjectPtr obj,long data,Str255 buf)
{
    long cTag = 0;
    long cLast = cTag + ListLength(ObjectClassList(obj));
    long xTag = cLast + 1;
    long xLast = xTag + ListLength(ObjectSearchList(obj));
    long pTag = xLast + 1;
    long pLast = pTag + ListLength(ObjectProperties(obj));
    long sTag = pLast + 1;
    long sLast = sTag + ListLength(ObjectSharedProperties(obj));
    if (data == cTag)
	CopyString("\PParent Objects:",buf);
    else if (data > cTag && data <= cLast)
	ObjectPrint(Car(ListElement(ObjectClassList(obj),data - cTag - 1)),buf);
    else if (data == xTag)
	CopyString("\PSearch List:",buf);
    else if (data > xTag && data <= xLast)
	ObjectPrint(Car(ListElement(ObjectSearchList(obj),data - xTag - 1)),buf);
    else if (data == pTag)
	CopyString("\PProperties:",buf);
    else if (data > pTag && data <= pLast)
	ObjectPrint(Car(ListElement(ObjectProperties(obj),data - pTag - 1)),buf);
    else if (data == sTag)
	CopyString("\PShared Properties:",buf);
    else if (data > sTag && data <= sLast)
	ObjectPrint(Car(ListElement(ObjectSharedProperties(obj),data - sTag - 1)),buf);
    else
	CopyString("\P<Shouldn't happen!>",buf);
}

ObjectPtr ObjInspectorOpen(ObjectPtr obj,long data)
{
    long cTag = 0;
    long cLast = cTag + ListLength(ObjectClassList(obj));
    long xTag = cLast + 1;
    long xLast = xTag + ListLength(ObjectSearchList(obj));
    long pTag = xLast + 1;
    long pLast = pTag + ListLength(ObjectProperties(obj));
    long sTag = pLast + 1;
    long sLast = sTag + ListLength(ObjectSharedProperties(obj));
    if (data == cTag)
	return nil;
    else if (data > cTag && data <= cLast)
	return Car(ListElement(ObjectClassList(obj),data - cTag - 1));
    else if (data == xTag)
	return nil;
    else if (data > xTag && data <= xLast)
	return Car(ListElement(ObjectSearchList(obj),data - xTag - 1));
    else if (data == pTag)
	return nil;
    else if (data > pTag && data <= pLast)
	return PropertyValue(Car(ListElement(ObjectProperties(obj),data - pTag - 1)));
    else if (data == sTag)
	return nil;
    else if (data > sTag && data <= sLast)
	return PropertyValue(Car(ListElement(ObjectSharedProperties(obj),data - sTag - 1)));
    else
	return nil;
}


/*** CONS ***/

void ConsPrint(ObjectPtr obj,Str255 buf);
long ConsSize(ObjectPtr obj);
void ConsScan(ObjectPtr obj);
int ConsWrite(ObjectPtr obj,StreamHandle fh);
int ConsRead(ObjectPtr obj,StreamHandle fh);
long ConsInspectorCount(ObjectPtr obj);
void ConsInspectorPrint(ObjectPtr obj,long data,Str255 buf);
ObjectPtr ConsInspectorOpen(ObjectPtr obj,long data);

TypeDispatch ConsDispatch = {
    ConsTypeTag,
    ConsPrint,
    ConsSize,
    ObjCopy,
    ConsScan,
    ConsWrite,
    ConsRead,
    ConsInspectorCount,
    ConsInspectorPrint,
    ConsInspectorOpen
};

ObjectPtr Cons(ObjectPtr car,ObjectPtr cdr)
{
    ObjectPtr new;
    cpush(cdr);
    cpush(car);
    new = AllocateObjectMemory(sizeof(ConsObject));
    SetObjectDispatch(new,&ConsDispatch);
    SetCar(new,pop());
    SetCdr(new,pop());
    return new;
}

void ConsPrint(ObjectPtr obj,Str255 buf)
{
    DefaultObjectPrint(obj,"\PList",buf);
}

long ConsSize(ObjectPtr obj)
{
    return sizeof(ConsObject);
}

void ConsScan(ObjectPtr obj)
{
    SetCar(obj,CopyObject(Car(obj)));
    SetCdr(obj,CopyObject(Cdr(obj)));
}

int ConsWrite(ObjectPtr obj,StreamHandle fh)
{
    return WriteLong(ConsTypeTag,fh)
    &&     WritePointer(Car(obj),fh)
    &&     WritePointer(Cdr(obj),fh);
}

int ConsRead(ObjectPtr obj,StreamHandle fh)
{
    ObjectPtr car,cdr;
    if (!ReadPointer(&car,fh)
    ||  !ReadPointer(&cdr,fh))
	return FALSE;
    SetObjectDispatch(obj,&ConsDispatch);
    SetCar(obj,car);
    SetCdr(obj,cdr);
    return TRUE;
}

long ConsInspectorCount(ObjectPtr obj)
{
    return ListLength(obj);
}

void ConsInspectorPrint(ObjectPtr obj,long data,Str255 buf)
{
    obj = ListElement(obj,data);
    if (ConsP(obj))
	ObjectPrint(Car(obj),buf);
    else
	CopyString("\P<Missing!>",buf);
}

ObjectPtr ConsInspectorOpen(ObjectPtr obj,long data)
{
    obj = ListElement(obj,data);
    return ConsP(obj) ? Car(obj) : nil;
}


/*** PROPERTY OBJECT ***/

void PropPrint(ObjectPtr obj,Str255 buf);
long PropSize(ObjectPtr obj);
void PropScan(ObjectPtr obj);
int PropWrite(ObjectPtr obj,StreamHandle fh);
int PropRead(ObjectPtr obj,StreamHandle fh);
long PropInspectorCount(ObjectPtr obj);
void PropInspectorPrint(ObjectPtr obj,long data,Str255 buf);
ObjectPtr PropInspectorOpen(ObjectPtr obj,long data);

TypeDispatch PropertyDispatch = {
    PropertyTypeTag,
    PropPrint,
    PropSize,
    ObjCopy,
    PropScan,
    PropWrite,
    PropRead,
    PropInspectorCount,
    PropInspectorPrint,
    PropInspectorOpen
};

ObjectPtr NewPropertyObject(ObjectPtr tag,ObjectPtr value)
{
    ObjectPtr new;
    cpush(value);
    cpush(tag);
    new = AllocateObjectMemory(sizeof(PropertyObject));
    SetObjectDispatch(new,&PropertyDispatch);
    SetPropertyTag(new,pop());
    SetPropertyValue(new,pop());
    return new;
}

void PropPrint(ObjectPtr obj,Str255 buf)
{
    Str255 tag;
    ObjectPrint(PropertyTag(obj),tag);
    PrintTagAndValue(tag,PropertyValue(obj),buf);
}

long PropSize(ObjectPtr obj)
{
    return sizeof(PropertyObject);
}

void PropScan(ObjectPtr obj)
{
    SetPropertyTag(obj,CopyObject(PropertyTag(obj)));
    SetPropertyValue(obj,CopyObject(PropertyValue(obj)));
}

int PropWrite(ObjectPtr obj,StreamHandle fh)
{
    return WriteLong(PropertyTypeTag,fh)
    &&     WritePointer(PropertyTag(obj),fh)
    &&     WritePointer(PropertyValue(obj),fh);
}

int PropRead(ObjectPtr obj,StreamHandle fh)
{
    ObjectPtr tag,value;
    if (!ReadPointer(&tag,fh)
    ||  !ReadPointer(&value,fh))
	return FALSE;
    SetObjectDispatch(obj,&PropertyDispatch);
    SetPropertyTag(obj,tag);
    SetPropertyValue(obj,value);
    return TRUE;
}

long PropInspectorCount(ObjectPtr obj)
{
    return 2;
}

void PropInspectorPrint(ObjectPtr obj,long data,Str255 buf)
{
    switch (data) {
    case 0:
	PrintTagAndValue("\PTag",PropertyTag(obj),buf);
	break;
    case 1:
	PrintTagAndValue("\PValue",PropertyValue(obj),buf);
	break;
    }
}

ObjectPtr PropInspectorOpen(ObjectPtr obj,long data)
{
    switch (data) {
    case 0:
	return PropertyTag(obj);
    case 1:
	return PropertyValue(obj);
    }
}


/*** SYMBOL OBJECT ***/

void SymPrint(ObjectPtr obj,Str255 buf);
long SymSize(ObjectPtr obj);
void SymScan(ObjectPtr obj);
int SymWrite(ObjectPtr obj,StreamHandle fh);
int SymRead(ObjectPtr obj,StreamHandle fh);
long SymInspectorCount(ObjectPtr obj);
void SymInspectorPrint(ObjectPtr obj,long data,Str255 buf);
ObjectPtr SymInspectorOpen(ObjectPtr obj,long data);

TypeDispatch SymbolDispatch = {
    SymbolTypeTag,
    SymPrint,
    SymSize,
    ObjCopy,
    SymScan,
    SymWrite,
    SymRead,
    SymInspectorCount,
    SymInspectorPrint,
    SymInspectorOpen
};

ObjectPtr NewSymbolObject(ObjectPtr printName)
{
    ObjectPtr new;
    cpush(printName);
    new = AllocateObjectMemory(sizeof(SymbolObject));
    SetObjectDispatch(new,&SymbolDispatch);
    SetSymbolPackage(new,nilObject);
    SetSymbolPrintName(new,pop());
    SetSymbolValue(new,nilObject);
    return new;
}

void SymPrint(ObjectPtr obj,Str255 buf)
{
    ObjectPtr pname = PackageName(SymbolPackage(obj));
    ObjectPtr name = SymbolPrintName(obj);
    if (SymbolPackage(obj) == symbolPackage || obj == nilObject) {
	long size = StringSize(name);
	if (size > 255) size = 255;
	BlockMove(StringDataAddress(name),&buf[1],size);
	buf[0] = size;
    }
    else {
	long size = StringSize(pname);
	if (size > 255) size = 255;
	BlockMove(StringDataAddress(pname),&buf[1],size);
	if ((buf[0] = size) < 255) {
	    buf[++buf[0]] = ':';
	    size = StringSize(name);
	    if (buf[0] + size > 255) size = 255 - buf[0];
	    BlockMove(StringDataAddress(name),&buf[buf[0] + 1],size);
	    buf[0] += size;
	}
    }
}

long SymSize(ObjectPtr obj)
{
    return sizeof(SymbolObject);
}

void SymScan(ObjectPtr obj)
{
    SetSymbolPackage(obj,CopyObject(SymbolPackage(obj)));
    SetSymbolPrintName(obj,CopyObject(SymbolPrintName(obj)));
    SetSymbolValue(obj,CopyObject(SymbolValue(obj)));
}

int SymWrite(ObjectPtr obj,StreamHandle fh)
{
    return WriteLong(SymbolTypeTag,fh)
    &&     WritePointer(SymbolPackage(obj),fh)
    &&     WritePointer(SymbolPrintName(obj),fh)
    &&     WritePointer(SymbolValue(obj),fh);
}

int SymRead(ObjectPtr obj,StreamHandle fh)
{
    ObjectPtr package,printName,value;
    if (!ReadPointer(&package,fh)
    ||  !ReadPointer(&printName,fh)
    ||  !ReadPointer(&value,fh))
	return FALSE;
    SetObjectDispatch(obj,&SymbolDispatch);
    SetSymbolPackage(obj,package);
    SetSymbolPrintName(obj,printName);
    SetSymbolValue(obj,value);
    return TRUE;
}

long SymInspectorCount(ObjectPtr obj)
{
    return 3;
}

void SymInspectorPrint(ObjectPtr obj,long data,Str255 buf)
{
    switch (data) {
    case 0:
	PrintTagAndValue("\PPackage",SymbolPackage(obj),buf);
	break;
    case 1:
	PrintTagAndValue("\PPrint name",SymbolPrintName(obj),buf);
	break;
    case 2:
	PrintTagAndValue("\PValue",SymbolValue(obj),buf);
	break;
    }
}

ObjectPtr SymInspectorOpen(ObjectPtr obj,long data)
{
    switch (data) {
    case 0:
	return SymbolPackage(obj);
    case 1:
	return SymbolPrintName(obj);
    case 2:
	return SymbolValue(obj);
    }
}


/*** STRING OBJECT ***/

void StrPrint(ObjectPtr obj,Str255 buf);
long StrSize(ObjectPtr obj);
void StrScan(ObjectPtr obj);
int StrWrite(ObjectPtr obj,StreamHandle fh);
int StrRead(ObjectPtr obj,StreamHandle fh);

TypeDispatch StringDispatch = {
    StringTypeTag,
    StrPrint,
    StrSize,
    ObjCopy,
    StrScan,
    StrWrite,
    StrRead,
    DumInspectorCount,
    DumInspectorPrint,
    DumInspectorOpen
};

ObjectPtr NewStringObject(unsigned char *data,long size)
{
    long allocSize = sizeof(StringObject) + RoundToLong(size);
    ObjectPtr new = AllocateObjectMemory(allocSize);
    SetObjectDispatch(new,&StringDispatch);
    SetStringSize(new,size);
    if (data)
	BlockMove(data,StringDataAddress(new),size);
    else {
	unsigned char *p;
	for (p = StringDataAddress(new); --size >= 0; )
	    *p++ = '\0';
    }
    return new;
}

ObjectPtr NewCStringObject(char *str)
{
    return NewStringObject((unsigned char *)str,strlen(str));
}

void StrPrint(ObjectPtr obj,Str255 buf)
{
    long size = StringSize(obj);
    buf[1] = '"';
    if (size > 253) {
	buf[0] = 255;
	BlockMove(StringDataAddress(obj),&buf[2],254);
    }
    else {
	buf[0] = size + 2;
	BlockMove(StringDataAddress(obj),&buf[2],size);
	buf[size + 2] = '"';
    }
}

long StrSize(ObjectPtr obj)
{
    return sizeof(StringObject) + RoundToLong(StringSize(obj));
}

void StrScan(ObjectPtr obj)
{
}

int StrWrite(ObjectPtr obj,StreamHandle fh)
{
    long size = StringSize(obj);
    unsigned char *p = StringDataAddress(obj);
    if (!WriteLong(StringTypeTag,fh) || !WriteLong(size,fh))
	return FALSE;
    for (; --size >= 0; ++p)
	if (StreamPutC(fh,*p))
	    return FALSE;
    return TRUE;
}

int StrRead(ObjectPtr obj,StreamHandle fh)
{
    unsigned char *p = StringDataAddress(obj);
    long size;
    int c;
    if (!ReadLong(&size,fh))
	return FALSE;
    SetObjectDispatch(obj,&StringDispatch);
    SetStringSize(obj,size);
    while (--size >= 0) {
	if ((c = StreamGetC(fh)) == StreamEOF)
	    return FALSE;
	*p++ = c;
    }
    return TRUE;
}


/*** VECTOR OBJECT ***/

void VecPrint(ObjectPtr obj,Str255 buf);
long VecSize(ObjectPtr obj);
void VecScan(ObjectPtr obj);
int VecWrite(ObjectPtr obj,StreamHandle fh);
int VecRead(ObjectPtr obj,StreamHandle fh);
long VecInspectorCount(ObjectPtr obj);
void VecInspectorPrint(ObjectPtr obj,long data,Str255 buf);
ObjectPtr VecInspectorOpen(ObjectPtr obj,long data);

TypeDispatch VectorDispatch = {
    VectorTypeTag,
    VecPrint,
    VecSize,
    ObjCopy,
    VecScan,
    VecWrite,
    VecRead,
    VecInspectorCount,
    VecInspectorPrint,
    VecInspectorOpen
};

ObjectPtr NewVectorObject(ObjectPtr *data,long size)
{
    long allocSize = sizeof(VectorObject) + size * sizeof(ObjectPtr);
    ObjectPtr new = AllocateObjectMemory(allocSize);
    SetObjectDispatch(new,&VectorDispatch);
    SetVectorSize(new,size);
    if (data)
	BlockMove(data,VectorDataAddress(new),size * sizeof(ObjectPtr));
    else {
	ObjectPtr *p;
	for (p = VectorDataAddress(new); --size >= 0; )
	    *p++ = nilObject;
    }
    return new;
}

void VecPrint(ObjectPtr obj,Str255 buf)
{
    DefaultObjectPrint(obj,"\PVector",buf);
}

long VecSize(ObjectPtr obj)
{
    return sizeof(VectorObject) + VectorSize(obj) * sizeof(ObjectPtr);
}

void VecScan(ObjectPtr obj)
{
    long i;
    for (i = 0; i < VectorSize(obj); ++i)
    	SetVectorElement(obj,i,CopyObject(VectorElement(obj,i)));
}

int VecWrite(ObjectPtr obj,StreamHandle fh)
{
    long i,size = VectorSize(obj);
    if (!WriteLong(VectorTypeTag,fh) || !WriteLong(size,fh))
	return FALSE;
    for (i = 0; i < size; ++i)
	if (!WritePointer(VectorElement(obj,i),fh))
	    return FALSE;
    return TRUE;
}

int VecRead(ObjectPtr obj,StreamHandle fh)
{
    ObjectPtr value;
    long i,size;
    if (!ReadLong(&size,fh))
	return FALSE;
    SetObjectDispatch(obj,&VectorDispatch);
    SetVectorSize(obj,size);
    for (i = 0; i < size; ++i) {
	if (!ReadPointer(&value,fh))
	    return FALSE;
	SetVectorElement(obj,i,value);
    }
    return TRUE;
}

long VecInspectorCount(ObjectPtr obj)
{
    return VectorSize(obj);
}

void VecInspectorPrint(ObjectPtr obj,long data,Str255 buf)
{
    ObjectPrint(VectorElement(obj,data),buf);
}

ObjectPtr VecInspectorOpen(ObjectPtr obj,long data)
{
    return VectorElement(obj,data);
}


/*** METHOD OBJECT ***/

void MethodPrint(ObjectPtr obj,Str255 buf);
long MethodSize(ObjectPtr obj);
void MethodScan(ObjectPtr obj);
int MethodWrite(ObjectPtr obj,StreamHandle fh);
int MethodRead(ObjectPtr obj,StreamHandle fh);
long MethodInspectorCount(ObjectPtr obj);
void MethodInspectorPrint(ObjectPtr obj,long data,Str255 buf);
ObjectPtr MethodInspectorOpen(ObjectPtr obj,long data);

TypeDispatch MethodDispatch = {
    MethodTypeTag,
    MethodPrint,
    MethodSize,
    ObjCopy,
    MethodScan,
    MethodWrite,
    MethodRead,
    MethodInspectorCount,
    MethodInspectorPrint,
    MethodInspectorOpen
};

ObjectPtr NewMethodObject(ObjectPtr literals,ObjectPtr code)
{
    ObjectPtr new;
    cpush(code);
    cpush(literals);
    new = AllocateObjectMemory(sizeof(MethodObject));
    SetObjectDispatch(new,&MethodDispatch);
    SetMethodClass(new,nilObject);
    SetMethodLiterals(new,pop());
    SetMethodCode(new,pop());
    return new;
}

void MethodPrint(ObjectPtr obj,Str255 buf)
{
    DefaultObjectPrint(obj,"\PMethod",buf);
}

long MethodSize(ObjectPtr obj)
{
    return sizeof(MethodObject);
}

void MethodScan(ObjectPtr obj)
{
    SetMethodClass(obj,CopyObject(MethodClass(obj)));
    SetMethodLiterals(obj,CopyObject(MethodLiterals(obj)));
    SetMethodCode(obj,CopyObject(MethodCode(obj)));
}

int MethodWrite(ObjectPtr obj,StreamHandle fh)
{
    return WriteLong(MethodTypeTag,fh)
    &&     WritePointer(MethodClass(obj),fh)
    &&     WritePointer(MethodLiterals(obj),fh)
    &&     WritePointer(MethodCode(obj),fh);
}

int MethodRead(ObjectPtr obj,StreamHandle fh)
{
    ObjectPtr methodClass,literals,code;
    if (!ReadPointer(&methodClass,fh)
    ||  !ReadPointer(&literals,fh)
    ||  !ReadPointer(&code,fh))
	return FALSE;
    SetObjectDispatch(obj,&MethodDispatch);
    SetMethodClass(obj,methodClass);
    SetMethodLiterals(obj,literals);
    SetMethodCode(obj,code);
    return TRUE;
}

long MethodInspectorCount(ObjectPtr obj)
{
    return 3;
}

void MethodInspectorPrint(ObjectPtr obj,long data,Str255 buf)
{
    switch (data) {
    case 0:
	PrintTagAndValue("\PClass",MethodClass(obj),buf);
	break;
    case 1:
	PrintTagAndValue("\PCode",MethodCode(obj),buf);
	break;
    case 2:
	PrintTagAndValue("\PLiterals",MethodLiterals(obj),buf);
	break;
    }
}

ObjectPtr MethodInspectorOpen(ObjectPtr obj,long data)
{
    switch (data) {
    case 0:
	return MethodClass(obj);
    case 1:
	return MethodCode(obj);
    case 2:
	return MethodLiterals(obj);
    }
}


/*** PACKAGE OBJECT ***/

void PackPrint(ObjectPtr obj,Str255 buf);
long PackSize(ObjectPtr obj);
void PackScan(ObjectPtr obj);
int PackWrite(ObjectPtr obj,StreamHandle fh);
int PackRead(ObjectPtr obj,StreamHandle fh);
long PackInspectorCount(ObjectPtr obj);
void PackInspectorPrint(ObjectPtr obj,long data,Str255 buf);
ObjectPtr PackInspectorOpen(ObjectPtr obj,long data);

TypeDispatch PackageDispatch = {
    PackageTypeTag,
    PackPrint,
    PackSize,
    ObjCopy,
    PackScan,
    PackWrite,
    PackRead,
    PackInspectorCount,
    PackInspectorPrint,
    PackInspectorOpen
};

ObjectPtr NewPackageObject(ObjectPtr name)
{
    ObjectPtr new;
    cpush(name);
    new = AllocateObjectMemory(sizeof(PackageObject));
    SetObjectDispatch(new,&PackageDispatch);
    SetPackageName(new,pop());
    SetPackageHashTable(new,nilObject);
    return new;
}

void PackPrint(ObjectPtr obj,Str255 buf)
{
    DefaultObjectPrint(obj,"\PPackage",buf);
}

long PackSize(ObjectPtr obj)
{
    return sizeof(PackageObject);
}

void PackScan(ObjectPtr obj)
{
    SetPackageName(obj,CopyObject(PackageName(obj)));
    SetPackageHashTable(obj,CopyObject(PackageHashTable(obj)));
}

int PackWrite(ObjectPtr obj,StreamHandle fh)
{
    return WriteLong(PackageTypeTag,fh)
    &&     WritePointer(PackageName(obj),fh)
    &&     WritePointer(PackageHashTable(obj),fh);
}

int PackRead(ObjectPtr obj,StreamHandle fh)
{
    ObjectPtr name,hashTable;
    if (!ReadPointer(&name,fh)
    ||  !ReadPointer(&hashTable,fh))
	return FALSE;
    SetObjectDispatch(obj,&PackageDispatch);
    SetPackageName(obj,name);
    SetPackageHashTable(obj,hashTable);
    return TRUE;
}

long PackInspectorCount(ObjectPtr obj)
{
    return 2;
}

void PackInspectorPrint(ObjectPtr obj,long data,Str255 buf)
{
    switch (data) {
    case 0:
	PrintTagAndValue("\PName",PackageName(obj),buf);
	break;
    case 1:
	PrintTagAndValue("\PHash Table",PackageHashTable(obj),buf);
	break;
    }
}

ObjectPtr PackInspectorOpen(ObjectPtr obj,long data)
{
    switch (data) {
    case 0:
	return PackageName(obj);
    case 1:
	return PackageHashTable(obj);
    }
}


/*** C METHOD OBJECT ***/

void CMethodPrint(ObjectPtr obj,Str255 buf);
long CMethodSize(ObjectPtr obj);
void CMethodScan(ObjectPtr obj);
int CMethodWrite(ObjectPtr obj,StreamHandle fh);
int CMethodRead(ObjectPtr obj,StreamHandle fh);
long CMethodInspectorCount(ObjectPtr obj);
void CMethodInspectorPrint(ObjectPtr obj,long data,Str255 buf);
ObjectPtr CMethodInspectorOpen(ObjectPtr obj,long data);

TypeDispatch CMethodDispatch = {
    CMethodTypeTag,
    CMethodPrint,
    CMethodSize,
    ObjCopy,
    CMethodScan,
    CMethodWrite,
    CMethodRead,
    CMethodInspectorCount,
    CMethodInspectorPrint,
    CMethodInspectorOpen
};

ObjectPtr NewCMethodObject(char *name,void (*handler)(void))
{
    ObjectPtr new;
    new = AllocateObjectMemory(sizeof(CMethodObject));
    SetObjectDispatch(new,&CMethodDispatch);
    SetCMethodName(new,name);
    SetCMethodHandler(new,handler);
    return new;
}

void CMethodPrint(ObjectPtr obj,Str255 buf)
{
    DefaultObjectPrint(obj,"\PCMethod",buf);
}

long CMethodSize(ObjectPtr obj)
{
    return sizeof(CMethodObject);
}

void CMethodScan(ObjectPtr obj)
{
}

int CMethodWrite(ObjectPtr obj,StreamHandle fh)
{
    char *p = CMethodName(obj);
    if (!WriteLong(CMethodTypeTag,fh))
	return FALSE;
    for (; *p != '\0'; ++p)
	if (StreamPutC(fh,*p))
	    return FALSE;
    StreamPutC(fh,'\0');
    return TRUE;
}

int CMethodRead(ObjectPtr obj,StreamHandle fh)
{
    extern FunctionTableEntry functionTable[];
    char name[256],*p = name;
    FunctionTableEntry *f;
    int c;
    while ((c = StreamGetC(fh)) != '\0') {
	if (c == StreamEOF)
	    return FALSE;
	*p++ = c;
    }
    *p = '\0';
    for (f = functionTable; f->name != 0; ++f)
	if (strcmp(f->name,name) == 0) {
	    SetObjectDispatch(obj,&CMethodDispatch);
	    SetCMethodName(obj,f->name);
	    SetCMethodHandler(obj,f->handler);
	    return TRUE;
	}
    return FALSE;
}

long CMethodInspectorCount(ObjectPtr obj)
{
    return 2;
}

void CMethodInspectorPrint(ObjectPtr obj,long data,Str255 buf)
{
    switch (data) {
    case 0:
	PrintTagAndString("\PName",CMethodName(obj),buf);
	break;
    case 1:
	PrintTagAndValue("\PHandler",BoxNumber((long)CMethodHandler(obj)),buf);
	break;
    }
}

ObjectPtr CMethodInspectorOpen(ObjectPtr obj,long data)
{
    switch (data) {
    case 0:
	return nil;
    case 1:
	return nil;
    }
}


/*** BROKEN HEART ***/

void BrokenHeartPrint(ObjectPtr obj,Str255 buf);
long BrokenHeartSize(ObjectPtr obj);
ObjectPtr BrokenHeartCopy(ObjectPtr obj);
void BrokenHeartScan(ObjectPtr obj);
int BrokenHeartWrite(ObjectPtr obj,StreamHandle fh);
int BrokenHeartRead(ObjectPtr obj,StreamHandle fh);

TypeDispatch BrokenHeartDispatch = {
    BrokenHeartTypeTag,
    BrokenHeartPrint,
    BrokenHeartSize,
    BrokenHeartCopy,
    BrokenHeartScan,
    BrokenHeartWrite,
    BrokenHeartRead,
    DumInspectorCount,
    DumInspectorPrint,
    DumInspectorOpen
};

void BrokenHeartPrint(ObjectPtr obj,Str255 buf)
{
    DefaultObjectPrint(obj,"\PBrokenHeart",buf);
}

long BrokenHeartSize(ObjectPtr obj)
{
    return sizeof(BrokenHeart);
}

ObjectPtr BrokenHeartCopy(ObjectPtr obj)
{
    return ForwardingAddress(obj);
}

void BrokenHeartScan(ObjectPtr obj)
{
    /* should never get here! */
}

int BrokenHeartWrite(ObjectPtr obj,StreamHandle fh)
{
    /* should never get here! */
}

int BrokenHeartRead(ObjectPtr obj,StreamHandle fh)
{
    /* should never get here! */
}


/* miscellaneous functions */

ObjectPtr GetProperty(ObjectPtr obj,ObjectPtr tag)
{
    ObjectPtr p;
    for (p = ObjectProperties(obj); p != nilObject; p = Cdr(p))
	if (PropertyTag(Car(p)) == tag)
	    return Car(p);
    return GetSharedProperty(obj,tag);
}

ObjectPtr GetSharedProperty(ObjectPtr obj,ObjectPtr tag)
{
    ObjectPtr p;
    for (p = ObjectSharedProperties(obj); p != nilObject; p = Cdr(p))
	if (PropertyTag(Car(p)) == tag)
	    return Car(p);
    return nil;
}

ObjectPtr GetInheritedProperty(ObjectPtr obj,ObjectPtr tag)
{
    ObjectPtr c,p;
    for (c = ObjectSearchList(obj); c != nilObject; c = Cdr(c))
	if ((p = GetSharedProperty(Car(c),tag)) != nil)
	    return p;
    return nil;
}

long ListLength(ObjectPtr list)
{
    long len = 0;
    while (ConsP(list)) {
	list = Cdr(list);
	++len;
    }
    return len;
}

ObjectPtr ListElement(ObjectPtr list,long n)
{
    while (ConsP(list)) {
	if (--n < 0)
	    return list;
	list = Cdr(list);
    }
    return nilObject;
}

ObjectPtr CopyPropertyList(ObjectPtr plist)
{
    ObjectPtr new;
    cpush(nilObject);
    cpush(nilObject);
    cpush(plist);
    for (; *sp != nilObject; *sp = Cdr(*sp)) {
	new = Cons(NewPropertyObject(PropertyTag(Car(*sp)),PropertyValue(Car(*sp))),nilObject);
	if (sp[1] == nilObject)
	    sp[1] = sp[2] = new;
	else {
	    SetCdr(sp[1],new);
	    sp[1] = new;
	}
    }
    drop(2);
    return pop();
}

ObjectPtr InternCString(ObjectPtr package,char *str)
{
    cpush(package);
    *sp = InternSymbol(*sp,NewStringObject((unsigned char *)str,strlen(str)));
    return pop();
}

ObjectPtr InternSymbol(ObjectPtr package,ObjectPtr printName)
{
    unsigned char *p1,*p2;
    ObjectPtr sym,p;
    long cnt;
    for (p = PackageHashTable(package); p != nilObject; p = Cdr(p)) {
	sym = Car(p);
	p1 = StringDataAddress(printName);
	p2 = StringDataAddress(SymbolPrintName(sym));
	cnt = StringSize(printName);
	if (cnt == StringSize(SymbolPrintName(sym))) {
	    while (cnt > 0) {
		if (*p1++ != *p2++)
		    break;
		--cnt;
	    }
	    if (cnt == 0)
		return sym;
	}
    }
    cpush(package);
    sym = NewSymbolObject(printName);
    SetSymbolPackage(sym,*sp);
    SetPackageHashTable(*sp,Cons(sym,PackageHashTable(*sp)));
    return Car(PackageHashTable(pop()));
}

void DefaultObjectPrint(ObjectPtr obj,Str255 tag,Str255 buf)
{
    unsigned char *dst = &buf[1];
    buf[0] = tag[0] + 3;
    *dst++ = '<';
    BlockMove(&tag[1],dst,tag[0]);
    dst += tag[0];
    NumToString((long)obj,dst);
    buf[0] += *dst; *dst = ' ';
    buf[buf[0]] = '>';
}

void CopyString(Str255 src,Str255 dst)
{
    BlockMove(src,dst,src[0] + 1);
}

void PrintTagAndValue(Str255 tag,ObjectPtr obj,Str255 buf)
{
    unsigned char *src,*dst;
    int srclen,dstlen;
    Str255 val;
    ObjectPrint(obj,val);
    dst = &buf[1]; dstlen = 255;
    src = &tag[1]; srclen = tag[0];
    while (--srclen >= 0)
	if (dstlen > 0) {
	    *dst++ = *src++;
	    --dstlen;
	}
    if (dstlen >= 2) {
	*dst++ = ':';
	*dst++ = ' ';
	dstlen -= 2;
    }
    src = &val[1]; srclen = val[0];
    while (--srclen >= 0)
	if (dstlen > 0) {
	    *dst++ = *src++;
	    --dstlen;
	}
    buf[0] = dst - &buf[1];
}

void PrintTagAndString(Str255 tag,char *str,Str255 buf)
{
    unsigned char *src,*dst;
    int srclen,dstlen;
    dst = &buf[1]; dstlen = 255;
    src = &tag[1]; srclen = tag[0];
    while (--srclen >= 0)
	if (dstlen > 0) {
	    *dst++ = *src++;
	    --dstlen;
	}
    if (dstlen >= 2) {
	*dst++ = ':';
	*dst++ = ' ';
	dstlen -= 2;
    }
    for (; *str != '\0'; ++str)
	if (dstlen > 0) {
	    *dst++ = *str;
	    --dstlen;
	}
    buf[0] = dst - &buf[1];
}

void AllocateMemorySpaces(long size,long stackSize)
{
    /* make the memory spaces */
    CheckNIL(oldSpace = NewMemorySpace(size));
    CheckNIL(newSpace = NewMemorySpace(size));

    /* make and initialize the stack */
    CheckNIL(stack = (ObjectPtr *)osalloc(stackSize * sizeof(ObjectPtr)));
    stackTop = stack + stackSize;
    ResetStack();
}

void FreeMemorySpaces(void)
{
    osfree((char *)oldSpace);
    osfree((char *)newSpace);
    osfree((char *)stack);
}
       
void InitObjectMemory(long size,long stackSize)
{
    /* first allocate the memory */
    AllocateMemorySpaces(size,stackSize);
    
    /* make the nil symbol */
    nilObject = NewSymbolObject(NewStringObject((unsigned char *)"nil",3));
    SetSymbolPackage(nilObject,nilObject);
    SetSymbolValue(nilObject,nilObject);

    /* initialize the packages */
    symbolPackage = NewPackageObject(NewStringObject((unsigned char *)"symbol",6));
    wordPackage = NewPackageObject(NewStringObject((unsigned char *)"word",4));
        
    /* make the true and false symbols */
    trueObject = InternCString(symbolPackage,"t");
    SetSymbolValue(trueObject,trueObject);
    falseObject = nilObject;
}

MemorySpace *NewMemorySpace(long size)
{
    MemorySpace *space;
    if ((space = (MemorySpace *)osalloc(sizeof(MemorySpace) + size)) != nil) {
	space->base = (unsigned char *)space + sizeof(MemorySpace);
	space->free = space->base;
	space->top = space->base + size;
    }
    return space;
}

ObjectPtr AllocateObjectMemory(long size)
{
    ObjectPtr p;
    if (newSpace->free + size > newSpace->top) {
	CollectGarbage();
	if (newSpace->free + size > newSpace->top)
	    InsufficientMemory();
    }
    p = (ObjectPtr)newSpace->free;
    newSpace->free += size;
    return p;
}

void CollectGarbage(void)
{
    unsigned char *scan;
    ObjectPtr obj,*p;
    MemorySpace *ms;
    long pcoff;
    
    /* reverse the memory spaces */
    ms = oldSpace;
    oldSpace = newSpace;
    newSpace = ms;

    /* reset the new space pointers */
    newSpace->free = scan = newSpace->base;
    
    /* copy the root objects */
    nilObject = CopyObject(nilObject);
    trueObject = CopyObject(trueObject);
    falseObject = CopyObject(falseObject);
    symbolPackage = CopyObject(symbolPackage);
    wordPackage = CopyObject(wordPackage);
    CopyRootObjects();
    
    /* copy the stack */
    for (p = sp; p < stackTop; ++p)
	*p = CopyObject(*p);
	
    /* copy the current code object and fixup cbase and pc */
    if (code != nil) {
	pcoff = pc - cbase;
	code = CopyObject(code);
	cbase = StringDataAddress(MethodCode(code));
	pc = cbase + pcoff;
    }
    
    /* scan and copy until all accessible objects have been copied */
    while (scan < newSpace->free) {
	obj = (ObjectPtr)scan;
	scan += ObjectSize(obj);
	ScanObject(obj);
    }
    
    /* cleanup after garbage collection */
    GarbageCollectionDone();
}

int SaveWorkspace(StreamHandle fh)
{
    unsigned char *next;
    ObjectPtr obj;
    
    /* first compact memory */
    CollectGarbage();
    
    /* write the file header */
    WriteLong((long)(newSpace->top - newSpace->base),fh);
    WriteLong((long)(stackTop - stack),fh);
    WriteLong((long)(newSpace->free - newSpace->base),fh);
    
    /* write the root objects */
    WritePointer(nilObject,fh);
    WritePointer(symbolPackage,fh);
    WritePointer(wordPackage,fh);
    WritePointer(trueObject,fh);
    WritePointer(falseObject,fh);
    
    /* write the heap */
    for (next = newSpace->base; next < newSpace->free; next += ObjectSize(obj)) {
	obj = (ObjectPtr)next;
	if (!ObjectDispatch(obj)->write(obj,fh))
	    return FALSE;
    }
    return TRUE;
}

/* WARNING:  this must be in the same order as the xxxTypeTag values!!! */
static TypeDispatch *tagToDispatch[] = {
  &BrokenHeartDispatch,
  &NumberDispatch,
  &ObjectDispatch,
  &ConsDispatch,
  &PropertyDispatch,
  &SymbolDispatch,
  &StringDispatch,
  &VectorDispatch,
  &MethodDispatch,
  &PackageDispatch,
  &CMethodDispatch
};

int RestoreWorkspace(StreamHandle fh)
{
    long size,stackSize,tmp;
    unsigned char *free;
    ObjectPtr obj;
    
    /* free the old memory spaces */
    FreeMemorySpaces();
    
    /* read the file header */
    ReadLong(&size,fh);
    ReadLong(&stackSize,fh);
    ReadLong(&tmp,fh);
    
    /* create the new memory spaces */
    AllocateMemorySpaces(size,stackSize);
    
    /* compute first free address after the load */
    free = newSpace->base + tmp;
    
    /* read the root objects */
    ReadPointer(&nilObject,fh);
    ReadPointer(&symbolPackage,fh);
    ReadPointer(&wordPackage,fh);
    ReadPointer(&trueObject,fh);
    ReadPointer(&falseObject,fh);
    
    /* read the heap */
    while (newSpace->free < free) {
	ReadLong(&tmp,fh);
	obj = (ObjectPtr)newSpace->free;
	if (!tagToDispatch[(int)tmp]->read(obj,fh))
	    return FALSE;
	newSpace->free += ObjectSize(obj);
    }
    return TRUE;
}

int WriteLong(long n,StreamHandle fh)
{
    return StreamPutC(fh,(n >> 24)) != StreamEOF
    &&     StreamPutC(fh,(n >> 16)) != StreamEOF
    &&     StreamPutC(fh,(n >>  8)) != StreamEOF
    &&     StreamPutC(fh, n       ) != StreamEOF;
}

int ReadLong(long *pn,StreamHandle fh)
{
    int c;
    if ((c = StreamGetC(fh)) == StreamEOF)
	return FALSE;
    *pn = (long)c << 24;
    if ((c = StreamGetC(fh)) == StreamEOF)
	return FALSE;
    *pn |= (long)c << 16;
    if ((c = StreamGetC(fh)) == StreamEOF)
	return FALSE;
    *pn |= (long)c << 8;
    if ((c = StreamGetC(fh)) == StreamEOF)
	return FALSE;
    *pn |= (long)c;
    return TRUE;
}

int WritePointer(ObjectPtr p,StreamHandle fh)
{
    return NumberP(p) ? WriteLong((long)p,fh)
    		      : WriteLong((long)((unsigned char *)p - newSpace->base),fh);
}

int ReadPointer(ObjectPtr *pp,StreamHandle fh)
{
    long n;
    if (!ReadLong(&n,fh))
	return FALSE;
    *pp = NumberP(n) ? (ObjectPtr)n : (ObjectPtr)(newSpace->base + n);
    return TRUE;
}

void StackOverflow(void)
{
    error("stack overflow");
}

void ResetStack(void)
{
    sp = fp = stackTop;
    code = nil;
}




