/************************************************************************
*
* Program: GINA System LISP
* Module ginalisp.c
* Version 0.1, January 1994.
* Version 0.2, October 1994.
* Version 0.3, February 1995.
* Version 0.4, August 1995.
*
* Copyright 1994, 1995, Jeff Standish.  (jestandi@cs.indiana.edu)
* All rights reserved.
* Permission is hereby granted for unrestricted non-commercial use,
* provided full attribution of source code origin is included.
*
* Version 0.1
* This module contains the main LISP evaluation functions for parsing
* in LISP expressions, evaluating them, and then printing out the
* results.  The actual LISP functions are defined within another module.
*
* Version 0.2
* Revised into standard C, removing dependencies upon special functions
* for accessing memory via 8088.
*
* Version 0.3
* Corrected a bug in the (method...) function where <self> was reset
* before the function arguments where resolved, which required that
* eval_user_func() take a new self pointer and deal with the hassle of
* resetting <self>.
*
* Version 0.4
* Few changes initially have been made.  Mostly just adding comments
* to the code, filling in the function block descriptions, and changing
* the error messages to use the func_warning() function to display
* both the warning message and the code which generated it.
*
************************************************************************/

#include <stdio.h>
#include "ginas.h"


extern NODEZ *warningbind;	/* flag for minor warning messages */

	/* flag for whether to print garbage collection messages */
extern NODEZ *garbmesgbind;

	/* file pointer for input file */
extern FILE *lispfile;
extern int  linenumber;
extern int  debuglevel;

	/* data for parser_lisp */
#define LINELEN   2048
char token[LINELEN], fname[50];
int  stringtoken, thischar, nextchar;

	/* trees for global variables, local variables, and functions */
extern NODEZ *global, *locals, *functions;

	/* info on built-in functions */
extern FUNCDATA funclist[];

	/* count of nodes allocated since last garbage collection */
extern int nodetally;
extern int gcsize;	/* number of nodes to be allocated be for garb coll */

	/* location of "self" binding */
extern NODEZ *selfbind;

	/* array of truth values indicating whether indexed char is special */
char specialchar[128] = {		/* space & ' ( ) ^ */
0,0,0,0,0,0,0,0,0,0,	/*   0-9 */
0,0,0,0,0,0,0,0,0,0,	/*  10-19 */
0,0,0,0,0,0,0,0,0,0,	/*  20-29 */
0,0,1,0,0,0,0,0,1,1,	/*  30-39 */
1,1,0,0,0,0,0,0,0,0,	/*  40-49 */
0,0,0,0,0,0,0,0,0,0,	/*  50-59 */
0,0,0,0,0,0,0,0,0,0,	/*  60-69 */
0,0,0,0,0,0,0,0,0,0,	/*  70-79 */
0,0,0,0,0,0,0,0,0,0,	/*  80-89 */
0,0,0,0,1,0,0,0,0,0,	/*  90-99 */
0,0,0,0,0,0,0,0,0,0,	/* 100-109 */
0,0,0,0,0,0,0,0,0,0,	/* 110-119 */
0,0,0,0,0,0,0,0		/* 120-127 */
};


/************************************************************************
*
* gina_lisp() - this function serves as the main interactive lisp
*		interpreter, either reading the input from a file,
*		or from the keyboard if no file is given
*
* filename  - name of file to load, if null then read from stdin
* docollect - will do garbage collection only if this flag is set,
*		this flag should only be set when called from the
*		top level of the interpreter, if set from anywhere else,
*		it is possible that nodes in temporary use by internal
*		functions will be freed up while still in use
*
************************************************************************/

#ifdef _ANSI_
void gina_lisp(char *filename, int docollect)
#else
gina_lisp(filename, docollect)
char *filename;
int  docollect;
#endif
{
    int   exitflag;
    NODEZ *head, *result;
    char  oldtoken[LINELEN], oldfname[50];
    int   oldthischar, oldnextchar, oldline;
    FILE  *oldlispfile;

	/* record current values of input stream */
    if (filename != NULL) {
	oldlispfile = lispfile;
	strcpy(oldtoken, token);
	strcpy(oldfname, fname);
	oldthischar = thischar;
	oldnextchar = nextchar;
	oldline = linenumber;
    }

	/* empty 2-byte lookahead buffer */
    head = NULL;
    thischar = '\n';
    nextchar = '\n';

	/* if a file name is given, open that file for input,
	 * otherwise default to standard input */
    if (filename != NULL)
	load_file(filename);
    else
	lispfile = stdin;

	/* loop until hit end of input file, or until the exitflag is
	 * set by an executed LISP function */
    exitflag = 0;
    while (!exitflag && (lispfile != NULL)) {

	    /* if reading from standard input, prompt user for instruction */
	if (lispfile == stdin) {
	    if (debuglevel)
		printf("debug %d> ", debuglevel);
	    else
		printf("GL> ");
	}

	    /* preload next token, and break loop if at end of file */
	get_token();
	if (lispfile == NULL)
	    break;

	    /* parse the input stream into a lisp structure */
	head = parse_lisp();
	result = evaluate(head, &exitflag);

	    /* if reading from standard input, display the results of
		evaluating the lisp expression */
	if (lispfile == stdin) {
	    pretty_print_tree(result, 0, 0);
	    printf("\n");
	}

	    /* if garbage collection is turned on and a sufficient number
		of cons nodes have been allocated, then perform collection */
	if (docollect && (nodetally >= gcsize))
	    collect_garbage(garbmesgbind->value.ptr.cdr != NULL);
    }

	/* restore the previous values for the input stream */
    if (filename != NULL) {
	linenumber = oldline;
	nextchar = oldnextchar;
	thischar = oldthischar;
	strcpy(fname, oldfname);
	strcpy(token, oldtoken);
	lispfile = oldlispfile;
    }
}



/************************************************************************
*
* evaluate() - Evaluate the given lisp expression, and return its result.
*	Set <exitflag> to interrupt flow control.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *evaluate(NODEZ *head, int *exitflag)
#else
NODEZ *evaluate(head, exitflag)
NODEZ *head;
int   *exitflag;
#endif
{
    NODEZ *ptr1, *ptr2, *result;

    *exitflag = 0;

	/* make sure not passed an empty list */
    if (head == NULL)
	return (NULL);

	/* strings, numbers, objects, and classes evaluate to themselves */
    else if ((head->type == TYPEnumber) || (head->type == TYPEstring)
		|| (head->type == TYPEobject) || (head->type == TYPEclass))
	return (head);

	/* should be a binding */
    else if (head->type == TYPEidname) {
	if ((ptr1 = find_binding(head->value.idptr)) != NULL)
	    return (ptr1->value.ptr.cdr);
	else if (warningbind->value.ptr.cdr != NULL)
	    func_warning("unknown binding \"%s\"", head->value.idptr->name,
		head);
	return (NULL);
    }

	/* should be a function */
    else if (head->type == TYPElisthead) {
	if (head->value.ptr.car == NULL) {
	    func_warning("invalid function, nil is not a function", NULL,
			head);
	    return (NULL);
	}

	    /* built-in functions */
	ptr1 = head->value.ptr.car;
	if (ptr1->type == TYPEfuncname) {
	    if (funclist[ptr1->value.number].argnum < 0) {
		if (argument_minimum(-funclist[ptr1->value.number].argnum,
				funclist[ptr1->value.number].name, head))
		    return (NULL);
	    }
	    else if (argument_check(funclist[ptr1->value.number].argnum,
				funclist[ptr1->value.number].name, head))
		return (NULL);
	    return (*funclist[ptr1->value.number].funcptr)(head, exitflag);
	}

		/* handle case of user-defined function */
	else if (ptr1->type == TYPEidname) {
	    ptr2 = selfbind->value.ptr.cdr;

		/* if self has a value, and that object has this method,
		   then evaluate this method */
	    if ((ptr2 != NULL) && ((ptr2 = find_method(good_ob_ptr(ptr2),
						ptr1->value.idptr)) != NULL))
		result = eval_user_func(head, ptr2, NULL);

		/* otherwise, look for a regular function of this name */
	    else if ((ptr2 = find_function(ptr1->value.idptr)) != NULL)
		result = eval_user_func(head, ptr2, NULL);

		/* otherwise, bitch and gripe */
	    else {
		if (warningbind->value.ptr.cdr != NULL)
		    func_warning("unknown function \"%s\"",
			ptr1->value.idptr->name, head);
		result = NULL;
	    }
	}
	else {
	    func_warning("function expected", NULL, head);
	    result = NULL;
	}
    }
    else if (head->type == TYPEfuncname) {
	func_warning("invalid use of symbol \"%s\"",
		funclist[head->value.number].name, head);
	result = head;
    }
    else {
	printf("Error: (4) unknown node type %d\n", head->type);
	result = NULL;
    }

    return (result);
}



/************************************************************************
*
* eval_user_func() - Evaluate a user-defined function.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *eval_user_func(NODEZ *head, NODEZ *fptr, NODEZ *newself)
#else
NODEZ *eval_user_func(head, fptr, newself)
NODEZ *head, *fptr, *newself;
#endif
{
    int   exitflag;
    NODEZ *ptr, *result, *scopeptr, *oldself;

	/* fptr is function definition, head is function call */
    fptr = fptr->value.ptr.cdr;

	/* create new local scope, scopenode is local header */
    scopeptr = new_node();
    scopeptr->type = TYPElisthead;
    scopeptr->value.ptr.cdr = eval_binding_list(head->value.ptr.cdr,
						fptr->value.ptr.car, head);
    fptr = fptr->value.ptr.cdr;

	/* now that locals have been assigned, place new scope
	   on top of local binding tree */
    scopeptr->value.ptr.car = locals;
    locals = scopeptr;

	/* if a new self pointer is given, save old one and change to new */
    if (newself != NULL) {
	oldself = selfbind->value.ptr.cdr;
	selfbind->value.ptr.cdr = newself;
    }

	/* evaluate the function */
    ptr = fptr;
    result = NULL;
    while (ptr != NULL) {
	result = evaluate(ptr->value.ptr.car, &exitflag);
	if (exitflag)
	    break;
	ptr = ptr->value.ptr.cdr;
    }

	/* if self pointer was changed, change it back to what it was */
    if (newself != NULL)
	selfbind->value.ptr.cdr = oldself;

	/* eliminate local bindings */
    locals = locals->value.ptr.car;

    return (result);
}



/************************************************************************
*
* eval_binding_list() - given a list of values to evaluate and a list of
*	binding names, evaluate the values and assign the results to the
*	respective names
*
************************************************************************/

#ifdef _ANSI_
NODEZ *eval_binding_list(NODEZ *valuelist, NODEZ *namelist, NODEZ *head)
#else
NODEZ *eval_binding_list(valuelist, namelist, head)
NODEZ *valuelist, *namelist, *head;
#endif
{
    int   dud;
    NODEZ *result, *nameptr;

	/* loop through the list of parameters for the function */
    result = NULL;
    while (namelist != NULL) {

	    /* normal form of the list is a list of parameter names */
	if (namelist->type == TYPElisthead) {
	    nameptr = namelist->value.ptr.car;

		/* handle being given nil as parameter name, or not a
		   valid identifier */
	    if ((nameptr == NULL) || (nameptr->type != TYPEidname)) {
		func_warning("variable name must be an identifier",
			NULL, head);
	    }

		/* just to be safe, allow for a dotted pair or the end of
		   the parameter list */
	    else if ((valuelist == NULL) || (valuelist->type != TYPElisthead)){
		result = assign_binding(nameptr->value.idptr,
			 evaluate(valuelist, &dud), result);
		valuelist = NULL;
	    }

		/* otherwise default to the normal format of a parameter */
	    else
		result = assign_binding(nameptr->value.idptr,
			 evaluate(valuelist->value.ptr.car, &dud), result);

		/* advance the list pointers for the parameter list and
		   list of parameter values */
	    namelist = namelist->value.ptr.cdr;
	    if (valuelist != NULL)
		valuelist = valuelist->value.ptr.cdr;
	}

	    /* handle occurrence of a dotted pair in a parameter list */
	else if (namelist->type == TYPEidname) {
	    result = assign_binding(namelist->value.idptr, 
				    eval_bind_list_rec(valuelist), result);
	    namelist = NULL;
	}

	    /* in case given a dotted pair with a non-identifier as parameter
		list */
	else {
	    func_warning("function variable name must be an identifier",
			NULL, head);
	    break;
	}
    }

    return (result);
}

/* recursively cons up a list of remaining parameter values for when given
 * a dotted pair in the list of parameter names
 */
#ifdef _ANSI_
NODEZ *eval_bind_list_rec(NODEZ *valuelist)
#else
NODEZ *eval_bind_list_rec(valuelist)
NODEZ *valuelist;
#endif
{
    int dud;

    if ((valuelist == NULL) || (valuelist->type != TYPElisthead))
	return (evaluate(valuelist, &dud));

    return (constructor(evaluate(valuelist->value.ptr.car, &dud),
			eval_bind_list_rec(valuelist->value.ptr.cdr)));
}



/************************************************************************
*
* is_equal_tree() - Given two trees, return true if they are equal to
*	one another.
*
************************************************************************/

#ifdef _ANSI_
int is_equal_tree(NODEZ *tree1, NODEZ *tree2)
#else
int is_equal_tree(tree1, tree2)
NODEZ *tree1, *tree2;
#endif
{
	/* recursively compare all elements of list, and abort as soon as
	   find any inequality between the lists */
    while (1) {

	    /* if either one is nil, then return true if both are nil */
	if ((tree1 == NULL) || (tree2 == NULL))
	    return (tree1 == tree2);

	    /* if both nodes do not have the same type, they are not equal */
	if (tree1->type != tree2->type)
	    return 0;

	    /* numbers and id's can be easily compared */
	if ((tree1->type == TYPEfuncname) || (tree1->type == TYPEidname)
			|| (tree1->type == TYPEnumber))
	    return (tree1->value.number == tree2->value.number);

	    /* for strings must compare the two strings */
	else if (tree1->type == TYPEstring)
	    return (!strcmp(tree1->value.idptr->name,tree2->value.idptr->name));

	    /* for lists, must recursively compare all components */
	else if (tree1->type == TYPElisthead) {
	    if (!is_equal_tree(tree1->value.ptr.car, tree2->value.ptr.car))
		return 0;
	}

	    /* for pointers to objects or classes, pointer must be equal */
	else if ((tree1->type == TYPEobject) || (tree1->type == TYPEclass))
	    return (tree1->value.ptr.car == tree2->value.ptr.car);

	else {
	    printf("Error: comparison of unknown types\n");
	    return 0;
	}

	tree1 = tree1->value.ptr.cdr;
	tree2 = tree2->value.ptr.cdr;
    }
}



/************************************************************************
*
* constructor() - Given two list nodes, cons them together and return
*	the resulting list.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *constructor(NODEZ *carptr, NODEZ *cdrptr)
#else
NODEZ *constructor(carptr, cdrptr)
NODEZ *carptr, *cdrptr;
#endif
{
    NODEZ *ptr;

    ptr = new_node();
    ptr->type = TYPElisthead;
    ptr->value.ptr.car = carptr;
    ptr->value.ptr.cdr = cdrptr;

    return (ptr);
}



/************************************************************************
*
* quoter() - Given a lisp expression, wrap the (quote ...) function
*	around it.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *quoter(NODEZ *ptr)
#else
NODEZ *quoter(ptr)
NODEZ *ptr;
#endif
{
    NODEZ *head, *nodeptr;

    nodeptr = new_node();
    nodeptr->type = TYPEfuncname;
    nodeptr->value.number = QUOTEnum;
    head = constructor(nodeptr, constructor(ptr, NULL));

    return (head);
}



/************************************************************************
*
* good_ob_ptr() - given a pointer to a TYPEobject node, return the pointer
*		  to the object, or NULL if an invalid node
*
************************************************************************/

#ifdef _ANSI_
OBJECT *good_ob_ptr(NODEZ *ptr)
#else
OBJECT *good_ob_ptr(ptr)
NODEZ *ptr;
#endif
{
    if ((ptr == NULL) || (ptr->type != TYPEobject))
	return (NULL);

    return ((ptr->value.objptr));
}



/************************************************************************
*
* obj_to_node() - Given an object pointer, convert it into a lisp node.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *obj_to_node(OBJECT *obptr)
#else
NODEZ *obj_to_node(obptr)
OBJECT *obptr;
#endif
{
    NODEZ *ptr;

    if (obptr == NULL)
	return (NULL);

    ptr = new_node();
    ptr->type = TYPEobject;
    ptr->value.objptr = obptr;

    return (ptr);
}



/************************************************************************
*
* obj_to_idname() - Given an object pointer, return its name in a lisp
*	node.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *obj_to_idname(OBJECT *obptr)
#else
NODEZ *obj_to_idname(obptr)
OBJECT *obptr;
#endif
{
    NODEZ *ptr;

    if (obptr == NULL)
	return (NULL);

    ptr = new_node();
    ptr->type = TYPEidname;
    ptr->value.idptr = obptr->idptr;

    return (ptr);
}



/************************************************************************
*
* make_findobject() - Given an object pointer, construct a lisp function
*	call which will find the named object, and return this expression.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *make_findobj(OBJECT *obptr)
#else
NODEZ *make_findobj(obptr)
OBJECT *obptr;
#endif
{
    NODEZ *ptr;

    if (obptr == NULL)
	return (NULL);

    ptr = new_node();
    ptr->type = TYPEfuncname;
    ptr->value.number = FINDOBJECTnum;

    return (constructor(ptr, constructor(quoter(obj_to_idname(obptr)), NULL)));
}



/************************************************************************
*
* assign_binding() - Given an identifier and a binding value, cons them
*	together and attach them to the given list of bindings, and then
*	return the new list of bindings.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *assign_binding(IDNODE *idptr, NODEZ *binding, NODEZ *list)
#else
NODEZ *assign_binding(idptr, binding, list)
IDNODE *idptr;
NODEZ  *binding, *list;
#endif
{
    NODEZ *ptr1;

    ptr1 = new_node();
    ptr1->type = TYPEidname;
    ptr1->value.idptr = idptr;

    return (constructor(constructor(ptr1, binding), list));
}



/************************************************************************
*
* assign_global() - Given an identifier and a binding value, cons them
*	together and add them to the list of global bindings.
*
************************************************************************/

#ifdef _ANSI_
void assign_global(IDNODE *idptr, NODEZ *binding)
#else
assign_global(idptr, binding)
IDNODE *idptr;
NODEZ  *binding;
#endif
{
    global = assign_binding(idptr, binding, global);
}



/************************************************************************
*
* find_binding() - Search for the given identifer in the list of local
*	bindings.  If not there, search for it in the list of global
*	bindings.  If found, return the binding for the identifier, such
*	that the car is the identifier name, and the cdr is the value of
*	the binding.  If no such binding is found, return NULL.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *find_binding(IDNODE *idptr)
#else
NODEZ *find_binding(idptr)
IDNODE *idptr;
#endif
{
    NODEZ *ptr, *ptr2, *ptr3;

	/* search through the list of local bindings, and return the
	   binding if it is found */
    if (locals != NULL) {
	ptr = locals->value.ptr.cdr;
	while (ptr != NULL) {

		/* ptr2 = root of binding, ptr3 = idname */
	    ptr2 = ptr->value.ptr.car;
	    ptr3 = ptr2->value.ptr.car;
	    if (idptr == ptr3->value.idptr)
		return (ptr2);

	    ptr = ptr->value.ptr.cdr;
	}
    }

	/* if reach here, then no such local binding exists, so proceed to
	   search through the list of global bindings for it */
    ptr = global;
    while (ptr != NULL) {

	    /* ptr2 = root of binding, ptr3 = idname */
	ptr2 = ptr->value.ptr.car;
	ptr3 = ptr2->value.ptr.car;
	if (idptr == ptr3->value.idptr)
	    return (ptr2);

	ptr = ptr->value.ptr.cdr;
    }

	/* if all else fails, then no such binding exists, so return NULL */
    return (NULL);
}



/************************************************************************
*
* find_function() - Search through the list of function bindings for the
*	given user-defined function.  If found, return the binding such
*	that the car is the function name and the cdr is the function
*	definition.  If no such function exists, return NULL.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *find_function(IDNODE *idptr)
#else
NODEZ *find_function(idptr)
IDNODE *idptr;
#endif
{
    NODEZ *ptr, *ptr2, *ptr3;

    ptr = functions;
    while (ptr != NULL) {

	    /* ptr2 = root of binding, ptr3 = idname */
	ptr2 = ptr->value.ptr.car;
	ptr3 = ptr2->value.ptr.car;
	if (idptr == ptr3->value.idptr)
	    return (ptr2);

	ptr = ptr->value.ptr.cdr;
    }

    return (NULL);
}



/************************************************************************
*
* argument_minimum() - Verify that the minimum number of arguments is
*	given for the named built-in function.  Return true if there is
*	an error.
*
************************************************************************/

#ifdef _ANSI_
int argument_minimum(int argnum, char *name, NODEZ *head)
#else
int argument_minimum(argnum, name, head)
int   argnum;
char  *name;
NODEZ *head;
#endif
{
    if (argument_num(head) < argnum) {
	func_warning("the %s function requires more arguments", name, head);
	return 1;
    }

    return 0;
}



/************************************************************************
*
* argument_check() - Verify that the correct number of arguments is
*	given for the named built-in function.  Return true if there is
*	an error.
*
************************************************************************/

#ifdef _ANSI_
int argument_check(int argnum, char *name, NODEZ *head)
#else
int argument_check(argnum, name, head)
int   argnum;
char  *name;
NODEZ *head;
#endif
{
    if (argument_num(head) != argnum) {
	func_warning("the %s function given incorrect number of arguments",
		name, head);
	return 1;
    }

    return 0;
}



/************************************************************************
*
* argument_num() - Return the number of arguments for a built-in function.
*
************************************************************************/

#ifdef _ANSI_
int argument_num(NODEZ *head)
#else
int argument_num(head)
NODEZ *head;
#endif
{
    int   i;
    NODEZ *ptr;

    i = -1;
    ptr = head;
    while (ptr != NULL) {
	++i;

	    /* if there is a dotted pair, give a warning and return zero
		as the number of arguments, so that the built-in functions
		will not be given any dotted pairs, except for those which
		do no require any arguments (those which require arguments
		will not be called, since the system will complain that
		they are not given a sufficient number of arguments */
	if (ptr->type != TYPElisthead) {
	    func_warning("cannot use dotted pairs with built-in functions",
			NULL, head);
	    return 0;
	}

	ptr = ptr->value.ptr.cdr;
    }

    return i;
}



/************************************************************************
*
* screen_print() - Given a LISP expression, print out that expression
*	without any parenthesis or quotes, using the print_string
*	function.  This allows the (print ...) function to handle any
*	kind of LISP expression given to it as an argument.
*
************************************************************************/

#ifdef _ANSI_
void screen_print(NODEZ *ptr)
#else
screen_print(ptr)
NODEZ *ptr;
#endif
{
    char   buffer[32];
    OBJECT *obptr;

    if (ptr == NULL)
	;
    else if (ptr->type == TYPElisthead) {
	screen_print(ptr->value.ptr.car);
	screen_print(ptr->value.ptr.cdr);
    }
    else if (ptr->type == TYPEstring)
	print_string(ptr->value.idptr->name);
    else if (ptr->type == TYPEnumber) {
	sprintf(buffer, "%d", ptr->value.number);
	print_string(buffer);
    }
    else if (ptr->type == TYPEfuncname)
	print_string(funclist[ptr->value.number].name);
    else if (ptr->type == TYPEidname)
	print_string(ptr->value.idptr->name);
    else if (ptr->type == TYPEobject) {
	obptr = ptr->value.objptr;
	print_string("&");
	print_string(obptr->idptr->name);
    }
    else if (ptr->type == TYPEclass) {
	obptr = ptr->value.objptr;
	print_string("^");
	print_string(obptr->idptr->name);
    }
    else
	printf("\nscreen_print() unknown type\n");
}



/************************************************************************
*
* dump_tree() - Dump the given tree to the output file in a format which
*	can be read back in to reconstruct the tree.  Note that in the
*	case of object pointers, the (findobject ...) function is wrapped
*	around the object's name, so that the pointer can be recovered
*	upon reloading the lisp code.
*
************************************************************************/

#ifdef _ANSI_
void dump_tree(NODEZ *ptr, FILE *outfile)
#else
dump_tree(ptr, outfile)
NODEZ *ptr;
FILE  *outfile;
#endif
{
    OBJECT *obptr;

    if (ptr == NULL)
	fprintf(outfile, "()");
    else if (ptr->type == TYPElisthead) {
	putc('(', outfile);
	while (ptr != NULL) {
	    if (ptr->type != TYPElisthead) {
		fprintf(outfile, ". ");
		dump_tree(ptr, outfile);
		break;
	    }

	    dump_tree(ptr->value.ptr.car, outfile);
	    ptr = ptr->value.ptr.cdr;

	    if (ptr != NULL)
		putc(' ', outfile);
	}

	fprintf(outfile, ")");
    }
    else if (ptr->type == TYPEfuncname)
	fprintf(outfile, funclist[ptr->value.number].name);
    else if (ptr->type == TYPEidname)
	fprintf(outfile, ptr->value.idptr->name);
    else if (ptr->type == TYPEnumber)
	fprintf(outfile, "%d", ptr->value.number);
    else if (ptr->type == TYPEstring)
	fprintf(outfile, "\"%s\"", ptr->value.idptr->name);
    else if (ptr->type == TYPEobject) {
	obptr = ptr->value.objptr;
	fprintf(outfile, "(findobject '%s)", obptr->idptr->name);
    }
    else if (ptr->type == TYPEclass) {
	printf("Error: dump_tree(): invalid class pointer\n");
	exit(1);
    }
    else {
	printf("Error: (1) unknown node type %d\n", ptr->type);
	exit(1);
    }
}



/************************************************************************
*
* print_tree() - Print the given LISP expression as-is, without making
*	any changes to the tree.  No formatting is done to the expression,
*	so no new lines or fancy indentation are bothered with.
*
************************************************************************/

#ifdef _ANSI_
void print_tree(NODEZ *ptr, FILE *outfile)
#else
print_tree(ptr, outfile)
NODEZ *ptr;
FILE  *outfile;
#endif
{
    OBJECT *obptr;

    if (ptr == NULL)
	fprintf(outfile, "nil");
    else if (ptr->type == TYPElisthead) {
	putc('(', outfile);
	while (ptr != NULL) {
	    if (ptr->type != TYPElisthead) {
		fprintf(outfile, ". ");
		print_tree(ptr, outfile);
		break;
	    }

	    print_tree(ptr->value.ptr.car, outfile);
	    ptr = ptr->value.ptr.cdr;

	    if (ptr != NULL)
		putc(' ', outfile);
	}

	fprintf(outfile, ")");
    }
    else if (ptr->type == TYPEfuncname)
	fprintf(outfile, funclist[ptr->value.number].name);
    else if (ptr->type == TYPEidname)
	fprintf(outfile, ptr->value.idptr->name);
    else if (ptr->type == TYPEnumber)
	fprintf(outfile, "%d", ptr->value.number);
    else if (ptr->type == TYPEstring)
	fprintf(outfile, "\"%s\"", ptr->value.idptr->name);
    else if (ptr->type == TYPEobject) {
	obptr = ptr->value.objptr;
	fprintf(outfile, "&%s", obptr->idptr->name);
    }
    else if (ptr->type == TYPEclass) {
	obptr = ptr->value.objptr;
	fprintf(outfile, "^%s", obptr->idptr->name);
    }
    else {
	printf("Error: (1) unknown node type %d\n", ptr->type);
	exit(1);
    }
}



/************************************************************************
*
* pretty_print_tree() - Essentially identical in purpose to print_tree(),
*	this function can only print to standard output, and attempts
*	to nicely format the expression to make it easily readable.
*
*	Unfortunately, the current version of this function breaks down
*	for large or complex expressions, and they end up being rather
*	ugly in terms of appearance.
*
************************************************************************/

#ifdef _ANSI_
int pretty_print_tree(NODEZ *ptr, int indent, int length)
#else
int pretty_print_tree(ptr, indent, length)
NODEZ *ptr;
int   indent, length;
#endif
{
    int    len, treetype;
    OBJECT *obptr;

    len = tree_length(ptr);

    if (ptr == NULL) {
	printf("nil");
	length += 3;
    }
    else if (ptr->type == TYPElisthead) {
	putchar('(');
	++length;
	treetype = is_funcall(ptr);
	++indent;
	while (ptr != NULL) {
	    if (ptr->type != TYPElisthead) {
		printf(". ");
		length += 2;
		length = pretty_print_tree(ptr, indent, length);
		break;
	    }

	    if (treetype == 0)
		length = pretty_print_tree(ptr->value.ptr.car, indent, length);
	    else if (treetype == 1) {
		len = tree_length(ptr->value.ptr.car);
		length = pretty_print_tree(ptr->value.ptr.car, indent, length);
		++treetype;
	    }
	    else if (treetype == 2) {
		indent += len + 1;
		length = pretty_print_tree(ptr->value.ptr.car, indent, length);
		++treetype;
	    }
	    else {
		if (length > indent)
		    length = pretty_indent(indent);
		length = pretty_print_tree(ptr->value.ptr.car, indent, length);
	    }

	    ptr = ptr->value.ptr.cdr;

	    if (ptr != NULL) {
		if (length > 70)
		    length = pretty_indent(indent);
		else {
		    putchar(' ');
		    ++length;
		}
	    }
	}
	putchar(')');
	++length;
    }
    else if (ptr->type == TYPEfuncname) {
	printf(funclist[ptr->value.number].name);
	length += len;
    }
    else if (ptr->type == TYPEidname) {
	printf(ptr->value.idptr->name); 
	length += len;
    }
    else if (ptr->type == TYPEnumber) {
	printf("%d", ptr->value.number);
	length += len;
    }
    else if (ptr->type == TYPEstring) {
	printf("\"%s\"", ptr->value.idptr->name);
	length += len;
    }
    else if (ptr->type == TYPEobject) {
	obptr = ptr->value.objptr;
	printf("&%s", obptr->idptr->name);
	length += len;
    }
    else if (ptr->type == TYPEclass) {
	obptr = ptr->value.objptr;
	printf("^%s", obptr->idptr->name);
	length += len;
    }
    else {
	printf("Error: (2) unknown node type %d\n", ptr->type);
	exit(1);
    }

    return (length);
}



/************************************************************************
*
* pretty_indent() - Print out spaces for indentation, using both tabs
*	and spaces to reduce the number of characters printed.
*
************************************************************************/

#ifdef _ANSI_
int pretty_indent(int indent)
#else
int pretty_indent(indent)
int indent;
#endif
{
    int i;

    if (indent) {
	putchar('\n');
	for (i = indent / 8; i > 0; --i)
	    putchar('\t');

	for (i = indent % 8; i > 0; --i)
	    putchar(' ');
    }

    return (indent);
}



/************************************************************************
*
* is_funcall() - Returns true if this given list appears to be a function
*	call.  Assumes ptr of TYPElisthead.
*
************************************************************************/

#ifdef _ANSI_
int is_funcall(NODEZ *ptr)
#else
int is_funcall(ptr)
NODEZ *ptr;
#endif
{
    NODEZ *carptr, *cdrptr;

    if (ptr != NULL) {
	carptr = ptr->value.ptr.car;
	cdrptr = ptr->value.ptr.cdr;
	if ((carptr != NULL)
	     && ((carptr->type == TYPEfuncname)
	       || ((cdrptr != NULL)
		 && (carptr->type == TYPEidname)
		 && (cdrptr->type == TYPElisthead)
		 && ((cdrptr->value.ptr.car == NULL)
		   || (cdrptr->value.ptr.car->type == TYPElisthead)))))
	    return 1;
    }

    return 0;
}



/************************************************************************
*
* tree_length() - Attempts to determine the number of characters required
*	to print out the given expression, so that formatting can be
*	guessed at by the calling function.
*
************************************************************************/

#ifdef _ANSI_
int tree_length(NODEZ *ptr)
#else
int tree_length(ptr)
NODEZ *ptr;
#endif
{
    int    len;
    OBJECT *obptr;
    char   numbuf[32];

    if (ptr == NULL)
	len = 3;
    else if (ptr->type == TYPElisthead) {
	len = 2;
	while (ptr != NULL) {
	    if (ptr->type != TYPElisthead) {
		len += 2 + tree_length(ptr);
		break;
	    }

	    len += tree_length(ptr->value.ptr.car);
	    ptr = ptr->value.ptr.cdr;

	    if (ptr != NULL)
		++len;
	}
    }
    else if (ptr->type == TYPEfuncname)
	len = strlen(funclist[ptr->value.number].name);
    else if (ptr->type == TYPEidname)
	len = strlen(ptr->value.idptr->name);
    else if (ptr->type == TYPEnumber) {
	sprintf(numbuf, "%d", ptr->value.number);
	len = strlen(numbuf);
    }
    else if (ptr->type == TYPEstring)
	len = 2 + strlen(ptr->value.idptr->name);
    else if ((ptr->type == TYPEobject) || (ptr->type == TYPEclass)) {
	obptr = ptr->value.objptr;
	len = strlen(obptr->idptr->name);
    }
    else {
	printf("Error: (3) unknown node type %d\n", ptr->type);
	exit(1);
    }

    return (len);
}



/************************************************************************
*
* load_file() - Open up the named file for input, or crash if no such
*	file can be found or opened successfully.
*
************************************************************************/

#ifdef _ANSI_
void load_file(char *name)
#else
load_file(name)
char *name;
#endif
{
    FILE *fopen();

    strcpy(fname, name);

    if ((lispfile = fopen(name, "r")) == NULL) {
	printf("Error: unable to open file \"%s\".\n", name);
	exit(1);
    }
    linenumber = 1;

/*
    printf("Opening file \"%s\" for input.\n", name);
*/
}



/************************************************************************
*
* parse_lisp() - Parse the input stream into a lisp expression and return
*	the resulting tree structure.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *parse_lisp(void)
#else
NODEZ *parse_lisp()
#endif
{
    int   i;
    NODEZ *nodeptr, *head;

    head = NULL;

    if (stringtoken) {				/* handle strings */
	head = new_node();
	head->type = TYPEstring;
	head->value.idptr = find_idstring(token);
    }
    else if (specialchar[*token]) {		/* handle special characters */
	if (*token == '(') {
	    get_token();

	    while ((*token != ')') || stringtoken) {
		if ((*token == '.') && !stringtoken) {
		    if (head == NULL) {
			printf("Error: dot cannot be first element of list\n");
			nodeptr = head = new_node();
			head->type = TYPElisthead;
		    }
		    get_token();
		    nodeptr->value.ptr.cdr = parse_lisp();
		    get_token();
		    break;
		}

		if (head == NULL)
		    nodeptr = head = new_node();
		else {
		    nodeptr->value.ptr.cdr = new_node();
		    nodeptr = nodeptr->value.ptr.cdr;
		}

		nodeptr->type = TYPElisthead;
		nodeptr->value.ptr.car = parse_lisp();
		get_token();
	    }
	}
	else if (*token == '\'') {		/* convert quotes */
	    get_token();
	    head = quoter(parse_lisp());
	}
	else if (*token == '.') {		/* incorrect dotted pair */
	    printf("Error: incorrect use of dotted pair notation\n");
	    get_token();
	}
	else if ((*token == '^') || (*token == '&')) {	/* class/obj ptr */
	    printf("Error: hit object/class pointer on input\n");
	    exit(1);
	}
	else
	    printf("Syntax error: \"%s\"\n", token);
    }
    else {
	i = find_function_num(token);
	if (i >= 0) {				/* built-in functions */
	    head = new_node();
	    head->type = TYPEfuncname;
	    head->value.number = i;
	}
	else if (!strcmp(token, "nil")) {	/* nil evaluates to nil */
	    head = NULL;
	}
	else if (is_integer(token)) {		/* integers */
	    head = new_node();
	    head->type = TYPEnumber;
	    head->value.number = str2int(token);
	}
	else {					/* default to an idname */
	    head = new_node();
	    head->type = TYPEidname;
	    head->value.idptr = find_idstring(token);
	}
    }

    return (head);
}



/************************************************************************
*
* fund_function_num() - If the given string is the name of a built-in
*	function, return the id number of that function (its offset in
*	funclist[]).  Return -1 if this is not a built-in function.
*
************************************************************************/

#ifdef _ANSI_
int find_function_num(char *name)
#else
int find_function_num(name)
char *name;
#endif
{
    int i, low, mid, high;

	/* perform a binary search in funclist[] for the given name */
    low = 0;
    high = MAXFUNC - 1;
    while (low <= high) {
	mid = (low + high) / 2;
	i = strcmp(name, funclist[mid].name);
	if (i < 0)
	    high = mid - 1;
	else if (i > 0)
	    low = mid + 1;
	else
	    return (mid);
    }

    return (-1);
}



/************************************************************************
*
* get_character() - Read one character from the input stream.  This
*	character is placed in the global variable <thischar>, while
*	the next character is placed into <nextchar>, allowing for a
*	2-byte lookahead buffer.
*
************************************************************************/

#ifdef _ANSI_
void get_character(void)
#else
get_character()
#endif
{
	/* make sure not attempting to read past the end of the file */
    if ((thischar == EOF) && (lispfile == NULL))
	printf("Error: unexpected end of input file.\n");

	/* a hack to flush the 2-byte buffer for standard input, so that
	   the newline character can be returned before prompting for
	   another line of input */
    else if ((lispfile == stdin) && (nextchar == '\n') && (thischar != '\n')) {
	thischar = nextchar;
    }

	/* otherwise, do the normal look-one-ahead */
    else {
	thischar = nextchar;
	if (lispfile != NULL) {
	    if ((nextchar = getc(lispfile)) == EOF) {
/*
		printf("End of file \"%s\" reached.\n", fname);
*/
		fclose(lispfile);
		lispfile = NULL;
	    }
	    else if (nextchar == '\t')
		nextchar = ' ';
	}
	if (thischar == '\n')
	    ++linenumber;
    }
}



/************************************************************************
*
* get_token() - Pull one token from the input stream, and place it in
*	in the global variable <token>.
*
************************************************************************/

#ifdef _ANSI_
void get_token(void)
#else
get_token()
#endif
{
    char *tokenptr;
    int  buflen;

	/* eat any irrelevant stuff (whitespace and comments) */
    for (;;) {

	    /* skip white space and blank lines */
	while ((thischar == ' ') || (thischar == '\n'))
	    get_character();

	    /* stop if have not hit a comment */
	if (thischar != ';')
	    break;

	    /* skip comments */
	while ((thischar != '\n') && (thischar != EOF))
	    get_character();
    }

    tokenptr = token;
    *tokenptr = '\0';
    stringtoken = 0;

    if (thischar == EOF)
	return;
    else if (specialchar[thischar]) {
	*tokenptr = thischar;
	++tokenptr;
	get_character();
    }
    else if (thischar == '"') {
	stringtoken = 1;
	buflen = LINELEN;
	get_character();
	for (;;) {
	    if (thischar == '\\') {
		if (nextchar == '\n') {
		    get_character();
		    get_character();
		    while (thischar == ' ')
			get_character();
		}
		else {
		    *tokenptr = thischar;
		    ++tokenptr;
		    --buflen;
		    get_character();
		    *tokenptr = thischar;
		    ++tokenptr;
		    --buflen;
		    get_character();
		}
	    }
	    else if (thischar == '"') {
		get_character();
		break;
	    }
	    else if (thischar == '\n') {
		warning_mesg("new line found within a string -- ignored");
		get_character();
	    }
	    else {
		*tokenptr = thischar;
		++tokenptr;
		get_character();
		if ((--buflen) < 5) {
		    printf("Error: string/token buffer overflow.\n");
		    exit(1);
		}
	    }
	}
    }
    else {
	do {
	    *tokenptr = thischar;
	    ++tokenptr;
	    get_character();
	} while ((thischar != '\n') && (thischar != EOF)
				    && !specialchar[thischar]);
    }

    *tokenptr = '\0';
}



/************************************************************************
*
* warning_mesg() - Display a warning message about a parsing error on
*	the input stream, including the file name and line number if
*	reading from an input file.
*
************************************************************************/

#ifdef _ANSI_
void warning_mesg(char *strg)
#else
warning_mesg(strg)
char *strg;
#endif
{
    printf("Warning");
    if (lispfile != stdin)
	printf(", %s line %d", fname, linenumber);
    printf(": %s\n", strg);
}
