/* alpng.c: Allegro png loader
    Converts a png image file into the BITMAP format used by allegro
    By L. Ross Raszewski <lraszewski@justice.loyola.edu>

    Based upon loadpng, by Peter Wang
    This program uses the libpng PNG library
*/



/* loadpng, Allegro wrapper routines for libpng
 * by Peter Wang (tjaden@psynet.net), December 1999.
 * 
 * Because all I really did was copy and paste copy out of libpng's example.c,
 * the loadpng is being released under the same licence as libpng itself.
 * 
 * COPYRIGHT NOTICE:
 *
 * The loadpng Library is supplied "AS IS".  The Contributing Authors
 * disclaim all warranties, expressed or implied, including, without
 * limitation, the warranties of merchantability and of fitness for any 
 * purpose.  The Contributing Authors no liability for direct, indirect,
 * incidental, special, exemplary, or consequential damages, which may
 * result from the use of the loadpng Library, even if advised of the 
 * possibility of such damage.
 *
 * Permission is hereby granted to use, copy, modify, and distribute this
 * source code, or portions hereof, for any purpose, without fee, subject
 * to the following restrictions:
 * 1. The origin of this source code must not be misrepresented.
 * 2. Altered versions must be plainly marked as such and must not be
 *    misrepresented as being the original source.
 * 3. This Copyright notice may not be removed or altered from any source or
 *    altered source distribution.
 *
 * The Contributing Authors specifically permit, without fee, and encourage
 * the use of this source code as a component to supporting the PNG file
 * format in commercial products.  If you use this source code in a product, 
 * acknowledgment is not required but would be appreciated.
 */


#include <png.h>
#include <allegro.h>
#include <allegro/aintern.h>



double _png_screen_gamma = -1;
int _png_compression_level = Z_BEST_COMPRESSION;



/* get_gamma:
 *  Get screen gamma value one of three ways. 
 */
static double get_gamma()
{
    char *gamma_str;
   
    if (_png_screen_gamma != -1) 
	return _png_screen_gamma;
    
    /* Use the environment variable if available.  
     * 2.2 is a good guess for PC monitors.
     * 1.1 is good for my laptop. 
     */
    gamma_str = getenv("SCREEN_GAMMA");
    return (gamma_str) ? atof(gamma_str) : 2.2;
}







/* load_png:
 *  Load a PNG file, doing colour coversion if required.
 */
BITMAP *PNGtoBITMAP(FILE *fp)
{
    PALETTE pal;
    BITMAP *bmp;
    png_structp png_ptr;
    png_infop info_ptr;
    png_uint_32 width, height, rowbytes;
    int bit_depth, color_type, interlace_type;
    double image_gamma, screen_gamma;
    int intent;
    int bpp, dest_bpp;
    int number_passes, pass, y;

    /* Create and initialize the png_struct with the desired error handler
     * functions.  If you want to use the default stderr and longjump method,
     * you can supply NULL for the last three parameters.  We also supply the
     * the compiler header file version, so that we know if the application
     * was compiled with a compatible version of the library.  
     */
    png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, (void *)NULL, NULL, NULL);
    
    /* Allocate/initialize the memory for image information. */
    info_ptr = png_create_info_struct(png_ptr);

    /* Set error handling if you are using the setjmp/longjmp method (this is
     * the normal method of doing things with libpng).  REQUIRED unless you
     * set up your own error handlers in the png_create_read_struct() earlier.
     */
    if (setjmp(png_ptr->jmpbuf)) {
	/* Free all of the memory associated with the png_ptr and info_ptr */
	png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
	/* If we get here, we had a problem reading the file */
	return NULL;
    }
    
    /* Use Allegro packfile routines. */
   png_init_io(png_ptr,fp);
    
    /* We have already read some of the signature. */
    png_set_sig_bytes(png_ptr, 0);
    
    /* The call to png_read_info() gives us all of the information from the
     * PNG file before the first IDAT (image data chunk).  
     */
    png_read_info(png_ptr, info_ptr);
    
    png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth, &color_type,
		 &interlace_type, NULL, NULL);
    
    rowbytes = png_get_rowbytes(png_ptr, info_ptr);
    
    /* Extract multiple pixels with bit depths of 1, 2, and 4 from a single
     * byte into separate bytes (useful for paletted and grayscale images).
     */
    png_set_packing(png_ptr);

    /* Expand grayscale images to the full 8 bits from 1, 2, or 4 bits/pixel */
    if ((color_type == PNG_COLOR_TYPE_GRAY) && (bit_depth < 8))
	png_set_expand(png_ptr);

    /* Convert grayscale to RGB triplets */
    if ((color_type == PNG_COLOR_TYPE_GRAY) ||
	(color_type == PNG_COLOR_TYPE_GRAY_ALPHA))
	png_set_gray_to_rgb(png_ptr);

    /* Tell libpng to handle the gamma conversion for us. */
    screen_gamma = get_gamma();
    
    if (png_get_sRGB(png_ptr, info_ptr, &intent)) { 
	png_set_sRGB(png_ptr, info_ptr, intent);
    }
    else {
	if (png_get_gAMA(png_ptr, info_ptr, &image_gamma))
	    png_set_gamma(png_ptr, screen_gamma, image_gamma);
	else
	    png_set_gamma(png_ptr, screen_gamma, 0.50);
    }
    
    /* Dither RGB files down to 8 bit palette or reduce palettes
     * to the number of colors available on your screen.
     */
	if (color_type & PNG_COLOR_MASK_COLOR) {
	    int num_palette, i;
	    png_colorp palette;
	    
	    if (png_get_PLTE(png_ptr, info_ptr, &palette, &num_palette)) {
		/* We don't actually dither, we just copy the palette. */
		for (i = 0; ((i < num_palette) && (i < 256)); i++) {
		    pal[i].r = palette[i].red >> 2;		/* 256 -> 64 */
		    pal[i].g = palette[i].green >> 2;
		    pal[i].b = palette[i].blue >> 2;		
		}
		
		for (; i < 256; i++)
		    pal[i].r = pal[i].g = pal[i].b = 0;
	    }
	}
    
    /* Flip RGB pixels to BGR. */
    if (makecol24(255, 0, 0) > makecol24(0, 0, 255))
	png_set_bgr(png_ptr);

    /* Turn on interlace handling. */
    number_passes = png_set_interlace_handling(png_ptr);

    /* Call to gamma correct and add the background to the palette
     * and update info structure.
     */
    png_read_update_info(png_ptr, info_ptr);
    
    /* Allocate the memory to hold the image using the fields of info_ptr. */
    if (color_type == PNG_COLOR_TYPE_GRAY)
	bpp = 24;
    else if (color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
	bpp = 32;
    else
	bpp = rowbytes * 8 / width;

    /* Allegro cannot handle less than 8 bpp. */
    if (bpp < 8)
        bpp = 8;

    dest_bpp = _color_load_depth(bpp); //, (bpp == 32));
    bmp = create_bitmap_ex(bpp, width, height);

    /* Read the image, one line at a line (easier to debug!) */
    for (pass = 0; pass < number_passes; pass++) {
	for (y = 0; y < height; y++)
	    png_read_rows(png_ptr, &bmp->line[y], NULL, 1);
    }
    
    if (dest_bpp != bpp)
	bmp = _fixup_loaded_bitmap(bmp, pal, dest_bpp);
    
    /* Read rest of file, and get additional chunks in info_ptr. */
    png_read_end(png_ptr, info_ptr);
    
    /* Clean up after the read, and free any memory allocated. */
    png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);

    /* Done. */
    return bmp;  
}

