
import org.ifarchive.glk.*;

/* model.c: Model program for Glk API, version 0.4.
    Designed by Andrew Plotkin <erkyrath@netcom.com>
    http://www.edoc.com/zarf/glk/index.html
    This program is in the public domain.
*/

/* Java conversion by John Elliott, 30 December 2001. */

/* This is a simple model of a text adventure which uses the Glk API.
    It shows how to input a line of text, display results, maintain a
    status window, write to a transcript file, and so on. */

class Model
{
/* The story and status windows. */
	Window mainWin = null;
	Window statusWin = null;

/* A file reference for the transcript file. */
	FileRef scriptRef = null;
/* A stream for the transcript file, when it's open. */
	Stream scriptStr = null;

/* Your location. This determines what appears in the status line. */
	int currentRoom;

/* A flag indicating whether you should look around. */
	boolean needLook; 

	public static void main(String argv[])
	{
		System.err.println("This program is a Glk program - it can't be run standalone.");	
	}
/* The glk_main() function is called by the Glk system; it's the main entry
    point for your program. */
	public static void glkMain()
	{
		try
		{
			Model m = new Model();
			m.run();
		}
		catch (Throwable e)
		{
			e.printStackTrace(System.err);
			System.err.flush();
		}
	}

	void run()
	{
	    /* Open the main window. */
		mainWin = Window.open(null, 0, 0, Window.wintype_TextBuffer, new Integer(1));
    		if (mainWin == null) 
		{
	        /* It's possible that the main window failed to open. There's
       		   nothing we can do without it, so exit. */
	        return; 
    		}
  
    		/* Set the current output stream to print to it. */
		Glk.setWindow(mainWin);
 
    /* Open a second window: a text grid, above the main window, three lines
        high. It is possible that this will fail also, but we accept that. */
		statusWin = Window.open(mainWin, Window.winmethod_Above | Window.winmethod_Fixed, 
       		 			3, Window.wintype_TextGrid, new Integer(0));
		Glk.putString("Model Glk Program\nAn Interactive Model Glk Program\n");
	    	Glk.putString("By Andrew Plotkin.\nRelease 5.\n");
		Glk.putString("Type \"help\" for a list of commands.\n");
    
		currentRoom = 0; /* Set initial location. */
		needLook = true; 
    
		while (true) 
		{
			boolean gotLine;
			int len, cx;
/*
	This is the only way to get a pointer to String in Java. 
	"command[0]" will be populated by line events; to pass in
	initial text in the input buffer, simply assign to it before
	requesting the event.	
*/
			String[] command = new String[1];
			command[0] = new String();	  
 
       			Event ev = new Event();
        
       			drawStatusWin();
        
        		if (needLook) 
			{
				needLook = false;
				Glk.putString("\n");
				Glk.setStyle(Stream.style_Subheader);

			        if (currentRoom == 0) Glk.putString("The Room\n");
            			else Glk.putString("A Different Room\n");
            			Glk.setStyle(Stream.style_Normal);
            			Glk.putString("You're in a room of some sort.\n");
        		}
        		Glk.putString("\n>");
        /* We request up to 255 characters (Java doesn't care, but the Glk
	  library does). When the event takes place, command[0] will be 
	  a valid Java string containing the command typed. */
			mainWin.requestLineEvent(command, 255);
       
			gotLine = false;
			while (!gotLine) 
       			{ 
				Glk.select(ev);
            
           			 switch (ev.type) 
				 {
                			case Event.evtype_LineInput:
                    			if (ev.win == mainWin) 
					{
                			        gotLine = true;
                        /* Really the event can *only* be from mainWin,
                            because we never request line input from the
                            status window. But we do a paranoia test,
                            because commandbuf is only filled if the line
                            event comes from the mainWin request. If the
                            line event comes from anywhere else, we ignore
                            it. */
                    			}
                    			break;
                    
                			case Event.evtype_Arrange:
                    /* Windows have changed size, so we have to redraw the
                        status window. */
                    			drawStatusWin();
                    			break;
            			}
        		}
        
	        /* commandbuf now contains a line of input from the main window.
		You would now run your parser and do something with it. */
      
		/* [JCE] Convert string to lowercase and trim whitespace 
			in one swell foop. */ 
		command[0] = Glk.stringToLower(command[0]).trim();
        
       	 /* cmd now points to a nice null-terminated string. We'll do the
       	     simplest possible parsing. */
       		if (command[0].equals("")) {
			Glk.putString("Excuse me?\n");
        	}
	        else if (command[0].equals("help")) {
	            verbHelp();
	        }
	        else if (command[0].equals("move")) {
	            verbMove();
	        }
	        else if (command[0].equals("jump")) {
	            verbJump();
	        }
	        else if (command[0].equals("yada")) {
	            verbYada();
	        }
	        else if (command[0].equals("quit")) {
	            verbQuit();
	        }
	        else if (command[0].equals("save")) {
	            verbSave();
	        }
	        else if (command[0].equals("restore")) {
	            verbRestore();
	        }
	        else if (command[0].equals("script")) {
	            verbScript();
	        }
	        else if (command[0].equals("unscript")) {
	            verbUnscript();
	        }
	        else if (command[0].equals("iterate")) {
	            verbIterate();
	        }
	        else if (command[0].equals("dispatch")) {
	            verbDispatch();
	        }
	        else if (command[0].equals("testdisp")) {
	            verbTestDisp();
	        }
	        else {
	            Glk.putString("I don't understand the command \"" + command[0] + "\".\n");
	        }
    	}
}

	void drawStatusWin()
	{
		String roomName;
		long width;	
  	 
   		 if (statusWin == null) {
        /* It is possible that the window was not successfully 
            created. If that's the case, don't try to draw it. */
	       		return;
    		}
    
		if (currentRoom == 0) 
			roomName = "The Room";
    		else
			roomName = "A Different Room";
   
		Glk.setWindow(statusWin);
		statusWin.clear();
   
		width  = statusWin.getWidth(); 
    
    /* Print the room name, centered. */
    		statusWin.moveCursor((width - roomName.length()) / 2, 1);
    		Glk.putString(roomName);
    
    /* Draw a decorative compass rose in the upper right. */
		statusWin.moveCursor(width - 3, 0);
		Glk.putString("\\|/");
		statusWin.moveCursor(width - 3, 1);
		Glk.putString("-*-");
		statusWin.moveCursor(width - 3, 2);
		Glk.putString("/|\\");
		Glk.setWindow(mainWin);
	}

	boolean yesOrNo()
	{
		String command[] = new String[1];
		command[0] = new String();
		int cx;
		int len;
		boolean gotLine;
		Event ev = new Event();
    
   		drawStatusWin();
    
    /* This loop is identical to the main command loop in glk_main(). */
    
    		while (true) 
		{
			mainWin.requestLineEvent(command, 255);
       			gotLine = false;
			while (!gotLine)
			{ 
            			Glk.select(ev);
            
           			switch (ev.type) 
				{
                			case Event.evtype_LineInput:
                    			if (ev.win == mainWin) 
					{
                        			gotLine = true;
                    			}
                    			break;
                   	 
                			case Event.evtype_Arrange:
                    			drawStatusWin();
                    			break;
				}
            		}
			command[0] = Glk.stringToUpper(command[0].trim());

			if (command[0].length() > 0 && command[0].charAt(0) == 'Y') return true;	
			if (command[0].length() > 0 && command[0].charAt(0) == 'N') return false;	
            
       	 		Glk.putString("Please enter \"yes\" or \"no\": ");
    		}
	}

	void verbHelp()
	{
	    Glk.putString("This model only understands the following commands:\n");
	    Glk.putString("HELP: Display this list.\n");
	    Glk.putString("JUMP: A verb which just prints some text.\n");
	    Glk.putString("YADA: A verb which prints a very long stream of text.\n");
	    Glk.putString("MOVE: A verb which prints some text, and also changes the status line display.\n");
	    Glk.putString("SCRIPT: Turn on transcripting, so that output will be echoed to a text file.\n");
	    Glk.putString("UNSCRIPT: Turn off transcripting.\n");
	    Glk.putString("SAVE: Write fake data to a save file.\n");
	    Glk.putString("RESTORE: Read it back in.\n");
	    Glk.putString("ITERATE: List all Glk objects in the system\n");
	    Glk.putString("DISPATCH: List all Glk constants and functions\n");
	    Glk.putString("TESTDISP: Call a few Glk functions via the dispatch interfaces\n");
	    Glk.putString("QUIT: Quit and exit.\n");
	}

	void verbJump()
	{
		Glk.putString("You jump on the fruit, spotlessly.\n");
	}

	private final int NUMWORDS = 13;
	private final String wordCapList[] = {
   		     	"Ga", "Bo", "Wa", "Mu", "Bi", "Fo", "Za", "Mo", "Ra", 
			"Po", "Ha", "Ni", "Na" };
	private final String wordList[] = {
        		"figgle", "wob", "shim", "fleb", "moobosh", "fonk", 
			"wabble", "gazoon", "ting", "floo", "zonk", "loof", 
			"lob" };

	private int wcount1 = 0;
	private int wcount2 = 0;
	private int wstep   = 1;
	private int jx      = 0;
	
	void verbYada()
	{
    /* This is a goofy (and overly ornate) way to print a long paragraph. 
        It just shows off line wrapping in the Glk implementation. */
    		int ix;
    		boolean first = true;
    
		for (ix=0; ix<85; ix++) {
        		if (ix > 0) {
            			Glk.putString(" ");
        		}
                
        		if (first) {
            			Glk.putString(wordCapList[(ix / 17) % NUMWORDS]);
            			first = false;
        		}
        
        		Glk.putString(wordList[jx]);
        		jx = (jx + wstep) % NUMWORDS;
        		wcount1++;
        		if (wcount1 >= NUMWORDS) {
            			wcount1 = 0;
            			wstep++;
            			wcount2++;
            			if (wcount2 >= NUMWORDS-2) {
                			wcount2 = 0;
                			wstep = 1;
            			}
        		}
        
		if ((ix % 17) == 16) {
            		Glk.putString(".");
            		first = true;
        		}
    		}
    	Glk.putChar('\n');
	}

	void verbMove()
	{
		currentRoom = (currentRoom+1) % 2;
		needLook = true;
    
		Glk.putString("You walk for a while.\n");
	}

	void verbQuit()
	{
		Glk.putString("Are you sure you want to quit? ");
		if (yesOrNo()) {
       			Glk.putString("Thanks for playing.\n");
        		Glk.exit();
		/* glk_exit() actually stops the process; it does not return. */
    		}
	}

void verbScript()
{
    if (scriptStr != null) {
        Glk.putString("Scripting is already on.\n");
        return;
    }
    
    /* If we've turned on scripting before, use the same file reference; 
        otherwise, prompt the player for a file. */
    if (scriptRef == null) {
        scriptRef = FileRef.createByPrompt(
            FileRef.fileusage_Transcript | FileRef.fileusage_TextMode, 
            FileRef.filemode_WriteAppend);
        if (scriptRef == null) {
            Glk.putString("Unable to place script file.\n");
            return;
        }
    }
    
    /* Open the file. */
    scriptStr = scriptRef.openFile(FileRef.filemode_WriteAppend);
    if (scriptStr == null) {
        Glk.putString("Unable to write to script file.\n");
        return;
    }
    Glk.putString("Scripting on.\n");
    mainWin.setEchoStream(scriptStr);
    scriptStr.putString("This is the beginning of a transcript.\n");
}

void verbUnscript()
{
    if (scriptStr == null) {
        Glk.putString("Scripting is already off.\n");
        return;
    }
    
    /* Close the file. */
    scriptStr.putString("This is the end of a transcript.\n\n");
    scriptStr.close();
    Glk.putString("Scripting off.\n");
    scriptStr = null;
}

void verbSave()
{
    int ix;
    FileRef saveRef;
    Stream saveStr;
    
    saveRef = FileRef.createByPrompt(
        FileRef.fileusage_SavedGame | FileRef.fileusage_BinaryMode, 
        FileRef.filemode_Write);
    if (saveRef == null) {
        Glk.putString("Unable to place save file.\n");
        return;
    }
    
    saveStr = saveRef.openFile(FileRef.filemode_Write);
    if (saveStr == null) {
        Glk.putString("Unable to write to save file.\n");
        saveRef.destroy();
        return;
    }

    saveRef.destroy(); /* We're done with the file ref now. */
    
    /* Write some binary data. */
    for (ix=0; ix<256; ix++) {
        saveStr.putChar((char)ix);
    }
   
    saveStr.close(); 
    
    Glk.putString("Game saved.\n");
}

void verbRestore()
{
    int ix;
    boolean err;
    int ch;
    FileRef saveRef;
    Stream saveStr;
    
    saveRef = FileRef.createByPrompt(
        FileRef.fileusage_SavedGame | FileRef.fileusage_BinaryMode, 
        FileRef.filemode_Read);
    if (saveRef == null) {
        Glk.putString("Unable to find save file.\n");
        return;
    }
    
    saveStr = saveRef.openFile(FileRef.filemode_Read);
    if (saveStr == null) {
        Glk.putString("Unable to read from save file.\n");
        saveRef.destroy();
        return;
    }

    saveRef.destroy(); /* We're done with the file ref now. */
    
    /* Read some binary data. */
    err = false;
    
    for (ix=0; ix<256; ix++) {
        ch = saveStr.getChar();
        if (ch == -1) {
            Glk.putString("Unexpected end of file.\n");
            err = true;
            break;
        }
        if (ch != ix) {
            Glk.putString("This does not appear to be a valid saved game.\n");
            err = true;
            break;
        }
    }
    
    saveStr.close();
    
    if (err) {
        Glk.putString("Failed.\n");
        return;
    }
    
    Glk.putString("Game restored.\n");
}


// [JCE] Added a verbIterate() to test iteration

	void verbIterate()
	{
		Window  w;
		Stream  s;
		FileRef f; 
		Object  rocks[] = new Object[1];

		Glk.setStyle(Stream.style_Header);
		Glk.putString("Windows in system:\n");	
		Glk.setStyle(Stream.style_Normal);
		for (w = Window.iterate(null, rocks); w != null; 
		     w = Window.iterate(w,    rocks))
		{
			Glk.putString(w.toString());
			Glk.putChar('\n');
		}
		Glk.setStyle(Stream.style_Header);
		Glk.putString("Streams in system:\n");	
		Glk.setStyle(Stream.style_Normal);
		for (s = Stream.iterate(null, rocks); s != null; 
		     s = Stream.iterate(s,    rocks))
		{
			Glk.putString(s.toString());
			Glk.putChar('\n');
		}
		Glk.setStyle(Stream.style_Header);
		Glk.putString("FileRefs in system:\n");	
		Glk.setStyle(Stream.style_Normal);
		for (f = FileRef.iterate(null, rocks); f != null; 
		     f = FileRef.iterate(f,    rocks))
		{
			Glk.putString(f.toString());
			Glk.putChar('\n');
		}
	}
	

	void verbDispatch()
	{
		IntConstant ic;
		Function fn;
		int n, max;

		Glk.setStyle(Stream.style_Header);
		Glk.putString("Constants in system:\n");	
		Glk.setStyle(Stream.style_Normal);
		max = IntConstant.count();
		for (n = 0; n < max; n++)	
		{
			ic = IntConstant.get(n);
			Glk.putString(ic.name + " = " + Long.toString(ic.value));
			Glk.putChar('\n');
		} 
		Glk.setStyle(Stream.style_Header);
		Glk.putString("Functions in system:\n");	
		max = Function.count();
		Glk.setStyle(Stream.style_Preformatted);
		for (n = 0; n < max; n++)	
		{
			fn = Function.get(n);
			Glk.putString(fn.name + " = " + Integer.toString(fn.id));
			Glk.putChar('\n');
			Glk.putString(fn.parsedPrototype());
			Glk.putChar('\n');
		} 
		Glk.setStyle(Stream.style_Normal);
	}

	void verbTestDisp()
	{
		Object[] params;
		Object[] event;
		Reference ref;
		OpaqueObject stream_id;
		byte[] example;
		Event e = new Event();

		// Test glk_window_get_stream(), passing the main window.
 		params = new Object[2];
		params[0] = mainWin;
		params[1] = new Reference(null); // will receive result
		Glk.dispatchCall(44, params);

		ref = (Reference)params[1];
		String s = "null";
		if (ref.target != null) s = ref.target.toString();
		// Print actual result, and expected result.
		Glk.putString("These two should be identical:\n");
		Glk.setStyle(Stream.style_Preformatted);
		Glk.putString("Through the dispatch layer: " + s + "\n");
		Glk.putString("Through normal code:        " + 
				mainWin.getStream().toString() + "\n");
		Glk.setStyle(Stream.style_Normal);
		stream_id = (OpaqueObject)ref.target;

		// Test glk_put_string_stream 
		params = new Object[2];
		params[0] = stream_id;
		params[1] = new String("Press SPACE to continue");	
		Glk.dispatchCall(131, params);
		// Test glk_request_char_event()
		params = new Object[1];
		params[0] = mainWin;
		Glk.dispatchCall(210, params);
		// Test glk_select(). 
		params = new Object[1];
		params[0] = new Reference(null);
		do	
		{
			Glk.dispatchCall(192, params);
			ref = (Reference)params[0];
			event = (Object[]) (ref.target);
			e.type = ((Number)event[0]).intValue();
       			if (e.type == Event.evtype_Arrange) drawStatusWin();
		} while (e.type != Event.evtype_CharInput);
		Glk.putString("\nGot a character event; here it is:\n");
		for (int n = 0; n < 4; n++)
		{
			Glk.putString(Integer.toString(n) + ": " + event[n].toString() + "\n");	
		}
		Glk.putString("Now type something.\n>> ");	
		// Test glk_request_line_event
		example = new byte[255];
		example[0] = 't'; example[1] = 'h'; example[2] = 'i';	
		example[3] = 'n'; example[4] = 'g';
		params = new Object[3];
		params[0] = mainWin;
		params[1] = example;
		params[2] = new Integer(5);
		Glk.dispatchCall(208, params);
		do
		{
			// Event requested. Now read back in the normal way; because
			// we didn't request it using Glk.requestLineEvent(),
			// it won't have been converted into a nice String for us.
			Glk.select(e);
       			if (e.type == Event.evtype_Arrange) drawStatusWin();
		} while (e.type != Event.evtype_LineInput);
		Glk.putString("You typed: '");	
		// Arrays don't get changed by dispatchCall() - the array 
		// that comes out is the same as the array that was put in.
		Glk.putString(new String(example, 0, (int)e.val1));
		Glk.putString("'\n");
		// Test the C-style interface on glk_put_buffer_stream().
		UniversalUnion uargs[] = new UniversalUnion[4];
		byte[] buffer = "C-style dispatch works.\n".getBytes();
		uargs[0] = new UniversalUnion(stream_id, OpaqueObject.clStream);
		uargs[1] = new UniversalUnion(true);
		uargs[2] = new UniversalUnion(buffer);
		uargs[3] = new UniversalUnion(1, buffer.length);
		Glk.dispatchCall(0x0085, uargs);
		uargs[2].release();	

		// And, finally, a function with no arguments...
		Glk.dispatchCall(0x0003, (Object[])null);	// glk_tick()
		Glk.dispatchCall(0x0003, (UniversalUnion[])null);	
	}

}
