
import org.ifarchive.glk.*;

/* multiwin.c: Sample program for Glk API, version 0.4.
    Designed by Andrew Plotkin <erkyrath@netcom.com>
    http://www.edoc.com/zarf/glk/index.html
    This program is in the public domain.
*/

/* Java conversion by John Elliott, 30 December 2001. */

/* This example demonstrates multiple windows and timed input in the
    Glk API. */



/* [JCE] Also added example opaque object and array registries.
 * Note that these classes do not do any printing directly, since 
 * they may be called while there are line events pending or from within
 * Glk printing functions. Instead they output to a Queue. Queue.flush()
 * is called when line input is requested. */

class Queue
{
	String text;
	Window window;
	
	Queue(Window w) 
	{ 
		window = w; 
		text = "";
	}

	void add(String s) 
	{ 
		text += s; 
	}
	// flush() is called whenever a line event is requested. If it's
	// for "our" window, print out the current buffer.
	// For best results, do this in the main window with the timer 
	// turned on.
	void flush(Window w) 
	{ 
		if (window != w) return;
		w.getStream().putString("\n" + text); 
		text = ""; 
	}
}

class MyObjectRegistry implements ObjectRegistry
{
	Queue  queue;
	int count;

	MyObjectRegistry(Queue q)
	{
		queue = q;
		count = 1;
	}

	public Object reg(OpaqueObject o, int glkClass)
	{
		queue.add("Registered object " + Integer.toString(count) +
				 " of class " + Integer.toString(glkClass) + "\n");
		return new Integer(count++);
	}

	public void unreg(OpaqueObject o, int glkClass, Object disprock)
	{
		String dr = "(null)";
		if (disprock != null) dr = disprock.toString();
		queue.add("Unregistered object " + dr + 
				 " of class " + Integer.toString(glkClass) + "\n");
	}
}


class MyArrayRegistry implements ArrayRegistry
{
	Queue  queue;
	int count;

	MyArrayRegistry(Queue q)
	{
		queue = q;
		count = 1;
	}

	public Object reg(Object o, String typecode)
	{
		queue.add("Registered array " + Integer.toString(count) +
				 " of type " + typecode + "\n");
		return new Integer(count++);
	}

	public void unreg(Object o, String typecode, Object disprock)
	{
		String dr = "(null)";
		if (disprock != null) dr = disprock.toString();
		queue.add("Unregistered array " + dr + " of type " + 
				typecode + "\n");
	}
}


class MultiWin
{
/* The story and status windows. */
	Window mainwin1 = null;
	Window mainwin2 = null;
	Window statuswin = null;
/* Key windows don't get stored in a global variable; we'll find them
    by iterating over the list and looking for this rock value. */
	static final Integer KEYWINROCK = new Integer(97);

/* For the two main windows, we keep a flag saying whether that window
    has a line input request pending. (Because if it does, we need to
    cancel the line input before printing to that window.) */
	boolean inputpending1, inputpending2;
/* When we cancel line input, we should remember how many characters
    had been typed. This lets us restart the input with those characters
    already in place. */
// Not used in the Java version, because Window.requestLineEvent() returns
// Strings which know how long they are.
// 	int already1, already2;

/* There's a three-second timer which can be on or off. */
	boolean timerOn = false;

// [JCE] This queue buffers messages from the object registry. */
	Queue queue = null;

/* The glk_main() function is called by the Glk system; it's the main entry
    point for your program. */
	public static void glkMain()
	{
		MultiWin mw = new MultiWin();
		mw.main();
	}


	void main()
	{
		String commandBuf1[] = new String[1];
		String commandBuf2[] = new String[1];
		commandBuf1[0] = "";	// For mainwin1
		commandBuf2[0] = "";	// For mainwin2

    		/* Open the main windows. */
		mainwin1 = Window.open(null, 0, 0, Window.wintype_TextBuffer, new Integer(1));
    		if (mainwin1 == null) {
        /* It's possible that the main window failed to open. There's
            nothing we can do without it, so exit. */
		        return; 
    		}
    
    /* Open a second window: a text grid, above the main window, five 
        lines high. It is possible that this will fail also, but we accept 
        that. */
	    statuswin = Window.open(mainwin1, 
       		Window.winmethod_Above | Window.winmethod_Fixed, 
	        5, Window.wintype_TextGrid);
        
    /* And a third window, a second story window below the main one. */
	    mainwin2 = Window.open(mainwin1, 
       		Window.winmethod_Below | Window.winmethod_Proportional, 
      		50, Window.wintype_TextBuffer);
        
    /* We're going to be switching from one window to another all the
        time. So we'll be setting the output stream on a case-by-case
        basis. Every function that prints must set the output stream
        first. (Contrast model.c, where the output stream is always the
        main window, and every function that changes that must set it
        back afterwards.) */
   
	    Glk.setWindow(mainwin1); 
	    Glk.putString("Multiwin\nAn Interactive Sample Glk Program\n");
	    Glk.putString("By Andrew Plotkin.\nRelease 1.\n");
	    Glk.putString("Type \"help\" for a list of commands.\n");
    
	    Glk.setWindow(mainwin2);
	    Glk.putString("Note that the upper left-hand window accepts character input.");
	    Glk.putString(" Hit 'h' to split the window horizontally, 'v' to split the");
	    Glk.putString(" window vertically, 'c' to close a window,");
	    Glk.putString(" and any other key (including special keys) to display key");
	    Glk.putString(" codes. All new windows accept these same keys as well.\n\n");
	    Glk.putString("This bottom window accepts normal line input.\n");
    
	    if (statuswin != null) {
	        /* For fun, let's open a fourth window now, splitting the status
	            window. */
	        Window keywin;
	        keywin = Window.open(statuswin, 
	            Window.winmethod_Left | Window.winmethod_Proportional, 
	            66, Window.wintype_TextGrid, KEYWINROCK);
	        if (keywin != null) {
	            keywin.requestCharEvent();
	        }
	    }
    
	    /* Draw the key window now, since we don't draw it every input (as
	        we do the status window. */
	    drawKeyWins();

	    inputpending1 = false;
	    inputpending2 = false;
    
    while (true) 
	{
	String cmd = null;
	boolean doneloop;
        int len;
	Window whichwin = null;
        Event ev = new Event();
        
        drawStatusWin();
        /* We're not redrawing the key windows every command. */
        
        /* Either main window, or both, could already have line input
            pending. If so, leave that window alone. If there is no
            input pending on a window, set a line input request, but
            keep around any characters that were in the buffer already. */
        
        if ((mainwin1 != null) && !inputpending1) {
            Glk.setWindow(mainwin1);
	// [JCE] Flush any output from the object / array registries
	// if it was directed to this window.
	    if (queue != null) queue.flush(mainwin1);
            Glk.putString("\n>");
            /* We request up to 255 characters. The buffer can hold 256, 
                but we are going to stick a null character at the end, so 
                we have to leave room for that. Note that the Glk library 
                does *not* put on that null character. */
            mainwin1.requestLineEvent(commandBuf1, 255);
            inputpending1 = true;
        }
        
        if ((mainwin2 != null)&& !inputpending2) {
            Glk.setWindow(mainwin2);
	// [JCE] Flush any output from the object / array registries
	// if it was directed to this window.
	    if (queue != null) queue.flush(mainwin2);
            Glk.putString("\n>");
            /* See above. */
            mainwin2.requestLineEvent(commandBuf2, 255); 
            inputpending2 = true;
        }
        
        doneloop = false;
        while (!doneloop) {
        
            /* Grab an event. */
            Glk.select(ev);
            
            switch (ev.type) {
            
                case Event.evtype_LineInput:
                    /* If the event comes from one main window or the other,
                        we mark that window as no longer having line input
                        pending. We also set commandBuf to point to the
                        appropriate buffer. Then we leave the event loop. */
                    if (mainwin1 != null && ev.win == mainwin1) {
                        whichwin = mainwin1;
                        inputpending1 = false;
                        cmd = commandBuf1[0];
                        doneloop = true;
                    }
                    else if (mainwin2 != null && ev.win == mainwin2) {
                        whichwin = mainwin2;
                        inputpending2 = false;
                        cmd = commandBuf2[0];
                        doneloop = true;
                    }
                    break;
                    
                case Event.evtype_CharInput:
                    /* It's a key event, from one of the keywins. We
                        call a subroutine rather than exiting the
                        event loop (although I could have done it
                        that way too.) */
// [JCE]
// Note that the key is cast to int. If this isn't done, then it won't
// match any of the Glk.keycode_ constants, since they're integers 
// (and negative) while ev.val1 is a long (and positive). I can't
// make the constants into longs because then they can't be used
// in a switch() statement.
// Blame the Java designers for not putting in an unsigned type.
//
                    performKey(ev.win, (int)ev.val1);
                    break;
                
                case Event.evtype_Timer:
                    /* It's a timer event. This does exit from the event
                        loop, since we're going to interrupt input in
                        mainwin1 and then re-print the prompt. */
                    whichwin = null;
                    cmd = null; 
                    doneloop = true;
                    break;
                    
                case Event.evtype_Arrange:
                    /* Windows have changed size, so we have to redraw the
                        status window and key window. But we stay in the
                        event loop. */
                    drawStatusWin();
                    drawKeyWins();
                    break;
            }
        }
        
        if (cmd == null) {
            /* It was a timer event. */
            performTimer();
            continue;
        }
        /* It was a line input event. cmd now points at a line of input
            from one of the main windows. */
       
        /* Squash to lower-case, and trim. */
	cmd = Glk.stringToLower(cmd).trim();
        
        /* cmd now points to a nice null-terminated string. We'll do the
            simplest possible parsing. */
        if (cmd.equals("")) {
            Glk.setWindow(whichwin);
            Glk.putString("Excuse me?\n");
        }
        else if (cmd.equals("help")) {
            verbHelp(whichwin);
        }
        else if (cmd.equals("jump")) {
            verbJump(whichwin);
        }
        else if (cmd.equals("yada")) {
            verbYada(whichwin);
        }
        else if (cmd.equals("both")) {
            verbBoth(whichwin);
        }
        else if (cmd.equals("clear")) {
            verbClear(whichwin);
        }
        else if (cmd.equals("page")) {
            verbPage(whichwin);
        }
        else if (cmd.equals("pageboth")) {
            verbPageBoth(whichwin);
        }
        else if (cmd.equals("timer")) {
            verbTimer(whichwin);
        }
        else if (cmd.equals("untimer")) {
            verbUntimer(whichwin);
        }
	else if (cmd.equals("registry")) {
	    verbRegistry(whichwin);
	}
	else if (cmd.equals("unregistry")) {
	    verbUnregistry(whichwin);
	}
        else if (cmd.equals("chars")) {
            verbChars(whichwin);
        }
        else if (cmd.equals("quit")) {
            verbQuit(whichwin);
        }
        else {
            Glk.setWindow(whichwin);
            Glk.putString("I don't understand the command \"");
            Glk.putString(cmd);
            Glk.putString("\".\n");
        }
        // Instead of setting "already" to 0, set the command buffers to blank 
	// strings, which has the same effect. 
        if (whichwin == mainwin1) commandBuf1[0] = "";
        else if (whichwin == mainwin2) commandBuf2[0] = "";
    }
}

	void drawStatusWin()
	{
 	    long width,height;
	    if (statuswin == null) {
	        /* It is possible that the window was not successfully 
       		   created. If that's the case, don't try to draw it. */
	        return;
    	    }
    
	    Glk.setWindow(statuswin);
	    statuswin.clear();
	    width  = statuswin.getWidth();
	    height = statuswin.getHeight();
    
    /* Draw a decorative compass rose in the center. */
    width = (width/2);
    if (width > 0)
        width--;
    height = (height/2);
    if (height > 0)
        height--;
        
    statuswin.moveCursor(width, height+0);
    Glk.putString("\\|/");
    statuswin.moveCursor(width, height+1);
    Glk.putString("-*-");
    statuswin.moveCursor(width, height+2);
    Glk.putString("/|\\");
    
}

/* This draws some corner decorations in *every* key window -- the
    one created at startup, and any later ones. It finds them all
    with glk_window_iterate. */
void drawKeyWins()
{
    Window win;
    Object[] rocks = new Object[1];
    long width, height;
    
    for (win = Window.iterate(null, rocks); win != null;
            win = Window.iterate(win, rocks)) {
        if (rocks[0] == KEYWINROCK) {
            Glk.setWindow(win);
            win.clear();
	    width = win.getWidth();
	    height= win.getHeight();
	    win.moveCursor(0,0);
            Glk.putChar('O');
	    win.moveCursor(width-1,0);
            Glk.putChar('O');
	    win.moveCursor(0, height-1);
            Glk.putChar('O');
            win.moveCursor(width-1, height-1);
            Glk.putChar('O');
        }
    }
}

/* React to character input in a key window. */
void performKey(Window win, int key)
{
    long width, height;
    int len;
    int ix;
    String buf;
    String keyname;
    
    if (key == 'h' || key == 'v') {
        Window newwin;
        int loc;
        /* Open a new keywindow. */
        if (key == 'h')
            loc = Window.winmethod_Right | Window.winmethod_Proportional;
        else
            loc = Window.winmethod_Below | Window.winmethod_Proportional;
        newwin = Window.open(win, 
            loc, 50, Window.wintype_TextGrid, KEYWINROCK);
        /* Since the new window has rock value KEYWINROCK, the
            draw_keywins() routine will redraw it. */
        if (newwin != null) {
            /* Request character input. In this program, only keywins
                get char input, so the CharInput events always call
                perform_key() -- and so the new window will respond
                to keys just as this one does. */
            newwin.requestCharEvent();
            /* We now have to redraw the keywins, because any or all of
                them could have changed size when we opened newwin.
                glk_window_open() does not generate Arrange events; we
                have to do the redrawing manually. */
            drawKeyWins();
        }
        /* Re-request character input for this window, so that future
            keys are accepted. */
        win.requestCharEvent();
        return;
    }
    else if (key == 'c') {
        /* Close this keywindow. */
        win.close();
        /* Again, any key windows could have changed size. Also the
            status window could have (if this was the last key window). */
        drawKeyWins();
        drawStatusWin();
        return;
    }
    
    /* Print a string naming the key that was just hit. */
   
    switch (key) {
        case ' ':
            keyname = "space";
            break;
        case Glk.keycode_Left:
            keyname = "left";
            break;
        case Glk.keycode_Right:
            keyname = "right";
            break;
        case Glk.keycode_Up:
            keyname = "up";
            break;
        case Glk.keycode_Down:
            keyname = "down";
            break;
        case Glk.keycode_Return:
            keyname = "return";
            break;
        case Glk.keycode_Delete:
            keyname = "delete";
            break;
        case Glk.keycode_Escape:
            keyname = "escape";
            break;
        case Glk.keycode_Tab:
            keyname = "tab";
            break;
        case Glk.keycode_PageUp:
            keyname = "page up";
            break;
        case Glk.keycode_PageDown:
            keyname = "page down";
            break;
        case Glk.keycode_Home:
            keyname = "home";
            break;
        case Glk.keycode_End:
            keyname = "end";
            break;
        default:
		// Note that since Java thinks these constants are negative,
		// the comparison operators are reversed.
            if (key <= Glk.keycode_Func1 && key >= Glk.keycode_Func12) {
                keyname = "function key";
            }
		// and for the same reason, the ">= 0" check has had to be
		// added.
            else if (key < 32 && key >= 0) {
                keyname = "ctrl-" + (char)('@' + key);
            }
            else if (key <= 255 && key >= 0) {
                keyname = "" + (char)key;
            }
            else {
                keyname = "unknown key";
            }
            break;
    }
   
    buf = "Key: " + keyname; 
    
    len = buf.length();
    
    /* Print the string centered in this window. */
    Glk.setWindow(win);
    width  = win.getWidth();
    height = win.getHeight();
    win.moveCursor(0, height/2);
    for (ix=0; ix<width; ix++)
        Glk.putChar(' ');
        
    width = width/2;
    len = len/2;
    
    if (width > len)
        width = width-len;
    else
        width = 0;
    
    win.moveCursor(width, height/2);
    Glk.putString(buf);
    
    /* Re-request character input for this window, so that future
        keys are accepted. */
    win.requestCharEvent();
}

/* React to a timer event. This just prints "Tick" in mainwin1, but it
    first has to cancel line input if any is pending. */
void performTimer()
{
    Event ev = new Event();
    
    if (mainwin1 == null) return;
    
    if (inputpending1) {
        mainwin1.cancelLineEvent(ev);
// Not used in Java; the length is commandBuf1[0].length()
//        if (ev.type == Event.evtype_Line1Input) already1 = ev.val1;
        inputpending1 = false;
    }

    Glk.setWindow(mainwin1);
    Glk.putString("Tick.\n");
}

/* This is a utility function. Given a main window, it finds the
    "other" main window (if both actually exist) and cancels line
    input in that other window (if input is pending.) It does not
    set the output stream to point there, however. If there is only
    one main window, this returns 0. */
Window printToOtherwin(Window win)
{
    Window otherwin = null;
    Event ev = new Event();

    if (win == mainwin1) {
        if (mainwin2 != null) {
            otherwin = mainwin2;
            mainwin2.cancelLineEvent(ev);
// Not used in Java; the length is commandBuf2[0].length()
//            if (ev.type == Event.evtype_LineInput) already2 = ev.val1;
            inputpending2 = false;
        }
    }
    else if (win == mainwin2) {
        if (mainwin1 != null) {
            otherwin = mainwin1;
            mainwin1.cancelLineEvent(ev); 
// Not used in Java; the length is commandBuf1[0].length()
//            if (ev.type == Event.evtype_LineInput) already1 = ev.val1;
            inputpending1 = false;
        }
    }
    
    return otherwin;
}

void verbHelp(Window win)
{
    Glk.setWindow(win);
    
    Glk.putString("This model only understands the following commands:\n");
    Glk.putString("HELP: Display this list.\n");
    Glk.putString("JUMP: Print a short message.\n");
    Glk.putString("YADA: Print a long paragraph.\n");
    Glk.putString("BOTH: Print a short message in both main windows.\n");
    Glk.putString("CLEAR: Clear one window.\n");
    Glk.putString("PAGE: Print thirty lines, demonstrating paging.\n");
    Glk.putString("PAGEBOTH: Print thirty lines in each window.\n");
    Glk.putString("TIMER: Turn on a timer, which ticks in the upper ");
    Glk.putString("main window every three seconds.\n");
    Glk.putString("UNTIMER: Turns off the timer.\n");
    Glk.putString("CHARS: Prints the entire Latin-1 character set.\n");
    Glk.putString("REGISTRY: Monitor object and array creation/destruction. " +
	          "For best results, run this in the main window with the " +
		  "timer turned on.\n");
    Glk.putString("UNREGISTRY: Stop monitoring.\n");
    Glk.putString("QUIT: Quit and exit.\n");
}

void verbJump(Window win)
{
    Glk.setWindow(win);
    
    Glk.putString("You jump on the fruit, spotlessly.\n");
}

/* Print some text in both windows. This uses print_to_otherwin() to
    find the other window and prepare it for printing. */
void verbBoth(Window win)
{
    Window otherwin;
    
    Glk.setWindow(win);
    Glk.putString("Something happens in this window.\n");
    
    otherwin = printToOtherwin(win);
    
    if (otherwin != null) {
        Glk.setWindow(otherwin);
        Glk.putString("Something happens in the other window.\n");
    }
}

/* Clear a window. */
void verbClear(Window win)
{
    win.clear();
}

/* Print thirty lines. */
void verbPage(Window win)
{
    int ix;
    
    Glk.setWindow(win);
    for (ix=0; ix<30; ix++) {
        Glk.putString(Integer.toString(ix));
        Glk.putChar('\n');
    }
}

/* Print thirty lines in both windows. This gets fancy by printing
    to each window alternately, without setting the output stream,
    by using Stream.putString() instead of Glk.putString(). 
    There's no particular difference; this is just a demonstration. */
void verbPageBoth(Window win)
{
    int ix;
    Window otherwin;
    Stream str, otherstr;
    String buf;
 
    str = win.getStream();
    otherwin = printToOtherwin(win);
    if (otherwin != null) 
        otherstr = otherwin.getStream();
    else
        otherstr = null;

    for (ix=0; ix<30; ix++) {
        buf = Integer.toString(ix) + "\n";
        str.putString(buf);
        if (otherstr != null)
            otherstr.putString(buf);
    }
}

/* Turn on the timer. The timer prints a tick in mainwin1 every three
    seconds. */
void verbTimer(Window win)
{
    Glk.setWindow(win);
    
    if (timerOn) {
        Glk.putString("The timer is already running.\n");
        return;
    }
    
    if (Glk.gestalt(Glk.gestalt_Timer) == 0) {
        Glk.putString("Your Glk library does not support timer events.\n");
        return;
    }
    
    Glk.putString("A timer starts running in the upper window.\n");
    Glk.requestTimerEvents(3000); /* Every three seconds. */
    timerOn = true;
}

/* Turn off the timer. */
void verbUntimer(Window win)
{
    Glk.setWindow(win);
    
    if (!timerOn) {
        Glk.putString("The timer is not currently running.\n");
        return;
    }
    
    Glk.putString("The timer stops running.\n");
    Glk.requestTimerEvents(0);
    timerOn = false;
}

/* Print every character, or rather try to. */
void verbChars(Window win)
{
    char ix;
    
    Glk.setWindow(win);
    
    for (ix=0; ix<256; ix++) {
        Glk.putString(Integer.toString(ix));
        Glk.putString(": ");
        Glk.putChar(ix);
        Glk.putChar('\n');
    }
}

    static final int NUMWORDS = 13;
    static final String wordcaplist[] = {
        "Ga", "Bo", "Wa", "Mu", "Bi", "Fo", "Za", "Mo", "Ra", "Po",
            "Ha", "Ni", "Na"
    };
    static final String wordlist[] = {
        "figgle", "wob", "shim", "fleb", "moobosh", "fonk", "wabble",
            "gazoon", "ting", "floo", "zonk", "loof", "lob",
    };
    int wcount1 = 0;
    int wcount2 = 0;
    int wstep = 1;
    int jx = 0;


void verbYada(Window win)
{
    /* This is a goofy (and overly ornate) way to print a long paragraph. 
        It just shows off line wrapping in the Glk implementation. */
    int ix;
    boolean first = true;
    
    Glk.setWindow(win);
    
    for (ix=0; ix<85; ix++) {
        if (ix > 0) {
            Glk.putString(" ");
        }
                
        if (first) {
            Glk.putString(wordcaplist[(ix / 17) % NUMWORDS]);
            first = false;
        }
        
        Glk.putString(wordlist[jx]);
        jx = (jx + wstep) % NUMWORDS;
        wcount1++;
        if (wcount1 >= NUMWORDS) {
            wcount1 = 0;
            wstep++;
            wcount2++;
            if (wcount2 >= NUMWORDS-2) {
                wcount2 = 0;
                wstep = 1;
            }
        }
        
        if ((ix % 17) == 16) {
            Glk.putString(".");
            first = true;
        }
    }
    
    Glk.putChar('\n');
}

	void verbQuit(Window win)
	{
	    Glk.setWindow(win);
   	 
	    Glk.putString("Thanks for playing.\n");
	    Glk.exit();
	    /* glk_exit() actually stops the process; it does not return. */
	}

	void verbRegistry(Window win)
	{
		queue = new Queue(win);
		OpaqueObject.setRegistry(new MyObjectRegistry(queue));
		Glk.setArrayRegistry(new MyArrayRegistry(queue));
	}

	void verbUnregistry(Window win)
	{
		OpaqueObject.setRegistry(null);
		Glk.setArrayRegistry(null);
	}
}
