
package org.ifarchive.glk;

/** The Glk class contains static functions wrapping non-OO Glk functions.
 *
 * @version 1.0
 * @author John Elliott
 */

public class Glk
{

 // Attempt to load the C glue code that links us to GLK.
 static {                                                                     
         try 
	 {                                                                  
            System.loadLibrary( "jniglk" ); 
         }                                                                      
         catch( UnsatisfiedLinkError e ) 
         {                                      
            System.err.println(                                                 
               "Could not load native code for Glk library support." ); 
            System.exit(1);                                                     
         }                                                                      
   }
	private Glk() {}	// Hide the constructor

	private static InterruptListener st_interruptListener;

	/* Constants for calls to gestalt() */ 
	/** Return the Glk version supported by the library.
	  * <ul>
	  * <li>Bits 31-16 = major version number
	  * <li>Bits 15-8  = minor version number
	  * <li>Bits  7-0  = revision number
	  * </ul> */
	public static final int gestalt_Version = 0;
	/** Test if a character can be input in character input. 
	 *  <p>Pass the character as the second parameter to 
	 *     the gestalt() call.</p>
	 *  <p>Returns 1 if the character can be input, 0 if not. */
        public static final int gestalt_CharInput = 1;
	/** Test if a character can be input in line input. 
	 *  <p>Pass the character as the second parameter to 
	 *     the gestalt() call.</p>
	 *  <p>Returns 1 if the character can be input, 0 if not. */
        public static final int gestalt_LineInput = 2;
        /** Test if a character can be output.
         *  <p>Pass the character as the second parameter to 
         *     the gestalt() call.</p>
	 *  <p>Pass an array with one entry to get the number of screen 
	 *    cells this character will take up when printed.
         *  <p>Returns one of gestalt_CharOutput_CannotPrint,
	 *     gestalt_CharOutput_ApproxPrint, gestalt_CharOutput_ExactPrint</p>
	 */
	public static final int gestalt_CharOutput = 3;
	/** Character cannot be meaningfully printed.
	 * @see #gestalt_CharOutput */
	public static final int gestalt_CharOutput_CannotPrint = 0;
	/** Character may not look right. For example, "ae" may be used
	 *  for the one-character "ae" ligature, or "e" for an accented "e".
	 * @see #gestalt_CharOutput */
        public static final int gestalt_CharOutput_ApproxPrint = 1;
	/** Character will be printed exactly as defined.
	 * @see #gestalt_CharOutput */
        public static final int gestalt_CharOutput_ExactPrint = 2;
	/** Test if mouse input is supported in a window type. 
	 *  <p>Pass the window type as the second parameter to  
	 *     the gestalt() call.</p>
	 *  <p>Returns 1 if mouse input is possible, 0 if not. */
	public static final int gestalt_MouseInput = 4;
	/** Test if timer events are supported. 
	 *  <p>Returns 1 if they are, 0 if not. */
	public static final int gestalt_Timer = 5;
	/** Test if graphics functions are supported. 
	 *  <p>Returns 1 if they are, 0 if not. */
	public static final int gestalt_Graphics = 6;
	/** Test if images can be drawn in a particular window type.
	 *  <p>Pass the window type as the second parameter to  
	 *     the gestalt() call.</p>
	 *  <p>Returns 1 if the window supports images, 0 if not. */
	public static final int gestalt_DrawImage = 7;
	/** Test if sound functions are supported. 
	 *  <p>Returns 1 if they are, 0 if not. */
	public static final int gestalt_Sound = 8;
	/** Test if the volume can be set for sound channels.
	 *  <p>Returns 1 if it can, 0 if not. */
	public static final int gestalt_SoundVolume = 9;
	/** Test if the library supports sound notification events.
	 *  <p>Returns 1 if it does, 0 if not. */
	public static final int gestalt_SoundNotify = 10;
	/** Test if hyperlink functions are supported. 
	 *  <p>Returns 1 if they are, 0 if not. */
	public static final int gestalt_Hyperlinks = 11;
	/** Test if a window type allows hyperlinks. 
	 *  <p>Pass the window type as the second parameter to  
	 *     the gestalt() call.</p>
	 *  <p>Returns 1 if the window allows hyperlinks, 0 if not. */
	public static final int gestalt_HyperlinkInput = 12;
	/** Test if the library supports music sound resources.
	 *  <p>Returns 1 if it does, 0 if not. */
	public static final int gestalt_SoundMusic = 13;
	/** Test if alpha channels are supported in images. 
	 *  <p>Returns 1 if they are, 0 if not. */
	public static final int gestalt_GraphicsTransparency = 14;

	// Keycodes
	/** This code is returned if a key was pressed for which there is no Glk representation. */
	public static final int keycode_Unknown  = 0xffffffff;
	/** This code is returned by the cursor left key. */
	public static final int keycode_Left     = 0xfffffffe;
	/** This code is returned by the cursor right key. */
	public static final int keycode_Right    = 0xfffffffd;
	/** This code is returned by the cursor up key. */
	public static final int keycode_Up       = 0xfffffffc;
	/** This code is returned by the cursor down key. */
	public static final int keycode_Down     = 0xfffffffb;
	/** This code is returned by the Enter / Return key. */
	public static final int keycode_Return   = 0xfffffffa;
	/** This code is returned by the Delete and Backspace keys. */
	public static final int keycode_Delete   = 0xfffffff9;
	/** This code is returned by the Escape key. */
	public static final int keycode_Escape   = 0xfffffff8;
	/** This code is returned by the Tab key. */
	public static final int keycode_Tab      = 0xfffffff7;
	/** This code is returned by the PageUp key. */
	public static final int keycode_PageUp   = 0xfffffff6;
	/** This code is returned by the PageDown key. */
	public static final int keycode_PageDown = 0xfffffff5;
	/** This code is returned by the Home key. */
	public static final int keycode_Home     = 0xfffffff4;
	/** This code is returned by the End key. */
	public static final int keycode_End      = 0xfffffff3;
	/** This code is returned by the F1 key. */
	public static final int keycode_Func1    = 0xffffffef;
	/** This code is returned by the F2 key. */
	public static final int keycode_Func2    = 0xffffffee;
	/** This code is returned by the F3 key. */
	public static final int keycode_Func3    = 0xffffffed;
	/** This code is returned by the F4 key. */
	public static final int keycode_Func4    = 0xffffffec;
	/** This code is returned by the F5 key. */
	public static final int keycode_Func5    = 0xffffffeb;
	/** This code is returned by the F6 key. */
	public static final int keycode_Func6    = 0xffffffea;
	/** This code is returned by the F7 key. */
	public static final int keycode_Func7    = 0xffffffe9;
	/** This code is returned by the F8 key. */
	public static final int keycode_Func8    = 0xffffffe8;
	/** This code is returned by the F9 key. */
	public static final int keycode_Func9    = 0xffffffe7;
	/** This code is returned by the F10 key. */
	public static final int keycode_Func10   = 0xffffffe6;
	/** This code is returned by the F11 key. */
	public static final int keycode_Func11   = 0xffffffe5;
	/** This code is returned by the F12 key. */
	public static final int keycode_Func12   = 0xffffffe4;
	/** This is the number of keycode_* constants. */
	public static final int keycode_MAXVAL   = 28; 


	/* JNI implementations of GLK methods */

	// Note that we use integer IDs here. winid_t and friends have
	// to be cast to integer (and vice versa) down in the C glue 
	// code.
	static native void jniglk_exit();
	static native void jniglk_set_interrupt_handler(boolean enable);
	static native void jniglk_tick();	
	static native long jniglk_gestalt(long sel, long param);
	static native long jniglk_gestalt_ext(long sel, long param, long arr[]);
	static native char jniglk_char_to_lower(char ch);
	static native char jniglk_char_to_upper(char ch);

	// Window calls	
	static native int jniglk_window_get_root();
	static native int jniglk_window_open(int split, int method, 
				 long size, int wintype, int rock);
	static native void jniglk_window_close(int win, StreamResult r);

	static native int jniglk_window_get_arrangement_key(int win);
	static native long jniglk_window_get_arrangement_size(int win);
	static native int  jniglk_window_get_arrangement_method(int win);
	static native void jniglk_window_set_arrangement(int win, int method, long size, long key);
	static native long jniglk_window_get_width(int win);
	static native long jniglk_window_get_height(int win);
	static native long jniglk_window_get_rock(int win);
	static native int  jniglk_window_get_type(int win);
	static native int  jniglk_window_get_parent(int win);
	static native int  jniglk_window_get_sibling(int win);
	static native void jniglk_window_clear(int win);
	static native void jniglk_window_move_cursor(int win, long xpos, long ypos);
	static native int  jniglk_window_get_stream(int win);
	static native void jniglk_window_set_echo_stream(int win, int str);
	static native int  jniglk_window_get_echo_stream(int win);
	static native void jniglk_set_window(int win);

	// Character calls
	static native int  jniglk_stream_open_file(int frefid, int fmode, int rock);
	static native int  jniglk_stream_open_memory(int bufid, int buflen, int fmode, int rock);
	static native void jniglk_stream_close(int str, StreamResult r);
	static native long jniglk_stream_get_rock(int str);
	static native void jniglk_stream_set_position(int str, int pos, int seekmode);
	static native long jniglk_stream_get_position(int str);
	static native void jniglk_set_stream(int str);
	static native int  jniglk_get_stream();

	// Note that glk_put_string() and glk_put_string_stream() are not wrapped;
	// the Java level code converts them to put_buffer() and put_buffer_stream() 
	// calls. 
	static native void jniglk_put_char(byte ch);
	static native void jniglk_put_buffer(byte ch[]);
	static native void jniglk_set_style(int style);
	
	static native int  jniglk_get_char_stream(int str);
	static native int  jniglk_get_line_stream(int str, byte ch[]);
	static native int  jniglk_get_buffer_stream(int str, byte ch[]);
	static native void jniglk_put_char_stream(int str, byte ch);
	static native void jniglk_put_buffer_stream(int str, byte ch[]);
	static native void jniglk_set_style_stream(int str, int style);

	static native void jniglk_stylehint_set  (int wintype, int styl, int hint, int val);
	static native void jniglk_stylehint_clear(int wintype, int styl, int hint);
	static native boolean jniglk_style_distinguish(int winid, int styl1, int styl2);
	static native boolean jniglk_style_measure(int winid, int styl, int hint, int[] result);
	

	// Fileref calls
	static native int  jniglk_fileref_create_temp(int usage, int rock);
	static native int  jniglk_fileref_create_by_name(int usage, String name, int rock);
	static native int  jniglk_fileref_create_by_prompt(int usage, int fmode, int rock);
	static native int  jniglk_fileref_create_from_fileref(int usage, long fref, int rock);
	static native void jniglk_fileref_destroy(long fref);
	static native long jniglk_fileref_get_rock(long fref);
	static native void jniglk_fileref_delete_file(long fref);
	static native boolean jniglk_fileref_does_file_exist(long fref);
	
	// Event calls. This call wraps select() and select_poll(), since the 
	// code to populate e can be shared.
	static native void jniglk_select(Event e, boolean poll);

	static native void jniglk_request_timer_events(long millisecs);
	static native void jniglk_request_line_event (int win, int bufid, int buflen, int initLen);
	static native void jniglk_request_char_event (int win);
	static native void jniglk_request_mouse_event(int win);
	static native void jniglk_cancel_line_event  (int win, Event e);
	static native void jniglk_cancel_char_event  (int win);
	static native void jniglk_cancel_mouse_event (int win);

	// Image functions
	static native boolean jniglk_image_draw(int win, int image, int val1, int val2);
	static native boolean jniglk_image_draw_scaled(int win, int image, int val1, int val2, long width, long height);
	static native boolean jniglk_image_get_info(int image, int[]results);

	static native void jniglk_window_flow_break(int win);

	static native void jniglk_window_erase_rect(int win, int left, int top, long width, long height);
	static native void jniglk_window_fill_rect(int win, int color, int left, int top, long width, long height);
	static native void jniglk_window_set_background_color(int win, int color);
	// Sound functions
	static native int jniglk_schannel_create(int rock);
	static native void jniglk_schannel_destroy(int chan);
	static native int jniglk_schannel_get_rock(int chan);

	static native boolean jniglk_schannel_play(int chan, int snd);
	static native boolean jniglk_schannel_play_ext(int chan, int snd, int repeats, int notify);
	static native void jniglk_schannel_stop(int chan);
	static native void jniglk_schannel_set_volume(int chan, int vol);

	static native void jniglk_sound_load_hint(int snd, boolean flag);
	// Hyperlink functions
	static native void jniglk_set_hyperlink(int linkval);
	static native void jniglk_set_hyperlink_stream(int str, int linkval);
	static native void jniglk_request_hyperlink_event(int win);
	static native void jniglk_cancel_hyperlink_event(int win);


	// Dispatch functions
	static native void   jnigidispatch_call(int func, UniversalUnion args[]);
	static native String jnigidispatch_prototype(int func);
	static native int    jnigidispatch_count_classes();
	static native int    jnigidispatch_count_intconst();
	static native IntConstant jnigidispatch_get_intconst(int index);
	static native int    jnigidispatch_count_functions();
	static native Function jnigidispatch_get_function(int index);	
	static native Function jnigidispatch_get_function_by_id(int index);	

	// Blorb functions
	static native int    jnigiblorb_set_resource_map(int str);

	// Functions with no exact equivalent in glk.h
	static native void jniglk_register_registry  ();
	static native int  jniglk_lock_byte_array    (byte[] buf);
	static native void jniglk_unlock_byte_array  (byte[] buf, int bufid);
	static native int  jniglk_lock_short_array   (short[] buf);
	static native void jniglk_unlock_short_array (short[] buf, int bufid);
	static native int  jniglk_lock_int_array     (int[] buf);
	static native void jniglk_unlock_int_array   (int[] buf, int bufid);
	static native int  jniglk_lock_long_array    (long[] buf);
	static native void jniglk_unlock_long_array  (long[] buf, int bufid);
	static native int  jniglk_lock_char_array    (char[] buf);
	static native void jniglk_unlock_char_array  (char[] buf, int bufid);

	// Callbacks from JNI to normal code. These all catch exceptions 
	// before the exception can get out into C.
	static void jniglk_interrupt()
	{
		try		
		{	
			if (st_interruptListener != null) st_interruptListener.interrupt();
		}
                catch (Throwable e)
                {
                        e.printStackTrace(System.err);
                        System.err.flush();
                }
	}

        static int jniglk_reg(int glkClass)
        {
		try
		{
			return OpaqueObject.reg(glkClass);
		}
                catch (Throwable e)
                {
                        e.printStackTrace(System.err);
                        System.err.flush();
                }
		return 0;
        }

        static void jniglk_unreg(int glkClass, int disprock)
	{
		try
		{
			OpaqueObject.unreg(glkClass, disprock);
		}
                catch (Throwable e)
                {
                        e.printStackTrace(System.err);
                        System.err.flush();
                }
	}

        static void jniglk_array_reg(int bufid, int len, String typecode)
        {
		try
		{
			GlkArray.reg(bufid, len, typecode);
		}
                catch (Throwable e)
                {
                        e.printStackTrace(System.err);
                        System.err.flush();
                }
        }

        static void jniglk_array_unreg(int bufid, String typecode)
	{
		try
		{
			GlkArray.unreg(bufid, typecode);
		}
                catch (Throwable e)
                {
                        e.printStackTrace(System.err);
                        System.err.flush();
                }
	}
	
	//////////////////////////////////////////////////////////////////////////
	// The public face of the class. This would be the same public
	// face presented by a native Glk library, if one existed.

	/** A default main() in case jniglk wasn't told what class to load */
	public static void glkMain()
	{
		Window w = Window.open(null, 0, 0, Window.wintype_TextBuffer, null);
		if (w == null) return;

		setWindow(w);
		setStyle(Stream.style_Header);
		putString("JniGlk default screen\n");
		setStyle(Stream.style_Normal);
		putString("If you are seeing this, JniGlk is working properly " +
			  "but you have not specified the class to load. " +
			  "To do this, add the class name to the JniGlk command: " +
			  "\n\n  jniglk ");
		setStyle(Stream.style_Emphasized);
		putString("class");
		setStyle(Stream.style_Normal);
		putString("\n\nFor example:\n\n");	
		setStyle(Stream.style_Input);
		putString("  jniglk org.raif-pool.Xyzzy\n");	
		setStyle(Stream.style_Normal);
	}


	/** Terminate immediately. This function does not return. */
	public static void exit()
	{
		jniglk_exit();
	}

	/** Set an interrupt handler.
	 * <p>Most platforms have some provision for interrupting a program --
	 * command-period on the Macintosh, control-C in Unix, possibly a window
	 * manager menu item, or other possibilities. This can happen at any time,
	 * including while execution is nested inside one of your own functions,
	 * or inside a Glk library function.
	 *
	 * <p>If you need to clean up critical resources, you can specify an interrupt
	 * handler class - one that implements the InterruptListener interface.
	 *
	 * <p>Initially there is no interrupt handler. You can reset to not having
	 * any by calling setInterruptHandler(null).
	 *
	 * <p>If you call glk_set_interrupt_handler() with a new handler function
	 * while an older one is set, the new one replaces the old one. Glk does
	 * not try to queue interrupt handlers.
	 * 
	 * @see InterruptListener
	 * @param i The interface that will be called if the process is 
	 *         interrupted. 
	 */
	public static void setInterruptHandler(InterruptListener i)
	{
		st_interruptListener = i;
		jniglk_set_interrupt_handler(i != null);
	}

	/** Allow the underlying GLK library to yield if necessary. 
	 *
	 * <p>Many platforms have some annoying thing that has to be done every so
	 * often, or the gnurrs come from the voodvork out and eat your computer.</p>
	 * 
	 * <p>Well, not really. But you should call glk_tick() every so often, just
	 * in case. It may be necessary to yield time to other applications in a
	 * cooperative-multitasking OS, or to check for player interrupts in an
	 * infinite loop.
	 * 
	 * <p> This call is fast; in fact, on average, it does nothing at all. So
	 * you can call it often. 
	 */
	public static void tick()
	{
		jniglk_tick();
	}

	/** Request information about the capabilities of the API. 
	 * 
	 * <p>This is exactly the same as the two-parameter version of gestalt(), 
	 * except that the second parameter is always 0.
	 *  
         * @see #gestalt(long,long)
	 * @param sel The capability you are asking about 
	 * @return Information on the requested capability, or 0 if no information is available.
	 */
        public static long gestalt(long sel) 
	{ 
		return gestalt(sel,0); 
	}


	/** Request information about the capabilities of the API. 
	 *
	 * @param sel The capability you are asking about 
	 * @param value Additional information on the parameter.
	 * @return Information on the requested capability, or 0 if no information is available.
	 */
        public static long gestalt(long sel, long param)
        {
        	return jniglk_gestalt(sel, param);
	}

	/** Request information about the capabilities of the API. 
	 *
	 * @param sel The capability you are asking about 
	 * @param value Additional information on the parameter.
	 * @param arr More values to pass if required.
	 * @return Information on the requested capability, or 0 if no information is available.
	 */
	public static long gestalt(long sel, long param, long arr[])
	{
		return jniglk_gestalt_ext(sel, param, arr);
	}

	// Use Glk's upper/lower methods on ISO-8859-1 characters, and 
	// Java's own on others.
	/** Convert a character to lower case 
	 *  @param c The character to convert
	 *  @return The lowercase version, or (if c is not uppercase) c */
	public static char charToLower(char c)
	{
		if (c >= 0 && c <= 0xFF) return jniglk_char_to_lower(c);
		return Character.toLowerCase(c);
	}
	
	/** Convert a character to upper case 
	 *  @param c The character to convert
	 *  @return The uppercase version, or (if c is not lowercase) c */
	public static char charToUpper(char c)
	{
		if (c >= 0 && c <= 0xFF) return jniglk_char_to_upper(c);
		return Character.toUpperCase(c);
	}

	// And convenience calls for Strings
	/** Convert a string to lower case
	 * @param s The string to convert
	 * @return The string with any uppercase characters converted to lower
	 */
	public static String stringToLower(String s)
	{
		StringBuffer t = new StringBuffer(s.length());
		for (int n = 0; n < s.length(); n++)
			 t.append(charToLower(s.charAt(n)));
		return new String(t);
	}

	/** Convert a string to upper case
	 * @param s The string to convert
	 * @return The string with any lowercase characters converted to upper
	 */
	public static String stringToUpper(String s)
	{
		StringBuffer t = new StringBuffer(s.length());
		for (int n = 0; n < s.length(); n++)
			 t.append(charToUpper(s.charAt(n)));
		return new String(t);
	}

	/** Wait for an event and return it.
	 *
	 * @param e An Event which will be populated when this function returns
	 */
	public static void select    (Event e) 
	{ 
		jniglk_select(e, false); 
		e.win = Window.findWindow(e.winid);
		// If it's a line input event, parse the buffer
		// back to a string
		if (e.win != null && e.type == Event.evtype_LineInput)
		{
			e.win.endLineEvent(e);
		}
	}

	/** Check for an internally spawned event.
	 * <p>The first question you now ask is, what is an internally-spawned
	 * event? selectPoll() does *not* check for or return evtype_CharInput,
	 * evtype_LineInput, or evtype_MouseInput events. It is intended for you
	 * to test conditions which may have occurred while you are computing, and
	 * not interfacing with the player. For example, time may pass during slow
	 * computations; you can use selectPoll() to see if a evtype_Timer
	 * event has occured.</p>
	 * <p>At the moment, glk_select_poll() checks for evtype_Timer, and possibly
	 * evtype_Arrange and evtype_SoundNotify events. </p>
	 * 
	 * @param e An Event which will be populated when this function returns
	 */
	public static void selectPoll(Event e) 
	{ 
		jniglk_select(e, true); 
		e.win = Window.findWindow(e.winid);
		// If it's a line input event (which it shouldn't be, but better
		// safe than sorry), parse the buffer back to a string
		if (e.win != null && e.type == Event.evtype_LineInput)
		{
			e.win.endLineEvent(e);
		}
	}

	/** Ask Glk to start sending timer events. 
	 * Timer events will be sent repeatedly until you call this method
	 * with millisecs = 0. Timer events can be missed if other events
	 * also arrive.
	 * @param millisecs Time spent between each event, 0 to cancel events. 
	 */
	public static void requestTimerEvents(long millisecs)
	{
		jniglk_request_timer_events(millisecs);	
	}

	/** Set the current output stream. 
	 *  <p>This doesn't gain you as much in Java as it does in C, because there are no
	 *  global functions, so the difference in typing is between <code>Glk.putChar()</code> and 
	 *  <code>myStream.putChar()</code> as opposed to <code>glk_put_char(c)</code> and 
	 *  <code>glk_put_char_stream(my_stream, c)</code></p> 
	 *
	 *  @param s The stream to be used as the default for output. It can be null. */
	public static void setStream(Stream s)
	{
		if (s != null) jniglk_set_stream(s.m_dispid);
		else	       jniglk_set_stream(0);
	}

	/** Set the current output stream. 
	 * @return The current output stream, or null if there is none. 
	 * @see #setStream(Stream) */
	public static Stream getStream()
	{
		return Stream.findStream(jniglk_get_stream());
	}


	/** Set the current stream to a window stream.
	 *  This is exactly identical to setStream(w.getStream()).
	 * 
	 * @param w The window whose stream will be used.
	 * @see #setStream(Stream)
	 */
	public static void setWindow(Window w)
	{
		if (w != null) 
		{
			jniglk_set_window(w.m_dispid);
		}
		else	jniglk_set_window(0);
	}

	/** Write a character to the currently selected stream.
	 * @see #setStream
	 * @see Stream#putChar(char)
	 * @param c The character to write */
	public static void putChar  (char c)   { jniglk_put_char(charToByte(c)); }
	/** Write a string to the currently selected stream.
	 * @see #setStream
	 * @see Stream#putString(String)
	 * @param s The string to write */
	public static void putString(String s) 
	{ 
		jniglk_put_buffer(stringToBuffer(s, false)); 
	}

	/** Write a character array to the currently selected stream.
	 * @see #setWindow
	 * @see Stream#putBuffer(char[])
	 * @param c The character array to write */
	public static void putBuffer(char c[]) 
	{
		jniglk_put_buffer(charsToBuffer(c, false)); 
	}
	/** Write part of a character array to the currently selected stream.
	 * @see #setWindow
	 * @see Stream#putBuffer(char[],int,int)
	 * @param c The character array to write
	 * @param first The first element to write
	 * @param len The number of elements to write */
	public static void putBuffer(char c[], int first, int len) 
	{ 
		jniglk_put_buffer(charsToBuffer(c, first, len, false)); 
	}
	/** Set output style for the currently selected stream. 
	 * @see #setWindow
	 * @see Stream#setStyle(int)
	 * @param style The style to select */
	public static void setStyle (int style) { jniglk_set_style(style); }

	/** Initialise the library. This is called from the runtime and 
	 * should not be called by programs using the library. */
	static boolean init()
	{
		try
		{
			// Create maps of Glk <--> Java objects
			OpaqueObject.init();
			jniglk_register_registry();
			return true;
		}
                catch (Throwable e)
                {
                        e.printStackTrace(System.err);
                        System.err.flush();
                }
		return false;
	}

//
// If the character is printable in ISO-8859-1, return it; else '?'
//
	static byte  charToByte(char c)
	{
		if ((c & 0xFF00) != 0) return '?';
		return (byte)(c & 0xFF);
	}


/** Convert a Java string to an ISO-8859-1 buffer. 
 * @param s The string to convert.
 * @param terminate true to add a 0 to the end of the buffer, else false.
 */
	static byte[] stringToBuffer(String s, boolean terminate)
	{
		if (s == null) return stringToBuffer("(null)", terminate);
		int len, bufsiz;
		
		bufsiz = len = s.length();
		if (terminate) ++bufsiz;
	
		byte[] buf = new byte[bufsiz];
		for (int n = 0; n < len; n++) 
		{
			buf[n] = charToByte(s.charAt(n));	
		}
		if (terminate) buf[len] = 0;
		return buf;
	}
	

/** Convert a Java character array to an ISO-8859-1 buffer. 
 * @param c The array to convert.
 * @param terminate true to add a 0 to the end of the buffer, else false.
 */
	static byte[] charsToBuffer(char c[], boolean terminate)
	{
		return charsToBuffer(c, 0, c.length, terminate);
	}

/** Convert part of a Java character array to an ISO-8859-1 buffer. 
 * @param c The array to convert.
 * @param first The index of first character to convert.
 * @param len The number of characters to convert.
 * @param terminate true to add a 0 to the end of the buffer, else false.
 */
	static byte[] charsToBuffer(char c[], int first, int len, boolean terminate)
	{
		int bufsiz = len;
		if (terminate) ++bufsiz;

		byte[] buf = new byte[bufsiz];
		for (int n = 0; n < len; n++) 
		{
			buf[n] = charToByte(c[first + n]);	
		}
		if (terminate) buf[len] = 0;
		return buf;
	}

	/** Set a stylehint for a window class.
	 * @param wintype The type of window to set the hint for, or Window.wintype_AllTypes 
	 * @param style The style to set (Stream.style_*)
	 * @param hint The stylehint selector (Stream.stylehint_*) 
	 * @param val The new value of the hint. */
        public static void styleHintSet  (int wintype, int styl, int hint, int val)
	{
		jniglk_stylehint_set(wintype, styl, hint, val);
	}


	/** Unset a stylehint for a window class.
	 * @param wintype The type of window to set the hint for, or Window.wintype_AllTypes 
	 * @param style The style to set (Stream.style_*)
	 * @param hint The stylehint selector (Stream.stylehint_*)  */
        public static void styleHintClear(int wintype, int styl, int hint)
	{
		jniglk_stylehint_clear(wintype, styl, hint);
	}

	/** Are two styles visually distinguishable?
	 * @param w The window in which to check
	 * @param styl1 The first style  (Stream.style_*)
	 * @param styl2 The second style (Stream.style_*)
	 * @return true if the two styles can be distinguished, otherwise false.
	 */
        public static boolean styleDistinguish(Window w, int styl1, int styl2)
	{
		return jniglk_style_distinguish(w.m_dispid, styl1, styl2);	
	}

	/** Test an attribute of a style in the given window.
	 * @param w The window in which to check.
	 * @param styl The style to test (Stream.style_*).
	 * @param hint The hint to try and read (Stream.stylehint_*)
	 * @return true if the attribute could be determined, otherwise false.
	 */
        public static boolean styleMeasure(Window w, int styl, int hint, int[] result)
	{
		return jniglk_style_measure(w.m_dispid, styl, hint, result);
	}


	/** Convert a list of Java arguments into a list of UniversalUnions.
	  If pass1, just counts how many arguments there are to convert.
	  If pass2, actually puts the results into uargs.
	  @param pass1 True to count arguments, false to convert them.
          @param proto Prototypes for each argument.
	  @param args The arguments. 	
	  @param uargs The array of UniversalUnions which will be filled. */
	static int convertArgs(boolean pass1, Prototype[] proto, Object args[],
					 UniversalUnion uargs[])
	{
		int nu, na, dispid;

		if (proto == null) return 0;	// No arguments

		nu = 0;
		for (na = 0; na < proto.length; na++)
		{
			Object arg = args[na];

			nu = UniversalUnion.encodeObject(pass1, args[na], 
							  proto[na], nu, uargs);
		}
/* debugging code
		if (!pass1)
		{
			for (na = 0; na < nu; na++)
			{
				System.out.println(uargs[na].toString());
			}
		} */
		return nu;
	}

	/** Retrieve return values from an array of UniversalUnions.
          @param proto Prototypes for each argument.
	  @param args The arguments. 	
	  @param uargs The array of UniversalUnions which will be read. */
	static void unconvertArgs(Prototype proto[], Object args[], UniversalUnion uargs[])
	{
		int nu, na, dispid;

		if (proto == null) return;	// No arguments

		nu = 0;
		for (na = 0; na < proto.length; na++)
		{
			nu = UniversalUnion.decodeObject(args, na, proto[na], nu, uargs);
		}
	}

	/** Dispatch interface. 
	 * <p>In Java, the normal dispatch interface is a bit different from
	 * the one that the C library provides. Each argument corresponds
	 * exactly to one entry in the args[] array. Argument encoding works
	 * like this:
	 * <ul>
	 * <li>Basic types (integers, characters) are passed in as instances
	 *   of the Java wrapper classes (Integer, Character etc.)
	 * <li>Opaque objects are passed in as-is.
	 * <li>To pass NULL to a reference argument, pass null.
	 * <li>To pass an actual reference to a reference argument, pass a
	 * Reference object whose target is the current value. For example,
	 * to pass in a reference to a window, use:
	 * <pre>
	 *      args[n] = new Reference(theWindow);
	 * </pre>
	 * A pass-out reference (one that will be filled in by the library)
	 * can be done like this:
	 * <pre>
	 *      args[n] = new Reference(null);
	 * </pre>
	 * <li>Arrays are passed in as the array itself - eg:
	 * <pre>
	 *      byte array[] = new byte[64];
	 *      args[n] = array;
	 * </pre>
	 * <li>A reference to a structure is passed in a Reference to 
	 *    an array of Objects, one for each field. For example:
	 * <pre>
	 * 	Object[] structure = new Object[2];
	 *	structure[0] = mainWindow;
	 *	structure[1] = new Integer(3);
	 * 	args[0] = new Reference(structure);
	 * </pre>
	 * </ul>
	 * <p>Here are two examples: calling glk_select():</p>
	 * <pre> 
	 *	Event e = new Event();
	 *	Object args[] = new Object[1];
	 *	args[0] = new Reference(null);
	 * 	Glk.dispatchCall(0x00C0, args);
	 *	Object out[] = (Object[])( ((Reference)args[0]).target );
	 *	ev.type   = (Number)(out[0]).intVal();
	 *	ev.window = (Window)(out[1]);
	 *	ev.val1   = (Number)(out[2]).longVal();
	 *	ev.val2   = (Number)(out[3]).longVal();
	 * </pre>
	 * and calling glk_put_buffer_stream:
	 * <pre>
	 *	byte buf[64];
	 *	...populate buf...
	 * 	Object args[] = new Object[2];
	 *	args[0] = myStream;
	 * 	args[1] = buf;
	 *	Glk.dispatchCall(0x0085, args);
	 * </pre>
	 * <p><strong>An important note:</strong> Any Reference argument 
	 * passed into a function won't necessarily come back referring to
	 * the same object. For example, if you did this for glk_select():</p>
	 * <pre> 
	 *	Event e = new Event();
	 * 	Object in[] = new Object[4];
	 *	Object args[] = new Object[1];
	 *	args[0] = new Reference(in);
	 * 	Glk.dispatchCall(0x00C0, args);
	 *	Object out[] = (Object[])( ((Reference)args[0]).target );
	 * </pre>	
	 * then the array in[] would not be the same as the array out[], 
	 * and it would be out[] that was populated.</p>
	 *
	 * @param func The Glk function ID
	 * @param args The argument list, in Java form.
	 */
	public static void dispatchCall(int func, Object args[])   
	{
		UniversalUnion uargs[] = null;
		String proto = Function.prototype(func);
		Prototype p[] = Prototype.parseString(proto);
		int na;

		// Two passes will be needed. The first one works out how
		// many arguments to pass; the second then creates them.
		na = convertArgs(true, p, args, null);	
		if (na != 0)	// No arguments.
		{
			uargs = new UniversalUnion[na];
			convertArgs(false, p, args, uargs); 
		}
		dispatchCall(func, uargs);
		// Convert returned values back to Java.
		if (na != 0) unconvertArgs(p, args, uargs);
	}

	/** Dispatch interface (C-style version).
	 * <p>This version of the Dispatch interface does no marshalling,
	 * so your calling code will have to do all the setting up of 
	 * UniversalUnions (1 or 2 per reference, 1 or 3 per array, etc.)
	 * <p>The UniversalUnion documentation contains some examples of
	 * how to do this for various different types.
	 *
	 * @param func ID of Glk function to call.
	 * @param uargs Marshalled arguments. 
	 * @see #dispatchCall(int,java.lang.Object[])
	 * @see UniversalUnion
	 */
	public static void dispatchCall(int func, UniversalUnion uargs[])
	{
		jnigidispatch_call(func, uargs);
	}

        // Hyperlinks
	/** Set the current hyperlink value in the current output stream.
	 * A link value is any non-zero integer; zero indicates no link. 
	 * Subsequent text output is considered to make up the body of the 
	 * link, which continues until the link value is changed 
	 * (or set to zero).
	 * @param linkval The link value, 0 for no hyperlink.
	 */
        public static void setHyperlink(int linkval)
        {
                jniglk_set_hyperlink(linkval);
        }

	/** Set a retained array registry. It will be called when Glk
	 *  retains or releases arrays.
	 * @param r An instance of your array registry. */
	public static void setArrayRegistry(ArrayRegistry r)
	{
		GlkArray.setArrayRegistry(r);
	}

}
