
package org.ifarchive.glk;

/** Code to deal with parsing a Glk prototype string.
 * The Prototype class represents one entry; you can use 
 * Prototype.parseString() to parse a string to an array of Prototypes.
 * 
 * @see #parseString(java.lang.String)
 * @author John Elliott
 * @version 1.0
 */

public class Prototype
{
	/** Index of first character of this prototype in the prototype string. */
	int pspos;
	/** Index of first character of next prototype in the prototype string. */
	int psnext;
	/** The prototype's major type. One of: 
 	 * <ul>
	 * <li>I : Integer
	 * <li>C : Character
	 * <li>S : C-style string
	 * <li>F : Float (not currently supported)
	 * <li>Q : Opaque object
	 * <li>[ : Structure
	 * </ul>
	 */
	public char major;	
	/** The prototype's minor type.
	 * <ul>
	 * <li>For an integer: u for unsigned, s for signed. 
	 * <li>For a character: u for unsigned, s for signed, n for native. 
	 * <li>For an opaque object: Glk class + 'a'.
	 * <li>For others: zero 
	 * </ul> */
	public char minor;

	/** One or more of the modifiers MOD_* corresponding to modifier
	 * characters in the prototype string. */
	public int modifiers;	// See below
	/** This is a reference parameter being passed in. */
	public static final int MOD_REF_IN   = 1;
	/** This is a reference parameter being passed out. */
	public static final int MOD_REF_OUT  = 2;
	/** This is a reference parameter which can't be null. */
	public static final int MOD_NOTNULL  = 4;
	/** This is a structure parameter. */
	public static final int MOD_STRUCT   = 8;
	/** This is an array parameter. */
	public static final int MOD_ARRAY    = 16;
	/** This array may be retained by Glk. The Java binding ignores this
         * and assumes all arrays may be retained. */
	public static final int MOD_RETAINED = 32;

	/** Prototypes for each field in the structure.
 	 * Null if this prototype is not for a structure. */
	public Prototype structure[];	// Structure fields

	/** Is this a reference argument? 
	 * @return true if this is a pass-in or pass-out reference. */
	public boolean isRef()
	{
		return ((modifiers & (MOD_REF_IN | MOD_REF_OUT)) != 0);
	}

	/** Is this reference non-null? 
	 * @return true if this reference can't be null. */
	public boolean isNotNull()
	{
		return ((modifiers & MOD_NOTNULL) != 0);
	}
	
	/** Is this prototype for an array?
	 * @return true if this prototype is for an array parameter. */
	public boolean isArray()
	{
		return ((modifiers & MOD_ARRAY) != 0);
	}

	/** Convert to a printable string. 
	 * @return a printed representation of this object. */
	public String toString()
	{
		String s = "Prototype: Major type = " + major;
		if (minor != 0)
			s += " Minor type = " + minor;
		if ((modifiers & MOD_REF_IN)   != 0) s += " Reference in";
		if ((modifiers & MOD_REF_OUT)  != 0) s += " Reference out";
		if ((modifiers & MOD_NOTNULL)  != 0) s += " Not null";
		if ((modifiers & MOD_RETAINED) != 0) s += " Retained";
		if ((modifiers & MOD_STRUCT)   != 0) 
		{
			int m = structure.length;
			s += " Struct[";
			for (int n = 0; n < m; n++)
			{
				s += ";maj="+Integer.toString(structure[n].major);
				if (structure[n].minor != 0)
				{
					s += ",min="+Integer.toString(structure[n].minor);
				}
			} 
			s += "]";
		}
		if ((modifiers & MOD_ARRAY)    != 0) s += " Array";
		return s;
	}

	/** Parse an entry in a prototype string.
	 * @param proto The string to parse.
	 * @param np The first character to look at. */
	Prototype(String proto, int np)
	{
		StringBuffer st = new StringBuffer();
		pspos = np;
		modifiers = 0;
		boolean retv = false;

		// Parse any modifier characters before the main argument.
		while (proto.charAt(np) < 'A' || proto.charAt(np) > 'Z') 
		{
			switch(proto.charAt(np))
			{
				case '&': modifiers |= MOD_REF_IN | MOD_REF_OUT; 
					break;	// Reference I,O
				case '<': modifiers |= MOD_REF_OUT; break; 	// Ref O
				case '>': modifiers |= MOD_REF_IN; break; 	// Ref I
				case '+': modifiers |= MOD_NOTNULL; break;	// not NULL
				case ':': modifiers |= MOD_NOTNULL | MOD_REF_OUT; 
					  retv = true; break;	// "<+"
				case '[': // Structure. Break out its 
					// members into their own little string
					// and recursively parse that.
					  modifiers |= MOD_STRUCT;
					  ++np;
					  while (proto.charAt(np) != ']')
					  {
						st.append(proto.charAt(np));
					 	++np;
				  	  }
					  structure = parseString(new String(st));
					  major = '[';
		  			  minor = 0;
					  psnext = ++np;
					  return;
				case '#': modifiers |= MOD_ARRAY;  break;	// Array
				case '!': modifiers |= MOD_RETAINED; break;	// Retained
			}
			++np;
		}
	
		// Proto[np] is now the major class
		major = proto.charAt(np);
		switch(major)
		{
			case 'I': case 'C': case 'Q': 
				minor = proto.charAt(np+1);
				np += 2;
				break;
			default: minor = 0;
				 np++;
				 break;
		}
		psnext = np;
	}

	/** Parse a Glk argument string.
	 *  Converts the encoded string into an array of Prototype objects.
	 * @param proto The string to parse, which must be a complete Glk
	 *          argument string or structure definition string.
	 * @return  an array of Prototype objects, one for each argument.
	 */
	public static Prototype[] parseString(String proto)
	{
                StringBuffer sargc = new StringBuffer();
                int argc;
		int np = 0;
		int na;
		Prototype a[];
		
                while (Character.isDigit(proto.charAt(np)))
                {
                        sargc.append(proto.charAt(np));
                        ++np;
                }
                argc = Integer.valueOf(new String(sargc)).intValue();	
		if (argc == 0) return null;

		a = new Prototype[argc];
		for (na = 0; na < argc; na++)
		{
			a[na] = new Prototype(proto, np);
			np = a[na].psnext;
		}	
		return a;
	}
}
