
#include <stdio.h>	/* for sprintf */
#include <stdlib.h>	/* for malloc */
#include <string.h>	/* for strlen */
#include <stdarg.h>

#include "glk.h"
#include "gi_dispa.h"
#include "gi_blorb.h"
#include "glkstart.h"
#include "org_ifarchive_glk_Glk.h"

#define PACKAGE "org/ifarchive/glk/"

#define GLUI32(x)    ((glui32)((x) & 0xFFFFFFFFL))

#define WINID_T(x)      ((winid_t )dispid_to_object(__FUNCTION__,gidisp_Class_Window, (x)))
#define STRID_T(x)      ((strid_t )dispid_to_object(__FUNCTION__,gidisp_Class_Stream, (x)))
#define FREFID_T(x)     ((frefid_t)dispid_to_object(__FUNCTION__,gidisp_Class_Fileref, (x)))
#define SCHANNELID_T(x) ((schanid_t)dispid_to_object(__FUNCTION__,gidisp_Class_Schannel, (x)))
#define JBYTE_PTR(x)    ((jbyte  *) dispid_to_array (__FUNCTION__,x))
#define JSHORT_PTR(x)   ((jshort *) dispid_to_array (__FUNCTION__,x))
#define JINT_PTR(x)     ((jint   *) dispid_to_array (__FUNCTION__,x))
#define JCHAR_PTR(x)    ((jchar  *) dispid_to_array (__FUNCTION__,x))
#define JLONG_PTR(x)    ((jlong  *) dispid_to_array (__FUNCTION__,x))
#define WINMAP(x)	object_to_dispid(__FUNCTION__,gidisp_Class_Window, (x))
#define STRMAP(x)	object_to_dispid(__FUNCTION__,gidisp_Class_Stream, (x))
#define FREFMAP(x)	object_to_dispid(__FUNCTION__,gidisp_Class_Fileref, (x))
#define SCHANNELMAP(x)	object_to_dispid(__FUNCTION__,gidisp_Class_Schannel, (x))

static void jniglk_exit(void);	/* Forward declarations */
static void  *dispid_to_object(char *func, glui32 glkClass, glui32 dispid);
static glui32 object_to_dispid(char *func, glui32 glkClass, void *object); 
static void  *dispid_to_array(char *func, glui32 dispid);
static glui32 array_to_dispid(char *func, void *object); 


/* Any fatal error in jniglk comes here. It is printed on the Glk root 
 * window (if available) and to standard error. */
static void diewith(char *s, ...)
{
	va_list ap;
	char buf[4000];
	winid_t root;
	strid_t stream;

	strcpy(buf, "jniglk fatal error: ");	
	va_start(ap, s);
	vsprintf(buf + strlen(buf), s, ap);
	va_end(ap);

	root = glk_window_get_root();
	if (root)
	{
		stream = glk_window_get_stream(root);
		glk_set_style_stream(stream, style_Alert);
		glk_put_char_stream(stream, '\n');
		glk_put_string_stream(stream, buf);
		glk_put_char_stream(stream, '\n');
	}
	fprintf(stderr, "%s\n", buf);
	jniglk_exit();
	glk_exit();
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1exit
  (JNIEnv *env, jclass c)
{
	jniglk_exit();
	glk_exit();
}

static JNIEnv    *cbEnv;

static void jniglk_interrupt(void)
{
	jclass clazz;
	jmethodID method;

	clazz = (*cbEnv)->FindClass(cbEnv, PACKAGE "Glk");
	if (clazz == NULL)
	{
		fprintf(stderr, "jniglk_interrupt(): Can't find Glk class\n");
		return;
	}
	method = (*cbEnv)->GetStaticMethodID(cbEnv, clazz, "jniglk_interrupt", "()V");
	if (method == NULL)
	{
		fprintf(stderr, "jniglk_interrupt(): Can't find Glk.jniglk_interrupt method\n");
	}
	(*cbEnv)->CallStaticVoidMethod(cbEnv, clazz, method);
}



JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1set_1interrupt_1handler
  (JNIEnv *env, jclass c, jboolean enable)
{
	cbEnv = env;
	if (enable) glk_set_interrupt_handler(jniglk_interrupt);
	else	    glk_set_interrupt_handler(NULL);	
}



JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1tick
  (JNIEnv *env, jclass c)
{
	glk_tick();
}



JNIEXPORT jlong JNICALL Java_org_ifarchive_glk_Glk_jniglk_1gestalt
  (JNIEnv *env, jclass c, jlong sel, jlong value)
{
	return glk_gestalt( GLUI32(sel), GLUI32(value));
}

JNIEXPORT jlong JNICALL Java_org_ifarchive_glk_Glk_jniglk_1gestalt_1ext
  (JNIEnv *env, jclass c, jlong sel, jlong value, jlongArray jarr)
{
	glui32 arrlen;
	glui32 *arr;
	glui32 res, n;
	jlong *larray;

	if (arr == NULL) return glk_gestalt( GLUI32(sel), GLUI32(value));
	arrlen = (*env)->GetArrayLength(env, jarr);
	arr = malloc(arrlen * sizeof(glui32));
	if (!arr) 
	{
		jclass clazz = (*env)->FindClass(env, "java/lang/NullPointerException");
		(*env)->ThrowNew(env, clazz, "Cannot allocate memory for glk_gestalt_ext");
		return 0;
	}
	larray = (*env)->GetLongArrayElements(env, jarr, NULL);
	for (n = 0; n < arrlen; n++) arr[n] = GLUI32(larray[n]);
	res = glk_gestalt_ext(GLUI32(sel), GLUI32(value), arr, arrlen);
	for (n = 0; n < arrlen; n++) larray[n] = arr[n];
	(*env)->ReleaseLongArrayElements(env, jarr, larray, 0);
	free(arr);
	return res;
}


JNIEXPORT jchar JNICALL Java_org_ifarchive_glk_Glk_jniglk_1char_1to_1lower
  (JNIEnv *env, jclass c, jchar ch)
{
	return glk_char_to_lower((unsigned char)ch);
}


JNIEXPORT jchar JNICALL Java_org_ifarchive_glk_Glk_jniglk_1char_1to_1upper
  (JNIEnv *env, jclass c, jchar ch)
{
	return glk_char_to_upper((unsigned char)ch);
}



JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1root
  (JNIEnv *env, jclass c)
{
	return WINMAP(glk_window_get_root());
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1open
  (JNIEnv *env, jclass c, jint split, jint method, jlong size, jint wintype, jint rock)
{
	return WINMAP(glk_window_open( WINID_T(split), GLUI32(method), 
				GLUI32(size), GLUI32(wintype), rock));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1close
  (JNIEnv *env, jclass c, jint win, jobject result)
{
	jclass   rc;
	jfieldID rid, wid;
	stream_result_t res;

	glk_window_close( WINID_T(win), &res);

	if (result == NULL)
	{
		return;
	}
	rc  = (*env)->GetObjectClass(env, result);
	rid = (*env)->GetFieldID(env, rc, "readCount", "J");
	wid = (*env)->GetFieldID(env, rc, "writeCount", "J");
	(*env)->SetLongField(env, result, rid, res.readcount);
	(*env)->SetLongField(env, result, wid, res.writecount);
}



JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1arrangement_1key
  (JNIEnv *env, jclass c, jint winid)
{
	winid_t key;
	glui32 method, size;
	glk_window_get_arrangement(WINID_T(winid), &method, &size, &key);
	return WINMAP(key);
}

JNIEXPORT jlong JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1arrangement_1size
  (JNIEnv *env, jclass c, jint winid)
{
	winid_t key;
	glui32 method, size;
	glk_window_get_arrangement(WINID_T(winid), &method, &size, &key);
	return size;
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1arrangement_1method
  (JNIEnv *env, jclass c, jint winid)
{
	winid_t key;
	glui32 method, size;
	glk_window_get_arrangement(WINID_T(winid), &method, &size, &key);
	return method;
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1set_1arrangement
  (JNIEnv *env, jclass c, jint win, jint method, jlong size, jlong key)
{
	glk_window_set_arrangement(WINID_T(win), GLUI32(method), GLUI32(size), WINID_T(key));
}


JNIEXPORT jlong JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1width
  (JNIEnv *env, jclass c, jint winid)
{
	glui32 w;
	glk_window_get_size(WINID_T(winid), &w, NULL);
	return w;
}

JNIEXPORT jlong JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1height
  (JNIEnv *env, jclass c, jint winid)
{
	glui32 h;
	glk_window_get_size(WINID_T(winid), NULL, &h);
	return h;
}

JNIEXPORT jlong JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1rock
  (JNIEnv *env, jclass c, jint winid)
{
	return glk_window_get_rock(WINID_T(winid));
}


JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1type
  (JNIEnv *env, jclass c, jint winid)
{
	return glk_window_get_type(WINID_T(winid));
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1parent
  (JNIEnv *env, jclass c, jint winid)
{
	return WINMAP(glk_window_get_parent(WINID_T(winid)));
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1sibling
  (JNIEnv *env, jclass c, jint winid)
{
	return WINMAP(glk_window_get_sibling(WINID_T(winid)));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1clear
  (JNIEnv *env, jclass c, jint winid)
{
	glk_window_clear(WINID_T(winid));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1move_1cursor
  (JNIEnv *env, jclass c, jint winid, jlong x, jlong y)
{
	glk_window_move_cursor(WINID_T(winid), GLUI32(x), GLUI32(y));
}


JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1stream
  (JNIEnv *env, jclass e, jint win)
{
	return STRMAP(glk_window_get_stream(WINID_T(win)));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1set_1echo_1stream
  (JNIEnv *env, jclass c, jint win, jint str)
{
	glk_window_set_echo_stream(WINID_T(win), STRID_T(str));

}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1get_1echo_1stream
  (JNIEnv *env, jclass c, jint win)
{
	return STRMAP(glk_window_get_echo_stream(WINID_T(win)));
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1set_1window
  (JNIEnv *env, jclass c, jint win)
{
	glk_set_window(WINID_T(win));
}


JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1stream_1open_1file
  (JNIEnv *env, jclass c, jint fref, jint mode, jint rock)
{
	return STRMAP(glk_stream_open_file(FREFID_T(fref), mode, rock));
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1stream_1open_1memory
  (JNIEnv *env, jclass c, jint bufid, jint buflen, jint mode, jint rock)
{
	return STRMAP(glk_stream_open_memory(JBYTE_PTR(bufid), buflen, mode, rock));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1stream_1close
  (JNIEnv *env, jclass c, jint strid, jobject result)
{
        jclass   rc;
        jfieldID rid, wid;
        stream_result_t res;
	
        glk_stream_close( STRID_T(strid), &res);

        if (result == NULL)
        {
                return;
        }
        rc  = (*env)->GetObjectClass(env, result);
        rid = (*env)->GetFieldID(env, rc, "readCount", "J");
        wid = (*env)->GetFieldID(env, rc, "writeCount", "J");
        (*env)->SetLongField(env, result, rid, res.readcount);
        (*env)->SetLongField(env, result, wid, res.writecount);

}


JNIEXPORT jlong JNICALL Java_org_ifarchive_glk_Glk_jniglk_1stream_1get_1rock
  (JNIEnv *env, jclass c, jint strid)
{
	return glk_stream_get_rock(STRID_T(strid));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1stream_1set_1position
  (JNIEnv *env, jclass c, jint strid, jint pos, jint seekmode)
{
	glk_stream_set_position(STRID_T(strid), pos, seekmode);
}

JNIEXPORT jlong JNICALL Java_org_ifarchive_glk_Glk_jniglk_1stream_1get_1position
  (JNIEnv *env, jclass c, jint strid)
{
	return glk_stream_get_position(STRID_T(strid));
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1set_1stream
  (JNIEnv *env, jclass c, jint stream)
{
	glk_stream_set_current(STRID_T(stream));
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1get_1stream
  (JNIEnv *env, jclass c)
{
	return STRMAP(glk_stream_get_current());
}





JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1put_1char
  (JNIEnv *env, jclass c, jbyte ch)
{
	glk_put_char((unsigned char)ch);
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1put_1buffer
  (JNIEnv *env, jclass c, jarray arr)
{
	glui32 len = (*env)->GetArrayLength(env, arr);
	jbyte *buf = (*env)->GetByteArrayElements(env, arr, NULL);
	glk_put_buffer((unsigned char *)buf, len);
	(*env)->ReleaseByteArrayElements(env, arr, buf, 0);	
}



JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1set_1style
  (JNIEnv *env, jclass c, jint i)
{
	glk_set_style((glui32)i);
}


JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1get_1char_1stream
  (JNIEnv *env, jclass c, jint strid)
{
	glui32 i = glk_get_char_stream(STRID_T(strid));
	if (i == 0xFFFFFFFFL) return -1;
	return (jint)i;
}


JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1get_1line_1stream
  (JNIEnv *env, jclass c, jint str, jbyteArray arr)
{
	jint i;

	glui32 len = (*env)->GetArrayLength(env, arr);
	jbyte *buf = (*env)->GetByteArrayElements(env, arr, NULL);
	i = glk_get_line_stream(STRID_T(str), buf, len);
	(*env)->ReleaseByteArrayElements(env, arr, buf, 0);	
	return i;
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1get_1buffer_1stream
  (JNIEnv *env, jclass c, jint str, jbyteArray arr)
{
	jint i;

	glui32 len = (*env)->GetArrayLength(env, arr);
	jbyte *buf = (*env)->GetByteArrayElements(env, arr, NULL);
	i = glk_get_buffer_stream(STRID_T(str), buf, len);
	(*env)->ReleaseByteArrayElements(env, arr, buf, 0);	
	return i;
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1put_1char_1stream
  (JNIEnv *env, jclass c, jint str, jbyte ch)
{
	glk_put_char_stream(STRID_T(str), (unsigned char)ch);
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1put_1buffer_1stream
  (JNIEnv *env, jclass c, jint str, jarray arr)
{
	glui32 len = (*env)->GetArrayLength(env, arr);
	jbyte *buf = (*env)->GetByteArrayElements(env, arr, NULL);
	glk_put_buffer_stream(STRID_T(str), (unsigned char *)buf, len);
	(*env)->ReleaseByteArrayElements(env, arr, buf, 0);	
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1set_1style_1stream
  (JNIEnv *env, jclass c, jint str, jint i)
{
	glk_set_style_stream(STRID_T(str), (glui32)i);
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1stylehint_1set
  (JNIEnv *env, jclass c, jint wintype, jint styl, jint hint, jint val)
{
	glk_stylehint_set(wintype, styl, hint, val);
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1stylehint_1clear
  (JNIEnv *env, jclass c, jint wintype, jint styl, jint hint)
{
	glk_stylehint_clear(wintype, styl, hint);
}

JNIEXPORT jboolean JNICALL Java_org_ifarchive_glk_Glk_jniglk_1style_1distinguish
  (JNIEnv *env, jclass c, jint winid, jint styl1, jint styl2)
{
	return glk_style_distinguish(WINID_T(winid), styl1, styl2) ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT jboolean JNICALL Java_org_ifarchive_glk_Glk_jniglk_1style_1measure
  (JNIEnv *env, jclass c, jint winid, jint styl, jint hint, jintArray result)
{
	glui32 res;
	jint ires;
	jboolean rv;
	
	rv = glk_style_measure(WINID_T(winid), styl, hint, &res);
	if (result)
	{
		ires = res;
		(*env)->SetIntArrayRegion(env, result, 0, 1, &ires);
	}
	return rv;
}



JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1fileref_1create_1temp
  (JNIEnv *env, jclass c, jint usage, jint rock)
{
	return FREFMAP(glk_fileref_create_temp(usage,rock));	
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1fileref_1create_1by_1name
  (JNIEnv *env, jclass c, jint usage, jstring name, jint rock)
{
	const char *utfname = (*env)->GetStringUTFChars(env, name, NULL);
	jlong id = FREFMAP(glk_fileref_create_by_name(usage, (char *)utfname, rock));
	(*env)->ReleaseStringUTFChars(env, name, utfname);
	return id;
}



JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1fileref_1create_1by_1prompt
  (JNIEnv *env, jclass c, jint usage, jint fmode, jint rock)
{
	return FREFMAP(glk_fileref_create_by_prompt(usage, fmode, rock));
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1fileref_1create_1from_1fileref
  (JNIEnv *env, jclass c, jint i, jlong fref, jint rock)
{
	return FREFMAP(glk_fileref_create_from_fileref(i, FREFID_T(fref), rock));
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1fileref_1destroy
  (JNIEnv *env, jclass c, jlong fref)
{
	glk_fileref_destroy(FREFID_T(fref));
}


JNIEXPORT jlong JNICALL Java_org_ifarchive_glk_Glk_jniglk_1fileref_1get_1rock
  (JNIEnv *env, jclass c, jlong fref)
{
	return glk_fileref_get_rock(FREFID_T(fref));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1fileref_1delete_1file
  (JNIEnv *env, jclass c, jlong fref)
{
	glk_fileref_delete_file(FREFID_T(fref));
}

JNIEXPORT jboolean JNICALL Java_org_ifarchive_glk_Glk_jniglk_1fileref_1does_1file_1exist
  (JNIEnv *env, jclass c, jlong fref)
{
	return glk_fileref_does_file_exist(FREFID_T(fref)) ? JNI_TRUE : JNI_FALSE;
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1select
  (JNIEnv *env, jclass c, jobject event, jboolean poll)
{
        jclass   ec;
        jfieldID etype, ewin, eval1, eval2;

	event_t ev;

	if (poll) glk_select_poll(&ev); 
	else	  glk_select(&ev);

	if (event == NULL) return;
	ec    = (*env)->GetObjectClass(env, event);
	etype = (*env)->GetFieldID(env, ec, "type",  "I");
	ewin  = (*env)->GetFieldID(env, ec, "winid", "I");
	eval1 = (*env)->GetFieldID(env, ec, "val1",  "J");
	eval2 = (*env)->GetFieldID(env, ec, "val2",  "J");	
	(*env)->SetIntField (env, event, etype, ev.type);
	(*env)->SetIntField (env, event, ewin,  WINMAP(ev.win));
	(*env)->SetLongField(env, event, eval1, ev.val1);
	(*env)->SetLongField(env, event, eval2, ev.val2);
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1request_1timer_1events
  (JNIEnv *env, jclass c, jlong millis)
{
	glk_request_timer_events(GLUI32(millis));
}



JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1request_1line_1event
  (JNIEnv *env, jclass c, jint winid, jint bufid, jint buflen, jint initLen)
{
	glk_request_line_event(WINID_T(winid), JBYTE_PTR(bufid), buflen - 1, initLen);
}





JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1request_1char_1event
  (JNIEnv *env, jclass c, jint winid)
{
	glk_request_char_event(WINID_T(winid));
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1request_1mouse_1event
  (JNIEnv *env, jclass c, jint winid)
{
	glk_request_mouse_event(WINID_T(winid));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1cancel_1line_1event
  (JNIEnv *env, jclass c, jint winid, jobject event)
{
	event_t e, *pe;
        jclass   ec;
        jfieldID etype, ewin, eval1, eval2;

	if (event) pe = &e; else pe = NULL;
	glk_cancel_line_event(WINID_T(winid), pe);
	if (event == NULL) return;
	ec    = (*env)->GetObjectClass(env, event);
	etype = (*env)->GetFieldID(env, ec, "type",  "I");
	ewin  = (*env)->GetFieldID(env, ec, "winid", "I");
	eval1 = (*env)->GetFieldID(env, ec, "val1",  "J");
	eval2 = (*env)->GetFieldID(env, ec, "val2",  "J");	
	(*env)->SetIntField (env, event, etype, e.type);
	(*env)->SetIntField (env, event, ewin,  WINMAP(e.win));
	(*env)->SetLongField(env, event, eval1, e.val1);
	(*env)->SetLongField(env, event, eval2, e.val2);
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1cancel_1char_1event
  (JNIEnv *env, jclass c, jint winid)
{
	glk_cancel_char_event(WINID_T(winid));
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1cancel_1mouse_1event
  (JNIEnv *env, jclass c, jint winid)
{
	glk_cancel_mouse_event(WINID_T(winid));
}


JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jnigidispatch_1count_1classes
  (JNIEnv *env, jclass c)
{
	return gidispatch_count_classes();
}


/* The opaque object registry. We store the mapping between pointers and 
 * dispatch IDs at the C level; however, Java still needs to be called when
 * objects are registered or unregistered so that it can create or destroy
 * its wrapper objects. */
static JNIEnv *regEnv;

/*******************************************************************
 * The following functions are used to manipulate variable-size 
 * arrays of mappings (the type jniglk_vector is used to represent
 * these). There will be one vector for each class of opaque object,
 * and one for retained arrays. 
 *******************************************************************/
 
typedef struct 
{
	void *obj;
	glui32 dispid;
} jniglk_mapping;

typedef struct
{
	int count;
	int max;
	jniglk_mapping *mapping;
} jniglk_vector;


static void mem_err()
{
	diewith("Out of memory");
}

/* If a vector has not been initialised, initialise it, allocating space
 * for 16 entries. */
static void vector_check(jniglk_vector *v)
{
	if (v->max == 0)
	{
		v->mapping = malloc(16 * sizeof(jniglk_mapping));
		if (v->mapping == NULL) mem_err();
		v->max = 16;
	}
}

/* Add an entry to a vector. If the vector is full, double its size. */

static void vector_add(jniglk_vector *v, void *obj, glui32 dispid)
{
	int n;
	if (v->count == v->max)
	{
		jniglk_mapping *m2 = malloc(v->max * 2 * sizeof(jniglk_mapping));
		if (!m2) mem_err();
		for (n = 0; n < v->count; n++) m2[n] = v->mapping[n];
		v->max *= 2;
		free(v->mapping);
		v->mapping = m2;
	}
	v->mapping[v->count].obj    = obj;
	v->mapping[v->count].dispid = dispid;
	++v->count;
}

/* Delete an entry from a vector. */

static void vector_del(jniglk_vector *v, void *obj, glui32 dispid)
{
	int n, m;
	for (n = 0; n < v->count; n++)
	{
		if (v->mapping[n].dispid == dispid &&
		    v->mapping[n].obj    == obj) break;
	}
	if (n >= v->count) return;	/* Object was never registered */

	/* Move all the objects above (if present) down one. */
	--(v->count);	
	for (m = n; m < v->count; m++)
	{
		v->mapping[m] = v->mapping[m+1];
	}
}



/* The opaque object registry */
static jniglk_vector *st_reg = NULL;
/* The retained array registry */
static jniglk_vector st_arrays = { 0, 0, 0 };
static int st_maxclass = 0;
static glui32 st_bufid    = 1;


/* Create a C registry to store a mapping between dispatch IDs and
 * C pointers. Each entry holds mappings for one Glk class. */
static void check_alloc_reg()
{
	int n;

	if (!st_maxclass) st_maxclass = gidispatch_count_classes();

	if (st_reg == NULL)
	{
		st_reg = malloc(st_maxclass * sizeof(jniglk_vector));
		if (st_reg == NULL) mem_err();
		for (n = 0; n < st_maxclass; n++) 
		{
			memset(&st_reg[n], 0, sizeof(jniglk_vector)); 
		}
	}
	for (n = 0; n < st_maxclass; n++) vector_check(&st_reg[n]);
}


/* Deallocate the object<-->dispid mapping */
static void dealloc_reg()
{
	int n;

	if (!st_maxclass) st_maxclass = gidispatch_count_classes();

	if (st_reg == NULL) return;
	for (n = 0; n < st_maxclass; n++)
	{
		if (st_reg[n].mapping) free(st_reg[n].mapping);
	}
	free(st_reg);
	st_reg = NULL;
}

/* Debugging code: Dump out the persistent object registry.
static void dump_reg()
{
	int n, m;	
	for (n = 0; n < st_maxclass; n++)
	{
		fprintf(stderr, "Class %d objects:\n", n);
		for (m = 0; m < st_reg[n].count; m++)
		{
			fprintf(stderr, "dispid=%3ld objid=%x\n",
				st_reg[n].mapping[m].dispid,
				(int)st_reg[n].mapping[m].obj);
		}
	}
}
*/


/* Register an object<-->dispid mapping */
static void c_reg(void *obj, glui32 objclass, glui32 dispid)
{
	if (!dispid) return;	/* null reference */
	check_alloc_reg();
	if (objclass >= st_maxclass)
	{
		fprintf(stderr, "jniglk: Object class %ld out of range\n", objclass);
		return;
	}
	vector_add(&st_reg[objclass], obj, dispid);
	
	/* dump_reg(); debugging */
}



/* Unregister an object<-->dispid mapping */
static void c_unreg(void *obj, glui32 objclass, glui32 dispid)
{
	if (!dispid) return;	/* null reference */
	check_alloc_reg();
	if (objclass >= st_maxclass)
	{
		fprintf(stderr, "jniglk: Object class %ld out of range\n", objclass);
		return;
	}
	vector_del(&st_reg[objclass], obj, dispid);
	/* dump_reg(); debugging */
}

/* Given the object's dispid, find its Glk pointer */
static void *dispid_to_object(char *func, glui32 objclass, glui32 dispid)
{
	jniglk_vector *v;
	int n;

	if (dispid == 0) return NULL;

	check_alloc_reg();
	if (objclass >= st_maxclass)
	{
		fprintf(stderr, "jniglk: Object class %ld out of range in %s\n", objclass, func);
		return NULL;
	}
	v = &st_reg[objclass];
	for (n = 0; n < v->count; n++)
	{
		if (v->mapping[n].dispid == dispid) return v->mapping[n].obj;
	}
	diewith("Could not map object %ld to Glk object in %s\n", dispid, func);
	return NULL;
}


/* We could just use the registry, but... why not do it the easy way? */
static glui32 object_to_dispid(char *func, glui32 glkClass, void *object)
{
	if (object == NULL) return 0;
	return gidispatch_get_objrock(object, glkClass).num;
}


static void *dispid_to_array(char *func, glui32 dispid)
{
	int n;
	
	vector_check(&st_arrays);
	for (n = 0; n < st_arrays.count; n++)
	{
		if (st_arrays.mapping[n].dispid == dispid)
			return st_arrays.mapping[n].obj;
	}
	/* Should never happen */
	return NULL;		
}



static glui32 array_to_dispid(char *func, void *array)
{
	int n;
	
	vector_check(&st_arrays);
	for (n = 0; n < st_arrays.count; n++)
	{
		if (st_arrays.mapping[n].obj == array)
			return st_arrays.mapping[n].dispid;
	}
	/* Should never happen */
	return 0;
}



/* Register the appearance of a new object, and get its dispatch ID */
static gidispatch_rock_t jnireg(void *obj, glui32 objclass)
{
	jclass clazz;
	jmethodID method;
	gidispatch_rock_t rc;	

	rc.num = 0;
	clazz = (*regEnv)->FindClass(regEnv, PACKAGE "Glk");
	if (clazz == NULL)
	{
		diewith("reg() can't find Glk class\n");
	}
	method = (*regEnv)->GetStaticMethodID(regEnv, clazz, "jniglk_reg", "(I)I");
	if (method == NULL)
	{
		diewith("reg() can't find Glk.jniglk_reg method\n");
	}
	rc.num = (*regEnv)->CallStaticIntMethod(regEnv, clazz, method, objclass);
	
	/* Now we want to record this mapping at the C level. */
	c_reg(obj, objclass, rc.num);
	return rc;
}

/* Register the disappearance of an object. */
static void jniunreg(void *obj, glui32 objclass, gidispatch_rock_t rc)
{
	jclass clazz;
	jmethodID method;

	clazz = (*regEnv)->FindClass(regEnv, PACKAGE "Glk");
	if (clazz == NULL)
	{
		diewith("unreg() can't find Glk class\n");
	}
	method = (*regEnv)->GetStaticMethodID(regEnv, clazz, "jniglk_unreg", "(II)V");
	if (method == NULL)
	{
		diewith("unreg() can't find Glk.jniglk_unreg method\n");
	}
	(*regEnv)->CallStaticVoidMethod(regEnv, clazz, method, objclass, rc.num);
	/* Record this mapping at the C level. */
	c_unreg(obj, objclass, rc.num);
}


/* Glk retains an array */
static gidispatch_rock_t arrayreg(void *array, glui32 len, char *typecode)
{
	jclass clazz;
	jmethodID method;
	jstring typestring;
	gidispatch_rock_t rc;	
	int dispid = array_to_dispid("arrayreg", array);

	rc.num = 0;
	clazz = (*regEnv)->FindClass(regEnv, PACKAGE "Glk");
	if (clazz == NULL)
	{
		diewith("reg() can't find Glk class\n");
	}
	method = (*regEnv)->GetStaticMethodID(regEnv, clazz, "jniglk_array_reg", "(IILjava/lang/String;)V");
	if (method == NULL)
	{
		diewith("reg() can't find Glk.jniglk_array_reg method\n");
	}
	typestring = (*regEnv)->NewStringUTF(regEnv, typecode);
	(*regEnv)->CallStaticVoidMethod(regEnv, clazz, method, dispid, len, typestring);
	rc.num = dispid;
	
	return rc;
}

/* Glk unretains an object. */
static void arrayunreg(void *array, glui32 len, char *typecode,
				gidispatch_rock_t objrock)
{
	jclass clazz;
	jmethodID method;
	jstring typestring;

	clazz = (*regEnv)->FindClass(regEnv, PACKAGE "Glk");
	if (clazz == NULL)
	{
		diewith("unreg() can't find Glk class\n");
	}
	method = (*regEnv)->GetStaticMethodID(regEnv, clazz, "jniglk_array_unreg", "(ILjava/lang/String;)V");
	if (method == NULL)
	{
		diewith("unreg() can't find Glk.jniglk_array_unreg method\n");
	}
	typestring = (*regEnv)->NewStringUTF(regEnv, typecode);
	(*regEnv)->CallStaticVoidMethod(regEnv, clazz, method, objrock.num, typestring);
}







JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1register_1registry
  (JNIEnv *env, jclass c)
{
	regEnv   = env;
	gidispatch_set_object_registry(jnireg, jniunreg);
	gidispatch_set_retained_registry(arrayreg, arrayunreg);
}



JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1lock_1byte_1array
  (JNIEnv *env, jclass c, jbyteArray arr)
{
	int id;
	void *buf = (*env)->GetByteArrayElements(env, arr, NULL);
	if (!buf) return 0;
	id = st_bufid++;
	if (!st_bufid) ++st_bufid;
	vector_check(&st_arrays);
	vector_add(&st_arrays, buf, id);
	return id; 
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1unlock_1byte_1array
  (JNIEnv *env, jclass c, jbyteArray arr, jint bufid)
{
	jbyte *buf = JBYTE_PTR(bufid);

	(*env)->ReleaseByteArrayElements(env, arr, buf, 0);	
	vector_del(&st_arrays, buf, bufid);
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1lock_1short_1array
  (JNIEnv *env, jclass c, jshortArray arr)
{
	int id;
	void *buf = (*env)->GetShortArrayElements(env, arr, NULL);
	if (!buf) return 0;
	id = st_bufid++;
	if (!st_bufid) ++st_bufid;
	vector_check(&st_arrays);
	vector_add(&st_arrays, buf, id);
	return id; 
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1unlock_1short_1array
  (JNIEnv *env, jclass c, jshortArray arr, jint bufid)
{
	jshort *buf = JSHORT_PTR(bufid);

	(*env)->ReleaseShortArrayElements(env, arr, buf, 0);	
	vector_del(&st_arrays, buf, bufid);
}



JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1lock_1int_1array
  (JNIEnv *env, jclass c, jintArray arr)
{
	int id;
	void *buf = (*env)->GetIntArrayElements(env, arr, NULL);
	if (!buf) return 0;
	id = st_bufid++;
	if (!st_bufid) ++st_bufid;
	vector_check(&st_arrays);
	vector_add(&st_arrays, buf, id);
	return id; 
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1unlock_1int_1array
  (JNIEnv *env, jclass c, jintArray arr, jint bufid)
{
	jint *buf = JINT_PTR(bufid);

	(*env)->ReleaseIntArrayElements(env, arr, buf, 0);	
	vector_del(&st_arrays, buf, bufid);
}


JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1lock_1long_1array
  (JNIEnv *env, jclass c, jlongArray arr)
{
	int id;
	void *buf = (*env)->GetLongArrayElements(env, arr, NULL);
	if (!buf) return 0;
	id = st_bufid++;
	if (!st_bufid) ++st_bufid;
	vector_check(&st_arrays);
	vector_add(&st_arrays, buf, id);
	return id; 
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1unlock_1long_1array
  (JNIEnv *env, jclass c, jlongArray arr, jint bufid)
{
	jlong *buf = JLONG_PTR(bufid);

	(*env)->ReleaseLongArrayElements(env, arr, buf, 0);	
	vector_del(&st_arrays, buf, bufid);
}


JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1lock_1char_1array
  (JNIEnv *env, jclass c, jcharArray arr)
{
	int id;
	void *buf = (*env)->GetCharArrayElements(env, arr, NULL);
	if (!buf) return 0;
	id = st_bufid++;
	if (!st_bufid) ++st_bufid;
	vector_check(&st_arrays);
	vector_add(&st_arrays, buf, id);
	return id; 
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1unlock_1char_1array
  (JNIEnv *env, jclass c, jcharArray arr, jint bufid)
{
	jchar *buf = JCHAR_PTR(bufid);

	(*env)->ReleaseCharArrayElements(env, arr, buf, 0);	
	vector_del(&st_arrays, buf, bufid);
}


static void jniglk_exit(void)
{
	dealloc_reg();
	if (st_arrays.mapping) 
	{
		free(st_arrays.mapping); 
		st_arrays.mapping = NULL; 
	}
}




/* Make a dispatch call. Java has created an array of UniversalUnions, 
   which we shall now proceed to convert to/from gluniversal_ts. */
JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jnigidispatch_1call
  (JNIEnv *env, jclass cla, jint funcnum, jobjectArray arr)
{
	int n, len;
	gluniversal_t *params;
	jobject par;
	jfieldID ids[11];
	jclass clazz;
	int convert;
	jint i = 0;
	jint cl = 0;
	jlong l = 0;
	jchar c = 0;

	clazz = (*env)->FindClass(env, PACKAGE "UniversalUnion");
	if (clazz == NULL) diewith("Can't find UniversalUnion class");	
	ids[ 0] = (*env)->GetFieldID(env, clazz, "m_type",      "I");
	ids[ 1] = (*env)->GetFieldID(env, clazz, "m_uint",      "J");
	ids[ 2] = (*env)->GetFieldID(env, clazz, "m_sint",      "I");
	ids[ 3] = (*env)->GetFieldID(env, clazz, "m_dispid",    "I");
	ids[ 4] = (*env)->GetFieldID(env, clazz, "m_uch",       "C");
	ids[ 5] = (*env)->GetFieldID(env, clazz, "m_sch",       "C");
	ids[ 6] = (*env)->GetFieldID(env, clazz, "m_ch",        "C");
	ids[ 7] = (*env)->GetFieldID(env, clazz, "m_charstrid", "I");
	ids[ 8] = (*env)->GetFieldID(env, clazz, "m_bufid",     "I");
	ids[ 9] = (*env)->GetFieldID(env, clazz, "m_ptrflag",   "I");
	ids[10] = (*env)->GetFieldID(env, clazz, "m_glkClass", "I");

	if (!arr) len = 0;
 	else      len = (*env)->GetArrayLength(env, arr);	
	if (len) params = malloc(len * sizeof(gluniversal_t));
	else     params = NULL;
	if (len && !params) mem_err();

	for (n = 0; n < len; n++)
	{
		par = (*env)->GetObjectArrayElement(env, arr, n);	
		if (!par) convert = 0;
		else 	  convert = (*env)->GetIntField(env, par, ids[0]);
		
		switch(convert)
		{
			case 1: l = (*env)->GetLongField(env, par, ids[convert]);
				break;
			case 3: cl = (*env)->GetIntField(env, par, ids[10]);
				/* and fall through... */
			case 2:
			case 7:
			case 8:
			case 9: i = (*env)->GetIntField(env, par, ids[convert]);
				break;
			case 4:
			case 5:
			case 6: c = (*env)->GetCharField(env, par, ids[convert]);	
				break;
		}
		switch(convert)
		{
			case 0: params[n].uint = 0; break; 
			case 1: params[n].uint = GLUI32(l); break; 
			case 2: params[n].sint = i; break; 
			case 3: params[n].opaqueref = dispid_to_object(__FUNCTION__,cl, i); break; 
			case 4: params[n].uch = (c & 0xFF); break; 
			case 5: params[n].sch = (c & 0xFF); break; 
			case 6: params[n].ch  = (c & 0xFF); break; 
			case 7: params[n].charstr = dispid_to_array(__FUNCTION__,i); break;
			case 8: params[n].array   = dispid_to_array(__FUNCTION__,i); break;
			case 9: params[n].ptrflag = i; break;
		}
	}
	/* The "params" array has now been populated. Call Glk. */
	gidispatch_call(funcnum, len, params);
	/* Now copy back out from "params" back to the array. */
	for (n = 0; n < len; n++)
	{
		par = (*env)->GetObjectArrayElement(env, arr, n);	
		if (!par) convert = 0;
		else 	  convert = (*env)->GetIntField(env, par, ids[0]);
		
		switch(convert)
		{
			case 0: i = 0; break;
			case 1: l = params[n].uint; break;
			case 2: i = params[n].sint; break; 
			case 3: i = object_to_dispid(__FUNCTION__, cl, params[n].opaqueref);
			case 4: c = params[n].uch; break; 
			case 5: c = params[n].sch; break; 
			case 6: c = params[n].ch; break; 
			case 7: break; /* Strings are always read-only. */
			case 8: i = array_to_dispid(__FUNCTION__,params[n].array); break;
			case 9: i = params[n].ptrflag; break;
		}
		switch(convert)
		{
			case 1: (*env)->SetLongField(env, par, ids[convert], l);
				break;
			case 2:
			case 3:
			case 7:
			case 8:
			case 9: (*env)->SetIntField(env, par, ids[convert], i);
				break;
			case 4:
			case 5:
			case 6: (*env)->SetCharField(env, par, ids[convert], c);	
				break;
		}
	}
	if (params) free(params);
}

JNIEXPORT jstring JNICALL Java_org_ifarchive_glk_Glk_jnigidispatch_1prototype
  (JNIEnv *env, jclass c, jint func)
{
	char *s = gidispatch_prototype(func);
	
	if (!s) return NULL;
	return (*env)->NewStringUTF(env, s);
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jnigidispatch_1count_1intconst
  (JNIEnv *env, jclass c)
{
	return gidispatch_count_intconst();
}

JNIEXPORT jobject JNICALL Java_org_ifarchive_glk_Glk_jnigidispatch_1get_1intconst
  (JNIEnv *env, jclass c, jint index)
{
	jobject obj;
	jstring str;
	jlong v;
	jclass clazz;
	jmethodID methodID;

	gidispatch_intconst_t *ic = gidispatch_get_intconst(index);
	if (!ic) return NULL;

	str = (*env)->NewStringUTF(env, ic->name);
	v   = ic->val;

	clazz    = (*env)->FindClass(env, PACKAGE "IntConstant");
	if (!clazz) diewith("Can't find IntConstant class");
	methodID = (*env)->GetMethodID(env, clazz, "<init>", "(Ljava/lang/String;J)V"); 	
	if (!clazz) diewith("Can't find IntConstant constructor");
	obj = (*env)->NewObject(env, clazz, methodID, str, v);
	return obj;
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jnigidispatch_1count_1functions
  (JNIEnv *env, jclass c)
{
	return gidispatch_count_functions();
}

JNIEXPORT jobject JNICALL Java_org_ifarchive_glk_Glk_jnigidispatch_1get_1function
  (JNIEnv *env, jclass c, jint index)
{
	jobject obj;
	jstring str;
	jint v;
	jclass clazz;
	jmethodID methodID;

	gidispatch_function_t *fn = gidispatch_get_function(index);
	if (!fn) return NULL;

	str = (*env)->NewStringUTF(env, fn->name);
	v   = fn->id;

	clazz    = (*env)->FindClass(env, PACKAGE "Function");
	if (!clazz) diewith("Can't find Function class");
	methodID = (*env)->GetMethodID(env, clazz, "<init>", "(Ljava/lang/String;I)V"); 	
	if (!clazz) diewith("Can't find Function constructor");
	obj = (*env)->NewObject(env, clazz, methodID, str, v);
	return obj;
}

JNIEXPORT jobject JNICALL Java_org_ifarchive_glk_Glk_jnigidispatch_1get_1function_1by_1id
  (JNIEnv *env, jclass c, jint index)
{
	jobject obj;
	jstring str;
	jint v;
	jclass clazz;
	jmethodID methodID;

	gidispatch_function_t *fn = gidispatch_get_function_by_id(index);
	if (!fn) return NULL;

	str = (*env)->NewStringUTF(env, fn->name);
	v   = fn->id;

	clazz    = (*env)->FindClass(env, PACKAGE "Function");
	if (!clazz) diewith("Can't find Function class");
	methodID = (*env)->GetMethodID(env, clazz, "<init>", "(Ljava/lang/String;I)V"); 	
	if (!clazz) diewith("Can't find Function constructor");
	obj = (*env)->NewObject(env, clazz, methodID, str, v);
	return obj;
}

/*
 * Image functions
 */
#ifdef GLK_MODULE_IMAGE
JNIEXPORT jboolean JNICALL Java_org_ifarchive_glk_Glk_jniglk_1image_1draw
  (JNIEnv *env, jclass c, jint win, jint image, jint val1, jint val2)
{
	return glk_image_draw(WINID_T(win), image, val1, val2) ? JNI_TRUE : JNI_FALSE; 
}

JNIEXPORT jboolean JNICALL Java_org_ifarchive_glk_Glk_jniglk_1image_1draw_1scaled
  (JNIEnv *env, jclass c, jint win, jint image, jint val1, jint val2, jlong w, jlong h)
{
	return glk_image_draw_scaled(WINID_T(win), image, val1, val2, GLUI32(w), GLUI32(h)) ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT jboolean JNICALL Java_org_ifarchive_glk_Glk_jniglk_1image_1get_1info
  (JNIEnv *env, jclass c, jint image, jintArray result)
{
	glui32 w,h;
	jint iw, ih;
	jboolean res;

	res = glk_image_get_info(image, &w, &h) ? JNI_TRUE : JNI_FALSE;
	if (!result) return res;

	iw = w; ih = h;
	(*env)->SetIntArrayRegion(env, result, 0, 1, &iw);
	(*env)->SetIntArrayRegion(env, result, 1, 1, &ih);
	return res;
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1flow_1break
  (JNIEnv *env, jclass c, jint win)
{
	glk_window_flow_break(WINID_T(win));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1erase_1rect
  (JNIEnv *env, jclass c, jint win, jint x, jint y, jlong w, jlong h)
{
	glk_window_erase_rect(WINID_T(win), x, y, GLUI32(w), GLUI32(h));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1fill_1rect
  (JNIEnv *env, jclass c, jint win, jint colour, jint x, jint y, jlong w, jlong h)
{
	glk_window_fill_rect(WINID_T(win), colour, x, y, GLUI32(w), GLUI32(h));
}


JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1window_1set_1background_1color
  (JNIEnv *env, jclass c, jint window, jint colour)
{
	glk_window_set_background_color(WINID_T(window), colour);
}
#endif /* def GLK_MODULE_IMAGE */
#ifdef GLK_MODULE_SOUND
JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1schannel_1create
  (JNIEnv *env, jclass c, jint rock)
{
	return SCHANNELMAP(glk_schannel_create(rock));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1schannel_1destroy
  (JNIEnv *env, jclass c, jint chan)
{
	glk_schannel_destroy(SCHANNELID_T(chan));
}

JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jniglk_1schannel_1get_1rock
  (JNIEnv *env, jclass c, jint chan)
{
	return glk_schannel_get_rock(SCHANNELID_T(chan));
}

JNIEXPORT jboolean JNICALL Java_org_ifarchive_glk_Glk_jniglk_1schannel_1play
  (JNIEnv *env, jclass c, jint chan, jint snd)
{
	return glk_schannel_play(SCHANNELID_T(chan), snd) ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT jboolean JNICALL Java_org_ifarchive_glk_Glk_jniglk_1schannel_1play_1ext
  (JNIEnv *env, jclass c, jint chan, jint snd, jint repeats, jint notify)
{
	return glk_schannel_play_ext(SCHANNELID_T(chan), snd, repeats, notify) ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1schannel_1stop
  (JNIEnv *env, jclass c, jint chan)
{
	glk_schannel_stop(SCHANNELID_T(chan));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1schannel_1set_1volume
  (JNIEnv *env, jclass c, jint chan, jint vol)
{
	glk_schannel_set_volume(SCHANNELID_T(chan), vol);
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1sound_1load_1hint
  (JNIEnv *env, jclass c, jint snd, jboolean hint)
{
	glk_sound_load_hint(snd, hint);
}
#endif /* def GLK_MODULE_SOUND */
#ifdef GLK_MODULE_HYPERLINKS

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1set_1hyperlink
  (JNIEnv *env, jclass c, jint linkval)
{
	glk_set_hyperlink(linkval);
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1set_1hyperlink_1stream
  (JNIEnv *env, jclass c, jint stream, jint linkval)
{
	glk_set_hyperlink_stream(STRID_T(stream), linkval);
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1request_1hyperlink_1event
  (JNIEnv *env, jclass c, jint win)
{
	glk_request_hyperlink_event(WINID_T(win));
}

JNIEXPORT void JNICALL Java_org_ifarchive_glk_Glk_jniglk_1cancel_1hyperlink_1event
  (JNIEnv *env, jclass c, jint win)
{
	glk_cancel_hyperlink_event(WINID_T(win));
}
#endif /* def GLK_MODULE_HYPERLINKS */

/* Blorb */
JNIEXPORT jint JNICALL Java_org_ifarchive_glk_Glk_jnigiblorb_1set_1resource_1map
  (JNIEnv *env, jclass c, jint str)
{
	return giblorb_set_resource_map(STRID_T(str));
}


