!----------------------------------------------------------------------------
! converse.h - conversation management in Hugo
! Version 1.0, Copyright 2001 by Christopher Tate <ctate@acm.org>
!
! Also needs converse.g for grammar.
!
! Relies on HUGOLIB and VERBLIB -- #include this file after
! #including "hugolib.h" and "verblib.h"
!----------------------------------------------------------------------------

#ifclear _CONVERSE_H
#set _CONVERSE_H

!----------------------------------------------------------------------------
! !!! Things not implemented; may or may not be good ideas:
!
!	- Non-hierarchical topic backtracking [go back to earlier topics automatically even without tree structure]
!	- detection of 'abrupt' vs 'natural' topic changes, and NPC reaction based on it
!	- allow NPCs to veto topic changes - scope ceiling a la Emily, or some other mechanism?
!	- somehow hook Assertions into mutiple topics w/out duplicating?  a la Emily's 'pars' system.
!	- 'mood alteration' keying; a property in the Assertion that indicates how the NPC's mood should change in response to this assertion being run
!----------------------------------------------------------------------------

!----------------------------------------------------------------------------
! Entering conversation mode is done with >GREET character, or with
! >TALK TO character.  Changing topics is done with >TOPIC or >T
!----------------------------------------------------------------------------

! global variables used by the conversation mechanism:

array gAvailableAssertions[5]			! at any moment, the available conversational options
global gNumAvailableAssertions		! how many elements are 'valid' in gAvailableAssertions[]
global gDidSpeak								! did the player engage in conversation last turn?

!----------------------------------------------------------------------------
! an NPC is someone we can talk to.  NPCs use some custom properties:

property next_assertion alias d_to			! go to this Assertion automatically next turn if the player doesn't pick one explicitly
property force_assertion alias cant_go	! go to this Assertion automatically as soon as the player gives us a chance; makes SURE we get here
property last_assertion alias in_to			! what did we say last?
property current_topic alias out_to			! what's the NPC currently talking about?
property greeted alias e_to						! called when SpeakTo() has determined that conversation is actually ensuing

class NPC
{
	type npc
	next_assertion 0
	force_assertion 0
	last_assertion 0
	current_topic 0

	! by default, when you greet an NPC there is no set topic, and any stale cued assertions are flushed
	greeted
	{
		"You contemplate introductory remarks."
		gNumAvailableAssertions = 0
		self.current_topic = 0
		self.next_assertion = 0
		self.force_assertion = 0
	}

	! add "is female" and make pronouns "she", "her", "her", "herself" for female NPCs
	pronouns "he", "him", "his", "himself"
	is living, static
}

!----------------------------------------------------------------------------
! The global 'Conversation' object serves as the root of the topic tree.
! *ONLY* Topic objects should be placed 'in' the Conversation.

object Conversation
{
	type conversation
	is transparent, hidden, static
	in_scope 0				! will be placed in the player's scope at init time
}

!----------------------------------------------------------------------------
! a 'Topic' is a thing that can be discussed; the current subject of conversation
! is set via >TOPIC FOO or >ASK character ABOUT FOO.  Note that you must
! manually attach derived Topic objects in the proper place, either to the Conversation
! object directly, or to another Topic object (useful for providing scoping of topics).

class Topic
{
	type topic
	is hidden, container, openable, not open
	is transparent									! so you can 'see' subtopics
	is known											! so you can ask about it
	exclude_from_all true

	parse_rank
	{
		! prefer Topic objects over similarly-named 'tangible' objects when parsing >TOPIC obj
		if verbroutine = &DoTopic : return 1
		else : return -1		! 0 is the default parse_rank; we defer to all non-topic objects like this
	}
}

!----------------------------------------------------------------------------
! An "Assertion" is a conversational exchange, and encapsulates both a question
! or statement that the player can pose to an NPC and the NPCs response.
!
! long_desc, which displays the NPCs response to a given conversational gambit, should
! set up any topic chains etc. in the NPC when necessary; it is called with the global variable
! 'speaking' set to the current conversation target to facilitate this.

! attribute visited								! indicates that the Assertion has been said; generally this will be set by the engine
! property short_desc						! menu item text for this Assertion; should NOT end with a newline
! property long_desc							! full text to be displayed when the Assertion is selected by the player.
property Prerequisite alias out_to	! Usually a function property; returns 0 (false) if the Assertion is currently unavailable for any reason
property only_npc alias n_to			! unique NPC  to whom this Assertion applies.  This is a shortcut for putting npc-specific testing in Prerequisite.
property only_after alias e_to			! shortcut - only available immediately after the only_after Assertion is said.

class Assertion
{
	type assertion

	! short_desc		! prints the text of the menu item for this Assertion, *without* a trailing newline
	! long_desc		! performs the NPC's side of the conversational exchange

	Prerequisite			! by default all Assertions are available, and disappear once they've been selected; override this for custom behavior
	{
		if self is not visited : return true
		else : return false
	}
	only_npc 0
	only_after 0

	is not known		! don't trouble the parser with these objects
}

!----------------------------------------------------------------------------
! SetCurrentTopic(topic) - sets the current topic and builds the list of available
! conversational gambits.
!
! The 'obj' argument is the topic object to use, and is set to be the speaking NPC's
! current_topic.  If there are no available assertions in the given topic, the routine
! backs up the topic tree until it either finds some available assertions or hits a root
! topic, i.e. one whose parent is the global Conversation object.

routine SetCurrentTopic(obj)
{
	local a, i

	speaking.current_topic = obj

	i = 0
	for a in obj
	{
		! 'visited' testing is left to the Prerequisite function
		if a.type = assertion and
			(a.only_npc = 0 or a.only_npc = speaking) and
			(a.only_after = 0 or a.only_after = speaking.last_assertion) and
			a.Prerequisite
		{
			i++
			gAvailableAssertions[i] = a
			if i = gAvailableAssertions[] : break		! stop if we hit our limit
		}
	}

	gNumAvailableAssertions = i

	! if we didn't find any valid assertions, and this was not a
	! top-level topic, move up the topic tree and try again.
	if i = 0 and parent(obj).type = topic
	{
		SetCurrentTopic(parent(obj))
	}
}

!----------------------------------------------------------------------------
! SayAssertion(assertion, showPrompt) - display and record a conversational exchange.
! If showPrompt = true, the player's side of the exchange is displayed, otherwise only
! the NPC's side is shown.  This is useful when calling from the event that continues
! conversations even in the absence of prompting from the player.  This can be called
! even when there is no active conversation.
!
! If showPrompt is true, the player's side of the conversation (theAssert.short_desc)
! is displayed before the NPCs response (theAssert.long_desc).  This is done by calling
! the PrintAssertionPrompt() routine, which you can 'replace' to get custom formatting.
! By default, PrintAssertionPrompt() displays the player's question/comment enclosed
! in double quotes, followed by a single newline.

routine SayAssertion(theAssert, showPrompt)
{
	! Run the Assertion, mark it 'visited', and rebuild the assertion list. Note that
	! the assertion is marked 'visited' *before* its *_desc functions are called,
	! so those functions can cleanly mark it not visited to override the default
	! behavior, if desired.

	theAssert is visited

	if showPrompt
	{
		PrintAssertionPrompt(theAssert)					! player's half of the exchange, if designated
	}
	""																			! blank line before the NPC's response, always
	Indent																	! indent properly
	run theAssert.long_desc									! the NPC's response itself
	if speaking
	{
		speaking.last_assertion = theAssert				! remember what was last said
		SetCurrentTopic(speaking.current_topic)		! resetting the current topic rebuilds the available assertion list
	}
}

routine PrintAssertionPrompt(theAssert)
{
	Indent
	print "\"";
	run theAssert.short_desc		! should not end with a newline!
	"\""
}

!----------------------------------------------------------------------------
! Routines to cue an Assertion for a given NPC to say next time around.
! To clear an NPCs cued Assertion, call CueAssertion(theNPC, 0)

routine CueAssertion(theNPC, theAssertion)
{
	if theNPC ~= nothing : theNPC.next_assertion = theAssertion
}

routine ForceCueAssertion(theNPC, theAssertion)
{
	if theNPC ~= nothing : theNPC.force_assertion = theAssertion
}

!----------------------------------------------------------------------------
! Cueing messages in the future - fuse-based.  Note that because this uses
! a single fuse, you can have only one pending cue at a time.  If you want more
! than one NPC's conversation cued up at once, or multiple cues, just copy &
! extend this fuse concept.

#ifclear NO_FUSES

fuse cue_fuse
{
	misc 0							! refers to the NPC where the Assertion should be queued
	next_assertion 0			! the Assertion to cue
}

event in cue_fuse
{
	if not self.tick		! when the timer runs out
	{
		CueAssertion(cue_fuse.misc, cue_fuse.next_assertion)
		Deactivate(cue_fuse)
	}
}

routine CueAhead(theNPC, theAssertion, theDelay)
{
	cue_fuse.misc = theNPC
	cue_fuse.next_assertion = theAssertion
	Activate(cue_fuse, theDelay)
}

#endif		! ifclear NO_FUSES

!----------------------------------------------------------------------------
! Global event to handle cued assertions.  Non-forced assertions are favored over
! forced ones, since the forced ones will never be discarded automatically.

event
{
	if speaking and not gDidSpeak
	{
		local a

		a = nothing
		if speaking.next_assertion
		{
			a = speaking.next_assertion
			speaking.next_assertion = nothing		! i.e. CueAssertion(speaking, nothing)
		}
		elseif speaking.force_assertion
		{
			a = speaking.force_assertion
			speaking.force_assertion = nothing	! i.e. ForceCueAssertion(speaking, nothing)
		}
		if a
		{
			SayAssertion(a, false)
			PrintStatusline
			event_flag = 2		! force an interruption
		}
	}
	gDidSpeak = false
}

!----------------------------------------------------------------------------
! INITIALIZATION
! Before any conversation can occur, InitConverse() must be called.  This must
! happen *AFTER* the 'player' global has been initialized.

routine InitConverse
{
	speaking = 0
	gNumAvailableAssertions = 0

	! Put the Conversation tree in the player's scope in order to
	! provide access to the topic tree regardless of location.
	PutInScope(Conversation, player)
}

!----------------------------------------------------------------------------
! VERB ROUTINES FOR CONVERSATION

routine DoGreet
{
	local obj
	if object
	{
		obj = object
		object = nothing
		SpeakTo(obj)
	}
	else
	{
		ConverseVMessage(&DoGreet)		! whom do you want to greet?
		return false
	}
	return true
}

routine DoTopic
{
	if speaking = nothing
	{
		ConverseVMessage(&DoTopic, 1)	! "you're not talking to anybody!"
		return false
	}

	if object = nothing
	{
		ConverseVMessage(&DoTopic, 2)      ! "Ask about something specific..."
		return false
	}

	! If, despite all of our parse_rank adjustments, the parser has come up with a
	! non-Topic object here, we issue a relevant message and fail.
	if object.type ~= topic
	{
		ConverseVMessage(&DoTopic, 3)		! not an available topic
		return false
	}

	! okay, we have a new topic - flush any cued non-critical conversation under
	! the old topic
	if object ~= speaking.current_topic
	{
		speaking.next_assertion = nothing		! i.e. CueAssertion(speaking, nothing)
	}

	! Set the new topic
	SetCurrentTopic(object)

	! suitable feedback that the topic change has happened
	ConverseVMessage(&DoTopic, 4)		! "(You contemplate your options.)"

	PrintStatusline		! rebuild the status bar manually since this action doesn't reenter main
	return false			! change of topic doesn't cost a turn
}

routine DoConverse
{
	! this routine actually performs a conversational exchange

	! be nice to have a string-to-number conversion routine....
	local which
	select word[1]
		case "1" : which = 1
		case "2" : which = 2
		case "3" : which = 3
		case "4" : which = 4
		case "5" : which = 5

	if which <= gNumAvailableAssertions
	{
		! valid option, so clear any cued Assertion and run the newly- selected one.  Doing
		! so in this order is necessary so that the cued assertion doesn't get cleared after the
		! newly-run one has just established it.

		speaking.next_assertion = nothing		! i.e. CueAssertion(speaking, nothing)
		gDidSpeak = true
		SayAssertion(gAvailableAssertions[which], true)
		return true
	}
	else
	{
		ConverseVMessage(&DoConverse)		! "that option is not currently available"
		return false
	}
}

!----------------------------------------------------------------------------
! The Converse library uses its own VMessage-like system.  To override these,
! replace NewConverseVMessages(r, num, a, b), just like the ordinary verblib.h
! VMessage / NewVMessages system.

routine ConverseVMessage(r, num, a, b)
{
	if NewConverseVMessages(r, num, a, b):  return

	select r
	case &DoGreet
	{
		"It's unclear whom do you wish to greet."
	}

	case &DoTopic
	{
		select num
			case 1: "You're not currently conversing with anyone."
			case 2: "Please be more specific about what topic you wish to introduce."
			case 3: "You are unsure how to broach that subject."
			case 4: "(You contemplate your options.)"
	}

	case &DoConverse
	{
		"That option is not currently available."
	}
}

routine NewConverseVMessages(r, num, a, b)      ! The NewConverseVMessages routine may be
{                                       ! REPLACED, and should return true
	return false                    ! if a replacement message <num>
}                                       ! exists for routine <r>

!----------------------------------------------------------------------------
! REPLACEMENTS FOR HUGOLIB ROUTINES
!----------------------------------------------------------------------------


!----------------------------------------------------------------------------
! DoWait() in hugolib defaults to 3 turns, which can interfere with queued-conversation
! event management.  This replacement makes the default 1 turn rather than 3.

replace DoWait(count)                   ! count argument is from DoWaitUntil
{
	if object = 0
		count = 1
	elseif count = 0
		count = object

	VMessage(&DoWait)                ! "Time passes..."
	event_flag = 0
	while --count
	{
		Main
		if event_flag
			if not KeepWaiting
				return
	}
	event_flag = 0
	return true
}

!----------------------------------------------------------------------------
! custom statusline when using the Converse system

replace PrintStatusline
{
	! because we might be changing the status line's height,
	! clear the old one *before* recalculating its size
	window display.statusline_height
	{
		cls
	}

	if speaking ~= 0
		display.statusline_height = 7
	else
		display.statusline_height = 1

	Font(BOLD_OFF | ITALIC_OFF | UNDERLINE_OFF | PROP_OFF)
	window display.statusline_height
	{
		color SL_TEXTCOLOR, SL_BGCOLOR
		cls			! make sure we've drawn the entire status bar in the proper colors

		locate 2, 1
		if speaking ~= 0
			print "Talking to: "; capital speaking.name;
		else
			print "No conversation";
		print to (display.linelength / 2);

		if not light_source
			print "In the dark";
		else
			print capital location.name;

		! now print the available conversational options
		if speaking ~= 0
		{
			local line
			for (line = 1; line <= gNumAvailableAssertions; line++)
			{
				locate 2, (line+2)		! skip a line
				print number line; ": ";
				run gAvailableAssertions[line].short_desc
			}
		}
	}
	color TEXTCOLOR, BGCOLOR
	Font(DEFAULT_FONT)
}

!----------------------------------------------------------------------------
! SpeakTo is called by the engine when a command is addressed to a
! character object via:  SpeakTo(character)
!
! For example, the input line
!
!   <character>, get the <object>
!
! will call SpeakTo(<character>), with the global object containing <object>,
! and the verbroutine global set to &DoGet
!
! The Converse library replaces this routine to make the proper calls to
! DoGreet() when the player addresses an NPC, rather than using the DoHello
! DoAsk/DoTalkTo mechanism.

replace SpeakTo(char)
{
	speaking = 0

	if char is not living
	{
		ParseError(6)
		return
	}

	AssignPronoun(char)

	if not FindObject(char, location)
	{
		ParseError(11, char)
		return
	}
	elseif char = player
	{
		Message(&Speakto, 1)    ! "Stop talking to yourself..."
		return
	}
	elseif object is not known
	{
		ParseError(10, object)
		return
	}

	if char is unfriendly
		jump IgnorePlayerReplaced

	speaking = char

	select verbroutine
#ifclear NO_VERBS
		case 0, &DoGreet, &DoHello			! just an NPC name is like >GREET NPC, as is >NPC, HELLO
		{
			run char.greeted
			return true			! this takes a turn
		}
#endif  ! ifclear NO_VERBS

		case else
		{

! If the character can respond to a request, this should be dealt with by
! an order_response property routine; order_response--if it exists--should
! return false if there is no response for the given verbroutine

:TryResponseReplaced

			if not char.order_response
				jump IgnorePlayerReplaced
			return true
		}
	return false

:IgnorePlayerReplaced

	if not char.ignore_response
		Message(&Speakto, 4, char)      ! "X ignores you."
	speaking = 0
	return true
}

!----------------------------------------------------------------------------
! The ASK/TELL verbroutines get stubbed out with the Converse system; now they all just
! give instructions on how to engage in conversation.  Replace the PrintConverseUsage()
! routine in order to customize the message; just make sure your replacement returns
! false.

replace DoHello
{
	! Handle like >GREET
	if object.type = npc
	{
		SpeakTo(object)
		return true
	}
	else : return PrintConverseUsage
}

replace DoTalk
{
	! redirect to >GREET in the "TALK TO npc" case
	if object.type = npc and xobject = 0
	{
		verbroutine = &DoGreet
		SpeakTo(object)
		return true
	}
	else : return PrintConverseUsage
}

replace DoAsk
{
	return PrintConverseUsage
}

replace DoAskQuestion
{
	return PrintConverseUsage
}

replace DoTell
{
	return PrintConverseUsage
}

routine PrintConverseUsage
{
	"\nTo enter into conversation with a character, type \BGREET <Character Name>\b.  You can then
	enter the numbers corresponding to the available conversational gambits shown in the status
	bar area.  If there are no such gambits available, or if you wish to direct the conversation to a
	particular subject, at any time during a conversation you may enter \BTOPIC <subject>\b to
	change the matter under discussion.  Selecting a new topic does not count as a turn."

	"\n\BTALK TO\b and \BHELLO\b are synonyms for \BGREET\b, and \BTOPIC\b can be abbreviated \BT\b."

	return false
}

#endif
