/*
	HEALLEG.C

	Non-portable functions specific to DJGPP/Allegro (32-bit build)

	for the Hugo Engine

	Copyright (c) 1995-2006 by Kent Tessman

	NOTE:  See HEBLANK.C for elaboration/further documentation of
	these interface routines (as well as equivalents for other
	compilers/versions such as HEQUICKC.C, HEDJGPP.C, HELCC.C,
	etc.).

	(Also note that since we're using Allegro, this port must link in
	the Allegro library "liballego.a".)
*/


#include "heheader.h"

#include <bios.h>
#include <dpmi.h>
#include <go32.h>
#include <sys/farptr.h>

#include "mikmod.h"

#define line line_unused
#include "allegro.h"
#undef line

/* Function prototypes: */
void hugo_addcommand(void);
int hugo_color(int c);
void hugo_restorecommand(int n);
#if defined (NO_LATIN1_CHARSET)
void hugo_stripaccents(char *a);
#endif

/* Specific to HEALLEG.C: */
void ConstrainCursor(void);
void PrintPortInformation(void);
void PrintGRchar(int c);
void UpdateScreen(void);

#if defined (DEBUGGER)
void *AllocMemory(size_t size);
#endif

#if defined (FRONT_END)
int he_main(int argc, char **argv);     /* from he.c */
void Banner(void);
#endif

void hugo_initpalette(void);            /* from hejpeg.c */

int InitPlayer(void);			/* from hesound.c */
void hugo_stopmusic(void);
void hugo_stopsample(void);
void SuspendAudio(void);
void ResumeAudio(void);

int getch(void);			/* C library */


/* Defined Hugo colors: */
#define HUGO_BLACK         0
#define HUGO_BLUE          1
#define HUGO_GREEN         2
#define HUGO_CYAN          3
#define HUGO_RED           4
#define HUGO_MAGENTA       5
#define HUGO_BROWN         6
#define HUGO_WHITE         7
#define HUGO_DARK_GRAY     8
#define HUGO_LIGHT_BLUE    9
#define HUGO_LIGHT_GREEN   10
#define HUGO_LIGHT_CYAN    11
#define HUGO_LIGHT_RED     12
#define HUGO_LIGHT_MAGENTA 13
#define HUGO_YELLOW        14
#define HUGO_BRIGHT_WHITE  15

#define HOME_KEY        0x200
#define END_KEY         0x201
#define DELETE_KEY      0x202
#define INSERT_KEY      0x203
#define BACKSPACE_KEY   0x204
#define CTRL_LEFT_KEY   0x205
#define CTRL_RIGHT_KEY  0x206

int insert_mode = true;

/* Since HEALLEG.C provides its own command history: */
#define HISTORY_SIZE    16              /* for command-line editing */
int hcount = 0;
char *history[HISTORY_SIZE];


/* Now some variables and constants to manage graphics/text display--
   all specific to HEALLEG.C
*/
int text_windowleft, text_windowtop, text_windowright, text_windowbottom,
	text_windowwidth, text_windowheight;
int FONT_WIDTH = 8, FONT_HEIGHT;
unsigned long text_font_addr;

int print_cursor = 1;   /* since Allegro is always in graphics mode, a
			   cursor must be explicitly displayed */
char override_latin1_conversion = false;

const int SVGA_DT = GFX_AUTODETECT;         /* constants from <allegro.h> */
const int VGA_DT = GFX_MODEX; //GFX_VGA;
const int NONE_DT = GFX_TEXT;

/* From hejpeg.c: */
extern int SCREENCOLORS;
extern int display_type, color_space;
extern int HUGO_PALETTE[];              /* for Hugo text colors */
extern int reduce_quality;

/* Global video hardware settings */
int color_depth, real_display_type;
int display_width, display_height;

/* Copied from jpeglib.h: */
typedef enum
{
	JCS_UNKNOWN,            /* error/unspecified */
	JCS_GRAYSCALE,          /* monochrome */
	JCS_RGB,                /* red/green/blue */
	JCS_YCbCr,              /* Y/Cb/Cr (also known as YUV) */
	JCS_CMYK,               /* C/M/Y/K */
	JCS_YCCK                /* Y/Cb/Cr/K */
} J_COLOR_SPACE;

/* Various variables to help mimic a text display: */
int current_text_color = DEF_FCOLOR, current_back_color = DEF_BGCOLOR,
	current_text_col = 1, current_text_row = 1, current_text_font = 0;

BITMAP *screen_buffer;
char screen_needs_repainting = 0;

/* From hesound.c: */
extern char no_audio;
extern char audio_suspended;
extern char player_started;
extern UNIMOD *modfile;
extern char music_is_playing;
extern MSAMPLE *sample;
extern char sample_is_playing;

#if !defined (NO_LATIN1_CHARSET)
#define LATIN1_START 0xa0
char latin1_to_ibm[] =
{
	0x20, 0xad, 0x9b, 0x9c,  ' ', 0x9d,  ' ', 0x15,	/* a0 - a7 */
	 ' ',  'c', 0xa6, 0xae, 0xaa, 0xc4,  'R',  ' ',	/* a8 - af */
	0xf8, 0xf1, 0xfd, 0xfc, 0x27, 0xe6,  ' ', 0xfa,	/* b0 - b7 */
	 ' ',  ' ', 0xa7, 0xaf, 0xac, 0xab,  ' ', 0xa8,	/* b8 - bf */
	 'A',  'A',  'A',  'A', 0x8e, 0x8f, 0x92, 0x80,	/* c0 - c7 */
	 'E', 0x90,  'E',  'E',  'I',  'I',  'I',  'I',	/* c8 - cf */
	 'D', 0xa5,  'O',  'O',  'O',  'O', 0x99,  'x',	/* d0 - d7 */
	 'O',  'U',  'U',  'U', 0x9a,  'Y',  ' ', 0xe1,	/* d8 - df */
	0x85, 0xa0, 0x83,  'a', 0x84, 0x86, 0x91, 0x87,	/* e0 - e7 */
	0x8a, 0x82, 0x88, 0x89, 0x8d, 0xa1, 0x8c, 0x8b,	/* e8 - ef */
	 'd', 0xa4, 0x95, 0xa2, 0x93,  'o', 0x94, 0xf6,	/* f0 - f7 */
	 'o', 0x97, 0xa3, 0x96, 0x81,  'y',  ' ', 0x98	/* f8 - ff */
};
#endif


#if defined (FRONT_END)
/* main

	Replaces the old main() in HE.C (since FRONT_END is #defined),
	since we need to process MS-DOS-specific switches for graphics
	mode setting.
*/

int main(int argc, char *argv[])
{
	unsigned int i, j;
	int arg, r;
	int small_font = false;
	__dpmi_regs regs;


/* Initialize display presets: */

	color_depth = 8;        /* default is 8-bit palette */

	/* These will vary from mode to mode: */
	real_display_type = display_type = SVGA_DT;
	display_width = 800, display_height = 600;

	color_space = JCS_RGB;          /* from jpeglib.h */


/* Parse the command line */

	for (arg=1; arg<argc; arg++)
	{
		if (argv[arg][0]=='-')
		{
			for (i=1; i<strlen(argv[arg]); i++)
			{
				switch (tolower(argv[arg][i]))
				{
					case 'h':
						real_display_type = display_type = SVGA_DT;
						r = (argv[arg][++i]);
						if (r=='1')
						{
							i+=3,
							display_width = 1024,
							display_height = 768;
							break;
						}
						else if (r=='8')
						{
							i+=2,
							display_width = 800,
							display_height = 600;
							break;
						}
						else if (r=='6')
						{
							i+=2,
							display_width = 640,
							display_height = 480;
						}
						else
							display_width = 800,
							display_height = 600;
						break;
#if !defined (DEBUGGER)
					case 'l':
						real_display_type = display_type = VGA_DT;
						display_width = 320, display_height = 240;
						//display_width = 320, display_height = 200;
						break;
#endif

					case 't':
						display_type = NONE_DT;
						break;

					case 'm':
						color_depth = 32;
						break;
					case '1':	/* (undocumented test switch) */
						i++;	/* '6' in "16" */
						color_depth = 16;
						break;
					case 'p':
						color_depth = 8;
						break;

					case 'b':
						color_space = JCS_GRAYSCALE;
						break;

					case 's':
						small_font = true;
						break;

					case 'r':
						reduce_quality = true;
						break;

					case 'x':
						no_audio = true;
						break;

					case '?':
						PrintPortInformation();
						exit(0);

					default:
						goto PrintSwitches;
				}
			}
		}
	}

	/* Now skim through and remove any switches, since
	   ParseCommandLine() in HEMISC.C doesn't know a switch
	   from a hole in the ground
	*/
	for (i=1; i<(unsigned)argc; i++)
	{
		if (argv[i][0]=='-')
		{
			for (j=i; j<(unsigned)argc-1; j++)
				argv[j] = argv[j+1];
			arg--, argc--, i--;
		}
	}

#if !defined (IOTEST)
	if (argc==1)
	{
PrintSwitches:
		Banner();
		printf("SWITCHES:");
		printf("\t-h     High resolution\n");
		printf("\t\t-hnnnn (nnnn is 1024, 800, or 640 pixels)\n");
#if !defined (DEBUGGER)
		printf("\t\t-l     Low resolution\n");
#endif
		printf("\t\t-t     Text-only\n\n");
		
		printf("\t\t-m     Millions of colors (slower scrolling)\n");
		printf("\t\t-16    16-bit color\n");
		printf("\t\t-p     Palette mode (faster scrolling - default)\n\n");
		
		printf("\t\t-b     Black-and-white graphics mode\n");
		printf("\t\t-s     Small font (high-res/text only)\n");
		printf("\t\t-r     Reduce quality (faster image loading)\n");
		printf("\t\t-x     Disable sound/music\n");
		printf("\t\t-?     Additional information\n");
		exit(0);
	}
#endif	/* !defined (IOTEST) */


/* Initialize the graphical font parameters */

	/* "Large" font only for 640x480 */
	if (real_display_type!=SVGA_DT) small_font = true;

	/* Calculate dimensions for small font... */
	if (small_font)
	{
		FONT_HEIGHT = 8;
		regs.h.bh = 0x03;	/* font code for VGA 8x8 */
	}

	/* ...or regular-sized font */
	else
	{
		FONT_HEIGHT = 16;
		regs.h.bh = 0x06;	/* font code for VGA 8x16 */
	}

	/* Find the address of the font table using Int 10h */
	regs.h.ah = 0x11;	/* function */
	regs.h.al = 0x30;	/* sub-function */
	__dpmi_int(0x10, &regs);
	text_font_addr = (unsigned long)(regs.x.es*16L + regs.x.bp);

/* Get Allegro up and running */

	allegro_init();

	/* Make sure we have hardware that'll display graphics */
	if (color_space==JCS_GRAYSCALE) color_depth = 8;
	set_color_depth(color_depth);
	while (set_gfx_mode(real_display_type, display_width, display_height, 0, 0))
	{
		if (color_depth==8)     /* can't get any lower than 8 bits */
		{
			set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
			printf("Fatal Error:  Unable to support requested video mode\n");
			printf("(Try a different resolution or color mode)\n");

			/* Because pre-Win95 (or NT) doesn't automatically switch to
			   full-screen if necessary
			*/
			if (windows_version && (windows_version<4 || windows_version>=0x100))
				printf("\n(Must be started from full-screen DOS mode)\n");
			exit(-1);
		}
		else
			set_color_depth(color_depth-=8);
	}
	screen_buffer = create_bitmap(display_width, display_height);

	/* Initialize the palette */
	hugo_initpalette();


/* Initialize Allegro sound */

//	install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL);


/* Now set up MikMod */

	if (!no_audio)
	{
		if (!InitPlayer()) audio_suspended = true;
	}

/* Call the engine entry point--we won't return to this function
   until the engine terminates
*/
	r = he_main(argc, argv);


/* Before terminating the engine */

	/* Free sample if one was playing */
	if (sample_is_playing) hugo_stopsample();

	/* Free music if it was playing */
	if (music_is_playing) hugo_stopmusic();

	/* Clean up the sound driver */
	if (!no_audio && !audio_suspended)
	{
		MD_PlayStop();
		MD_Exit();
	}


	return r;			/* returned from he_main() */
}
#endif  /* if defined (FRONT_END) */

void PrintPortInformation(void)
{
#if !defined (IOTEST)
	Banner();
#endif
	printf(
"\nThis version of the Hugo Engine has SVGA-enabled graphics support.  Its\n\
functioning modes are high-resolution (640x480, 800x600, 1024x768) and low-\n\
resolution (320x200).  Color depths range from 8-bit (256 colors) to 32-bit\n\
(millions of colors) depending on video hardware.  When running graphics under\n\
Windows, the Hugo Engine should be started from full-screen DOS mode.  Music\n\
and sound effects may be played through a supported sound device.\n");
	printf(
"\n(Function keys:  ALT-R to restore a corrupted display; ALT-S to toggle\n\
sound on/off)\n");
	printf(
"\nThis port is powered by Allegro by Shawn Hargreaves (and others).  JPEG\n\
graphics display is based in part on the work of the Independent JPEG Group.\n\
MikMod sound system used under license.\n");
}


/*
    MEMORY ALLOCATION:
*/

void *hugo_blockalloc(long num)
{
	return malloc(num*sizeof(char));
}

void hugo_blockfree(void *block)
{
	free(block);
}


/*
    FILENAME MANAGEMENT:

    Different operating systems have different ways of naming files.
    The following routines are simply required to know and be able to
    dissect/build the components of a particular filename,
    storing/restoring the components via the specified char arrays.

    For example, in MS-DOS:

	hugo_splitpath("C:\HUGO\FILES\HUGOLIB.H", ...)
		becomes:  C:, HUGO\FILES, HUGOLIB, H

    and

	hugo_makepath(..., "C:", "HUGO\FILES", "HUGOLIB", "H")
		becomes:  C:\HUGO\FILES\HUGOLIB.H

    The appropriate equivalent nomenclature should be used for the
    operating system in question.
*/

void hugo_splitpath(char *path, char *drive, char *dir, char *fname, char *ext)
{
	fnsplit(path, drive, dir, fname, ext);  /* from libc */
}

void hugo_makepath(char *path, char *drive, char *dir, char *fname, char *ext)
{
	fnmerge(path, drive, dir, fname, ext);  /* from libc */
}


/* hugo_getfilename

    Loads the name of the filename to save or restore (as specified by
    the argument <a>) into the line[] array.
*/

void hugo_getfilename(char *a, char *b)
{
	unsigned int i, p;

/* An alternate method using the Allegro GUI routines:

	char path[MAXPATH];

	gui_fg_color = HUGO_PALETTE[15];
	gui_bg_color = HUGO_PALETTE[1];
	gui_mg_color = HUGO_PALETTE[8];

	strcpy(path, b);

	sprintf(line, "Path/filename %s", a);

	if (file_select(line, path, NULL))
		strcpy(line, path);
	else
		strcpy(line, "");
*/

	sprintf(line, "Enter path and filename %s.", a);

	AP(line);

	sprintf(line,"%c(Default is %s): \\;", NO_CONTROLCHAR, b);
	AP(line);

	p = var[prompt];
	var[prompt] = 0;        /* null string */

	RunInput();

	var[prompt] = p;

	remaining = 0;

	strcpy(line, "");
	if (words==0)
		strcpy(line, b);
	else
	{
		for (i=1; i<=(unsigned int)words; i++)
			strcat(line, word[i]);
	}
}


/* hugo_overwrite

    Checks to see if the given filename already exists, and prompts to
    replace it.  Returns true if file may be overwritten.
*/

int hugo_overwrite(char *f)
{
	FILE *tempfile;

	if (!(tempfile = fopen(f, "rb")))       /* if file doesn't exist */
		return true;

	fclose(tempfile);

	sprintf(pbuffer, "Overwrite existing \"%s\" (Y or N)? ", f);

/* An alternate method using the Allegro GUI routines:

	if (alert("File Already Exists", pbuffer, "", "&Yes", "&No", 'y', 'n'))
		return true;
	else
		return false;
*/
	RunInput();

	if (words==1 && (!strcmp(strupr(word[1]), "Y") || !strcmp(strupr(word[1]), "YES")))
		return true;

	return false;
}


/* hugo_closefiles

    Closes all open files.  NOTE:  If the operating system automatically
    closes any open streams upon exit from the program, this function may
    be left empty.
*/

void hugo_closefiles()
{
	fclose(game);
	if (script) fclose(script);
	if (io) fclose(io);
	if (record) fclose(record);
}


/* hugo_getkey

    Returns the next keystroke waiting in the keyboard buffer.  It is
    expected that hugo_getkey() will return the following modified
    keystrokes:

	up-arrow        11 (CTRL-K)
	down-arrow      10 (CTRL-J)
	left-arrow       8 (CTRL-H)
	right-arrow     21 (CTRL-U)
*/

int hugo_getkey(void)
{
	int b;

	if (screen_needs_repainting) UpdateScreen();

	while (!(b = _bios_keybrd(_KEYBRD_READY)));

	b = _bios_keybrd(_KEYBRD_READ);

	if (b==-1) b = 3;       /* CTRL-BREAK */

	/* A minor hack because we don't want the left-arrow and backspace
	   to have the same effect:
	*/
	if ((b&0xff)==8) return BACKSPACE_KEY;

	/* With this BIOS call, function keys are returned with
	   0x00 in the bottom 8 bits and the scan code in the top 8.
	*/

	if ((b&0xff)==0)
	{
		switch (b>>8)
		{
			case KEY_UP:
				{b = 11; break;}
			case KEY_LEFT:
				{b = 8; break;}
			case KEY_RIGHT:
				{b = 21; break;}
			case KEY_DOWN:
				{b = 10; break;}
			case KEY_HOME:
				{b = HOME_KEY; break;}
			case KEY_END:
				{b = END_KEY; break;}
			case KEY_INSERT:
				{b = INSERT_KEY; break;}
			case KEY_DEL:
				{b = DELETE_KEY; break;}
			case 115:
				{b = CTRL_LEFT_KEY; break;}
			case 116:
				{b = CTRL_RIGHT_KEY; break;}

			case KEY_R:             /* redraw SVGA screen */
			{
#if !defined (DEBUGGER)
				if (real_display_type==SVGA_DT)
#else
				if (active_screen==GAME && real_display_type==SVGA_DT)
#endif
					set_color_depth(color_depth);
					set_gfx_mode(real_display_type, display_width, display_height, 0, 0);
					hugo_initpalette();
//					hugo_clearfullscreen();
					UpdateScreen();
				b = 0;
				break;
			}

			case KEY_S:		/* toggle sound on/off */
			{
				if (audio_suspended)
					ResumeAudio();
				else
					SuspendAudio();
				b = 0;
				break;
			}

			default:
				b += 256;
		}
		return b;
	}
#if !defined (NO_LATIN1_CHARSET)
	/* Convert DOS extended-ASCII codes to Latin1 */
        else if ((b&0xff) > 128)
	{
		int i;
		
		for (i=LATIN1_START; i<=0xff; i++)
		{
			if ((unsigned char)latin1_to_ibm[i-LATIN1_START]==(b&0xff))
			{
				b = i;
				break;
			}
		}
		if (i>0xff) b = ' ';	/* invalid character */
	}
#endif

#if !defined (DEBUGGER)
	if (b==3)
#else
	if (b==3 && active_screen==GAME)
#endif
	{
		hugo_cleanup_screen();
		printf("Hugo Engine shutting down due to CTRL-C interrupt\n");
		exit(-1);
	}

	return b&0xff;          /* strip everything but character */
}


/* hugo_getline

    Gets a line of input from the keyboard, storing it in <buffer>.

    (NOTE:  The function keys used here are MS-DOS specific.
    They have been chosen to somewhat mirror the command-line editing
    keys of MS-DOS.  For other systems, the "switch (b)" lines
    will likely need to be changed.)

    The advantage of using this instead of an available library input
    function is when building the Debugger--pressing the Tab key switches
    to the debug screen; if a command such as Restart, Restore, etc. is
    given, the engine must know to abandon the input.

    Note also that this input function assumes a non-proportional font.
*/

#define CURRENT_CHAR(c) ((c<(int)strlen(buffer))?buffer[c]:' ')

void hugo_getline(char *p)
{
	char ch[2];
	int a, b, thiscommand;
	int c;                          /* c is the character being added */
	int x, y, oldx, oldy;

NewPrompt:
	hugo_settextcolor(fcolor);
	hugo_setbackcolor(bgcolor);

	strcpy(buffer, "");
	c = 0;
	thiscommand = hcount;

	oldx = (currentpos+charwidth)/charwidth;
	oldy = currentline;
	y = oldy;
	x = oldx + hugo_textwidth(p)/charwidth;

	hugo_print(p);

GetKey:

	hugo_settextpos(x, y);
	if (print_cursor)
	{
		override_latin1_conversion = true;
		sprintf(ch, "%c", 219);
		hugo_print(ch);
		override_latin1_conversion = false;
	}

	b = hugo_getkey();

	hugo_settextcolor(icolor);
	hugo_setbackcolor(bgcolor);

	/* Now, start key checking */
	switch (b)
	{
#if defined (DEBUGGER)
		case (9):                       /* Tab */
		{
			during_input = true;
			Debugger();
			during_input = false;

			/* If the debugger is stopping execution, quitting,
			   etc., a blank string must be returned in the
			   input buffer.
			*/
			if (debugger_collapsing)
			{
				strcpy(buffer, "");
				if (active_screen==GAME)
					hugo_scrollwindowup();
				return;
			}

			goto GetKey;
		}
#endif

		case (13):                      /* Enter */
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			full = 0;

			if (script) fprintf(script, "%s%s\n", p, buffer);

			while (kbhit())
				getch();
			hugo_settextpos(1, y + 2);
			if (y >= physical_windowheight/lineheight - 1)
				hugo_scrollwindowup();

			strcpy(buffer, Rtrim(buffer));
			hugo_addcommand();
#if defined (NO_LATIN1_CHARSET)
			hugo_stripaccents(buffer);
#endif

			return;
		}
		case INSERT_KEY:
		{
			insert_mode = !insert_mode;
			goto GetKey;
		}
		case BACKSPACE_KEY:
		case DELETE_KEY:
		{
			if (strlen(buffer)>0)
			{
				if (b==BACKSPACE_KEY)
				{
					if (c==0) goto GetKey;
					c--;
				}

				for (a=c; a<=(int)strlen(buffer); a++)
					buffer[a] = buffer[a+1];

				if (b==BACKSPACE_KEY) x--;

				hugo_settextpos(oldx+hugo_strlen(p), y);
				hugo_print(buffer);
				hugo_settextpos(oldx+hugo_strlen(p)+strlen(buffer), y);
				hugo_print("  ");
			}
			goto GetKey;
		}
		case (8):                       /* left-arrow */
		{
			if (c > 0)
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				x--, c--;
			}
			goto GetKey;
		}
		case (21):                      /* right-arrow */
		{
			if (c<(int)strlen(buffer))
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				x++, c++;
			}
			goto GetKey;
		}
		case CTRL_LEFT_KEY:
		{
			if (c)
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				do
				{
					do
						c--, x--;
					while (c && buffer[c-1]!=' ');
				}
				while (c && buffer[c]==' ');
			}
			goto GetKey;
		}
		case CTRL_RIGHT_KEY:
		{
			if (c<(int)strlen(buffer))
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				do
				{
					do
						c++, x++;
					while (c<(int)strlen(buffer) &&
						buffer[c-1]!=' ');
				}
				while (c<(int)strlen(buffer) && buffer[c]==' ');
			}
			goto GetKey;
		}
		case HOME_KEY:
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			c = 0;
			x = oldx + hugo_strlen(p);
			goto GetKey;
		}
		case END_KEY:
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			c = strlen(buffer);
			x = oldx + hugo_strlen(p) + strlen(buffer);
			goto GetKey;
		}
		case (27):                      /* Escape */
		case (24):                      /* CTRL-X */
		case (KEY_ESC<<8):
		{
BlankLine:
			hugo_settextpos(oldx+hugo_strlen(p), y);

			memset(buffer, ' ', text_windowwidth);

			buffer[text_windowwidth-oldx-hugo_strlen(p)] = '\0';
			hugo_print(buffer);
			hugo_settextpos(oldx, y);

			goto NewPrompt;
		}
		case (KEY_F3<<8):               /* F3 */
		case (11):                      /* up-arrow */
		{
			if (--thiscommand<0)
			{
				thiscommand = 0;
				goto GetKey;
			}
			a = strlen(buffer);
RestoreCommand:
			hugo_restorecommand(thiscommand);
			x = oldx + strlen(buffer) + hugo_strlen(p);
			hugo_settextpos(oldx+hugo_strlen(p), y);
			hugo_print(buffer);
			while (a >= (int)strlen(buffer))
				{hugo_print(" ");
				a--;}
			hugo_settextpos(x-1, y);
			c = strlen(buffer);
			goto GetKey;
		}
		case (10):                      /* down-arrow */
		{
			a = strlen(buffer);
			if (++thiscommand>=hcount) goto BlankLine;
			goto RestoreCommand;
		}
	}

	/* Disallow invalid keystrokes */
	if (b < 32 || b>=256) goto GetKey;

	/* Hugo circa v2.2 allowed '^' and '~' for '\n' and '\"',
	   respectively
	*/
	if (game_version<=22 && (b=='^' || b=='~')) goto GetKey;

	if (oldx+strlen(buffer) >= text_windowwidth-2)
		goto GetKey;

	hugo_settextpos(x++, y);
	sprintf(ch, "%c", b);                   /* add the new character */
	hugo_print(ch);
	buffer[strlen(buffer)+1] = '\0';
	if (c<(int)strlen(buffer) && insert_mode)
	{
		hugo_settextpos(x, y);
		hugo_print(buffer+c);
		for (a=strlen(buffer); a>c; a--)
			buffer[a] = buffer[a-1];
	}
	buffer[c] = (char)b;

	c++;

	goto GetKey;
}


/* hugo_waitforkey

    Provided to be replaced by multitasking systems where cycling while
    waiting for a keystroke may not be such a hot idea.
*/

int hugo_waitforkey(void)
{
	UpdateScreen();

	while (_bios_keybrd(_KEYBRD_READY))     /* clear key buffer */
		hugo_getkey();

/*
	while (!_bios_keybrd(_KEYBRD_READY));
*/
	return hugo_getkey();
}


/* hugo_iskeywaiting

    Returns true if a keypress is waiting to be retrieved.
*/

int hugo_iskeywaiting(void)
{
	return _bios_keybrd(_KEYBRD_READY)?1:0;
}


/* hugo_timewait

    Waits for 1/n seconds.  Returns false if waiting is unsupported.
*/

int hugo_timewait(int n)
{
	static volatile int repaint_interval = 0;
	
	delay(1000/n);
	
	if (screen_needs_repainting)
	{
		/* Repaint only ever 10th of a second in order to not bog
		   things down
		*/
		if ((repaint_interval+=1000/n) >= 100)
		{
			UpdateScreen();
			repaint_interval = 0;
		}
	}
		
	return true;
}


/*
    COMMAND HISTORY:

    To store/retrieve player inputs for editing.
*/

void hugo_addcommand(void)
{
	int i;

	if (!strcmp(buffer, "")) return;

	if (hcount>=HISTORY_SIZE)
	{
		hugo_blockfree(history[0]);
		for (i=0; i<HISTORY_SIZE-1; i++)
			history[i] = history[i+1];
		hcount = HISTORY_SIZE-1;
	}

	/* Because the debugger might use (up to) all available memory for
	   code line storage, a different means of memory allocation is
	   needed (at least in MS-DOS due to limited available memory to
	   begin with).
	*/
#if !defined (DEBUGGER)
	if ((history[hcount] = (char *)hugo_blockalloc((long)((strlen(buffer)+1)*sizeof(char))))==NULL)
#else
	if ((history[hcount] = (char *)AllocMemory((size_t)((strlen(buffer)+1)*sizeof(char))))==NULL)
#endif
	{
		hugo_blockfree(history[0]);
		if (hcount)
		{
			for (i=0; i<hcount; i++)
				history[i] = history[i+1];
			hcount--;
		}
		return;
	}

	for (i=0; i<=(int)strlen(buffer); i++)
		history[hcount][i] = buffer[i];
	hcount++;
}

void hugo_restorecommand(int n)
{
	int i;

	if (n < 0 || (n>=hcount && hcount!=HISTORY_SIZE-1)) return;

	i = 0;
	do
		buffer[i] = history[n][i];
	while (history[n][i++]!='\0');
}


/* hugo_stripaccents */

#if defined (NO_LATIN1_CHARSET)

void hugo_stripaccents(char *a)
{
	/* If there is no Latin1 character set, accented characters
	   should be converted to ASCII equivalents at input time.
	   (NOTE:  These >127 extended ASCII codes are not 
	   machine-independent.) */

	int i;

	for (i=0; i<(int)strlen(a); i++)
	{
		if ((unsigned char)a[i]>=0x80)
		{
			switch ((unsigned char)a[i])
			{
				case 131:               /* 'A' accents */
				case 132:
				case 133:
				case 134:
				case 160:
					{a[i] = 'a';
					break;}
				case 142:
				case 143:
					{a[i] = 'A';
					break;}

				case 135:               /* 'C' cedilla */
					{a[i] = 'c';
					break;}
				case 128:
					{a[i] = 'C';
					break;}

				case 130:               /* 'E' accents */
				case 136:
				case 137:
				case 138:
					{a[i] = 'e';
					break;}
				case 144:
					{a[i] = 'E';
					break;}

				case 139:               /* 'I' accents */
				case 140:
				case 141:
				case 161:
					{a[i] = 'i';
					break;}

				case 164:               /* 'N' tilde */
					{a[i] = 'n';
					break;}
				case 165:
					{a[i] = 'N';
					break;}

				case 147:               /* 'O' accents */
				case 148:
				case 149:
				case 162:
					{a[i] = 'o';
					break;}
				case 153:
					{a[i] = 'O';
					break;}

				case 129:               /* 'U' accents */
				case 150:
				case 151:
				case 163:
					{a[i] = 'u';
					break;}
				case 154:
					{a[i] = 'U';
					break;}

				case 152:               /* 'Y' accents */
					{a[i] = 'y';
					break;}

				case 174:               /* '<<' */
				case 175:               /* '>>' */
					{a[i] = '\"';
					break;}

				default:
				{
					strcpy(a+i, a+i+1);
				}
			}
		}
	}
}

#endif	/* if defined (NO_LATIN1_CHARSET) */


/*
    DISPLAY CONTROL:
*/


/* Because Allegro uses a buffered system to display, first writing everything
   to a memory bitmap, then blitting it to the screen:
*/
void UpdateScreen(void)
{
	blit(screen_buffer, screen, 0, 0, 0, 0, display_width, display_height);
	screen_needs_repainting = false;
}


#if !defined (DEBUGGER)         /* If the debugger is being built, it will
				   take care of these */
void hugo_init_screen(void)
{
/* Does whatever has to be done to initially set up the display. */

	/* NOTE:  Allegro is initialized in main(), above */
}

void hugo_cleanup_screen(void)
{
/* Does whatever has to be done to clean up the display pre-termination. */

	allegro_exit();
}

#endif  /* !defined (DEBUGGER) */


int hugo_hasgraphics(void)
{
	if (display_type==NONE_DT)
		return false;
	return true;
}


void hugo_setgametitle(char *t)
{}


void hugo_clearfullscreen(void)
{
/* Clears everything on the screen, moving the cursor to the top-left
   corner of the screen */

	clear_to_color(screen_buffer, HUGO_PALETTE[current_back_color]);
	screen_needs_repainting = true;

	/* Must be set: */
	currentpos = 0;
	currentline = 1;
}

void hugo_clearwindow(void)
{
/* Clears the currently defined window, moving the cursor to the top-left
   corner of the window */

	rectfill(screen_buffer,
		physical_windowleft, physical_windowtop,
		physical_windowright, physical_windowbottom,
		HUGO_PALETTE[current_back_color]);
	screen_needs_repainting = true;

	/* Must be set: */
	currentpos = 0;
	currentline = 1;
}

void hugo_settextmode(void)
{
/* This function does whatever is necessary to set the system up for
   a standard text display */

	/* For the Allegro port, screen dimensions are in pixels
	   and printing is non-proportional.
	*/

	FIXEDCHARWIDTH = charwidth = FONT_WIDTH;
	FIXEDLINEHEIGHT = lineheight = FONT_HEIGHT;

	/* Must be set (as character or pixel coordinates, as
	   applicable):
	*/
	SCREENWIDTH = SCREEN_W;
	SCREENHEIGHT = SCREEN_H;

	/* Must be set: */
	hugo_settextwindow(1, 1,
		SCREENWIDTH/FIXEDCHARWIDTH, SCREENHEIGHT/FIXEDLINEHEIGHT);
}


void hugo_settextwindow(int left, int top, int right, int bottom)
{
/* Again, coords. are passed as text coordinates with the top corner (1, 1),
   and here must be converted to pixel coordinates.
*/
	/* Must be set (as character, not pixel, coordinates): */
	/* (Note that these are Allegro-specific parameters) */
	text_windowtop = top;        text_windowbottom = bottom;
	text_windowleft = left;      text_windowright = right;

	text_windowwidth = text_windowright-text_windowleft+1;
	text_windowheight = text_windowbottom-text_windowtop+1;


	/* Must be set (as pixel coordinates): */
	/* (Engine-required parameters) */
	physical_windowleft = (left-1)*FIXEDCHARWIDTH;
	physical_windowtop = (top-1)*FIXEDLINEHEIGHT;
	physical_windowright = right*FIXEDCHARWIDTH-1;
	physical_windowbottom = bottom*FIXEDLINEHEIGHT-1;
	physical_windowwidth = (right-left+1)*FIXEDCHARWIDTH;
	physical_windowheight = (bottom-top+1)*FIXEDLINEHEIGHT;

	ConstrainCursor();
}


void hugo_settextpos(int x, int y)
{
/* The top-left corner of the current active window is (1, 1).

   (In other words, if the screen is being windowed so that the top row
   of the window is row 4 on the screen, the (1, 1) refers to the 4th
   row on the screen, and (2, 1) refers to the 5th.)

   This function must also properly set currentline and currentpos (where
   currentline is a the current character line, and currentpos may be
   either in pixels or characters, depending on the measure being used).
*/
	/* Must be set: */
	currentline = y;
	currentpos = (x-1)*charwidth;   /* Note:  zero-based */

	current_text_col = text_windowleft-1+x;
	current_text_row = text_windowtop-1+y;
	ConstrainCursor();

	/* In case we're circumventing a linefeed */
	screen_needs_repainting = true;
}


void ConstrainCursor(void)
{
	/* ConstrainCursor() is needed since HEALLEG.C's cursor-
	   positioning doesn't automatically pay attention to any
	   currently-defined window.
	*/
	if (current_text_col > text_windowright) current_text_col = text_windowright;
	if (current_text_col < text_windowleft) current_text_col = text_windowleft;
	if (current_text_row > text_windowbottom) current_text_row = text_windowbottom;
	if (current_text_row < text_windowtop) current_text_row = text_windowtop;
}


void hugo_print(char *a)
{
        char last_was_newline = false;
	int i, len;

	len = strlen(a);

	for (i=0; i<len; i++)
	{
		last_was_newline = false;

		switch (a[i])
		{
			case '\n':
				++current_text_row;
                                last_was_newline = true;
				break;
			case '\r':
CarriageReturn:
				current_text_col = text_windowleft;
				break;
			default:
			{
				char c;
				
				c = a[i];
#if !defined (NO_LATIN1_CHARSET)
				if (!override_latin1_conversion)
				{
				    if ((unsigned char)a[i] >= LATIN1_START)
					c = latin1_to_ibm[(unsigned char)a[i]-LATIN1_START];
				}
#endif
				PrintGRchar((unsigned char)c);

				if (++current_text_col > text_windowright)
				{
					current_text_col = text_windowleft;
					++current_text_row;
				}
			}
		}

		if (current_text_row > text_windowbottom)
		{
			current_text_row = text_windowbottom;
			hugo_scrollwindowup();
		}
	}

        /* Because '\n' should imply '\r' */
	if (last_was_newline)
        {
                last_was_newline = false;
                goto CarriageReturn;
        }
}


/* PrintGRchar

	Prints a text character to the graphics screen using the
	default text font.
*/

void PrintGRchar(int c)
{
	int x, y, pcolor, col, row, font_row;

	for (row=0; row<FONT_HEIGHT; row++)
	{
		font_row = (int)_farpeekb(_dos_ds, text_font_addr+c*FONT_HEIGHT+row)&0xff;

		/* Double each pixel for boldface */
		if (current_text_font & BOLD_FONT)
			font_row |= font_row>>1;

		/* Add a bottom line for underline */
		if ((current_text_font & UNDERLINE_FONT) && row==FONT_HEIGHT-1)
			font_row = 0xffff;

		/* Shift the top half of the character for italic */
		if ((current_text_font & ITALIC_FONT) && row<FONT_HEIGHT/2)
			font_row = font_row>>1;

		for (col=0; col<FONT_WIDTH; col++)
		{
			/* If the font says a pixel should be plotted here */
			pcolor = HUGO_PALETTE[(font_row & 1<<(FONT_WIDTH-col-1))
				?current_text_color:current_back_color];

			x = (current_text_col-1)*FONT_WIDTH+col;
			y = (current_text_row-1)*FONT_HEIGHT+row;

			putpixel(screen_buffer, x, y, pcolor);
		}
	}
}


void hugo_scrollwindowup()
{
	/* Basically just copies a hunk of screen to an upward-
	   shifted position and fills in the screen behind/below it
	*/
	blit(screen_buffer, screen_buffer,
		(text_windowleft-1)*charwidth, (text_windowtop)*lineheight,
		(text_windowleft-1)*charwidth, (text_windowtop-1)*lineheight,
		text_windowwidth*charwidth, (text_windowheight-1)*lineheight);

	rectfill(screen_buffer,
		(text_windowleft-1)*charwidth, (text_windowbottom-1)*lineheight,
		text_windowright*charwidth-1, text_windowbottom*lineheight-1,
		HUGO_PALETTE[current_back_color]);

        UpdateScreen();
}


void hugo_font(int f)
{
	current_text_font = f;
}


/* Constants needed for coercion of colors in the event that we're forcing
   to grayscale (needed by hugo_settextcolor() and hugo_setbackcolor()):
*/
#define BW_DIM_BACK      8
#define BW_DIM_FORE     11
#define BW_BRIGHT_BACK  12
#define BW_BRIGHT_FORE  15

void hugo_settextcolor(int c)           /* foreground (print) color */
{
	current_text_color = hugo_color(c);

	/* If we're forcing a black-and-white palette, it is necessary
	   to tweak the colors--otherwise 0-15 are just varying shades
	   of black-->gray-->white
	*/
	if (color_space==JCS_GRAYSCALE && current_text_color!=0)
	{
		if (current_text_color>=8) current_text_color = BW_BRIGHT_FORE;
		else current_text_color = BW_DIM_FORE;
	}
}

void hugo_setbackcolor(int c)           /* background color */
{
	current_back_color = hugo_color(c);

	/* As with hugo_settextcolor(), it's necessary to tweak the
	   "color" if we're forcing to grayscale
	*/
	if (color_space==JCS_GRAYSCALE && current_back_color!=0)
	{
		if (current_back_color>=8) current_back_color = BW_BRIGHT_BACK;
		else current_back_color = BW_DIM_BACK;
	}
}

int hugo_color(int c)
{
	if (c==16)      c = DEF_FCOLOR;
	else if (c==17) c = DEF_BGCOLOR;
	else if (c==18) c = DEF_SLFCOLOR;
	else if (c==19) c = DEF_SLBGCOLOR;
	else if (c==20) c = hugo_color(fcolor);

/* Uncomment this block of code and change "c = ..." values if the system
   palette differs from the Hugo palette.

   If colors are unavailable on the system in question, it may suffice
   to have black, white, and brightwhite (i.e. boldface).  It is expected
   that colored text will be visible on any other-colored background.

	switch (c)
	{
		case HUGO_BLACK:	 c = 0;  break;
		case HUGO_BLUE:		 c = 1;  break;
		case HUGO_GREEN:	 c = 2;  break;
		case HUGO_CYAN:		 c = 3;  break;
		case HUGO_RED:		 c = 4;  break;
		case HUGO_MAGENTA:	 c = 5;  break;
		case HUGO_BROWN:	 c = 6;  break;
		case HUGO_WHITE:	 c = 7;  break;
		case HUGO_DARK_GRAY:	 c = 8;  break;
		case HUGO_LIGHT_BLUE:	 c = 9;  break;
		case HUGO_LIGHT_GREEN:	 c = 10; break;
		case HUGO_LIGHT_CYAN:	 c = 11; break;
		case HUGO_LIGHT_RED:	 c = 12; break;
		case HUGO_LIGHT_MAGENTA: c = 13; break;
		case HUGO_YELLOW:	 c = 14; break;
		case HUGO_BRIGHT_WHITE:	 c = 15; break;
*/
	return c;
}


/* CHARACTER AND TEXT MEASUREMENT */

int hugo_charwidth(char a)
{
	if (a==FORCED_SPACE)
		return FIXEDCHARWIDTH;	  /* same as ' ' */

	else if ((unsigned char)a >= ' ') /* alphanumeric characters */

		return FIXEDCHARWIDTH;	  /* because printing under MS-DOS
					     is non-proportional */
	return 0;
}

int hugo_textwidth(char *a)
{
	int i, slen, len = 0;

	slen = (int)strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else
			len += hugo_charwidth(a[i]);
	}

	return len;
}

int hugo_strlen(char *a)
{
	int i, slen, len = 0;

	slen = (int)strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else len++;
	}

	return len;
}


#if !defined (COMPILE_V25)
int hugo_hasvideo(void)
{
	return false;
}

int hugo_playvideo(HUGO_FILE infile, long reslength,
	char loop_flag, char background, int volume)
{
	fclose(infile);
	return true;
}

void hugo_stopvideo(void)
{}
#endif
