package iagecompiler;

import java.io.*;

/**
 *  Data class. Handles all file input and output for IAGE, as well
 *  as being the global data store for all the other classes.
 */
public class data {

  /** Used to determine if the current file the data class is reading has reached the end */
  boolean fileeof = false;
  /** Set when IAGE is reading an encrypted IAGE Format 2 file */
  boolean isencrypted = false;
  /** Turns on debug trace */
  public static final boolean debugging = true;
  public static mainframe theframe;

  /** Maximum number of user-defined flags available */
  public static int maxflags = 1000;
  /** IP port to listen on */
  public static int ipport = 1111;
  /** Default player name for new connections entering the server */
  public static String defaultnewplayername = "<new player>";
  /** Set when the engine is in a position to start and data loading is complete */
  public static boolean canstartenginenow = false;
  /** Current number of client connections being served */
  public static int numberofconnections = 0;
  /** Next unique generated player ID */
  public static int nextplayerID = 1;

  /** Adverb object collection */
  public static iagecollection oadverbs = new iagecollection();
  /** Verb object collection */
  public static iagecollection overbs = new iagecollection();
  /** Noun object collection */
  public static iagecollection onouns = new iagecollection();
  /** Game object */
  public static game ogame = new game();
  /** Location object collection */
  public static iagecollection olocations = new iagecollection();
  /** NPC object collection */
  public static iagecollection ocharacters = new iagecollection();
  /** Item object collection */
  public static iagecollection oitems = new iagecollection();
  /** Code modules */
  public static iagecollection omodules = new iagecollection();
  /** Code collection for runafterinput event */
  public static runafterinput orunafterinput = new runafterinput();
  /** Code collection for runbeforeinput event */
  public static runbeforeinput orunbeforeinput = new runbeforeinput();
  /** Global error object */
  //public static error oerror = new error();
  /** Player collection. Contains details for all connected players */
  public static iagecollection oplayers = new iagecollection();
  /** Message object collection */
  public static iagecollection omessages = new iagecollection();
  /** Flag array */
  //public static flag[] oflags = new flag[maxflags];

  public static String openfilename = "";

  /*** Bare constructor */
  public data() {
  }

  /**
   *  Creates a new data object by attempting to load the passed file.
   */
  public data(String sfile) {

	// Clear all data
	canstartenginenow = false;
	numberofconnections = 0;
	nextplayerID = 1;
	oadverbs = new iagecollection();
	overbs = new iagecollection();
	onouns = new iagecollection();
	ogame = new game();
	olocations = new iagecollection();
	ocharacters = new iagecollection();
	oitems = new iagecollection();
	omodules = new iagecollection();
	orunafterinput = new runafterinput();
	orunbeforeinput = new runbeforeinput();
	//oerror = new error();
	oplayers = new iagecollection();
	omessages = new iagecollection();
	//flag[] oflags = new flag[maxflags];

    // Set our marker to not end of file
    fileeof = false;

    // Open the file for read
    File fh = new File(sfile);

    // Bomb if the file isn't there
    if ( ! fh.exists()) {
      vdu.println("file - " + sfile + " does not exist.");
      return;
    }

    // Open a reader on it
    FileInputStream in;
    try {

      in = new FileInputStream(fh);

      vdu.println(fh.getName() + " opened.");
      
      // Store the file name.
      data.openfilename = sfile;

      // Read first line - should be file format version.
      String buffline = readline(in);

      if (buffline.indexOf("IAGE 2") != -1) {
        // Cool - it's file format version 2!
        // Check to see if it's encrypted
        vdu.println("IAGE File Format 2.0 detected.");
        buffline = readline(in);
        if (buffline.indexOf("1") != -1) {
          // yep - it's encrypted
          vdu.println("File is encrypted.");
          isencrypted = true;
        }
        else
        {
          vdu.println("File is not encrypted.");
          isencrypted = false;
        }

        // Start our pass through the file now, looking for special
        // header blocks
        // (GAME, LOCATIONS, ITEMS, CHARACTERS, MESSAGES, NOUNS, ADVERBS, VERBS,
        // RUNBEFOREINPUT, RUNAFTERINPUT)

        vdu.println("Loading data, please wait...");
        while ( ! fileeof ) {

          buffline = decrypt(readline(in));

          if (comparestring(buffline, "GAME") == true) {
            // Activate game reader
            vdu.print("Reading game properties...");
            loadgame(in);
          }
          if (comparestring(buffline, "LOCATIONS") == true) {
            // Activate locations reader
            vdu.print("Reading locations...");
            loadlocations(in);
          }
          if (comparestring(buffline, "ITEMS") == true) {
            // Activate items reader
            vdu.print("Reading objects...");
            loaditems(in);
          }
          if (comparestring(buffline, "CHARACTERS") == true) {
            // Activate characters reader
            vdu.print("Reading characters...");
            loadcharacters(in);
          }
          if (comparestring(buffline, "MESSAGES") == true) {
            // Activate messages reader
            vdu.print("Reading messages...");
            loadmessages(in);
          }
          if (comparestring(buffline, "NOUNS") == true) {
            // Activate nouns reader
            vdu.print("Reading dictionary (nouns)...");
            loadnouns(in);
          }
          if (comparestring(buffline, "ADVERBS") == true) {
            // Activate adverbs reader
            vdu.print("Reading dictionary (adverbs)...");
            loadadverbs(in);
          }
          if (comparestring(buffline, "VERBS") == true) {
            // Activate verbs reader
            vdu.print("Reading dictionary (verbs)...");
            loadverbs(in);
          }
          if (comparestring(buffline, "MODULES") == true) {
            // Activate module reader
            vdu.print("Reading code modules...");
            loadmodules(in);
          }
          if (comparestring(buffline, "RUNBEFOREINPUT") == true) {
            // Activate runbeforeinput reader
            vdu.print("Reading event code (1 of 2)...");
            loadrunbeforeinput(in);
          }
          if (comparestring(buffline, "RUNAFTERINPUT") == true) {
            // Activate runafterinput reader
            vdu.print("Reading event code (2 of 2)...");
            loadrunafterinput(in);
          }
        }
        vdu.println("Load complete.");
      }
      else
      {
        // What could it be? Who knows!
        vdu.println(sfile + " has an unrecognised file format.");
        return;
      }
    }
    catch(Exception e) {
      vdu.println("Error occurred: " + e.getMessage());
      e.printStackTrace();
      return;
    }
    // Since we must have had no errors to get here - everything
    // must be ok
    data.canstartenginenow = true;
  }

  /**
   *  Private routine used internally - reads a complete line of data
   *  by reading from a passed file handle one byte at a time until it
   *  finds a Chr(13) followed by a Chr(10).
   */
  private String readline(FileInputStream fs) {
    // Reads a complete line of data from a file, throwing away the
    // Chr(13) and Chr(10)s.
    StringBuffer sb = new StringBuffer("");
    Integer iob;
    int nb = -1;
    //byte[] by;

    while (true) {

      // Read next byte from stream
      try {
        nb = fs.read();
      }
      catch(Exception e) {
        vdu.println("Error reading from file - " + e.getMessage());
        return sb.toString();
      }

      // if it's a -1 or a 13, better quit and return
      if (nb == 13) {
        try {
          nb = fs.read(); // ditch the chr(10) as well
        }
        catch(Exception e) {
          vdu.println("Error reading from file - " + e.getMessage());
          return sb.toString();
        }
        return sb.toString();
      }

      if (nb == -1) {
        // no more data - return what we have
        fileeof = true;
        return sb.toString();
      }

      // Otherwise, append our jobbie into the string buffer
      iob = new Integer(nb);
      byte[] by = { iob.byteValue()};
      sb.append(new String(by));

    }

  }
  
  /*** Writes a line of text to the outputstream specified */
  public void writeline(FileOutputStream out, String s) {
  	
  	s = encrypt(s);
  	byte[] ba = s.getBytes();
  	try {
  		out.write(ba);
  		// Dump a Windows type carriage return - it's a bytestream, who cares!
  		// The read process looks for Windows type carriage returns anyway
  		byte[] bb = { 13, 10 };
  		out.write(bb);
  	}
  	catch (Exception e) {
  		e.printStackTrace();
  		vdu.println("Error writing to file - " + e.getMessage());
  	}
  	
  }
  
  /*** Writes a line of unencrypted text to the outputstream specified */
  public void writelineunencrypted(FileOutputStream out, String s) {
  	
  	byte[] ba = s.getBytes();
  	try {
  		out.write(ba);
  		// Dump a Windows type carriage return - it's a bytestream, who cares!
  		// The read process looks for Windows type carriage returns anyway
  		byte[] bb = { 13, 10 };
  		out.write(bb);
  	}
  	catch (Exception e) {
  		e.printStackTrace();
  		vdu.println("Error writing to file - " + e.getMessage());
  	}
  	
  }

  /*** Writes an iagecode object out as a property */
  public void writecode(FileOutputStream out, String propname, iagecode thecode) {
  	int i = 1;
  	while (i <= thecode.getCount()) {
  	  writeline(out, propname + " = " + thecode.get(i));
  	  i++;	
  	}
  }

  /** Loads game object */
  private void loadgame(FileInputStream in) {

    String buff;

    buff = decrypt(readline(in));
    while (buff.indexOf("END") == -1 && ! fileeof) {

      if (comparebuff(buff, "MaxItemsCanCarry") == true) {ogame.MaxItemsCanCarry = getpropvalue_long(buff);}
      if (comparebuff(buff, "MaxWeightCanCarry") == true) {ogame.MaxWeightCanCarry = getpropvalue_long(buff);}
      if (comparebuff(buff, "MaxSizeCanCarry") == true) {ogame.MaxSizeCanCarry = getpropvalue_long(buff);}
      if (comparebuff(buff, "StartingLocation") == true) {ogame.StartingLocation = getpropvalue_long(buff);}
      if (comparebuff(buff, "RepeatDescription") == true) {ogame.RepeatDescription = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "ShowAvailableExits") == true) {ogame.ShowAvailableExits = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "AllowPersist") == true) {ogame.AllowPersist = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "SinglePlayerGame") == true) {ogame.SinglePlayerGame = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "MaxUsers") == true) {ogame.MaxUsers = (int)getpropvalue_long(buff);}
      if (comparebuff(buff, "IDEPassword") == true) {ogame.IDEPassword = getpropvalue_string(buff);}
      if (comparebuff(buff, "Name") == true) {ogame.Name = getpropvalue_string(buff);}
      if (comparebuff(buff, "OverrideSecondaryNouns") == true) {ogame.OverrideSecondaryNouns = getpropvalue_string(buff);}
      if (comparebuff(buff, "MediaBase") == true) {ogame.MediaBase = getpropvalue_string(buff);}
      if (comparebuff(buff, "UsingIAGECombat") == true) {ogame.UsingIAGECombat = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "UsingIAGEMoney") == true) {ogame.UsingIAGEMoney = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "WideDisplay") == true) {ogame.WideDisplay = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "RealTimeNPCs") == true) {ogame.RealTimeNPCs = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "PlayersStayDead") == true) {ogame.PlayersStayDead = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "NPCsStayDead") == true) {ogame.NPCsStayDead = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "DefaultHitPoints") == true) {ogame.DefaultHitPoints = getpropvalue_long(buff);}
      if (comparebuff(buff, "DefaultDamage") == true) {ogame.DefaultDamage = getpropvalue_long(buff);}
      if (comparebuff(buff, "DefaultMoney") == true) {ogame.DefaultMoney = getpropvalue_long(buff);}      
      if (comparebuff(buff, "DefaultChanceOfHitting") == true) {ogame.DefaultChanceOfHitting = getpropvalue_long(buff);}      
      if (comparebuff(buff, "ChanceOfHittingIncrementForKill") == true) {ogame.ChanceOfHittingIncrementForKill = getpropvalue_long(buff);}      
      if (comparebuff(buff, "DamageIndicatorIncrementForKill") == true) {ogame.DamageIndicatorIncrementForKill = getpropvalue_long(buff);}      
      if (comparebuff(buff, "OnDisplayBanner") == true) {ogame.OnDisplayBanner.add(getuntrimmedvalue(buff));}
      if (comparebuff(buff, "OnAfterInputImmediate") == true) {ogame.OnAfterInputImmediate.add(getuntrimmedvalue(buff));}
      if (comparebuff(buff, "OnQuit") == true) {ogame.OnQuit.add(getuntrimmedvalue(buff));}
      if (comparebuff(buff, "OnStart") == true) {ogame.OnStart.add(getuntrimmedvalue(buff));}
      if (comparebuff(buff, "OnScore") == true) {ogame.OnScore.add(getuntrimmedvalue(buff));}
      if (comparebuff(buff, "OnInitialise") == true) {ogame.OnInitialise.add(getuntrimmedvalue(buff));}

      buff = decrypt(readline(in));
    }
    vdu.println("OK.");
  }

  /** Loads locations */
  private void loadlocations(FileInputStream in) {

    String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      location l = new location();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {l.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Name") == true) {l.Name = getpropvalue_string(buff);}
        if (comparebuff(buff, "ImagePath") == true) {l.ImagePath = getpropvalue_string(buff);}
        if (comparebuff(buff, "Description") == true) {l.Description = getpropvalue_string(buff);}
        if (comparebuff(buff, "IsDark") == true) {l.IsDark = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CustomProperties") == true) {l.CustomProperties = getpropvalue_string(buff);}
        if (comparebuff(buff, "N") == true) {l.N = getpropvalue_long(buff);}
        if (comparebuff(buff, "S") == true) {l.S = getpropvalue_long(buff);}
        if (comparebuff(buff, "E") == true) {l.E = getpropvalue_long(buff);}
        if (comparebuff(buff, "W") == true) {l.W = getpropvalue_long(buff);}
        if (comparebuff(buff, "U") == true) {l.U = getpropvalue_long(buff);}
        if (comparebuff(buff, "D") == true) {l.D = getpropvalue_long(buff);}
        if (comparebuff(buff, "NE") == true) {l.NE = getpropvalue_long(buff);}
        if (comparebuff(buff, "NW") == true) {l.NW = getpropvalue_long(buff);}
        if (comparebuff(buff, "SE") == true) {l.SE = getpropvalue_long(buff);}
        if (comparebuff(buff, "SW") == true) {l.SW = getpropvalue_long(buff);}
        if (comparebuff(buff, "OnInput") == true) {l.OnInput.add(getuntrimmedvalue(buff));}
        if (comparebuff(buff, "OnDisplay") == true) {l.OnDisplay.add(getuntrimmedvalue(buff));}
        buff = decrypt(readline(in));
      }
      // Add the location to the collection
      olocations.add(l);
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(olocations.getCount()) + " Locations).");
  }

  /** Loads items */
  private void loaditems(FileInputStream in) {
  	
  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      item i = new item();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {i.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Name") == true) {i.Name = getpropvalue_string(buff);}
        if (comparebuff(buff, "CurrentLocation") == true) {i.CurrentLocation = getpropvalue_long(buff);}
        if (comparebuff(buff, "Weight") == true) {i.Weight = getpropvalue_long(buff);}
        if (comparebuff(buff, "Description") == true) {i.Description = getpropvalue_string(buff);}
        if (comparebuff(buff, "CustomProperties") == true) {i.CustomProperties = getpropvalue_string(buff);}
        if (comparebuff(buff, "UserBooleans") == true) {i.UserBooleans = getpropvalue_string(buff);}
        if (comparebuff(buff, "NounID") == true) {i.NounID = getpropvalue_long(buff);}
        if (comparebuff(buff, "IsLightSource") == true) {i.IsLightSource = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsLit") == true) {i.IsLit = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsWorn") == true) {i.IsWorn = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsWeapon") == true) {i.IsWeapon = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "Transparent") == true) {i.Transparent = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "HasSurface") == true) {i.HasSurface = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CanBeLaidOn") == true) {i.CanBeLaidOn = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CanBeStoodOn") == true) {i.CanBeStoodOn = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CanBeSatOn") == true) {i.CanBeSatOn = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CanBeGotIn") == true) {i.CanBeGotIn = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "DefaultExamine") == true) {i.DefaultExamine = getpropvalue_string(buff);}
        if (comparebuff(buff, "IsContainer") == true) {i.IsContainer = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsEdible") == true) {i.IsEdible = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "EdibleHitPoints") == true) {i.EdibleHitPoints = getpropvalue_long(buff);}
        if (comparebuff(buff, "IsWearable") == true) {i.IsWearable = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsReadable") == true) {i.IsReadable = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "ReadableText") == true) {i.ReadableText = getpropvalue_string(buff);}
        if (comparebuff(buff, "Size") == true) {i.Size = getpropvalue_long(buff);}
        if (comparebuff(buff, "DamageIndicator") == true) {i.DamageIndicator = getpropvalue_long(buff);}
        if (comparebuff(buff, "Invisible") == true) {i.Invisible = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsFixed") == true) {i.IsFixed = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CanOpenClose") == true) {i.CanOpenClose = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "OpenCloseState") == true) {i.OpenCloseState = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "FixedMessage") == true) {i.FixedMessage = getpropvalue_string(buff);}
        if (comparebuff(buff, "IsSubItem") == true) {i.IsSubItem = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "SubItemOf") == true) {i.SubItemOf = getpropvalue_long(buff);}
        if (comparebuff(buff, "OnAction") == true) {i.OnAction.add(getuntrimmedvalue(buff));}
        buff = decrypt(readline(in));
      }
      // Add the item to the collection
      data.oitems.add(i);
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.oitems.getCount()) + " Objects).");

  }

  /** Loads NPCs */
  private void loadcharacters(FileInputStream in) {
  	
  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      character c = new character();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {c.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Name") == true) {c.Name = getpropvalue_string(buff);}
        if (comparebuff(buff, "CurrentLocation") == true) {c.CurrentLocation = getpropvalue_long(buff);}
        if (comparebuff(buff, "Description") == true) {c.Description = getpropvalue_string(buff);}
        if (comparebuff(buff, "CustomProperties") == true) {c.CustomProperties = getpropvalue_string(buff);}
        if (comparebuff(buff, "NounID") == true) {c.NounID = getpropvalue_long(buff);}
        if (comparebuff(buff, "TimerInterval") == true) {c.TimerInterval = getpropvalue_long(buff);}
        if (comparebuff(buff, "DefaultExamine") == true) {c.DefaultExamine = getpropvalue_string(buff);}
        if (comparebuff(buff, "HitPoints") == true) {c.HitPoints = getpropvalue_long(buff);}
        if (comparebuff(buff, "DamageIndicator") == true) {c.DamageIndicator = getpropvalue_long(buff);}
        if (comparebuff(buff, "AutoAttack") == true) {c.AutoAttack = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "AttackWhenAttacked") == true) {c.AttackWhenAttacked = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "Money") == true) {c.Money = getpropvalue_long(buff);}
        if (comparebuff(buff, "OnTimer") == true) {c.OnTimer.add(getuntrimmedvalue(buff));}
        if (comparebuff(buff, "OnTalk") == true) {c.OnTalk.add(getuntrimmedvalue(buff));}
        if (comparebuff(buff, "OnAction") == true) {c.OnAction.add(getuntrimmedvalue(buff));}
        buff = decrypt(readline(in));
      }
      // Add the npc to the collection
      data.ocharacters.add(c);
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.ocharacters.getCount()) + " NPCs).");

  }

  /** Loads verbs */
  private void loadverbs(FileInputStream in) {

  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      verb v = new verb();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {v.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Text") == true) {v.Text = getpropvalue_string(buff);}
        buff = decrypt(readline(in));
      }
      // Add the verb to the collection
      data.overbs.add(v);
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.overbs.getCount()) + " Verbs).");

  }

  /** Loads code modules */
  private void loadmodules(FileInputStream in) {

  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      codemodule co = new codemodule();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
      	if (comparebuff(buff, "ID") == true) {co.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "N") == true) {co.Name = getpropvalue_string(buff);}
        if (comparebuff(buff, "C") == true) {co.Code.add(getpropvalue_string(buff));}
        buff = decrypt(readline(in));
      }
      // Add the codemodule to the collection
      data.omodules.add(co);
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.omodules.getCount()) + " Code modules).");

  }

  /** Loads adverbs */
  private void loadadverbs(FileInputStream in) {

  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      adverb v = new adverb();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {v.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Text") == true) {v.Text = getpropvalue_string(buff);}
        buff = decrypt(readline(in));
      }
      // Add the adverb to the collection
      data.oadverbs.add(v);
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.oadverbs.getCount()) + " Adverbs).");

  }

  /** Loads nouns */
  private void loadnouns(FileInputStream in) {
  	
  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      noun v = new noun();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {v.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Text") == true) {v.Text = getpropvalue_string(buff);}
        buff = decrypt(readline(in));
      }
      // Add the verb to the collection
      data.onouns.add(v);
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.onouns.getCount()) + " Nouns).");

  }

  /** Loads run before input event */
  private void loadrunbeforeinput(FileInputStream in) {

  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "$END$") && !fileeof) {
      // Add the code to the collection
      data.orunbeforeinput.code.add(buff);
      buff = decrypt(readline(in));
      
    }
    vdu.println("OK (" + Integer.toString(data.orunbeforeinput.code.getCount()) + " lines).");

  }

  /** Loads run after input event */
  private void loadrunafterinput(FileInputStream in) {
  	
  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "$END$") && !fileeof) {
      // Add the code to the collection
      data.orunafterinput.code.add(buff);
      buff = decrypt(readline(in));
      
    }
    vdu.println("OK (" + Integer.toString(data.orunafterinput.code.getCount()) + " lines).");

  }

  /** Loads messages in from file */
  private void loadmessages(FileInputStream in) {
  	
  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      message v = new message();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {v.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Text") == true) {v.Text = getpropvalue_string(buff);}
        buff = decrypt(readline(in));
      }
      // Add the message to the collection
      data.omessages.add(v);
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.omessages.getCount()) + " Messages).");

  }

  /**
   *  Returns everything after the equals sign formatted as long.
   */
  private long getpropvalue_long(String sbuff) {
	// if first char is space remove it
	String p = getvalue(sbuff);
    return Long.parseLong(p);
  }

  /**
   *  Returns everything after the equals sign formatted as string
   */
  private String getpropvalue_string(String sbuff) {
    return getvalue(sbuff); // Get and return as string
  }

  /**
   *  Returns everything after the equals sign formatted as boolean.
   */
  private boolean getpropvalue_boolean(String sbuff) {
    String p = getvalue(sbuff); // Get as string
    if (p.equals("true") || p.equals("True")) return true; else return false;
  }

  /**
   *  Throws away everything upto and including the equals sign
   *  in the passed string.
   */
  private String getvalue(String sbuff) {
    // Returns a string containing the property value only
    String paramvalue = sbuff.substring(sbuff.indexOf("=") + 1, sbuff.length());
    
    // Return empty string if we just have a space
    if (paramvalue.startsWith(" ") && paramvalue.length() == 1) {
    	return "";
	}
    paramvalue = removeleadingspaces(paramvalue);
    paramvalue = removetrailingspaces(paramvalue);
    
    return paramvalue;
  }
  
  /**
   * Throws away everything upto and including the equals sign
   * in the passed string and DOES NOT TRIM it - this is vital
   * for code values.
   */
  private String getuntrimmedvalue(String sbuff) {
  	
  	// Returns a string containing the property value only
    String paramvalue = sbuff.substring(sbuff.indexOf("=") + 1, sbuff.length());
    
    // Return empty string if we just have a space
    if (paramvalue.startsWith(" ") && paramvalue.length() == 1) {
    	return "";
	}
    
    return paramvalue;
  }

  /**
   *  Throws everything away after and including the equals sign
   *  in string one. Then compares to string 2. If they match,
   *  returns a true value.
   */
  private boolean comparebuff(String buff, String compareto) {
    // Returns true if the specified property name forms the
    // first part of the string

    // Determine if we have an equals - ditch now if not
    if (buff.indexOf("=") == -1) {return false;}

    // Get the first part of the string
    String prop = buff.substring(0, buff.indexOf("=") -1);
    
    // ditch spaces
    prop = removeleadingspaces(prop);
    prop = removetrailingspaces(prop);
    
    // Compare
    if (prop.equals(compareto)) {
      return true;
    }
    else {
      return false;
    }
  }
  
  /**
   *  Returns true or false depending on whether the two strings
   *  passed in are the same. This routine will ditch whitespace
   *  from the first string before comparison.
   */
  private boolean comparestring(String buff, String compareto) {
    // Returns true if the strings are equal

    // ditch spaces
    buff = removeleadingspaces(buff);
    buff = removetrailingspaces(buff);

    // Compare
    if (buff.equals(compareto)) {
      return true;
    }
    else {
      return false;
    }
  }

  /*** Strips all the leading spaces from a string */
  public static String removeleadingspaces(String s) {
  	
  	while (s.startsWith(" ")) {
  		if (s.length() > 0) s = s.substring(1, s.length())	;
  	}
  	return s;
  	
  }

  /*** Strips all the trailing spaces from a string */
  public static String removetrailingspaces(String s) {
  	
  	while (s.endsWith(" ")) {
  		if (s.length() > 0) s = s.substring(0, s.length() - 1);
  	}
  	return s;
  	
  }

  /*** Removes all spaces, leading and trailing */
  public static String trimstring(String s) {
  	s = removeleadingspaces(s);
  	s = removetrailingspaces(s);
  	return s;
  }

  private String decrypt(String s) {
    // Decrypts a string if the isencrypted flag is set

    // If flag is not set, return the same string passed in
    if (!isencrypted) {return s;}
	
		final String Hex = "0123456789ABCDEF";
		byte[] ba = new byte[8192];
		int bpos = 0;
		int ihi = 0;
		int ilo = 0;
		String shi = "";
		String slo = "";
		String out = "";
		
		for (int i = 0; i < s.length(); i++) {
			
			// Read hi byte
			shi = s.substring(i, i + 1);
			// Calculate decimal value from position in Hex string
			ihi = Hex.indexOf(shi) * 16;
			
			// Next
			i++;
			
			// Read lo byte
			slo = s.substring(i, i + 1);
			// Calculate decimal value from position in Hex
			ilo = Hex.indexOf(slo);
			
			// Enter
			ba[bpos] = Byte.parseByte(Integer.toString(ihi + ilo));
			bpos++;
			
		}
		
		// Dump byte stream to a string
		out = new String(ba);
		
		// Truncate to correct length
		out = out.substring(0, bpos);
		
		return out;
	
  }
  
  public static String encrypthex(String s) {
  	    
  	    // Encrypts a string to hex

		final String Hex = "0123456789ABCDEF";
		byte[] ba = s.getBytes();
		double cb = 0;
		double hi = 0;
		double lo = 0;
		int ihi = 0;
		int ilo = 0;
		String shi = "";
		String slo = "";
		String out = "";
		String hexrep = "";
		
		for (int i = 0; i < ba.length; i++) {
			
			// Convert byte to double
			cb = (double) ba[i];
			
			// Calculate high byte
			hi = cb / 16;
			
			// Throw away all after decimal point
			int iihi = (int) hi;
			hi = (double) iihi;
			
			// Calculate lo byte
			lo = cb - (hi * 16);
			
			shi = Double.toString(hi);
			slo = Double.toString(lo);
			
			if (shi.endsWith(".0")) { shi = shi.substring(0, shi.length() - 2); }
			if (slo.endsWith(".0")) { slo = slo.substring(0, slo.length() - 2); }
			
			ihi = Integer.parseInt(shi);
			ilo = Integer.parseInt(slo);
			
			// Construct hex string
			hexrep = Hex.substring(ihi, ihi + 1);
			hexrep += Hex.substring(ilo, ilo + 1);
			
			// Append
			out += hexrep;
			
		}
		
		return out;

  }
  
  public static String encrypt(String s) {
  	    
  	    // Encrypts a string to ASCII + 2
  	    // Much faster and smaller than hex

		byte[] ba = s.getBytes();
		byte[] ob = new byte[ba.length];
		String out = "";
		
		for (int i = 0; i < ba.length; i++) {
			// Shift byte 2 places
			ob[i] = ba[i];
			ob[i] = (byte)(((int)ob[i]) + 2);
		}
		
		out = new String(ob);
		return out;

  }
  
  /*** Converts booleans to strings. 
     * It's not like I'm talking rocket science here and it's only a little routine, but
     * come on Sun, how hard would it have been to put a "toString" method in the Boolean
     * wrapper? */
  public static String booleanToString(boolean b) {
  	if (b) {
  		return "true";
  	}
  	else
  	{
  		return "false";
  	}
  }
  
  /*** Saves the data back to the file it came from. Effectively re-writes it. 
     * Set the encrypted flag if you want the file encrypted while saving.
     */
  public void SaveData(boolean encrypted) {
  		
  		int i = 1;
  		location l = null;
  		item im = null;
  		character cc = null;
  		verb v = null;
  		noun n = null;
  		adverb a = null;
  		message m = null;
  		codemodule mod = null;
  		isencrypted = encrypted;
  		
  		try {
	  		File fh = new File(openfilename);
	  		FileOutputStream out = new FileOutputStream(fh);
	  		
	  		vdu.println("Saving " + fh.getName());
	  		vdu.print("Writing header...");
	  		
	  		// Write header
	  		writelineunencrypted(out, "IAGE 2");
	  		if (encrypted) {
	  			writelineunencrypted(out, "2");	
	  		}
	  		else
	  		{
	  			writelineunencrypted(out, "0");
	  		}
	  		
	  		// Blank line with char 0 to fool hex editors
	  		byte[] ba = {0};
	  		writeline(out, new String(ba));
	  		vdu.println("OK");
	  		
	  		// Game properties
	  		
	  		vdu.print("Writing game properties...");
	  		writeline(out, "GAME");
	  		writeline(out, "");
	  		writeline(out, "MaxItemsCanCarry = " + Long.toString(ogame.MaxItemsCanCarry));
	  		writeline(out, "MaxWeightCanCarry = " + Long.toString(ogame.MaxWeightCanCarry));
	  		writeline(out, "MaxSizeCanCarry = " + Long.toString(ogame.MaxSizeCanCarry));
	  		writeline(out, "StartingLocation = " + Long.toString(ogame.StartingLocation));
	  		writeline(out, "RepeatDescription = " + booleanToString(ogame.RepeatDescription));
	  		writeline(out, "ShowAvailableExits = " + booleanToString(ogame.ShowAvailableExits));
	  		writeline(out, "AllowPersist = " + booleanToString(ogame.AllowPersist));
	  		writeline(out, "SinglePlayerGame = " + booleanToString(ogame.SinglePlayerGame));
	  		writeline(out, "MaxUsers = " + Integer.toString(ogame.MaxUsers));
	  		writeline(out, "IDEPassword = " + ogame.IDEPassword);
	  		writeline(out, "Name = " + ogame.Name);
	  		writeline(out, "OverrideSecondaryNouns = " + ogame.OverrideSecondaryNouns);
	  		writeline(out, "MediaBase = " + ogame.MediaBase);
	  		writeline(out, "UsingIAGECombat = " + booleanToString(ogame.UsingIAGECombat));
	  		writeline(out, "UsingIAGEMoney = " + booleanToString(ogame.UsingIAGEMoney));
	  		writeline(out, "WideDisplay = " + booleanToString(ogame.WideDisplay));
	  		writeline(out, "RealTimeNPCs = " + booleanToString(ogame.RealTimeNPCs));
	  		writeline(out, "PlayersStayDead = " + booleanToString(ogame.PlayersStayDead));
	  		writeline(out, "NPCsStayDead = " + booleanToString(ogame.NPCsStayDead));
	  		writeline(out, "DefaultHitPoints = " + Long.toString(ogame.DefaultHitPoints));
	  		writeline(out, "DefaultDamage = " + Long.toString(ogame.DefaultDamage));
	  		writeline(out, "DefaultMoney = " + Long.toString(ogame.DefaultMoney));
	  		writeline(out, "DefaultChanceOfHitting = " + Long.toString(ogame.DefaultChanceOfHitting));
	  		writeline(out, "ChanceOfHittingIncrementForKill = " + Long.toString(ogame.ChanceOfHittingIncrementForKill));
	  		writeline(out, "DamageIndicatorIncrementForKill = " + Long.toString(ogame.DamageIndicatorIncrementForKill));
	  		writecode(out, "OnDisplayBanner", ogame.OnDisplayBanner);
	  		writecode(out, "OnAfterInputImmediate", ogame.OnAfterInputImmediate);
	  		writecode(out, "OnQuit", ogame.OnQuit);
	  		writecode(out, "OnStart", ogame.OnStart);
	  		writecode(out, "OnScore", ogame.OnScore);
	  		writecode(out, "OnInitialise", ogame.OnInitialise);
	  		writeline(out, "");
	  		writeline(out, "END");
	  		vdu.println("OK");
	  		
	  		// Locations
	  		
	  		vdu.print("Writing locations...");
	  		writeline(out, "");
	  		writeline(out, "LOCATIONS");
	  		writeline(out, "");
	  		i = 1;
	  		while (i <= data.olocations.getCount()) {
	  			l = (location) data.olocations.get(i);
	  			writeline(out, "ID = " + Long.toString(l.ID));
	  			writeline(out, "Name = " + l.Name);
	  			writeline(out, "ImagePath = " + l.ImagePath);
	  			writeline(out, "Description = " + l.Description);
	  			writeline(out, "IsDark = " + booleanToString(l.IsDark));
	  			writeline(out, "CustomProperties = " + l.CustomProperties);
	  			writeline(out, "N = " + Long.toString(l.N));
	  			writeline(out, "S = " + Long.toString(l.S));
	  			writeline(out, "E = " + Long.toString(l.E));
	  			writeline(out, "W = " + Long.toString(l.W));
	  			writeline(out, "U = " + Long.toString(l.U));
	  			writeline(out, "D = " + Long.toString(l.D));
	  			writeline(out, "NE = " + Long.toString(l.NE));
	  			writeline(out, "NW = " + Long.toString(l.NW));
	  			writeline(out, "SE = " + Long.toString(l.SE));
	  			writeline(out, "SW = " + Long.toString(l.SW));
	  			writecode(out, "OnInput", l.OnInput);
	  			writecode(out, "OnDisplay", l.OnDisplay);
	  			if (i < data.olocations.getCount()) writeline(out, "BREAK");
	  			i++;	
	  		}
	  		writeline(out, "");
	  		writeline(out, "END");
	  		vdu.println("OK (" + Integer.toString(i - 1) + " locations)");
	  		
	  		// Items
	  		
	  		vdu.print("Writing items...");
	  		writeline(out, "");
	  		writeline(out, "ITEMS");
	  		writeline(out, "");
	  		i = 1;
	  		while (i <= data.oitems.getCount()) {
	  			im = (item) data.oitems.get(i);
	  			writeline(out, "ID = " + Long.toString(im.ID));
	  			writeline(out, "Name = " + im.Name);
	  			writeline(out, "CurrentLocation = " + Long.toString(im.CurrentLocation));
	  			writeline(out, "Weight = " + Long.toString(im.Weight));
	  			writeline(out, "Description = " + im.Description);
	  			writeline(out, "CustomProperties = " + im.CustomProperties);
	  			writeline(out, "UserBooleans = " + im.UserBooleans);
	  			writeline(out, "NounID = " + Long.toString(im.NounID));
	  			writeline(out, "IsLightSource = " + booleanToString(im.IsLightSource));
	  			writeline(out, "IsLit = " + booleanToString(im.IsLit));
	  			writeline(out, "IsWorn = " + booleanToString(im.IsWorn));
	  			writeline(out, "IsWeapon = " + booleanToString(im.IsWeapon));
	  			writeline(out, "Transparent = " + booleanToString(im.Transparent));
	  			writeline(out, "HasSurface = " + booleanToString(im.HasSurface));
	  			writeline(out, "CanBeLaidOn = " + booleanToString(im.CanBeLaidOn));
	  			writeline(out, "CanBeStoodOn = " + booleanToString(im.CanBeStoodOn));
	  			writeline(out, "CanBeSatOn = " + booleanToString(im.CanBeSatOn));
	  			writeline(out, "CanBeGotIn = " + booleanToString(im.CanBeGotIn));
	  			writeline(out, "DefaultExamine = " + im.DefaultExamine);
	  			writeline(out, "IsContainer = " + booleanToString(im.IsContainer));
	  			writeline(out, "IsEdible = " + booleanToString(im.IsEdible));
	  			writeline(out, "EdibleHitPoints = " + Long.toString(im.EdibleHitPoints));
	  			writeline(out, "IsWearable = " + booleanToString(im.IsWearable));
	  			writeline(out, "IsReadable = " + booleanToString(im.IsReadable));
	  			writeline(out, "ReadableText = " + im.ReadableText);
	  			writeline(out, "Size = " + Long.toString(im.Size));
	  			writeline(out, "DamageIndicator = " + Long.toString(im.DamageIndicator));
	  			writeline(out, "Invisible = " + booleanToString(im.Invisible));
	  			writeline(out, "IsFixed = " + booleanToString(im.IsFixed));
	  			writeline(out, "CanOpenClose = " + booleanToString(im.CanOpenClose));
	  			writeline(out, "OpenCloseState = " + booleanToString(im.OpenCloseState));
	  			writeline(out, "FixedMessage = " + im.FixedMessage);
	  			writeline(out, "IsSubItem = " + booleanToString(im.IsSubItem));
	  			writeline(out, "SubItemOf = " + Long.toString(im.SubItemOf));
	  			writecode(out, "OnAction", im.OnAction);
				if (i < data.oitems.getCount()) writeline(out, "BREAK");
	  			i++;	
	  		}
	  		writeline(out, "");
	  		writeline(out, "END");
	  		vdu.println("OK (" + Integer.toString(i - 1) + " items)");

	  		// Characters
	  		
	  		vdu.print("Writing NPCs...");
	  		writeline(out, "");
	  		writeline(out, "CHARACTERS");
	  		writeline(out, "");
	  		i = 1;
	  		while (i <= data.ocharacters.getCount()) {
	  			cc = (character) data.ocharacters.get(i);
	  			writeline(out, "ID = " + Long.toString(cc.ID));
	  			writeline(out, "Name = " + cc.Name);
	  			writeline(out, "CurrentLocation = " + Long.toString(cc.CurrentLocation));
	  			writeline(out, "Description = " + cc.Description);
	  			writeline(out, "NounID = " + Long.toString(cc.NounID));
	  			writeline(out, "TimerInterval = " + Long.toString(cc.TimerInterval));
	  			writeline(out, "DefaultExamine = " + cc.DefaultExamine);
	  			writeline(out, "HitPoints = " + Long.toString(cc.HitPoints));
	  			writeline(out, "DamageIndicator = " + Long.toString(cc.DamageIndicator));
	  			writeline(out, "AutoAttack = " + booleanToString(cc.AutoAttack));
	  			writeline(out, "AttackWhenAttacked = " + booleanToString(cc.AttackWhenAttacked));
	  			writeline(out, "Money = " + Long.toString(cc.Money));
	  			writecode(out, "OnTimer", cc.OnTimer);
	  			writecode(out, "OnTalk", cc.OnTalk);
	  			writecode(out, "OnAction", cc.OnAction);
				if (i < data.ocharacters.getCount()) writeline(out, "BREAK");
	  			i++;	
	  		}
	  		writeline(out, "");
	  		writeline(out, "END");
	  		vdu.println("OK (" + Integer.toString(i - 1) + " characters)");	  	
	  		
	  		// Verbs
	  		
	  		vdu.print("Writing Verbs...");
	  		writeline(out, "");
	  		writeline(out, "VERBS");
	  		writeline(out, "");
	  		i = 1;
	  		while (i <= data.overbs.getCount()) {
	  			v = (verb) data.overbs.get(i);
	  			writeline(out, "ID = " + Long.toString(v.ID));
	  			writeline(out, "Text = " + v.Text);
				if (i < data.overbs.getCount()) writeline(out, "BREAK");
	  			i++;	
	  		}
	  		writeline(out, "");
	  		writeline(out, "END");
	  		vdu.println("OK (" + Integer.toString(i - 1) + " verbs)");	  
	  		
	  		// nouns
	  		
	  		vdu.print("Writing nouns...");
	  		writeline(out, "");
	  		writeline(out, "NOUNS");
	  		writeline(out, "");
	  		i = 1;
	  		while (i <= data.onouns.getCount()) {
	  			n = (noun) data.onouns.get(i);
	  			writeline(out, "ID = " + Long.toString(n.ID));
	  			writeline(out, "Text = " + n.Text);
				if (i < data.onouns.getCount()) writeline(out, "BREAK");
	  			i++;	
	  		}
	  		writeline(out, "");
	  		writeline(out, "END");
	  		vdu.println("OK (" + Integer.toString(i - 1) + " nouns)");	  
	  		
	  		// adverbs
	  		
	  		vdu.print("Writing adverbs...");
	  		writeline(out, "");
	  		writeline(out, "ADVERBS");
	  		writeline(out, "");
	  		i = 1;
	  		while (i <= data.oadverbs.getCount()) {
	  			a = (adverb) data.oadverbs.get(i);
	  			writeline(out, "ID = " + Long.toString(a.ID));
	  			writeline(out, "Text = " + a.Text);
				if (i < data.oadverbs.getCount()) writeline(out, "BREAK");
	  			i++;	
	  		}
	  		writeline(out, "");
	  		writeline(out, "END");
	  		vdu.println("OK (" + Integer.toString(i - 1) + " adverbs)");	  

	  		// messages
	  		
	  		vdu.print("Writing messages...");
	  		writeline(out, "");
	  		writeline(out, "MESSAGES");
	  		writeline(out, "");
	  		i = 1;
	  		while (i <= data.omessages.getCount()) {
	  			m = (message) data.omessages.get(i);
	  			writeline(out, "ID = " + Long.toString(m.ID));
	  			writeline(out, "Text = " + m.Text);
				if (i < data.omessages.getCount()) writeline(out, "BREAK");
	  			i++;	
	  		}
	  		writeline(out, "");
	  		writeline(out, "END");
	  		vdu.println("OK (" + Integer.toString(i - 1) + " messages)");	  

	  		// Modules
	  		
	  		vdu.print("Writing code modules...");
	  		writeline(out, "");
	  		writeline(out, "MODULES");
	  		writeline(out, "");
	  		i = 1;
	  		while (i <= data.omodules.getCount()) {
	  			mod = (codemodule) data.omodules.get(i);
	  			writeline(out, "ID = " + mod.ID);
	  			writeline(out, "N = " + mod.Name);
	  			writecode(out, "C", mod.Code);
				if (i < data.omodules.getCount()) writeline(out, "BREAK");
	  			i++;	
	  		}
	  		writeline(out, "");
	  		writeline(out, "END");
	  		vdu.println("OK (" + Integer.toString(i - 1) + " code modules)");	
	  		
	  		// Run before input
	  		vdu.print("Writing event code (1 of 2)...");
	  		writeline(out, "");
	  		writeline(out, "RUNBEFOREINPUT");
	  		i = 1;
	  		while (i <= data.orunbeforeinput.code.getCount()) {
	  			writeline(out, data.orunbeforeinput.code.get(i));
	  			i++;	
	  		}
	  		writeline(out, "$END$");
	  		vdu.println("OK (" + Integer.toString(i - 1) + " lines)");
	  		
	  		// Run after input
	  		vdu.print("Writing event code (2 of 2)...");
	  		writeline(out, "");
	  		writeline(out, "RUNAFTERINPUT");
	  		i = 1;
	  		while (i <= data.orunafterinput.code.getCount()) {
	  			writeline(out, data.orunafterinput.code.get(i));
	  			i++;	
	  		}
	  		writeline(out, "$END$");
	  		vdu.println("OK (" + Integer.toString(i) + " lines)");
	  		
	  		vdu.println("Save complete.");

	  	}
	  	catch(Exception e) {
	  		e.printStackTrace();
	  		vdu.println("Error opening output file - " + e.getMessage());
	  	}
  }
  
  public static void cleardata() {
  	// Clear all data
	data.oadverbs = new iagecollection();
	data.overbs = new iagecollection();
	data.onouns = new iagecollection();
	data.ogame = new game();
	data.olocations = new iagecollection();
	data.ocharacters = new iagecollection();
	data.oitems = new iagecollection();
	data.omodules = new iagecollection();
	data.orunafterinput = new runafterinput();
	data.orunbeforeinput = new runbeforeinput();
	//oerror = new error();
	data.oplayers = new iagecollection();
	data.omessages = new iagecollection();
	}
  
}