package iageserver;

import java.io.*;
import java.net.*;
import iageconnector;

/**
 *  Data class. Handles all file input and output for IAGE, as well
 *  as being the global data store for all the other classes.
 */
public class data extends Thread {

  /** Used to determine if the current file the data class is reading has reached the end */
  static boolean fileeof = false;
  /** Set when IAGE is reading an encrypted IAGE Format 2 file */
  static boolean isencrypted = false;
  /** Encryption type used in file - 0 = none, 1 = hex, 2 = ascii + 2*/
  static int encrypttype = 0;
  /** Turns on debug trace */
  public static final boolean debugging = false;

  /** Maximum number of user-defined flags available */
  public static int maxflags = 1000;
  /** IP port to listen on */
  public static int ipport = 1111;
  /** Server package version number */
  public static String versionno = "Version 020416";
  /** Display copyright and version message */
  public static String internalversion = "Internet Adventure Game Engine ServPlet" + versionno + "\nWritten by R.Rawson-Tetley\nCopyright(c)2000-2002, R.Rawson-Tetley";
  /** Default player name for new connections entering the server */
  public static String defaultnewplayername = "<new player>";
  /** Set when the engine is in a position to start and data loading is complete */
  public static boolean canstartenginenow = false;
  /** Current number of client connections being served */
  public static int numberofconnections = 0;
  /** Next unique generated player ID */
  public static int nextplayerID = 1;
  /** Path of the loaded game file */
  public static String gamepath = "";
  /** Adverb object collection */
  public static iagecollection oadverbs = new iagecollection();
  /** Verb object collection */
  public static iagecollection overbs = new iagecollection();
  /** Noun object collection */
  public static iagecollection onouns = new iagecollection();
  /** Game object */
  public static game ogame = new game();
  /** Location object collection */
  public static iagecollection olocations = new iagecollection();
  /** NPC object collection */
  public static iagecollection ocharacters = new iagecollection();
  /** Item object collection */
  public static iagecollection oitems = new iagecollection();
  /** Code modules collection */
  public static iagecollection omodules = new iagecollection();
  /** Code collection for runafterinput event */
  public static runafterinput orunafterinput = new runafterinput();
  /** Code collection for runbeforeinput event */
  public static runbeforeinput orunbeforeinput = new runbeforeinput();
  /** Global error object */
  public static error oerror = new error();
  /** Global NPC timer object */
  public static npctimer onpctimer = null;
  /** Player collection. Contains details for all connected players */
  public static iagecollection oplayers = new iagecollection();
  /** Message object collection */
  public static iagecollection omessages = new iagecollection();
  /** Flag array */
  public static flag[] oflags = new flag[maxflags];
  /** URL of the file to load */
  public String absoluteurl = "";
  /** Reference to the IAGE connector object to call back when 
      we are ready to start the server running */
  public iageconnector theconnector = null;
  /** The progress character indicator */
  private int prg = 0;

  public void run() {
  	this.loaddata();
  }

  /**
   *  Creates a new data object by attempting to load the passed file.
   */
  public void loaddata() {

	// Clear all data
	canstartenginenow = false;
	numberofconnections = 0;
	nextplayerID = 1;
	oadverbs = new iagecollection();
	overbs = new iagecollection();
	onouns = new iagecollection();
	ogame = new game();
	olocations = new iagecollection();
	ocharacters = new iagecollection();
	oitems = new iagecollection();
	omodules = new iagecollection();
	orunafterinput = new runafterinput();
	orunbeforeinput = new runbeforeinput();
	oerror = new error();
	oplayers = new iagecollection();
	omessages = new iagecollection();
	oflags = new flag[maxflags];
	
	// Yield to let the screen redraw
	this.yield();

	// Reset all our flags
	int flagcnt = 1;
	while (flagcnt < maxflags) {
		flag f = new flag();
		oflags[flagcnt] = f;
		flagcnt++;	
	}

	this.yield();

    // Set our marker to not end of file
    fileeof = false;

	// Grab URL
	URL ourl;
	try {
		ourl = new URL(absoluteurl);
	}
	catch (IOException e) {
		vdu.errprintln("URL - " + absoluteurl + " cannot be resolved - " + e.getMessage());
		return;
	}

	// Open a connection on it
	InputStreamReader in;
	try {
		in = new InputStreamReader(ourl.openStream());
	}
	catch (Exception e) {
		vdu.errprintln("Error opening reader on " + absoluteurl + " - " + e.getMessage());
		return;
	}

	this.yield();

    try {

      // Read first line - should be file format version.
      String buffline = readline(in);

      if (buffline.indexOf("IAGE 2") != -1) {
        // Cool - it's file format version 2!
        // Check to see if it's encrypted
        vdu.consoleconnector.location.setText("IAGE File Format 2.0 detected.");
        buffline = readline(in);
        if (buffline.indexOf("0") == -1) {
          // yep - it's encrypted
          vdu.consoleconnector.location.setText("File is encrypted.");
          isencrypted = true;
          // Work out the encryption type and store it
		  if (buffline.indexOf("1") != -1) {
			  encrypttype = 1; // hex
		  }
		  if (buffline.indexOf("2") != -1) {
		  	  encrypttype = 2; // ascii shift + 2
		  }
        }
        else
        {
          vdu.consoleconnector.location.setText("File is not encrypted.");
          isencrypted = false;
        }

        // Start our pass through the file now, looking for special
        // header blocks
        // (GAME, LOCATIONS, ITEMS, CHARACTERS, MESSAGES, NOUNS, ADVERBS, VERBS,
        // RUNBEFOREINPUT, RUNAFTERINPUT)

        vdu.consoleconnector.location.setText("Loading data, please wait...");
        while ( ! fileeof ) {

          buffline = decrypt(readline(in));

		  this.yield();

          if (comparestring(buffline, "GAME") == true) {
            // Activate game reader
            vdu.consoleconnector.location.setText("Reading game properties...");
            loadgame(in);
          }
          
          if (comparestring(buffline, "LOCATIONS") == true) {
            // Activate locations reader
            vdu.consoleconnector.location.setText("Reading locations...");
            loadlocations(in);
          }
          
          if (comparestring(buffline, "ITEMS") == true) {
            // Activate items reader
            vdu.consoleconnector.location.setText("Reading objects...");
            loaditems(in);
          }
          
          if (comparestring(buffline, "CHARACTERS") == true) {
            // Activate characters reader
            vdu.consoleconnector.location.setText("Reading characters...");
            loadcharacters(in);
          }
          
          if (comparestring(buffline, "MESSAGES") == true) {
            // Activate messages reader
            vdu.consoleconnector.location.setText("Reading messages...");
            loadmessages(in);
          }
          
          if (comparestring(buffline, "NOUNS") == true) {
            // Activate nouns reader
            vdu.consoleconnector.location.setText("Reading dictionary (nouns)...");
            loadnouns(in);
          }
          
          if (comparestring(buffline, "ADVERBS") == true) {
            // Activate adverbs reader
            vdu.consoleconnector.location.setText("Reading dictionary (adverbs)...");
            loadadverbs(in);
          }
          
          if (comparestring(buffline, "VERBS") == true) {
            // Activate verbs reader
            vdu.consoleconnector.location.setText("Reading dictionary (verbs)...");
            loadverbs(in);
          }
          
          if (comparestring(buffline, "MODULES") == true) {
            // Activate module reader
            vdu.print("Reading code modules...");
            loadmodules(in);
          }
          
          if (comparestring(buffline, "RUNBEFOREINPUT") == true) {
            // Activate runbeforeinput reader
            vdu.consoleconnector.location.setText("Reading event code (1 of 2)...");
            loadrunbeforeinput(in);
          }
          
          if (comparestring(buffline, "RUNAFTERINPUT") == true) {
            // Activate runafterinput reader
            vdu.consoleconnector.location.setText("Reading event code (2 of 2)...");
            loadrunafterinput(in);
          }
          
        }
        vdu.println("Load complete.");
        
        // Start everything going
        theconnector.Begin();
      }
      else
      {
        // What could it be? Who knows!
        vdu.errprintln(absoluteurl + " has an unrecognised file format.");
        return;
      }
    }
    catch(Exception e) {
      vdu.errprintln("Error occurred: " + e.getMessage());
      e.printStackTrace();
      return;
    }
    // Since we must have had no errors to get here - everything
    // must be ok
    data.canstartenginenow = true;
  }

  /**
   *  Reads a complete line of data
   *  by reading from a passed file handle one byte at a time until it
   *  finds a Chr(13) followed by a Chr(10).
   */
  public static String readline(InputStreamReader fs) {
    // Reads a complete line of data from a file, throwing away the
    // Chr(13) and Chr(10)s.
    StringBuffer sb = new StringBuffer("");
    Integer iob;
    int nb = -1;
    boolean readabyte = false;
    //byte[] by;

    while (true) {

      // Read next byte from stream
      try {
        nb = fs.read();
      }
      catch(Exception e) {
        vdu.println("Error reading from file - " + e.getMessage());
        return sb.toString();
      }

	  // if we have a 10 and nothing else so far,
	  // simply ignore it and carry on
	  if (nb == 10 && !readabyte) {
	  	  try {
	        nb = fs.read();
	      }
	      catch(Exception e) {
	        vdu.println("Error reading from file - " + e.getMessage());
	        return sb.toString();
	      }	
	  }

      // if it's a a 13 or 10, better quit and return
      if (nb == 13 || nb == 10) {
        try {
          nb = fs.read(); // ditch the chr(10) as well
        }
        catch(Exception e) {
          vdu.println("Error reading from file - " + e.getMessage());
          return sb.toString();
        }
        return sb.toString();
      }

      if (nb == -1) {
        // no more data - return what we have
        fileeof = true;
        return sb.toString();
      }

      // Otherwise, append our jobbie into the string buffer
      readabyte = true;
      iob = new Integer(nb);
      byte[] by = { iob.byteValue()};
      sb.append(new String(by));

    }

  }

  /** Loads game object */
  private void loadgame(InputStreamReader in) {

    String buff;

    buff = decrypt(readline(in));
    while (buff.indexOf("END") == -1 && ! fileeof) {

      if (comparebuff(buff, "MaxItemsCanCarry") == true) {ogame.MaxItemsCanCarry = getpropvalue_long(buff);}
      if (comparebuff(buff, "MaxWeightCanCarry") == true) {ogame.MaxWeightCanCarry = getpropvalue_long(buff);}
      if (comparebuff(buff, "MaxSizeCanCarry") == true) {ogame.MaxSizeCanCarry = getpropvalue_long(buff);}
      if (comparebuff(buff, "StartingLocation") == true) {ogame.StartingLocation = getpropvalue_long(buff);}
      if (comparebuff(buff, "RepeatDescription") == true) {ogame.RepeatDescription = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "ShowAvailableExits") == true) {ogame.ShowAvailableExits = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "AllowPersist") == true) {ogame.AllowPersist = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "SinglePlayerGame") == true) {ogame.SinglePlayerGame = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "MaxUsers") == true) {ogame.MaxUsers = (int) getpropvalue_long(buff);}
      if (comparebuff(buff, "IDEPassword") == true) {ogame.IDEPassword = getpropvalue_string(buff);}
      if (comparebuff(buff, "Name") == true) {ogame.Name = getpropvalue_string(buff);}
      if (comparebuff(buff, "OverrideSecondaryNouns") == true) {ogame.OverrideSecondaryNouns = getpropvalue_string(buff);}
      if (comparebuff(buff, "MediaBase") == true) {ogame.MediaBase = getpropvalue_string(buff);}
      if (comparebuff(buff, "UsingIAGECombat") == true) {ogame.UsingIAGECombat = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "UsingIAGEMoney") == true) {ogame.UsingIAGEMoney = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "WideDisplay") == true) {game.WideDisplay = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "RealTimeNPCs") == true) {ogame.RealTimeNPCs = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "PlayersStayDead") == true) {ogame.PlayersStayDead = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "NPCsStayDead") == true) {ogame.NPCsStayDead = getpropvalue_boolean(buff);}
      if (comparebuff(buff, "DefaultHitPoints") == true) {ogame.DefaultHitPoints = getpropvalue_long(buff);}
      if (comparebuff(buff, "DefaultDamage") == true) {ogame.DefaultDamage = getpropvalue_long(buff);}
      if (comparebuff(buff, "DefaultMoney") == true) {ogame.DefaultMoney = getpropvalue_long(buff);}      
      if (comparebuff(buff, "DefaultChanceOfHitting") == true) {ogame.DefaultChanceOfHitting = getpropvalue_long(buff);}      
      if (comparebuff(buff, "ChanceOfHittingIncrementForKill") == true) {ogame.ChanceOfHittingIncrementForKill = getpropvalue_long(buff);}      
      if (comparebuff(buff, "DamageIndicatorIncrementForKill") == true) {ogame.DamageIndicatorIncrementForKill = getpropvalue_long(buff);}      
      if (comparebuff(buff, "OnAfterInputImmediate") == true) {ogame.OnAfterInputImmediate.add(getpropvalue_string(buff));}
      if (comparebuff(buff, "OnDisplayBanner") == true) {ogame.OnDisplayBanner.add(getpropvalue_string(buff));}
      if (comparebuff(buff, "OnQuit") == true) {ogame.OnQuit.add(getpropvalue_string(buff));}
      if (comparebuff(buff, "OnStart") == true) {ogame.OnStart.add(getpropvalue_string(buff));}
      if (comparebuff(buff, "OnScore") == true) {ogame.OnScore.add(getpropvalue_string(buff));}
      if (comparebuff(buff, "OnInitialise") == true) {ogame.OnInitialise.add(getpropvalue_string(buff));}

      buff = decrypt(readline(in));
    }
    vdu.println("OK.");
  }

  /** Loads locations */
  private void loadlocations(InputStreamReader in) {

    String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      location l = new location();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {l.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Name") == true) {l.Name = getpropvalue_string(buff);}
        if (comparebuff(buff, "ImagePath") == true) {l.ImagePath = getpropvalue_string(buff);}
        if (comparebuff(buff, "Description") == true) {l.Description = getpropvalue_string(buff);}
        if (comparebuff(buff, "IsDark") == true) {l.IsDark = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CustomProperties") == true) {l.CustomProperties = getpropvalue_string(buff);}
        if (comparebuff(buff, "N") == true) {l.N = getpropvalue_long(buff);}
        if (comparebuff(buff, "S") == true) {l.S = getpropvalue_long(buff);}
        if (comparebuff(buff, "E") == true) {l.E = getpropvalue_long(buff);}
        if (comparebuff(buff, "W") == true) {l.W = getpropvalue_long(buff);}
        if (comparebuff(buff, "U") == true) {l.U = getpropvalue_long(buff);}
        if (comparebuff(buff, "D") == true) {l.D = getpropvalue_long(buff);}
        if (comparebuff(buff, "NE") == true) {l.NE = getpropvalue_long(buff);}
        if (comparebuff(buff, "NW") == true) {l.NW = getpropvalue_long(buff);}
        if (comparebuff(buff, "SE") == true) {l.SE = getpropvalue_long(buff);}
        if (comparebuff(buff, "SW") == true) {l.SW = getpropvalue_long(buff);}
        if (comparebuff(buff, "OnInput") == true) {l.OnInput.add(getpropvalue_string(buff));}
        if (comparebuff(buff, "OnDisplay") == true) {l.OnDisplay.add(getpropvalue_string(buff));}
        buff = decrypt(readline(in));
      }
      // Add the location to the collection
      olocations.add(l);
      
      vdu.consoleconnector.location.setText("Reading locations... " + getNextProgress());
      
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(olocations.getCount()) + " Locations).");
  }

  /** Loads items */
  private void loaditems(InputStreamReader in) {
  	
  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      item i = new item();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {i.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Name") == true) {i.Name = getpropvalue_string(buff);}
        if (comparebuff(buff, "CurrentLocation") == true) {i.CurrentLocation = getpropvalue_long(buff);}
        if (comparebuff(buff, "Weight") == true) {i.Weight = getpropvalue_long(buff);}
        if (comparebuff(buff, "Description") == true) {i.Description = getpropvalue_string(buff);}
        if (comparebuff(buff, "CustomProperties") == true) {i.CustomProperties = getpropvalue_string(buff);}
        if (comparebuff(buff, "UserBooleans") == true) {i.UserBooleans = getpropvalue_string(buff);}
        if (comparebuff(buff, "NounID") == true) {i.NounID = getpropvalue_long(buff);}
        if (comparebuff(buff, "IsLightSource") == true) {i.IsLightSource = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsLit") == true) {i.IsLit = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsWorn") == true) {i.IsWorn = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsWeapon") == true) {i.IsWeapon = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "Transparent") == true) {i.Transparent = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "HasSurface") == true) {i.HasSurface = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CanBeLaidOn") == true) {i.CanBeLaidOn = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CanBeStoodOn") == true) {i.CanBeStoodOn = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CanBeSatOn") == true) {i.CanBeSatOn = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CanBeGotIn") == true) {i.CanBeGotIn = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "DefaultExamine") == true) {i.DefaultExamine = getpropvalue_string(buff);}
        if (comparebuff(buff, "IsContainer") == true) {i.IsContainer = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsEdible") == true) {i.IsEdible = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "EdibleHitPoints") == true) {i.EdibleHitPoints = getpropvalue_long(buff);}
        if (comparebuff(buff, "IsWearable") == true) {i.IsWearable = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsReadable") == true) {i.IsReadable = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "ReadableText") == true) {i.ReadableText = getpropvalue_string(buff);}
        if (comparebuff(buff, "Size") == true) {i.Size = getpropvalue_long(buff);}
        if (comparebuff(buff, "DamageIndicator") == true) {i.DamageIndicator = getpropvalue_long(buff);}
        if (comparebuff(buff, "Invisible") == true) {i.Invisible = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "IsFixed") == true) {i.IsFixed = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "CanOpenClose") == true) {i.CanOpenClose = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "OpenCloseState") == true) {i.OpenCloseState = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "FixedMessage") == true) {i.FixedMessage = getpropvalue_string(buff);}
        if (comparebuff(buff, "IsSubItem") == true) {i.IsSubItem = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "SubItemOf") == true) {i.SubItemOf = getpropvalue_long(buff);}
        if (comparebuff(buff, "OnAction") == true) {i.OnAction.add(getpropvalue_string(buff));}
        buff = decrypt(readline(in));
      }
      // Add the item to the collection
      data.oitems.add(i);
      
      vdu.consoleconnector.location.setText("Reading objects... " + getNextProgress());
      
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.oitems.getCount()) + " Objects).");

  }

  /** Loads NPCs */
  private void loadcharacters(InputStreamReader in) {
  	
  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      character c = new character();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {c.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Name") == true) {c.Name = getpropvalue_string(buff);}
        if (comparebuff(buff, "CurrentLocation") == true) {c.CurrentLocation = getpropvalue_long(buff);}
        if (comparebuff(buff, "Description") == true) {c.Description = getpropvalue_string(buff);}
        if (comparebuff(buff, "CustomProperties") == true) {c.CustomProperties = getpropvalue_string(buff);}
        if (comparebuff(buff, "NounID") == true) {c.NounID = getpropvalue_long(buff);}
        if (comparebuff(buff, "TimerInterval") == true) {c.TimerInterval = getpropvalue_long(buff);}
        if (comparebuff(buff, "DefaultExamine") == true) {c.DefaultExamine = getpropvalue_string(buff);}
        if (comparebuff(buff, "HitPoints") == true) {c.HitPoints = getpropvalue_long(buff);}
        if (comparebuff(buff, "DamageIndicator") == true) {c.DamageIndicator = getpropvalue_long(buff);}
        if (comparebuff(buff, "AutoAttack") == true) {c.AutoAttack = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "AttackWhenAttacked") == true) {c.AttackWhenAttacked = getpropvalue_boolean(buff);}
        if (comparebuff(buff, "Money") == true) {c.Money = getpropvalue_long(buff);}
        if (comparebuff(buff, "OnTimer") == true) {c.OnTimer.add(getpropvalue_string(buff));}
        if (comparebuff(buff, "OnTalk") == true) {c.OnTalk.add(getpropvalue_string(buff));}
        if (comparebuff(buff, "OnAction") == true) {c.OnAction.add(getpropvalue_string(buff));}
        buff = decrypt(readline(in));
      }
      // Add the npc to the collection
      data.ocharacters.add(c);
      
      vdu.consoleconnector.location.setText("Reading characters... " + getNextProgress());
      
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.ocharacters.getCount()) + " NPCs).");

  }

  /** Loads verbs */
  private void loadverbs(InputStreamReader in) {

  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      verb v = new verb();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {v.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Text") == true) {v.Text = getpropvalue_string(buff);}
        buff = decrypt(readline(in));
      }
      // Add the verb to the collection
      data.overbs.add(v);
      
      vdu.consoleconnector.location.setText("Reading dictionary (verbs)... " + getNextProgress());
      
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.overbs.getCount()) + " Verbs).");

  }

  /** Loads adverbs */
  private void loadadverbs(InputStreamReader in) {

  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      adverb v = new adverb();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {v.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Text") == true) {v.Text = getpropvalue_string(buff);}
        buff = decrypt(readline(in));
      }
      // Add the adverb to the collection
      data.oadverbs.add(v);
      
      vdu.consoleconnector.location.setText("Reading dictionary (adverbs)... " + getNextProgress());
      
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.oadverbs.getCount()) + " Adverbs).");

  }
  
    /** Loads code modules */
  private void loadmodules(InputStreamReader in) {

  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      codemodule co = new codemodule();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
      	if (comparebuff(buff, "ID") == true) {co.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "N") == true) {co.Name = getpropvalue_string(buff);}
        if (comparebuff(buff, "C") == true) {co.Code.add(getpropvalue_string(buff));}
        buff = decrypt(readline(in));
      }
      // Add the codemodule to the collection
      data.omodules.add(co);
      
      vdu.consoleconnector.location.setText("Reading code modules... " + getNextProgress());
      
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.omodules.getCount()) + " Code modules).");

  }

  /** Loads nouns */
  private void loadnouns(InputStreamReader in) {
  	
  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      noun v = new noun();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {v.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Text") == true) {v.Text = getpropvalue_string(buff);}
        buff = decrypt(readline(in));
      }
      // Add the verb to the collection
      data.onouns.add(v);
      
      vdu.consoleconnector.location.setText("Reading dictionary (nouns)... " + getNextProgress());
      
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.onouns.getCount()) + " Nouns).");

  }

  /** Loads run before input event */
  private void loadrunbeforeinput(InputStreamReader in) {

  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "$END$") && !fileeof) {
      // Add the code to the collection
      data.orunbeforeinput.code.add(buff);
      buff = decrypt(readline(in));
      
    }
    vdu.println("OK (" + Integer.toString(data.orunbeforeinput.code.getCount()) + " lines).");

  }

  /** Loads run after input event */
  private void loadrunafterinput(InputStreamReader in) {
  	
  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "$END$") && !fileeof) {
      // Add the code to the collection
      data.orunafterinput.code.add(buff);
      buff = decrypt(readline(in));
      
    }
    vdu.println("OK (" + Integer.toString(data.orunafterinput.code.getCount()) + " lines).");

  }

  /** Loads messages in from file */
  private void loadmessages(InputStreamReader in) {
  	
  	String buff;
    buff = decrypt(readline(in));
    while (!comparestring(buff, "END") && !fileeof) {
      message v = new message();
      while (!comparestring(buff, "BREAK") && !fileeof && !comparestring(buff, "END")) {
        if (comparebuff(buff, "ID") == true) {v.ID = getpropvalue_long(buff);}
        if (comparebuff(buff, "Text") == true) {v.Text = getpropvalue_string(buff);}
        buff = decrypt(readline(in));
      }
      // Add the message to the collection
      data.omessages.add(v);
      
      vdu.consoleconnector.location.setText("Reading messages... " + getNextProgress());
      
      // If it wasn't an END instruction, keep on reading
      if (!comparestring(buff, "END")) {buff = decrypt(readline(in));}
    }
    vdu.println("OK (" + Integer.toString(data.omessages.getCount()) + " Messages).");

  }
  
  
  /** Returns the next progress character for animated characters
      during download */
  private String getNextProgress() {
  	
  	if (prg	== 0) {
  		prg++;
  		return "|";	
  	}
  	
  	if (prg == 1) {
  		prg++;
  		return "\\";
  	}
  	
  	if (prg == 2) {
  		prg++;
  		return "-";
  	}
  	
  	if (prg == 3) {
  		prg = 0;
  		return "/";
  	}
  	
  	return "|";
  	
  }  

  /**
   *  Returns everything after the equals sign formatted as long.
   */
  public static long getpropvalue_long(String sbuff) {
	// if first char is space remove it
	String p = getvalue(sbuff);
    return Long.parseLong(p);
  }

  /**
   *  Returns everything after the equals sign formatted as string
   */
  public static String getpropvalue_string(String sbuff) {
    return getvalue(sbuff); // Get and return as string
  }

  /**
   *  Returns everything after the equals sign formatted as boolean.
   */
  public static boolean getpropvalue_boolean(String sbuff) {
    String p = getvalue(sbuff); // Get as string
    if (p.equals("true") || p.equals("True")) return true; else return false;
  }

  /**
   *  Throws away everything upto and including the equals sign
   *  in the passed string.
   */
  public static String getvalue(String sbuff) {
    // Returns a string containing the property value only
    String paramvalue = sbuff.substring(sbuff.indexOf("=") + 1, sbuff.length());
    
    // Return empty string if we just have a space
    if (paramvalue.startsWith(" ") && paramvalue.length() == 1) {
    	return "";
	}
    paramvalue = removeleadingspaces(paramvalue);
    paramvalue = removetrailingspaces(paramvalue);
    
    return paramvalue;
  }

  /**
   *  Throws everything away after and including the equals sign
   *  in string one. Then compares to string 2. If they match,
   *  returns a true value.
   */
  public static boolean comparebuff(String buff, String compareto) {
    // Returns true if the specified property name forms the
    // first part of the string

    // Determine if we have an equals - ditch now if not
    if (buff.indexOf("=") == -1) {return false;}

    // Get the first part of the string
    String prop = buff.substring(0, buff.indexOf("=") -1);
    
    // ditch spaces
    prop = data.removeleadingspaces(prop);
    prop = data.removetrailingspaces(prop);
    
    // Compare
    if (prop.equals(compareto)) {
      return true;
    }
    else {
      return false;
    }
  }
  
  /**
   *  Returns true or false depending on whether the two strings
   *  passed in are the same. This routine will ditch whitespace
   *  from the first string before comparison.
   */
  private boolean comparestring(String buff, String compareto) {
    // Returns true if the strings are equal

    // ditch spaces
    buff = removeleadingspaces(buff);
    buff = removetrailingspaces(buff);
    // Compare
    if (buff.equals(compareto)) {
      return true;
    }
    else {
      return false;
    }
  }

  /*** Strips all the leading spaces from a string */
  public static String removeleadingspaces(String s) {
  	
  	s =  s.trim();
  	return s;
  	
  }

  /*** Strips all the trailing spaces from a string */
  public static String removetrailingspaces(String s) {
  	
  	s = s.trim();
  	return s;
  	
  }

  /*** Removes all spaces, leading and trailing */
  public static String trimstring(String s) {
  	s = s.trim();
  	return s;
  }

  public static String decrypt(String s) {
    // Decrypts a string if the isencrypted flag is set

    // If flag is not set, return the same string passed in
    if (!isencrypted) {return s;}
    
    // Check the encryption type
    
    // HEX
    if (encrypttype == 1) {

		final String Hex = "0123456789ABCDEF";
		byte[] ba = new byte[s.length() / 2];
		int bpos = 0;
		int ihi = 0;
		int ilo = 0;
		String shi = "";
		String slo = "";
		String out = "";
		
		for (int i = 0; i < s.length(); i++) {
			
			// Read hi byte
			shi = s.substring(i, i + 1);
			// Calculate decimal value from position in Hex string
			ihi = Hex.indexOf(shi) * 16;
			
			// Next
			i++;
			
			// Read lo byte
			slo = s.substring(i, i + 1);
			// Calculate decimal value from position in Hex
			ilo = Hex.indexOf(slo);
			
			// Enter
			ba[bpos] = Byte.parseByte(Integer.toString(ihi + ilo));
			bpos++;
			
		}
		
		// Dump byte stream to a string
		out = new String(ba);
		
		// Truncate to correct length
		out = out.substring(0, bpos);
		
		return out;
	}
	else
	{

		byte[] ba = s.getBytes();
		byte[] ob = new byte[ba.length];
		String out = "";
		
		for (int i = 0; i < ba.length; i++) {
			// Shift byte 2 places
			ob[i] = ba[i];
			ob[i] = (byte)(((int)ob[i]) - 2);
		}
		
		out = new String(ob);
		return out;
	}
		
  }
  
  public static String encrypt(String s) {
  	    
  	    // Encrypts a string

		final String Hex = "0123456789ABCDEF";
		byte[] ba = s.getBytes();
		double cb = 0;
		double hi = 0;
		double lo = 0;
		int ihi = 0;
		int ilo = 0;
		String shi = "";
		String slo = "";
		String out = "";
		String hexrep = "";
		
		for (int i = 0; i < ba.length; i++) {
			
			// Convert byte to double
			cb = (double) ba[i];
			
			// Calculate high byte
			hi = cb / 16;
			
			// Throw away all after decimal point
			int iihi = (int) hi;
			hi = (double) iihi;
			
			// Calculate lo byte
			lo = cb - (hi * 16);
			
			shi = Double.toString(hi);
			slo = Double.toString(lo);
			
			if (shi.endsWith(".0")) { shi = shi.substring(0, shi.length() - 2); }
			if (slo.endsWith(".0")) { slo = slo.substring(0, slo.length() - 2); }
			
			ihi = Integer.parseInt(shi);
			ilo = Integer.parseInt(slo);
			
			// Construct hex string
			hexrep = Hex.substring(ihi, ihi + 1);
			hexrep += Hex.substring(ilo, ilo + 1);
			
			// Append
			out += hexrep;
			
		}
		
		return out;

  }
  
  /*** Converts booleans to strings. 
     * It's not like I'm talking rocket science here and it's only a little routine, but
     * come on Sun, how hard would it have been to put a "toString" method in the Boolean
     * wrapper? */
  public static String booleanToString(boolean b) {
  	if (b) {
  		return "true";
  	}
  	else
  	{
  		return "false";
  	}
  }
  /*** Converts a string to a boolean */
  public static boolean stringToBoolean(String s) {
  	return s.equalsIgnoreCase("true");
  }
}