! NpAct.h - library to integrate NPCs and the player, for Inform.
!   Version 1.0 (19-Sep-2001)
!
! by Matt Albrecht - groboclown@users.sourceforge.net
!
! (If you need to edit this file, note that indentations are 4 spaces
! and tabs are not used.)
!
! This has been donated to the Public Domain.  Use, abuse, and don't blame me.
!
! To prevent cluttering the global namespace, the NPC-Action stuff
! all begins with "npcact_" (in any case).
! Library private routines, constants, and variables begin with "npcact__".

! For details, see the included NpcAct.html

System_file;


Ifndef NPCACT__INCLUDED;
Constant NPCACT__INCLUDED;

Include "classutl";

! Constants to allow debug information to be displayed at various details.
! General pattern:
!           0 - no debugging
!           1 - high level reports
!           ...
!           5 - everything that goes on
Ifndef NPCACT_TRACE_LOGIC;
    Constant NPCACT_TRACE_LOGIC 0;
Endif;
Ifndef NPCACT_TRACE_PLAYER;
    Constant NPCACT_TRACE_PLAYER 0;
Endif;

! A general flag - set if you want erroneus states to be visually displayed
!Constant NPCACT_CHECK_ERRORS;

! Optimize the attribute usage to minimize the number of them used.
Constant NPCACT_OPTIMIZE_ATTRIBUTES;

! Define attributes
Attribute npcact_att_monitor;
Ifdef NPCACT_OPTIMIZE_ATTRIBUTES;
Attribute npcact_att_character    alias on;
Ifnot;
Attribute npcact_att_character;
Endif;


! Common property
Property npcact__next;


! Monitor Overrideable Properties
Property npcact_Before;
Property npcact_After;
Property npcact_Canceled;

! Character Overrideable Properties
Property npcact_Turn;

! Action Overrideable Properties
Property npcact_Setup;
Property npcact_Execute;
Property npcact_CanceledResponse;
Property npcact_verb;


Message "Adding NPC-Action library";


! Private global internal variables

! Maximum number of monitors in scope at one time, per action
Ifndef NPCACT__MAX_MONITOR_COUNT;
    Constant NPCACT__MAX_MONITOR_COUNT 8*32;
Endif;

Array npcact__v_seenplayer -> (NPCACT__MAX_MONITOR_COUNT / 8);

!------------------------------------------------------------------------------
! Override the library functions.
!------------------------------------------------------------------------------

! Calls NpcAct_NewRoom when the NPC-Action library part of the routine is
! finished.
[ NewRoom
    ;
    
    move NpcAct_Imp to player;
    
    NpcAct_NewRoom();
];


Ifndef NpcAct_NewRoom;
[ NpcAct_NewRoom; ];
Endif;

    
!------------------------------------------------------------------------------
! The Monitor Class    
!------------------------------------------------------------------------------

Class NpcAct_Monitor
    with
        !----------------------------------------------------------------------
        ! Player Input methods
        !       data-collecting routines.
        !----------------------------------------------------------------------
        
        react_before
            [;
#Iftrue NPCACT_TRACE_PLAYER >= 1;
print "[NpcAct Player: Character ",(name)self,": react_before.]^";
#Endif;
                ! Player performed an action
                return NpcAct__ActionProcessor.npcact__PlayerBefore( self );
            ],
! react_before and orders are always called before life
!        life
!            [;
!#Ifdef NPCACT_TRACE_PLAYER;
!print "NPC ",(name)self,": life^";
!#Endif;
!            ],
        orders
            [;
#Iftrue NPCACT_TRACE_PLAYER >= 1;
print "[NpcAct Player: NPC ",(name)self,": orders.]^";
#Endif;
                return NpcAct__ActionProcessor.npcact__PlayerBefore( self );
            ],
        

        !----------------------------------------------------------------------
        ! Public Overrideable methods
        !----------------------------------------------------------------------
        
        npcact_Before
            [
                _actionList ! parameter
                ;
                
                ! subclass defined
            ],
        
        npcact_After
            [
                _actionList ! parameter
                ;
                
                ! subclass defined
            ],
        
        npcact_Canceled
            [
                _actionList _canceller ! parameters
                ;
                
                ! subclass defined
            ],
        

        !----------------------------------------------------------------------
        ! Private methods
        !----------------------------------------------------------------------
        
        
        
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Inner variables useful only to this class.
        ! You should never touch these.
        npcact__action nothing,
        npcact__next nothing,
        npcact__index -1,
        
    has
        npcact_att_monitor
    ;


    
!------------------------------------------------------------------------------
! The Base NPC Class
!------------------------------------------------------------------------------


Class NpcAct_Character
    with
        !----------------------------------------------------------------------
        ! general methods - names are required by outside stuff.
        !----------------------------------------------------------------------
        classutl_constructor
            [;
                ! Check for possible errors.
                
                ! currently, nothing to check
                
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: ",(name)self," is in the default Character constructor.]^";
#Endif;
            ],
        
        
        !----------------------------------------------------------------------
        ! abstract methods specific to AiNPC subclasses / instances
        !----------------------------------------------------------------------
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Performed during the NPC's turn.
        !
        ! Overridden by sub-class
        npcact_Turn
            [
                ;
                
            ],
        
            
        
        !----------------------------------------------------------------------
        ! public methods specific to NpcAct
        !   - these methods should not be overridden by subclasses.
        !----------------------------------------------------------------------
        
            
        
        !----------------------------------------------------------------------
        ! private properties specific to NpcAct
        !   - these methods should not be overridden by subclasses, nor should
        !     they be called by anthing except the AiNPC library.
        !----------------------------------------------------------------------
        
            
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Inner variables useful only to this class.
        ! You should never touch these.
        npcact__action nothing,
        npcact__next nothing,
    has
        animate
        npcact_att_character
    ;

    
    
!------------------------------------------------------------------------------
! Helper Class - The commonly used character monitor
!------------------------------------------------------------------------------

Class NpcAct_CharacterMonitor
    class NpcAct_Character
    class NpcAct_Monitor;



!------------------------------------------------------------------------------
! The Singleton Action List Object    
!------------------------------------------------------------------------------

! Bits which define what can and can't be executed in a listener. Set if it can
! be executed.
Constant NPCACT__REGISTRATION_NONE 0;
Constant NPCACT__REGISTRATION 1;
Constant NPCACT__REGISTRATION_BEFORE 2;
Constant NPCACT__REGISTRATION_AFTER 4;
Constant NPCACT__REGISTRATION_CANCEL 8;

Object NpcAct__ActionList
    with

        !----------------------------------------------------------------------
        ! Public methods
        !----------------------------------------------------------------------
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_Register
            [
                _action;  ! parameter
                
                if (self.npcact__CheckAction( _action, NPCACT__REGISTRATION,
                    "Register" ))
                {
                    self.npcact__InvalidateAction( _action );
                    rtrue;
                }
                
                self.npcact__Prepend( _action );
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_RegisterBefore
            [
                _action;  ! parameter
                
                if (self.npcact__CheckAction( _action,
                    NPCACT__REGISTRATION_BEFORE, "RegisterBefore" ))
                {
                    self.npcact__InvalidateAction( _action );
                    rtrue;
                }
                
                self.npcact__Prepend( _action );
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_RegisterAfter
            [
                _action;  ! parameter
                
                if (self.npcact__CheckAction( _action,
                    NPCACT__REGISTRATION_AFTER, "RegisterAfter" ))
                {
                    self.npcact__InvalidateAction( _action );
                    rtrue;
                }
                
                self.npcact__Append( _action );
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_CurrentAction
            [;
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: CurrentAction: ",(name)self.npcact__head,"]^";
#Endif;
                return self.npcact__head;
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_Cancel
            [;
                
                if (self.npcact__ValidateState( NPCACT__REGISTRATION_CANCEL,
                    "Cancel" ))
                {
                    rtrue;
                }
                
                if (self.npcact__head ~= nothing)
                {
                    self.npcact__head.npcact__state = NPCACT__ACTION_CANCELED;
                }
            ],
            

        !----------------------------------------------------------------------
        ! Private properties
        !----------------------------------------------------------------------
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Pull the current action off the list, and add it to the executed list.
        npcact__EndAction
            [
                _action; ! local
                
                _action = self.npcact__head;
                if (_action ~= nothing)
                {
                    self.npcact__head = _action.npcact__next;
                    _action.npcact__next = self.npcact__executed;
                    self.npcact__executed = _action;
#Iftrue NPCACT_TRACE_LOGIC >= 1;
print "[NpcAct Logic: EndAction: new current action = ",(name)self.npcact__head,"]^";
#Endif;
                }
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Clean up the list after the round has finished.
        npcact__Cleanup
            [
                        ! parameters
                _action _oldaction; !locals
                
                _action = self.npcact__executed;
                while (_action ~= nothing)
                {
                    ! Clean up the action

                    ! de-register the action
                    _action.npcact__state = NPCACT__ACTION_EMPTY;
                    
                    
                    !XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
                    ! Add more cleanup?
                    
                    
                    _oldaction = _action;
                    
                    ! go to next action
                    _action = _action.npcact__next;
                    
                    ! clean up link in old action
                    _oldaction.npcact__next = nothing;
                }
                
                self.npcact__executed = nothing;
            ],
            
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact__state NPCACT__REGISTRATION_NONE,
            

        !----------------------------------------------------------------------
        ! All Private parts *MUST* occur after the last non-private part
        !----------------------------------------------------------------------
            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        private npcact__InvalidateAction
            [
                _action; ! parameter
                
                if (_action ~= nothing && _action ofclass NpcAct_Action)
                {
                    _action.npcact__state = NPCACT__ACTION_INVALID;
                }
            ],

        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        private npcact__CheckAction
            [
                _action _bits _name; ! parameters
                if (_action == nothing)
                {
#Ifdef NPCACT_CHECK_ERRORS;
                    print "** ",(string)_name," of nothing Action **^";
#Endif;
                    rtrue;
                }
                if (~~(_action ofclass NpcAct_Action))
                {
#Ifdef NPCACT_CHECK_ERRORS;
                    print "** ",(string)_name," of non-action ",
                        (name)_action," **^";
#Endif;
                    rtrue;
                }
#Iftrue NPCACT_TRACE_LOGIC >= 4;
print "[NpcAct Logic: Checking registration: Action ",(name)_action,"]^";
#Endif;
                if (_action.npcact_IsRegistered())
                {
#Ifdef NPCACT_CHECK_ERRORS;
                    print "** ",(string)_name," Action ",(name)_action,
                        " already registered. **^";
#Endif;
                    rtrue;
                }

                
#Iftrue NPCACT_TRACE_LOGIC >= 4;
print "[NpcAct Logic: Checking source: Action ",(name)_action,"]^";
#Endif;
                if (_action.npcact_source == nothing)
                {
#Ifdef NPCACT_CHECK_ERRORS;
                    print "** ",(string)_name," Action ",(name)_action,
                        " has no source character. **^";
#Endif;
                    rtrue;
                }
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: Checking source not player: Action ",(name)_action,"]^";
#Endif;
                if (_action.npcact_source ~= player)
                {
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: Checking character properties: source ",(name)_action.npcact_source,"]^";
#Endif;
                    if (~~(_action.npcact_source ofclass NpcAct_Character) ||
                        ~~(_action.npcact_source ofclass NpcAct_Monitor))
                    {
#Ifdef NPCACT_CHECK_ERRORS;
                        print "** ",(string)_name," Action ",(name)_action,
                            " has non-character/monitor source ",
                            (name)_action.npcact_source,". **^";
#Endif;
                        rtrue;
                    }
                    
                    ! Check for double registration
                    if (_action.npcact_source.npcact__action ~= nothing)
                    {
#Ifdef NPCACT_CHECK_ERRORS;
                        print "** ",(string)_name," Action ",(name)_action,
                            " has character/monitor source ",
                            (name)_action.npcact_source," already registered
                            with action ",
                            (name)_action.npcact_source.npcact__action,". **^";
#Endif;
                        rtrue;
                    }
                }
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: Checking list state]^";
#Endif;
                
                return self.npcact__ValidateState( _bits, _name );
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        private npcact__ValidateState
            [
                _bits _name; ! parameters
                
#Iftrue NPCACT_TRACE_LOGIC >= 4;
print "[NpcAct Logic: List state = ",self.npcact__state,", bits = ",_bits,"]^";
#Endif;
                if ((self.npcact__state & _bits) == 0)
                {
#Ifdef NPCACT_CHECK_ERRORS;
                    print "** ",(string)_name," occured when not
                        allowed. **^";
#Endif;
                    rtrue;
                }
                rfalse;
            ],
                
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        private npcact__Prepend
            [
                _action; ! parameter

                _action.npcact__next = self.npcact__head;
                self.npcact__head = _action;
                
                self.npcact__RegisterAction( _action );
#Iftrue NPCACT_TRACE_LOGIC >= 1;
print "[NpcAct Logic: Prepend: new current action = ",(name)self.npcact__head,"]^";
#Endif;
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        private npcact__Append
            [
                _action; ! parameter

                if (self.npcact__head == nothing)
                {
                    return self.npcact__Prepend( _action );
                }
                _action.npcact__next = self.npcact__head.npcact__next;
                self.npcact__head.npcact__next = _action;
                
                self.npcact__RegisterAction( _action );
#Iftrue NPCACT_TRACE_LOGIC >= 1;
print "[NpcAct Logic: Append: new current action = ",(name)self.npcact__head,"]^";
#Endif;
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        private npcact__RegisterAction
            [
                _action; ! parameter
                
                if (_action.npcact_source == player)
                {
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: Registering the player action]^";
#Endif;
                    NpcAct_Imp.npcact__action = _action;
                }
                else
                {
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: Registering action = ",(name)_action," with ",(name)_action.npcact_source,"]^";
#Endif;
                    _action.npcact_source.npcact__action = _action;
                }
                _action.npcact__state = NPCACT__ACTION_PRE_BEFORE;
            ],
        
        ! The current head in the list of actions.
        private npcact__head nothing,
        
        ! The current head pointing to the list of already executed actions.
        ! Used for clean-up tasks.
        private npcact__executed nothing,
        
    has
    ;
    
    
!------------------------------------------------------------------------------
! The Base Action Object
!       Due to the nature of subclassing and object sharing, I suggest
!       an architecture of giving each AiNPC one Action object (hence the class
!       being concealed and scenery).  Then, before the object is registered,
!       specify a subclass to set the object to.  These "subclasses" should be
!       singleton global objects.
!
!       The npcact_verb variable should correspond to, say, ##Look.
!------------------------------------------------------------------------------

! State of the action
Constant NPCACT__ACTION_EMPTY 0;
Constant NPCACT__ACTION_PRE_BEFORE 1;
Constant NPCACT__ACTION_IN_BEFORE 2;
Constant NPCACT__ACTION_AFTER_BEFORE 3;
Constant NPCACT__ACTION_CANCELED 4;
Constant NPCACT__ACTION_COMPLETE 5;
Constant NPCACT__ACTION_INVALID 6;


Class NpcAct_Action
    with
        !----------------------------------------------------------------------
        ! Public methods (should be untouched)
        !----------------------------------------------------------------------
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Construct the current Action (including reinitialization) based on
        ! the given subclass
        !
        ! If _npc is not given, it will use the parent object.
        ! All known data and subclassed methods are copied to this action
        ! from the subclass object.
        npcact_Construct
            [
                _template   ! parameter
                _npc;       ! optional parameter
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: Inside base Actor Construct.]^";
#Endif;
                
#Ifdef NPCACT_CHECK_ERRORS;
                if (_template == nothing)
                {
                    "** Action ",(name)self," is being constructed with a
                    non-existent template. **";
                }
                if (~~(_template ofclass NpcAct_Action))
                {
                    "** Action ",(name)self," is being constructed with
                    a non-template object ",(name)_template,". **";
                }
#Endif;
                if (_npc == nothing)
                {
                    _npc = parent(self);
#Ifdef NPCACT_CHECK_ERRORS;
                    if (_npc == nothing)
                    {
                        "** Action ",(name)self," constructed without
                        an NPC owner. **";
                    }
#Endif;
                }
#Ifdef NPCACT_CHECK_ERRORS;
                if (~~(_npc ofclass NpcAct_Character) ||
                    ~~(_npc ofclass NpcAct_Monitor))
                {
                    "** Action ",(name)self," is being constructed with a
                    non-Character/Monitor owner ",(the)_npc,". **";
                }
#Endif;
                
                self.npcact__reversed = false;
                self.npcact_source = _npc;
                self.npcact_actor = _npc;
                self.npcact__state = NPCACT__ACTION_EMPTY;

                ! Copy data
                self.npcact_Setup = _template.npcact_Setup;
                self.npcact_Execute = _template.npcact_Execute;
                self.npcact_data = _template.npcact_data;
                self.npcact_verb = _template.npcact_verb;
                self.npcact_noun = _template.npcact_noun;
                self.npcact_second = _template.npcact_second;
                self.npcact_CanceledResponse =
                    _template.npcact_CanceledResponse;
                if (_template.npcact_actor ~= nothing)
                {
                    self.npcact_actor = _template.npcact_actor;
                }
            ],

            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_PlayerInScope
            [
                 ! parameter
                ret; ! locals
                
                ret = (TestScope( self.npcact_source ));
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: PlayerInScope for Action ",(name)self," = ",ret,"]^";
#Endif;
                return ret;
            ],

            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_IsPending
            [
                ; ! parameters
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: IsPending: Action ",(name)self,".npcact__state = ",self.npcact__state,"]^";
#Endif;
                return (self.npcact__state ==
                    NPCACT__ACTION_PRE_BEFORE or
                    NPCACT__ACTION_IN_BEFORE or
                    NPCACT__ACTION_AFTER_BEFORE);
            ],

            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_IsCanceled
            [
                ; ! parameters
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: IsCanceled: Action ",(name)self,".npcact__state = ",self.npcact__state,"]^";
#Endif;
                
                return (self.npcact__state == NPCACT__ACTION_CANCELED);
            ],

            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_IsComplete
            [
                ; ! parameters
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: IsComplete: Action ",(name)self,".npcact__state = ",self.npcact__state,"]^";
#Endif;
                
                return (self.npcact__state == NPCACT__ACTION_COMPLETE);
            ],

            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_IsRegistered
            [
                ; ! parameters
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: IsRegistered: Action ",(name)self,".npcact__state = ",self.npcact__state,"]^";
#Endif;
                
                return (self.npcact__state ~= NPCACT__ACTION_EMPTY);
            ],

            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_IsReversed
            [
                ; ! parameters
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: IsReversed: Action ",(name)self,".npcact__state = ",self.npcact__state,"]^";
#Endif;
                
                return (self.npcact__reversed == true);
            ],

            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_IsDone
            [
                ; ! parameters
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: IsDone: Action ",(name)self,".npcact__state = ",self.npcact__state,"]^";
#Endif;
                
                return (self.npcact__state ==
                    NPCACT__ACTION_CANCELED or NPCACT__ACTION_COMPLETE or
                    NPCACT__ACTION_INVALID );
                
            ],

            
        !----------------------------------------------------------------------
        ! Public variables (should only be modified by template Setup)
        !----------------------------------------------------------------------
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact_source nothing,
        npcact_actor nothing,
        npcact_verb -1,
        npcact_noun nothing,
        npcact_second nothing,
        npcact_secondVerb -1,
        npcact_data nothing,


        
        !----------------------------------------------------------------------
        ! Public Methods assigned from a Subclass during assembly
        !----------------------------------------------------------------------
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Sets up the subclassed object before registration.
        ! Parameters are based on subclass.
        npcact_Setup
            [;
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Perform the Action.  Never takes parameters.
        npcact_Execute
            [;
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Perform a response if the action was canceled. Only performed if the
        ! property is not nothing and a Routine.
!        npcact_CanceldResponse [ _canceller; ],
        npcact_CanceledResponse nothing,
        

            
        !----------------------------------------------------------------------
        ! Private properties (should be untouched)
        !----------------------------------------------------------------------
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Returns true if the action has reversed itself, or false if it
        ! cannot reverse itself.
        npcact__Reverse
            [
                 ! parameters
                t; ! locals
                
                if (self.npcact__reversed == true ||
                    self.npcact_IsComplete() == false ||
                    self.npcact_second == nothing ||
                    self.npcact_secondVerb < 0)
                {
                    rfalse;
                }
                
                self.npcact_Execute = nothing;
                self.npcact_CanceledResponse = nothing;
                
                self.npcact_verb = npcact_secondVerb;
                
                t = self.npcact_second;
                self.npcact_second = self.npcact_noun;
                self.npcact_noun = t;
                
                self.npcact__state = NPCACT__ACTION_PRE_BEFORE;
                self.npcact__reversed = true;
                
                rtrue;
            ],
        
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact__state NPCACT__ACTION_EMPTY,
        npcact__canceller nothing,
        npcact__next nothing,
        npcact__reversed false,
        
    has
        scenery;


        
        
!------------------------------------------------------------------------------
! The Singleton Imp Object
!       - the character associated with the player
!       - not a daemon - it should be moved to the player before the 
!------------------------------------------------------------------------------

Object NpcAct_Imp
    class NpcAct_Monitor
    with
        npcact_Before
            [
                _actionList ! parameter
                _action _x; !locals

#Iftrue NPCACT_TRACE_PLAYER >= 1;
                _action = _actionList.npcact_CurrentAction();
                if (_action.npcact_source == player)
                {
                    print "[NpcAct Player: Imp responding to player action Before.]^";
                }
#Endif;
                ! Safe object loop - monitors may move themselves around.
                objectloop( _x ofclass NpcAct_Monitor )
                {
                    if (parent(_x) == self)
                    {
                        _x.npcact_Before( _actionList );
                    }
                }
            ],
        
        npcact_After
            [
                _actionList ! parameter
                _action x; !locals
                

#Iftrue NPCACT_TRACE_PLAYER >= 1;
                _action = _actionList.npcact_CurrentAction();
                if (_action.npcact_source == player)
                {
                    print "[NpcAct Player: Imp responding to player action After.]^";
                }
#Endif;
                objectloop( x in self && x ofclass NpcAct_Monitor )
                {
                    x.npcact_After( _actionList );
                }
            ],
        
        npcact_Canceled
            [
                _actionList ! parameter
                _action x; !locals
                

#Iftrue NPCACT_TRACE_PLAYER >= 1;
                _action = _actionList.npcact_CurrentAction();
                if (_action.npcact_source == player)
                {
                    print "[NpcAct Player: Imp responding to player action Canceled.]^";
                }
#Endif;
                objectloop( x in self && x ofclass NpcAct_Monitor )
                {
                    x.npcact_Canceled( _actionList );
                }
            ],
    has
        concealed
        scenery
        ~npcact_att_character
        ;

NpcAct_Action -> NpcAct__ImpAction
    with
        ! build the action object from the player's action
        npcact_Construct
            [
                ;
                
                self.npcact_source = player;
                self.npcact_actor = actor;
                self.npcact_verb = action;
                self.npcact_noun = noun;
                self.npcact_second = second;
                self.npcact_data = nothing;
            ],
        npcact_Setup
            [
                ;
#Ifdef NPCACT_CHECK_ERRORS;
                "** ",(name)self," had setup called - should never be called
                **";
#Endif;
            ],
        npcact_Execute
            [
                ;
#Ifdef NPCACT_CHECK_ERRORS;
                "** ",(name)self," had execute called - should never be called
                **";
#Endif;
            ],
    ;

        
!------------------------------------------------------------------------------
! The Singleton Action Processor Object
!       - integrated with the Character List for optimization reasons
!       - This is where the action is, boys.
!------------------------------------------------------------------------------

! Note: make sure that all call-back functions have the ActionList bits set
!       correctly
Object NpcAct__ActionProcessor
    with

        !----------------------------------------------------------------------
        ! general methods - names are required by outside stuff.
        !----------------------------------------------------------------------
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        classutl_constructor
            [
                _x _i;  ! locals
#Iftrue NPCACT_TRACE_LOGIC >= 5;
print "[NpcAct Logic: ",(name)self," is in the constructor.]^";
#Endif;
                

                ! Start this as a daemon process.
                StartDaemon( self );
                

                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                ! CharacterList constructor
                objectloop( _x ofclass NpcAct_Character )
                {
                    _x.npcact__next = self.npcact__head;
                    self.npcact__head = _x;
                }
                
                ! Globals setup
                for (_i = 0 : _i < NPCACT__MAX_MONITOR_COUNT : ++_i)
                {
                    NpcAct__SetFlag( npcact__v_seenplayer, _i, false );
                }
            ],
        
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        daemon
            [
                _x _i _rout;
#Iftrue NPCACT_TRACE_LOGIC >= 2;
print "[NpcAct Logic: ActionProcessor: start Daemon.]^";
#Endif;
                
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                ! Perform post-player actions
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                
                
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                ! Finish passing Before action to each monitor in scope of the
                ! player
                objectloop( _x ofclass NpcAct_Monitor )
                {
                    if (_x has npcact_att_monitor &&
                        _x.npcact__index >= 0 &&
                        NpcAct__FlagOn( npcact__v_seenplayer, _x.npcact__index)
                            == false &&
                        TestScope( _x ) == true)
                    {
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: Call Before on ",(name)_x,"]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;
                        NpcAct__CallMonitor( _x, NPCACT__BEFORE );
                    }
                }

                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                ! Call the post action method for all monitors around the player
                _rout = NPCACT__AFTER;
                if (NpcAct__ImpAction.npcact_IsCanceled())
                {
                    _rout = NPCACT__CANCELED;
                }
                objectloop( _x ofclass NpcAct_Monitor )
                {
                    if (TestScope( _x ) == true)
                    {
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: Call Cancel/After on ",(name)_x,"]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;
                        NpcAct__CallMonitor( _x, _rout );
                    }
                }
                NpcAct__ActionList.npcact__EndAction();
                
                
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                ! Finish up all remaining actions from the Player responses
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                
                self.npcact__RunRemainingActions();
                
                
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                ! Perform character turns
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                
                objectloop( _x ofclass NpcAct_Character &&
                        _x has npcact_att_character)
                {
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: Call Turn on ",(name)_x,"]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;
                    NpcAct__CallMonitor( _x, NPCACT__TURN );
                    
                    self.npcact__ActionLogic( _x );
                }
                
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                ! Finish up all remaining actions
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                
                self.npcact__RunRemainingActions();
                
                
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                ! Perform cleanup duties
                !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

#Iftrue NPCACT_TRACE_LOGIC >= 2;
print "[NpcAct Logic: ActionProcessor: cleanup.]^";
#Endif;
                for (_i = 0 : _i < NPCACT__MAX_MONITOR_COUNT : ++_i)
                {
                    NpcAct__SetFlag( npcact__v_seenplayer, _i, false );
                }
                objectloop( _x ofclass NpcAct_Monitor )
                {
                    _x.npcact__index = -1;
                }
                objectloop( _x ofclass NpcAct_Character )
                {
                    _x.npcact__action = nothing;
                }
                
                self.npcact__flagcount = 0;
                self.npcact__playerRegistered = false;
                NpcAct__ActionList.npcact__Cleanup();
            ],
        


        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! Perform before / cancel / execute / after loop
        npcact__ActionLogic
            [
                _character ! parameter
                _x _lastAction _routType _isCanceled; ! locals
                
#Ifndef NPCACT_CHECK_ERRORS;
                if (_character.npcact__action ~= nothing &&
                    _character.npcact__action ~=
                        NpcAct__ActionList.npcact_CurrentAction())
                {
                    "** Character ",(name)_character," has action ",
                        (name)_character.npcact__action,", but doesn't match
                        current action ",
                        (name)NpcAct__ActionList.npcact_CurrentAction(),". **";
                }
#Endif;
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: ActionLogic: char = ",(name)_character," (action = ",(name)_character.npcact__action,")]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;
                
                ! Only run this loop if the character which requested the action
                ! has an action that has not executed.
                if (_character.npcact__action ~= nothing &&
                    _character.npcact__action.npcact_IsDone() == false)
                {
                    _lastAction = _character.npcact__action;
                    
                    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                    ! Perform Before calls
                    
                    _lastAction.npcact__state = NPCACT__ACTION_IN_BEFORE;
                    if (_lastAction hasnt concealed)
                    {
                        objectloop(
                            _x has npcact_att_monitor &&
                            _x ofclass NpcAct_Monitor &&
                            TestScope( _x, _character ) )
                        {
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: Call Before on ",(name)_x," (last action = ",(name)_lastAction,")]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;
                            _isCanceled = _lastAction.npcact_IsCanceled();
                            NpcAct__CallMonitor( _x, NPCACT__BEFORE,
                                _lastAction has locked );
                            if (_isCanceled == false &&
                                _lastAction.npcact_IsCanceled() == true)
                            {
                                ! found canceller
                                _lastAction.npcact__canceller = _x;
                            }
                            
                            if (_lastAction ~=
                                NpcAct__ActionList.npcact_CurrentAction())
                            {
#Ifndef NPCACT_CHECK_ERRORS;
                                if (~~(_x ofclass NpcAct_Character))
                                {
                                    print "** monitor ",(name)_x," registered an
                                        action, but is not a character. **^";
                                }
#Endif;
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: Action registered in Before on ",(name)_x," (last action = ",(name)_lastAction,")]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;
                                self.npcact__ActionLogic( _x );
                            }
                        }
                    }


                    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                    ! Perform action
                    
                    ! Set the ActionList bits before running the action
                    NpcAct__ActionList.npcact__state =
                        NPCACT__REGISTRATION_NONE;
                    
                    _routType = 0;
                    if (_lastAction.npcact_IsCanceled())
                    {
#Ifndef NPCACT_CHECK_ERRORS;
                        if (_lastAction has locked or concealed)
                        {
                            print "** Action ",(name)_lastAction," should be un
                                -cancelable, but was canceled by ",
                                (name)_lastAction.npcact__canceller,". **^";
                        }
                        if (_lastAction.npcact__canceller == nothing)
                        {
                            print "** Action ",(name)_lastAction," was canceled,
                            but no monitor took responsibility for cancelling it
                            (npcact__canceller == nothing). **^";
                        }
#Endif;
                        _routType = NPCACT__CANCELED;
                        if (NpcAct__CanCallMethod( _lastAction,
                            npcact_CanceledResponse ) == true)
                        {
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: Calling CanceledResponse on action ",(name)_lastAction,"]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;
                            _lastAction.npcact_CanceledResponse(
                                _lastAction.npcact__canceller );
                        }
                    }
                    else
                    {
                        _lastAction.npcact__state = NPCACT__ACTION_AFTER_BEFORE;
                        _routType = NPCACT__AFTER;
                        if (NpcAct__CanCallMethod( _lastAction,
                            npcact_Execute ) == true)
                        {
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: Calling Execute on action ",(name)_lastAction,"]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;
                            _lastAction.npcact_Execute();
                        }
                    }
                    
                    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                    ! Perform After / Cancel calls
                    
                    if (_lastAction hasnt concealed)
                    {
                        objectloop(
                            _x has npcact_att_monitor &&
                            _x ofclass NpcAct_Monitor &&
                            TestScope( _x, _character ) )
                        {
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: Calling After/Canceled on monitor ",(name)_x,"]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;
                            NpcAct__CallMonitor( _x, _routType );
                        }
                    }
                    _lastAction.npcact__state = NPCACT__ACTION_COMPLETE;
                    
                    if (_lastAction.npcact__Reverse())
                    {
                        ! perform the action in reverse mode
                        self.npcact__ActionLogic( _character );
                        
                        ! the above will remove this action from the list,
                        ! hence the else statement below.
                    }
                    else
                    {
                        ! Only end the action if it did not just run the normal
                        ! mode, followed by a reverse (i.e. we may be in the
                        ! reverse mode).
                        NpcAct__ActionList.npcact__EndAction();
                    }
                }
            ],
        
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! A monitor encountered a player action
        npcact__PlayerBefore
            [
                _monitor   ! parameter
                _ret;   ! locals
                
                ! The monitor is not active
                if (_monitor hasnt npcact_att_monitor)
                {
                    rfalse;
                }
                
                
#Ifdef NPCACT_CHECK_ERRORS;
                if (self.npcact__flagcount >= NPCACT__MAX_MONITOR_COUNT)
                {
                    print "** Error: too many monitors in scope (maximum is ",
                        NPCACT__MAX_MONITOR_COUNT,"). **^";
                    rfalse;
                }
#Endif;
                _monitor.npcact__index = self.npcact__flagcount++;
                
                ! create the action object
                NpcAct__ImpAction.npcact_Construct();
                
                if (self.npcact__playerRegistered == false)
                {
#Iftrue NPCACT_TRACE_LOGIC >= 2;
print "[NpcAct Logic: Registering Player Action]^";
#Endif;
                    NpcAct__ActionList.npcact__state = NPCACT__REGISTRATION,
                    NpcAct__ActionList.npcact_Register( NpcAct__ImpAction );
                    self.npcact__playerRegistered = true;
                    NpcAct__ImpAction.npcact__state = NPCACT__ACTION_IN_BEFORE;
                }
                
                NpcAct__SetFlag( npcact__v_seenplayer, _monitor.npcact__index,
                    true );
                
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: Calling Before on ",(name)_monitor," for Player Action]^";
#Endif;
                _ret = NpcAct__CallMonitor( _monitor, NPCACT__BEFORE );
#Iftrue NPCACT_TRACE_PLAYER >= 3;
print "[NpcAct Player: react_before -> ~",(name)_monitor,"~.Before returned ",_ret,"]^";
#Endif;
                if (_ret == false)
                {
                    _ret = NpcAct__ImpAction.npcact_IsCanceled();
#Iftrue NPCACT_TRACE_PLAYER >= 3;
print "[NpcAct Player: was player action canceled? ",_ret,"]^";
#Endif;
                }
                
#Iftrue NPCACT_TRACE_PLAYER >= 2;
print "[NpcAct Player: react_before will return ",_ret,"]^";
#Endif;

                return _ret;
            ],

        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        npcact__RunRemainingActions
            [
                _x; ! locals
                
                while (NpcAct__ActionList.npcact_CurrentAction() ~= nothing)
                {
                    _x = NpcAct__ActionList.npcact_CurrentAction().
                        npcact_source;
#Iftrue NPCACT_TRACE_LOGIC >= 3;
print "[NpcAct Logic: Calling remaining action on monitor/character ",(name)_x,"]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;
                    self.npcact__ActionLogic( _x );
                }
            ],
        
        
        !----------------------------------------------------------------------
        ! CharacterList Properties
        !----------------------------------------------------------------------
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! resets the list for the begining of a turn.
        npcact__ResetCharacterList
            [;
                self.npcact__current = self.npcact__head;
            ],
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ! returns the next character in the list, and advances the list.
        npcact__NextCharacter
            [
                _npc; ! locals
                
                _npc = self.npcact__current;
                if (_npc ~= nothing)
                {
                    self.npcact__current = _npc.npcact__next;
                }
                return _npc;
            ],
            
        
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        private npcact__head nothing,
        private npcact__current nothing,
        private npcact__flagcount 0,
        private npcact__playerRegistered false,
    has
    ;
    

    
        
!------------------------------------------------------------------------------
! Private helper functions
!------------------------------------------------------------------------------

! Flag code - based on flags.h by Adam Cadre, with my own optimizations
[ NpcAct__SetFlag
    flags bit onoff     ! parameters
    index bitmask count;      ! locals
    
    ! 2^3 = 8
    
    count = -3;
    @log_shift bit count -> index;
    bitmask = bit & 7;
    @log_shift 1 bitmask -> bitmask;
    
    if (onoff)
    {
        flags->index = flags->index | bitmask;
    }
    else
    {
        flags->index = flags->index & ~bitmask;
    }
];

[ NpcAct__FlagOn
    flags bit        ! parameters
    index bitmask count;   ! locals
    
    count = -3;
    @log_shift bit count -> index;
    bitmask = bit & 7;
    @log_shift 1 bitmask -> bitmask;
    
    return (flags->index & bitmask ~= 0);
];


Constant NPCACT__BEFORE   0;
Constant NPCACT__CANCELED 1;
Constant NPCACT__AFTER    2;
Constant NPCACT__TURN     3;

! Calls the monitor's call-back method, and sets up the list's bits
[ NpcAct__CallMonitor
    _monitor _routType ! parameter
    _actionIsLocked ! optional parameter
    _mask _act;  ! locals

#Iftrue NPCACT_TRACE_LOGIC >= 4;
print "[NpcAct Logic: Calling Call-Back on monitor/character ",(name)_monitor," with routine ",_routType,"]^";
NpcAct__ActionList.npcact_CurrentAction();
#Endif;

    
#Ifdef NPCACT_CHECK_ERRORS;
    if (_monitor == nothing)
    {
        "** Nothing monitor being called. **^";
    }
    if (_routType ~= NPCACT__TURN)
    {
        if (~~(_monitor ofclass NpcAct_Monitor))
        {
            "** Monitor ",(name)_monitor," without Monitor class being
                called. **^";
        }
        if (_monitor hasnt npcact_att_monitor)
        {
            "** Monitor ",(name)_monitor," without Monitor attribute being
                called. **^";
        }
    }
    else
    {
        if (~~(_monitor ofclass NpcAct_Character))
        {
            "** Character ",(name)_monitor," without Character class being
                called. **^";
        }
        if (_monitor hasnt npcact_att_character)
        {
            "** Character ",(name)_monitor," without Character attribute being
                called. **^";
        }
    }
#Endif;
    
    
    _mask = NPCACT__REGISTRATION_NONE;
    switch (_routType)
    {
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        NPCACT__BEFORE:
            if (_actionIsLocked == true)
            {
                _mask = NPCACT__REGISTRATION_NONE;
            }
            else
            {
                _mask = NPCACT__REGISTRATION_CANCEL;
            }
            if (_monitor ofclass NpcAct_Character)
            {
                _mask = _mask |
                    NPCACT__REGISTRATION_BEFORE | NPCACT__REGISTRATION_AFTER;
            }
            NpcAct__ActionList.npcact__state = _mask;
            
            if (NpcAct__CanCallMethod( _monitor, npcact_Before ) == false)
            {
                ! don't execute action, since it's not really an action.
                rfalse;
            }
            
            return _monitor.npcact_Before( NpcAct__ActionList );
            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        NPCACT__CANCELED:
            if (_monitor ofclass NpcAct_Character)
            {
                _mask = NPCACT__REGISTRATION;
            }
            NpcAct__ActionList.npcact__state = _mask;
            _act = NpcAct__ActionList.npcact_CurrentAction();
            
            if (NpcAct__CanCallMethod( _monitor, npcact_Canceled ) == false)
            {
                ! don't execute action, since it's not really an action.
                rfalse;
            }
            
            return _monitor.npcact_Canceled( NpcAct__ActionList,
                _act.npcact__canceller );
            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        NPCACT__AFTER:
            if (_monitor ofclass NpcAct_Character)
            {
                _mask = NPCACT__REGISTRATION_AFTER;
            }
            NpcAct__ActionList.npcact__state = _mask;
            
            if (NpcAct__CanCallMethod( _monitor, npcact_After ) == false)
            {
                ! don't execute action, since it's not really an action.
                rfalse;
            }
            
            return _monitor.npcact_After( NpcAct__ActionList );
            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        NPCACT__TURN:
            _mask = NPCACT__REGISTRATION;
            NpcAct__ActionList.npcact__state = _mask;
            
            if (NpcAct__CanCallMethod( _monitor, npcact_Turn ) == false)
            {
                ! don't execute action, since it's not really an action.
                rfalse;
            }
            
            return _monitor.npcact_Turn( NpcAct__ActionList );
            
        !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        default:
#Ifdef NPCACT_CHECK_ERRORS;
            "** Unknown routine (",_routType,") called on monitor ",
                (name)_monitor,". **^";
#Endif;
    }
];


! Checks if the method can be called on the object
[ NpcAct__CanCallMethod
    _obj _method; ! parameters
    
    return (_obj provides _method &&
        _obj._method ~= nothing &&
        metaclass(_obj._method) == Routine);
];

Endif; ! NPCACT_INCLUDED
