#charset "us-ascii"
#include <adv3.h>
#include <en_us.h>

/* 
 *   This file mainly defines new actions used in the game, together with 
 *   modified or extended grammar for existing actions.
 */

DefineIAction(Crawl)
    execAction()
    {
        "There\'s no need to crawl --- you can walk. ";
    }
;

VerbRule (Crawl)
    'crawl' | 'creep'
    : CrawlAction
    verbPhrase = 'crawl/crawling'
;

VerbRule (CrawlDir)
    ('crawl' | 'creep') singleDir
    : CrawlAction
    verbPhrase = 'crawl/crawling around'
;

DefineTAction(GoUnder);

VerbRule (GoUnder)
    ('go' | 'crawl' | 'get' | 'creep') 'under' singleDobj
    : GoUnderAction
    verbPhrase = 'go/going under (what)'
;


/* For the brake lever: */
DefineTAction(Release);

VerbRule (Release)
    'release' dobjList
    : ReleaseAction
    verbPhrase = 'release/releasing (what)'
;



/* New synonyms for 'take': */
VerbRule(Grab)
    ('grab' | 'hold') dobjList
    : TakeAction
    verbPhrase = 'grab/grabbing (what)'
;

/* To allow 'jump off of': */
VerbRule(JumpOffOf)
    'jump' 'off' 'of' singleDobj
    : JumpOffAction
    verbPhrase = 'jump/jumping (off of what)'
    askDobjResponseProd = singleNoun
;

/* And how about 'discuss' as a synonym for 'tell about'? */
VerbRule(Discuss)
    ('discuss') singleTopic 'with' singleDobj
    : TellAboutAction
    verbPhrase = 'discuss/discussing (what) (with whom)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
;

VerbRule(DiscussImplicit)
    'discuss' singleTopic
    : TellAboutAction
    verbPhrase = 'discuss/discussing (what) (with whom)'
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty direct object phrase */
        dobjMatch = new EmptyNounPhraseProd();
        dobjMatch.responseProd = singleNoun;
    }
;

VerbRule(DiscussWhat)
    [badness 500] 'discuss' singleDobj
    : TellAboutAction
    verbPhrase = 'discuss/discussing (what) (with whom)'
    askDobjResponseProd = singleNoun
    omitIobjInDobjQuery = true
    construct()
    {
        /* set up the empty topic phrase */
        topicMatch = new EmptyNounPhraseProd();
        topicMatch.responseProd = aboutTopicPhrase;
    }
;

/* 
 *   This is almost useless, but not quite, as we're going to make the 
 *   orange work a little more nicely:
 */
DefineTAction(Peel);

VerbRule(Peel)
    'peel' singleDobj
    : PeelAction
    verbPhrase = 'peel/peeling (what)'
;

/* Also For the orange: */
DefineTAction(FruitPick);

VerbRule(FruitPick)
    'pick' singleDobj
    : FruitPickAction
    verbPhrase = 'pick/picking (what)'
;


/* To allow 'put X near Y' or 'put X by Y', for instance: */
VerbRule(PutNear)
    ('put' | 'place' | 'set') dobjList ('near' | 'by' | 'beneath' | 'underneath')
    singleIobj
    : PutUnderAction
    verbPhrase = 'put/putting (what) (near what)'
;

VerbRule(MoveNear)
    ('push' | 'move') dobjList ('near' | 'by' | 'beneath' | 'underneath') singleIobj
    : MoveToAction
    verbPhrase = 'move/moving (what) (to what)'
    askIobjResponseProd = toSingleNoun
    omitIobjInDobjQuery = true
;

/* 
 *   Adding 'roll,' which has no value except as a synonym for 'push' with 
 *   the wheelchair and skateboard:
 */
VerbRule(RollTravelDir)
    ('roll') singleDobj singleDir
    : PushTravelDirAction
    verbPhrase = ('roll/rolling (what) ' + dirMatch.dir.name)
;

VerbRule(RollTravelClimbUp)
    ('roll') singleDobj
    'up' singleIobj
    : PushTravelClimbUpAction
    verbPhrase = 'roll/rolling (what) (up what)'
    omitIobjInDobjQuery = true
;

VerbRule(Roll)
    ('roll') dobjList
    : PushAction
    verbPhrase = 'roll/rolling (what)'
;

VerbRule(RollTo)
    ('roll') dobjList ('to' | 'under' | 'underneath' | 'near' | 'beneath') singleIobj
    : MoveToAction
    verbPhrase = 'roll/rolling (what) (to what)'
    askIobjResponseProd = toSingleNoun
    omitIobjInDobjQuery = true
;

/* For the jar: */
VerbRule(Smash)
    ('smash' | 'shatter') dobjList
    : BreakAction
    verbPhrase = 'smash/smashing (what)'
;

/* Likewise, for the jar, in order to allow 'throw jar against wall': */
replace VerbRule(ThrowAt)
    ('throw' | 'toss') dobjList ('at' | 'against') singleIobj
    : ThrowAtAction
    verbPhrase = 'throw/throwing (what) (at what)'
    askIobjResponseProd = atSingleNoun
;

DefineTAction(Shake);

VerbRule(Shake)
    'shake' singleDobj
    : ShakeAction
    verbPhrase = 'shake/shaking (what)'
;

/* ...and because the jar is on a shelf: */
VerbRule(LookOn)
    'look' 'on' singleDobj
    : ExamineAction
    verbPhrase = 'look/looking on (what)'
;

/* A tester wanted to 'turn on tv with remote': */
DefineTIAction (TurnOnWith);

VerbRule(TurnOnWith)
    (('activate' | ('turn' | 'switch') 'on') dobjList
    | ('turn' | 'switch') dobjList 'on') ('with' | 'using') singleIobj
    : TurnOnWithAction
    verbPhrase = 'turn/turning on (what) (with what)'
;
       

/* 
 *   Added 'rake' and 'shift' to the possible wording, also 'scoop' and 
 *   'scoop up'.
 */
VerbRule(GetWith)
    ('get' | 'take' | 'rake' |  'shift' | 'scoop') singleDobj ('with' | 'using')
    singleIobj
    : MoveWithAction
    verbPhrase = 'get/getting (what) (with what)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
    omitIobjInDobjQuery = true
;

VerbRule(ScoopUpWith)
    'scoop' 'up' singleDobj ('in' | 'with') singleIobj
    : MoveWithAction
    verbPhrase = 'scoop/scooping up (what) (in what)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
    omitIobjInDobjQuery = true
;    

VerbRule(Ascend)
    'ascend' singleDobj
    : ClimbUpAction
    verbPhrase = 'ascend/ascending (what)'
    askDobjResponseProd = singleNoun
;

VerbRule(Descend)
    'descend' singleDobj
    : ClimbDownAction
    verbPhrase = 'descend/descending (what)'
    askDobjResponseProd = singleNoun
;

modify VerbRule(Climb)
    'climb' ('over' | ) singleDobj
    :
;

modify VerbRule(MoveTo)
    ('push' | 'move') dobjList ('to' | 'under' | 'beneath' ) singleIobj
    :
;

DefineTAction(Ring);

VerbRule(Ring)
    'ring' singleDobj
    : RingAction
    verbPhrase = 'ring/ringing (what)'
;



DefineTIAction(FillWith)
;

VerbRule(FillWith)
    'fill' singleDobj ('with' | 'from') singleIobj
    : FillWithAction
    verbPhrase = 'fill/filling (what) (with what)'
;
VerbRule(DipIn)
    ('dip' | 'submerge') singleDobj 'in' singleIobj
    : FillWithAction
    verbPhrase = 'dip/dipping (what) (in what)'
;

VerbRule(FillWithWhat)
    [badness 500] 'fill' singleDobj
    : FillWithAction
    verbPhrase = 'fill/filling (what) (with what)'
    construct()
    {
        /* set up the empty indirect object phrase */
        iobjMatch = new EmptyNounPhraseProd();
        iobjMatch.responseProd = withSingleNoun;
    }
    announceDefaultObject(obj, whichObj, resolvedAllObjects)
    {
        if(whichObj == IndirectObject)
            return spSuffix('from')
            + obj.getAnnouncementDistinguisher().theName(obj);
        return inherited(obj, whichObj, resolvedAllObjects);
    }
;

DefineTAction(Empty)
;
VerbRule(Empty)
    'empty' singleDobj
    : EmptyAction
    verbPhrase = 'empty/emptying (what)'
;

DefineTAction(KnockOn)
;

VerbRule(KnockOn)
    'knock' ('on'|'at'|) singleDobj
    : KnockOnAction
    verbPhrase = 'knock/knocking (on what)'
;

/* Allow SKATE <DIRN>. */

DefineAction(Skate, TravelAction)
    checkAction()
    {
        if(!gActor.canSee(skateboard))
            failCheck('There\'s no skateboard in sight. ');
        
        if(!gActor.canTouch(skateboard))
            failCheck('You don\'t have your skateboard. ');
        
        if(!gActor.getOutermostRoom.ofKind(OutdoorRoom))
            failCheck(skateboard.cannotRideIndoorsMsg);
    }
    
    execAction()
    {
        if(skateboard.location != gActor.getOutermostRoom)
            tryImplicitAction(PutOn, skateboard, gActor.getOutermostRoom.roomFloor);
        
        if(!gActor.isIn(skateboard))
            tryImplicitAction(StandOn, skateboard);
        
        if(!gActor.isIn(skateboard))
            failCheck('You need to be standing on your skateboard to do that.
                ');      
                
        local oldRoom = gActor.getOutermostRoom();
        local action = new TravelDirAction(getDirection);
        action.doActionOnce();
        if(gActor.getOutermostRoom != oldRoom)
            reportBefore('You skate ' + getDirection.name + '. ');
            
    }
    failCheck(msg) { delegated Thing(msg); }
;

VerbRule(Skate)
    ('skate' | 'skateboard') singleDir
    : SkateAction
    verbPhrase = 'skate/skating (where)'
;

DefineTAction(Tear)
;

VerbRule(Tear)
    (('tear' | 'rip') ('up' | ) dobjList) |
    ('tear' | 'rip') dobjList 'up'
    : TearAction
    verbPhrase = 'tear/tearing (what)'
;

modify VerbRule(Clean)
    ('clean' | 'wipe' | 'dust') dobjList
    : 
;

modify VerbRule(MoveWith)
    ('move' | 'push' | 'prod' | 'poke')  singleDobj 'with' singleIobj
    :
;

/* Modified verb rule to add additional phrasings. */
modify VerbRule(PutIn)
    ('put' | 'place' | 'set'| 'insert' ) dobjList
        ('in' | 'into' | 'in' 'to' | 'inside' | 'inside' 'of'|'through' |
         'thru') singleIobj
    :
;

/* Extended phrasing of GoThrough */
modify VerbRule(GoThrough)
    ('walk' | 'go' |'climb' | 'clamber' | 'crawl' ) ('through' | 'thru')
        singleDobj
    : 
;

/* Extend Examine grammar */
modify VerbRule(Examine)
    ('examine' | 'inspect' | 'x' | 'look' 'at' | 'l' 'at' | 'look' | 'view'
     'see' | 'l' ) dobjList
    :
;

modify VerbRule(PutInWhat)
    [badness 500] ('put' | 'place' | 'insert') dobjList
    :
;

modify VerbRule(TalkTo)
    ('greet' | 'call' | 'say' 'hello' 'to' | 'talk' 'to') singleDobj
    :
;

/* 
 *   Added handling for these tired old magic words, just in case someone 
 *   insists on trying them.
 */
DefineIAction(XYZZY)
    execAction()
    {
        response.doScript();   
    }
    response: StopEventList {
    [
        'You\'ve obviously done this kind of thing before; unfortunately that
        magic word doesn\'t work here (after all, this isn\'t exactly a
        <i>Colossal Adventure</i>). ',
        'As I said, that magic word doesn\'t work here; perhaps you should try
        something more constructive. ',
        'What part of <.q>that magic word doesn\'t work here<./q> do you find so
        hard to understand? ',
        'Okay, you asked for it: suddenly there\'s a loud bang and a cloud of
        vile-smelling puce-coloured smoke, which slowly clears away to reveal
        that... <i>absolutely nothing has changed!</i> Happy now? (Of course
        you only imagined it all). '
    ]
    }
    /* Since this doesn't do anything, we won't make it consume any time */
    actionTime = 0
;

VerbRule(XYZZY)
    ('say' | 'cry' | 'incant' | ) ('xyzzy' | 'plover')
    : XYZZYAction
    verbPhrase = 'say/saying XYZZY'
;

/* 
 *   Add handling for another newbie verb ... note that in release 1, it 
 *   doesn't work very consistently!
 */
DefineTAction(GoTo)    
;

VerbRule(GoTo)
    ('go' | 'walk' | 'move') ('over'|'down'|'up'| ) 
    ('to' | 'towards' | 'around' |'past') singleDobj
    : GoToAction
    verbPhrase = 'go/going (to what)'
;

/* Extend VagueTravel grammar */
modify VerbRule(VagueTravel) 
    ('go' | 'walk') ('around'|'about'|'away'| )
    : 
;

/* Remove GO TO from the phrasing of EnterAction */
modify VerbRule(Enter)
    ('enter' | 'in' | 'into' | 'in' 'to'
     | ('walk' | 'go') ('in' | 'in' 'to' | 'into')) singleDobj
    :
;

VerbRule(Carry)
    'carry' dobjList
    : TakeAction
    verbPhrase = 'carry/carrying (what)'
;

DefineTAction(CarryDir);

VerbRule(CarryDir)
    'carry' dobjList singleDir
    : CarryDirAction
    verbPhrase = ('carry/carrying (what) ' + dirMatch.dir.name)
;

/* 
 *   Adding 'slip' so people can try to slip the flyer under the door.  
 *   And while we're at it, let's allow 'beneath'.
 */
replace VerbRule(PutUnder)
    ('put' | 'place' | 'set' | 'slip') dobjList ('under' | 'beneath') singleIobj
    : PutUnderAction
    verbPhrase = 'put/putting (what) (under what)'
;

DefineTAction(Water);

VerbRule(Water)
   ('water' | 'irrigate' | 'moisten') singleDobj
    : WaterAction
    verbPhrase = 'water/watering (what)'
;

/* 
 *   Allowing 'pry' to move the keys: 'open with' is now a synonym for 'pry 
 *   with', mainly for the jar; if we had anything else that could be opened 
 *   or pried, this might get weird.
 */
VerbRule(PryWith)
    ('pry' | 'open') singleDobj 'with' singleIobj
    : MoveWithAction
    verbPhrase = 'pry/prying (what) (with what)'
    askDobjResponseProd = singleNoun
    askIobjResponseProd = withSingleNoun
    omitIobjInDobjQuery = true
;

/* For the ocarina: */
DefineTAction(Play);

VerbRule(Play)
    ('play' | ('blow' ('in' | 'into'))) singleDobj
    : PlayAction
    verbPhrase = 'play/playing (what)'
;

/* New grammar for 'play music on ocarina': */
DefineTIAction(PlayOn);

VerbRule(PlayOn)
    'play' singleDobj ('on' | 'using' | 'with') singleIobj
    : PlayOnAction
    verbPhrase = 'play/playing (what) (on what)'
;

/* For the hairbrush: */

DefineTAction(Brush);

VerbRule(Brush)
    ('brush' | 'comb') dobjList
    : BrushAction
    verbPhrase = 'brush/brushing (what)'
;

DefineTIAction(BrushWith)
;

VerbRule(BrushWith)
    ('brush' | 'comb') dobjList 'with' singleIobj
    : BrushWithAction
    verbPhrase = 'brush/brushing (what) (with what)'
;

/* To allow phrasing ATTACH X TO Y WITH SCREWDRIVER */

DefineTIAction(AttachWithScrewdriver)
;

VerbRule(AttachWithSrewdriver)
    ('attach' | 'fix' | 'fasten' | 'screw') singleDobj 'to' singleIobj
    'with' ('the' |) 'screwdriver'
    : AttachWithScrewdriverAction
    verbPhrase = 'attach/attaching (what) (to what)'
;


/* Add handling to Thing for all the new actions we've just defined above */

modify Thing
    dobjFor(GoUnder)
    {
        verify()
        {
            illogical('{That dobj/he} {is} not something you can get under. ');
        }
    }

    dobjFor(Release) asDobjFor(Drop)

    dobjFor(Peel) 
    {
        preCond = [touchObj]
        verify() {  illogical('That\'s not something you can peel. ');   }
    }

    dobjFor(Shake) 
    {
        preCond = [touchObj]
        verify() {}
        check() 
        {
            failCheck ('Shaking {the dobj/him} has no obvious effect. ');
        }
    }

    dobjFor(TurnOnWith) 
    {
        preCond = [objVisible]
        verify() {}
        check() 
        {
            failCheck ('{The dobj/he} is not something that can be activated
                indirectly. ');
        }
    }
    
    iobjFor(TurnOnWith) 
    {
        preCond = [objHeld]
        verify() {}
        check() 
        {
            failCheck ('{The iobj/he} cannot be used for activating things
                indirectly. ');
        }
    }

    dobjFor(Ring) 
    {
        verify() {  illogical ('{The dobj/he} cannot be rung. ');  }
    }

    dobjFor(Empty) 
    {
        preCond = [objHeld]
        verify() {illogical(cannotEmptyMsg); } 
    }
    cannotEmptyMsg = '{That dobj/he} {is} not something {you/he} can empty. '
    
    dobjFor(Tear)
    {
        preCond = [touchObj]
        verify() 
        {
            if (!isPaper) 
                illogical(cannotTearMsg);
        }
        check() 
        {
            failCheck('Better not --- you might need it later. ');
        }
    }
    cannotTearMsg = '{That dobj/he} {is} not something {you/he} can tear. '
    
    dobjFor(FillWith)  {  verify() {  illogical(cannotFillMsg);  } }
    iobjFor(FillWith)  { verify() { illogical(cannotFillWithMsg); } }
    cannotFillMsg = '{That dobj/he} {is} not something {you/he} can fill. '
    cannotFillWithMsg = '{You/he} can\'t fill anything with {that iobj/him}. '
    
    dobjFor(KnockOn)
    {
        preCond = [touchObj]
        verify() {}
        action()  {   mainReport(knockNoEffectMsg);  }
    }
    knockNoEffectMsg = 'This has remarkably little effect. '
    
    dobjFor(Examine)
    {
        action()
        {
            inherited();
            wordingCheck();
        }
    }   
    
    wordingCheck()
    {
        if(gAction.getEnteredVerbPhrase == 'look (dobj)' &&
           !wordingCheckExplained)
        {
            wordingCheckExplained = true;
            "<.p>(I assume you meant LOOK AT <<gDobj.theName.toUpper>>, which
            you can abbreviate to X <<gDobj.name.toUpper>>. LOOK really needs
            to be followed by AT, IN, UNDER, BEHIND or THROUGH to make it clear
            what you mean)<.p>";
        }
    }
    wordingCheckExplained = nil
    
    
    dobjFor(GoTo)
    {
        preCond = [objVisible]
        verify() 
        {
            if(isIn(gActor.getOutermostRoom))
                illogicalNow('There\'s no need to use any such command to move
                    yourself around within a location. ');
        }
        
        action()
        {
            local conn = gActor.getOutermostRoom.getConnectorTo(gActor,
                getOutermostRoom);
            if(conn)
            {
                "(going to <<getOutermostRoom.destName>>)\b";
                nestedAction(TravelVia, conn);
                "<.p>(For future reference, it would be more reliable to use one
                of the standard compass-direction movement commands, like EAST
                or NORTH, to move around)<.p>";
            }
            else
                failCheck('{The dobj/he} {is} too far away for {you/him} to
                    reach like that; you\'ll have use explicit movement
                    commands (NORTH, SOUTH, EAST, WEST etc.) ');
        }
    }
    
    dobjFor(PushTravel)
    {
        verify()
        {
            if(isIn(gActor))
                illogical('Since you\'re carrying {it dobj/him} that seems
                    rather pointless. ');
            else
                illogical('It would be far simpler just to pick {it dobj/him} up
                    and carry {it dobj/him}. ');
            
        }
    }
        
    dobjFor(CarryDir) 
    {
        preCond = [objHeld]
        verify() {}
        action() 
        {
            "No need to be specific about carrying things in any particular
            direction; simply use a travel command (such as NORTH) while
            you\'re holding them. ";
        }
    }
    
    dobjFor(Water)
    {
        verify()
        { 
            illogical ('{The dobj/he} {is} not in need of watering. ');
        }
    }

    dobjFor(FruitPick)
    {
        verify()
        {
            illogical ('{The dobj/he} {is} not something you can pick. Possibly
                you meant to type \'pick up\'. ');
        }
    }
   
    dobjFor(Play)
    {
        verify()
        {
            illogical ('{The dobj/he} {is} not something you can play. ');
        }
    }
    
    dobjFor(PlayOn) 
    {
        verify() {}
        check() 
        {
            if (gIobj == ocarina) 
                failCheck ('{The dobj/he} is not something that can be played
                    on the ocarina. ');
            else
                failCheck ('That makes no sense! ');
        }
    }
    
    iobjFor(PlayOn) 
    {
        verify() {}
        check() 
        {
            failCheck ('{The iobj/he} is not a musical instrument. ');
        }
    }
   
   
    dobjFor(Brush) 
    {
        preCond = [touchObj]
        verify() {}
        check() {}
        action() 
        {
            if(brush.isHeldBy(gActor))
                replaceAction(BrushWith, self, brush);
            
            "You brush the surface of {the dobj/him} with your fingertips. ";
            
        }
    }
    
    dobjFor(BrushWith)
    {
        preCond = [touchObj]
        verify() {}
        check()   {    failCheck(futileToBrushWithMsg);  }
        action() { }
    }
    
    iobjFor(BrushWith)
    {
        preCond = [objHeld]
        verify() { illogical(cannotBrushWithMsg); }
    }
    
    futileToBrushWithMsg = 'Applying {the iobj/him} to {the dobj/him} seems
        rather unnecessary. '
    cannotBrushWithMsg = 'You can\'t very well brush anything with {that
        iobj/him}. '
    
    dobjFor(AttachWithScrewdriver)
    {
        preCond = [touchObj, new ObjectPreCondition(screwdriver, objHeld) ]
        
        action() { replaceAction(AttachTo, gDobj, gIobj); }
    }
    
    iobjFor(AttachWithScrewdriver) asIobjFor(AttachTo)
    
    
    tasteDesc = "{It dobj/he} really do{es}n't look like the kind of thing you
        want to put in your mouth. "
;

/* Allow GO TO decoration like GO TO THING */
modify Decoration
    dobjFor(GoTo)
    {
        verify() 
        { 
            inherited Thing; 
            logicalRank(80, 'decoration');
        }
        action() { inherited Thing; }
    }
;


/* Modified handling of 'Yell' */
modify YellAction
    checkAction()  {   gActor.getOutermostRoom.yellCheck();  }
    execAction()   {   gActor.getOutermostRoom.yellDesc();   }
;


modify Door
    dobjFor(KnockOn)
    {
        verify()
        {
            /* A door is a likely target of a Knock On command */
            logicalRank(120, 'knockable');
        }         
        
    }   
    knockNoEffectMsg = '{You/he} knock{s} on {the dobj/him}, but nobody answers. '
;

modify Enterable
    dobjFor(GoTo) asDobjFor(Enter)
;
    


//---------------------------------------------------------------------------------
// Debugging verbs start here.

#ifdef __DEBUG

DefineIAction(ShowAllHints)
    execAction()
    {
        me.showAllHints = true;
        "Showing all hints. ";
    }
;
VerbRule(ShowAllHints)
    'sah' | 'showallhints'
    : ShowAllHintsAction
    verbPhrase = 'show/showing all hints'
;

DefineIAction(RepairLift)
    execAction()
    {
        elevator.isWorking = true;
        "You cry <q>ROTAVELE!</q> in a loud voice, and the elevator is
        magically repaired. ";
    }
;

VerbRule(RepairLift)
    'repair' ('lift' | 'elevator') | 'rotavele'
    : RepairLiftAction
    verbPhrase = 'repair/repairing the elevator'
;

DefineTAction(MagicUnlock)
;

VerbRule(MagicUnlock)
    'magic' 'unlock' singleDobj
    : MagicUnlockAction
    verbPhrase = 'magically unlock/unlocking (what)'
;

DefineTAction(Banish)
;

VerbRule(Banish)
    'banish' dobjList
    : BanishAction
    verbRule = 'banish/banishing (what)'
;

modify Thing
    dobjFor(Banish)
    {
        verify() {}
        action()
        {
            "{The dobj/he} vanish{es} in a cloud of choking green smoke. ";
            moveIntoForTravel(nil);
        }
    }
;

DefineIAction(WaterElf)
    execAction()
    {
        elf.isWatered = true;
        elf.setCurState(elfAlert);
        elfAlert.previousState = elfWatered;
        "You gesture magically and cry, <.q>Ali-kazaam!<./q> The elf responds
        by becoming watered. ";
    }
;
VerbRule(WaterElf)
    'waterelf'
    : WaterElfAction
    verbRule = 'water/watering the elf'
;

modify Lockable
    dobjFor(MagicUnlock)
    {
        verify()
        {
            if(!isLocked)
                illogicalAlready('{The dobj/he} is not locked. ');
        }
        action()
        {
            makeLocked(nil);
            "OPEN SESAME! {The dobj/he} is magically unlocked.";
        }
    }
;

#endif