#include <adv3.h>
#include <en_us.h>


lawnWest: OutdoorRoom
    'Lawn, West End'
    'the west end of the lawn'
    'west end of the lawn'
    "Emma and her family live close to the middle of town, but you could hardly 
    tell by looking at their backyard. It's huge! The lawn stretches southwest 
    to a big tree with a swing hanging from it and east to the cubby house by the 
    fence.  There don't seem to be any good hiding places right here, but the 
    big clump of bushes to the northwest would make a good hiding spot. The back 
    door of the house, where Mum and the other grownups are, is south, but 
    inside is out of bounds, and so is the shed to the north."
    
    /* connections */
    northwest = lotsaBushes
    southwest = underTree
    east = lawnEast
    south = enterHouse
    north = enterShed
    
    /* wording */
    actorInPrep = 'at'
    
    /* vocabulary */
    vocabWords = 'west lawn back yard/backyard/end/lawn/west'
    
    /* ground */
    roomParts = static inherited - ordinaryGround + lawnGrass
;



lawnEast: OutdoorRoom
    'Lawn, East End'
    'the east end of the lawn'
    'east end of the lawn'
    "There aren't many places to hide <<gActor.isIn(self) ? '' : 't'>>here. 
    The lawn is big and neat and empty 
    and although there are a few plants around the edges, they're much too 
    small to hide anybody. The shed to the north would be a good place to hide,
    but it's out of bounds. The cubby house to the northeast or the bushes or
    big tree at the other end of the lawn would probably be much better places 
    to look. "
    
    /* connections */
    west = lawnWest
    northeast = cubbyArea
    north = enterShed
    south = enterHouse
    
    /* wording */
    actorInPrep = 'at'
    
    /* vocabulary */
    vocabWords = 'east lawn end/lawn/east'
    
    /* ground */
    roomParts = static inherited - ordinaryGround + lawnGrass
;


/* 
 *   let the player refer to the "other end of the lawn" and handle it 
 *   gracefully
 */
grammar nounWord(otherEndOfLawn):
    'other' 'end' ('of' ('the' | ) 
                   ('lawn' | ('back' | ) 'yard' | 'backyard') | )
    : SpecialNounPhraseProd
    
    /* 
     *   If the player is at the east end of the lawn, match the west end, 
     *   or vice versa. If she's somewhere else, don't match at all.
     */
    getMatchList()
    {
        local loc = gPlayerChar.getOutermostRoom();
        if (loc == lawnWest)
            return [lawnEast];
        if (loc is in (lawnEast, cubbyArea))
            return [lawnWest];
        return [];
    }
    
    /* 
     *   this futzing around is apparently necessary because there are 
     *   different possible ways for the player to enter this phrase
     */
    getAdjustedTokens() 
    {
        local origToks = getOrigTokenList();
        local adjToks = new Vector(origToks.length());
        
        foreach (local tok in origToks) 
        {
            adjToks.append(tok[3]);
            if (tok[3] is in ('end', 'lawn', 'yard', 'backyard'))
                adjToks.append(&noun);
            else
                adjToks.append(&adjective);
        }
        
        return adjToks;
    }
;



cubbyArea: OutdoorRoom
    'By the Cubby House'
    'the area with the cubby house'
    'area with the cubby house'
    "This is a really cool cubby house. It's big and has a ladder on one side 
    and a slide on the other side that goes down into a sandpit. Behind it 
    there's a bush that would make a good hiding spot.<.p>
    The lawn stretches off to the southwest and a little path leads south to the 
    vegetable garden. Or you could look behind the shed to the northwest. "
    
    /* connections */
    northwest = behindShed
    southwest = lawnEast
    south = gardenPathN
    up: FakeConnector
    {
        travelDesc()
        {
            replaceAction(Enter, cubbyHouse);
        }
    }
    
    actorInName = 'near the cubby house'
;


+ gardenPathN: PathPassage
    -> gardenPathS
    'little narrow brick path'
    'path'
    "A narrow brick path runs south towards the vegetable garden. It looks
    new --- all the bricks lie flat and there are not weeds growing between
    them. "
;



behindShed: OutdoorRoom
    'Behind the Shed'
    'the area behind the shed'
    'back of the shed'
    "The grass here hasn't been cut for a long time. It tangles around your 
    feet and nearly trips you up sometimes. There might be snakes in it or 
    anything --- well, maybe not a person, but anything smaller. There's a lot of 
    bushes to the southwest and the cubby house is off to the southeast. The 
    only place anyone could really hide here is behind a pile of old junk leaning 
    up against the shed. "
    
    /* connections */
    southeast = cubbyArea
    southwest = lotsaBushes
    
    actorInName = 'behind the shed'
    actorIntoName = 'into the area behind the shed'
;



lotsaBushes: OutdoorRoom
    desc = "<<gActor.isIn(self) ? 'This' : 'That'>> part of the backyard is 
        filled with a bunch of bushes and stuff.
        A squiggly line of rocks marks off a rough triangle in the corner of 
        the yard --- the lawn is outside the triangle, and inside is just heaps
        of bushes, growing wild. When Emma was showing you around, she said 
        this was a special kind of garden. Something like <q>wildebeest 
        garden</q> or <q>wilfulness garden</q>, some long word. It doesn't 
        look very gardenish, though.
        <<gActor.isIn(self) ? '<.p>Anyway, there are plenty of hiding places in 
            these bushes --- looking in them all will take a while. There\'s a 
            big tree to the south, or you can try 
            behind the shed to the northeast. The lawn is southeast.' : ''>> "
    
    /* 
     *   room name - randomly display "wilderness garden", "wildebeest 
     *   garden", or "wilfulness garden"
     */
    nameList = ['wilderness', 'wilfulness', 'wildebeest']
    
    roomName()
    {
        return name().allToTitleCase();
    }
    
    name()
    {
        local nm = rand(nameList) + ' garden';
        lastName = nm;
        return nm;
    }
    
    /* 
     *   Don't regenerate the name for the status line - use whichever name 
     *   we printed last.
     */
    lastName = nil
    
    statusName(actor)
    {
        /* say the last-used name in title case */
        say(lastName.allToTitleCase());
        
        /* add the actor's posture parenthetically if necessary */
        actor.actorRoomNameStatus(self);
    }
    
    /* understand all variation on "wilderness garden" */
    vocabWords = 'wilfulness wilderness wildebeest garden'
    
    /* connections */
    northeast = behindShed
    south = underTree
    southeast = lawnWest
;



underTree: OutdoorRoom
    'Under the Big Tree'
    'the base of the big tree'
    'big tree'
    "This tree must be pretty old. It's much too big for you to get your arms 
    around it, and the ground is all lumpy from the roots running underground.
    The lowest branches are just out of reach, but if you stood on the swing 
    hanging from one of them you could probably pull yourself up. The leaves 
    are thick enough that someone could hide up there.<.p>
    The lawn is northeast and the front garden is southeast. To the north is a
    big patch of bushes. "
    
    /* connections */
    north = lotsaBushes
    northeast = lawnWest
    southeast = frontGarden
    up()
    {
        return ((gActor && gActor.isPlayerChar) ? climbTree : upTree);
    }
    
    actorInPrep = 'beneath'
    
    roomParts = static inherited - ordinaryGround + lumpyGround
;



climbTree: TravelMessage, EventList
    destination = upTree
    eventList = [
        new function
        {
            "You <<gActor.isIn(swing) && (gActor.posture == standing) ?
              '' : 'stand on the swing and '>> grab the branch. Bracing 
            yourself against the tree trunk, you pull yourself up as hard as 
            you can. Not quite... and then, with a last desperate effort, you 
            scramble onto the branch. From here, it's easy. ";
        },
        'You clamber up into the leaves. You\'re getting good at this. '
    ]
;



upTree: OutdoorRoom
    'In the Big Tree'
    'the top of the big tree'
    'top of the big tree'
    
    desc()
    {
        "You perch on a thick branch, clinging tightly with your knees. It's cooler up
        here, shady, with a gentle breeze that ruffles your hair. Peering down through the 
        leaves, you can see everything, but you're pretty sure no one can see you.
        Neighbourhood noises trickle through: a dog barks, a car slows down, somebody
        laughs. ";
        /* 
         *   TODO: remove or rewrite the list of noises? it seems a bit odd 
         *   that you hear the same noises every time you LOOK (maybe select 
         *   a few noises randomly from a list, or is that ridiculous 
         *   overkill for a simple effect?)
         */
        
        /* 
         *   TODO: reconsider this - does it imply something more than "you 
         *   and Tiana can hear each other in this position"? like "Tiana is 
         *   hurt, you should help her" or "there's something interesting 
         *   down in the bushes, you should investigate"?
         */
        if (!talkToTianaHinted && tiana.isIn(lotsaBushes) && 
            !tiana.isPlayingHnS)
        {
            "Down in the patch of bushes, {the Tiana/she} stumbles over 
            something and you hear her mutter, <q>Ow!</q> ";
            talkToTianaHinted = true;
        }
    }
    
    /* 
     *   The first time the player is here, if Tiana is down in the bushes, 
     *   give a hint that you can talk to her without climbing down the tree.
     */
    talkToTianaHinted = nil
    
    /* connections */
    down: TravelMessage
    {
        -> underTree
        "{You/he} wriggle{s} off the branch and drop{s} to the ground. "
        
        /* You don't need to stand up before climbing down the tree. */
        actorTravelPreCond(actor)
        {
            return [];
        }
    }
    
    actorInPrep = 'at'
    
    /* 
     *   There's no floor at the top of the tree - obviously. (Code stolen 
     *   from the adv3 Floorless class, which isn't appropriate here because 
     *   I don't want dropped things to fall to the bottom of the tree.)
     */
    roomParts = perInstance(inherited().subset({x: !x.ofKind(Floor)}))
    
    /* The only possible posture at the top of the tree is sitting. */
    defaultPosture = sitting
;



frontGarden: OutdoorRoom
    'Front Garden'
    "The front garden is almost as quiet as the back yard. Occasional cars pass
    by, but most of the time the road is empty. It's a much more formal garden 
    than the back, though. Flower beds edge the 
    lawn --- the plants are mostly far too small to hide anyone, but some of 
    the shrubs might. A stone bird bath squats in the middle of the lawn, and a
    path curves northeast around the house to the vegetable garden. The cars in 
    the driveway could be a good hiding place too. <<gActor.isIn(self) ? 'If not, 
        there\'s the big tree to the northwest. ' : ''>>"
    
    
    /* connections */
    northwest = underTree
    northeast = vegeGarden
    north = enterHouse
    south = road
    
    /* vocabulary */
    vocabWords = 'front yard/garden'
    
    /* ground */
    roomParts = static inherited - ordinaryGround + lawnGrass
;


+ gardenPathSW: PathPassage
    -> gardenPathNE
    'narrow little brick path'
    'path'
    "A little brick path runs northeast around the corner of the
    house. It looks new --- the bricks all lie flat and there are
    no weeds growing up between them. "
;



vegeGarden: OutdoorRoom
    'Vegetable Garden'
    "<<gActor.isIn(self) ? 'A path runs along the side of the house from 
    the front garden southwest to the cubby house at the north. ' : ''>>
    Between the path and the fence, vegetable plants of various kinds 
    stand in tidy rows. None of them are big enough to hide anyone, 
    though. "
    
    /* connections */
    north = cubbyArea
    southwest = frontGarden
    
    /* vocabulary */
    vocabWords = 'vege vegetable garden'
    
    dobjFor(Eat) remapTo(Eat, vegePlants)
    
    /* 
     *   When the player isn't cheating, Tiana should always get to the 
     *   hiding spot before the PC. If the player is deliberately hurrying, 
     *   this may require some trickery. So before entering the vege garden 
     *   from the front garden, check where Tiana is up to, and if she isn't 
     *   by the cubby ready to hide, make sure she is.
     *
     *   Yes, this is blatant authorial cheating.
     *
     *   The point of having the cheat happen at this point is so that the 
     *   player sees Tiana near the cubby before she hides, thus (hopefully) 
     *   creating an illusion that I DIDN'T just spirit her across the map.
     */
    enteringRoom(traveler)
    {
        /* if the PC is playing hide and seek, skip this code */
        if (traveler.isPlayingHnS())
            return;
        
        /* otherwise go ahead and cheat */
        if (traveler.isPlayerChar() && !traveler.isCheating &&
            !bushNearCubby.searchedBy(tiana))
        {
            /* move Tiana to the cubby area */
            if (!tiana.isIn(cubbyArea))
                tiana.moveIntoForTravel(cubbyArea);
            
            /* give her the knowledge of Emma's hiding spot */
            bushNearCubby.searchedByList += tiana;
            
            /* supposedly, the last thing she did was search the bush */
            tiana.lastAction = SearchAction.createActionInstance();
            tiana.lastAction.dobjCur_ = bushNearCubby;
        }
    }
;


+ gardenPathAmbig: PathPassage, AskConnector
    'narrow little brick path'
    'path'
    "The path curves round here, leaving the front garden to the 
    southwest and heading north towards the cubby house. It looks new --- 
    the bricks all lie flat and there are no weeds growing up between them. "
    
    /* ask about the other two ends of the path */
    travelAction = EnterAction
    travelObjs = [gardenPathS, gardenPathNE]
    
    askDisambig(targetActor, promptTxt, curMatchList, 
                fullMatchList, requiredNum, askingAgain, dist)
    {
        "The path leads in two directions from here. Which way do
        you want to go, north to the cubby house or southwest to
        the front garden? ";
    }
    
    /* 
     *   more likely than the other two paths, if the player doesn't mention 
     *   a specific path
     */
    vocabLikelihood = 50
;


+ gardenPathS: PathPassage
    'narrow little brick north n path to (cubby) (house) (the) 
    n/north/path'
    'path to the cubby house'
    desc = delegated gardenPathAmbig();
;


+ gardenPathNE: PathPassage
    'narrow little brick south-west southwest sw path to (the) (front) 
    (garden) (yard) south-west/southwest/sw/path'
    'path to the front garden'
    desc = delegated gardenPathAmbig();
;



enterHouse: DeadEndConnector
    apparentDestName = 'the house'
    travelDesc()
    {
        if (gPlayerChar.isPlayingHnS)
        {
            "You stick your head round the door, but Mum shoos you away. Emma's 
            mum did tell you all to play outside (and not get into mischief). ";
        }
        else
        {
            "You slip into the house and curl up next to Mum's chair. She glances
            at you and asks, <q>Tired?</q> but doesn't wait for an answer.<.p>
            A while later the others come in. <q>Why did you run off like that?</q>
            Emma demands. <q>We were waiting ages for you to come!</q><.p>
            You say nothing. ";
            endGame('You have given up');
        }
    }
    isOutOfBounds = true
;



enterShed: DeadEndConnector
    apparentDestName = 'the shed'
    travelDesc()
    {
        if (gPlayerChar.isPlayingHnS)
        {
            "You stick your head into the shed, but it's empty. 
            <<gPlayerChar.roleInHnS == roleHiding ? 'No good hiding spots there.
                Anyway, ' : 'And'>> Emma's dad probably wouldn't like you messing
            around in there. ";
        }
        else
        {
            "You slip into the shed, pull the door closed behind you, and curl up 
            in a corner. It's quiet in here. You didn't want to play that silly
            game anyway.<.p>
            It is a long time before the door opens again. <q>There you 
            are!</q> Emma says. <q>What were you doing? <i>You're</i> not supposed
            to be the one hiding, stupid! We were waiting ages and ages for 
            you!</q>";
            endGame('You have given up');
        }
    }
    isOutOfBounds = true
;



/* TODO: similar dead end for walking out onto the road? */



/* Darkness - the place you end up in when you close your eyes. */
dark: DarkRoom
    roomDarkName = 'Eyes Closed'
;



/* All of the areas have lines of sight to other areas. */
DistanceConnector [lawnWest, lawnEast, cubbyArea];
DistanceConnector [lawnWest, lawnEast, lotsaBushes, underTree, upTree];
DistanceConnector [underTree, upTree, frontGarden];
DistanceConnector [frontGarden, vegeGarden];
DistanceConnector [vegeGarden, cubbyArea];
DistanceConnector [cubbyArea, behindShed];
DistanceConnector [behindShed, lotsaBushes];


/* 
 *   Darkness is also sense-connected to several other rooms; this allows 
 *   the PC to talk to the other characters while her eyes are closed.
 */
DistanceConnector
    [dark, lawnWest, lawnEast, cubbyArea, underTree]
    
    /* Don't include visual information. */
    transSensingThru(sense)
    {
        if (sense == sight)
            return opaque;
        else
            return inherited(sense);
    }
;
