////////////////////////////////////////////////////////////////////////
//  
//  AltSkelB
//  Release:    1.0
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  This file based on the TADS Workbench Sample and is
//  (c) 1999, 2000 Michael J. Roberts and has been ported to
//  Alt and elaborated upon.
//
//  This file is one of the Alt Library demonstration games
//  and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#include <stdif.h>

#pragma C+
 
modify story
    startingLocation = entryway
    title = "A Basic Starter Game"
    headline = "An Interactive Fiction"
    introduction = "Welcome to TADS Alternate Library..."
;

/*
 *  Define the rucksack. Alt's SackItem class makes this a Wearable
 *  OpenableContainer that automatically tidies things away for the
 *  player when their hands are "full".
 */
rucksack: SackItem
    location = Me
    wearer = Me
    noun = 'rucksack' 'sack'
    sDesc = "rucksack"
;

/*
 *   We must define the initial location, where the player starts the
 *   game.
 *   
 *   We'll make entryway an object of class "Room", which is the basic
 *   class defined in Alt for a location in the game.  
 */
entryway: Room
    /* 
     *   The "sDesc" is the short description of the location.  This is
     *   displayed when the player enters or re-enters the location. 
     */
    sDesc = "Entryway"

    /* 
     *   The "lDesc" is the long description of the location.  This is
     *   displayed when the player enters the location for the first time,
     *   or types "look" while in the location. 
     */
    lDesc =
    {
        "This large, formal entryway is slightly intimidating:
        the walls are lined with somber portraits of gray-haired
        men from decades past; a medieval suit of armor";

        /* if the axe is in the armor, list it specially */
        if (axe.isIn(suitOfArmor))
            ", posed with battle axe at the ready,";

        " towers over a single straight-backed wooden chair.
        The front door leads back outside to the south.  A
        hallway leads north. There is a small sign here.";
    }

    /* 
     *   To the north is the hallway.  Set the "north" property to the
     *   destination room object.  Other direction properties that we
     *   could set: east, west, north, up, down, plus the diagonals: ne,
     *   nw, se, sw.  We can also set "in" and "out".  
     */
    north = hallway

    /* 
     *   To the south is the front door; set the "south" property to refer
     *   to the door object.
     */
    south = frontDoor

    /* 
     *   set the "out" direction to be the same as south, since going
     *   south leads outside 
     */
    out = frontDoor
;

/*
 *   Define the front door.
 *
 *  The frontDoor is a Doorway, not a Floating object, Doorway should
 *  define a location and leadsTo.
 */
frontDoor: Doorway
    /* the door is in the starting room */
    location = entryway

    /* define the words the player can use to refer to the door */
    noun = 'door'
    adjective = 'front'

    /* the short description */
    sDesc = "front door"

    /*
     *   Define where the door goes if you go through it.  Normally, we'd
     *   just set this property to a room object.  In this case, though,
     *   we don't want the player to be able to go through the door, since
     *   this is one of the boundaries of the game.  So, when the player
     *   tries to go through the door, simply display a message about why
     *   they can't, then return "nil" - this is a special value in TADS
     *   that means that the destination is empty.  
     */
    leadsTo = {
        "You'd rather stay in the house for now. ";
        return nil;
    }
    whenOpen = "The front door is open."
    whenClosed = "The front door is closed."
;

/*
 *   Define the chair.  The Alt "ChairItem" class is helpful here.
 */
chair: ChairItem
    location = entryway
    sDesc = "wooden chair"
    noun = 'chair'
    adjective = 'straight-backed' 'wooden'
;

/*
 *  Define Joe. Joe is an Actor class object, which means that he can't
 *  be carried around.
 */
joe: Actor
    location = entryway
    noun = 'joe'
    sDesc = "Joe"
    aDesc = "Joe"
    theDesc = "Joe"
    isHim = true
    actorDesc = "Joe is here, pacing the floor."
    actorAction(v, d, p, i) = {}
;

/*
 *  Define a ball. The Alt "Item" class is useful here. This object also
 *  defines an initial attribute, which displays until the Player
 *  Character moves the object.
 */
ball: Item
    location = entryway
    sDesc = "beach ball"
    noun = 'ball'
    adjective = 'beach'
    initial = "There's a beach ball here."
;

/*
 *  Define a glass bottle. Since the bottle is openable and made of
 *  glass we can use the Transparent interface along with
 *  OpenableContainer. Transparent defines a material attribute of 
 *  "glass", which allows sight to pass through the object, but not
 *  sound, smell, or touch.
 */
bottle: Transparent, OpenableContainer
    location = entryway
    sDesc = "glass bottle"
    noun = 'glass'
    adjective = 'bottle'
    whenOpen = "The glass bottle is open."
    whenClosed = "The glass bottle is closed."
;

/*
 *   Define the suit of armor.  It can't be moved, so it's a "FixedItem"
 *   (defined in Alt).  
 */
suitOfArmor: FixedItem
    location = entryway
    sDesc = "suit of armor"
    noun = 'suit' 'armor'
    adjective = 'medieval'
    lDesc = {
        "It's a suit of plate-mail armor that looks suitable
        for a very tall knight. ";

        /* if I'm holding the axe, so note */
        if (axe.isIn(self))
            "The armor is posed with a huge battle-axe held
            at the ready. ";
    }

    /* 
     *   This is a little unusual, but set the armor to be a "quiet
     *   container."  We initially have the axe located in the armor, even
     *   though the armor isn't a container in the normal sense.  The
     *   "Qcontainer" flag will keep the axe from being listed separately
     *   in the contents of the room. 
     */
    isQcontainer = true
;

/*
 *   Battle Axe 
 */
axe: Item
    sDesc = "battle axe"

    /* it starts out in the suit of armor */
    location = suitOfArmor

    /* 
     *   Define vocabulary.  Note that we provide some synonyms for things
     *   that really could be separate objects (the blade, the dried blood
     *   on the blade), but which aren't important enough to be
     *   implemented separately. 
     */
    noun = 'axe' 'blood' 'blade' 'edge'
    adjective = 'steel' 'battle' 'dried'
    lDesc = "It's a large steel battle axe.  A little bit of
             dried blood on the edge of the blade makes the authenticity
             of the equipment quite credible. "
;

/*
 *   Define the portraits.  We don't want to define several individual
 *   portraits, so define a single object that refers to the portraits
 *   collectively. 
 */
portraits: FixedItem
    location = entryway
    noun = 'portraits' 'pictures' 'men'
    adjective = 'somber' 'gray-haired'
    sDesc = "portraits"

    /* 
     *   set the "isThem" flag, so that default messages in adv.t refer to
     *   this object in the plural
     */
    isThem = true

    /* 
     *   Define an "aDesc": this is used in messages to refer to the
     *   object with an indefinite pronoun (such as "a" or "an").
     *   Normally, the Alt base class would define this as "a
     *   <<sDesc>>", but this isn't appropriate for a collective object -
     *   "a portraits" doesn't sound right.
     */
    aDesc = "a portrait"
;

/*
 *  Define a sign. Alt's Wysiwyg class makes it easy to create signs and
 *  maps. In this case, however, we use a ReadableItem class object and 
 *  generate the sign using Alt's box() function.
 */
sign: Fixture, ReadableItem
	location = entryway
	sDesc = "sign"
	noun = 'sign'
	readDesc = {
	    /*
	     *  The box() function generates a Wysiwyg display from the list
	     *  of single-quoted strings passed as its argument.
	     */
        box(['Welcome'
		    'To'
		    'ALT.T'
	        ]);
	}
;

/*
 *  A listGroup for listing hats together.
 *
 *  ListGroupParenCount - one of the pre-defined ListGroup classes
 */
hatListGroup: ListGroupParenCount
    groupPrefix = "hats"
;

/*
 *  A class for Hats
 */
class HatItem: Item
    listTogether = hatListGroup
    sDesc = "<<groupDesc>> hat"
    noun = 'hat'
    plural = 'hats'
    pluralDesc = "hats"
;

fez: HatItem 
    location = entryway
    noun = 'fez' 
    groupDesc = "fez" 
    sDesc="fez"
;
panama: HatItem 
    location = entryway
    noun = 'panama' 
    groupDesc = "Panama"
;
sombrero: HatItem 
    location = entryway
    noun = 'sombrero' 
    groupDesc = "sombrero" 
    sDesc="sombrero"
;

/*
 *  A ListGroup for listing coins together.
 */
coinListGroup: ListGroupParenCount
    groupPrefix = "coins"
;

/*
 *  A class for silver coins.
 */
class SilverCoin: Item
    location = entryway
    isEquivalent = true
    plural = 'coins'
    noun = 'coin'
    adjective = 'silver'
    sDesc = "silver coin"
    pluralDesc = "silver coins"
    groupDesc = "silver"
    pluralGroupDesc = "silver"
    listTogether = coinListGroup
;

coin1: SilverCoin;
coin2: SilverCoin;
coin3: SilverCoin;
coin4: SilverCoin;

/*
 *  A class for bronze coins.
 */
class BronzeCoin: Item
    location = entryway
    isEquivalent = true
    plural = 'coins'
    noun = 'coin'
    adjective = 'bronze'
    sDesc = "bronze coin"
    pluralDesc = "bronze coins"
    groupDesc = "bronze"
    pluralGroupDesc = "bronze"
    listTogether = coinListGroup
;

coin5: BronzeCoin;
coin6: BronzeCoin;

/*
 *  A class for gold coins.
 */
class GoldCoin: Item
    location = entryway
    isEquivalent = true
    plural = 'coins'
    noun = 'coin'
    adjective = 'gold'
    sDesc = "gold coin"
    pluralDesc = "gold coins"
    groupDesc = "gold"
    pluralGroupDesc = "gold"
    listTogether = coinListGroup
;

coin7: GoldCoin;
coin8: GoldCoin;
coin9: GoldCoin;

/*
 *  A ListGroup for listing scrabble pieces together.
 *
 *  ListGroupMessage - provides a prefix and suffix message for the
 *  grouped objects.
 */
scrabbleTileListGroup: ListGroupMessage
  groupPrefix = "the letters "
  groupSuffix = "from a Scrabble set"
;

/*
 *  A class for scrabble pieces.
 */
class ScrabbleTile: Item
  listTogether = scrabbleTileListGroup
  sDesc = "Scrabble letter <<groupDesc>>"
  adjective = 'letter'
  plural = 'letters'
;

x: ScrabbleTile
    location = entryway
    groupDesc = "X"
    noun = 'x'
;

y: ScrabbleTile
    location = entryway
    groupDesc = "Y"
    noun = 'y'
;

z: ScrabbleTile
    location = entryway
    groupDesc = "Z"
    noun = 'z'
;

/*
 *  Define a candle. Lightsources have lightsOn set to true when they
 *  are providing light.
 */
candle: Lightsource
    location = entryway
    sDesc = "candle"
    noun = 'candle'
    lightsOn = true
;

/*
 *   The hallway to the north of the entryway 
 *
 *  The hallway demonstrates the use of initial and firstSeen in room
 *  descriptions.
 */
hallway: Room
    sDesc = "Hallway"
    lDesc = "This broad, dimly-lit corridor runs north and south. "

    south = entryway
    north = kitchen
    initial = "As you enter the hallway a feeling of nostalgia
        overwhelms you."
    firstSeen = "You pause to wonder, where have you seen this
        room before?"
;

/*
 *  Define the kitchen. A "DarkRoom" is a Room that doesn't provide its
 *  own light and depends upon a Lightsource.
 */
kitchen: DarkRoom
    sDesc = "Kitchen"
    lDesc = "This is a surprisingly cramped kitchen, equipped with
             antiques: the stove is a huge black iron contraption,
             and there doesn't even seem to be a refrigerator.  A
             hallway lies to the south. "
    south = hallway
    north = pantryDoor
;

/*
 *  The stove is fixed, since we don't want the player to be able 
 *  to move it, and we'd also like it to be a container that can be 
 *  opened and closed, so make it both a Fixture and an 
 *  OpenableContainer.
 *
 *  Since the stove's material is "paper" it allows sounds and smells to
 *  pass, but not sight or touch. We can therefore smell the bread when 
 *  it is inside the stove.
 */
stove: Fixture, OpenableContainer
    location = kitchen
    noun = 'stove' 'oven' 'contraption' 'door'
    adjective = 'huge' 'black' 'iron' 'stove'
    material = paper
    sDesc = "stove"
    lDesc = {
        "It's a huge black iron cube, with a front door that swings
        open sideways.  The door is currently
        <<self.isOpen ? "open" : "closed">>. ";

        /* list my contents if there's anything inside */
        if (self.isOpen && itemCount(self.contents) != 0)
            "Inside the stove you can see <<listCont(self)>>. ";
    }

    /* it starts off closed */
    isOpen = nil
;

/*
 *   A FoodItem is something that we can eat 
 */
loaf: FoodItem
    location = stove
    noun = 'loaf' 'bread' 'crust'
    adjective = 'fresh' 'golden-brown' 'brown'
    sDesc = "loaf of bread"
    lDesc = "It's a fresh loaf with a golden-brown crust."
    smellDesc = "The smell of fresh baked bread permeates the room."

    /* 
     *   override the "doEat" method - this is called when the player eats
     *   the item 
     */
    doEat(actor) = {
        "You tear off a piece and eat it; it's delicious.  You tear off
        a little more, then a little more, and before long the whole loaf
        is gone. ";

        /* make the bread vanish by moving it to "nil" */
        self.moveInto(nil);
    }
;

/*
 *  Define the pantry door. This is a Door class object, which means
 *  that it Floats between locations. It defines a foundIn property that
 *  indicates both where it is located and where it leads.
 *
 *  The door is also a Connector. Connector class objects act as
 *  conduits for sense passing between locations (in this case between
 *  the kitchen and the pantry). Connectors must define a
 *  connectorMaterial, which determines what kind of senses will pass 
 *  through the Connector. 
 */
pantryDoor: Connector, Door
    foundIn = [ kitchen, pantry ]
    sDesc = "pantry door"
    noun = 'door'
    adjective = 'pantry'
    /*
     *  Allows sound and smell to pass through it.
     */
    connectorMaterial = paper
    /*
     *  If the door is open then we can smell and hear things in the
     *  kitchen; otherwise we cannot.
     */
    canSenseLocations(sense, isVantage) = {
        if (self.isOpen)
            return self.connectorMaterial.senseThru(sense);
        else return nil;
    }
    /*
     *  The initial attribute can be used to indicate whether the
     *  door is open or closed. We could also use whenOpen and
     *  whenClosed for thise messages.
     */
    initial = "There is a pantry door here. It is currently
        <<self.isOpen ? "open" : "closed">>. "
;

/*
 *  Define the pantry.
 */
pantry: DarkRoom
    sDesc = "Pantry"
    lDesc = "This is a long narrow room used for storing food."
    south = pantryDoor
;

/*
 *  Define the cupboard. Alt's "EnterableOpenable" class can be used to
 *  make objects that the player can enter. These locations act like
 *  DarkRooms when closed and like NestedRooms when opened.
 */
cupboard: Fixture, EnterableOpenable
    location = pantry
    sDesc = "large wooden cupboard"
    noun = 'cupboard'
    adjective = 'large' 'wooden'
    initial = "\^<<self.aDesc>> stands against the far end of the room."
    lDesc = "It's a large wooden cupboard like your grandmother 
        used to own."
    /*
     *  An EnterableOpenable should define an "inside description" for
     *  display when an actor is inside the object.
     */
    insideDesc = "Despite it's outward appearance, the inside of 
        <<self.theDesc>> is a cramped and confining space."
;

#pragma C-
