////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: Animate 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _ANIMATE_H_
#define _ANIMATE_H_

#include <qcontainer.h>
#include <movefloating.t>

#pragma C+

/*
 *  Animate: Qcontainer
 *
 *  Just like an Actor object, except that the player can
 *  manipulate the Animate like an ordinary Item.  Useful for certain
 *  types of actors, such as small animals.
 */
class Animate: Qcontainer // A character in the game
    isAnimate = true          // flag that this is an Animate object
    isListed = nil          // described separately from room's contents
    bulk = 10               // and pretty bulky
    weight = 10             // actors are pretty heavy
    maxBulk = 20            // Number of objects that can be carried at once
    maxWeight = 50          // Weight that can be carried at once
    reachable = []
    posture = standing
    
    /* this should be used as an actor when ambiguous */
    preferredActor = true
    
    /*
     *  The senses the actor is capable of using.
     */
    sensesList = [ sight, sound, smell, touch ]
    
    lDesc = {
        if (itemCount(self.contents)) 
            "\^<<self.theDesc>> <<self.fmtAre>> 
            carrying <<listCont(self.contents)>>. ";
        else
            "\^<<self.theDesc>> appears to be empty-handed. ";
    }
    actorDesc = {
        "\^<<self.aDesc>> is here. ";
    }
    isCarrying(obj) = { return obj.isIn(self); }
    roomCheck(v) = {
        /*
         *   if the actor has a location, return its roomCheck, otherwise
         *   simply allow the command 
         */
        if (self.location)
            return self.location.roomCheck(v);
        else
            return true;
    }
    
    /*
     *  Animate objects sensibility is limited by the sensesList. If the
     *  sense is not in their sensesList then they can't sense the
     *  object. Integers and Strings are always sensible.
     */
    canSenseObj(sense, obj) = {
        if (isclass(obj, Integer) || isclass(obj, String))
            return true;
        else if (intersect(self.sensesList, [sense]) != nil)
            pass canSenseObj;
        else return nil;
    }
    
    /*
     *  Animate objects sensibility is limited by the sensesList. If the
     *  sense is not in their sensesList then they can't sense the
     *  object. Integers and Strings are always sensible.
     */
    canSensePreferred(sense, obj) = {
        if (isclass(obj, Integer) || isclass(obj, String)) {
            obj.sensedByPreferred = true;
            return true;
        } else if (intersect(self.sensesList, [sense]) != nil)
            pass canSensePreferred;
        else return nil;
    }

    actorAction(v, d, p, i) = {
        "\^<<self.theDesc>> <<self.doesDesc>>n't appear interested. ";
        exit;
    }
    
    verGrab(item) = {
        "\^<<self.theDesc>> <<self.isDesc>> carrying <<item.theDesc>> and
        won't let %youm% have <<item.itObjDesc>>. ";
    }
    
    verDoAskAbout(actor, iobj) = {}
    doAskAbout(actor, iobj) = {
        local lst, i, tot;
        
        lst = objwords(2);       // get actual words asked about
        tot = length(lst);
        if ((tot == 1 && (find(['it' 'them' 'him' 'her'], lst[1]) != nil))
        || tot == 0) {
            "\"Could you be more specific?\"";
            return;
        }
        
        // try to find a response for each word
        for (i = 1 ; i <= tot ; ++i) {
            if (self.askWord(lst[i], lst))
                return;
        }
        
        // didn't find anything to talk about
        self.disavow;
    }
    askWord(word, lst) = { return nil; }
    disavow = "\"I don't know much about that.\""

    verDoFollow(actor) = {
        "But <<self.theDesc>> is right here! ";
    }

    verIoGiveTo(actor) = {
        if (actor == self)
            "That wouldn't accomplish anything!";
    }
    ioGiveTo(actor, dobj) = {
        "\^<<self.theDesc>> rejects the offer.";
    }

    verIoPutIn(actor) = {
        "If you want to give something to <<self.theDesc>>, just say so.";
    }
    
    moveInto(newLoc) = {
        /*
         *  if we have a Follower then move it to our old location.
         */
        if (self.myFollower)
            self.myFollower.moveInto(self.location);
            
        /*
         *  move the object to its new location.
         */
        inherited.moveInto(newLoc);
        
        /*
         *  adjust Floating object locations if the object was the
         *  actor.
         */
        if (self == gActor())
            moveFloating(self);
    }
    
    /*
     *  move to a new location, notifying player of coming and going
     */
    travelTo(room) = {
        local oldLocVisible;
        local newLocVisible;
        
        /* if it's a Destination, travel to the Destination instead */
        while (room != nil && room.isDestination)
            room = room.destination;
        
        /* do nothing if going nowhere */
        if (room == nil)
            return;

        /* note whether the actor was previously visible */
        oldLocVisible = parserGetMe().canSenseObj(sight, self.location);

        /* note whether the actor will be visible in the new location */
        newLocVisible = parserGetMe().canSenseObj(sight, room);

        /* 
         *   if I'm leaving the player's location, and I was previously
         *   visible, and I'm not going to be visible after the move, and
         *   the player's location isn't dark, show the "leaving" message 
         */
        if (parserGetMe().location != nil
            && parserGetMe().location.isLit(parserGetMe())
            && oldLocVisible
            && !newLocVisible)
            self.sayLeaving;
        
        /* move to my new location */
        self.moveInto(room);

        /* 
         *   if I'm visible to the player at the new location, and I
         *   wasn't previously visible, and it's not dark, show the
         *   "arriving" message 
         */
        if (parserGetMe().location != nil
            && parserGetMe().location.isLit(parserGetMe())
            && newLocVisible
            && !oldLocVisible)
            self.sayArriving;
    }

    /*
     *  sayLeaving and sayArriving announce the actor's departure and arrival
     *  in the same room as the player.
     */
    sayLeaving = {
        self.location.dispParagraph;
        "\^<<self.theDesc>> leaves the area.";
    }
    sayArriving = {
        self.location.dispParagraph;
        "\^<<self.theDesc>> enters the area.";
    }

    /*  properties for the format strings */
    fmtYou = {  // subjective
        self.isThem ? "they" : self.isHim ? "he" : self.isHer ? "she" : "it";
    }
    fmtYoum = { // objective
        self.isThem ? "them" : self.isHim ? "him" : self.isHer ? "her" 
        : "it"; 
    }
    fmtYourself = { // reflexive
        self.isThem ? "themselves" : self.isHim ? "himself" : self.isHer
        ? "herself" : "itself";
    }
    fmtYour = { // possessive
        self.isThem ? "their" : self.isHim ? "his" : self.isHer ? "her"
        : "its";
    }
    fmtMe = {self.theDesc;}
    fmtYoure = {
        self.isThem ? "they're" : self.isHim ? "he's" : self.isHer ? 
        "she's" : "it's"; 
    }
    fmtYouve = {
        self.isThem ? "they've" : self.isHim ? "he's" : self.isHer ? 
        "she's" : "it's"; 
    }
    fmtHave = {self.isThem ? "have" : "has";}
    fmtDo = {self.isThem ? "do" : "does";}
    fmtAre = {self.isThem ? "are" : "is";}
    fmtS = {self.isThem ? "" : "s";}
    fmtEs = {self.isThem ? "" : "es";}
;

#pragma C-

#endif /* _ANIMATE_H_ */
