////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: Hider 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////
#ifndef _HIDER_H_
#define _HIDER_H_

#include <item.h>

#pragma C+

/*
 *  Hider: Item
 *
 *  This is a basic class of object that can hide other objects in various
 *  ways.  The UnderHider, BehindHider, and SearchHider classes
 *  are examples of Hider subclasses.  The class defines
 *  the method searchObj(actor, list), which is given the list
 *  of hidden items contained in the object (for example, this would be the
 *  underCont property, in the case of an UnderHider), and "finds"
 *  the object or objects. Its action is dependent upon a couple of other
 *  properties of the Hider object.  The serialSearch property,
 *  if true, indicates that Items in the list are to be found one at
 *  a time; if nil (the default), the entire list is found on the
 *  first try.  The autoTake property, if true, indicates that
 *  the actor automatically takes the Item or Items found; if nil, the
 *  Item or Items are moved to the actor's location.  The searchObj method
 *  returns the list with the found object or objects removed; the
 *  caller should assign this returned value back to the appropriate
 *  property (for example, UnderHider will assign the return value
 *  to underCont).
 *  
 *  Note that because the Hider is hiding something, this class
 *  overrides the normal verDoSearch method to display the
 *  message, "You'll have to be more specific about how you want
 *  to search that."  The reason is that the normal verDoSearch
 *  message ("You find nothing of interest") leads players to believe
 *  that the object was exhaustively searched, and we want to avoid
 *  misleading the player.  On the other hand, we don't want a general
 *  search to be exhaustive for most hider objects.  So, we just
 *  display a message letting the player know that the search was not
 *  enough, but we don't give away what they have to do instead.
 *  
 *  The objects hidden with one of the Hider classes must be
 *  of class Hidden.
 */
class Hider: Item
    serialSearch = nil             /* find everything in one try by default */
    autoTake = true               /* actor takes item when found by default */
    verDoSearch(actor) = {
        "%You%'ll have to be more specific about how %you% want%s%
        to search <<self.thatDesc>>. ";
    }
    searchObj(actor, list) = {
        local found, dest, i, tot;
        
        /* see how much we get this time */
        if (self.serialSearch) {
            found = [] + car(list);
            list = cdr(list);
        } else {
            found = list;
            list = nil;
        }
        
        /* set it(them) to the found item(s) */
        if (length(found) == 1)
            setit(found[1]);    // only one item - set 'it'
        else
            setit(found);       // multiple items - set 'them'
        
        /* figure destination */
        dest = actor;
        if (!self.autoTake)
            dest = dest.location;
        
        /* note what we found, and move it to destination */
        "%You% find%s% ";
        tot = length(found);
        i = 1;
        while (i <= tot) {
            found[i].aDesc;
            if (i+1 < tot)
                ", ";
            else if (i == 1 && tot == 2)
                " and ";
            else if (i+1 == tot && tot > 2)
                ", and ";
            
            found[i].moveInto(dest);
            i = i + 1;
        }
        
        /* say what happened */
        if (self.autoTake)
            ", which %you% take%s%. ";
        else
            "! ";
        
        if (list != nil && length(list) == 0)
            list = nil;
        return list;
    }
;

#pragma C-

#endif /* _HIDER_H_ */
