////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: Player 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _PLAYER_H_
#define _PLAYER_H_

#include <actor.h>

#pragma C+

/*
 *  Player: Actor
 *
 *  A default implementation of the Me object, which is the
 *  player character.  adv.t defines Player instead of
 *  Me to allow your game to override parts of the default
 *  implementation while still using the rest, and without changing
 *  adv.t itself.  To use Player unchanged as your player
 *  character, include this in your game:  "Me: Player;".
 *  
 *  The Player object defines all of the methods and properties
 *  required for an actor, with appropriate values for the player
 *  character.  The nouns "me" and "myself" are defined ("I"
 *  is not defined, because it conflicts with the "inventory"
 *  command's minimal abbreviation of "i" in certain circumstances,
 *  and is generally not compatible with the syntax of most player
 *  commands anyway).  The sDesc is "you"; the theDesc
 *  and aDesc are "yourself," which is appropriate for most
 *  contexts.  The maxBulk and maxWeight properties are
 *  set to 10 each; a more sophisticated Me might include the
 *  player's state of health in determining the maxWeight and
 *  maxBulk properties.
 */
class Player: Actor
    noun = 'me' 'myself'
    maxBulk = 10
    maxWeight = 10
    sDesc = "you"
    aDesc = "yourself"
    theDesc = "yourself"
    lDesc = "You look about the same as always. "
    roomCheck(v) = { return self.location.roomCheck(v); }
    actorAction(verb, dobj, prep, iobj) = {}
    travelTo(room) = {
        /* 
         *   if I'm not the current player character, inherit the default
         *   handling 
         */
        if (parserGetMe() != self) {
            /* 
             *   I'm not the current player character, so I'm just an
             *   ordinary actor - use the inherited version that ordinary
             *   actors use 
             */
            inherited.travelTo(room);
            return;
        }

        /* 
         *   if the room is not nil, travel to the new location; ignore
         *   attempts to travel to "nil", because direction properties and
         *   Destinations return nil after displaying a message explaining
         *   why the travel is impossible, hence we need do nothing more
         *   in these cases 
         */
        if (room != nil) {
            if (room.isDestination) {
                /* it's a Destination - travel to the Destination's destination */
                self.travelTo(room.destination);
            } else if (self.checkForDarkTravel(room)) {
                /* neither location is lit - travel in the dark */
                darkTravel();
            } else {
                /* notify the old location that we're leaving */
                if (self.location != nil)
                    self.location.leaveRoom(self);

                /* move to the new location */
                self.moveInto(room);

                /* notify the new location that we're entering */
                room.enterRoom(self);
            }
        }
    }
    checkForDarkTravel(newLoc) = {
        local oldLoc, oldLocLit, newLocLit;
        
        /* Save the current location */
        oldLoc = self.location;
        
        /* check for light at the current location */
        oldLocLit = self.location.isLit(self);
        
        if (!oldLocLit) {
            
            /* we don't have light, move the actor to new location */
            self.moveInto(newLoc);
            
            /* check for light at new location */
            newLocLit = self.location.isLit(self);
            
            /* move the actor back to old location */
            self.moveInto(oldLoc);
            
            /* check for light at new location */
            if (!newLocLit) 
                return true;
        }
        return nil;
    }
    verDoFollow(actor) = {
        if (actor == self)
            "You can't follow yourself! ";
    }
    ioGiveTo(actor, dobj) = {
        "\^<<self.fmtYou>> accept<<self.fmtS>> <<dobj.theDesc
        >> from %youm%. ";
        dobj.moveInto(parserGetMe());
    }

    // these properties are for the format strings
    fmtYou = {  // subjective
        self == parserGetMe() ? "you" : self.isThem ? "they" :
        self.isHim ? "he" : self.isHer ? "she" : "it";
    }
    fmtYoum = { // objective
        self == parserGetMe() ? "you" : self.isThem ? "them" :
        self.isHim ? "him" : self.isHer ? "her" : "it";
    }
    fmtYourself = { // reflexive
        self == parserGetMe() ? "yourself" : self.isThem ? "themselves"
        : self.isHim ? "himself" : self.isHer ? "herself" : "itself"; 
    }
    fmtYour = { // possessive
        self == parserGetMe() ? "your" : self.isThem ? "their" :
        self.isHim ? "his" : self.isHer ? "hers" : "its"; 
    }
    fmtMe = {
        self == parserGetMe() ? "me" : self.isThem ? "them" : self.isHim
        ? "him" : self.isHer ? "her" : "it";
    }
    fmtYoure = {
        self == parserGetMe() ? "you're" : self.isThem ? "they're" :
        self.isHim ? "he's" : self.isHer ? "she's" : "it's"; 
    }
    fmtYouve = {
        self == parserGetMe() ? "you've" : self.isThem ? "they've" :
        self.isHim ? "he's" : self.isHer ? "she's" : "it's";
    }
    fmtHave = {
        self == parserGetMe() ? "have" : self.isThem ? "have" : "has";
    }
    fmtDo = {
        self == parserGetMe() ? "do" : self.isThem ? "do" : "does";
    }
    fmtAre = {
        self == parserGetMe() ? "are" : self.isThem ? "are" : "is";
    }
    fmtS = {
        self == parserGetMe() ? "" : self.isThem ? "" : "s";
    }
    fmtEs = {
        self == parserGetMe() ? "" : self.isThem ? "" : "es";
    }
;

/*
 *   "Me" is the initial player's actor; the parser automatically uses the
 *   object named "Me" as the player character object at the beginning of
 *   the game.  We'll provide a default definition simply by creating an
 *   object that inherits from the basic player object, Player.
 *   
 *   Note that you can change the player character object at any time
 *   during the game by calling switchPlayer(newPlayer).  You can also create
 *   additional player character objects, if you want to let the player
 *   take the role of different characters in the course of the game, by
 *   creating additional objects that inherit from Player.  (Inheriting
 *   from Player isn't required for player character objects -- you can
 *   define your own objects from scratch -- but it makes it a lot easier,
 *   since Player has a lot of code pre-defined for you.)  
 */
Me: Player
;

#pragma C-

#endif /* _PLAYER_H_ */
