////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: listContGen() 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _LIST_CONT_GEN_T_
#define _LIST_CONT_GEN_T_

#include <itemcount.t>
#include <isindistinguishable.t>
#include <sayprefixcount.t>

#pragma C+

/*
 *  listContGen: function(obj, desc, flags, indent)
 *
 *  This is a general-purpose object lister routine; the other object lister
 *  routines call this routine to do their work.  This function can take an
 *  object, in which case it lists the contents of the object, or it can take
 *  a list, in which case it simply lists the items in the list.  The flags
 *  parameter specifies what to do.  LCG_TALL makes the function display a "tall"
 *  listing, with one object per line; if LCG_TALL isn't specified, the function
 *  displays a "wide" listing, showing the objects as a comma-separated list.
 *  When LCG_TALL is specified, the indent parameter indicates how many
 *  tab levels to indent the current level of listing, allowing recursive calls
 *  to display the contents of listed objects.  LCG_CHECKVIS makes the function
 *  check the visibility of the top-level object before listing its contents.
 *  LCG_RECURSE indicates that we should recursively list the contents of the
 *  objects we list; we'll use the same flags on recursive calls, and use one
 *  higher indent level.  To specify multiple flags, combine them with the
 *  bitwise-or (|) operator.
 */
listContGen: function(obj, desc, flags, indent) {
    local i, count, tot, list, cur, dispTot, prefixCount;
    local listGroupList = [];

    /*
     *   Get the list.  If the "obj" parameter is already a list, use it
     *   directly; otherwise, list the contents of the object. 
     */
    switch(datatype(obj)) {
    case 2:
        /* it's an object - list its contents */
        list = obj.contents;

        /* 
         *   if the CHECKVIS flag is specified, check to make sure the
         *   contents of the object are visible; if they're not, don't
         *   list anything 
         */
        if ((flags & LCG_CHECKVIS) != 0) {
            local contvis;

            /* determine whether the contents are visible */
            contvis = (!isclass(obj, Openable)
                        || (isclass(obj, Openable) && obj.isOpen)
                        || obj.canSenseContents(sight, true));

            /* if they're not visible, don't list the contents */
            if (!contvis)
                return;
        }
        break;
        
    case 7:
        /* it's already a list */
        list = obj;
        break;

    default:
        /* ignore other types entirely */
        return;
    }

    /* count the items in the list */
    tot = length(list);

    /* we haven't displayed anything yet */
    count = 0;

    /* 
     *   Count how many items we're going to display -- this may be fewer
     *   than the number of items in the list, because some items might
     *   not be listed at all (isListed = nil), and we'll list each group
     *   of equivalent objects with a single display item (with a count of
     *   the number of equivalent objets) 
     */
    dispTot = itemCount(list, flags);

    /* iterate through the list */
    for (i = 1 ; i <= tot ; ++i) {
        /* get the current object */
        cur = list[i];

        /* 
         *  This code builds the listGroupList and registers each object 
         *  with its listGroup object. LCG_NOLISTGROUP is passed to
         *  listContGen() from showList() and skips over the grouping 
         *  mechanism.
         */
        if ((flags & LCG_NOLISTGROUP) != LCG_NOLISTGROUP
        && proptype(cur, &listTogether) == DTY_OBJECT) {
            if (find(listGroupList, cur.listTogether) == nil) {
                listGroupList += cur.listTogether;
                cur.listTogether.beginList;
            }
            cur.listTogether.addList(cur);     
        /* if this object is to be listed, figure out how to show it */
        } else if ((flags & LCG_LISTALL) == LCG_LISTALL || cur.isListed) {
            /* presume there is only one such object (i.e., no equivalents) */
            prefixCount = 1;
            
            /*
             *   if this is one of more than one equivalent items, list it
             *   only if it's the first one, and show the number of such
             *   items along with the first one 
             */
            if (cur.isEquivalent) {
                local before, after;
                local j;
                local sc;

                /* get this object's superclass */
                sc = firstsc(cur);

                /* scan for other objects equivalent to this one */
                for (before = after = 0, j = 1 ; j <= tot ; ++j) {
                    if (isIndistinguishable(cur, list[j])) {
                        if (j < i) {
                            /*
                             *   note that objects precede this one, and
                             *   then look no further, since we're just
                             *   going to skip this item anyway
                             */
                            ++before;
                            break;
                        } else ++after;
                    }
                }
                
                /*
                 *   if there are multiple such objects, and this is the
                 *   first such object, list it with the count prefixed;
                 *   if there are multiple and this isn't the first one,
                 *   skip it; otherwise, go on as normal 
                 */
                if (before == 0)
                    prefixCount = after;
                else
                    continue;
            }

            /* display the appropriate separator before this item */
            if ((flags & LCG_TALL) != 0) {
                local j;
                
                /* "tall" listing - indent to the desired level */
                "\n";
                for (j = 1; j <= indent; ++j)
                    "\t";
            } else {
                /* 
                 *   "wide" (paragraph-style) listing - add a comma, and
                 *   possibly "and", if this isn't the first item
                 */
                if (count > 0) {
                    if (count+1 < dispTot)
                        ", ";
                    else if (count == 1)
                        " and ";
                    else
                        ", and ";
                }
            }
            
            /* list the object, along with the number of such items */
            if (prefixCount == 1) {
                if ((flags & LCG_SHOWPREFIX) == LCG_SHOWPREFIX) {
                    sayPrefixCount(prefixCount); " ";
                }
                if (desc == &theDesc)
                    cur.theDesc;
                else if (desc == &sDesc)
                    cur.sDesc;
                else if (desc == &groupDesc)
                    cur.groupDesc;
                else
                    /* there's only one object - show the singular description */
                    cur.aDesc;
            } else {
                /* 
                 *   there are multiple equivalents for this object -
                 *   display the number of the items and the plural
                 *   description 
                 */
                if (desc == &theDesc)
                    "the ";
                sayPrefixCount(prefixCount); " ";
                if (desc == &groupDesc)
                    cur.pluralGroupDesc;
                else
                    cur.pluralDesc;
            }
            
            if (gVerb() == iVerb
            ||gVerb() == iTallVerb
            ||gVerb() == iWideVerb)
                cur.invDesc;
                
            /* increment the number of displayed items */
            ++count;

            /* 
             *   if this is a "tall" listing, and there's at least one
             *   item contained inside the current item, list the item's
             *   contents recursively, indented one level deeper 
             */
            if ((flags & LCG_RECURSE) != 0 
            && itemCount(cur.contents) != 0) {
                /* 
                 *   if this is a "wide" listing, show the contents in
                 *   parentheses 
                 */
                if ((flags & LCG_TALL) == 0) {
                    if (cur.isSurface)
                        " (upon which %you% see%s% ";
                    else
                        " (which contains ";
                }
                
                /* show the recursive listing, indented one level deeper */
                listContGen(cur, desc, flags, indent + 1);

                /* close the parenthetical, if we opened one */
                if ((flags & LCG_TALL) == 0)
                    ")";
            }
        }
    }
    /*
     *  Process the Grouped items. First we decide how to separate the
     *  general listing from the group listing -- if an "and" is
     *  necessary. Next we process each ListGroup object, determining
     *  how to separate each grouping. showList() should eventually call
     *  listContGen() (with LCG_NOLISTGROUP) and allow listContGen() to
     *  format the listing of the elements within each group.
     */
    dispTot = 0;
    tot = length(listGroupList);
    for (i = 1; i <= tot; ++i) {
        if (itemCount(listGroupList[i].groupList_, flags)) ++dispTot;
    }
    // This "and" separates the general listing from group listing.
    if (count > 0 && tot > 0) {
        if (count == 1) {
            if (tot == 1)
                " and ";
            else
                ", ";
        } else {
            if (count+1 < count+dispTot)
                ", ";
            else if (count == 1)
                " and ";
            else
                ", and ";
        }
    }
    count = 0;
    for (i = 1; i <= tot; ++i) {
        cur = listGroupList[i]; 
        if (count > 0 && itemCount(cur.groupList_) > 0) {
            // This "and" separates one group from another.
            if (count+1 < dispTot)
                "; ";
            else if (count == 1)
                " and ";
            else
                "; and ";
        }
        // showList is passed the parameters of listContGen() except obj
        cur.showList(desc, flags, indent);
        ++count;
    }
}

#pragma C-

#endif /* _LIST_CONT_GEN_T_ */
