//
//  XTStringUtilsTests.m
//  TadsTerp
//
//  Created by Rune Berg on 24/04/14.
//  Copyright (c) 2014 Rune Berg. All rights reserved.
//

#import <XCTest/XCTest.h>
#import "XTStringUtils.h"


@interface XTStringUtilsTests : XCTestCase

@end


@implementation XTStringUtilsTests

- (void)setUp
{
    [super setUp];
}

- (void)tearDown
{
    // Put teardown code here. This method is called after the invocation of each test method in the class.
}

- (void)test_stringEndsWith
{
	XCTAssertTrue([XTStringUtils string:@"a" endsWith:@"a"]);
	XCTAssertTrue([XTStringUtils string:@"ab" endsWith:@"b"]);
	XCTAssertTrue([XTStringUtils string:@"abc" endsWith:@"c"]);
	XCTAssertTrue([XTStringUtils string:@"ab" endsWith:@"ab"]);
	XCTAssertTrue([XTStringUtils string:@"abc" endsWith:@"abc"]);
	XCTAssertTrue([XTStringUtils string:@"abcdef" endsWith:@"cdef"]);
	XCTAssertTrue([XTStringUtils string:@"game.gam" endsWith:@".gam"]);
	XCTAssertTrue([XTStringUtils string:@"game.t3" endsWith:@".t3"]);

	XCTAssertFalse([XTStringUtils string:@"a" endsWith:@"A"]);
	XCTAssertFalse([XTStringUtils string:@"A" endsWith:@"a"]);
	XCTAssertFalse([XTStringUtils string:@"ab" endsWith:@"aB"]);
	XCTAssertFalse([XTStringUtils string:@"a" endsWith:@"b"]);
	XCTAssertFalse([XTStringUtils string:@"ab" endsWith:@"a"]);
	XCTAssertFalse([XTStringUtils string:@"abc" endsWith:@"b"]);

	XCTAssertFalse([XTStringUtils string:nil endsWith:nil]);
	XCTAssertFalse([XTStringUtils string:@"" endsWith:nil]);
	XCTAssertFalse([XTStringUtils string:@"a" endsWith:nil]);
	XCTAssertFalse([XTStringUtils string:@"abc" endsWith:nil]);

	XCTAssertFalse([XTStringUtils string:nil endsWith:@""]);
	XCTAssertFalse([XTStringUtils string:@"" endsWith:@""]);
	XCTAssertFalse([XTStringUtils string:@"a" endsWith:@""]);
	XCTAssertFalse([XTStringUtils string:@"abc" endsWith:@""]);
}

- (void)test_stringEndsWithCaseInsensitive
{
	XCTAssertTrue([XTStringUtils string:@"a" endsWithCaseInsensitive:@"a"]);
	XCTAssertTrue([XTStringUtils string:@"a" endsWithCaseInsensitive:@"A"]);
	XCTAssertTrue([XTStringUtils string:@"A" endsWithCaseInsensitive:@"A"]);
	XCTAssertTrue([XTStringUtils string:@"A" endsWithCaseInsensitive:@"a"]);
	XCTAssertTrue([XTStringUtils string:@"ab" endsWithCaseInsensitive:@"b"]);
	XCTAssertTrue([XTStringUtils string:@"abc" endsWithCaseInsensitive:@"c"]);
	XCTAssertTrue([XTStringUtils string:@"ab" endsWithCaseInsensitive:@"ab"]);
	XCTAssertTrue([XTStringUtils string:@"abc" endsWithCaseInsensitive:@"abc"]);
	XCTAssertTrue([XTStringUtils string:@"abc" endsWithCaseInsensitive:@"ABC"]);
	XCTAssertTrue([XTStringUtils string:@"ABC" endsWithCaseInsensitive:@"abc"]);
	XCTAssertTrue([XTStringUtils string:@"ABC" endsWithCaseInsensitive:@"ABC"]);
	XCTAssertTrue([XTStringUtils string:@"abcdef" endsWithCaseInsensitive:@"cdef"]);
	XCTAssertTrue([XTStringUtils string:@"game.gam" endsWithCaseInsensitive:@".gam"]);
	XCTAssertTrue([XTStringUtils string:@"game.gam" endsWithCaseInsensitive:@".GAM"]);
	XCTAssertTrue([XTStringUtils string:@"GAME.GAM" endsWithCaseInsensitive:@".gam"]);
	XCTAssertTrue([XTStringUtils string:@"GAME.T3" endsWithCaseInsensitive:@".t3"]);
	
	XCTAssertFalse([XTStringUtils string:@"a" endsWithCaseInsensitive:@"b"]);
	XCTAssertFalse([XTStringUtils string:@"ab" endsWithCaseInsensitive:@"a"]);
	XCTAssertFalse([XTStringUtils string:@"abc" endsWithCaseInsensitive:@"b"]);
	
	XCTAssertFalse([XTStringUtils string:nil endsWithCaseInsensitive:nil]);
	XCTAssertFalse([XTStringUtils string:@"" endsWithCaseInsensitive:nil]);
	XCTAssertFalse([XTStringUtils string:@"a" endsWithCaseInsensitive:nil]);
	XCTAssertFalse([XTStringUtils string:@"abc" endsWithCaseInsensitive:nil]);
	XCTAssertFalse([XTStringUtils string:nil endsWithCaseInsensitive:@""]);
	
	XCTAssertFalse([XTStringUtils string:@"" endsWithCaseInsensitive:@""]);
	XCTAssertFalse([XTStringUtils string:@"a" endsWithCaseInsensitive:@""]);
	XCTAssertFalse([XTStringUtils string:@"abc" endsWithCaseInsensitive:@""]);
}

//+ (BOOL)string:(NSString *)string1 isEqualToCaseInsensitive:(NSString *)string2

- (void)test_isEqualToCaseInsensitive
{
	XCTAssertFalse([XTStringUtils string:nil isEqualToCaseInsensitive:nil]);
	XCTAssertFalse([XTStringUtils string:nil isEqualToCaseInsensitive:@"a"]);
	XCTAssertFalse([XTStringUtils string:@"a" isEqualToCaseInsensitive:nil]);

	XCTAssertTrue([XTStringUtils string:@"a" isEqualToCaseInsensitive:@"a"]);
	XCTAssertTrue([XTStringUtils string:@"A" isEqualToCaseInsensitive:@"a"]);
	XCTAssertTrue([XTStringUtils string:@"a" isEqualToCaseInsensitive:@"A"]);
	XCTAssertTrue([XTStringUtils string:@"A" isEqualToCaseInsensitive:@"A"]);

	XCTAssertFalse([XTStringUtils string:@"a" isEqualToCaseInsensitive:@"b"]);
}

- (void)test_filterRepeatedNewlines
{
	XCTAssertEqualObjects(nil, [XTStringUtils filterRepeatedNewlines:nil]);
	XCTAssertEqualObjects(@"", [XTStringUtils filterRepeatedNewlines:@""]);
	XCTAssertEqualObjects(@"a", [XTStringUtils filterRepeatedNewlines:@"a"]);
	XCTAssertEqualObjects(@"abc", [XTStringUtils filterRepeatedNewlines:@"abc"]);
	XCTAssertEqualObjects(@"\n", [XTStringUtils filterRepeatedNewlines:@"\n"]);
	XCTAssertEqualObjects(@"\n", [XTStringUtils filterRepeatedNewlines:@"\n\n"]);
	XCTAssertEqualObjects(@"\n", [XTStringUtils filterRepeatedNewlines:@"\n\n\n"]);
	XCTAssertEqualObjects(@"a\nb", [XTStringUtils filterRepeatedNewlines:@"a\nb"]);
	XCTAssertEqualObjects(@"ab\nbc", [XTStringUtils filterRepeatedNewlines:@"ab\n\nbc"]);
	XCTAssertEqualObjects(@"ab\nb\nc", [XTStringUtils filterRepeatedNewlines:@"ab\n\nb\nc"]);
}

- (void)test_findRangeOfTrailingWhitespaceInLastParagraph
{
	NSRange range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:nil];
	XCTAssertEqual(NSNotFound, range.location);

	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@""];
	XCTAssertEqual(NSNotFound, range.location);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\n"];
	XCTAssertEqual(NSNotFound, range.location);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"a"];
	XCTAssertEqual(NSNotFound, range.location);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"abc"];
	XCTAssertEqual(NSNotFound, range.location);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"a\n"];
	XCTAssertEqual(NSNotFound, range.location);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\na"];
	XCTAssertEqual(NSNotFound, range.location);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\t"];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@" "];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:UNICHAR_EN_SPACE];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(1, range.length);

	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"a " UNICHAR_EN_SPACE];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(2, range.length);

	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:ZERO_WIDTH_SPACE];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\t\t"];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"  "];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:ZERO_WIDTH_SPACE ZERO_WIDTH_SPACE];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:ZERO_WIDTH_SPACE @" "];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:ZERO_WIDTH_SPACE @"\t"];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:ZERO_WIDTH_SPACE @" \t "];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(4, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\t\t\t\t\t"];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(5, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"    "];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(4, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"x\t"];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"x "];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"x\t\t"];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"x\t "];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\n\ta"];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\na\t"];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\n\t"];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\n "];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\n\n "];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\nab\n "];
	XCTAssertEqual(4, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\n\t\t"];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\n  "];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\n\t\t\t\t"];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(4, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"\n    "];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(4, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"abc\n\t"];
	XCTAssertEqual(4, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"abc\n  "];
	XCTAssertEqual(4, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"ab\n\t\t\t"];
	XCTAssertEqual(3, range.location);
	XCTAssertEqual(3, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"ab\n\t  "];
	XCTAssertEqual(3, range.location);
	XCTAssertEqual(3, range.length);
	
	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"abc\n\t\t\t\n\t"];
	XCTAssertEqual(8, range.location);
	XCTAssertEqual(1, range.length);

	range = [XTStringUtils findRangeOfTrailingWhitespaceInLastParagraph:@"abc\n\t \t\n \t"];
	XCTAssertEqual(8, range.location);
	XCTAssertEqual(2, range.length);
}

- (void)test_indexInStringOfCharAtRowColumn
{
	{
	NSString *stringNil = nil;
	XCTAssertEqual(0, [XTStringUtils indexInString:stringNil ofCharAtRow:0 column:0]);
	XCTAssertEqual(0, [XTStringUtils indexInString:stringNil ofCharAtRow:0 column:1]);
	XCTAssertEqual(0, [XTStringUtils indexInString:stringNil ofCharAtRow:1 column:0]);
	XCTAssertEqual(0, [XTStringUtils indexInString:stringNil ofCharAtRow:1 column:1]);
	XCTAssertEqual(0, [XTStringUtils indexInString:stringNil ofCharAtRow:10 column:10]);
	}
	{
	NSString *string0 = @"";
	XCTAssertEqual(0, [XTStringUtils indexInString:string0 ofCharAtRow:0 column:0]);
	XCTAssertEqual(0, [XTStringUtils indexInString:string0 ofCharAtRow:0 column:1]);
	XCTAssertEqual(0, [XTStringUtils indexInString:string0 ofCharAtRow:1 column:0]);
	XCTAssertEqual(0, [XTStringUtils indexInString:string0 ofCharAtRow:1 column:1]);
	XCTAssertEqual(0, [XTStringUtils indexInString:string0 ofCharAtRow:10 column:10]);
	}
	{
	NSString *string1 = @"1";
	XCTAssertEqual(0, [XTStringUtils indexInString:string1 ofCharAtRow:0 column:0]);
	XCTAssertEqual(1, [XTStringUtils indexInString:string1 ofCharAtRow:0 column:1]);
	XCTAssertEqual(1, [XTStringUtils indexInString:string1 ofCharAtRow:0 column:2]);
	XCTAssertEqual(1, [XTStringUtils indexInString:string1 ofCharAtRow:1 column:0]);
	XCTAssertEqual(1, [XTStringUtils indexInString:string1 ofCharAtRow:1 column:1]);
	XCTAssertEqual(1, [XTStringUtils indexInString:string1 ofCharAtRow:10 column:10]);
	}
	{
	NSString *string5 = @"12345";
	XCTAssertEqual(0, [XTStringUtils indexInString:string5 ofCharAtRow:0 column:0]);
	XCTAssertEqual(1, [XTStringUtils indexInString:string5 ofCharAtRow:0 column:1]);
	XCTAssertEqual(4, [XTStringUtils indexInString:string5 ofCharAtRow:0 column:4]);
	XCTAssertEqual(5, [XTStringUtils indexInString:string5 ofCharAtRow:0 column:5]);
	XCTAssertEqual(5, [XTStringUtils indexInString:string5 ofCharAtRow:1 column:0]);
	XCTAssertEqual(5, [XTStringUtils indexInString:string5 ofCharAtRow:1 column:1]);
	XCTAssertEqual(5, [XTStringUtils indexInString:string5 ofCharAtRow:10 column:10]);
	}
	{
	NSString *string2_3_1 = @"12\n123\n1";
							//012 3456 7
	XCTAssertEqual(0, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:0 column:0]);
	XCTAssertEqual(1, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:0 column:1]);
	XCTAssertEqual(2, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:0 column:2]);
	XCTAssertEqual(2, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:0 column:3]);
	XCTAssertEqual(2, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:0 column:4]);
	XCTAssertEqual(3, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:1 column:0]);
	XCTAssertEqual(4, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:1 column:1]);
	XCTAssertEqual(5, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:1 column:2]);
	XCTAssertEqual(6, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:1 column:3]);
	XCTAssertEqual(6, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:1 column:4]);
	XCTAssertEqual(6, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:1 column:5]);
	XCTAssertEqual(7, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:2 column:0]);
	XCTAssertEqual(8, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:2 column:1]);
	XCTAssertEqual(8, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:2 column:2]);
	XCTAssertEqual(8, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:2 column:3]);
	XCTAssertEqual(8, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:3 column:0]);
	XCTAssertEqual(8, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:3 column:1]);
	XCTAssertEqual(8, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:3 column:2]);
	XCTAssertEqual(8, [XTStringUtils indexInString:string2_3_1 ofCharAtRow:10 column:10]);
	}
}

- (void)test_rangeOfLongestLineIn
{
	NSRange rol;
	
	rol = [XTStringUtils rangeOfLongestLineIn:nil];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(0, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@""];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(0, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"a"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(1, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"abc"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(3, rol.length);

	rol = [XTStringUtils rangeOfLongestLineIn:@"a\n"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(1, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"ab\n"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(2, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"abc\n"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(3, rol.length);

	rol = [XTStringUtils rangeOfLongestLineIn:@"a\na"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(1, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"ab\na"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(2, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"abc\na"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(3, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"abc\nabc"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(3, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\na"];
	XCTAssertEqual(1, rol.location);
	XCTAssertEqual(1, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\nab"];
	XCTAssertEqual(1, rol.location);
	XCTAssertEqual(2, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\nabcd"];
	XCTAssertEqual(1, rol.location);
	XCTAssertEqual(4, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"a\nab"];
	XCTAssertEqual(2, rol.location);
	XCTAssertEqual(2, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"ab\nabc"];
	XCTAssertEqual(3, rol.location);
	XCTAssertEqual(3, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"ab\nabc\na\n"];
	XCTAssertEqual(3, rol.location);
	XCTAssertEqual(3, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"ab\nabc\na\n\nABCD"];
	XCTAssertEqual(10, rol.location);
	XCTAssertEqual(4, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"ab\nabc\na\n\nABCD\n"];
	XCTAssertEqual(10, rol.location);
	XCTAssertEqual(4, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\na\n"];
	XCTAssertEqual(1, rol.location);
	XCTAssertEqual(1, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\nab\n"];
	XCTAssertEqual(1, rol.location);
	XCTAssertEqual(2, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\n\nab\n"];
	XCTAssertEqual(2, rol.location);
	XCTAssertEqual(2, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\na\nAB\nz"];
	XCTAssertEqual(3, rol.location);
	XCTAssertEqual(2, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\na\nab\nABC"];
	XCTAssertEqual(6, rol.location);
	XCTAssertEqual(3, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\n"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(0, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\n\n"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(0, rol.length);
	
	rol = [XTStringUtils rangeOfLongestLineIn:@"\n\n\n"];
	XCTAssertEqual(0, rol.location);
	XCTAssertEqual(0, rol.length);
}

- (void)test_removeLastCharFrom
{
	NSMutableString *string;
	unichar ch;

	string = [NSMutableString stringWithString:@"abc"];
	ch = [XTStringUtils removeLastCharFrom:string];
	XCTAssertEqualObjects(@"ab", string);
	XCTAssertEqual('c', ch);
	
	string = [NSMutableString stringWithString:@"ab"];
	ch = [XTStringUtils removeLastCharFrom:string];
	XCTAssertEqualObjects(@"a", string);
	XCTAssertEqual('b', ch);
	
	string = [NSMutableString stringWithString:@"a"];
	ch = [XTStringUtils removeLastCharFrom:string];
	XCTAssertEqualObjects(@"", string);
	XCTAssertEqual('a', ch);
	
	string = [NSMutableString stringWithString:@""];
	ch = [XTStringUtils removeLastCharFrom:string];
	XCTAssertEqualObjects(@"", string);
	XCTAssertEqual(0, ch);

	string = nil;
	ch = [XTStringUtils removeLastCharFrom:string];
	XCTAssertNil(string);
	XCTAssertEqual(0, ch);
}

- (void)test_numericPrefix
{
	NSString *string = @"3";
	NSString *numPrefix = [XTStringUtils numericPrefix:string];
	XCTAssertEqualObjects(@"3", numPrefix);
	
	string = @"379";
	numPrefix = [XTStringUtils numericPrefix:string];
	XCTAssertEqualObjects(@"379", numPrefix);
	
	string = @"3x";
	numPrefix = [XTStringUtils numericPrefix:string];
	XCTAssertEqualObjects(@"3", numPrefix);
	
	string = @"x";
	numPrefix = [XTStringUtils numericPrefix:string];
	XCTAssertEqualObjects(@"", numPrefix);
	
	string = @"x3";
	numPrefix = [XTStringUtils numericPrefix:string];
	XCTAssertEqualObjects(@"", numPrefix);
	
	string = @"";
	numPrefix = [XTStringUtils numericPrefix:string];
	XCTAssertEqualObjects(@"", numPrefix);
	
	string = nil;
	numPrefix = [XTStringUtils numericPrefix:string];
	XCTAssertNil(numPrefix);
}

- (void)test_findRangeOfLastParagraph
{
	NSRange range;
	
	range = [XTStringUtils findRangeOfLastParagraph:nil];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@""];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"a"];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"abc"];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(3, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"\n"];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"\na"];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"\nabc"];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(3, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"a\n"];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"a\nb"];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"a\nbc"];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"ab\n"];
	XCTAssertEqual(3, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"\n\n"];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"\n\na"];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"a\nbc\ndef"];
	XCTAssertEqual(5, range.location);
	XCTAssertEqual(3, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"a\n"];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils findRangeOfLastParagraph:@"a\nb"];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(1, range.length);
}

- (void)testSplitStringBySeparatorIncludeSeparatorEntries_1
{
	NSArray *array;
	NSString *sep = @",";

	array = [XTStringUtils splitString:@"a,bb,ccc" bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(3, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	XCTAssertEqualObjects(@"bb", array[1]);
	XCTAssertEqualObjects(@"ccc", array[2]);
	
	array = [XTStringUtils splitString:@"a,bb" bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(2, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	XCTAssertEqualObjects(@"bb", array[1]);
	
	array = [XTStringUtils splitString:@"a" bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	
	array = [XTStringUtils splitString:@"" bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	
	array = [XTStringUtils splitString:nil bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(0, array.count);
	
	//-----
	
	array = [XTStringUtils splitString:@"a," bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(2, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	XCTAssertEqualObjects(@"", array[1]);
	
	array = [XTStringUtils splitString:@"a,," bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(3, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	XCTAssertEqualObjects(@"", array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	
	array = [XTStringUtils splitString:@",a" bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(2, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(@"a", array[1]);
	
	array = [XTStringUtils splitString:@",,a" bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(3, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(@"", array[1]);
	XCTAssertEqualObjects(@"a", array[2]);
	
	array = [XTStringUtils splitString:@",a," bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(3, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(@"a", array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	
	array = [XTStringUtils splitString:@"a,,c" bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(3, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	XCTAssertEqualObjects(@"", array[1]);
	XCTAssertEqualObjects(@"c", array[2]);
	
	array = [XTStringUtils splitString:@"," bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(2, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(@"", array[1]);
	
	array = [XTStringUtils splitString:@",," bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(3, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(@"", array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	
	array = [XTStringUtils splitString:@",,," bySeparator:sep includeSeparatorEntries:NO];
	XCTAssertEqual(4, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(@"", array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	XCTAssertEqualObjects(@"", array[3]);
	
	//-------
	
	array = [XTStringUtils splitString:@"a,b,c" bySeparator:@"" includeSeparatorEntries:NO];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"a,b,c", array[0]);
	
	array = [XTStringUtils splitString:@"" bySeparator:@"" includeSeparatorEntries:NO];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	
	array = [XTStringUtils splitString:nil bySeparator:@"" includeSeparatorEntries:NO];
	XCTAssertEqual(0, array.count);
	
	array = [XTStringUtils splitString:@"a,b,c" bySeparator:nil includeSeparatorEntries:NO];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"a,b,c", array[0]);
	
	array = [XTStringUtils splitString:@"" bySeparator:nil includeSeparatorEntries:NO];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	
	array = [XTStringUtils splitString:nil bySeparator:nil includeSeparatorEntries:NO];
	XCTAssertEqual(0, array.count);
}

- (void)testSplitStringBySeparatorIncludeSeparatorEntries_2
{
	NSArray *array;
	NSString *sep = @",";
	
	array = [XTStringUtils splitString:@"a,bb,ccc" bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(5, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"bb", array[2]);
	XCTAssertEqualObjects(sep, array[3]);
	XCTAssertEqualObjects(@"ccc", array[4]);
	
	array = [XTStringUtils splitString:@"a,bb" bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(3, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"bb", array[2]);
	
	array = [XTStringUtils splitString:@"a" bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	
	array = [XTStringUtils splitString:@"" bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	
	array = [XTStringUtils splitString:nil bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(0, array.count);
	
	//-----
	
	array = [XTStringUtils splitString:@"a," bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(3, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	
	array = [XTStringUtils splitString:@"a,," bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(5, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	XCTAssertEqualObjects(sep, array[3]);
	XCTAssertEqualObjects(@"", array[4]);
	
	array = [XTStringUtils splitString:@",a" bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(3, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"a", array[2]);
	
	array = [XTStringUtils splitString:@",,a" bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(5, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	XCTAssertEqualObjects(sep, array[3]);
	XCTAssertEqualObjects(@"a", array[4]);
	
	array = [XTStringUtils splitString:@",a," bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(5, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"a", array[2]);
	XCTAssertEqualObjects(sep, array[3]);
	XCTAssertEqualObjects(@"", array[4]);
	
	array = [XTStringUtils splitString:@"a,,c" bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(5, array.count);
	XCTAssertEqualObjects(@"a", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	XCTAssertEqualObjects(sep, array[3]);
	XCTAssertEqualObjects(@"c", array[4]);
	
	array = [XTStringUtils splitString:@"," bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(3, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	
	array = [XTStringUtils splitString:@",," bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(5, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	XCTAssertEqualObjects(sep, array[3]);
	XCTAssertEqualObjects(@"", array[4]);
	
	array = [XTStringUtils splitString:@",,," bySeparator:sep includeSeparatorEntries:YES];
	XCTAssertEqual(7, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	XCTAssertEqualObjects(sep, array[1]);
	XCTAssertEqualObjects(@"", array[2]);
	XCTAssertEqualObjects(sep, array[3]);
	XCTAssertEqualObjects(@"", array[4]);
	XCTAssertEqualObjects(sep, array[5]);
	XCTAssertEqualObjects(@"", array[6]);
	
	//-------
	
	array = [XTStringUtils splitString:@"a,b,c" bySeparator:@"" includeSeparatorEntries:YES];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"a,b,c", array[0]);
	
	array = [XTStringUtils splitString:@"" bySeparator:@"" includeSeparatorEntries:YES];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	
	array = [XTStringUtils splitString:nil bySeparator:@"" includeSeparatorEntries:YES];
	XCTAssertEqual(0, array.count);
	
	array = [XTStringUtils splitString:@"a,b,c" bySeparator:nil includeSeparatorEntries:YES];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"a,b,c", array[0]);
	
	array = [XTStringUtils splitString:@"" bySeparator:nil includeSeparatorEntries:YES];
	XCTAssertEqual(1, array.count);
	XCTAssertEqualObjects(@"", array[0]);
	
	array = [XTStringUtils splitString:nil bySeparator:nil includeSeparatorEntries:YES];
	XCTAssertEqual(0, array.count);
}

//-------

- (void)test_rangeOfNextParagraphInInRange_1
{
	NSRange range;
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:0];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(4, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:1];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(3, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:2];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:3];
	XCTAssertEqual(3, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:4];
	XCTAssertEqual(4, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:5];
	XCTAssertEqual(5, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:6];
	XCTAssertEqual(6, range.location);
	XCTAssertEqual(4, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:7];
	XCTAssertEqual(7, range.location);
	XCTAssertEqual(3, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:8];
	XCTAssertEqual(8, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:9];
	XCTAssertEqual(9, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:10];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:11];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"abc\nd\nefgh" fromLoc:100];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	//------
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"\n" fromLoc:0];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"\n" fromLoc:1];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	//------
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"\n\n\n" fromLoc:0];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"\n\n\n" fromLoc:1];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"\n\n\n" fromLoc:2];
	XCTAssertEqual(2, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"\n\n\n" fromLoc:3];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	//------
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"a" fromLoc:0];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(1, range.length);

	range = [XTStringUtils rangeOfNextParagraphIn:@"a" fromLoc:1];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	//------
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"ab" fromLoc:0];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(2, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"ab" fromLoc:1];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"ab" fromLoc:2];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);

	//------
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"\na" fromLoc:0];
	XCTAssertEqual(0, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"\na" fromLoc:1];
	XCTAssertEqual(1, range.location);
	XCTAssertEqual(1, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"\na" fromLoc:2];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
}

- (void)test_rangeOfNextParagraphInInRange_2
{
	NSRange range;
	
	range = [XTStringUtils rangeOfNextParagraphIn:nil fromLoc:0];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:nil fromLoc:1];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:nil fromLoc:5];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);

	range = [XTStringUtils rangeOfNextParagraphIn:@"" fromLoc:0];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"" fromLoc:1];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
	
	range = [XTStringUtils rangeOfNextParagraphIn:@"" fromLoc:5];
	XCTAssertEqual(NSNotFound, range.location);
	XCTAssertEqual(0, range.length);
}

@end




