#charset "us-ascii"
#include <adv3.h>

/*

Binary Heap

Steve Breslin, 2004
steve.breslin@gmail.com

A Binary Heap represents a sorted vector, designed so that the first
element of the underlying vector is always the one with the smallest
"prop" value. (The "prop" is defined when the binary heap is created:
see below.)

It defines three main methods:

  - append(elem)  == add a new element
  - getFirst()    == retrieve the first element (removes the element)
  - update(elem)  == re-sort the vector after lowering elem.(myProp)

To create a BinaryHeap object, do this:

    new BinaryHeap(<prop>, <length>?, <[initial elements]>?)

This is the same as creating a normal vector, except the first argument
is new:

<prop> should be the property pointer for the property that is used for
       sorting the vector: binaryHeap[1] will always be the element of
       the heap with the lowest <prop> value.

<length> is the length of the vector. This argument is optional, but if
         you are providing a list of initial elements, this value is
         required.

<[initial elements]> are the initial values to pass to the vector. This
         argument is optional. The initial elements must be in the form
         of a list (i.e., enclosed in square brackets).

Please note that this is a "partial" implementation of a binary heap, in
the sense that it does not specially define all of the intrinsic methods
associated with a normal Vector, but instead passes off the work to the
underlying Vector.

This *could* cause problems, if used incorrectly. For example, if you
try removeElement, the element will be removed from the underlying
vector as expected, but the binary heap will no longer be properly
sorted. However, if you take care to use the class properly, you should
have no trouble.

If you have an application where you want to remove an element or
something, let me know and I'll work up a more robust implementation.

*/

/*------------------------------------------------------------------

Please note that the elements in the vector are not in linear order: the
second element does not necessarily have the second-smallest myProp, and
the last element does not necessarily have the largest myProp. For the
organization details, see below.

You could always just use a normal vector and do something like this
every time you want to find the element with the lowest myProp value...

  // assume the first element in the vector has the lowest myProp
  local lowest = myVector[1];
  // cycle through the vector
  foreach(local elem in myVector)
    // if elem has a lower myProp then lowest, update lowest
    if(elem.(myProp) < lowest.(myProp))
      lowest = elem;
  // now lowest is the element in the vector with the lowest myProp

  // or:
  lowest = myVector.sort(nil, { a, b: a.myProp - b.myProp })[1];

...but the Binary Heap is drastically faster, especially when the vector
is large.

There are several other sorted vector structures available, but the
binary heap is the fastest, and the best in most cases.

*/

/*------------------------------------------------------------------

For the curious:
================

You can use the Binary Heap just fine without bothering with the
implementation details. But in case you're curious, here's some
explanation.

How the vector is set up:
-------------------------

Basically, we're representing a binary tree in the underlying vector.

A binary tree is a tree where each element has one parent and two
children. A picture normally helps:

              a
             / \
            /   \
           b     c
          /\     /\
         /  \   /  \
         d  e   f  g

The letters represent elements in the tree. a is the "first parent" (aka
"root"), and has two children, b and c. b has two children, d and e; and
c has two children, f and g. If we had a bigger tree, the children of d
would be h and i; and the children of e would be j and k; and so on.

We represent a tree in a vector very simply, by converting the location
on the tree into indexes of the underlying vector. To convert the above
tree into a vector, it would be simply [a,b,c,d,e,f,g].

I hope this is clear so far, but it is perhaps helpful to understand the
above in different terms: the parent of any element is at that element's
index/2; the children of an element is at that element's index*2 and
(*2)+1. Here are two examples, for clarity and review:

Q: what is g's parent?
A: myVector[myVector.indexOf(g)/2]

Q: what are b's children?
A: myVector[myVector.indexOf(b)*2] and myVector[myVector.indexOf(b)*2+1]

Ok, now you know how the vector is set up to represent a binary tree,
but what you really want to know is...

How the binary heap works:
==========================

There's only one rule: the children of any element cannot have lower
myProp values than their immediate parent. 

Every time we add or remove an element, we reorder the vector so that
this rule still applies.

What's so great about this? We only need to visit a small fraction of
the elements to maintain the vector to follow this rule.

To add an element, we add it to the end of the vector. If it has a
smaller myProp value than its parent, it switches places with its
parent, repeating this process as necessary.

To remove the first element, we remove it, and move the last element to
the first index. This new "first element" changes places with its child
with the lower myProp value, repeating this process until its myProp is
lower than both children's myProp values.

That's it.

I wrote this explanation for you, so if you read this and still don't
understand, it's my fault. Please let me know and I will revise this
explanation or comment the code below.

*/

class BinaryHeap: object

    myProp = nil // should be a property pointer
    myVector = nil // the underlying vector for the heap

// still trying to figure out the right construction for
// allowing the list argument without the length argument first.

    construct(prop, [args]) {
        myProp = prop;
        if(args) {
            if(args.length > 2) {
#ifdef DEBUG__
                "Binary Heap exception. Wrong number of arguments.";
#endif
                exit;
            }
            if(args.length == 2) {
                myVector = new Vector(args[1]);
                foreach(local elem in args[2])
                    append(elem);
            }
            else
            {
                local popList = nil;
                local cnt = 0;
                try {
                    cnt = args[1].length;
                }
                finally {
                    if(!cnt) {
                        popList = args[1];
                        cnt = 8;
                    }
                }
                myVector = new Vector(cnt);
                if(popList)
                    foreach(local elem in popList)
                        append(elem);
            }
        }
        else
            myVector = new Vector(8);
    }

    append(elem) {
        local idx, temp;
        myVector.append(elem);
        idx = myVector.length();

        while(idx > 1) {
            if(myVector[idx].(myProp) <= myVector[idx/2].(myProp)) {
                temp = myVector[idx/2];
                myVector[idx/2] = myVector[idx];
                myVector[idx] = temp;
                idx = idx/2;
            }
            else break;
        }
    }

    getFirst() {
        local temp, i = 1, j, len;
        local ret = myVector[1];
        myVector.removeElementAt(1);

        len = myVector.length;
        if(!len)
            return ret;

        myVector.prepend(myVector[len]);
        myVector.removeElementAt(len+1);

        while ( (j=2*i) <= len ) {
            if ( j<len && myVector[j].(myProp) > myVector[j+1].(myProp) )
                j++;
            if ( myVector[i].(myProp) > myVector[j].(myProp) ) {
                temp = myVector[j];
                myVector[j] = myVector[i];
                myVector[i] = temp;
                i = j;
            }
            else break;
        }
        return ret;
    }

    /* Note: we don't bother handling if the new value of
     * myVector[idx].(myProp) is lower than the old value. In most
     * applications this doesn't happen. Let me know if you want this
     * support, and I'll be happy to add it.
     */
    update(elem) {
        local temp;
        local idx = myVector.indexOf(elem);
        while(idx > 1) {
            if(myVector[idx].(myProp) <= myVector[idx/2].(myProp)) {
                temp = myVector[idx/2];
                myVector[idx/2] = myVector[idx];
                myVector[idx] = temp;
                idx = idx/2;
            }
            else break;
        }
    }

    propNotDefined(prop, [args]) {
      myVector.(prop)(args...);
    }

    /* I don't see why this is necessary. propNotDefined should pick it
     * up, I think. Unless the length for an object is defined as nil,
     * which would be silly.
     */
    length() { return myVector.length; }
;


