/* advdbs.c - adventure database access routines */
/*
	Copyright (c) 1986, by David Michael Betz
	All rights reserved
*/
#pragma lint -1
#pragma noroot

#if STACKDEBUG
#pragma debug 24
#endif

#if OPTIMIZE
#pragma optimize 9
#endif

#include "allint.h" /* TUU */
#include "advint.h"
#include "advdbs.h"
#include <stdlib.h> /* TUU */
#include <string.h> /* TUU */
#include <fcntl.h> /* TUU for open */
#include <errno.h> /* TUU for errno */

#ifdef __ORCAC__
#include <setjmp.h>
#define RMODE (O_RDONLY|O_BINARY)
#elif MAC
/**#include <fnctl.h>   TUU already included****/ 
#define RMODE	(O_RONLY|O_BINARY)
#else
#include <setjmp.h>
#define RMODE	0
#endif

/* global variables */
MyInt h_init;	/* initialization code */
MyInt h_update;	/* update code */
MyInt h_before;	/* before handler code */
MyInt h_after;	/* after handler code */
MyInt h_error;	/* error handling code */

FILE *datafd;	/* data file descriptor*/

/***** TUU was
int datafd;
*****/

/* external variables */
extern jmp_buf restart;

/* external routines in <stdlib.h> TUU 
extern char *malloc();
*****/

/* table base addresses */
char *wtable;	/* word table */
char *wtypes;	/* word type table */
MyInt wcount;	/* number of words */
char *otable;	/* object table */
MyInt ocount;	/* number of objects */
char *atable;	/* action table */
MyInt acount;	/* number of actions */
char *vtable;	/* variable table */
MyInt vcount;	/* number of variables */
char *data;	/* base of data tables */
char *base;	/* current base address */
char *dbase;	/* base of the data space */
char *cbase;	/* base of the code space */
MyInt length;	/* length of resident data structures */

/* data file header */
static char hdr[HDR_SIZE];

/* save parameters */
static long saveoff;	/* save data file offset */
static char *save;	/* save area base address */
static MyInt slen;	/* save area length */

/* db_init - read and decode the data file header */
void db_init(char *name)
{
#ifdef DEBUG
#pragma debug 0
#endif

    MyInt woff,ooff,aoff,voff,n;
    char fname[50];
    long TMPLONG;
    int TMPINT;

    /* get the data file name */
    strcpy(fname,name);
#ifndef MAC
    strcat(fname,".dat");
#endif

    /* open the data file */
/****       orig
   datafd=open(fname,RMODE);        
   TMPINT=datafd;
   if (TMPINT == -1)    TUU was datafd==-1 
	error("can't open data file");
*****/

    if ((datafd = fopen(fname,"r")) == NIL) /* TUU updated*/
	error("can't open data file");

    /* read the header */

    TMPLONG=fread(hdr,1,HDR_SIZE,datafd);
    if (TMPLONG != (long) HDR_SIZE)
	{
         printf("fread read %lu bytes.. wanted %u bytes\n",TMPLONG,HDR_SIZE);
         error("bad data file");
        }

/*** orig
      TMPINT=read(datafd,hdr,HDR_SIZE);
      if (TMPINT != HDR_SIZE)
	{
         if (TMPINT==-1)
           printf("errno is %d\n",errno);
         else
           printf("asked for %d, got %d bytes.\n",HDR_SIZE,TMPINT);
         error("bad data file");
        }
****/

    complement(hdr,HDR_SIZE);
    base = hdr;

    /* check the magic information */
    if (strncmp(&hdr[HDR_MAGIC],"ADVSYS",6) != 0)
	error("not an adventure data file");

    /* check the version number */
/* TUU UPDATED */
    n=getword(HDR_VERSION);
    if (n != VERSION) {    /*changed*/ 
    	sprintf(fname,"Wrong version number! Is:%ld, should be:%d",n,VERSION);
    	error(fname);
    }

/*** orig
    if ((n = getword(HDR_VERSION)) < 101 || n > VERSION)
	error("wrong version number");
****/
    /* decode the resident data length header field */
    length = getword(HDR_LENGTH);

    /* allocate space for the resident data structure */
    if ((data = malloc(length)) == 0)
	error("insufficient memory");

    /* compute the offset to the data */
    saveoff = (long)getword(HDR_DATBLK) * 512L;	

    /* read the resident data structure */
/* TUU UPDATED */
    fseek(datafd,saveoff,0);
    if ((TMPLONG=fread(data,1,length,datafd)) != length)
       {
        printf("fread read %lu bytes.. wanted %lu bytes\n",TMPLONG,length);
        error("bad data file");
       }

/*** orig
    TMPLONG = lseek(datafd,saveoff,0);
    if (TMPLONG==-1)
      printf("errno is %d\n",errno);
    TMPINT=read(datafd,data,(int)length);
    if (TMPINT != (int) length)
      {
       printf("errno is %d\n",errno);
       error("bad data file");
      }
*****/

    complement(data,length);

    /* get the table base addresses */
    wtable = data + (woff = getword(HDR_WTABLE));
    wtypes = data + getword(HDR_WTYPES) - 1;
    otable = data + (ooff = getword(HDR_OTABLE));
    atable = data + (aoff = getword(HDR_ATABLE));
    vtable = data + (voff = getword(HDR_VTABLE));

    /* get the save data area */
    saveoff += (long)getword(HDR_SAVE);
    save = data + getword(HDR_SAVE);
    slen = getword(HDR_SLEN);

    /* get the base of the data and code spaces */
    dbase = data + getword(HDR_DBASE);
    cbase = data + getword(HDR_CBASE);

    /* initialize the message routines */
    msg_init(datafd,getword(HDR_MSGBLK));

    /* get the code pointers */
    h_init = getword(HDR_INIT);
    h_update = getword(HDR_UPDATE);
    h_before = getword(HDR_BEFORE);
    h_after = getword(HDR_AFTER);
    h_error = getword(HDR_ERROR);

    /* get the table lengths */
    base = data;
    wcount = getword(woff); 
    ocount = getword(ooff);
    acount = getword(aoff);
    vcount = getword(voff);

    /* setup the base of the resident data */
    base = dbase;

    /* set the object count */
    setvalue(V_OCOUNT,ocount);
#ifdef DEBUG
#pragma debug 2
#endif
}

/* db_save - save the current database */
MyInt db_save(void)
{
    return (advsave(&hdr[HDR_ANAME],20,save,slen) ? T : NIL);
}

/* db_restore - restore a saved database */
MyInt db_restore(void)
{
    return (advrestore(&hdr[HDR_ANAME],20,save,slen) ? T : NIL);
}

/* db_restart - restart the current game */
MyInt db_restart(void)
{
/* TUU UPDATED  */
    fseek(datafd,saveoff,0);
    if (fread(save,1,slen,datafd) != slen)
	return (NIL);

/*** orig
    lseek(datafd,saveoff,0);
    if (read(datafd,save,slen) != slen)
	return (NIL);
****/
    complement(save,slen);
    setvalue(V_OCOUNT,ocount);
    longjmp(restart,1);
}

/* complement - complement a block of memory */
void complement(char *adr,MyInt len)
{
    for (; len--; adr++)
	*adr = ~(*adr + 30);
}

/* findword - find a word in the dictionary */
MyInt findword(char *word)
{
    char sword[WRDSIZE+1];
    MyInt wrd,i;

    /* shorten the word */
    strncpy(sword,word,WRDSIZE); sword[WRDSIZE] = 0;

    /* look up the word */
    for (i = 1; i <= wcount; i++) {
	wrd = getwloc(i);
	if (strcmp(base+wrd+2,sword) == 0)
	    return (getword(wrd));
    }
    return (NIL);
}

/* wtype - return the type of a word */
MyInt wtype(MyInt wrd)
{
    return (wtypes[wrd]);
}

/* match - match an object against a name and list of adjectives */
MyInt match(MyInt obj,MyInt noun,MyInt *adjs)
{
    MyInt *aptr;

    if (!hasnoun(obj,noun))
	return (FALSE);
    for (aptr = adjs; *aptr != NIL; aptr++)
	if (!hasadjective(obj,*aptr))
	    return (FALSE);
    return (TRUE);
}

/* checkverb - check to see if this is a valid verb */
MyInt checkverb(MyInt *verbs)
{
    MyInt act;

    /* look up the action */
    for (act = 1; act <= acount; act++)
	if (hasverb(act,verbs))
	    return (act);
    return (NIL);
}

/* findaction - find an action matching a description */
MyInt findaction(MyInt *verbs,MyInt preposition,MyInt flag)
{
    MyInt act,mask;

    /* look up the action */
    for (act = 1; act <= acount; act++) {
	if (preposition && !haspreposition(act,preposition))
	    continue;
	if (!hasverb(act,verbs))
	    continue;
	mask = ~getabyte(act,A_MASK);
	if ((flag & mask) == (getabyte(act,A_FLAG) & mask))
	    return (act);
    }
    return (NIL);
}

/* getp - get the value of an object property */
MyInt getp(MyInt obj,MyInt prop)
{
    MyInt p;

    for (; obj; obj = getofield(obj,O_CLASS))
	if (p = findprop(obj,prop))
	    return (getofield(obj,p));
    return (NIL);
}

/* setp - set the value of an object property */
MyInt setp(MyInt obj,MyInt prop,MyInt val)
{
    MyInt p;

    for (; obj; obj = getofield(obj,O_CLASS))
	if (p = findprop(obj,prop))
	    return (putofield(obj,p,val));
    return (NIL);
}

/* findprop - find a property */
MyInt findprop(MyInt obj,MyInt prop)
{
    MyInt n,i,p;

    n = getofield(obj,O_NPROPERTIES);
    for (i = p = 0; i < n; i++, p += 4)
	if ((getofield(obj,O_PROPERTIES+p) & ~P_CLASS) == prop)
	    return (O_PROPERTIES+p+2);
    return (NIL);
}

/* hasnoun - check to see if an object has a specified noun */
MyInt hasnoun(MyInt obj,MyInt noun)
{
    while (obj) {
	if (inlist(getofield(obj,O_NOUNS),noun))
	    return (TRUE);
	obj = getofield(obj,O_CLASS);
    }
    return (FALSE);
}

/* hasadjective - check to see if an object has a specified adjective */
MyInt hasadjective(MyInt obj,MyInt adjective)
{
    while (obj) {
	if (inlist(getofield(obj,O_ADJECTIVES),adjective))
	    return (TRUE);
	obj = getofield(obj,O_CLASS);
    }
    return (FALSE);
}

/* hasverb - check to see if this action has this verb */
MyInt hasverb(MyInt act,MyInt *verbs)
{
    MyInt link,word,*verb;

    /* get the list of verbs */
    link = getafield(act,A_VERBS);

    /* look for this verb */
    while (link != NIL) {
	verb = verbs;
	word = getword(link+L_DATA);
	while (*verb != NIL && word != NIL) {
	    if (*verb != getword(word+L_DATA))
		break;
	    verb++;
	    word = getword(word+L_NEXT);
	}
	if (*verb == NIL && word == NIL)
	    return (TRUE);
	link = getword(link+L_NEXT);
    }
    return (FALSE);
}

/* haspreposition - check to see if an action has a specified preposition */
MyInt haspreposition(MyInt act,MyInt preposition)
{
    return (inlist(getafield(act,A_PREPOSITIONS),preposition));
}

/* inlist - check to see if a word is an element of a list */
MyInt inlist(MyInt link,MyInt word)
{
    while (link != NIL) {
	if (word == getword(link+L_DATA))
	    return (TRUE);
	link = getword(link+L_NEXT);
    }
    return (FALSE);
}

/* getofield - get a field from an object */
MyInt getofield(MyInt obj,MyInt off)
{
    return (getword(getoloc(obj)+off));
}

/* putofield - put a field into an object */
MyInt putofield(MyInt obj,MyInt off,MyInt val)
{
    return (putword(getoloc(obj)+off,val));
}

/* getafield - get a field from an action */
MyInt getafield(MyInt act,MyInt off)
{
    return (getword(getaloc(act)+off));
}

/* getabyte - get a byte field from an action */
MyInt getabyte(MyInt act,MyInt off)
{
    return (getbyte(getaloc(act)+off));
}

/* getoloc - get an object from the object table */
MyInt getoloc(MyInt n)
{
    if (n < 1 || n > ocount)
	nerror("object number out of range: %d",n);
    return (getdword(otable+n+n));
}

/* getaloc - get an action from the action table */
MyInt getaloc(MyInt n)
{
    if (n < 1 || n > acount)
	nerror("action number out of range: %d",n);
    return (getdword(atable+n+n));
}

/* getvalue - get the value of a variable from the variable table */
MyInt getvalue(MyInt n)
{
    if (n < 1 || n > vcount)
	nerror("variable number out of range: %d",n);
    return (getdword(vtable+n+n));
}

/* setvalue - set the value of a variable in the variable table */
MyInt setvalue(MyInt n,MyInt v)
{
    if (n < 1 || n > vcount)
	nerror("variable number out of range: %d",n);
    return (putdword(vtable+n+n,v));
}

/* getwloc - get a word from the word table */
MyInt getwloc(MyInt n)
{
    if (n < 1 || n > wcount)
	nerror("word number out of range: %d",n);
    return (getdword(wtable+n+n));
}

/* getword - get a word from the data array */
MyInt getword(MyInt n)
{
    return (getdword(base+n));
}

/* putword - put a word into the data array */
MyInt putword(MyInt n,MyInt w)
{
    return (putdword(base+n,w));
}

/* getbyte - get a byte from the data array */
MyInt getbyte(MyInt n)
{
    return (*(base+n) & 0xFF);
}

/* getcbyte - get a code byte */
MyInt getcbyte(MyInt n)
{
    return (*(cbase+n) & 0xFF);
}

/* getcword - get a code word */
MyInt getcword(MyInt n)
{
    return (getdword(cbase+n));
}

/* getdword - get a word from the data array */
MyInt getdword(char *p)
{
    return (((*p & 0xFF) | (*(p+1) << 8))&0xFFFF);
}

/* putdword - put a word into the data array */
MyInt putdword(char *p,MyInt w)
{
    *p = w; *(p+1) = w >> 8;
    return (w);
}

/* nerror - handle errors with numeric arguments */
void nerror(char *fmt,MyInt n)
{
    char buf[100];
    sprintf(buf,fmt,n);
    error(buf);
}
