/* funcs.c: All the Floo operators.
    Designed by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/floo/index.html
*/

#include "glk.h"
#include "floo.h"
#include "gi_dispa.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/* Pointers to the error names. These are kept available so that they
    can be quickly passed to exec_error(). */
obj_t *err_undefined = NULL;
obj_t *err_stackunderflow = NULL;
obj_t *err_rangecheck = NULL;
obj_t *err_typecheck = NULL;
obj_t *err_undefinedresult = NULL;
obj_t *err_unmatchedmark = NULL;

/* More objects which it's useful to keep around. */
static obj_t *trueobj, *falseobj, *nullobj;

static obj_t *define_id_ffunc(char *id, int (*funcptr)(void));

static int ffun_pop(void);
static int ffun_dup(void);
static int ffun_exch(void);
static int ffun_index(void);
static int ffun_copy(void);
static int ffun_roll(void);
static int ffun_exec(void);
static int ffun_if(void);
static int ffun_ifelse(void);
static int ffun_eq(void);
static int ffun_ne(void);
static int ffun_gt(void);
static int ffun_ge(void);
static int ffun_lt(void);
static int ffun_le(void);
static int ffun_xor(void);
static int ffun_or(void);
static int ffun_and(void);
static int ffun_not(void);
static int ffun_bitshift(void);
static int ffun_add(void);
static int ffun_sub(void);
static int ffun_mul(void);
static int ffun_idiv(void);
static int ffun_mod(void);
static int ffun_neg(void);
static int ffun_abs(void);
static int ffun_mark(void);
static int ffun_antimark(void);
static int ffun_bind(void);
static int ffun_load(void);
static int ffun_def(void);
static int ffun_put(void);
static int ffun_get(void);
static int ffun_astore(void);
static int ffun_aload(void);
static int ffun_array(void);
static int ffun_putinterval(void);
static int ffun_getinterval(void);
static int ffun_length(void);
static int ffun_for(void);
static int ffun_loop(void);
static int ffun_repeat(void);
static int ffun_continue(void);
static int ffun_exit(void);
static int ffun_stop(void);
static int ffun_string(void);
static int ffun_strlen(void);
static int ffun_strcat(void);
static int ffun_cvs(void);
static int ffun_cvx(void);
static int ffun_cvlit(void);
static int ffun_cvn(void);
static int ffun_echo(void);
static int ffun_echostack(void);
static int ffun_handleerror(void);
static int ffun_glk(void);

/* Set everything up. Returns TRUE for ok. */
int init_sysdict()
{
    obj_t *ob;
    obj_t *stopob, *glkob, *handleerrorob;
    int ix, count;
    
    /* Create special objects. */
    
    ob = new_obj(otyp_Boolean);
    ob->u.num = 0;
    floodict_put(sysdict, atomdict_find("false", -1, FALSE), ob);
    falseobj = obj_newref(ob);
    
    ob = new_obj(otyp_Boolean);
    ob->u.num = 1;
    floodict_put(sysdict, atomdict_find("true", -1, FALSE), ob);
    trueobj = obj_newref(ob);
    
    ob = new_obj(otyp_Null);
    floodict_put(sysdict, atomdict_find("null", -1, FALSE), ob);
    nullobj = obj_newref(ob);
    
    /* Create a bazillion operators. */
    
    define_id_ffunc("pop", &ffun_pop);
    define_id_ffunc("dup", &ffun_dup);
    define_id_ffunc("exch", &ffun_exch);
    define_id_ffunc("index", &ffun_index);
    define_id_ffunc("copy", &ffun_copy);
    define_id_ffunc("roll", &ffun_roll);

    define_id_ffunc("def", &ffun_def);
    define_id_ffunc("load", &ffun_load);
    define_id_ffunc("bind", &ffun_bind);

    define_id_ffunc("exec", &ffun_exec);
    define_id_ffunc("if", &ffun_if);
    define_id_ffunc("ifelse", &ffun_ifelse);
    define_id_ffunc("repeat", &ffun_repeat);
    define_id_ffunc("for", &ffun_for);
    define_id_ffunc("loop", &ffun_loop);
    define_id_ffunc("continue", &ffun_continue);
    define_id_ffunc("exit", &ffun_exit);
    stopob = define_id_ffunc("stop", &ffun_stop);

    define_id_ffunc("add", &ffun_add);
    define_id_ffunc("sub", &ffun_sub);
    define_id_ffunc("mul", &ffun_mul);
    define_id_ffunc("idiv", &ffun_idiv);
    define_id_ffunc("mod", &ffun_mod);
    define_id_ffunc("neg", &ffun_neg);
    define_id_ffunc("abs", &ffun_abs);
    define_id_ffunc("bitshift", &ffun_bitshift);
    define_id_ffunc("and", &ffun_and);
    define_id_ffunc("or",  &ffun_or);
    define_id_ffunc("xor", &ffun_xor);
    define_id_ffunc("not", &ffun_not);

    define_id_ffunc("eq", &ffun_eq);
    define_id_ffunc("ne", &ffun_ne);
    define_id_ffunc("gt", &ffun_gt);
    define_id_ffunc("ge", &ffun_ge);
    define_id_ffunc("lt", &ffun_lt);
    define_id_ffunc("le", &ffun_le);

    define_id_ffunc("length", &ffun_length);
    define_id_ffunc("get", &ffun_get);
    define_id_ffunc("put", &ffun_put);
    define_id_ffunc("getinterval", &ffun_getinterval);
    define_id_ffunc("putinterval", &ffun_putinterval);
    define_id_ffunc("array", &ffun_array);
    define_id_ffunc("astore", &ffun_astore);
    define_id_ffunc("aload", &ffun_aload);
    ob = define_id_ffunc("mark", &ffun_mark);
    floodict_put(sysdict, atomdict_find("[", 1, FALSE), obj_newref(ob));
    define_id_ffunc("]", &ffun_antimark);

    define_id_ffunc("string", &ffun_string);
    define_id_ffunc("strlen", &ffun_strlen);
    define_id_ffunc("strcat", &ffun_strcat);
    define_id_ffunc("cvs", &ffun_cvs);

    define_id_ffunc("cvx", &ffun_cvx);
    define_id_ffunc("cvlit", &ffun_cvlit);
    define_id_ffunc("cvn", &ffun_cvn);

    define_id_ffunc("echo", &ffun_echo);
    define_id_ffunc("echostack", &ffun_echostack);
    
    handleerrorob = define_id_ffunc("handleerror", &ffun_handleerror);

    /* Fill in placeholder values in the errinfodict dictionary. */
    floodict_put(errinfodict, atomdict_find("command", -1, FALSE), 
        new_nullref());
    floodict_put(errinfodict, atomdict_find("errorname", -1, FALSE), 
        new_nullref());

    /* Create the default error-handler procedure. Note that it uses names,
        rather than direct references to operators. This is to make it easier
        to override the error-handling system. */

    ob = new_obj_array(2, TRUE);
    ob->u.arr.o[0] = new_obj(otyp_XID);
    ob->u.arr.o[0]->u.name.atom = atomdict_find("handleerror", -1, FALSE);
    ob->u.arr.o[1] = new_obj(otyp_XID);
    ob->u.arr.o[1]->u.name.atom = atomdict_find("stop", -1, FALSE);
    
    /* Create all the error objects. */
    
    err_undefined = new_obj(otyp_ID);
    err_undefined->u.name.atom = atomdict_find("undefined", -1, FALSE);
    floodict_put(errdict, err_undefined->u.name.atom, ob);
    err_stackunderflow = new_obj(otyp_ID);
    err_stackunderflow->u.name.atom = atomdict_find("stackunderflow", -1, FALSE);
    floodict_put(errdict, err_stackunderflow->u.name.atom, ob);
    err_rangecheck = new_obj(otyp_ID);
    err_rangecheck->u.name.atom = atomdict_find("rangecheck", -1, FALSE);
    floodict_put(errdict, err_rangecheck->u.name.atom, ob);
    err_typecheck = new_obj(otyp_ID);
    err_typecheck->u.name.atom = atomdict_find("typecheck", -1, FALSE);
    floodict_put(errdict, err_typecheck->u.name.atom, ob);
    err_undefinedresult = new_obj(otyp_ID);
    err_undefinedresult->u.name.atom = atomdict_find("undefinedresult", -1, FALSE);
    floodict_put(errdict, err_undefinedresult->u.name.atom, ob);
    err_unmatchedmark = new_obj(otyp_ID);
    err_unmatchedmark->u.name.atom = atomdict_find("unmatchedmark", -1, FALSE);
    floodict_put(errdict, err_unmatchedmark->u.name.atom, ob);
    
    glkob = define_id_ffunc("glk", &ffun_glk);
    
    /* Create all the constants and procedures for Glk interface. */
    
    count = gidispatch_count_functions();
    for (ix=0; ix<count; ix++) {
        gidispatch_function_t *fn = gidispatch_get_function(ix);
        char buf[64];
        obj_t *obarr;
        atom_t *atom;
        sprintf(buf, "glk_%s", fn->name);
        obarr = new_obj_array(2, TRUE);
        obarr->u.arr.o[0] = new_obj(otyp_Integer);
        obarr->u.arr.o[0]->u.num = fn->id;
        obarr->u.arr.o[1] = obj_newref(glkob);
        atom = atomdict_find(buf, -1, TRUE);
        floodict_put(sysdict, atom, obarr);
    }
    count = gidispatch_count_intconst();
    for (ix=0; ix<count; ix++) {
        gidispatch_intconst_t *con = gidispatch_get_intconst(ix);
        atom_t *atom;
        ob = new_obj(otyp_Integer);
        ob->u.num = con->val;
        atom = atomdict_find(con->name, -1, FALSE);
        floodict_put(sysdict, atom, ob);
    }
    
    return TRUE;
}

/* Create an operator object with the given name and function. 
    id must be a static, null-terminated string. */
static obj_t *define_id_ffunc(char *id, int (*funcptr)(void))
{
    obj_t *ob = new_obj(otyp_FFunc);
    atom_t *atom = atomdict_find(id, -1, FALSE);
    ob->u.ffunc.funcptr = funcptr;
    ob->u.ffunc.defatom = atom;
    floodict_put(sysdict, atom, ob);
    return ob; /* nonref */
}

/* Return a new reference to the null object. */
obj_t *new_nullref()
{
    return obj_newref(nullobj);
}

/* And now, all the operators. */

static int ffun_pop()
{
    obj_t *ob = stack_pop(valst);
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_dup()
{
    obj_t *ob = stack_peek(valst, 0);
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    stack_push(valst, ob);
    return stat_Ok;
}

static int ffun_exch()
{
    obj_t *ob, *ob2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob = stack_pop(valst);
    ob2 = stack_pop(valst);
    stack_push(valst, ob);
    stack_push(valst, ob2);
    return stat_Ok;
}

static int ffun_index()
{
    obj_t *ob, *ob2;
    int val;
    ob = stack_pop(valst);
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    if (ob->type != otyp_Integer) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    val = ob->u.num;
    if (val < 0 || val >= stack_height(valst)) {
        stack_push(valst, ob);
        return exec_error(err_rangecheck);
    }
    delete_obj(ob);
    ob2 = stack_peek(valst, val);
    stack_push(valst, ob2);
    return stat_Ok;
}

static int ffun_copy()
{
    int ix, num;
    obj_t *ob, *ob2;
    ob = stack_pop(valst);
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    if (ob->type != otyp_Integer) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    num = ob->u.num;
    if (num < 0 || num > stack_height(valst)) {
        stack_push(valst, ob);
        return exec_error(err_rangecheck);
    }
    delete_obj(ob);
    for (ix=0; ix<num; ix++) {
        ob2 = stack_peek(valst, (num-1));
        stack_push(valst, ob2);
    }
    return stat_Ok;
}

static int ffun_roll()
{
    obj_t *ob, *ob2, *ob3;
    int ix, num, shift, pos;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    num = ob->u.num;
    shift = ob2->u.num;
    if (num < 0 || num > stack_height(valst)) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_rangecheck);
    }
    delete_obj(ob);
    delete_obj(ob2);
    if (num != 0 && shift != 0) {
        if (shift > 0)
            pos = shift % num;
        else 
            pos = (num-1) - ((-1-shift) % num);
        for (ix=0; ix<num; ix++) {
            ob3 = stack_peek(valst, pos);
            stack_push(exst, ob3);
            pos++;
            if (pos == num)
                pos = 0;
        }
        for (ix=0; ix<num; ix++) {
            ob3 = stack_pop(valst);
            delete_obj(ob3);
        }
        for (ix=0; ix<num; ix++) {
            ob3 = stack_pop(exst);
            stack_push(valst, ob3);
        }
    }
    return stat_Ok;
}

static int ffun_exec()
{
    obj_t *ob = stack_pop(valst);
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    stack_push(exst, ob);
    return stat_Ok;
}

static int ffun_def()
{
    obj_t *ob, *ob2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_ID) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    floodict_put(sysdict, ob->u.name.atom, ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_load()
{
    obj_t *ob, *ob2;
    ob = stack_pop(valst);
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    if (ob->type != otyp_ID && ob->type != otyp_XID) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    ob2 = floodict_get(sysdict, ob->u.name.atom);
    if (!ob2) {
        stack_push(valst, ob);
        return exec_error(err_undefined);
    }
    delete_obj(ob);
    stack_push(valst, ob2);
    return stat_Ok;
}

static int perform_bind(obj_t *ob)
{
    int ix;
    int len = ob->u.arr.len;
    for (ix=0; ix<len; ix++) {
        obj_t *ob2 = obj_newref(ob->u.arr.o[ix]);
        if (ob2->type == otyp_Proc) {
            perform_bind(ob2);
        }
        else if (ob2->type == otyp_XID) {
            obj_t *ob3 = floodict_get(sysdict, ob2->u.name.atom);
            if (ob3) {
                if (ob3->type == otyp_FFunc) {
                    delete_obj(ob->u.arr.o[ix]);
                    ob->u.arr.o[ix] = ob3;
                }
                else {
                    delete_obj(ob3);
                }
            }
        }
        delete_obj(ob2);
    }
    return stat_Ok;
}

static int ffun_bind()
{
    int res;
    obj_t *ob;
    ob = stack_pop(valst);
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    if (ob->type != otyp_Proc) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    res = perform_bind(ob);
    stack_push(valst, ob);
    return res;
}

static int ffun_if()
{
    obj_t *ob, *ob2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Boolean) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    if (ob->u.num) {
        stack_push(exst, ob2);
    }
    else {
        delete_obj(ob2);
    }
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_ifelse()
{
    obj_t *ob, *ob2, *ob3;
    if (stack_height(valst) < 3) {
        return exec_error(err_stackunderflow);
    }
    ob3 = stack_pop(valst);
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Boolean) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        stack_push(valst, ob3);
        return exec_error(err_typecheck);
    }
    if (ob->u.num) {
        stack_push(exst, ob2);
        delete_obj(ob3);
    }
    else {
        stack_push(exst, ob3);
        delete_obj(ob2);
    }
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_continue()
{
    return stat_Continue;
}

static int ffun_exit()
{
    return stat_Exit;
}

static int ffun_stop()
{
    return stat_Stop;
}

static int ffun_loop()
{
    obj_t *ob2 = stack_pop(valst);
    obj_t *obx;
    if (!ob2) {
        return exec_error(err_stackunderflow);
    }
    obx = new_obj(otyp_LoopFrame);
    obx->u.loopframe.obj = ob2;
    obx->u.loopframe.end = 0;
    obx->u.loopframe.inc = 0;
    obx->u.loopframe.cur = 0;
    obx->u.loopframe.pushflag = FALSE;
    stack_push(exst, obx);
    return stat_Ok;
}

static int ffun_repeat()
{
    obj_t *ob, *ob2, *obx;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    if (ob->u.num < 0) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_rangecheck);
    }
    obx = new_obj(otyp_LoopFrame);
    obx->u.loopframe.obj = ob2;
    obx->u.loopframe.end = ob->u.num;
    obx->u.loopframe.inc = 1;
    obx->u.loopframe.cur = 1;
    obx->u.loopframe.pushflag = FALSE;
    stack_push(exst, obx);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_for()
{
    obj_t *ob, *ob2, *ob3, *ob4, *obx;
    if (stack_height(valst) < 4) {
        return exec_error(err_stackunderflow);
    }
    ob4 = stack_pop(valst);
    ob3 = stack_pop(valst);
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer || ob3->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        stack_push(valst, ob3);
        stack_push(valst, ob4);
        return exec_error(err_typecheck);
    }
    obx = new_obj(otyp_LoopFrame);
    obx->u.loopframe.obj = ob4;
    obx->u.loopframe.end = ob3->u.num;
    obx->u.loopframe.inc = ob2->u.num;
    obx->u.loopframe.cur = ob->u.num;
    obx->u.loopframe.pushflag = TRUE;
    stack_push(exst, obx);
    delete_obj(ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int perform_eq(obj_t *ob, obj_t *ob2)
{
    int flag;
    if (ob->type != ob2->type)
        return FALSE;
    switch (ob->type) {
        case otyp_Null:
        case otyp_Mark:
            flag = TRUE;
            break;
        case otyp_Integer:
            flag = (ob->u.num == ob2->u.num);
            break;
        case otyp_Boolean:
            flag = (ob->u.num && ob2->u.num) || (!ob->u.num && !ob2->u.num);
            break;
        case otyp_String:
            flag = !strcmp(ob->u.str.s, ob2->u.str.s);
            break;
        case otyp_ID:
        case otyp_XID:
            flag = (ob->u.name.atom == ob2->u.name.atom);
            break;
        default:
            flag = (ob == ob2);
            break;
    }
    return flag;
}

static int ffun_eq()
{
    obj_t *ob, *ob2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (perform_eq(ob, ob2))
        stack_push(valst, obj_newref(trueobj));
    else
        stack_push(valst, obj_newref(falseobj));
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_ne()
{
    obj_t *ob, *ob2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (!perform_eq(ob, ob2))
        stack_push(valst, obj_newref(trueobj));
    else
        stack_push(valst, obj_newref(falseobj));
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_gt()
{
    obj_t *ob, *ob2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    if (ob->u.num > ob2->u.num)
        stack_push(valst, obj_newref(trueobj));
    else
        stack_push(valst, obj_newref(falseobj));
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_ge()
{
    obj_t *ob, *ob2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    if (ob->u.num >= ob2->u.num)
        stack_push(valst, obj_newref(trueobj));
    else
        stack_push(valst, obj_newref(falseobj));
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_lt()
{
    obj_t *ob, *ob2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    if (ob->u.num < ob2->u.num)
        stack_push(valst, obj_newref(trueobj));
    else
        stack_push(valst, obj_newref(falseobj));
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_le()
{
    obj_t *ob, *ob2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    if (ob->u.num <= ob2->u.num)
        stack_push(valst, obj_newref(trueobj));
    else
        stack_push(valst, obj_newref(falseobj));
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_and()
{
    obj_t *ob, *ob2, *ob3;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != ob2->type) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    if (ob->type == otyp_Integer) {
        ob3 = new_obj(ob->type);
        if (!ob3) 
            floo_err("Out of memory.");
        ob3->u.num = (ob->u.num & ob2->u.num);
    }
    else if (ob->type == otyp_Boolean) {
        if (ob->u.num && ob2->u.num)
            ob3 = obj_newref(trueobj);
        else
            ob3 = obj_newref(falseobj);
    }
    else {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    stack_push(valst, ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_or()
{
    obj_t *ob, *ob2, *ob3;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != ob2->type) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    if (ob->type == otyp_Integer) {
        ob3 = new_obj(ob->type);
        if (!ob3) 
            floo_err("Out of memory.");
        ob3->u.num = (ob->u.num | ob2->u.num);
    }
    else if (ob->type == otyp_Boolean) {
        if (ob->u.num || ob2->u.num)
            ob3 = obj_newref(trueobj);
        else
            ob3 = obj_newref(falseobj);
    }
    else {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    stack_push(valst, ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_xor()
{
    obj_t *ob, *ob2, *ob3;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != ob2->type) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    if (ob->type == otyp_Integer) {
        ob3 = new_obj(ob->type);
        if (!ob3) 
            floo_err("Out of memory.");
        ob3->u.num = (ob->u.num ^ ob2->u.num);
    }
    else if (ob->type == otyp_Boolean) {
        if ((ob->u.num && !ob2->u.num) || (!ob->u.num && ob2->u.num))
            ob3 = obj_newref(trueobj);
        else
            ob3 = obj_newref(falseobj);
    }
    else {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    stack_push(valst, ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_not()
{
    obj_t *ob, *ob3;
    if (stack_height(valst) < 1) {
        return exec_error(err_stackunderflow);
    }
    ob = stack_pop(valst);
    if (ob->type == otyp_Integer) {
        ob3 = new_obj(ob->type);
        if (!ob3) 
            floo_err("Out of memory.");
        ob3->u.num = (~ob->u.num);
    }
    else if (ob->type == otyp_Boolean) {
        if (!ob->u.num)
            ob3 = obj_newref(trueobj);
        else
            ob3 = obj_newref(falseobj);
    }
    else {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    stack_push(valst, ob3);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_bitshift()
{
    obj_t *ob, *ob2, *ob3;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    ob3 = new_obj(otyp_Integer);
    if (!ob3) 
        floo_err("Out of memory.");
    if (ob2->u.num > 0)
        ob3->u.num = (((glui32)ob->u.num) << ((glui32)ob2->u.num));
    else if (ob2->u.num < 0)
        ob3->u.num = (((glui32)ob->u.num) >> ((glui32)(-ob2->u.num)));
    else
        ob3->u.num = ob->u.num;
    stack_push(valst, ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_add()
{
    obj_t *ob, *ob2, *ob3;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    ob3 = new_obj(otyp_Integer);
    if (!ob3) 
        floo_err("Out of memory.");
    ob3->u.num = (ob->u.num + ob2->u.num);
    stack_push(valst, ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_sub()
{
    obj_t *ob, *ob2, *ob3;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    ob3 = new_obj(otyp_Integer);
    if (!ob3) 
        floo_err("Out of memory.");
    ob3->u.num = (ob->u.num - ob2->u.num);
    stack_push(valst, ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_mul()
{
    obj_t *ob, *ob2, *ob3;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    ob3 = new_obj(otyp_Integer);
    if (!ob3) 
        floo_err("Out of memory.");
    ob3->u.num = (ob->u.num * ob2->u.num);
    stack_push(valst, ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_idiv()
{
    obj_t *ob, *ob2, *ob3;
    glsi32 val1, val2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    val1 = ob->u.num;
    val2 = ob2->u.num;
    if (val2 == 0) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_undefinedresult);
    }
    ob3 = new_obj(otyp_Integer);
    if (!ob3) 
        floo_err("Out of memory.");
    if (val2 < 0) {
        val2 = -val2;
        val1 = -val1;
    }
    if (val1 > 0) 
        ob3->u.num = (val1 / val2);
    else
        ob3->u.num = -(-val1 / val2);
    stack_push(valst, ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_mod()
{
    obj_t *ob, *ob2, *ob3;
    glsi32 val1, val2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer || ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    val1 = ob->u.num;
    val2 = ob2->u.num;
    if (val2 == 0) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_undefinedresult);
    }
    ob3 = new_obj(otyp_Integer);
    if (!ob3) 
        floo_err("Out of memory.");
    if (val2 < 0) {
        val2 = -val2;
        val1 = -val1;
        if (val1 > 0) 
            ob3->u.num = -(val1 % val2);
        else
            ob3->u.num = (-val1 % val2);
    }
    else {
        if (val1 > 0) 
            ob3->u.num = (val1 % val2);
        else
            ob3->u.num = -(-val1 % val2);
    }
    stack_push(valst, ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_neg()
{
    obj_t *ob, *ob3;
    if (stack_height(valst) < 1) {
        return exec_error(err_stackunderflow);
    }
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    ob3 = new_obj(otyp_Integer);
    if (!ob3) 
        floo_err("Out of memory.");
    ob3->u.num = (-ob->u.num);
    stack_push(valst, ob3);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_abs()
{
    obj_t *ob, *ob3;
    if (stack_height(valst) < 1) {
        return exec_error(err_stackunderflow);
    }
    ob = stack_pop(valst);
    if (ob->type != otyp_Integer) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    ob3 = new_obj(otyp_Integer);
    if (!ob3) 
        floo_err("Out of memory.");
    if (ob->u.num >= 0)
        ob3->u.num = ob->u.num;
    else
        ob3->u.num = -(ob->u.num);
    stack_push(valst, ob3);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_mark()
{
    obj_t *ob = new_obj(otyp_Mark);
    if (!ob) {
        floo_err("Out of memory.");
    }
    ob->u.num = 0;
    stack_push(valst, ob);
    return stat_Ok;
}

static int ffun_antimark()
{
    obj_t *ob, *ob2;
    int ix, jx;
    ix = stack_searchmark(valst, 0);
    if (ix < 0) 
        return exec_error(err_unmatchedmark);
    ob = new_obj_array(ix, FALSE);
    if (!ob) {
        floo_err("Out of memory.");
    }
    for (jx = ix-1; jx >= 0; jx--) {
        ob2 = stack_pop(valst);
        ob->u.arr.o[jx] = ob2;
    }
    ob2 = stack_pop(valst);
    delete_obj(ob2);
    stack_push(valst, ob);
    return stat_Ok;
}

static int ffun_length()
{
    obj_t *ob3;
    obj_t *ob = stack_pop(valst);
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    ob3 = new_obj(otyp_Integer);
    if (!ob3)
        floo_err("Out of memory.");
    if (ob->type == otyp_Array || ob->type == otyp_Proc) {
        ob3->u.num = ob->u.arr.len;
    }
    else if (ob->type == otyp_String) {
        ob3->u.num = ob->u.str.len;
    }
    else {
        delete_obj(ob3);
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    stack_push(valst, ob3);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_put()
{
    obj_t *ob, *ob2, *ob3;
    if (stack_height(valst) < 3) {
        return exec_error(err_stackunderflow);
    }
    ob3 = stack_pop(valst);
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        stack_push(valst, ob3);
        return exec_error(err_typecheck);
    }
    if (ob->type == otyp_Array || ob->type == otyp_Proc) {
        int ix = ob2->u.num;
        if (ix < 0 || ix >= ob->u.arr.len) {
            stack_push(valst, ob);
            stack_push(valst, ob2);
            stack_push(valst, ob3);
            return exec_error(err_rangecheck);
        }
        delete_obj(ob->u.arr.o[ix]);
        ob->u.arr.o[ix] = ob3;
        ob3 = NULL;
    }
    else if (ob->type == otyp_String) {
        int ix = ob2->u.num;
        if (ob3->type != otyp_Integer) {
            stack_push(valst, ob);
            stack_push(valst, ob2);
            stack_push(valst, ob3);
            return exec_error(err_typecheck);
        }
        if (ix == ob->u.str.len && ob3->u.num == 0) {
            /* special case */
            delete_obj(ob3);
        }
        else {
            if (ix < 0 || ix >= ob->u.str.len) {
                stack_push(valst, ob);
                stack_push(valst, ob2);
                stack_push(valst, ob3);
                return exec_error(err_rangecheck);
            }
            ob->u.str.s[ix] = (ob3->u.num & 0xff);
            delete_obj(ob3);
        }
        ob3 = NULL;
    }
    else {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        stack_push(valst, ob3);
        return exec_error(err_typecheck);
    }
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_get()
{
    obj_t *ob, *ob2, *ob3;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    if (ob->type == otyp_Array || ob->type == otyp_Proc) {
        int ix = ob2->u.num;
        if (ix < 0 || ix >= ob->u.arr.len) {
            stack_push(valst, ob);
            stack_push(valst, ob2);
            return exec_error(err_rangecheck);
        }
        ob3 = obj_newref(ob->u.arr.o[ix]);
        stack_push(valst, ob3);
    }
    else if (ob->type == otyp_String) {
        int ix = ob2->u.num;
        if (ix < 0 || ix > ob->u.str.len) {
            stack_push(valst, ob);
            stack_push(valst, ob2);
            return exec_error(err_rangecheck);
        }
        ob3 = new_obj(otyp_Integer);
        ob3->u.num = (ob->u.str.s[ix] & 0xff);
        stack_push(valst, ob3);
    }
    else {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_getinterval()
{
    obj_t *ob, *ob2, *ob3, *ob4;
    if (stack_height(valst) < 3) {
        return exec_error(err_stackunderflow);
    }
    ob3 = stack_pop(valst);
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob2->type != otyp_Integer || ob3->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        stack_push(valst, ob3);
        return exec_error(err_typecheck);
    }
    if (ob->type == otyp_Array || ob->type == otyp_Proc) {
        int ix;
        int beg = ob2->u.num;
        int num = ob3->u.num;
        if (beg < 0 || beg > ob->u.arr.len
            || num < 0 || beg+num > ob->u.arr.len) {
            stack_push(valst, ob);
            stack_push(valst, ob2);
            stack_push(valst, ob3);
            return exec_error(err_rangecheck);
        }
        ob4 = new_obj_array(num, (ob->type == otyp_Proc));
        if (!ob4)
            floo_err("Out of memory.");
        for (ix=0; ix<num; ix++)
            ob4->u.arr.o[ix] = obj_newref(ob->u.arr.o[beg+ix]);
        stack_push(valst, ob4);
    }
    else if (ob->type == otyp_String) {
        int ix;
        char *str;
        int beg = ob2->u.num;
        int num = ob3->u.num;
        if (beg < 0 || beg > ob->u.str.len
            || num < 0 || beg+num > ob->u.str.len) {
            stack_push(valst, ob);
            stack_push(valst, ob2);
            stack_push(valst, ob3);
            return exec_error(err_rangecheck);
        }
        ob4 = new_obj(otyp_String);
        str = (char *)malloc(sizeof(char) * (num+1));
        if (!ob4 || !str)
            floo_err("Out of memory.");
        if (num)
            memcpy(str, &(ob->u.str.s[beg]), num * sizeof(char));
        str[num] = '\0';
        ob4->u.str.s = str;
        ob4->u.str.len = num;
        stack_push(valst, ob4);
    }
    else {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        stack_push(valst, ob3);
        return exec_error(err_typecheck);
    }
    delete_obj(ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_putinterval()
{
    int ix;
    int pos, len;
    obj_t *ob, *ob2, *ob3, *ob4;
    if (stack_height(valst) < 3) {
        return exec_error(err_stackunderflow);
    }
    ob3 = stack_pop(valst);
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob2->type != otyp_Integer) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        stack_push(valst, ob3);
        return exec_error(err_typecheck);
    }
    pos = ob2->u.num;
    if (ob->type == otyp_Array || ob->type == otyp_Proc) {
        if (!(ob3->type == otyp_Array || ob->type == otyp_Proc)) {
            stack_push(valst, ob);
            stack_push(valst, ob2);
            stack_push(valst, ob3);
            return exec_error(err_typecheck);
        }
        len = ob3->u.arr.len;
        if (pos < 0 || pos > ob->u.arr.len || pos+len > ob->u.arr.len) {
            stack_push(valst, ob);
            stack_push(valst, ob2);
            stack_push(valst, ob3);
            return exec_error(err_rangecheck);
        }
        for (ix=0; ix<len; ix++) {
            ob4 = obj_newref(ob3->u.arr.o[ix]);
            delete_obj(ob->u.arr.o[pos+ix]);
            ob->u.arr.o[pos+ix] = ob4;
        }
    }
    else if (ob->type == otyp_String) {
        if (!(ob3->type == otyp_String)) {
            stack_push(valst, ob);
            stack_push(valst, ob2);
            stack_push(valst, ob3);
            return exec_error(err_typecheck);
        }
        len = ob3->u.str.len;
        if (pos < 0 || pos > ob->u.str.len || pos+len > ob->u.str.len) {
            stack_push(valst, ob);
            stack_push(valst, ob2);
            stack_push(valst, ob3);
            return exec_error(err_rangecheck);
        }
        if (len)
            memcpy(&(ob->u.str.s[pos]), &(ob3->u.str.s[0]), len * sizeof(char));
    }
    else {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        stack_push(valst, ob3);
        return exec_error(err_typecheck);
    }
    delete_obj(ob3);
    delete_obj(ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_array()
{
    obj_t *ob = stack_pop(valst);
    obj_t *ob2;
    int ix, len;
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    if (ob->type != otyp_Integer) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    len = ob->u.num;
    delete_obj(ob);
    ob2 = new_obj_array(len, FALSE);
    if (!ob2)
        floo_err("Out of memory.");
    for (ix=0; ix<len; ix++)
        ob2->u.arr.o[ix] = new_nullref();
    stack_push(valst, ob2);
    return stat_Ok;
}

static int ffun_astore()
{
    obj_t *ob = stack_pop(valst);
    int ix, len;
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    if (ob->type != otyp_Array) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    len = ob->u.arr.len;
    if (stack_height(valst) < len) {
        stack_push(valst, ob);
        return exec_error(err_stackunderflow);
    }
    for (ix = len-1; ix >= 0; ix--) {
        obj_t *ob2 = stack_pop(valst);
        if (ob->u.arr.o[ix])
            delete_obj(ob->u.arr.o[ix]);
        ob->u.arr.o[ix] = ob2;
    }
    stack_push(valst, ob);
    return stat_Ok;
}

static int ffun_aload()
{
    obj_t *ob = stack_pop(valst);
    int ix, len;
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    if (ob->type != otyp_Array) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    len = ob->u.arr.len;
    for (ix = 0; ix < len; ix++) {
        obj_t *ob2 = obj_newref(ob->u.arr.o[ix]);
        stack_push(valst, ob2);
    }   
    stack_push(valst, ob);
    return stat_Ok;
}

static int ffun_string()
{
    obj_t *ob = stack_pop(valst);
    obj_t *ob2;
    char *str;
    int ix, len;
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    if (ob->type != otyp_Integer) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    len = ob->u.num;
    delete_obj(ob);
    ob2 = new_obj(otyp_String);
    str = (char *)malloc(sizeof(char) * (len+1));
    if (!ob2 || !str)
        floo_err("Out of memory.");
    for (ix=0; ix<=len; ix++)
        str[ix] = '\0';
    ob2->u.str.s = str;
    ob2->u.str.len = len;
    stack_push(valst, ob2);
    return stat_Ok;
}

static int ffun_strlen()
{
    obj_t *ob = stack_pop(valst);
    obj_t *ob2;
    int ix, len;
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    if (ob->type != otyp_String) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    ob2 = new_obj(otyp_Integer);
    if (!ob2)
        floo_err("Out of memory.");
    ob2->u.num = strlen(ob->u.str.s);
    stack_push(valst, ob2);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_strcat()
{
    obj_t *ob, *ob2;
    int len, len2;
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob2 = stack_pop(valst);
    ob = stack_pop(valst);
    if (ob->type != otyp_String || ob2->type != otyp_String) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_typecheck);
    }
    len = strlen(ob->u.str.s);
    len2 = strlen(ob2->u.str.s);
    if (len+len2 > ob->u.str.len) {
        stack_push(valst, ob);
        stack_push(valst, ob2);
        return exec_error(err_rangecheck);
    }
    if (len2)
        memcpy(ob->u.str.s+len, ob2->u.str.s, (sizeof(char) * len2));
    ob->u.str.s[len+len2] = '\0';
    stack_push(valst, ob);
    delete_obj(ob2);
    return stat_Ok;
}

static int ffun_cvs()
{
    obj_t *ob, *ob2;
    int ix, len;
    char *cx;
    char numbuf[32];
    if (stack_height(valst) < 2) {
        return exec_error(err_stackunderflow);
    }
    ob = stack_pop(valst);
    ob2 = stack_pop(valst);
    if (ob->type != otyp_String) {
        stack_push(valst, ob2);
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    switch (ob2->type) {
        case otyp_String:
            cx = ob2->u.str.s;
            len = strlen(cx);
            break;
        case otyp_ID:
        case otyp_XID:
            cx = ob2->u.name.atom->str;
            len = ob2->u.name.atom->len;
            break;
        case otyp_Integer:
            num_to_str(numbuf, ob2->u.num);
            cx = numbuf;
            len = strlen(cx);
            break;
        case otyp_Boolean:
            if (ob2->u.num)
                cx = "true";
            else
                cx = "false";
            len = strlen(cx);
            break;
        default:
            cx = "--nostringval--";
            len = strlen(cx);
            break;
    }
    if (len > ob->u.str.len) {
        stack_push(valst, ob2);
        stack_push(valst, ob);
        return exec_error(err_rangecheck);
    }
    if (len)
        memcpy(ob->u.str.s, cx, len);
    ob->u.str.s[len] = '\0';
    stack_push(valst, ob);
    delete_obj(ob2);
    return stat_Ok;
}

static int ffun_cvx()
{
    obj_t *ob = stack_pop(valst);
    obj_t *ob2;
    int ix;
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    switch (ob->type) {
        case otyp_ID:
            ob2 = new_obj(otyp_XID);
            if (!ob2)
                floo_err("Out of memory.");
            ob2->u.name.atom = ob->u.name.atom;
            break;
        case otyp_Array:
            ob2 = new_obj_array(ob->u.arr.len, TRUE);
            if (!ob2)
                floo_err("Out of memory.");
            for (ix=0; ix<ob->u.arr.len; ix++)
                ob2->u.arr.o[ix] = obj_newref(ob->u.arr.o[ix]);
            break;
        default:
            ob2 = obj_newref(ob);
            break;
    }
    delete_obj(ob);
    stack_push(valst, ob2);
    return stat_Ok;
}

static int ffun_cvlit()
{
    obj_t *ob = stack_pop(valst);
    obj_t *ob2;
    int ix;
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    switch (ob->type) {
        case otyp_XID:
            ob2 = new_obj(otyp_ID);
            if (!ob2)
                floo_err("Out of memory.");
            ob2->u.name.atom = ob->u.name.atom;
            break;
        case otyp_Proc:
            ob2 = new_obj_array(ob->u.arr.len, FALSE);
            if (!ob2)
                floo_err("Out of memory.");
            for (ix=0; ix<ob->u.arr.len; ix++)
                ob2->u.arr.o[ix] = obj_newref(ob->u.arr.o[ix]);
            break;
        default:
            ob2 = obj_newref(ob);
            break;
    }
    delete_obj(ob);
    stack_push(valst, ob2);
    return stat_Ok;
}

static int ffun_cvn()
{
    int ix;
    obj_t *ob = stack_pop(valst);
    obj_t *ob2;
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    if (ob->type != otyp_String) {
        stack_push(valst, ob);
        return exec_error(err_typecheck);
    }
    ob2 = new_obj(otyp_ID);
    ob2->u.name.atom = atomdict_find(ob->u.str.s, -1, TRUE);
    delete_obj(ob);
    stack_push(valst, ob2);
    return stat_Ok;
}

static int ffun_echo()
{
    obj_t *ob = stack_pop(valst);
    if (!ob) {
        return exec_error(err_stackunderflow);
    }
    obj_stream(ob, 0);
    delete_obj(ob);
    return stat_Ok;
}

static int ffun_echostack()
{
    obj_t *ob;
    int ix;
    int len = stack_height(valst);
    glk_put_string("Stack: ");
    for (ix=0; ix<len; ix++) {
        ob = stack_peek(valst, (len-1)-ix);
        glk_put_char(' ');
        obj_stream(ob, 0);
        delete_obj(ob);
    }
    glk_put_string("\n");
    return stat_Ok;
}

static int ffun_handleerror()
{
    obj_t *ob;
    winid_t rootwin = glk_window_get_root();
    winid_t win = 0;
    strid_t str, oldstr;
    if (!rootwin) {
        win = glk_window_open(0, 0, 0, wintype_TextBuffer, 1);
    }
    else {
        win = glk_window_open(rootwin, winmethod_Below | winmethod_Fixed, 
            3, wintype_TextBuffer, 0);
    }
    if (win) {
        oldstr = glk_stream_get_current();
        str = glk_window_get_stream(win);
        glk_stream_set_current(str);
        glk_put_string("Floo error \"");
        ob = floodict_get(errinfodict, atomdict_find("errorname", -1, FALSE));
        glk_put_buffer_stream(str, ob->u.name.atom->str, ob->u.name.atom->len);
        delete_obj(ob);
        glk_put_string("\" during execution of ");
        ob = floodict_get(errinfodict, atomdict_find("command", -1, FALSE));
        obj_stream(ob, str);
        delete_obj(ob);
        glk_put_string("\n");
        ffun_echostack();
        glk_stream_set_current(oldstr);
    }
    return stat_Ok;
}

static int ffun_glk()
{
    floo_dispatch();
    return stat_Ok;
}
