/* gtw_grx.c: graphics windows (allegro version)
        for GlkDOS, curses.h/allegro implementation of the Glk API
    Designed by L. Ross Raszewski <lraszewski@justice.loyola.edu>
    
    based upon GlkTerm by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/index.html
*/

#include "gtpref.h"
#include "gtoption.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "glk.h"
#include "glkdos.h"
#include <allegro.h>
#include <math.h>

#include "gtw_grx.h"

extern BITMAP *Buffer;          // We need access to allegro's screen

/* converts a color entry from glk (24-bit) to calleg (16-bit) */
int colorcvt(glui32 col)
{
 return makecol((col & 0x00ff0000) >> 16,
                (col & 0x0000ff00) >> 8,
                (col & 0xff));
}

/* gtwindow.c hooks */

/* Convert window dimentions from pixels to screen units */
glui32 gli_graphics_scale(glui32 size, glui32 method)
{
 if (method & winmethod_Fixed)
  if (method & winmethod_Above || method & winmethod_Below)
   size = ceil( ((float)size) / text_height(font));
  else size = ceil( ((float)size)/text_length(font,"M"));
 return size;
}

/* Calculate the dimentions of a graphics window */
void gli_graphics_winsize(window_t *win, glui32 *wid, glui32 *hgt)
{
  window_graphics_t *dwin=win->data;

  if (dwin->data) {
        *wid=dwin->data->w;
        *hgt=dwin->data->h;
        }
  else  *wid = *hgt = 0;
}

/* window manipulation functions */

/* Create a new graphics window */
window_graphics_t *win_graphics_create(window_t *win)
{
    window_graphics_t *dwin =
        (window_graphics_t *)malloc(sizeof(window_graphics_t));
    dwin->owner = win;
    dwin->bgcolor=0x00FFFFFF;  /* White */
    dwin->data=NULL;
    dwin->dirty=0;
    return dwin;
}

/* destroy a graphics window */
void win_graphics_destroy(window_graphics_t *dwin)
{
 if (dwin->data) destroy_bitmap(dwin->data);
 free(dwin);
}

/* redimention a graphics window */
void win_graphics_rearrange(window_t *win, grect_t *box)
{
 BITMAP *bmp;
 window_graphics_t *dwin=win->data;
 int x, y;
 dwin->owner->bbox=*box;
 /* create a new BITMAP for the region */
 x = box->right-box->left;
 y = box->bottom-box->top;
 x *= text_length(font,"M");
 y *= text_height(font);
 bmp=create_bitmap(x,y);
 clear_to_color(bmp,colorcvt(dwin->bgcolor));
 /* print the old BITMAP over the new */
 if (dwin->data)
  {
   blit(dwin->data,bmp,0,0,0,0,dwin->data->w,dwin->data->h);
   destroy_bitmap(dwin->data);
  }
 dwin->dirty=1;
 dwin->data=bmp;
}

/* Force a full redraw of the graphic window */
void win_graphics_redraw(window_t *win)
{
 window_graphics_t *dwin=win->data;
 int x,y;
 /* Calculate the location of the graphic, and blit the window there. */
 x = win->bbox.left * text_length(font,"M");
 y = win->bbox.top * text_height(font);
 blit(dwin->data,Buffer,0,0,x,y,dwin->data->w,dwin->data->h);
 dwin->dirty=0;
}

/* Update a graphic window by redrawing it if it's dirty */
void win_graphics_update(window_t *win)
{
 window_graphics_t *dwin=win->data;
 if (dwin->dirty) win_graphics_redraw(win);
}

/* Blank a graphics window */
void win_graphics_clear(window_t *win)
{
 window_graphics_t *dwin=win->data;
 win_graphics_erase_rect(win,0,0,dwin->data->w,dwin->data->h);
}

/* graphics pad functions */

/* Erase rectangle */
void win_graphics_erase_rect(winid_t win,
    glsi32 left, glsi32 top, glui32 width, glui32 height)
{
 window_graphics_t *dwin=win->data;
 win_graphics_fill_rect(win,dwin->bgcolor,left,top,width,height);
}

/* draw rectangle */
void win_graphics_fill_rect(winid_t win, glui32 color,
    glsi32 left, glsi32 top, glui32 width, glui32 height)
{
 window_graphics_t *dwin=win->data;
 rectfill(dwin->data,left,top, left+width, top+height,colorcvt(color));
 dwin->dirty=1;
}

/* set bg */
void win_graphics_set_background_color(winid_t win, glui32 color)
{
 window_graphics_t *dwin=win->data;
 dwin->bgcolor=color;
}

BITMAP *gli_get_image(winid_t win,
    glsi32 left, glsi32 top, glui32 width, glui32 height)
{
 window_graphics_t *dwin=win->data; 
 BITMAP *bmp;
 bmp=create_bitmap(width,height);
 blit(dwin->data,bmp,left,top,0,0,width,height);
 return bmp;
}
void gli_free_bitmap(BITMAP *b)
{
 destroy_bitmap(b);
}
