

package org.ifarchive.glk;

/** A sound channel can play exactly one sound at a time.
 * @author John Elliott
 * @version 1.0
 */
public class SoundChannel extends OpaqueObject
{
	SoundChannel(int id)
	{
		super(clSoundChannel, id);
	}

	static SoundChannel findSoundChannel(int id)
	{
		return (SoundChannel)(find(clSoundChannel, id));
	}

        /** Iterate over sound channels.
         * @see OpaqueObject#iterate
         */
        public static SoundChannel iterate(SoundChannel f, Object r[])
        {
                return (SoundChannel)iterate(clSoundChannel,f,r);
        }

	
	/** Create a sound channel. 
	 * @return The channel created, or null if the library failed to create it. */	
	public static SoundChannel create()
	{
		return create(null);
	}
	/** Create a sound channel. 
	 * @param jrock The rock to give this channel.
	 * @return The channel created, or null if the library failed to create it. */	
	public static SoundChannel create(Object jrock)
	{
		int irock = castRock(jrock);
		int id = Glk.jniglk_schannel_create(irock);	
		SoundChannel f = findSoundChannel(id);
		if (f != null) f.m_jrock = jrock;
		return f;
	}

	protected void finalize() { if (m_dispid != 0) destroy(); }
	/** Destroy this sound channel. If a sound is playing it will
	 * be stopped and no notification event will be sent. */
	public void destroy()
	{
		Glk.jniglk_schannel_destroy(m_dispid);
		super.destroy();
	}
	
	/** Play a sound.  
	 * @param snd The numbber of the sound to play. 
	 * @return true if the sound started playing, else false. */
	public boolean play(int snd)
	{
		return Glk.jniglk_schannel_play(m_dispid, snd);
	}

	/** Play a sound.  
	 * @param snd The number of the sound to play. 
	 * @param repeats The number of times to repeat; -1 means forever.
	 * @param notify If nonzero, request a sound notification event when the sound stops playing.
	 * @return true if the sound started playing, else false. */
	public boolean play(int snd, int repeats, int notify)
	{
		return Glk.jniglk_schannel_play_ext(m_dispid, snd, repeats, notify);
	}

	/** Stop any sound in this channel. There will be no notification event. */	
	public void stop()
	{
		Glk.jniglk_schannel_stop(m_dispid);
	}
	
	/** Set the volumne in this channel.	
	 * Not all libraries support this function. You should test the
	 * gestalt_SoundVolume selector before you rely on it.
 	 * Full volume is 0x10000. Half volume is 0x8000 etc. 
	 * @param vol The volume to set.
	 */ 	
	public void setVolume(int vol)
	{
		Glk.jniglk_schannel_set_volume(m_dispid, vol);
	}

	/** Give the library a hint about whether the given sound should
	 * be loaded or not. If flag is true, the library may preload the
	 * sound or do other initialization, so that play() will be
	 * faster. If flag is false, the library may release memory or other
	 * resources associated with the sound.
	 * @param snd The number of the sound to hint about.
	 * @param flag True to hint that the sound should be preloaded; false to hint that it should be unloaded.
	 */
	public static void loadHint(int snd, boolean flag)
	{
		Glk.jniglk_sound_load_hint(snd, flag);
	}


        long getGlkRock()       
        {
                return Glk.jniglk_schannel_get_rock(m_dispid);
        }
}


