
package org.ifarchive.glk;

/** The UniversalUnion is a Glk dispatch argument.
 * <p>Use these if you want to marshal your own arguments to Glk.dispatchCall(). 
 * For example, to call glk_select():</p>
 *<pre> 
 *	Event ev;
 * 	UniversalUnion args[] = new UniversalUnion[5];
 * 	args[0] = new UniversalUnion(true);	// pointer flag
 * 	args[1] = new UniversalUnion(UniversalUnion.UINT, 0);	// uint
 * 	args[2] = new UniversalUnion(null, OpaqueObject.clWindow);
 *						// null pointer to Window
 *	args[3] = new UniversalUnion(UniversalUnion.UINT, 0);	// uint
 *	args[4] = new UniversalUnion(UniversalUnion.UINT, 0);	// uint
 *	Glk.dispatchCall(0x00C0, args);		// call glk_event()
 * 	ev.type = args[1].intVal();
 *	ev.win  = (Window)(args[2].opaqueVal);
 *	ev.val1 = args[3].longVal();
 *	ev.val2 = args[4].longVal();
 *</pre> 
 * <p>note that unlike in C-Glk, you have to populate args[] even for parameters
 * that are only used for output.</p>
 * <p>Here's an example using arrays to do glk_put_buffer_stream():
 * <pre>
 *     byte buf[64];
 *     ...populate buf...
 *     UniversalUnion args[] = new UniversalUnion[4];
 *     args[0] = new UniversalUnion(myStream, OpaqueObject.clStream); // stream
 *     args[1] = new UniversalUnion(true);		// pointer flag
 *     args[2] = new UniversalUnion(buf);		// byte array
 *     args[3] = new UniversalUnion(UniversalUnion.UINT, buf.length);	// buffer length
 *     Glk.dispatchCall(0x0085, args);			// glk_put_buffer()
 *     args[2].release();	// All arrays and strings must be released
 * </pre>
 * @see Glk#dispatchCall(int,org.ifarchive.glk.UniversalUnion[])
 * @author John Elliott
 * @version 1.0 */

public class UniversalUnion
{
	int  m_type = 0;	
	long m_uint = 0;	// m_type = 1
	int  m_sint = 0;	// m_type = 2
	int  m_dispid = 0;	// m_type = 3
	int  m_glkClass = 0;	// m_type = 3 (again)
	char m_uch = 0;		// m_type = 4
	char m_sch = 0;		// m_type = 5
	char m_ch = 0;		// m_type = 6
	int  m_charstrid = 0;	// m_type = 7
	int  m_bufid = 0;	// m_type = 8
	int  m_ptrflag = 0;	// m_type = 9	

	/** This object represents an unsigned integer.
	 * @see #getType() */
	public static final int UINT = 1;
	/** This object represents a signed integer. 
	 * @see #getType() */
	public static final int SINT = 2;
	/** This object represents an opaque object. 
	 * @see #getType() */
	public static final int OPAQUE = 3;
	/** This object represents an unsigned character. 
	 * @see #getType() */
	public static final int UCHAR = 4;
	/** This object represents a signed character. 
	 * @see #getType() */
	public static final int SCHAR = 5;
	/** This object represents a native character. 
	 * It can be signed or unsigned depending on the underlying C compiler.
	 * @see #getType() */
	public static final int CHAR = 6;
	/** This object represents a string. 
	 * @see #getType() */
	public static final int STRING = 7;
	/** This object represents an array.
	 * @see #getType() */
	public static final int ARRAY = 8;
	/** This object represents a pointer flag. 
	 * @see #getType() */
	public static final int PTRFLAG = 9;

	/** Get the type of this object */
	public int getType() { return m_type; }	

	boolean m_needRelease = false;

	/** Produce a printable representation of this object. 
	 * @return The printed representation. */
	public String toString()
	{
		String s = "UniversalUnion: type = " + Integer.toString(m_type);
		switch(m_type)
		{
			case UINT:    s += " uint = "    + Long.toString(m_uint); break;
			case SINT:    s += " sint = "    + Integer.toString(m_sint); break;
			case OPAQUE:  s += " dispid = "  + Integer.toString(m_dispid); 
				      s += " class = "   + Integer.toString(m_glkClass); break;
			case UCHAR:   s += " uch = "     + m_uch; break;
			case SCHAR:   s += " sch = "     + m_sch; break;
			case CHAR:    s += " ch = "      + m_ch; break;
			case STRING:  s += " strid = "   + Integer.toString(m_charstrid); break;
			case ARRAY:   s += " bufid = "   + Integer.toString(m_bufid); break;
			case PTRFLAG: s += " ptrflag = " + Integer.toString(m_ptrflag); break;
		}	
		return s;
	}

	/** Construct a UniversalUnion that holds a character. 
	 * @param type UCHAR, SCHAR or CHAR
	 * @param v The character to pass in. */
	public UniversalUnion(int type, char v)
	{
		m_type = type;
		switch(type)
		{
			case UCHAR: m_uch = v; break;	
			case SCHAR: m_sch = v; break;	
			case CHAR: m_ch = v; break;	
		}
	}

	/** Construct a UniversalUnion that holds an unsigned int. 
	 * @param v The value to pass in.*/
	public UniversalUnion(long v)
	{
		m_type = UINT;
		m_uint = v;	
	}

	/** Construct a UniversalUnion that holds an opaque object.
	 * @param type must be 3.
	 * @param v The object's dispatch ID.
	 * @param g The object's Glk class.
	 * @see OpaqueObject#clWindow 
	 * @see OpaqueObject#clStream 
	 * @see OpaqueObject#clFileRef */
	UniversalUnion(int type, int v, int g)
	{
		this(type,v);
		m_glkClass = g;	
	}
	/** Construct a UniversalUnion that holds an opaque object. 
	 * @param o The object to store. 
	 * @param glkc The object's Glk class (used only if the object itself is null).
	 */
	public UniversalUnion(OpaqueObject o, int glkc)
	{	
		m_type = OPAQUE;
		if (o == null) 
		{
			m_glkClass = glkc;
			m_dispid = 0;
		}
		else
		{
			m_glkClass = o.m_glkClass;
			m_dispid   = o.m_dispid;
		}
	}

	/** Construct a UniversalUnion that holds a pointer flag.
	 * @param b true if the pointer is valid, false if it's null. */
	public UniversalUnion(boolean b)
	{
		m_type = PTRFLAG;
		m_ptrflag = b ? 1 : 0;
	}

	/** Construct a UniversalUnion that holds an integer.
	 * @param type UINT or SINT
	 * @param v Value of integer.*/
	public UniversalUnion(int type, int v)
	{
		m_type = type;
		switch(type)
		{
			case UINT:    m_uint      = v; break;
			case SINT:    m_sint      = v; break;
			case OPAQUE:  m_dispid    = v; break;
			case STRING:  m_charstrid = v; break;
			case ARRAY:   m_bufid     = v; break;
			case PTRFLAG: m_ptrflag   = v; break;
		}
	}

	/** Construct a UniversalUnion that holds a C string.
	 * You must call release() on this object after you have 
	 * made the Glk call that uses it.
	 * @see #release()
	 * @param s The string to store. */
	public UniversalUnion(String s)
	{
		byte[] bytes = Glk.stringToBuffer(s, true);
		m_charstrid = GlkArray.create(bytes);
		m_type = STRING;	
		m_needRelease = true;
	}

	/** Construct a UniversalUnion that holds an array of bytes.
	 * You must call release() on this object after you have made 
	 * the Glk call that uses it.
	 * @see #release()
	 * @param bytes The array to store. */
	public UniversalUnion(byte []bytes) 
	{
		m_bufid = GlkArray.create(bytes);
		m_type = ARRAY;	
		m_needRelease = true;
	}

	/** Construct a UniversalUnion that holds an array of ints.
	 * You must call release() on this object after you have made 
	 * the Glk call that uses it.
	 * @see #release()
	 * @param ints The array to store. */
	public UniversalUnion(int[] ints) 
	{
		m_bufid = GlkArray.create(ints);
		m_type = ARRAY;	
		m_needRelease = true;
	}

	/** Construct a UniversalUnion that holds an array of shorts.
	 * You must call release() on this object after you have made 
	 * the Glk call that uses it.
	 * @see #release()
	 * @param shorts The array to store. */
	public UniversalUnion(short[] shorts) 
	{
		m_bufid = GlkArray.create(shorts);
		m_type = ARRAY;	
		m_needRelease = true;
	}

	/** Construct a UniversalUnion that holds an array of longs.
	 * You must call release() on this object after you have made 
	 * the Glk call that uses it.
	 * @see #release()
	 * @param longs The array to store. */
	public UniversalUnion(long[] longs) 
	{
		m_bufid = GlkArray.create(longs);
		m_type = ARRAY;	
		m_needRelease = true;
	}

	/** Construct a UniversalUnion that holds an array of characters.
	 * You must call release() on this object after you have made 
	 * the Glk call that uses it.
	 * @see #release()
	 * @param chars The array to store. */
	public UniversalUnion(char[] chars) 
	{
		m_bufid = GlkArray.create(chars);
		m_type = ARRAY;	
		m_needRelease = true;
	}


	/** Release any arrays or character strings associated with this object. 
	 * @return false if Glk has retained the array this object represents, else true.
	 */
	public boolean release()
	{
		m_needRelease = false;
		if (m_type == STRING) return GlkArray.destroy(m_charstrid);
		if (m_type == ARRAY) return GlkArray.destroy(m_bufid);
		return true;
	}

	/** Release any arrays or character strings. Unfortunately you can't
	 * guarantee finalize() will be called, which is why release() is 
	 * there. 
	 * @see #release()
	 */
	protected void finalize()
	{
		if (m_needRelease) release();
	}

	/** If this object holds something that can be expressed as a long,
	 * return it. 
	 * @return The value, if possible; else -1. */
	public long longVal()
	{
		switch(m_type)
		{
			case 1: return m_uint;
			case 2: return m_sint;
			case 4: return m_uch;
			case 5: return m_sch;
			case 6: return m_ch;
			case 9: return m_ptrflag;
		}
		return -1;
	}

	/** If this object holds something that can be expressed as an integer,
	 * return it. 
	 * @return The value, if possible; else -1. */
	public int intVal()
	{
		return (int)longVal();
	}

	/** If this object holds something that can be expressed as a character,
	 * return it. 
	 * @return The value, if possible; else 0xFFFF. */
	public char charVal()
	{
		return (char)longVal();
	}

	/** If this object holds something that can be expressed as an opaque 
	 * object, return it.
	 * @return an opaque object, or null. */
	public OpaqueObject opaqueVal()
	{
		if (m_type == OPAQUE) return OpaqueObject.find(m_glkClass, m_dispid);
		return null;
	}
	

	/* Decode this as a Java object */
	Object decode()
	{
		switch(m_type)
		{
			case UINT:   return new Long(m_uint);
                        case SINT:   return new Integer(m_sint);
                        case OPAQUE: return OpaqueObject.find(m_glkClass, m_dispid);
			case UCHAR:  return new Character(m_uch);	
			case SCHAR:  return new Character(m_sch);	
			case CHAR:   return new Character(m_ch);
		}
		return null;
	}


        void addArg(boolean pass1, int nu, UniversalUnion args[])
        {
                if (pass1) return;
                args[nu] = this;
        }


/** Encode an argument (passed as an Object) into one or more UniversalUnions.
 *  The object to encode must be of an appropriate type:
 *  <ul>
 *  <li>For a primitive type, must be of class Number, or null.
 *  <li>For a string, must be a char[], byte[], String or null.
 *  <li>For an opaque object, must be of class OpaqueObject, Number or null.
 *  <li>For a reference to something, must be of class Reference, or null.
 *      The target of the Reference must in turn be of the correct type.
 *  <li>For a struct, must be an array of Object, one entry for each member.
 *  </ul> 
 * @param pass1 Set to true not to encode, but just to count the number of 
 * @param arg   The object to encode. 
 * @param proto The parsed prototype for this argument.
 * @param nu    The first UniversalUnion to populate.
 * @param uarr  The array of UniversalUnions to populate. In pass 1 this is null.
 * @return The next entry to populate.
 */
	static int encodeObject(boolean pass1, Object arg, 
				Prototype proto, int nu, UniversalUnion uarr[])
	{
		int type = 0, glkClass = -1;
		int dispid;
		Object structArgs[];
		Number numVal;
		UniversalUnion u;

		// Deal with references (except to arrays, which are done
		// further down)
		if (proto.isRef() && (!proto.isArray()) )
		{
			Reference r = null;

			if (arg != null && arg instanceof Reference) 
				r = (Reference) arg;
			if (proto.isNotNull() && (arg == null)) r = new Reference();	

			if (r == null) 	// Null reference argument
			{
				new UniversalUnion(false).addArg(pass1, nu++, uarr);
				return nu;
			}
			// Pointer is valid.
			new UniversalUnion(true).addArg(pass1, nu++, uarr);
			arg = r.target;
		}
		// Convert from Glk encoding to internal type encoding 
		switch(proto.major)
		{
			case 'I': if (proto.minor == 's') type = SINT;
				  else type = UINT;
				  break;
			case 'C': if      (proto.minor == 'u') type = UCHAR;
				  else if (proto.minor == 's') type = SCHAR;
				  else type = CHAR;
				  break;
			case 'Q': type = OPAQUE;
				  glkClass = proto.minor - 'a'; 
				  break;
//			case 'F': type = FLOAT;	
//				  break;
			case 'S': type = STRING;
				  break;
			case '[': 
			// Structure. The argument must be an array of objects,
			// or null; if it's null, make it into an array of nulls.
				  structArgs = (Object[])arg;
				  if (structArgs == null) structArgs = new Object[proto.structure.length];
			// Encode each member of the struct.
				  for (int np = 0; np < proto.structure.length; np++)
				  {
					nu = encodeObject(pass1, structArgs[np], 
						proto.structure[np], nu, uarr);
				  }
				  return nu;
			default:	// Can't encode because don't know how
				  return nu;
		}
		if (proto.isArray())
		{
			if (arg == null)	// Null array pointer
			{
				new UniversalUnion(false).addArg(pass1, nu++, uarr);
				return nu;
			}
			new UniversalUnion(true).addArg(pass1, nu++, uarr);
			byte [] barray;
			int  [] iarray;
			int bufid = 0;	
			int len = -1;
			switch(type)
			{
				case 1:
				case 2: iarray = (int[])  arg;
					len = iarray.length;
					bufid = GlkArray.create(iarray);
					break;
				case 4:
				case 5:
				case 6: barray = (byte[]) arg; 
					len = barray.length;
					bufid = GlkArray.create(barray);
					break;
				default: System.err.println("jniglk: Cannot marshal an array of type " + Integer.toString(type));
					break;
			}
			if (len == -1) return nu;	
			new UniversalUnion(ARRAY, bufid).addArg(pass1, nu++, uarr);
			new UniversalUnion(len).addArg(pass1, nu++, uarr);
			return nu;
		}
		numVal = null;
		if (arg != null && arg instanceof Number) numVal = (Number)arg;
		else		numVal = new Integer(0);
		switch(type)
		{
			default:
				System.err.print("Can't encode type " + 
					Integer.toString(type));
				System.err.println(" " + proto.toString());
			case UINT:
			u = new UniversalUnion(numVal.longValue());
			u.addArg(pass1, nu++, uarr);
			break;

			case SINT:
			u = new UniversalUnion(SINT, numVal.intValue());
			u.addArg(pass1, nu++, uarr);
			break;
			// Encode an opaque object. I'll allow either
			// the object itself or its dispatch ID; what
			// comes out will always be the object.
			case OPAQUE:	
			if      (arg == null) dispid = 0;
			else if (arg instanceof OpaqueObject)	dispid = ((OpaqueObject)arg).m_dispid;
			else dispid = numVal.intValue();
			// Add the encoded object.	
			u = new UniversalUnion(OPAQUE, dispid, glkClass);
			u.addArg(pass1, nu++, uarr);
			break;

			case UCHAR: case SCHAR: case CHAR:
			u = new UniversalUnion(type, (char)numVal.intValue());
			u.addArg(pass1, nu++, uarr);
			break;

			case STRING: // Character string
			if      (arg == null) dispid = 0;
			else if (arg instanceof byte[]) dispid = GlkArray.create((byte[])arg);
			else if (arg instanceof char[]) dispid = GlkArray.create(Glk.charsToBuffer ((char[])arg, true));
			else if (arg instanceof String) dispid = GlkArray.create(Glk.stringToBuffer((String)arg, true));
			else dispid = 0;
			u = new UniversalUnion(STRING, dispid);
			u.addArg(pass1, nu++, uarr);
			break;
		}
	return nu;
	}


	static int decodeObject(Object args[], int na, 
				Prototype proto, int nu, UniversalUnion uarr[])
	{
		// Arrays are quite easy, because GlkArray.destroy() unmarshals
		// them. 
		if (proto.isArray())
		{
			int valid = uarr[nu++].m_ptrflag;
			if (valid == 0) return nu;	// Null array pointer
		// If the array was retained this will fail. We don't care.
			uarr[nu++].release();
			nu++;	// Skip array length
			return nu;	
		}

		// We only decode outbound reference parameters. However, 
		// we still have to skip over the others, and release
		// strings.
		if ((proto.modifiers & Prototype.MOD_REF_OUT) == 0) 
		{
			if (uarr[nu].m_type == 7) uarr[nu].release();
			// Skip by using encodeObject in its "sizing" mode.
			return encodeObject(true, args[na], proto, nu, null);
		}

		if (uarr[nu++].m_ptrflag == 0) return nu;	// Null reference
		if (proto.major == '[')	// Structure
		{
			Object[] structArgs = new Object[proto.structure.length];
			for (int np = 0; np < proto.structure.length; np++)
			{
				switch(uarr[nu].m_type)
				{
					case UINT:  case SINT:  case OPAQUE:
					case UCHAR: case SCHAR: case CHAR:
						structArgs[np] = uarr[nu++].decode();	
						break;
					default:
						System.err.println("Can't decode " + proto.structure[np].toString());
						break;
				}

			}
			args[na] = new Reference(structArgs);
			return nu;
		}
		switch(uarr[nu].m_type)
		{
				case UINT:  case SINT:  case OPAQUE:
				case UCHAR: case SCHAR: case CHAR:
				args[na] = new Reference(uarr[nu++].decode());
				return nu;
			// 7 = character string, never returned
			// 8 = array, handled higher up
			// 9 = pointer flag, never returned	
			default:
				System.err.println("Can't decode " + proto.toString());
		}	
		return nu;
	}
}
