/*
	HE_EPOC.CPP (by Simon Quinn)

	Ported from the UNIX/GCC version using STDLIB. Unfortunetly E32 Console has very 
	limited screen handling functions so I added an Eikon wrapper to use 
	the EIKConsole (epoc_console.cpp). No other use is made of the Eikon environment.

*/

#include <e32base.h>

#include <eikcfdlg.h>
#include <eikon.rsg>
#include <eikfutil.h>

#include "heheader.h"
#include "epoc_console.h"


CAdvConsoleClient *iConsole;	//EIK Console to interact with


/* Function prototypes: */
void hugo_addcommand(void);
void hugo_restorecommand(int);
int hugo_color(int c);

/* Specific to hegcc.c: */
void ConstrainCursor(void);

#define HISTORY_SIZE    16      /* for command-line editing */
int hcount = 0;
char *history[HISTORY_SIZE];

/* Now some variables and constants to manage text/graphics display--
   all specific to hegcc.c (taken from hedjgpp.c):
*/
int text_windowleft, text_windowtop, text_windowright, text_windowbottom,
	text_windowwidth;
int insert_mode = true;
int print_cursor = false;       /* probably never be needed */
int current_text_col, current_text_row, current_text_color, current_back_color;

#if defined (DEBUGGER)
void *AllocMemory(size_t size);
extern int fake_graphics_mode;
#endif


/*
    MEMORY ALLOCATION:

    hugo_blockalloc(), and hugo_blockfree() are necessary because of
    the way MS-DOS handles memory allocation across more than 64K.
    For most systems, these will simply be normal ANSI function calls.
*/

void *hugo_blockalloc(long num)
{
        return malloc(num * sizeof(char));
}

void hugo_blockfree(void *block)
{
        free(block);
}


/*
    FILENAME MANAGEMENT:

    Different operating systems will have their own ways of naming
    files.  The following routines are simply required to know and
    be able to dissect/build the components of a particular filename,
    storing/restoring the compenents via the specified char arrays.

    For example, in MS-DOS:

        hugo_splitpath("C:\HUGO\FILES\HUGOLIB.H", ...)
                becomes:  C:, HUGO\FILES, HUGOLIB, H

    and

        hugo_makepath(..., "C:", "HUGO\FILES", "HUGOLIB", "H")
                becomes:  C:\HUGO\FILES\HUGOLIB.H

    The appropriate equivalent nomenclature should be used for the
    operating system in question.
*/

void hugo_splitpath(char *path, char *drive, char *dir, char *fname, char *ext)
{
	char *file;
	char *extension;

	strcpy(drive,"");
	strcpy(dir,"");
	strcpy(fname,"");
	strcpy(ext,"");
	
	if ((file = strrchr(path,'/')) == 0)
	{
		if ((file = strrchr(path,':')) == 0) 
			file = path;
	}
	strncpy(dir,path,strlen(path)-strlen(file));
	*(dir+strlen(path)-strlen(file)) = 0;
	extension = strrchr(file,'.');
	if (extension != 0)
	{
		strncpy(fname,file,strlen(file)-strlen(extension));
		*(fname+strlen(file)-strlen(extension)) = 0;
		strcpy(ext,extension+1);
	}
	else 
		strcpy(fname,file);
	
	if (strcmp(dir, "") and fname[0]=='/') strcpy(fname, fname+1);
}

void hugo_makepath(char *path, char *drive, char *dir, char *fname, char *ext)
{
        if (*ext == '.') ext++;
        strcpy(path,drive);
        strcat(path,dir);
        switch (*(path+strlen(path)))
        {
        case '/':
        case ':':
                break;
        default:
                if (strcmp(path, "")) strcat(path,"/");
                break;
        }
        strcat(path,fname);
        if (strcmp(ext, "")) strcat(path,".");
        strcat(path,ext);
}


/*
    OVERWRITE:

    Checks to see if the given filename already exists, and prompts to
    replace it.  Returns true if file may be overwritten.
*/

int hugo_overwrite(char *f)
{
	FILE *tempfile;

	if (!(tempfile = fopen(f, "rb")))	/* if file doesn't exist */
		return true;

	fclose(tempfile);

	sprintf(pbuffer, "Overwrite existing \"%s\" (Y or N)?", f);
	RunInput();

	if (words==1 and (!strcmp(strupr(word[1]), "Y") or !strcmp(strupr(word[1]), "YES")))
		return true;

	return false;
}


/*
    CLOSEFILES:

    Closes all open files.  NOTE:  If the operating system automatically
    closes any open streams upon exit from the program, this function may
    be left empty.
*/

void hugo_closefiles()
{
}


/*
    GETFILENAME:

    Loads the name of the filename to save or restore (as specified by
    the argument <a>) into the line[] array.
*/

void hugo_getfilename(char *a, char *b)
{
	unsigned int i, p;

	sprintf(line, "Enter path and filename %s.", a);
	AP(line);

	sprintf(line,"%c(Default is %s): \\;", NO_CONTROLCHAR, b);
	AP(line);

	p = var[prompt];
	var[prompt] = 0;        /* null string */

	RunInput();

	var[prompt] = p;

	remaining = 0;

	strcpy(line, "");
	if (words==0)
		strcpy(line, b);
	else
	{
		for (i=1; i<=(unsigned int)words; i++)
			strcat(line, word[i]);
	}
}


/*
    GETKEY:

    Returns the next keystroke waiting in the keyboard buffer.  It is
    expected that hugo_getkey() will return the following modified
    keystrokes:

	up-arrow        11 (CTRL-K)
	down-arrow      10 (CTRL-J)
	left-arrow       8 (CTRL-H)
	right-arrow     21 (CTRL-U)

    Also, accented characters are converted to non-accented characters.
    Invalid keystrokes are returned as 255.
*/

int hugo_getkey(void)
{
	int b;

	TKeyCode key;

        /*
         * In case this is used in halfdelay() or nodelay() mode
         * need to make sure we don't pass bad data to getline()
         */
	key = iConsole->Getch();

	switch (key)
	{
		case EKeyUpArrow:
			{b = 11; break;}
		//For EPOC, leave leftarrow we will check code in hugo_getline
//		case EKeyLeftArrow:
//			{b = 8; break;}		//Why 8, this is delete?!!!!
		case EKeyRightArrow:
			{b = 21; break;}
		case EKeyDownArrow:
			{b = 10; break;}

		default:
			{b = key; break;}
			
	} 
	return b;
}

/*
    GETLINE

    Gets a line of input from the keyboard, storing it in <buffer>.

    (NOTE:  The function keys used here are QuickC/MS-DOS specific.
    They have been chosen to somewhat mirror the command-line editing
    keys of MS-DOS.  For other systems, the 'if (b==<value>)' lines
    will likely need to be changed.)

    If a similar library input function is available, it will almost
    certainly be preferable to use that; in QuickC such was unfortunately
    not the case.

    Note also that this input function assumes a non-proportional font.
*/

#define CURRENT_CHAR(c) ((c<(int)strlen(buffer))?buffer[c]:' ')

void hugo_getline(char *p)
{
	char ch[2];
	int a, b, thiscommand;
	int c;                          /* c is the character being added */
	int x, y, oldx, oldy;

NewPrompt:
	hugo_settextcolor(fcolor);
	hugo_setbackcolor(bgcolor);

	strcpy(buffer, "");
	c = 0;
	thiscommand = hcount;

	oldx = (currentpos+charwidth)/charwidth;
	oldy = currentline;
	y = oldy;
	x = oldx + hugo_strlen(p);

	hugo_print(p);

GetKey:

	hugo_settextpos(x, y);

	b = hugo_getkey();

	hugo_settextcolor(icolor);
	hugo_setbackcolor(bgcolor);

	/* Now, start key checking */
	switch (b)
	{
#if defined (DEBUGGER)
		case (9):                       /* Tab */
		{
			during_input = true;
			Debugger();
			during_input = false;

			/* If the debugger is stopping execution, quitting,
			   etc., a blank string must be returned in the
			   input buffer.
			*/
			if (debugger_collapsing)
			{
				strcpy(buffer, "");
				if (active_screen==GAME)
					hugo_scrollwindowup();
				return;
			}

			goto GetKey;
		}
#endif

		case (13):                      /* Enter */
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
/*                      addstr("  "); */

			hugo_print(ch);
			full = 0;

			if (script) fprintf(script, "%s%s\n", p, buffer);

			hugo_settextpos(1, y + 2);
			hugo_scrollwindowup();

			strcpy(buffer, Rtrim(buffer));
			hugo_addcommand();
#if defined (NO_LATIN1_CHARSET)
			hugo_stripaccents(buffer);
#endif
			return;
		}		
		case (8):		//Delete key
		{
			if (strlen(buffer)>0)
			{
				if (b==8)
				{
					if (c==0) goto GetKey;
					c--;
				}

				for (a=c; a<=(int)strlen(buffer); a++)
					buffer[a] = buffer[a+1];

				if (b==8) x--;

				hugo_settextpos(oldx+hugo_strlen(p), y);
				hugo_print(buffer);
				hugo_settextpos(oldx+hugo_strlen(p)+strlen(buffer), y);
				hugo_print("  ");
			}
			goto GetKey;
		} 
		case EKeyLeftArrow:               /* left-arrow */
		{
			if (c > 0)
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				x--, c--;
			}
			goto GetKey;
		}
		case EKeyRightArrow:
		case (21):                      /* right-arrow */
		{
			if (c<(int)strlen(buffer))
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				x++, c++;
			}
			goto GetKey;
		}
        case EKeyHome:
		case (1):			/* Ctrl+a */
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			c = 0;
			x = oldx + hugo_strlen(p);
			goto GetKey;
		}
		case EKeyEnd:
		case (5):			/* Ctrl+e */
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			c = strlen(buffer);
			x = oldx + hugo_strlen(p) + strlen(buffer);
			goto GetKey;
		}
		case (27):                      /* Escape */
		{
BlankLine:
			hugo_settextpos(oldx+hugo_strlen(p), y);

			memset(buffer, ' ', text_windowwidth);

			buffer[text_windowwidth-oldx-hugo_strlen(p)] = '\0';
			hugo_print(buffer);
			hugo_settextpos(oldx, y);

			goto NewPrompt;
		}

		case 10004:		//Zoom Out
		case 10003:		//Zoom In
			{
				iConsole->ScreenZoom();
				hugo_settextmode();
				hugo_clearfullscreen();
				goto NewPrompt;
			}

		case EKeyUpArrow:
		case (11):                      /* up-arrow */
		{
			if (--thiscommand<0)
			{
				thiscommand = 0;
				goto GetKey;
			}
			a = strlen(buffer);
RestoreCommand:
			hugo_restorecommand(thiscommand);
			x = oldx + strlen(buffer) + hugo_strlen(p);
			hugo_settextpos(oldx+hugo_strlen(p), y);
			hugo_print(buffer);
			while (a >= (int)strlen(buffer))
				{hugo_print(" ");
				a--;}
			hugo_settextpos(x-1, y);
			c = strlen(buffer);
			goto GetKey;
		}
		case EKeyDownArrow:
		case (10):                      /* down-arrow */
		{
			a = strlen(buffer);
			if (++thiscommand>=hcount) goto BlankLine;
			goto RestoreCommand;
		}
	}

	/* Disallow invalid keystrokes */
	if (b < 32 or b>=256) goto GetKey;
	
	/* Hugo circa v2.2 allowed '^' and '~' for '\n' and '\"',
	   respectively
	*/
	if (game_version<=22 and (b=='^' or b=='~')) goto GetKey;

	if (oldx + (int)strlen(buffer) >= text_windowwidth-2)
		goto GetKey;

	hugo_settextpos(x++, y);
	sprintf(ch, "%c", b);                   /* add the new character */
	hugo_print(ch);
	buffer[strlen(buffer)+1] = '\0';
	if (c<(int)strlen(buffer) and insert_mode)
	{
		hugo_settextpos(x, y);
		hugo_print(buffer+c);
		for (a=strlen(buffer); a>c; a--)
			buffer[a] = buffer[a-1];
	}
	buffer[c] = (char)b;

	c++;

	goto GetKey;
}


/* hugo_iskeywaiting

    Returns true if a keypress is waiting to be retrieved.
*/

int hugo_iskeywaiting(void)
{
	return false;
}

/*
    WAITFORKEY:

    Provided to be replaced by multitasking systems where cycling while
    waiting for a keystroke may not be such a hot idea.
*/

int hugo_waitforkey(void)
{
        iConsole->SetPos(current_text_col-1, current_text_row-1);
        
        return hugo_getkey();
}


/* hugo_timewait

    Waits for 1/n seconds.  Returns false if waiting is unsupported.
*/

int hugo_timewait(int n)
{

	User::After(1000000 / n);

//	wtimeout(current_window, 1000/n);

	return true;
}


/*
    COMMAND HISTORY:

    To store/retrieve player inputs for editing.
*/

void hugo_addcommand(void)
{
	int i;

	if (!strcmp(buffer, "")) return;

        if (hcount>=HISTORY_SIZE)
	{
		hugo_blockfree(history[0]);
		for (i=0; i<HISTORY_SIZE-1; i++)
			history[i] = history[i+1];
		hcount = HISTORY_SIZE-1;
	}

	/* Because the debugger might use (up to) all available memory for
	   code line storage, a different means of memory allocation is
	   needed (at least in MS-DOS due to limited available memory to
	   begin with).
	*/
#if !defined (DEBUGGER)
	if ((history[hcount] = (char *)hugo_blockalloc((long)((strlen(buffer)+1)*sizeof(char))))==NULL)
#else
	if ((history[hcount] = (char *)AllocMemory((size_t)((strlen(buffer)+1)*sizeof(char))))==NULL)
#endif
	{
		hugo_blockfree(history[0]);
		if (hcount)
		{
			for (i=0; i<hcount; i++)
				history[i] = history[i+1];
			hcount--;
		}
		return;
	}

	for (i=0; i<=(int)strlen(buffer); i++)
		history[hcount][i] = buffer[i];
	hcount++;
}

void hugo_restorecommand(int n)
{
	int i;

	if (n < 0 or (n>=hcount and hcount!=HISTORY_SIZE-1)) return;

	i = 0;
	do
		buffer[i] = history[n][i];
	while (history[n][i++]!='\0');
}


/*
    DISPLAY CONTROL:

    These functions are specific library calls to QuickC/MS-DOS
    for screen control.  Each _function() should be replaced by
    the equivalent call for the operating system in question.  If
    a particular operation is unavailable, in some cases it may be
    left empty.  In other cases, it may be necessary to write a
    brief function to achieve the desired effect.
*/


void hugo_setgametitle(char *t)
{
}

void hugo_clearfullscreen(void)
{
/* Clears everything on the screen, moving the cursor to the top-left
   corner of the screen */

	iConsole->ClearScreen();

	/* Must be set: */
	currentpos = 0;
	currentline = 1;
}


void hugo_clearwindow(void)
{
/* Clears the currently defined window, moving the cursor to the top-left
   corner of the window */

	iConsole->ClearChars(TRect(text_windowleft-1, text_windowtop-1, text_windowright, text_windowbottom), 0);

	/* Must be set: */
	currentpos = 0;
	currentline = 1;
}


void hugo_settextmode(void)
{
/* This function does whatever is necessary to set the system up for
   a standard text display */
	charwidth = FIXEDCHARWIDTH = 1;       /* again, for non-proportional */
	lineheight = FIXEDLINEHEIGHT = 1;

	/* Must be set: */
	SCREENWIDTH = iConsole->ScreenSize().iWidth;
	SCREENHEIGHT = iConsole->ScreenSize().iHeight;

	/* Must be set: */
	hugo_settextwindow(1, 1,
		SCREENWIDTH/FIXEDCHARWIDTH, SCREENHEIGHT/FIXEDLINEHEIGHT);
}

void hugo_settextwindow(int left, int top, int right, int bottom)
{
/* Once again, the arguments for the window are passed using the same
   unit of measurement as SCREENRIGHT, SCREENLEFT, PAGETOP, etc., whether
   that is pixels or numbers of characters.

   The text window, once set, represents the scrolling/bottom part of the
   screen. */

	/* Character, not pixel, coordinates: */
	text_windowtop = top;
	text_windowbottom = bottom;
	text_windowleft = left;
	text_windowright = right;
	text_windowwidth = text_windowright-text_windowleft+1;

	/* Must be set: */
	/* (Engine-required parameters) */
	physical_windowleft = (left-1)*FIXEDCHARWIDTH;
	physical_windowtop = (top-1)*FIXEDLINEHEIGHT;
	physical_windowright = right*FIXEDCHARWIDTH-1;
	physical_windowbottom = bottom*FIXEDLINEHEIGHT-1;
	physical_windowwidth = (right-left+1)*FIXEDCHARWIDTH;
	physical_windowheight = (bottom-top+1)*FIXEDLINEHEIGHT;

	ConstrainCursor();
}

void hugo_settextpos(int x, int y)
{
/* The top-left corner of the current active window is (1, 1).

   (In other words, if the screen is being windowed so that the top row
   of the window is row 4 on the screen, the (1, 1) refers to the 4th
   row on the screen, and (2, 1) refers to the 5th.)

   This function must also properly set currentline and currentpos (where
   currentline is a the current character line, and currentpos may be
   either in pixels or characters, depending on the measure being used).
*/
	/* Must be set: */
	currentline = y;
	currentpos = (x-1)*charwidth;   /* Note:  zero-based */

	current_text_col = text_windowleft-1+x;
	current_text_row = text_windowtop-1+y;
	ConstrainCursor();

	iConsole->SetPos(current_text_col-1, current_text_row-1);
}

void ConstrainCursor(void)
{
	/* ConstrainCursor() is needed since HEDJGPP.C's cursor-
	   positioning doesn't automatically pay attention to any
	   currently-defined window.
	*/
	if (current_text_col > text_windowright) current_text_col = text_windowright;
	if (current_text_col < text_windowleft) current_text_col = text_windowleft;
	if (current_text_row > text_windowbottom) current_text_row = text_windowbottom;
	if (current_text_row < text_windowtop) current_text_row = text_windowtop;
}

// Output is buffered to avoid slow printing in EPOC console.
// Characters are written as a string instead of a single char each loop
// Returns and new lines force a flush of the printable string
void hugo_print(char *a)
{
	char just_repositioned = false;
        char last_was_newline = false;
	int i, len;

	TBuf<256> tempchar;

	len = strlen(a);


	for (i=0; i<len; i++)
	{
		last_was_newline = false;

		switch (a[i])
		{
			case '\n':
//sqdebug
				if(tempchar.Length() > 0)
				{
					iConsole->Write(tempchar);
					tempchar.Zero();
				}

				++current_text_row;
				iConsole->SetPos(current_text_col - 1, current_text_row - 1);
				just_repositioned = true;
				last_was_newline = true;
				break;
			case '\r':
CarriageReturn:
				if(tempchar.Length() > 0)
				{
					iConsole->Write(tempchar);
					tempchar.Zero();
				}
				current_text_col = text_windowleft;
				iConsole->SetPos(current_text_col - 1, current_text_row - 1);
				just_repositioned = true;
				break;
			default:
			{
				tempchar.Append(a[i]);

				current_text_col++;

				if (current_text_col > text_windowright)
				{
					if(tempchar.Length() > 0)
					{
						iConsole->Write(tempchar);
						tempchar.Zero();
					}
					iConsole->SetPos(current_text_col - 1 - 1, current_text_row - 1);
					current_text_col = text_windowleft;
					++current_text_row;
				}
				just_repositioned = false;
			}
		}

		if (current_text_row > text_windowbottom)
		{
			current_text_row = text_windowbottom;
			hugo_scrollwindowup();
		}
	}

	if(tempchar.Length() > 0)
	{
		iConsole->Write(tempchar);
		tempchar.Zero();
	}


	if (!just_repositioned) 
		iConsole->SetPos(current_text_col - 1, current_text_row - 1);

        /* Because '\n' should imply '\r' */
	if (last_was_newline)
	{
		last_was_newline = false;
		goto CarriageReturn;
	}
}


void hugo_scrollwindowup()
{
	iConsole->ScrollChars(TRect(text_windowleft - 1,text_windowtop - 1,text_windowright,text_windowbottom), TPoint(0,1));
}


/* The following local varibles are used to keep track of the current
   foreground and background colors. */
int local_fore_color= DEF_FCOLOR;
int local_back_color = DEF_BGCOLOR;
int local_back_inverse = 0;			// Is the text colour equivalent of Inverse?
int real_attributes = 0;

// Converts the Hugo specified attributes, into EPOC attributes.
// Real_attributes stores the attribute settings between function calls
// so that they are switched on and off, never reset to the default

#define REAL_BOLD  1
#define REAL_ITALIC  2
#define REAL_UNDERLINE 4
#define REAL_BRITE_COLOR 8

void hugo_real_attributes_mapper(int attrib)
{
	int curs_attr=ATT_NORMAL;	

	if (attrib & UNDERLINE_FONT)
		curs_attr|=ATT_UNDERLINE;
		
	if (attrib & ITALIC_FONT)
		curs_attr|=ATT_ITALIC;

	if (attrib & BOLD_FONT)
		curs_attr|=ATT_BOLD;

	if (local_back_inverse)			// Variable set in hugo_setbackcolor
		curs_attr|=ATT_INVERSE;

//	curs_attr |= ATT_COLORMASK | (local_fore_color & 0x0F) | ((((local_back_color/2) & 0x0F)<<4));

	iConsole->SetAttr(curs_attr);
}

void hugo_font(int f)
{
	real_attributes &= ~(REAL_BOLD | REAL_ITALIC | REAL_UNDERLINE);

	if (f & UNDERLINE_FONT)
		real_attributes |= REAL_UNDERLINE;

	if (f & ITALIC_FONT)
		real_attributes |= REAL_ITALIC;

	if (f & BOLD_FONT)
		real_attributes |= REAL_BOLD;

	hugo_real_attributes_mapper(real_attributes);
}


//Not currently doing anything in EPOC
void hugo_settextcolor(int c)   /* foreground (print) color */
{
    local_fore_color=hugo_color(c);

//	if(local_fore_color > 0)
//		local_fore_color = 15;

}

//Make sure the equivalent of inverse colours
//are printed using the inverse attribute
void hugo_setbackcolor(int c)   /* background color */
{
	local_back_inverse = 0;
	local_back_color=hugo_color(c);

	//Set the inverse flag if the colour isn't the standard background colour
	//and is not the same as the default_bgcolor, which can be changed by
	//certain stories and is not necessarily the same as the DEF_BGCOLOR
	if(local_back_color != DEF_BGCOLOR && local_back_color != default_bgcolor)
	{
		local_back_inverse = 1;
		local_back_color = DEF_FCOLOR;
	}

//	if(local_fore_color > 0)
//		local_fore_color = 15;
}

int hugo_gettextcolor()
{
	return local_fore_color;
}

int hugo_getbackcolor()
{
	return local_back_color;
}

int hugo_color(int c)
{
	/* Color-setting functions should always pass the color through
	   hugo_color() in order to properly set default fore/background
	   colors:
	*/

	if (c==16)      c = DEF_FCOLOR;
	else if (c==17) c = DEF_BGCOLOR;
	else if (c==18) c = DEF_SLFCOLOR;
	else if (c==19) c = DEF_SLBGCOLOR;
	else if (c==20) c = hugo_color(fcolor);
 
	return c;
}


/* CHARACTER AND TEXT MEASUREMENT

   	For non-proportional printing, these should be pretty much as
	described below, as screen dimensions will be given in characters,
	not pixels.

	For proportional printing, screen dimensions need to be in
	pixels, and each width routine must take into account the
	current font and style.

	The hugo_strlen() function is used to give the length of
	the string not including any non-printing control characters.
*/

int hugo_charwidth(char a)
{
	if (a==FORCED_SPACE)
		return FIXEDCHARWIDTH;	  /* same as ' ' */

	else if ((unsigned char)a >= ' ') /* alphanumeric characters */

		return FIXEDCHARWIDTH;    /* because printing 
					     is non-proportional */
	return 0;
}

int hugo_textwidth(char *a)
{
	int i, slen, len = 0;

	slen = strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else
			len += hugo_charwidth(a[i]);
	}

	return len;
}

int hugo_strlen(char *a)
{
	int i, slen, len = 0;

	slen = strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else len++;
	}

	return len;
}


void hugo_init_screen(void)
{
}

void hugo_cleanup_screen(void)
{
        /*
         * This will return the terminal to its original settings
         */
	hugo_settextcolor(DEF_FCOLOR);
	hugo_setbackcolor(DEF_BGCOLOR);
	hugo_font(0);
	hugo_clearfullscreen();
}


int hugo_hasgraphics(void)
{
	return false;
}

int hugo_displaypicture(FILE *infile, long len)
{
        fclose(infile);         /* since infile will be open */
        return 1;
}

int hugo_playmusic(HUGO_FILE infile, long reslength, char loop_flag)
{
	fclose(infile);
	return true;	/* not an error */
}

void hugo_musicvolume(int vol)
{}

void hugo_stopmusic(void)
{}

int hugo_playsample(HUGO_FILE infile, long reslength, char loop_flag)
{
	fclose(infile);
	return true;	/* not an error */
}

void hugo_samplevolume(int vol)
{}

void hugo_stopsample(void)
{}

// The controlling function for the whole program, works similar to
// C main().
// Creates the console and starts the main program
void ConsoleMainL()
{
	char *argv[2] = {"","                                                                                 "};
	TInt numargs=2;
	RProcess me;

    TBuf<128> args=me.CommandLine();
	if(args.Length() == 0)
		numargs=1;
	args.ZeroTerminate();

	argv[1] = (char *) args.Ptr();

//	iConsole = NewAdvConsole();
	iConsole = new CAdvConsoleClient;

	//Create the title with the opened file
	TBuf<128> tempname=_L("Hugo ");
	tempname.Append(args);

	((CConsoleBase *)iConsole)->Create(tempname, TSize(80, 25));

	he_main(numargs, argv);

	delete iConsole;
//	CleanupStack::PopAndDestroy();

	CloseSTDLIB();

}

GLDEF_C TInt E32Main()
{
	__UHEAP_MARK;
	CTrapCleanup* cleanupStack = CTrapCleanup::New();
	TRAPD(error, ConsoleMainL());
	__ASSERT_ALWAYS(!error, User::Panic(_L("PEP"),error));
	delete cleanupStack;
	__UHEAP_MARKEND;
	return 0;
}


/* These functions are needed only when program is a DLL, which it is
 * under WINS.
 */
#if defined(__WINS__)
EXPORT_C TInt WinsMain(TAny *p)
{
//    RealCommandLine.Set(*(TDesC *) p);
    //process.CommandLine() = _L("-U Reebo");     // This doesn't work because CommandLine returns a copy.
    return E32Main();
}

GLDEF_C TInt E32Dll(TDllReason)
{
    return(KErrNone);
}
#endif
