/*
	hepalm.c

	Copyright (c) 1995-2006 by Kent Tessman
*/

#define printf(a)

#include "PalmHugo.h"
#include "PalmHugoRsc.h"
#include "heheader.h"
#include "small_fixed_font.h"

#include <PalmOS.h>


/* Function prototypes: */
int hugo_color(int c);
int hugo_hasgraphics(void);
void ConstrainCursor(void);
void DrawGraphicChar(char c, int x, int y);
void FlushBuffer(void);

Boolean getline_active = false;
int getline_x, getline_y;
Boolean processed_accelerator_key = false;
int current_text_color = 0, current_back_color = 0;
int current_input_font = 0;
int current_text_font = 0;
Boolean printed_something = false;
int newline_count = 0;
Boolean illegal_window = false, minimal_windowing = false;


/* Defined Hugo colors: */
#define HUGO_BLACK         0
#define HUGO_BLUE          1
#define HUGO_GREEN         2
#define HUGO_CYAN          3
#define HUGO_RED           4
#define HUGO_MAGENTA       5
#define HUGO_BROWN         6
#define HUGO_WHITE         7
#define HUGO_DARK_GRAY     8
#define HUGO_LIGHT_BLUE    9
#define HUGO_LIGHT_GREEN   10
#define HUGO_LIGHT_CYAN    11
#define HUGO_LIGHT_RED     12
#define HUGO_LIGHT_MAGENTA 13
#define HUGO_YELLOW        14
#define HUGO_BRIGHT_WHITE  15

/*
#define HISTORY_SIZE    16
int hcount = 0;
char *history[HISTORY_SIZE];
void hugo_addcommand(void);
void hugo_restorecommand(int n);
*/

#ifdef SCROLLBACK_DEFINED
char scrollback_buffer[SCROLLBACK_SIZE];
int scrollback_pos = 0;
#endif


/*
    MEMORY ALLOCATION:
    
    It's VERY IMPORTANT not to mix hugo_blockalloc() and MemPtrNew() calls in
    PalmHugo.  Memory allocated with hugo_blockalloc() must be freed with
    hugo_blockfree(); memory allocated with MemPtrNew() must be freed with
    MemPtrFree().
    
    (This is because the memory allocation process for hugo_blockalloc()
    may change from just a simple MemPtrNew() call.)
*/

void *hugo_blockalloc(long num)
{
	return MemPtrNew(num);
}

void hugo_blockfree(void *block)
{
	if (block) MemPtrFree(block);
}


/*
    FILENAME MANAGEMENT:
*/

/* Palm OS doesn't have paths or extensions: */

void hugo_splitpath(char *path, char *drive, char *dir, char *fname, char *ext)
{
        strcpy(fname, path);
}

void hugo_makepath(char *path, char *drive, char *dir, char *fname, char *ext)
{
	strcpy(path, fname);
}


/* hugo_getfilename */

void hugo_getfilename(char *usage, char *def_filename)
{
	UInt16 form;
	
	if (def_filename==scriptfile)
	{
		file_select_type = PH_SCRIPT_TYPE;
		form = FileSaveForm;
	}
	else if (def_filename==recordfile)
	{
		file_select_type = PH_RECORD_TYPE;
		if (StrStr(usage, "record"))
			form = FileSaveForm;
		else
			form = FileOpenForm;
	}
	else
	{
		file_select_type = PH_SAVE_TYPE;
		if (StrStr(usage, "save"))
			form = FileSaveForm;
		else
			form = FileOpenForm;
	}
	
	RunDialog(form);
	if (StrCompare(file_selected, ""))
	{
		StrCopy(line, file_selected);
		return;
	}
	if (file_select_error==PH_NO_FILES)
	{
		PopupMessage("Open File", "No valid files available");
	}
	
	StrCopy(line, "");
}


/* hugo_overwrite

    Checks to see if the given filename already exists, and prompts to
    replace it.  Returns true if file may be overwritten.

    Again, it may be preferable to replace this with something fancier.
*/

int hugo_overwrite(char *f)
{
	Boolean found = false, result = true;
	int i;
	
	for (i=0; i<MemNumCards(); i++)
	{
		if (DmFindDatabase(i, f))
		{
			found = true;
			break;
		}
	}
	
	if (!found) return true;
	
	// Use buffer since the filename is probably in line, thanks
	// to great design
	sprintf(buffer, "Overwrite existing \"%s\"?", f);
	if (FrmCustomAlert(ConfirmAlert, buffer, "", "")==0)
	{
		strcpy(buffer, "");
		result = true;
	}
	else
	{
		strcpy(buffer, "");
		result = false;
	}
	
	RestoreWindow();
	
	return result;
}


/* hugo_closefiles

    Closes all open files.  NOTE:  If the operating system automatically
    closes any open streams upon exit from the program, this function may
    be left empty.
*/

void hugo_closefiles()
{
/*
	fclose(game);
	if (script) fclose(script);
	if (io) fclose(io);
	if (record) fclose(record);
*/
}


/* hugo_getkey

    Returns the next keystroke waiting in the keyboard buffer.  It is
    expected that hugo_getkey() will return the following modified
    keystrokes:

	up-arrow        11 (CTRL-K)
	down-arrow      10 (CTRL-J)
	left-arrow       8 (CTRL-H)
	right-arrow     21 (CTRL-U)
*/

#define MAX_KEYPRESSES 32
int keypress_count = 0;
int keypress_queue[MAX_KEYPRESSES];

void PushKeypress(int k)
{
	if (keypress_count==MAX_KEYPRESSES) return;
	keypress_queue[keypress_count++] = k;
}

int PullKeypress(void)
{
	int i, k;

	if (keypress_count==0) return 0;

	k = keypress_queue[0];
	for (i=0; i<keypress_count-1; i++)
		keypress_queue[i] = keypress_queue[i+1];
	keypress_count--;
	keypress_queue[keypress_count] = 0;
	
	// Mouse button
	if (k==1 && keypress_count>=2)
	{
		display_pointer_x = keypress_queue[0];
		display_pointer_y = keypress_queue[1];
		for (i=0; i<keypress_count-2; i++)
			keypress_queue[i] = keypress_queue[i+2];
		keypress_queue[keypress_count-2] = 0;
		keypress_queue[keypress_count-1] = 0;
		keypress_count-=2;
	}

	return k;
}

int hugo_getkey(void)
{
/*
	int b;

	if (...b is a function key...)
	{
		switch (...the function given by b...)
		{
			case ...up-arrow...:
				{b = 11;
				break;}
			case ...left-arrow...:
				{b = 8;
				break;}
			case ...right-arrow...:
				{b = 21;
				break;}
			case ...down-arrow...:
				{b = 10;
				break;}
		}
	}
	return b;
*/
	int k;

	InsPtSetLocation(current_text_x, current_text_y);
	InsPtSetHeight(lineheight);
	if (!promptmore_waiting) InsPtEnable(true);
	
	while (!keypress_count)
	{
		ProcessPalmOSEvents();
	}

	InsPtEnable(false);
	
	k = PullKeypress();
	
	if (k==10)      k = 13; // Enter
	else if (k==12) k = 10; // Down
	else if (k==28) k = 8;  // Left
	else if (k==29) k = 21; // Right
	else if (k==9)  k = 27; // Tab = Escape
	
	if (k < 0) k = (unsigned char)k;

	return k;
}


/*
    GETLINE

    Gets a line of input from the keyboard, storing it in <buffer>.
*/

void hugo_getline(char *prmpt)
{
	int tempfont = currentfont;
	FormPtr frmP;
	FieldPtr fieldP;
	UInt16 fieldindex;
	RectangleType rect;
	int local_icolor, local_bgcolor;
	
	got_first_input = true;
	
GetNewLine:

	// Print the prompt
	hugo_settextcolor(fcolor);
	hugo_setbackcolor(bgcolor);
	hugo_print(prmpt);
	FlushBuffer();
	
	auto_resuming = false;

	getline_x = current_text_x;
	getline_y = current_text_y;
	
	if (during_player_input)
		current_input_font = currentfont;

	if (ph_prefs.use_colors)
	{
		if (palm_display_depth < 8)
		{
			if (bgcolor==DEF_BGCOLOR || bgcolor==7 || bgcolor==15)
			{
				// Normal
				local_icolor = DEF_FCOLOR;
				local_bgcolor = DEF_BGCOLOR;
			}
			else
			{
				// Inverse
				local_icolor = DEF_BGCOLOR;
				local_bgcolor = DEF_FCOLOR;
			}		
		}
		else
		{
			local_icolor = icolor;
			local_bgcolor = bgcolor;
		}
	}
	else
	{
		local_icolor = DEF_FCOLOR;
		local_bgcolor = DEF_BGCOLOR;
	}

	frmP = FrmGetActiveForm();
	fieldindex = FrmGetObjectIndex(frmP, MainInputField);
	fieldP = FrmGetObjectPtr(frmP, fieldindex);
	
	strcpy(buffer, "");
	FldDelete(fieldP, 0, FldGetTextLength(fieldP));
	
	// Set the bounds of the field to the prompt
	RctSetRectangle(&rect, current_text_x, current_text_y,
		physical_windowright-current_text_x+1, lineheight);
	FldSetBounds(fieldP, &rect);

	// Set the field colors to match the current colors
	if (romVersion >= 0x03503000)
	{
		WinIndexToRGB(hugo_color(local_icolor), &field_text);
		WinIndexToRGB(hugo_color(local_icolor), &field_caret);
		WinIndexToRGB(hugo_color(local_bgcolor), &field_back);

		UIColorSetTableEntry(UIFieldText, &field_text);
		UIColorSetTableEntry(UIFieldCaret, &field_caret);
		UIColorSetTableEntry(UIFieldBackground, &field_back);
	}

	// Put the field up
	FrmShowObject(frmP, fieldindex);
	FrmSetFocus(frmP, fieldindex);

	getline_active = true;
	while (getline_active)
		ProcessPalmOSEvents();
		
	// Need to get these again, in case the form changed due to
	// customizing the direction buttons or something:
	frmP = FrmGetActiveForm();
	fieldindex = FrmGetObjectIndex(frmP, MainInputField);
	fieldP = FrmGetObjectPtr(frmP, fieldindex);

	if (processed_shortcut)
	{
		FldDelete(fieldP, 0, FldGetTextLength(fieldP));
	}
	else if (FldGetTextPtr(fieldP))
		strcpy(buffer, FldGetTextPtr(fieldP));
	
	FrmHideObject(frmP, fieldindex);
	
	hugo_font(PROP_FONT);
	hugo_settextcolor(local_icolor);
	hugo_setbackcolor(bgcolor);

	SetWindowColors();

	// The form behind will redraw in the form fill color, so redraw
	// it just in case
	if (default_bgcolor != DEF_BGCOLOR)
	{
		WinEraseRectangle(&rect, 0);
	}

	if (processed_shortcut)
		RestoreWindow();

	hugo_print(buffer);
	
	current_text_x = getline_x;
	current_text_y = getline_y;

	hugo_font(tempfont);
	
	hugo_print("\r\n");
	hugo_settextcolor(fcolor);
	
	RestoreWindowColors();

	/* Copy the input to the scrollback buffer */
	hugo_sendtoscrollback(prmpt);
	hugo_sendtoscrollback(buffer);
	hugo_sendtoscrollback("\n");

	if (processed_shortcut)
	{
		switch (processed_shortcut)
		{
			case StorySaveStory:
				AP("[SAVE]");
				RunSave();
				break;
			case StoryRestoreStory:
				AP("[RESTORE]");
				RunRestore();
				break;
			case StoryRestartStory:
				AP("[RESTART]");
				if (FrmCustomAlert(ConfirmAlert, "Restart the current story?", "", "")==0)
					RunRestart();
				else
					AP("Not restarted.");
				break;
			case EditUndo:
				AP("[UNDO]");
				Undo();
				break;
		}
		AP("");
		processed_shortcut = false;

		// Add to the textbuffer the prompt we're about to reprint
		TB_AddWord(GetWord(var[prompt]), physical_windowleft, current_text_y,
			physical_windowleft+hugo_textwidth(GetWord(var[prompt])),
			current_text_y+lineheight-1);
			
		goto GetNewLine;
	}
	
#ifdef HISTORY_SIZE
	hugo_addcommand();
#endif
}


/*
    COMMAND HISTORY:

    To store/retrieve player inputs for editing.
*/

#ifdef HISTORY_SIZE
void hugo_addcommand(void)
{
	int i;

	if (!StrCompare(buffer, "")) return;

        if (hcount>=HISTORY_SIZE)
	{
		MemPtrFree(history[0]);
		for (i=0; i<HISTORY_SIZE-1; i++)
			history[i] = history[i+1];
		hcount = HISTORY_SIZE-1;
	}

	if ((history[hcount] = MemPtrNew((long)((strlen(buffer)+1)*sizeof(char))))==NULL)
	{
		MemPtrFree(history[0]);
		if (hcount)
		{
			for (i=0; i<hcount; i++)
				history[i] = history[i+1];
			hcount--;
		}
		return;
	}

	for (i=0; i<=(int)StrLen(buffer); i++)
		history[hcount][i] = buffer[i];
	hcount++;
}

void hugo_restorecommand(int n)
{
	int i;

	if (n < 0 or (n>=hcount and hcount!=HISTORY_SIZE-1)) return;

	i = 0;
	do
		buffer[i] = history[n][i];
	while (history[n][i++]!='\0');
}

#endif	// HISTORY_SIZE


/* hugo_waitforkey

    Provided to be replaced by multitasking systems where cycling while
    waiting for a keystroke may not be such a hot idea.
*/

int hugo_waitforkey(void)
{
	if (!printed_something)
	{
		PopupMessage("Display", "Image?");
		PushKeypress(' ');
		override_window_restore = true;
	}
	
	FlushBuffer();
	return hugo_getkey();
}


/* hugo_iskeywaiting

    Returns true if a keypress is waiting to be retrieved.
*/

int hugo_iskeywaiting(void)
{
	ProcessPalmOSEvents();
	
	FlushBuffer();
	if (keypress_count)
		return true;
	return false;	
}


/* hugo_timewait

    Waits for 1/n seconds.  Returns false if waiting is unsupported.
*/

int hugo_timewait(int n)
{
	if (n==1)
	{
		UInt32 secs = TimGetSeconds();
		
		// Can't process events here unless we reset to evtNoWait
		while (secs==TimGetSeconds())
		{
		}
	}

	ProcessPalmOSEvents();
	
	return true;
}


/* DISPLAY CONTROL:

   Briefly, the variables required to interface with the engine's output
   functions are:

   The currently defined window, in zero-based coordinates (either
   character coordinates or pixel coordinates, as appropriate):

	physical_windowleft, physical_windowtop,
	physical_windowright, physical_windowbottom,
	physical_windowwidth, physical_windowheight

   The currently selected font is described by the ..._FONT bitmasks in:

	currentfont

   The currently selected font's width and height:

	charwidth, lineheight

   The non-proportional/fixed-width font's width and height (i.e., equal
   to charwidth and lineheight when the current font is the fixed-width
   font):

	FIXEDCHARWIDTH, FIXEDLINEHEIGHT

   Must be set by hugo_settextpos(), hugo_clearfullscreen(), and
   hugo_clearwindow():

	currentpos, currentline
*/

void hugo_init_screen(void)
{
/* Does whatever has to be done to initially set up the display. */
}

int hugo_hasgraphics(void)
{
/* Returns true if the current display is capable of graphics display;
   returns 2 if graphics are being routed to an external window other
   than the main display. */

	return false;
}

void hugo_setgametitle(char *t)
{
/* If the system doesn't provide any way to set the caption for the
   screen/window, this can be empty */
}

void hugo_cleanup_screen(void)
{
/* Does whatever has to be done to clean up the display pre-termination. */
}

void hugo_clearfullscreen(void)
{
/* Clears everything on the screen, moving the cursor to the top-left
   corner of the screen */

  	RectangleType rect;
   	RctSetRectangle(&rect, 0, 0, SCREENWIDTH, SCREENHEIGHT);

 	FlushBuffer();

  	WinEraseRectangle(&rect, 0);
  	
	TB_Clear(0, 0, SCREENWIDTH, SCREENHEIGHT);
	
	/* Must be set: */
	currentpos = 0;
	currentline = 1;
}

void hugo_clearwindow(void)
{
/* Clears the currently defined window, moving the cursor to the top-left
   corner of the window */
   
   	RectangleType rect;

   	RctSetRectangle(&rect, physical_windowleft, physical_windowtop,
   		physical_windowwidth, physical_windowheight);
    		
	FlushBuffer();

   	if (ph_prefs.minimal_windows && illegal_window)
		return;
   	
   	// Try to figure out if we're clearing the entire screen, or at least
   	// the main window
   	if (physical_windowleft==0 && physical_windowright>=palm_screen_width-1 &&
   		physical_windowbottom>=palm_screen_height-1)
   	{
   		printed_something = false;
   	}
   		
   	SetWindowColors();

	WinEraseRectangle(&rect, 0);
	
	RestoreWindowColors();
   		
#ifdef SCROLLBACK_DEFINED
	/* Send a solid line to the scrollback buffer (unless the buffer is empty)... */
	if (!inwindow && scrollback_pos!=0 &&
		// ...preventing duplicate linebreaks
		((scrollback_pos>4) && scrollback_buffer[scrollback_pos-5]!='_'))
	{
		if (scrollback_buffer[scrollback_pos-1]!='\n')
			hugo_sendtoscrollback("\n");
		MemSet(line, 38, '_');
		sprintf(line+38, "\n\n");
		hugo_sendtoscrollback(line);
	}
#endif

	TB_Clear(physical_windowleft, physical_windowtop,
		physical_windowright, physical_windowbottom);
	TB_AddWin(physical_windowleft, physical_windowtop,
		physical_windowright, physical_windowbottom);
   		
	/* Must be set: */
	currentpos = 0;
	currentline = 1;
}

void hugo_settextmode(void)
{
/* This function does whatever is necessary to set the system up for
   a standard text display */
   
	SCREENWIDTH = palm_screen_width;
	SCREENHEIGHT = palm_screen_height;
	FIXEDCHARWIDTH = FONT_WIDTH;
  	FIXEDLINEHEIGHT = FONT_HEIGHT;
	charwidth = FIXEDCHARWIDTH;
 	lineheight = FIXEDLINEHEIGHT;
	
	FntSetFont(stdFont);
	WinSetUnderlineMode(noUnderline);
	
	/* Must be set: */
	hugo_settextwindow(1, 1,
		SCREENWIDTH/FIXEDCHARWIDTH, SCREENHEIGHT/FIXEDLINEHEIGHT);
}

void hugo_settextwindow(int left, int top, int right, int bottom)
{
/* Again, coords. are passed as text coordinates with the top corner (1, 1) */

 	FlushBuffer();

	/* Must be set (as pixel coordinates): */
	physical_windowleft = (left-1)*FIXEDCHARWIDTH;
	physical_windowtop = (top-1)*FIXEDLINEHEIGHT;
	physical_windowright = right*FIXEDCHARWIDTH-1;
	physical_windowbottom = bottom*FIXEDLINEHEIGHT-1;

	/* Correct for full-width windows where the right border would
	   otherwise be clipped to a multiple of charwidth, leaving a sliver
	   of the former window at the righthand side.
	*/
	if (right>=SCREENWIDTH/FIXEDCHARWIDTH)
		physical_windowright = SCREENWIDTH-1;
	if (bottom>=SCREENHEIGHT/FIXEDLINEHEIGHT)
		physical_windowbottom = SCREENHEIGHT-1;

	physical_windowwidth = physical_windowright-physical_windowleft+1;
	physical_windowheight = physical_windowbottom-physical_windowtop+1;
	
	// Arbitrary boundaries for what are illegal or not-very-useful windows on the
	// Palm display:
	minimal_windowing = ph_prefs.minimal_windows;
	if ((inwindow) && (physical_windowheight > palm_screen_height/2 ||
		physical_windowwidth < palm_screen_width/4))
	{
		illegal_window = true;
	}
	else
	{
		illegal_window = false;
	}

	ConstrainCursor();
}

void hugo_settextpos(int x, int y)
{
/* The top-left corner of the current active window is (1, 1). */

 	FlushBuffer();

	/* Must be set: */
	currentline = y;
	currentpos = (x-1)*FIXEDCHARWIDTH;   /* Note:  zero-based */

	/* current_text_x/row are calculated assuming that the
	   character position (1, 1) is the pixel position (0, 0)
	*/
	current_text_x = physical_windowleft + currentpos;
	current_text_y = physical_windowtop + (y-1)*lineheight;
	ConstrainCursor();
}

void ConstrainCursor(void)
{
	if (current_text_x < physical_windowleft)
		current_text_x = physical_windowleft;
//	if (current_text_x > physical_windowright-charwidth)
//		current_text_x = physical_windowright-charwidth;
	if (current_text_x > physical_windowright)
		current_text_x = physical_windowright;
	if (current_text_y < physical_windowtop)
		current_text_y = physical_windowtop;
}

void DrawGraphicChar(char c, int x, int y)
{
	int row, filled = 0;
	CustomPatternType pat;
	RectangleType rect;
	static char scaling_checked = 0;
	
	if (romVersion >= 0x05003000)
	{
		// Temporary hack to make text at least readable on 1.5-sized
		// displays
		if (!scaling_checked)
		{
			UInt32 attr;
			WinScreenGetAttribute(winScreenDensity, &attr);
			if (attr==kDensityOneAndAHalf)
				scaling_checked = 1;	// 1.5 density
			else				// non-odd scaling
				scaling_checked = -1;
		}
		if (scaling_checked > 0)
			WinSetCoordinateSystem(kCoordinatesNative);
	}

	// Until we figure out how to handle Latin-1 glyphs
	if ((unsigned char)c>=FONT_COUNT) c = '?';
	
	WinSetPatternType(customPattern);

	while (filled < FONT_HEIGHT)
	{
		int start_fill = filled;
		
		row = (y+filled) % 8;
		while (row<8 && filled<FONT_HEIGHT)
		{
			pat[row] = text_font[c][filled];
			
			// Since a glyph is 4 bits wide, while a pattern is 8
			if ((x/FIXEDCHARWIDTH)&1)
				pat[row] >>= 4;

			// Shift the top half of the character for italic
			if ((current_text_font & ITALIC_FONT) && filled<FONT_HEIGHT/2+1)
				pat[row] >>= 1;

			row++;
			filled++;
		}

		// Add a bottom line for underline or bold (since bold 4-pixel-wide
		// glyphs don't look good)
		if (filled==FONT_HEIGHT)
		{
			if ((current_text_font & UNDERLINE_FONT) || (current_text_font & BOLD_FONT))
			{
				pat[row-1] = 0xff;
			}
		}

		WinSetPattern(&pat);
		RctSetRectangle(&rect, x, y+start_fill, FIXEDCHARWIDTH, filled-start_fill);
		WinFillRectangle(&rect, 0);
	}	

	WinSetPatternType(blackPattern);

	if (scaling_checked > 0)
		WinSetCoordinateSystem(kCoordinatesStandard);
}


#define MAX_FLUSH_BUFFER 128
static char flush_buffer[MAX_FLUSH_BUFFER] = "";
static int flush_x, flush_y, flush_len;

/* We use FlushBuffer() so that we can buffer multiple calls to hugo_print()
 * in order to save on more expensive drawing calls.
 */
 
void FlushBuffer(void)
{
	if (flush_len)
	{
		flush_buffer[flush_len] = '\0';
		
		SetWindowColors();

		if ((currentfont & PROP_FONT) || romVersion < 0x03503000)
		{
			WinDrawChars(flush_buffer, flush_len, flush_x, flush_y);
		}
		else
		{
			int i;
			for (i=0; i<flush_len; i++)
			{
				DrawGraphicChar(flush_buffer[i], flush_x+i*FIXEDCHARWIDTH, flush_y);
			}
		}

		RestoreWindowColors();

		flush_len = 0;
	}
}


void hugo_print(char *a)
{
/* Essentially the same as printf() without formatting, since printf()
   generally doesn't take into account color setting, font changes,
   windowing, etc.

   The newline character '\n' must be explicitly included at the end of
   a line in order to produce a linefeed.  The new cursor position is set
   to the end of this printed text.  Upon hitting the right edge of the
   screen, the printing position wraps to the start of the next line.
*/
	int i, len;
	
	if (ph_prefs.minimal_windows && illegal_window)
	{
		// If we're in an illegal window, trick the engine into
		// thinking it hasn't printed anything
		currentpos = 0;
		currentline = 1;
		full = 0;
		return;
	}
	
	len = strlen(a);

	for (i=0; i<len; i++)
	{
		if (a[i]>' ') printed_something = true;

		switch (a[i])
		{
			case '\n':
				current_text_y += lineheight;
				newline_count++;
				break;
			case '\r':
				current_text_x = physical_windowleft;
				break;
			default:
			{
		                if (!inwindow) newline_count = 0;

				if (flush_len==0)
				{
					flush_x = current_text_x;
					flush_y = current_text_y;
				}
				flush_buffer[flush_len++] = a[i];
				if (flush_len>=MAX_FLUSH_BUFFER-2) FlushBuffer();
				
				/* Increment the horizontal screen position by
				   the character width
				*/
				current_text_x += hugo_charwidth(a[i]);
			}
		}
		
		/* If we've passed the bottom of the window, align to the bottom edge */
		if (current_text_y > physical_windowbottom-lineheight+1)
		{
			int temp_lh = lineheight;

			FlushBuffer();
			SetWindowColors();

			lineheight = current_text_y - physical_windowbottom+lineheight;
			current_text_y -= lineheight;
			// hugo_scrollwindowup() will increment this:
			newline_count--;
			hugo_scrollwindowup();
			lineheight = temp_lh;

			RestoreWindowColors();
		}
	}
}


/* hugo_sendtoscrollback

	For copying printed text to the scrollback window buffer.
*/
#ifdef SCROLLBACK_DEFINED
void hugo_sendtoscrollback(char *a)
{
	int i, num;

	/* If the scrollback buffer is getting close to running out of room,
	   shift it backward, truncating at the head of the buffer
	*/
	if (scrollback_pos >= SCROLLBACK_SIZE-(int)StrLen(a))
	{
		num = (StrLen(a)>200)?StrLen(a):200;
		StrCopy(scrollback_buffer, scrollback_buffer+num);
		scrollback_pos -= num;
		for (i=scrollback_pos; i<SCROLLBACK_SIZE; i++)
			scrollback_buffer[i] = '\0';
	}
	
	num = StrLen(a);
	if (num>SCROLLBACK_SIZE-1) num = SCROLLBACK_SIZE-1;

	for (i=0; i<num; i++)
	{
		if (a[i]=='\n')
			scrollback_buffer[scrollback_pos++] = '\n';

		else if ((unsigned char)a[i]>=' ')
			scrollback_buffer[scrollback_pos++] = a[i];
	}
	scrollback_buffer[scrollback_pos] = '\0';
}
#endif

void hugo_scrollwindowup()
{
	/* Scroll the current text window up one line */
	RectangleType scrollrect, clearrect;
	
	if (auto_resuming && !got_first_input) return;
	
	if (++newline_count > 1 && ph_prefs.no_blank_lines) return;

	SetWindowColors();

	RctSetRectangle(&scrollrect, physical_windowleft, physical_windowtop,
		physical_windowwidth, physical_windowheight);
	WinScrollRectangle(&scrollrect, winUp, lineheight, &clearrect);
	WinEraseRectangle(&clearrect, 0);
	
	RestoreWindowColors();
	
	TB_Scroll();
}

void hugo_font(int f)
{
/* The <f> argument is a mask containing any or none of:
   BOLD_FONT, UNDERLINE_FONT, ITALIC_FONT, PROP_FONT.

   If charwidth and lineheight change with a font change, these must be
   reset here as well.
*/
	static int last_font_created = -1;

	if (f==last_font_created) return;
	last_font_created = f;

	FlushBuffer();

#ifndef PLAIN_FONTS_ONLY
	if (f & UNDERLINE_FONT)
		WinSetUnderlineMode(solidUnderline);
	else
		WinSetUnderlineMode(noUnderline);
	
	if (f & ITALIC_FONT)
	{
		WinSetUnderlineMode(grayUnderline);
	}
	
	if (f & BOLD_FONT)
		FntSetFont(boldFont);
	else
#endif
		FntSetFont(stdFont);

	if (f & PROP_FONT)
		lineheight = FntLineHeight();
	else
		lineheight = FIXEDLINEHEIGHT;
		
	// May cause display problems with [MORE] prompt if the proportional
	// font is shorter than the fixed font
	if (lineheight < FIXEDLINEHEIGHT)
		lineheight = FIXEDLINEHEIGHT;

	current_text_font = f;
}

void hugo_settextcolor(int c)           /* foreground (print) color */
{
	/* Set the foreground color to hugo_color(c) */
	
//	WinSetForeColor(hugo_color(c));
	FlushBuffer();
	current_text_color = c;
}

void hugo_setbackcolor(int c)           /* background color */
{
	/* Set the background color to hugo_color(c) */
	
//	WinSetBackColor(hugo_color(c));
	FlushBuffer();
	current_back_color = c;
}

int hugo_color(int c)
{
	/* Color-setting functions should always pass the color through
	   hugo_color() in order to properly set default fore/background
	   colors:
	*/
	
	RGBColorType rgb;
	
	if (romVersion < 0x03503000) return 0;

	if (c==16)      return UIColorGetTableEntryIndex(UIObjectForeground); //c = DEF_FCOLOR;
	else if (c==17) return UIColorGetTableEntryIndex(UIFormFill); //c = HUGO_WHITE;
	else if (c==18) c = HUGO_WHITE;
	else if (c==19) c = HUGO_BLUE;
	else if (c==20) return hugo_color(fcolor);	/* match foreground */

	switch (c)
	{
		case HUGO_BLACK:
			rgb.r = 0x00; rgb.g = 0x00; rgb.b = 0x00;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_BLUE:
			rgb.r = 0x00; rgb.g = 0x00; rgb.b = 0x7f;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_GREEN:
			rgb.r = 0x00; rgb.g = 0x7f; rgb.b = 0x00;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_CYAN:
			rgb.r = 0x00; rgb.g = 0x7f; rgb.b = 0x7f;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_RED:
			rgb.r = 0x7f; rgb.g = 0x00; rgb.b = 0x00;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_MAGENTA:
			rgb.r = 0x7f; rgb.g = 0x00; rgb.b = 0x7f;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_BROWN:
			rgb.r = 0x7f; rgb.g = 0x7f; rgb.b = 0x00;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_WHITE:
//			rgb.r = 0xbf; rgb.g = 0xbf; rgb.b = 0xbf;
			rgb.r = 0xff; rgb.g = 0xff; rgb.b = 0xff;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_DARK_GRAY:
			rgb.r = 0x40; rgb.g = 0x40; rgb.b = 0x40;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_LIGHT_BLUE:
			rgb.r = 0x00; rgb.g = 0x00; rgb.b = 0xff;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_LIGHT_GREEN:
			rgb.r = 0x00; rgb.g = 0xff; rgb.b = 0x00;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_LIGHT_CYAN:
			rgb.r = 0x00; rgb.g = 0xff; rgb.b = 0xff;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_LIGHT_RED:
			rgb.r = 0xff; rgb.g = 0x00; rgb.b = 0x00;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_LIGHT_MAGENTA:
			rgb.r = 0xff; rgb.g = 0x00; rgb.b = 0xff;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_YELLOW:
			rgb.r = 0xff; rgb.g = 0xff; rgb.b = 0x00;
			c = WinRGBToIndex(&rgb);
			break;
		case HUGO_BRIGHT_WHITE:
			rgb.r = 0xff; rgb.g = 0xff; rgb.b = 0xff;
			c = WinRGBToIndex(&rgb);
			break;
	}

	return c;
}


/* CHARACTER AND TEXT MEASUREMENT

	For non-proportional printing, screen dimensions will be given
	in characters, not pixels.

	For proportional printing, screen dimensions need to be in
	pixels, and each width routine must take into account the
	current font and style.

	The hugo_strlen() function is used to give the length of
	the string not including any non-printing control characters.
*/

int hugo_charwidth(char a)
{
	if (a==FORCED_SPACE)
		a = (' ');

	if ((unsigned char)a >= ' ')		/* alphanumeric characters */
	{
		if (!(currentfont & PROP_FONT))
			return FIXEDCHARWIDTH;	/* for non-proportional */
		else
			return FntCharWidth(a);
	}

	return 0;
}

int hugo_textwidth(char *a)
{
	int i, slen, len = 0;

	slen = (int)strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else
			len += hugo_charwidth(a[i]);
	}

	return len;
}

int hugo_strlen(char *a)
{
	int i, slen, len = 0;

	slen = (int)strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else len++;
	}

	return len;
}


void hugo_gettimeformatted(char *a)
{
	DateTimeType dt;
	TimSecondsToDateTime(TimGetSeconds(), &dt);
	sprintf(a, "%20d-%0d-%0d %0d:%0d:%0d",
		dt.year, dt.month, dt.day, dt.hour, dt.minute, dt.second);
}


/* Resource functions: */

int hugo_displaypicture(HUGO_FILE infile, long len)
{
        fclose(infile);		/* since infile will be open */
        return 1;
}

#if !defined (SOUND_SUPPORTED)

int hugo_playmusic(HUGO_FILE infile, long reslength, char loop_flag)
{
	fclose(infile);
	return true;	/* not an error */
}

void hugo_musicvolume(int vol)
{}

void hugo_stopmusic(void)
{}

int hugo_playsample(HUGO_FILE infile, long reslength, char loop_flag)
{
	fclose(infile);
	return true;	/* not an error */
}

void hugo_samplevolume(int vol)
{}

void hugo_stopsample(void)
{}

#endif	/* !defined (SOUND_SUPPORTED) */

#if !defined (COMPILE_V25)

int hugo_hasvideo(void)
{
	return false;
}

void hugo_stopvideo(void)
{}

int hugo_playvideo(HUGO_FILE infile, long reslength, char loop_flag)
{
	fclose(infile);
	return true;	/* not an error */
}

#endif	/* !defined (COMPILE_V25) */
