/* Copyright (c) 1998, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
Name
  vmpool.cpp - constant pool - swapping implementation
Function
  
Notes
  
Modified
  10/20/98 MJRoberts  - Creation
*/

#include <stdlib.h>
#include <memory.h>

#include "t3std.h"
#include "vmpool.h"


/* ------------------------------------------------------------------------ */
/*
 *   Swapping pool implementation.  This pool implementation keeps a set
 *   number of pages in memory at any given time.  Each time an address is
 *   translated, if the required page is not in memory, we'll discard the
 *   loaded page that was accessed (via address translation) least
 *   recently and re-use its space to load the needed page.  
 */

/*
 *   Delete the pool 
 */
CVmPoolSwap::~CVmPoolSwap()
{
    /* delete our in-memory pages, if any */
    delete_in_mem_pages();
}

/*
 *   delete our in-memory pages 
 */
void CVmPoolSwap::delete_in_mem_pages()
{
    /* if we allocated any pages, free them */
    if (mem_pages_ != 0)
    {
        size_t i;

        /* free each of the allocated pages */
        for (i = 0 ; i < max_mem_pages_ ; ++i)
        {
            /* free this page */
            if (mem_pages_[i].mem != 0)
                t3free(mem_pages_[i].mem);
        }

        /* free the list */
        t3free(mem_pages_);

        /* forget it */
        mem_pages_ = 0;
    }

    /* free the page correspondence array */
    if (pages_ptrs_ != 0)
    {
        /* free it */
        t3free(pages_ptrs_);

        /* forget it */
        pages_ptrs_ = 0;
    }
}

/*
 *   initialize the pool 
 */
void CVmPoolSwap::attach_backing_store(CVmPoolBackingStore *backing_store)
{
    size_t i;

    /* delete our in-memory list, if necessary */
    delete_in_mem_pages();
    
    /* do the normal initialization */
    CVmPool::attach_backing_store(backing_store);

    /*
     *   Ask the backing store how many pages we actually need.  If the
     *   backing store contains fewer pages than we've been told we can
     *   allocate, don't bother allocating more than we'll actually need.
     *   Note that we won't end up doing any swapping in this case, since
     *   we'll have enough pages in memory to hold all of the data in the
     *   pool.  
     */
    if (page_slots_ < max_mem_pages_)
        max_mem_pages_ = page_slots_;

    /* allocate the in-memory page list */
    mem_pages_ = (CVmPoolSwap_page *)
                 t3malloc(max_mem_pages_ * sizeof(*mem_pages_));

    /* 
     *   allocate the in-memory page list correspondence list - this list
     *   records the mem_pages_ object associated with each page in the
     *   pages_ list 
     */
    pages_ptrs_ = (CVmPoolSwap_page **)
                  t3malloc(page_slots_ * sizeof(pages_ptrs_[0]));

    /* make sure we had enough memory */
    if (mem_pages_ == 0 || pages_ptrs_ == 0)
        err_throw(VMERR_OUT_OF_MEMORY);

    /* nothing is in memory yet */
    memset(pages_ptrs_, 0, page_slots_ * sizeof(pages_ptrs_[0]));

    /* 
     *   Allocate the in-memory pages, and link them all into the free
     *   list 
     */
    for (i = 0 ; i < max_mem_pages_ ; ++i)
    {
        /* allocate memory for this page */
        mem_pages_[i].mem = (char *)t3malloc(page_size_);

        /* make sure that succeeded */
        if (mem_pages_[i].mem == 0)
            err_throw(VMERR_NO_MEM_FOR_PAGE);

        /* link it into the free list */
        mem_pages_[i].nxt = (i + 1 < max_mem_pages_ ? &mem_pages_[i+1] : 0);
        mem_pages_[i].prv = (i == 0 ? 0 : &mem_pages_[i-1]);
    }

    /* the first entry in the free list is the first page */
    free_ = &mem_pages_[0];

    /* nothing is in the lru list yet */
    lru_ = 0;
    mru_ = 0;
}

/*
 *   Translate an address, swapping in the page underlying the address if
 *   necessary. 
 */
const char *CVmPoolSwap::get_ptr(pool_ofs_t ofs)
{
    size_t slot;
    CVmPoolSwap_page *pg;
    size_t load_size;
    pool_ofs_t start_ofs;
    CVmPool_pg *info;
    
    /* get the page slot for this address */
    slot = get_page_for_ofs(ofs);
    info = get_page_info(slot);

    /* if the page is already in memory, this is trivial */
    if (info->mem != 0)
    {
        /* move the page to the most recently used position */
        make_page_mru(slot);

        /* return the page address */
        return info->mem + get_ofs_for_ofs(ofs);
    }

    /* if there's anything in the free list, allocate it to this page */
    if (free_ != 0)
    {
        /* pull it off the free list */
        pg = free_;
        free_ = free_->nxt;
    }
    else
    {
        /* 
         *   there's nothing in the free list - re-use the
         *   least-recently-used page 
         */
        pg = lru_;

        /* remove it from the list */
        remove_from_list(pg);

        /* the page's original slot is no longer present */
        get_page_info(pg->slot)->mem = 0;
        pages_ptrs_[pg->slot] = 0;
    }

    /* the page is no longer on any list */
    pg->nxt = pg->prv = 0;

    /* assign the page to its new slot */
    pg->slot = slot;

    /* point the slot to its new memory */
    info->mem = pg->mem;

    /* remember which object is in this slot */
    pages_ptrs_[slot] = pg;

    /* calculate the starting offset of the page */
    start_ofs = ((pool_ofs_t)slot) << log2_page_size_;

    /* load the page */
    load_size = backing_store_->vmpbs_get_page_size(start_ofs, page_size_);
    backing_store_->vmpbs_load_page(start_ofs, page_size_,
                                    load_size, pg->mem);

    /* move the page to the most recently used position */
    make_page_mru(slot);

    /* return the page address */
    return info->mem + get_ofs_for_ofs(ofs);
}

/*
 *   Move a page to the most-recently-used position on the cache list 
 */
void CVmPoolSwap::make_page_mru(size_t slot)
{
    CVmPoolSwap_page *pg;

    /* get the page object currently assigned to this slot */
    pg = pages_ptrs_[slot];

    /* if it's already at MRU, there's nothing to do */
    if (pg == mru_)
        return;

    /* remove it from the list */
    remove_from_list(pg);

    /* 
     *   add it as the most recently used item (i.e., at the end of the
     *   lru list)
     */
    pg->prv = mru_;
    pg->nxt = 0;
    if (mru_ != 0)
        mru_->nxt = pg;
    else
        lru_ = pg;
    mru_ = pg;
}

/*
 *   remove a page object from its list 
 */
void CVmPoolSwap::remove_from_list(CVmPoolSwap_page *pg)
{
    /* 
     *   if it's at the head, move the head; otherwise, if the next
     *   element is non-null, adjust the next element's back pointer 
     */
    if (pg == lru_)
        lru_ = pg->nxt;
    else if (pg->prv != 0)
        pg->prv->nxt = pg->nxt;

    /* 
     *   if it's at the end, move the end; otherwise, if the back pointer
     *   is non-null, adjust the previous element's next pointer 
     */
    if (pg == mru_)
        mru_ = pg->prv;
    else if (pg->nxt != 0)
        pg->nxt->prv = pg->prv;
}

