/* Copyright (c) 1999, 2000 by Kevin Forchione.  All Rights Reserved. */
/*
 *  TADS ADV.T/STD.T LIBRARY EXTENSION
 *  DOORITEM.T				
 *  version 2.0
 *
 *   	This file defines the multiloc class definition, which
 *      implements the dynamic creation of objects in multiple
 *      locations. In addition it defines doorItem and lockableDoorItem
 *      classes for implementing door objects.
 *
 *----------------------------------------------------------------------
 *  REQUIREMENTS
 *
 *      + HTML TADS 2.3.0 or later
 *      + Should be #included after ADV.T and STD.T     
 *
 *----------------------------------------------------------------------
 *  IMPORTANT LIBRARY INTERFACE AND MODIFICATION
 *
 *      The createMultiloc() function should be called from init(). 
 *
 *----------------------------------------------------------------------
 *  COPYRIGHT NOTICE
 *
 *  	You may modify and use this file in any way you want, provided that
 *		if you redistribute modified copies of this file in source form, the
 *   	copies must include the original copyright notice (including this
 *   	paragraph), and must be clearly marked as modified from the original
 *   	version.
 *
 *------------------------------------------------------------------------------
 *  REVISION HISTORY
 *
 *		31-Mar-99:	Creation.
 *		01-Apr-99:	Fixed infinite loop in multiloc.create
 *					Renamed class to doorItem and added lockableDoorItem.
 *		10-Jun-99:	Added references to objPtr in setIsopen() and setIslocked() 
 *					allowing code to reference the doorItem object to check for 
 *					open or locked conditions.
 */
 
#define __DOORITEM_MODULE_

createMultiloc: function;

/*
 *	multiloc: object
 *
 *	The general multiloc object is intended to be used for the generation of 
 *	dynamically created objects that can then be placed in the locations
 *	indicated by the foundIn property. Objects such as doors, floors, and 
 *	ceilings could be generated, allowing actors to interact with them when
 *	the parserGetMe() is not present. This class was originally intended as
 *	an alternative to floatingItem, but it offers some advantages to the coding
 *	of doorway objects also (see doorItem.)
 */
class multiloc: object
	objPtr = nil
	foundIn = []
	create( obj ) =
	{
		local c, o, l;
		c := self.foundIn;
		l := car( c );
		while( l )
		{
			o := new self;
			o.moveInto( l );
			c := cdr( c );
			l := car( c );	
		}
	}
;

/*
 *	doorItem: doorway, multiloc
 *
 *	The doorItem allows the author to code a single door in a game,
 *	requiring only an sdesc, noun, adjective, and a foundIn list. The 
 *	createMultiloc function called by commonInit uses the create method to
 *	dynamically create the two halves of a TADS doorway, link them together,
 *	assign their doordest properties, and finally modify the direction 
 *	properties of the doorway locations to point to the dynamic doorways.
 *
 *	list the locations you want the doors to be in and point to in foundIn. The
 *	list should contain exactly 2 different locations.
 */
class doorItem: doorway, multiloc
	/*
	 *	Because we are creating dynamically-created objects, yet want them to
	 *	have the same description as our 'door' object we have to use pointers
	 *	to get to the 'door' object's sdesc. This pointer is set up at the time
	 *	the doorway object is created.
	 */
	sdesc =
	{
		self.objPtr.sdesc;
	}
	adesc =
	{
		pass adesc;
	}
	thedesc =
	{
		pass thedesc;
	}
	mykey = nil	
	/*
	 *	This modification of doorway's setIsopen() method sets the isopen
	 *	attribute for the doorItem object, so that code can reference whether
	 *	it is open or closed (otherwise it will always remain in its initial
	 *	state.
	 */
    setIsopen(setting) =
    {
        /* update my status */
        self.isopen := setting;

        /* 
         *   if there's another side to this door, and its status is not
         *   already set to the new setting, update it as well (we don't
         *   update it if it's already been updated, since otherwise we'd
         *   recurse forever calling back and forth between the two sides) 
         */
        if (self.otherside != nil && self.otherside.isopen != setting)
            self.otherside.setIsopen(setting);
        if (self.objPtr != nil && self.objPtr.isopen != setting)
            self.objPtr.setIsopen(setting);
    }
	/*
	 *	This modification of doorway's setIslocked() method sets the islocked
	 *	attribute for the doorItem object, so that code can reference whether
	 *	it is locked or unlocked (otherwise it will always remain in its
	 *	initial state.
	 */
    setIslocked(setting) =
    {
        /* update my status */
        self.islocked := setting;

        /* if there's another side, update it as well */
        if (self.otherside != nil && self.otherside.islocked != setting)
            self.otherside.setIslocked(setting);
        if (self.objPtr != nil && self.objPtr.islocked != setting)
            self.objPtr.setIslocked(setting);
    }
	/*
	 *	This method will dynamically create two TADS doorways using the
	 *	information provided in the doorItem object coded by the author.
	 */
	create( obj ) =
	{		
		local o1, o2, loc1, loc2, s, v, w;
		
		/*
		 *	We check the door.foundIn for exactly 2 locations, otherwise we
		 *	dont set up the doors.
		 */
		if ( length( self.foundIn ) <> 2 ) return;

		/*
		 * 	Create the new doorItem objects.
		 */
		
		o1 := new doorItem;
		o2 := new doorItem;
		
		self.assignProp( o1, o2, obj );
	}
	assignProp( o1, o2, obj ) =
	{
		local loc1, loc2, v, w;
		/* 
		 * 	Get the locations from the foundIn list. This is used to give the
		 *	doors locations as well as doordest.
		 */
		
		loc1 := self.foundIn[ 1 ];
		loc2 := self.foundIn[ 2 ];

		/* 
		 * 	set up sdesc pointers to the door object. This allows the doors to 
		 * 	have the same short description that the author coded for the door.
		 */
		o1.objPtr := obj;
		o2.objPtr := obj;
		
		/* 
		 * 	set up the noun vocabulary, both doors at this stage are identical.
		 */
		v := getwords( self, &noun );
		w := car( v );
		while( w )
		{
			addword( o1, &noun, w );
			addword( o2, &noun, w );
			v := cdr( v );
			w := car( v );
		};
		
		/* 
		 * 	set up the adjective vocabulary 
		 */
		v := getwords( self, &adjective );
		w := car( v );
		while( w )
		{
			addword( o1, &adjective, w );
			addword( o2, &adjective, w );
			v := cdr( v );
			w := car( v );
		};
		
		/* 
		 *	Move each door to its location, set up its doordest, otherside, and
		 *	set the door.location travel direction to point to the appropriate
		 *	door.
		 */
		o1.moveInto( loc1 );			
		o1.doordest := loc2;
		o1.otherside := o2;
		
		if ( o1.objPtr.mykey )
			o1.mykey := o1.objPtr.mykey;

		self.assignDoorDir( loc1, o1 );
		
		o2.moveInto( loc2 );
		o2.doordest := loc1;
		o2.otherside := o1;
		
		if ( o2.objPtr.mykey )
			o2.mykey := o2.objPtr.mykey;		
		
		self.assignDoorDir( loc2, o2 );
	}
	doorDir = [ &north, &south, &east, &west, &ne, &nw, &se, &sw, &up, &down,
				&in, &out ]
	/*
	 *	This method looks for the object we have used as the model for our door
	 *	and assigns any direction property to the appropriate dynamically
	 *	created doorItem. This allows us to point our room to the door
	 *	object in our source code without having to know anything about the 
	 *	dynamically created doorway.
	 *
	 *	We only assign the doorway to a direction property if it returns an
	 *	object. If you want to affect anything special with code it should be
	 *	done in the leaveRoom/enterRoom.
	 */
	assignDoorDir( loc, obj ) =
	{
		local propPtr, plist;
		
		plist := doorDir;
		propPtr := car( plist );
		while ( propPtr )
		{
			if ( proptype( loc, propPtr ) = 2 and loc.(propPtr) = self ) 
				loc.(propPtr) := obj;
			plist := cdr( plist );
			propPtr := car( plist );
		}
	}
	/*
	 *	We handle the moving of the object in the create method.
	 */
	construct = {}		
;

/*
 *	lockableDoorItem: lockableDoorway, multiloc
 *
 *	The lockableDoorItem allows the author to code a single door in a game,
 *	requiring only an sdesc, noun, adjective, and a foundIn list. The 
 *	setupMultiloc function called by commonInit uses the create method to
 *	dynamically create the two halves of a TADS doorway, link them together,
 *	assign their doordest properties, and finally modify the direction 
 *	properties of the doorway locations to point to the dynamic doorways.
 *
 *	list the locations you want the doors to be in and point to in foundIn. The
 *	list should contain exactly 2 different locations.
 */
class lockableDoorItem: lockableDoorway, doorItem
	create( obj ) =
	{		
		local o1, o2, loc1, loc2, s, v, w;
		
		/*
		 *	We check the door.foundIn for exactly 2 locations, otherwise we
		 *	dont set up the doors.
		 */
		if ( length( self.foundIn ) <> 2 ) return;

		/*
		 * 	Create the new doorItem objects.
		 */
		
		o1 := new lockableDoorItem;
		o2 := new lockableDoorItem;
		
		self.assignProp( o1, o2, obj );
	}	
;

/*
 *	createMultiloc: function()
 *	
 *	This function should be called from init()(Do not call it from 
 *	preInit()). It loops over objects with the multiloc class, setting up
 *	a list that it then uses to setup the multiloc objects in your game.
 *
 *	The function must do this, because the objects it creates dynamically
 *	would be picked up by the firstobj/nextobj loop as they are created.
 */
createMultiloc: function()
{
	local o, c := [];
	
	for (o := firstobj(multiloc) ; o ; o := nextobj(o, multiloc))
	{
		c += o;
	};
	o := car( c );
	while( o )
	{
      	o.create( o );
      	c := cdr( c );
      	o := car( c );
	};
}
;
