#charset "us-ascii"
#include <adv3.h>
#include <en_us.h>

 /*
  * This file contains the language-dependent parts of the SayQuery
  * library extension.
  */


modify SuggestedSayTopic
   fullName()
   {
     if(suggestionSayGroup.tellStyle)
       return ('tell {it targetActor/him} that ' + name);
     else
       return ('say that ' + name);
   }
   
;

suggestionSayGroup: SuggestionListGroup
     
     tellStyle = nil
     
     groupPrefix()
     {
        if(tellStyle)
         "tell {it targetActor/him} ";
        else
         "say ";
     }
     showGroupItem(sublister, obj, options, pov, infoTab)
     {                 
       if(!obj.name.startsWith('that ')) 
         "that ";
       inherited(sublister, obj, options, pov, infoTab);            
     }

;

modify SuggestedQueryTopic 
  fullName = ('ask {it targetActor/him} ' + name)
;

suggestionQueryGroup: SuggestionListGroup     
     groupPrefix = "ask {it targetActor/him} "
     
;

//==========================================================

VerbRule(SayTo)
 ('say'|'answer'|'reply'|'s') singleTopic 'to' singleDobj
 | 'tell' singleDobj 'that' singleTopic
 :SayToAction
 verbRule = 'say/saying (what) (to whom)'
;

VerbRule(TellSomething)
    [badness 100] 'tell' singleDobj singleTopic
    : SayToAction
    verbPhrase = 'tell/telling (what) (what)'
;

VerbRule(SayToImplicit)
     ('say'|'answer'|'reply'|'s') ('that'|) singleTopic
     : SayToAction
     verbPhrase = 'say/saying (what) (to whom)'
     omitIobjInDobjQuery = true
     construct()
     {
         /* set up the empty direct object phrase */
         dobjMatch = new EmptyNounPhraseProd();
         dobjMatch.responseProd = toSingleNoun;
     }
;

 /*
  *  We restrict the grammar to questions starting with words like "who"
  *  of "what" or "why" to ensure we don't match any old nonsense like
  *  ASK FOO BOBBIT and to avoid masking an ASK ABOUT command, particularly
  *  one using the A abbreviation.
  *
  *  This may look convoluted, but it avoids having to define lots of
  *  different actions like AskWho, AskWhere, AskWhy and so on, while at
  *  the same time keeping the allowed grammar within acceptable bounds.
  */


#define query_word ('who'->qType|'where'->qType|'why'->qType|'what'->qType \
    |'when'->qType|'how'->qType|'which'->qType|'if'->qType|'whether'->qType \
    | 'whose'->qType | 'do'->qType |'does'->qType |'did'->qType |'have'->qType \
    |  'has'->qType |'had'->qType |'is'->qType |'are'->qType |'was'->qType \
    | 'were'->qType |'will'->qType |'shall'->qType |'would'->qType \
    |  'should'->qType )


VerbRule(Query)
  ('ask'|'a') singleDobj query_word singleTopic
  : QueryAction
  verbPhrase = 'ask/asking (whom) (what)'
  
  createTopicResolver(issuingActor, targetActor)
  {
     return new QueryTopicResolver(self, issuingActor, targetActor,
                                      topicMatch, whichMessageTopic,
                                      getTopicQualifierResolver(
                                          issuingActor, targetActor, nil));
  }

;

VerbRule(QueryImplicit)
  ('ask'|'a'|) query_word singleTopic
  : QueryAction
  verbPhrase = 'ask/asking (whom) (what)'
  omitIobjInDobjQuery = true
  construct()
  {
         /* set up the empty direct object phrase */
         dobjMatch = new EmptyNounPhraseProd();
         dobjMatch.responseProd = toSingleNoun;
  }
  
 createTopicResolver(issuingActor, targetActor)
 {
         return new QueryTopicResolver(self, issuingActor, targetActor,
                                      topicMatch, whichMessageTopic,
                                      getTopicQualifierResolver(
                                          issuingActor, targetActor, nil));
 }

  
;

 /*
  *  We return a nulWrapper, which in turn sets the resolved Topic to
  *  nulTopic if the player doesn't stipulate a topic in a Query Command.
  *  From the player's point of view, a command like ASK WHO may look an
  *  acceptable abbreviation: if this is entered it can potentially match
  *  any QueryTopic with a qType of 'who', since if nulTopic is passed
  *  to a QueryTopic's matchTopic() method, it is assumed to match.
  */

class QueryTopicResolver: ConvTopicResolver
  getDefaultObject(np)
  {  
       return [nulWrapper];
  }
;


DefineSystemAction(SayMode)
  execSystemAction()
  {
     mode = mode.toLower();
     "Say Mode set to <<mode>> ";
     switch(mode)
     {
        case 'strict':
          sayTopicManager.mode = SayTopicStrict;
          " (all SAY commands MUST start with SAY or S). ";
          break;
        case 'normal':
          sayTopicManager.mode = SayTopicNormal;
          " (the game will try to interpret unrecognized commands 
             as SAY commands). ";
          break;
        case 'lax':
          sayTopicManager.mode = SayTopicLax;
          " (the game will interpret all unrecognized commands as SAY
             commands if a conversation is in progress). ";
          break;
     }
  }
;

VerbRule(SayMode)
  ('set'|) 'say' 'mode' ('to'|) ('strict'->mode|'normal'->mode|'lax'->mode)
  : SayModeAction
  verbPhrase = 'set/setting say mode'
;

//=====================================================================

 /*
  *  Note that the first three methods modified here are used to see
  *  if the unrecognized command might work as the topic phrase
  *  of an implicit SAY command. If you modify these messages in your
  *  own code and you want this mechanism to carry on working, be sure
  *  to imitate the coding pattern below so that sayTopicManager.tryTopic(txt)
  *  is called, and a suitable error message is displayed if it returns nil.  
  */

modify playerMessages
     /* invalid command syntax */
     commandNotUnderstood(actor)
     {
        if(!sayTopicManager.tryTopic(actor, nil))
         inherited(actor);
     }
     
     wordIsUnknown(actor, txt)
     {
        if(!sayTopicManager.tryTopic(actor, txt))
           inherited(actor, txt);
     }
     askUnknownWord(actor, txt)
     {        
        if(!sayTopicManager.tryTopic(actor, txt))
           inherited(actor, txt);
     }
     
//     noMatchCannotSee(actor, txt)
//     {
//          if(!sayTopicManager.tryTopic(txt))
//           inherited(actor, txt);
//     }   
//     
//     noMatchNotAware(actor, txt)
//     {
//       if(!sayTopicManager.tryTopic(txt))
//           inherited(actor, txt);
//     }
     
     notValidInThisContext(actor, txt)
     {
        "The word <q><<txt>></q> was not understood because this
         conversational topic is no longer valid in this context. ";
         return true;
     }
     
     notValidOutsideConversation(actor, txt)
     {
        "The word <q><<txt>></q> in not valid in this story outside its
         particular conversational context(s). ";
        return true;
     }
;

modify playerActionMessages
    askVagueMsg = '<.parser>The story doesn&rsquo;t understand that command.
        Please use ASK ACTOR ABOUT TOPIC (or just A TOPIC), or make sure you
        include the appropriate interrogative word (WHO/WHAT/WHY/WHERE/HOW
        etc.) in your question, e.g. ASK ACTOR WHAT SOMETHING or A WHAT
        SOMETHING or WHAT SOMETHING.<./parser> '
;

/* 
 *   This is a function that can be called from game code to explain the 
 *   SayQuery Interface to players.
 */

explainSayQuery()
{
    "This game includes an extension to the standard TADS 3 conversation system
    that increases the range of what you can ask and say to NPCs.\b
    First, in addition to ASK ACTOR ABOUT TOPIC you can ask questions of the
    form ASK ACTOR HOW so-and-so, ASK ACTOR WHY so-and-so, ASK ACTOR WHO
    so-and-so and so on (e.g. ASK MAN WHO HE IS or ASK FRED WHERE WE ARE). In
    such questions the interrogative word (WHO/WHAT/WHY/WHERE/WHEN/WHETHER
    etc.) <i>must</i> be included in your question. When a conversation is in
    progress such questions may be abbrevated to ASK WHO HE IS or A WHO HE IS
    or even WHO HE IS (or, perhaps, WHO ARE YOU, if this has been allowed for).
    Clearly, you'll only get a meaningful (non-default) response to such a
    question if the game has provided one, which is likely to be the exception
    rather than the rule. You will normally need to use the TOPICS command to
    see what questions of this form are currently available.\b
    There is also a SAY command that allows you to say any arbitrary string to
    an actor, e.g. SAY BOO TO GOOSE, or equivalently, TELL GOOSE THAT BOO. When
    a conversation is in place this can be abbreviated to SAY BOO or S BOO or,
    under certain circumstances (see below) just BOO. What you say need not be
    restricted to one word: TELL AMANDA THAT YOU LOVE HER or SAY I AM MADLY IN
    LOVE WITH YOU are both legal SAY commands. Once again you will in general
    need to consult the topic inventory (either via the TOPICS command or when
    the game displays it spontaneously) to see what can usefully be said to an
    NPC at any particular time.\b
    On the default (normal) setting, the game will treat a command like YOU
    LOVE HER as an abbreviation for SAY YOU LOVE HER provided that this matches
    a currently available non-default response. If you prefer, you can use the
    command SAY MODE STRICT to enforce the requirement that any attempt to SAY
    something must be preceded by SAY or S (or ANSWER or REPLY), or SAY MODE
    LAX to have any otherwise unrecognized command treated as an attempt to
    SAY anything, or SAY MODE NORMAL to restore the default behaviour.<.p>";
}


//==========================================================

  /*
   *  This modification to DefaultQueryTopic allows a response incorporating
   *  the query word ('who', 'why' etc.). E.g.
   *
   *   + DefaultQueryTopic 
   *      "<q>\^<<queryWord>>...</q> you begin.\b
   *       <q>You ask too many questions,</q> he complains. "
   *   ;
   */
   
modify DefaultQueryTopic
  queryWord()
  {
    local qt = gAction.qType.toLower();
    if(qt is in (nil, 'if', 'whether'))
      return defaultQueryWord;
    
    return qt;
  }
  defaultQueryWord = 'what'
;