#charset "us-ascii"
#include <adv3.h>
#include <en_us.h>
#include "1mazes.h"

// -------------------------------------------------------------------
// TABLE OF CONTENTS
// -------------------------------------------------------------------

// NOTES
// DEBUG STUFF
// PYRAMID ROOM CLASSES
// PYRAMID ROOM CONNECTORS
// PYRAMID OBJECTS AND SCENERY
// PYRAMID 1 ROOMS
// PYRAMID 1 ROOM CONNECTORS
// PYRAMID 2 ROOMS
// PYRAMID 2 ROOM CONNECTORS
// PYRAMID "BLOCK GRIDS" (exit definitions)
// PYRAMID 999 ROOMS (commented out)
// PYRAMID 999 ROOM CONNECTORS (commented out)
// PYRAMID MAP FUNCTIONS

// -------------------------------------------------------------------
// NOTES
// -------------------------------------------------------------------

/*
Pyramid 1 solution:
23, 24, 25, 20, 19, 34, 33, 18, 13, 12, 11, 16, 21, 22, 17, 32, 29, 26, 7, 6, 1, 2, 3, 4, 5, 10, 15, 14, 31, 28, 9, 8, 27, 30, 35
E, E, N, W, U, W, D, N, W, W, S, S, E, N, U, N, N, D, W, N, E, E, E, E, S, S, W, U, N, D, W, U, S, U

Pyramid 2 solutions:
23, 22, 21, 16, 17, 32, 29, 26,  7,  2,  1,  6, 11, 12, 13, 18, 33, 34, 31, 28,  9, 14, 19, 24, 25, 20, 15, 10,  5,  4,  3,  8, 27, 30, 35
W, W, N, E, U, N, N, D, N, W, S, S, E, E, S, U, E, N, N, D, S, S, S, E, N, N, N, N, W, W, S, U, S, U
                                                                                                                                          
23, 22, 21, 16, 17, 32, 29, 26,  7,  2,  1,  6, 11, 12, 13, 18, 33, 34, 19, 24, 25, 20, 15, 14, 31, 28,  9, 10,  5,  4,  3,  8, 27, 30, 35
W, W, N, E, U, N, N, D, N, W, S, S, E, E, S, U, E, D, S, E, N, N, W, U, N, D, E, N, W, W, S, U, S, U
                                                                                                                                          
23, 22, 21, 16, 17, 32, 29, 26,  7,  2,  1,  6, 11, 12, 13, 14,  9, 28, 31, 34, 33, 18, 19, 24, 25, 20, 15, 10,  5,  4,  3,  8, 27, 30, 35
W, W, N, E, U, N, N, D, N, W, S, S, E, E, E, N, U, S, S, W, D, E, S, E, N, N, N, N, W, W, S, U, S, U
                                                                                                                                          
23, 22, 21, 16, 17, 32, 29, 26,  7,  2,  1,  6, 11, 12, 13, 14, 15, 20, 25, 24, 19, 18, 33, 34, 31, 28,  9, 10,  5,  4,  3,  8, 27, 30, 35
W, W, N, E, U, N, N, D, N, W, S, S, E, E, E, E, S, S, W, N, W, U, E, N, N, D, E, N, W, W, S, U, S, U
                                                                                                                                          

Put these room names in later:

Pyramid, First Level, Northwest Corner
Pyramid, First Level, North Edge, near Northwest Corner
Pyramid, First Level, North Edge
Pyramid, First Level, North Edge, near Northeast Corner
Pyramid, First Level, Northeast Corner

Pyramid, First Level, West Edge, near Northwest Corner
Pyramid, First Level, Middle, near Northwest Corner
Pyramid, First Level, Middle, near North Edge
Pyramid, First Level, Middle, near Northeast Corner
Pyramid, First Level, East Edge, near Northeast Corner

Pyramid, First Level, West Edge
Pyramid, First Level, Middle, near West Edge
Pyramid, First Level, Middle
Pyramid, First Level, Middle, near East Edge
Pyramid, First Level, East Edge

Pyramid, First Level, West Edge, near Southwest Corner
Pyramid, First Level, Middle, near Southwest Corner
Pyramid, First Level, Middle, near South Edge
Pyramid, First Level, Middle, near Southeast Corner
Pyramid, First Level, East Edge, near Southeast Corner

Pyramid, First Level, Southwest Corner
Pyramid, First Level, South Edge, near Southwest Corner
Pyramid, First Level, South Edge
Pyramid, First Level, South Edge, near Southeast Corner
Pyramid, First Level, Southeast Corner

Pyramid, Second Level, Northwest Corner
Pyramid, Second Level, North Edge
Pyramid, Second Level, Northeast Corner

Pyramid, Second Level, West Edge
Pyramid, Second Level, Middle
Pyramid, Second Level, East Edge

Pyramid, Second Level, Southwest Corner
Pyramid, Second Level, South Edge
Pyramid, Second Level, Southeast Corner

Pyramid, Third Level
*/

/*
       35
       
    26 27 28
    29 30 31
    32 33 34
    
  1  2  3  4  5
  6  7  8  9 10
 11 12 13 14 15
 16 17 18 19 20
 21 22 23 24 25
 
23,22,21,16,17,32,29,26,07,02,01,06,11,12,13,18,33,34,31,28,09,14,19,24,25,20,15,10,05,04,03,08,27,30
23,22,21,16,17,32,29,26,07,02,01,06,11,12,13,18,33,34,19,24,25,20,15,14,31,28,09,10,05,04,03,08,27,30
23,22,21,16,17,32,29,26,07,02,01,06,11,12,13,14,09,28,31,34,33,18,19,24,25,20,15,10,05,04,03,08,27,30
23,22,21,16,17,32,29,26,07,02,01,06,11,12,13,14,15,20,25,24,19,18,33,34,31,28,09,10,05,04,03,08,27,30
*/

// -------------------------------------------------------------------
// DEBUG STUFF
// -------------------------------------------------------------------

#ifdef __DEBUG

DefineIAction(Zxcv)
  execAction() {
    "Attempting to block the western passage. ";
    gPlayerChar.location.west.isWall = 1;
  }
;

VerbRule(Zxcv)
  'zxcv'
  : ZxcvAction
  verbPhrase = 'zxcv/zxcving'
;


pyr1Token: Fixture 'py1/pyr1/pyramid1' 'pyramid1' location=py1Rm23;
pyr2Token: Fixture 'py2/pyr2/pyramid2' 'pyramid2' location=py2Rm23;

pyramidTestWest: Room 'pyramidTestWest' 'pyramidTestWest'
  east = pyramidTestRoomConn
;

pyramidTestEast: Room 'pyramidTestEast' 'pyramidTestEast'
  west = pyramidTestRoomConn
;

pyramidTestRoomConn: RoomConnector
  room1 = pyramidTestWest
  room2 = pyramidTestEast
  // Connector is not actually passable, but seems to be so.
  isConnectorApparent(origin, actor) { return nil; }
  isConnectorPassable(origin, traveler) { return nil; }
  //canTravelerPass (traveler) { return nil; }
  //explainTravelBarrier (traveler) {    
  //  "No. ";
  //}
;

#endif // __DEBUG

// -------------------------------------------------------------------
// PYRAMID ROOM CLASSES
// -------------------------------------------------------------------

class PyramidRoom: Room
  roomName {
    switch (pyStr) {
      case '01': return 'Pyramid, First Level, Northwest Corner';
      case '02': return 'Pyramid, First Level, North Edge, near Northwest Corner';
      case '03': return 'Pyramid, First Level, North Edge';
      case '04': return 'Pyramid, First Level, North Edge, near Northeast Corner';
      case '05': return 'Pyramid, First Level, Northeast Corner';
      case '06': return 'Pyramid, First Level, West Edge, near Northwest Corner';
      case '07': return 'Pyramid, First Level, Middle, near Northwest Corner';
      case '08': return 'Pyramid, First Level, Middle, near North Edge';
      case '09': return 'Pyramid, First Level, Middle, near Northeast Corner';
      case '10': return 'Pyramid, First Level, East Edge, near Northeast Corner';
      case '11': return 'Pyramid, First Level, West Edge';
      case '12': return 'Pyramid, First Level, Middle, near West Edge';
      case '13': return 'Pyramid, First Level, Middle';
      case '14': return 'Pyramid, First Level, Middle, near East Edge';
      case '15': return 'Pyramid, First Level, East Edge';
      case '16': return 'Pyramid, First Level, West Edge, near Southwest Corner';
      case '17': return 'Pyramid, First Level, Middle, near Southwest Corner';
      case '18': return 'Pyramid, First Level, Middle, near South Edge';
      case '19': return 'Pyramid, First Level, Middle, near Southeast Corner';
      case '20': return 'Pyramid, First Level, East Edge, near Southeast Corner';
      case '21': return 'Pyramid, First Level, Southwest Corner';
      case '22': return 'Pyramid, First Level, South Edge, near Southwest Corner';
      case '23': return 'Pyramid, First Level, South Edge';
      case '24': return 'Pyramid, First Level, South Edge, near Southeast Corner';
      case '25': return 'Pyramid, First Level, Southeast Corner';
      case '26': return 'Pyramid, Second Level, Northwest Corner';
      case '27': return 'Pyramid, Second Level, North Edge';
      case '28': return 'Pyramid, Second Level, Northeast Corner';
      case '29': return 'Pyramid, Second Level, West Edge';
      case '30': return 'Pyramid, Second Level, Middle';
      case '31': return 'Pyramid, Second Level, East Edge';
      case '32': return 'Pyramid, Second Level, Southwest Corner';
      case '33': return 'Pyramid, Second Level, South Edge';
      case '34': return 'Pyramid, Second Level, Southeast Corner';
      case '35': return 'Pyramid Apex';
      default  : return 'Pyramid'; // this should never happen
    }
  }
  pyStr = '' //override
  desc {
    "The walls of this ancient structure are made of rough stone. ";
    describeDoorways;
    describeStairways;
  }
  describeDoorways {
    local openDoorways = [];
    local closedDoorways = [];
    if (pyDoorwayList == nil) {
      pyDoorwayList = [];
      if (north && !north.isWall) pyDoorwayList = pyDoorwayList.append([north,'north']);
      if (east  && !east .isWall) pyDoorwayList = pyDoorwayList.append([east ,'east' ]);
      if (south && !south.isWall) pyDoorwayList = pyDoorwayList.append([south,'south']);
      if (west  && !west .isWall) pyDoorwayList = pyDoorwayList.append([west ,'west' ]);
      //if (northeast && !northeast.isWall) pyDoorwayList = pyDoorwayList.append([northeast,'northeast']);
      //if (southeast && !southeast.isWall) pyDoorwayList = pyDoorwayList.append([southeast,'southeast']);
      //if (southwest && !southwest.isWall) pyDoorwayList = pyDoorwayList.append([southwest,'southwest']);
      //if (northwest && !northwest.isWall) pyDoorwayList = pyDoorwayList.append([northwest,'northwest']);
    }
    //local k=0; k=k/k;
    if (pyDoorwayList.length > 0) {
      for (local a = 1; a <= pyDoorwayList.length; a++) {
        if (pyDoorwayList[a][1].isConnectorPassable(bot.location, bot))
          openDoorways = openDoorways.append(pyDoorwayList[a][2]);
        else
          closedDoorways = closedDoorways.append(pyDoorwayList[a][2]);
      }
    }
    if (closedDoorways.length > 0) {
      "<<closedDoorways.length == 1 ? 'A doorway' : 'Doorways'>> to the ";
      for (local a = 1; a <= closedDoorways.length; a++) {
        say (closedDoorways[a]);
        if (closedDoorways.length > 2 && a < closedDoorways.length)
          ", ";
        if (a == closedDoorways.length - 1)
          " and ";
      }
      " <<closedDoorways.length == 1 ? 'has' : 'have'>> been sealed off";
    }
    if (openDoorways.length > 0) {
      if (closedDoorways.length > 0) ", while ";
      else "\^";
      "<<openDoorways.length == 1 ? 'an open doorway leads' 
        : 'open doorways lead'>> ";
      for (local a = 1; a <= openDoorways.length; a++) {
        say (openDoorways[a]);
        if (openDoorways.length > 2 && a < openDoorways.length)
          ", ";
        if (a == openDoorways.length - 1)
          " and ";
      }
    }
    if (pyDoorwayList.length > 0)
      ". ";
  }
  roomParts = [defaultFloor, defaultCeiling, pyrWalls]
  pyDoorwayList = nil
  describeStairways {
    local upVal, downVal;
    if (!up || up.isWall)
      upVal = 0;
    else if (!up.isConnectorPassable(bot.location, bot))
      upVal = 1;
    else
      upVal = 2;
    if (!down || down.isWall)
      downVal = 0;
    else if (!down.isConnectorPassable(bot.location, bot))
      downVal = 1;
    else
      downVal = 2;
    
    if (upVal == 0 && downVal == 0)
      { } // No staircase to describe in this case.
    else if (upVal == 2 && downVal == 2)
      "A spiral staircase here leads both down and up. ";
    else if (upVal == 2 && downVal == 0)
      "A spiral staircase here leads up. ";
    else if (upVal == 0 && downVal == 2)
      "A spiral staircase here leads down. ";
    else if (upVal == 2 && downVal == 1)
      "A spiral staircase here lies partially in ruin, although 
        you could go up it. ";
    else if (upVal == 1 && downVal == 2)
      "A spiral staircase here lies partially in ruin, although 
        you could go down it. ";
    else
      "A spiral staircase lies here in ruin. ";
  }
  beenThere = nil
  enteringRoom(traveler) {
    beenThere = true;
    nbmBanner.updateMe();
    //if (gKey()) { local k=0; k=k/k; }
  }
  holeSound = 'the hollow echoing of an ancient tomb'
  giveHint {
    if (bot.getOutermostRoom is in (py1Rm35, py2Rm35)) {
      "Sheila says, <.q>I think all we need to do here 
        is open the chest. It might be a little risky,
        but I<./s>m willing to try if you are.<./q> ";
      return;
    }
    glob.(hintProp)++;
    switch (glob.(hintProp)) {
      case 1:
        "Sheila says, ";
        if ((ofKind(Py1Rm) && pyr1Sign.hasBeenDescribed) ||
            (ofKind(Py2Rm) && pyr2Sign.hasBeenDescribed))
          "<.q>Well, you read the sign, and it said
            that an ancient treasure is
            supposedly hidden at the apex of the pyramid. 
            And it also said that in order to get it, you will need
            to explore the pyramid as thoroughly as possible.
            Both of these are important clues.<./q> ";
        else
          "<.q>In the first room of this maze, there is a sign.
            It says that there is an ancient treasure here,
            said to be hidden at the apex of the pyramid. 
            But it also says that in order to get it, you will need
            to explore the pyramid as thoroughly as possible.
            Both of those are important clues.<./q> ";
        if (glob.(potentialScoreProp) > (glob.(initialPsp) - 10))
          glob.(potentialScoreProp) = glob.(initialPsp) - 10;
        break;
      case 2:
        "Sheila says, <.q>As you may have noticed, it is easy
          to get to the top of the pyramid. But the treasure
          will not necessarily be waiting for you once you get there.
          The trick lies in that other clue from the sign:
          you need to explore the pyramid as thoroughly as possible.<./q> ";
        break;
      case 3:
        "Sheila says, <.q>Not only do you have to get to the apex,
          but you must get there after visiting
          all of the other rooms in the pyramid first. However, as you may have noticed,
          this is not altogether easy.
          When you leave a room, all the doorways of that room
          slam down, and its staircases collapse.
          As a result, you can<./s>t visit any room more than once.
          But you must still visit all the rooms before getting
          to the pyramid<./s>s apex.<./q> ";
        // Add note about RESET if you want.
        //"<p>Then Sheila adds, <.q>Incidentally, <./q> ";
        if (glob.(potentialScoreProp) > (glob.(initialPsp) - 20))
          glob.(potentialScoreProp) = glob.(initialPsp) - 20;
        break;
      case 4:
        "Sheila says, <.q>It may be somewhat challenging to visit 
            all the pyramid<./s>s rooms exactly once. To figure out how,
            it helps to look for things that would need to be part of 
            any conceivable solution. ";
        "<p><.q>For example, look at the room in the southeast corner 
            of the lower level. It borders on only two other rooms.
            Therefore, in order to prevent that room from becoming
            a dead end or cul-de-sac, you will need to visit that room 
            immediately after you first visit one of its neighbors. 
          <p><.q>There are plenty of other similar cases. 
            If you think along these lines, hopefully it will 
            help you solve the maze.<./q> ";
        if (glob.(potentialScoreProp) > (glob.(initialPsp) - 30))
          glob.(potentialScoreProp) = glob.(initialPsp) - 30;
        break;
      case 5:
        "Sheila says, <.q>I<./s>ve given you all the hints I can.
          All I can do now is give you the complete solution.
          Is that what you want?<./q> ";
        if (!sayYesOrNo()) {
          "<.q>No problem,<./q> says Sheila. <.q>Back to the game, then!<./q> ";
          glob.(hintProp)--;
        }
        else {
          giveCompleteSolution();
        }
        break;
      default:
        giveCompleteSolution();
        break;
    }
  }
  resetThisMazeSet {
    // If in Pyramid 1:
    if (bot.getOutermostRoom.ofKind(Py1Rm)) {
      glob.py1DoorwayTravelInstances = 0;
      glob.py1StairwayTravelInstances = 0;
      glob.pyramid1HintState = 0;
      pyr1Chest.moveInto(nil);
      pyr1AltarSign.moveInto(pyr1AltarSign.asscAltar);
      pyr1AltarSign.hasBeenDescribed = nil;
      for(local cur=firstObj(Py1Rm); cur != nil; cur=nextObj(cur,Py1Rm)) {
        cur.beenThere = nil;
      }
      for(local cur=firstObj(Py1Cn); cur != nil; cur=nextObj(cur,Py1Cn)) {
        cur.isBlocked = nil;
      }
    }
    // If in Pyramid 2:
    else {
      glob.py2DoorwayTravelInstances = 0;
      glob.py2StairwayTravelInstances = 0;
      glob.pyramid2HintState = 0;
      pyr2Chest.moveInto(nil);
      pyr2AltarSign.moveInto(pyr2AltarSign.asscAltar);
      pyr2AltarSign.hasBeenDescribed = nil;
      for(local cur=firstObj(Py2Rm); cur != nil; cur=nextObj(cur,Py2Rm)) {
        cur.beenThere = nil;
      }
      for(local cur=firstObj(Py2Cn); cur != nil; cur=nextObj(cur,Py2Cn)) {
        cur.isBlocked = nil;
      }
    }
    inherited();
  }
;
class Py1Rm: PyramidRoom
  bannerValue = pyramid1
  firstRoomInThisMazeSet = py1Rm23
  hintProp = &pyramid1HintState
  potentialScoreProp = &potentialScorePyramid1
  initialPsp = &initialPotentialScorePyramid1
  giveCompleteSolution {
    "Sheila says, <.q>Okay, here<./s>s the answer. ";
    if (bot.getOutermostRoom != firstRoomInThisMazeSet)
      "First, go back to the beginning of the maze by typing RESET. Then ";
    else
      "From the beginning of the maze, ";
    "go in the following directions:
      E, E, N, W, U, W, D, N, W, W, S, S, E, N, U, N, N, D, W, N, 
      E, E, E, E, S, S, W, U, N, D, W, U, S, and U.<./q> ";
    glob.(potentialScoreProp) = 0;
  }
;
class Py2Rm: PyramidRoom
  bannerValue = pyramid2
  firstRoomInThisMazeSet = py2Rm23
  hintProp = &pyramid2HintState
  potentialScoreProp = &potentialScorePyramid2
  initialPsp = &initialPotentialScorePyramid2
  giveCompleteSolution {
    "Sheila says, <.q>There are actually four different answers
      to this maze. Here<./s>s one of them. ";
    if (bot.getOutermostRoom != firstRoomInThisMazeSet)
      "First, go back to the beginning of the maze by typing RESET. Then ";
    else
      "From the beginning of the maze, ";
    "go in the following directions:
      W, W, N, E, U, N, N, D, N, W, S, S, E, E, E, E, S, S, W, N, 
      W, U, E, N, N, D, E, N, W, W, S, U, S, and U.<./q> ";
    glob.(potentialScoreProp) = 0;
  }
;

modify glob
  py1DoorwayTravelInstances = 0
  py2DoorwayTravelInstances = 0
  py1StairwayTravelInstances = 0
  py2StairwayTravelInstances = 0
  pyramid1HintState = 0
  pyramid2HintState = 0
;

// -------------------------------------------------------------------
// PYRAMID ROOM CONNECTORS
// -------------------------------------------------------------------

class PyramidConnector: RoomConnector
  isWall = nil
  isBlocked = nil
  isConnectorApparent(origin, actor) {
    if (isWall) return nil;
    if (isBlocked) return nil;
    return true;
  }
  canTravelerPass(traveler) {
    if (isWall) return nil;
    if (isBlocked) return nil;
    return true;
  } 
  explainTravelBarrier (traveler) {     
    local orig = bot.location;
    // Doorway, not stairway
    if (orig.north == self ||
        orig.south == self ||
        orig.east  == self ||
        orig.west  == self)
    {
      "That doorway has been sealed off with impenetrable stone. ";
    }
    // Stairway, not doorway
    else {
      local orig = bot.location;
      if ((orig.up   == self && orig.down && orig.down.canTravelerPass(bot)) ||
          (orig.down == self && orig.up   && orig.up  .canTravelerPass(bot)))
        "That part of the staircase has completely collapsed. ";
      else
        "That staircase has completely collapsed. ";
    }
  }
  isConnectorPassable(origin, traveler) {
    if (isWall) return nil;
    if (isBlocked) return nil;
    return true;
  }
  noteTraversal(traveler) {
    local orig = bot.location;
    //local dest = getDestination(bot.location, bot);
    
    // Doorway, not stairway
    if (orig.north == self ||
        orig.south == self ||
        orig.east  == self ||
        orig.west  == self)
    {
      if (glob.(doorwayTravelProp) <= 2)
        glob.(doorwayTravelProp)++;
      switch (glob.(doorwayTravelProp)) {
        case 1:
          "Just as Sheila goes through the doorway, she hears 
            a thunderous crash behind her. Wheeling around, 
            she eyes the doorway in astonishment. 
            It has been completely sealed off with stone! ";
          break;
        case 2:
          "Sheila wheels through the doorway as fast as she can. 
            Sure enough, once again a thunderous crash 
            follows in her wake, and she turns around 
            to see that the doorway has been sealed with stone. ";
          break;
        default:
          "Sheila wheels through the doorway, which then 
            crashes shut behind her. ";
          break;
      }
    }
    
    // Stairway, not doorway
    else {
      if (glob.(stairwayTravelProp) <= 2)
        glob.(stairwayTravelProp)++;
      if (glob.(stairwayTravelProp) == 1)
        "Sheila locks her wheels and lumbers 
          <<orig.up == self ? 'up' : 'down'>> the staircase 
          like a four-legged animal. When she gets to the 
          <<orig.up == self ? 'top' : 'bottom'>>, 
          she hears a thunderous crash behind her. 
          Turning around, she eyes the staircase in astonishment. 
          The stairs have completely collapsed! ";
      else
        "Sheila lumbers <<orig.up == self ? 'up' : 'down'>> 
          the staircase, which then collapses behind her. ";
      
      // Apex scenario
      if (   getDestination(bot.location, bot) 
          && getDestination(bot.location, bot).pyStr=='35'
          && allRoomsHaveBeenVisited(asscRoomClass)) {
        "<p>Then there is a loud grinding noise,
          which your bot soon discovers is coming from a large
          circular altar in the middle of the room.
          The whole top surface of the altar is turning
          itself upside down! When the rotation is complete,
          your bot is amazed to see a large bronze chest
          affixed to the top of the altar! ";
        asscAltar.asscChest.moveInto(asscAltar);
        if (asscAltar.asscSign.isIn(asscAltar))
          asscAltar.asscSign.moveInto(nil);
      }
    }
    
    // Common handling for both stairways and doorways
    if (orig.north && !orig.north.isWall) orig.north.isBlocked = true;
    if (orig.south && !orig.south.isWall) orig.south.isBlocked = true;
    if (orig.west  && !orig.west .isWall) orig.west .isBlocked = true;
    if (orig.east  && !orig.east .isWall) orig.east .isBlocked = true;
    if (orig.up    && !orig.up   .isWall) orig.up   .isBlocked = true;
    if (orig.down  && !orig.down .isWall) orig.down .isBlocked = true;
  }
  dobjFor(TravelVia) {
    action() {
      if (isWall)
        gActor.location.cannotTravel();
      else
        inherited;
    }
  }
;

allRoomsHaveBeenVisited(cl) {
  for (local cur = firstObj(cl); cur != nil; 
       cur = nextObj(cur, cl)) {
    if(!cur.beenThere && cur.pyStr != '35')
      return nil;
  }
  return true;
}

class Py1Cn: PyramidConnector
  doorwayTravelProp = &py1DoorwayTravelInstances
  stairwayTravelProp = &py1StairwayTravelInstances
  asscRoomClass = Py1Rm
  asscAltar = pyr1Altar
;
class Py2Cn: PyramidConnector
  doorwayTravelProp = &py2DoorwayTravelInstances
  stairwayTravelProp = &py2StairwayTravelInstances
  asscRoomClass = Py2Rm
  asscAltar = pyr2Altar
;

// -------------------------------------------------------------------
// PYRAMID OBJECTS AND SCENERY
// -------------------------------------------------------------------

pyrWalls: RoomPart
  vocabWords = 'rough stone (northern) (southern) (eastern) (western) 
    (north) (south) (east) (west) wall*walls'
  name = 'walls'
  desc = "The walls here made of rough stone. "
  isPlural = true
;

class PyramidDoorway: MultiLoc, Fixture
  // The PyramidDoorway objects are, of course, never anywhere
  // but in PyramidRoom rooms...
  initialLocationClass = PyramidRoom
  // ... and only in those rooms where the corresponding direction
  // does in fact have a doorway rather than a solid wall
  // (even if said doorway is blocked off).
  isInitiallyIn(obj) {
    if (obj.(asscDirProp) && !obj.(asscDirProp).isWall)
      return true;
    else
      return nil;
  }
  isBlocked = (bot.getOutermostRoom.(asscDirProp).isBlocked)
  desc {
    if (!isBlocked)
      "The <<asscDirName>>ern doorway is a simple rectangular passage,
        hewn from rough stone. ";
      //"The doorway is a simple rectangular passage, hewn from 
      //  rough stone, and leading <<asscDirName>>. ";
    else
      "The <<asscDirName>>ern doorway is a simple rectangular passage.
        It has been completely sealed off
        with a solid wall of impenetrable stone. ";
      //"The doorway is set in the <<asscDirName>>ern wall, but
      //  it has been completely blocked off with a solid wall 
      //  of impenetrable stone. ";
  }
  dobjFor(KnockOn) {
    verify { }
    check { }
    action {
      "<.q>Okay,<./q> says Sheila. Curling up her hand, she raps 
          her metal knuckles on the doorway.
        <p>Nobody answers. ";
    }
  }
  // Our verb handling here tries to emulate that of a ThroughPassage.
  // This is, in fact, a passage you can go through, though it can't belong to
  // the ThroughPassage class because of the unfortunate and kludgey way I linked 
  // the rooms with RoomConnectors rather than ThroughPassages.
  dobjFor(TravelVia) remapTo(TravelVia, bot.getOutermostRoom.(asscDirProp))
  //dobjFor(TravelVia) {
  //  verify { }
  //  check { }
  //  action {
  //    replaceAction(TravelVia, bot.getOutermostRoom.(asscDirProp));
  //  }
  //}
  dobjFor(GoThrough) remapTo(TravelVia, bot.getOutermostRoom.(asscDirProp))
  //dobjFor(GoThrough) remapTo(TravelVia, self)
  dobjFor(Enter) asDobjFor(GoThrough)
  dobjFor(LookThrough) {
    verify {
      if (isBlocked)
        illogical('That doorway has been sealed off with impenetrable stone. ');
    }
    action {
      mainReport(&nothingThroughPassageMsg);
    }
  }
;

enum blocked, okay;

// There is never more than one staircase in each room. Even if you
// can go both down and up, you do it by means of just one staircase
// object.
pyrStaircase: MultiLoc, Fixture
  'spiral rubble heap ruin stair stairs staircase stairway -'
  'spiral staircase'
  // The PyramidDoorway objects are, of course, never anywhere
  // but in PyramidRoom rooms...
  initialLocationClass = PyramidRoom
  // ... and only in those rooms where the corresponding direction
  // does in fact have a stairway 
  // (even if said stairway is blocked off).
  isInitiallyIn(obj) {
    if ((obj.up   && !obj.up.isWall) ||
        (obj.down && !obj.down.isWall))
      return true;
    else
      return nil;
  }
  upValue {
    if (!bot.getOutermostRoom.up || bot.getOutermostRoom.up.isWall)
      return nil;
    if (bot.getOutermostRoom.up.isBlocked)
      return blocked;
    return okay;
  }
  downValue {
    if (!bot.getOutermostRoom.down || bot.getOutermostRoom.down.isWall)
      return nil;
    if (bot.getOutermostRoom.down.isBlocked)
      return blocked;
    return okay;
  }
  //upIsWall      = (!bot.getOutermostRoom.up   || bot.getOutermostRoom.up  .isWall)
  //downIsWall    = (!bot.getOutermostRoom.down || bot.getOutermostRoom.down.isWall)
  //upIsBlocked   = ( bot.getOutermostRoom.up   && bot.getOutermostRoom.up  .isBlocked)
  //downIsBlocked = ( bot.getOutermostRoom.down && bot.getOutermostRoom.down.isBlocked)
  desc {
    // If stairway is in perfect condition...
    if ((upValue == okay || upValue == nil) && 
        (downValue == okay || downValue == nil))
    {
      if (upValue == okay)
        "A spiral staircase stands here, its stone stairs leading
          <<downValue == okay ? 'both down and up' : 'up'>>. ";
      else
        "A spiral staircase lies emerges from the floor here,
          its stone steps leading down. ";
    }
    // If stairway goes both up and down, and is only partially in ruin...
    else if ((upValue == okay && downValue == blocked) || 
             (downValue == okay && upValue == blocked)) {
      "A spiral staircase stands here. ";
      if (downValue == blocked)
        "Its stairs leading downward have completely collapsed,
          making it untraversable in that direction. 
          However, its stairs leading upward are 
          in perfectly good condition. ";
      else
        "Its stairs leading upward have collapsed into
          rubble at your bot<./s>s feet, making it untraversable
          in that direction,
          However, your bot could probably squeeze past the rubble
          to the stairs leading downward, which seem to be 
          in perfectly good condition. ";
    }
    // If stairway is entirely ruined...
    else {
      if (upValue != nil && downValue != nil)
        "A spiral staircase stands here. At some point in the past 
          it led both down and up, but now its stairs have completely 
          collapsed, making it untraversable in either
          direction. ";
      else if (upValue != nil)
        "A spiral staircase stands here. At some point in the past
          it led upwards, but its stone stairs have collapsed into
          rubble at your bot<./s>s feet, making it untraversable. ";
      else
        "A spiral staircase emerges from the floor here, but 
          its stone stairs have completely collapsed, 
          making it untraversable. ";
    }
  }
  // Our verb handling here tries to emulate that of a ThroughPassage.
  // This is, in fact, a passage you can go through, though it can't belong to
  // the ThroughPassage class because of the unfortunate and kludgey way I linked 
  // the rooms with RoomConnectors rather than ThroughPassages.
  dobjFor(TravelVia) {
    verify { }
    check {
      if (upValue != okay && downValue != okay) {
        sayCannotGo(nil);
        exit;
      }
      if (upValue == okay && downValue == okay) {
        "Sheila eyes the staircase uncertainly, then says,
          <.q>The stairs go both up and down from here.
          You<./s>ll have to tell me which way you want me to go.<./q> ";
        exit;
      }
    }
    action {
      if (upValue == okay)
        replaceAction(TravelVia, bot.getOutermostRoom.up);
      else
        replaceAction(TravelVia, bot.getOutermostRoom.down);
    }
  }
  dobjFor(Enter) remapTo(TravelVia, self)
  dobjFor(Climb) remapTo(TravelVia, self)
  dobjFor(ClimbUp) {
    verify { }
    check {
      if (upValue != okay && downValue != okay) {
        sayCannotGo(&up);
        exit;
      }
      else if (upValue == nil) {
        sayCannotGoNonexistent(&up);
        exit;
      }
      else if (upValue == blocked) {
        sayCannotGoBlocked(&up);
        exit;
      }
    }
    action {
      replaceAction(TravelVia, bot.getOutermostRoom.up);
    }
  }
  dobjFor(ClimbDown) {
    verify { }
    check {
      if (upValue != okay && downValue != okay) {
        sayCannotGo(&down);
        exit;
      }
      else if (downValue == nil) {
        sayCannotGoNonexistent(&down);
        exit;
      }
      else if (downValue == blocked) {
        sayCannotGoBlocked(&down);
        exit;
      }
    }
    action {
      replaceAction(TravelVia, bot.getOutermostRoom.down);
    }
  }
  sayCannotGo(val) {
    "Sheila eyes the ruined staircase skeptically, then says,
      <.q>This staircase has collapsed. It<./s>s simply 
      impossible for me to <<traverseVerb(val)>> it.<./q> ";
  }
  sayCannotGoNonexistent(val) {
    "Sheila eyes the staircase skeptically, then says,
      <.q>The staircase doesn<./s>t go <<val == &up ? 'up' : 'down'>>
      from here.<./q> ";
  }
  sayCannotGoBlocked(val) {
    "Sheila eyes the staircase skeptically, then says,
      <.q>The stairs <<val == &up ? 'above' : 'below'>>
      are completely collapsed. The only stairs that remain 
      traversable are the ones going <<val == &up ? 'up' : 'down'>>.<./q> ";
  }
  traverseVerb(val) {
    if (val == &down) return 'go down';
    if (val == &up  ) return 'go up'  ;
    return 'traverse';
  }
  //dobjFor(GoThrough) remapTo(TravelVia, bot.getOutermostRoom.(asscDirProp))
  ////dobjFor(GoThrough) remapTo(TravelVia, self)
  //dobjFor(LookThrough) {
  //  verify {
  //    if (isBlocked)
  //      illogical('That doorway has been sealed off with impenetrable stone. ');
  //  }
  //  action {
  //    mainReport(&nothingThroughPassageMsg);
  //  }
  //}
;

pyrDoorwayNorth: PyramidDoorway
  '(rough) (stone) north northern doorway/passage/exit*doorways*passages*exits'
  'north doorway'
  asscDirProp = &north
  asscDirName = 'north'
;
pyrDoorwaySouth: PyramidDoorway
  '(rough) (stone) south southern doorway/passage/exit*doorways*passages*exits'
  'south doorway'
  asscDirProp = &south
  asscDirName = 'south'
;
pyrDoorwayEast: PyramidDoorway
  '(rough) (stone) east eastern doorway/passage/exit*doorways*passages*exits'
  'east doorway'
  asscDirProp = &east
  asscDirName = 'east'
;
pyrDoorwayWest: PyramidDoorway
  '(rough) (stone) west western doorway/passage/exit*doorways*passages*exits'
  'west doorway'
  asscDirProp = &west
  asscDirName = 'west'
;

pyr1Sign: Readable, Fixture
  '(tourist) engraved metal pole/sign/marker' 'sign'
  @py1Rm23
  specialDesc = "There is a sign standing here. "
  desc =
    "A slender metal pole, embedded in the floor, rises and widens
        to form a handsome sign. In engraved letters it reads:
      <p><.q>Pyramids are one of Earth<./s>s most distinctive 
        forms of architecture, unlike anything found on Oo. 
        On ancient Earth, before modern construction methods 
        were invented, pyramids were the only type of structure 
        that could be built to a great height. 
        As a result, pyramids were built by the ancient inhabitants 
        of Egypt, Paris, and Las Vegas.
      <p><.q>And now you, too, can explore an Earth pyramid! 
        An ancient treasure is said to be hidden 
        at the apex of the pyramid. To get it, you will need 
        to explore the pyramid as thoroughly as possible. 
        Beware the dangers, and don't get trapped. 
        Will you be able to uncover the treasure?<./q> "
;

pyr2Sign: Readable, Fixture
  '(tourist) engraved metal pole/sign/marker' 'sign'
  @py2Rm23
  specialDesc = "There is a sign standing here. "
  desc =
    "A slender metal pole, embedded in the floor, rises and widens
        to form a handsome sign. In engraved letters it reads:
      <p><.q>One of the best things about Infocom<./s>s classic game 
        <i>Infidel</i> is that it makes no pretense of being 
        anything other than a treasure hunt. 
        Even better, it is set in an underground pyramid,
        and as a result, it has a certain unique atmosphere,
        while, thankfully, never pretending to be anything other
        than an underground crawl.
      <p><.q>And now you, too, can explore an Earth pyramid! 
        An ancient treasure is said to be hidden 
        at the apex of the pyramid. To get it, you will need 
        to explore the pyramid as thoroughly as possible. 
        Beware the dangers, and don't get trapped. 
        Will you be able to uncover the treasure?<./q> "
;

// Altar belongs to class Bed so that it's possible to put things
// on it, and so that, in addition, it's logical (but impermissible)
// to lie/sit down on it.
class Altar: Fixture
  'rough stone altar top'
  'altar'
  desc {
    "The altar is directly in the middle of the room. It is shaped
      like a circle, and is made of rough stone. ";
    if (!asscChest.isIn(self))
      "<p>Affixed to the top of the altar is a sign. ";
    else
      "<p>Affixed to the top of the altar is a bronze chest. ";
  }
  specialDesc =
    "Before your bot lies a large stone altar. 
      Affixed to the altar is 
      <<asscChest.isIn(self) ? 'a bronze chest' : 'a sign'>>. "
  //dobjFor(StandOn) asDobjFor(Board)
  //dobjFor(SitOn) asDobjFor(Board)
  //dobjFor(LieOn) asDobjFor(Board)
  //dobjFor(Climb) asDobjFor(Board)
  //dobjFor(ClimbUp) asDobjFor(Board)
  dobjFor(StandOn) remapTo(Board, self)
  dobjFor(SitOn) remapTo(Board, self)
  dobjFor(LieOn) remapTo(Board, self)
  dobjFor(Climb) remapTo(Board, self)
  dobjFor(ClimbUp) remapTo(Board, self)
  dobjFor(Board) {
    verify {
      nonObvious;
    }
    check { }
    action {
      if (!glob.altarFuseIsOn) {
        glob.altarFuseIsOn = true;
        ncYesNo.set(
          new function {
            gDobj.reallyClimbAltar;
          },
          '<.q>Oh, good,<./q> says Sheila. 
            <.q>That<./s>s probably a smart move.<./q> '
        );
        "Sheila looks around self-consciously, then says,
          <.q>Are you really sure you want me to climb up 
          on top of the altar? I mean, I don<./s>t want to desecrate 
          anything.<./q> ";
      }
      else { // the fuse is on.
        gDobj.reallyClimbAltar;
      }
    }
  }
  reallyClimbAltar {
    "<.q>All right, if you say so,<./q> says Sheila, 
        glancing around the room nervously. Then, heaving a heavy sigh,
        she locks her wheels, and boldly climbs up on top of the altar.
      <p>No sooner has Sheila done this, however, than a quiet 
        scraping noise comes from high above. One second later,
        a two-ton block of stone falls down on top of the altar, not only 
        smashing Sheila, but also crushing the altar itself. ";
    endGame(ftDeath);
  }
  iobjFor(PutOn) {
    verify { }
    check { }
    action {
      "Sheila glances around nervously, then reluctantly places
          {the dobj/him} on the altar. 
        <p>No sooner has Sheila done this, however, than a quiet 
          scraping noise comes from high above. One second later,
          a two-ton block of stone falls down on top of the altar,
          crushing it and reducing it to rubble. Furthermore, 
          the stone block then splits in two pieces, one of which
          falls directly on Sheila. ";
      endGame(ftDeath);
    }
  }
;
class AltarSign: Readable, Fixture
  'bronze metal sign'
  'sign'
  location = asscAltar
  desc {
    "The bronze sign is affixed to the top of the altar. 
        In engraved letters it reads:
      <p><.q>Alas, poor traveler, you are trapped. 
        You should have followed the instructions at the entrance 
        more carefully. Now you will die here.<./q> ";
    if (!hasBeenDescribed)
      "<p><.q>Uh oh. I don<./s>t like the sound of that,<./q> 
          says Sheila. <.q>Well, if the sign is telling the truth, 
          then I guess now would be a good time for you to type RESET. 
          Or, if you feel like giving up on this maze for now, 
          you can type ESCAPE.<./q> ";
    hasBeenDescribed = true;
  }
  hasBeenDescribed = nil
;
class AltarChest: Openable, Container, Fixture
  'bronze box/chest/patterns'
  'bronze chest'
  "The bronze chest in engraved with marvelous patterns. 
    More interesting than the patterns, however, 
    is the possibility that this box might contain
    any number of priceless treasures. "
  dobjFor(Open) {
    verify { }
    check { }
    action {
      "After studying the chest a moment, Sheila says, 
          <.q>Okay, now or never!<./q> With that, 
          she puts her hands to the lid of the chest, 
          and, very slowly and carefully, begins to lift. 
          The chest opens a crack, then opens further\ .\ .\ .
        <p>At that instant, a quiet 
          scraping noise comes from high above. One second later,
          from the ceiling directly above Sheila, 
          a two-ton block of stone falls down through the air.
        <p>During the same instant, however, the floor opens up 
          beneath Sheila<./s>s feet. She collapses down beneath the floor 
          just as the stone block crashes down upon the altar.
          With that, Sheila falls down and down\ .\ .\ . ";
      returnToCaveVictoriously('Well, I didn<./s>t make off 
        with the treasure, but at least I escaped with my life.');
    }
  }
;
pyr1Altar: Altar
  location = py1Rm35
  asscChest = pyr1Chest
  asscSign = pyr1AltarSign
;
pyr2Altar: Altar
  location = py2Rm35
  asscChest = pyr2Chest
  asscSign = pyr1AltarSign
;
pyr1AltarSign: AltarSign
  asscAltar = pyr1Altar
;
pyr2AltarSign: AltarSign
  asscAltar = pyr2Altar
;
pyr1Chest: AltarChest
  asscAltar = pyr1Altar
;
pyr2Chest: AltarChest
  asscAltar = pyr2Altar
;

// -------------------------------------------------------------------
// PYRAMID 1 ROOMS
// -------------------------------------------------------------------

py1Rm01: Py1Rm pyStr='01'                south=py1CnN01               east=py1CnE01            ;
py1Rm02: Py1Rm pyStr='02'                south=py1CnN02 west=py1CnE01 east=py1CnE02            ;
py1Rm03: Py1Rm pyStr='03'                south=py1CnN03 west=py1CnE02 east=py1CnE03            ;
py1Rm04: Py1Rm pyStr='04'                south=py1CnN04 west=py1CnE03 east=py1CnE04            ;
py1Rm05: Py1Rm pyStr='05'                south=py1CnN05 west=py1CnE04                          ;
py1Rm06: Py1Rm pyStr='06' north=py1CnN01 south=py1CnN06               east=py1CnE06            ;
py1Rm07: Py1Rm pyStr='07' north=py1CnN02 south=py1CnN07 west=py1CnE06 east=py1CnE07 up=py1CnU07;
py1Rm08: Py1Rm pyStr='08' north=py1CnN03 south=py1CnN08 west=py1CnE07 east=py1CnE08 up=py1CnU08;
py1Rm09: Py1Rm pyStr='09' north=py1CnN04 south=py1CnN09 west=py1CnE08 east=py1CnE09 up=py1CnU09;
py1Rm10: Py1Rm pyStr='10' north=py1CnN05 south=py1CnN10 west=py1CnE09                          ;
py1Rm11: Py1Rm pyStr='11' north=py1CnN06 south=py1CnN11               east=py1CnE11            ;
py1Rm12: Py1Rm pyStr='12' north=py1CnN07 south=py1CnN12 west=py1CnE11 east=py1CnE12 up=py1CnU12;
py1Rm13: Py1Rm pyStr='13' north=py1CnN08 south=py1CnN13 west=py1CnE12 east=py1CnE13 up=py1CnU13;
py1Rm14: Py1Rm pyStr='14' north=py1CnN09 south=py1CnN14 west=py1CnE13 east=py1CnE14 up=py1CnU14;
py1Rm15: Py1Rm pyStr='15' north=py1CnN10 south=py1CnN15 west=py1CnE14                          ;
py1Rm16: Py1Rm pyStr='16' north=py1CnN11 south=py1CnN16               east=py1CnE16            ;
py1Rm17: Py1Rm pyStr='17' north=py1CnN12 south=py1CnN17 west=py1CnE16 east=py1CnE17 up=py1CnU17;
py1Rm18: Py1Rm pyStr='18' north=py1CnN13 south=py1CnN18 west=py1CnE17 east=py1CnE18 up=py1CnU18;
py1Rm19: Py1Rm pyStr='19' north=py1CnN14 south=py1CnN19 west=py1CnE18 east=py1CnE19 up=py1CnU19;
py1Rm20: Py1Rm pyStr='20' north=py1CnN15 south=py1CnN20 west=py1CnE19                          ;
py1Rm21: Py1Rm pyStr='21' north=py1CnN16                              east=py1CnE21            ;
py1Rm22: Py1Rm pyStr='22' north=py1CnN17                west=py1CnE21 east=py1CnE22            ;
py1Rm23: Py1Rm pyStr='23' north=py1CnN18                west=py1CnE22 east=py1CnE23            ;
py1Rm24: Py1Rm pyStr='24' north=py1CnN19                west=py1CnE23 east=py1CnE24            ;
py1Rm25: Py1Rm pyStr='25' north=py1CnN20                west=py1CnE24                          ;

py1Rm26: Py1Rm pyStr='26'                south=py1CnN26               east=py1CnE26 down=py1CnU07;
py1Rm27: Py1Rm pyStr='27'                south=py1CnN27 west=py1CnE26 east=py1CnE27 down=py1CnU08;
py1Rm28: Py1Rm pyStr='28'                south=py1CnN28 west=py1CnE27               down=py1CnU09;
py1Rm29: Py1Rm pyStr='29' north=py1CnN26 south=py1CnN29               east=py1CnE29 down=py1CnU12;
py1Rm30: Py1Rm pyStr='30' north=py1CnN27 south=py1CnN30 west=py1CnE29 east=py1CnE30 down=py1CnU13 up=py1CnU30;
py1Rm31: Py1Rm pyStr='31' north=py1CnN28 south=py1CnN31 west=py1CnE30               down=py1CnU14;
py1Rm32: Py1Rm pyStr='32' north=py1CnN29                              east=py1CnE32 down=py1CnU17;
py1Rm33: Py1Rm pyStr='33' north=py1CnN30                west=py1CnE32 east=py1CnE33 down=py1CnU18;
py1Rm34: Py1Rm pyStr='34' north=py1CnN31                west=py1CnE33               down=py1CnU19;

py1Rm35: Py1Rm pyStr='35'                                                           down=py1CnU30;

// -------------------------------------------------------------------
// PYRAMID 1 ROOM CONNECTORS
// -------------------------------------------------------------------

py1CnN01: Py1Cn room1=py1Rm01 room2=py1Rm06 isWall=(blockGrid[11][1]);
py1CnN02: Py1Cn room1=py1Rm02 room2=py1Rm07 isWall=(blockGrid[11][2]);
py1CnN03: Py1Cn room1=py1Rm03 room2=py1Rm08 isWall=(blockGrid[11][3]);
py1CnN04: Py1Cn room1=py1Rm04 room2=py1Rm09 isWall=(blockGrid[11][4]);
py1CnN05: Py1Cn room1=py1Rm05 room2=py1Rm10 isWall=(blockGrid[11][5]);
py1CnN06: Py1Cn room1=py1Rm06 room2=py1Rm11 isWall=(blockGrid[13][1]);
py1CnN07: Py1Cn room1=py1Rm07 room2=py1Rm12 isWall=(blockGrid[13][2]);
py1CnN08: Py1Cn room1=py1Rm08 room2=py1Rm13 isWall=(blockGrid[13][3]);
py1CnN09: Py1Cn room1=py1Rm09 room2=py1Rm14 isWall=(blockGrid[13][4]);
py1CnN10: Py1Cn room1=py1Rm10 room2=py1Rm15 isWall=(blockGrid[13][5]);
py1CnN11: Py1Cn room1=py1Rm11 room2=py1Rm16 isWall=(blockGrid[15][1]);
py1CnN12: Py1Cn room1=py1Rm12 room2=py1Rm17 isWall=(blockGrid[15][2]);
py1CnN13: Py1Cn room1=py1Rm13 room2=py1Rm18 isWall=(blockGrid[15][3]);
py1CnN14: Py1Cn room1=py1Rm14 room2=py1Rm19 isWall=(blockGrid[15][4]);
py1CnN15: Py1Cn room1=py1Rm15 room2=py1Rm20 isWall=(blockGrid[15][5]);
py1CnN16: Py1Cn room1=py1Rm16 room2=py1Rm21 isWall=(blockGrid[17][1]);
py1CnN17: Py1Cn room1=py1Rm17 room2=py1Rm22 isWall=(blockGrid[17][2]);
py1CnN18: Py1Cn room1=py1Rm18 room2=py1Rm23 isWall=(blockGrid[17][3]);
py1CnN19: Py1Cn room1=py1Rm19 room2=py1Rm24 isWall=(blockGrid[17][4]);
py1CnN20: Py1Cn room1=py1Rm20 room2=py1Rm25 isWall=(blockGrid[17][5]);
py1CnN26: Py1Cn room1=py1Rm26 room2=py1Rm29 isWall=(blockGrid[ 3][1]);
py1CnN27: Py1Cn room1=py1Rm27 room2=py1Rm30 isWall=(blockGrid[ 3][2]);
py1CnN28: Py1Cn room1=py1Rm28 room2=py1Rm31 isWall=(blockGrid[ 3][3]);
py1CnN29: Py1Cn room1=py1Rm29 room2=py1Rm32 isWall=(blockGrid[ 5][1]);
py1CnN30: Py1Cn room1=py1Rm30 room2=py1Rm33 isWall=(blockGrid[ 5][2]);
py1CnN31: Py1Cn room1=py1Rm31 room2=py1Rm34 isWall=(blockGrid[ 5][3]);

py1CnE01: Py1Cn room1=py1Rm01 room2=py1Rm02 isWall=(blockGrid[10][1]);
py1CnE02: Py1Cn room1=py1Rm02 room2=py1Rm03 isWall=(blockGrid[10][2]);
py1CnE03: Py1Cn room1=py1Rm03 room2=py1Rm04 isWall=(blockGrid[10][3]);
py1CnE04: Py1Cn room1=py1Rm04 room2=py1Rm05 isWall=(blockGrid[10][4]);
py1CnE06: Py1Cn room1=py1Rm06 room2=py1Rm07 isWall=(blockGrid[12][1]);
py1CnE07: Py1Cn room1=py1Rm07 room2=py1Rm08 isWall=(blockGrid[12][2]);
py1CnE08: Py1Cn room1=py1Rm08 room2=py1Rm09 isWall=(blockGrid[12][3]);
py1CnE09: Py1Cn room1=py1Rm09 room2=py1Rm10 isWall=(blockGrid[12][4]);
py1CnE11: Py1Cn room1=py1Rm11 room2=py1Rm12 isWall=(blockGrid[14][1]);
py1CnE12: Py1Cn room1=py1Rm12 room2=py1Rm13 isWall=(blockGrid[14][2]);
py1CnE13: Py1Cn room1=py1Rm13 room2=py1Rm14 isWall=(blockGrid[14][3]);
py1CnE14: Py1Cn room1=py1Rm14 room2=py1Rm15 isWall=(blockGrid[14][4]);
py1CnE16: Py1Cn room1=py1Rm16 room2=py1Rm17 isWall=(blockGrid[16][1]);
py1CnE17: Py1Cn room1=py1Rm17 room2=py1Rm18 isWall=(blockGrid[16][2]);
py1CnE18: Py1Cn room1=py1Rm18 room2=py1Rm19 isWall=(blockGrid[16][3]);
py1CnE19: Py1Cn room1=py1Rm19 room2=py1Rm20 isWall=(blockGrid[16][4]);
py1CnE21: Py1Cn room1=py1Rm21 room2=py1Rm22 isWall=(blockGrid[18][1]);
py1CnE22: Py1Cn room1=py1Rm22 room2=py1Rm23 isWall=(blockGrid[18][2]);
py1CnE23: Py1Cn room1=py1Rm23 room2=py1Rm24 isWall=(blockGrid[18][3]);
py1CnE24: Py1Cn room1=py1Rm24 room2=py1Rm25 isWall=(blockGrid[18][4]);
py1CnE26: Py1Cn room1=py1Rm26 room2=py1Rm27 isWall=(blockGrid[ 2][1]);
py1CnE27: Py1Cn room1=py1Rm27 room2=py1Rm28 isWall=(blockGrid[ 2][2]);
py1CnE29: Py1Cn room1=py1Rm29 room2=py1Rm30 isWall=(blockGrid[ 4][1]);
py1CnE30: Py1Cn room1=py1Rm30 room2=py1Rm31 isWall=(blockGrid[ 4][2]);
py1CnE32: Py1Cn room1=py1Rm32 room2=py1Rm33 isWall=(blockGrid[ 6][1]);
py1CnE33: Py1Cn room1=py1Rm33 room2=py1Rm34 isWall=(blockGrid[ 6][2]);

py1CnU07: Py1Cn room1=py1Rm07 room2=py1Rm26 isWall=(blockGrid[ 7][1]);
py1CnU08: Py1Cn room1=py1Rm08 room2=py1Rm27 isWall=(blockGrid[ 7][2]);
py1CnU09: Py1Cn room1=py1Rm09 room2=py1Rm28 isWall=(blockGrid[ 7][3]);
py1CnU12: Py1Cn room1=py1Rm12 room2=py1Rm29 isWall=(blockGrid[ 8][1]);
py1CnU13: Py1Cn room1=py1Rm13 room2=py1Rm30 isWall=(blockGrid[ 8][2]);
py1CnU14: Py1Cn room1=py1Rm14 room2=py1Rm31 isWall=(blockGrid[ 8][3]);
py1CnU17: Py1Cn room1=py1Rm17 room2=py1Rm32 isWall=(blockGrid[ 9][1]);
py1CnU18: Py1Cn room1=py1Rm18 room2=py1Rm33 isWall=(blockGrid[ 9][2]);
py1CnU19: Py1Cn room1=py1Rm19 room2=py1Rm34 isWall=(blockGrid[ 9][3]);
py1CnU30: Py1Cn room1=py1Rm30 room2=py1Rm35 isWall=(blockGrid[ 1][1]);

// -------------------------------------------------------------------
// PYRAMID 2 ROOMS
// -------------------------------------------------------------------

py2Rm01: Py2Rm pyStr='01'                south=py2CnN01               east=py2CnE01            ;
py2Rm02: Py2Rm pyStr='02'                south=py2CnN02 west=py2CnE01 east=py2CnE02            ;
py2Rm03: Py2Rm pyStr='03'                south=py2CnN03 west=py2CnE02 east=py2CnE03            ;
py2Rm04: Py2Rm pyStr='04'                south=py2CnN04 west=py2CnE03 east=py2CnE04            ;
py2Rm05: Py2Rm pyStr='05'                south=py2CnN05 west=py2CnE04                          ;
py2Rm06: Py2Rm pyStr='06' north=py2CnN01 south=py2CnN06               east=py2CnE06            ;
py2Rm07: Py2Rm pyStr='07' north=py2CnN02 south=py2CnN07 west=py2CnE06 east=py2CnE07 up=py2CnU07;
py2Rm08: Py2Rm pyStr='08' north=py2CnN03 south=py2CnN08 west=py2CnE07 east=py2CnE08 up=py2CnU08;
py2Rm09: Py2Rm pyStr='09' north=py2CnN04 south=py2CnN09 west=py2CnE08 east=py2CnE09 up=py2CnU09;
py2Rm10: Py2Rm pyStr='10' north=py2CnN05 south=py2CnN10 west=py2CnE09                          ;
py2Rm11: Py2Rm pyStr='11' north=py2CnN06 south=py2CnN11               east=py2CnE11            ;
py2Rm12: Py2Rm pyStr='12' north=py2CnN07 south=py2CnN12 west=py2CnE11 east=py2CnE12 up=py2CnU12;
py2Rm13: Py2Rm pyStr='13' north=py2CnN08 south=py2CnN13 west=py2CnE12 east=py2CnE13 up=py2CnU13;
py2Rm14: Py2Rm pyStr='14' north=py2CnN09 south=py2CnN14 west=py2CnE13 east=py2CnE14 up=py2CnU14;
py2Rm15: Py2Rm pyStr='15' north=py2CnN10 south=py2CnN15 west=py2CnE14                          ;
py2Rm16: Py2Rm pyStr='16' north=py2CnN11 south=py2CnN16               east=py2CnE16            ;
py2Rm17: Py2Rm pyStr='17' north=py2CnN12 south=py2CnN17 west=py2CnE16 east=py2CnE17 up=py2CnU17;
py2Rm18: Py2Rm pyStr='18' north=py2CnN13 south=py2CnN18 west=py2CnE17 east=py2CnE18 up=py2CnU18;
py2Rm19: Py2Rm pyStr='19' north=py2CnN14 south=py2CnN19 west=py2CnE18 east=py2CnE19 up=py2CnU19;
py2Rm20: Py2Rm pyStr='20' north=py2CnN15 south=py2CnN20 west=py2CnE19                          ;
py2Rm21: Py2Rm pyStr='21' north=py2CnN16                              east=py2CnE21            ;
py2Rm22: Py2Rm pyStr='22' north=py2CnN17                west=py2CnE21 east=py2CnE22            ;
py2Rm23: Py2Rm pyStr='23' north=py2CnN18                west=py2CnE22 east=py2CnE23            ;
py2Rm24: Py2Rm pyStr='24' north=py2CnN19                west=py2CnE23 east=py2CnE24            ;
py2Rm25: Py2Rm pyStr='25' north=py2CnN20                west=py2CnE24                          ;

py2Rm26: Py2Rm pyStr='26'                south=py2CnN26               east=py2CnE26 down=py2CnU07;
py2Rm27: Py2Rm pyStr='27'                south=py2CnN27 west=py2CnE26 east=py2CnE27 down=py2CnU08;
py2Rm28: Py2Rm pyStr='28'                south=py2CnN28 west=py2CnE27               down=py2CnU09;
py2Rm29: Py2Rm pyStr='29' north=py2CnN26 south=py2CnN29               east=py2CnE29 down=py2CnU12;
py2Rm30: Py2Rm pyStr='30' north=py2CnN27 south=py2CnN30 west=py2CnE29 east=py2CnE30 down=py2CnU13 up=py2CnU30;
py2Rm31: Py2Rm pyStr='31' north=py2CnN28 south=py2CnN31 west=py2CnE30               down=py2CnU14;
py2Rm32: Py2Rm pyStr='32' north=py2CnN29                              east=py2CnE32 down=py2CnU17;
py2Rm33: Py2Rm pyStr='33' north=py2CnN30                west=py2CnE32 east=py2CnE33 down=py2CnU18;
py2Rm34: Py2Rm pyStr='34' north=py2CnN31                west=py2CnE33               down=py2CnU19;

py2Rm35: Py2Rm pyStr='35'                                                           down=py2CnU30;

// -------------------------------------------------------------------
// PYRAMID 2 ROOM CONNECTORS
// -------------------------------------------------------------------

py2CnN01: Py2Cn room1=py2Rm01 room2=py2Rm06 isWall=(blockGrid[11][1]);
py2CnN02: Py2Cn room1=py2Rm02 room2=py2Rm07 isWall=(blockGrid[11][2]);
py2CnN03: Py2Cn room1=py2Rm03 room2=py2Rm08 isWall=(blockGrid[11][3]);
py2CnN04: Py2Cn room1=py2Rm04 room2=py2Rm09 isWall=(blockGrid[11][4]);
py2CnN05: Py2Cn room1=py2Rm05 room2=py2Rm10 isWall=(blockGrid[11][5]);
py2CnN06: Py2Cn room1=py2Rm06 room2=py2Rm11 isWall=(blockGrid[13][1]);
py2CnN07: Py2Cn room1=py2Rm07 room2=py2Rm12 isWall=(blockGrid[13][2]);
py2CnN08: Py2Cn room1=py2Rm08 room2=py2Rm13 isWall=(blockGrid[13][3]);
py2CnN09: Py2Cn room1=py2Rm09 room2=py2Rm14 isWall=(blockGrid[13][4]);
py2CnN10: Py2Cn room1=py2Rm10 room2=py2Rm15 isWall=(blockGrid[13][5]);
py2CnN11: Py2Cn room1=py2Rm11 room2=py2Rm16 isWall=(blockGrid[15][1]);
py2CnN12: Py2Cn room1=py2Rm12 room2=py2Rm17 isWall=(blockGrid[15][2]);
py2CnN13: Py2Cn room1=py2Rm13 room2=py2Rm18 isWall=(blockGrid[15][3]);
py2CnN14: Py2Cn room1=py2Rm14 room2=py2Rm19 isWall=(blockGrid[15][4]);
py2CnN15: Py2Cn room1=py2Rm15 room2=py2Rm20 isWall=(blockGrid[15][5]);
py2CnN16: Py2Cn room1=py2Rm16 room2=py2Rm21 isWall=(blockGrid[17][1]);
py2CnN17: Py2Cn room1=py2Rm17 room2=py2Rm22 isWall=(blockGrid[17][2]);
py2CnN18: Py2Cn room1=py2Rm18 room2=py2Rm23 isWall=(blockGrid[17][3]);
py2CnN19: Py2Cn room1=py2Rm19 room2=py2Rm24 isWall=(blockGrid[17][4]);
py2CnN20: Py2Cn room1=py2Rm20 room2=py2Rm25 isWall=(blockGrid[17][5]);
py2CnN26: Py2Cn room1=py2Rm26 room2=py2Rm29 isWall=(blockGrid[ 3][1]);
py2CnN27: Py2Cn room1=py2Rm27 room2=py2Rm30 isWall=(blockGrid[ 3][2]);
py2CnN28: Py2Cn room1=py2Rm28 room2=py2Rm31 isWall=(blockGrid[ 3][3]);
py2CnN29: Py2Cn room1=py2Rm29 room2=py2Rm32 isWall=(blockGrid[ 5][1]);
py2CnN30: Py2Cn room1=py2Rm30 room2=py2Rm33 isWall=(blockGrid[ 5][2]);
py2CnN31: Py2Cn room1=py2Rm31 room2=py2Rm34 isWall=(blockGrid[ 5][3]);

py2CnE01: Py2Cn room1=py2Rm01 room2=py2Rm02 isWall=(blockGrid[10][1]);
py2CnE02: Py2Cn room1=py2Rm02 room2=py2Rm03 isWall=(blockGrid[10][2]);
py2CnE03: Py2Cn room1=py2Rm03 room2=py2Rm04 isWall=(blockGrid[10][3]);
py2CnE04: Py2Cn room1=py2Rm04 room2=py2Rm05 isWall=(blockGrid[10][4]);
py2CnE06: Py2Cn room1=py2Rm06 room2=py2Rm07 isWall=(blockGrid[12][1]);
py2CnE07: Py2Cn room1=py2Rm07 room2=py2Rm08 isWall=(blockGrid[12][2]);
py2CnE08: Py2Cn room1=py2Rm08 room2=py2Rm09 isWall=(blockGrid[12][3]);
py2CnE09: Py2Cn room1=py2Rm09 room2=py2Rm10 isWall=(blockGrid[12][4]);
py2CnE11: Py2Cn room1=py2Rm11 room2=py2Rm12 isWall=(blockGrid[14][1]);
py2CnE12: Py2Cn room1=py2Rm12 room2=py2Rm13 isWall=(blockGrid[14][2]);
py2CnE13: Py2Cn room1=py2Rm13 room2=py2Rm14 isWall=(blockGrid[14][3]);
py2CnE14: Py2Cn room1=py2Rm14 room2=py2Rm15 isWall=(blockGrid[14][4]);
py2CnE16: Py2Cn room1=py2Rm16 room2=py2Rm17 isWall=(blockGrid[16][1]);
py2CnE17: Py2Cn room1=py2Rm17 room2=py2Rm18 isWall=(blockGrid[16][2]);
py2CnE18: Py2Cn room1=py2Rm18 room2=py2Rm19 isWall=(blockGrid[16][3]);
py2CnE19: Py2Cn room1=py2Rm19 room2=py2Rm20 isWall=(blockGrid[16][4]);
py2CnE21: Py2Cn room1=py2Rm21 room2=py2Rm22 isWall=(blockGrid[18][1]);
py2CnE22: Py2Cn room1=py2Rm22 room2=py2Rm23 isWall=(blockGrid[18][2]);
py2CnE23: Py2Cn room1=py2Rm23 room2=py2Rm24 isWall=(blockGrid[18][3]);
py2CnE24: Py2Cn room1=py2Rm24 room2=py2Rm25 isWall=(blockGrid[18][4]);
py2CnE26: Py2Cn room1=py2Rm26 room2=py2Rm27 isWall=(blockGrid[ 2][1]);
py2CnE27: Py2Cn room1=py2Rm27 room2=py2Rm28 isWall=(blockGrid[ 2][2]);
py2CnE29: Py2Cn room1=py2Rm29 room2=py2Rm30 isWall=(blockGrid[ 4][1]);
py2CnE30: Py2Cn room1=py2Rm30 room2=py2Rm31 isWall=(blockGrid[ 4][2]);
py2CnE32: Py2Cn room1=py2Rm32 room2=py2Rm33 isWall=(blockGrid[ 6][1]);
py2CnE33: Py2Cn room1=py2Rm33 room2=py2Rm34 isWall=(blockGrid[ 6][2]);

py2CnU07: Py2Cn room1=py2Rm07 room2=py2Rm26 isWall=(blockGrid[ 7][1]);
py2CnU08: Py2Cn room1=py2Rm08 room2=py2Rm27 isWall=(blockGrid[ 7][2]);
py2CnU09: Py2Cn room1=py2Rm09 room2=py2Rm28 isWall=(blockGrid[ 7][3]);
py2CnU12: Py2Cn room1=py2Rm12 room2=py2Rm29 isWall=(blockGrid[ 8][1]);
py2CnU13: Py2Cn room1=py2Rm13 room2=py2Rm30 isWall=(blockGrid[ 8][2]);
py2CnU14: Py2Cn room1=py2Rm14 room2=py2Rm31 isWall=(blockGrid[ 8][3]);
py2CnU17: Py2Cn room1=py2Rm17 room2=py2Rm32 isWall=(blockGrid[ 9][1]);
py2CnU18: Py2Cn room1=py2Rm18 room2=py2Rm33 isWall=(blockGrid[ 9][2]);
py2CnU19: Py2Cn room1=py2Rm19 room2=py2Rm34 isWall=(blockGrid[ 9][3]);
py2CnU30: Py2Cn room1=py2Rm30 room2=py2Rm35 isWall=(blockGrid[ 1][1]);

// -------------------------------------------------------------------
// PYRAMID "BLOCK GRIDS" (exit definitions)
// -------------------------------------------------------------------

modify Py1Cn
  blockGrid = [
            [0],         //  1; 1
              
        [  1 , 0  ],     //  2; 1,2
        [0 , 0 , 0],     //  3; 1,2,3
        [  0 , 1  ],     //  4; 1,2
        [0 , 0 , 1],     //  5; 1,2,3
        [  1 , 0  ],     //  6; 1,2
        
        [0 , 0 , 0],     //  7; 1,2,3
                        
        [0 , 0 , 0],     //  8; 1,2,3
                        
        [0 , 0 , 0],     //  9; 1,2,3
                        
    [  0 , 0 , 0 , 0  ], // 10; 1,2,3,4
    [0 , 0 , 1 , 1 , 0], // 11; 1,2,3,4,5
    [  0 , 1 , 0 , 1  ], // 12; 1,2,3,4
    [1 , 0 , 1 , 1 , 0], // 13; 1,2,3,4,5
    [  0 , 0 , 0 , 0  ], // 14; 1,2,3,4
    [0 , 0 , 0 , 1 , 1], // 15; 1,2,3,4,5
    [  0 , 1 , 1 , 0  ], // 16; 1,2,3,4
    [0 , 0 , 0 , 0 , 0], // 17; 1,2,3,4,5
    [  0 , 1 , 0 , 0  ]  // 18; 1,2,3,4
  ]
;

modify Py2Cn
  blockGrid = [
            [0],         //  1; 1
              
        [  0 , 1  ],     //  2; 1,2
        [0 , 0 , 0],     //  3; 1,2,3
        [  0 , 0  ],     //  4; 1,2
        [0 , 0 , 0],     //  5; 1,2,3
        [  1 , 0  ],     //  6; 1,2
        
        [0 , 0 , 0],     //  7; 1,2,3
                        
        [1 , 0 , 0],     //  8; 1,2,3
                        
        [0 , 0 , 0],     //  9; 1,2,3
                        
    [  0 , 0 , 0 , 0  ], // 10; 1,2,3,4
    [0 , 0 , 0 , 0 , 0], // 11; 1,2,3,4,5
    [  0 , 0 , 0 , 0  ], // 12; 1,2,3,4
    [0 , 1 , 0 , 0 , 0], // 13; 1,2,3,4,5
    [  0 , 0 , 0 , 0  ], // 14; 1,2,3,4
    [1 , 0 , 0 , 0 , 0], // 15; 1,2,3,4,5
    [  0 , 0 , 0 , 0  ], // 16; 1,2,3,4
    [0 , 0 , 0 , 0 , 0], // 17; 1,2,3,4,5
    [  0 , 0 , 0 , 0  ]  // 18; 1,2,3,4
  ]
;

/*
modify Py1Cn
  blockGrid = [
            [0],         //  1; 1
              
        [  0 , 0  ],     //  2; 1,2
        [0 , 0 , 0],     //  3; 1,2,3
        [  0 , 0  ],     //  4; 1,2
        [0 , 0 , 0],     //  5; 1,2,3
        [  0 , 0  ],     //  6; 1,2
        
        [0 , 0 , 0],     //  7; 1,2,3
                        
        [0 , 0 , 0],     //  8; 1,2,3
                        
        [0 , 0 , 0],     //  9; 1,2,3
                        
    [  0 , 0 , 0 , 0  ], // 10; 1,2,3,4
    [0 , 0 , 0 , 0 , 0], // 11; 1,2,3,4,5
    [  0 , 0 , 0 , 0  ], // 12; 1,2,3,4
    [0 , 0 , 0 , 0 , 0], // 13; 1,2,3,4,5
    [  0 , 0 , 0 , 0  ], // 14; 1,2,3,4
    [0 , 0 , 0 , 0 , 0], // 15; 1,2,3,4,5
    [  0 , 0 , 0 , 0  ], // 16; 1,2,3,4
    [0 , 0 , 0 , 0 , 0], // 17; 1,2,3,4,5
    [  0 , 0 , 0 , 0  ]  // 18; 1,2,3,4
  ]
;
*/

/*
// -------------------------------------------------------------------
// PYRAMID 999 ROOMS (commented out)
// -------------------------------------------------------------------

py999Rm01: Py999Rm pyStr='01'                south=py999CnN01               east=py999CnE01            ;
py999Rm02: Py999Rm pyStr='02'                south=py999CnN02 west=py999CnE01 east=py999CnE02            ;
py999Rm03: Py999Rm pyStr='03'                south=py999CnN03 west=py999CnE02 east=py999CnE03            ;
py999Rm04: Py999Rm pyStr='04'                south=py999CnN04 west=py999CnE03 east=py999CnE04            ;
py999Rm05: Py999Rm pyStr='05'                south=py999CnN05 west=py999CnE04                          ;
py999Rm06: Py999Rm pyStr='06' north=py999CnN01 south=py999CnN06               east=py999CnE06            ;
py999Rm07: Py999Rm pyStr='07' north=py999CnN02 south=py999CnN07 west=py999CnE06 east=py999CnE07 up=py999CnU07;
py999Rm08: Py999Rm pyStr='08' north=py999CnN03 south=py999CnN08 west=py999CnE07 east=py999CnE08 up=py999CnU08;
py999Rm09: Py999Rm pyStr='09' north=py999CnN04 south=py999CnN09 west=py999CnE08 east=py999CnE09 up=py999CnU09;
py999Rm10: Py999Rm pyStr='10' north=py999CnN05 south=py999CnN10 west=py999CnE09                          ;
py999Rm11: Py999Rm pyStr='11' north=py999CnN06 south=py999CnN11               east=py999CnE11            ;
py999Rm12: Py999Rm pyStr='12' north=py999CnN07 south=py999CnN12 west=py999CnE11 east=py999CnE12 up=py999CnU12;
py999Rm13: Py999Rm pyStr='13' north=py999CnN08 south=py999CnN13 west=py999CnE12 east=py999CnE13 up=py999CnU13;
py999Rm14: Py999Rm pyStr='14' north=py999CnN09 south=py999CnN14 west=py999CnE13 east=py999CnE14 up=py999CnU14;
py999Rm15: Py999Rm pyStr='15' north=py999CnN10 south=py999CnN15 west=py999CnE14                          ;
py999Rm16: Py999Rm pyStr='16' north=py999CnN11 south=py999CnN16               east=py999CnE16            ;
py999Rm17: Py999Rm pyStr='17' north=py999CnN12 south=py999CnN17 west=py999CnE16 east=py999CnE17 up=py999CnU17;
py999Rm18: Py999Rm pyStr='18' north=py999CnN13 south=py999CnN18 west=py999CnE17 east=py999CnE18 up=py999CnU18;
py999Rm19: Py999Rm pyStr='19' north=py999CnN14 south=py999CnN19 west=py999CnE18 east=py999CnE19 up=py999CnU19;
py999Rm20: Py999Rm pyStr='20' north=py999CnN15 south=py999CnN20 west=py999CnE19                          ;
py999Rm21: Py999Rm pyStr='21' north=py999CnN16                              east=py999CnE21            ;
py999Rm22: Py999Rm pyStr='22' north=py999CnN17                west=py999CnE21 east=py999CnE22            ;
py999Rm23: Py999Rm pyStr='23' north=py999CnN18                west=py999CnE22 east=py999CnE23            ;
py999Rm24: Py999Rm pyStr='24' north=py999CnN19                west=py999CnE23 east=py999CnE24            ;
py999Rm25: Py999Rm pyStr='25' north=py999CnN20                west=py999CnE24                          ;

py999Rm26: Py999Rm pyStr='26'                south=py999CnN26               east=py999CnE26 down=py999CnU07;
py999Rm27: Py999Rm pyStr='27'                south=py999CnN27 west=py999CnE26 east=py999CnE27 down=py999CnU08;
py999Rm28: Py999Rm pyStr='28'                south=py999CnN28 west=py999CnE27               down=py999CnU09;
py999Rm29: Py999Rm pyStr='29' north=py999CnN26 south=py999CnN29               east=py999CnE29 down=py999CnU12;
py999Rm30: Py999Rm pyStr='30' north=py999CnN27 south=py999CnN30 west=py999CnE29 east=py999CnE30 down=py999CnU13 up=py999CnU30;
py999Rm31: Py999Rm pyStr='31' north=py999CnN28 south=py999CnN31 west=py999CnE30               down=py999CnU14;
py999Rm32: Py999Rm pyStr='32' north=py999CnN29                              east=py999CnE32 down=py999CnU17;
py999Rm33: Py999Rm pyStr='33' north=py999CnN30                west=py999CnE32 east=py999CnE33 down=py999CnU18;
py999Rm34: Py999Rm pyStr='34' north=py999CnN31                west=py999CnE33               down=py999CnU19;

py999Rm35: Py999Rm pyStr='35'                                                           down=py999CnU30;

// -------------------------------------------------------------------
// PYRAMID 999 ROOM CONNECTORS (commented out)
// -------------------------------------------------------------------

py999CnN01: Py999Cn room1=py999Rm01 room2=py999Rm06 isWall=(blockGrid[11][1]);
py999CnN02: Py999Cn room1=py999Rm02 room2=py999Rm07 isWall=(blockGrid[11][2]);
py999CnN03: Py999Cn room1=py999Rm03 room2=py999Rm08 isWall=(blockGrid[11][3]);
py999CnN04: Py999Cn room1=py999Rm04 room2=py999Rm09 isWall=(blockGrid[11][4]);
py999CnN05: Py999Cn room1=py999Rm05 room2=py999Rm10 isWall=(blockGrid[11][5]);
py999CnN06: Py999Cn room1=py999Rm06 room2=py999Rm11 isWall=(blockGrid[13][1]);
py999CnN07: Py999Cn room1=py999Rm07 room2=py999Rm12 isWall=(blockGrid[13][2]);
py999CnN08: Py999Cn room1=py999Rm08 room2=py999Rm13 isWall=(blockGrid[13][3]);
py999CnN09: Py999Cn room1=py999Rm09 room2=py999Rm14 isWall=(blockGrid[13][4]);
py999CnN10: Py999Cn room1=py999Rm10 room2=py999Rm15 isWall=(blockGrid[13][5]);
py999CnN11: Py999Cn room1=py999Rm11 room2=py999Rm16 isWall=(blockGrid[15][1]);
py999CnN12: Py999Cn room1=py999Rm12 room2=py999Rm17 isWall=(blockGrid[15][2]);
py999CnN13: Py999Cn room1=py999Rm13 room2=py999Rm18 isWall=(blockGrid[15][3]);
py999CnN14: Py999Cn room1=py999Rm14 room2=py999Rm19 isWall=(blockGrid[15][4]);
py999CnN15: Py999Cn room1=py999Rm15 room2=py999Rm20 isWall=(blockGrid[15][5]);
py999CnN16: Py999Cn room1=py999Rm16 room2=py999Rm21 isWall=(blockGrid[17][1]);
py999CnN17: Py999Cn room1=py999Rm17 room2=py999Rm22 isWall=(blockGrid[17][2]);
py999CnN18: Py999Cn room1=py999Rm18 room2=py999Rm23 isWall=(blockGrid[17][3]);
py999CnN19: Py999Cn room1=py999Rm19 room2=py999Rm24 isWall=(blockGrid[17][4]);
py999CnN20: Py999Cn room1=py999Rm20 room2=py999Rm25 isWall=(blockGrid[17][5]);
py999CnN26: Py999Cn room1=py999Rm26 room2=py999Rm29 isWall=(blockGrid[ 3][1]);
py999CnN27: Py999Cn room1=py999Rm27 room2=py999Rm30 isWall=(blockGrid[ 3][2]);
py999CnN28: Py999Cn room1=py999Rm28 room2=py999Rm31 isWall=(blockGrid[ 3][3]);
py999CnN29: Py999Cn room1=py999Rm29 room2=py999Rm32 isWall=(blockGrid[ 5][1]);
py999CnN30: Py999Cn room1=py999Rm30 room2=py999Rm33 isWall=(blockGrid[ 5][2]);
py999CnN31: Py999Cn room1=py999Rm31 room2=py999Rm34 isWall=(blockGrid[ 5][3]);

py999CnE01: Py999Cn room1=py999Rm01 room2=py999Rm02 isWall=(blockGrid[10][1]);
py999CnE02: Py999Cn room1=py999Rm02 room2=py999Rm03 isWall=(blockGrid[10][2]);
py999CnE03: Py999Cn room1=py999Rm03 room2=py999Rm04 isWall=(blockGrid[10][3]);
py999CnE04: Py999Cn room1=py999Rm04 room2=py999Rm05 isWall=(blockGrid[10][4]);
py999CnE06: Py999Cn room1=py999Rm06 room2=py999Rm07 isWall=(blockGrid[12][1]);
py999CnE07: Py999Cn room1=py999Rm07 room2=py999Rm08 isWall=(blockGrid[12][2]);
py999CnE08: Py999Cn room1=py999Rm08 room2=py999Rm09 isWall=(blockGrid[12][3]);
py999CnE09: Py999Cn room1=py999Rm09 room2=py999Rm10 isWall=(blockGrid[12][4]);
py999CnE11: Py999Cn room1=py999Rm11 room2=py999Rm12 isWall=(blockGrid[14][1]);
py999CnE12: Py999Cn room1=py999Rm12 room2=py999Rm13 isWall=(blockGrid[14][2]);
py999CnE13: Py999Cn room1=py999Rm13 room2=py999Rm14 isWall=(blockGrid[14][3]);
py999CnE14: Py999Cn room1=py999Rm14 room2=py999Rm15 isWall=(blockGrid[14][4]);
py999CnE16: Py999Cn room1=py999Rm16 room2=py999Rm17 isWall=(blockGrid[16][1]);
py999CnE17: Py999Cn room1=py999Rm17 room2=py999Rm18 isWall=(blockGrid[16][2]);
py999CnE18: Py999Cn room1=py999Rm18 room2=py999Rm19 isWall=(blockGrid[16][3]);
py999CnE19: Py999Cn room1=py999Rm19 room2=py999Rm20 isWall=(blockGrid[16][4]);
py999CnE21: Py999Cn room1=py999Rm21 room2=py999Rm22 isWall=(blockGrid[18][1]);
py999CnE22: Py999Cn room1=py999Rm22 room2=py999Rm23 isWall=(blockGrid[18][2]);
py999CnE23: Py999Cn room1=py999Rm23 room2=py999Rm24 isWall=(blockGrid[18][3]);
py999CnE24: Py999Cn room1=py999Rm24 room2=py999Rm25 isWall=(blockGrid[18][4]);
py999CnE26: Py999Cn room1=py999Rm26 room2=py999Rm27 isWall=(blockGrid[ 2][1]);
py999CnE27: Py999Cn room1=py999Rm27 room2=py999Rm28 isWall=(blockGrid[ 2][2]);
py999CnE29: Py999Cn room1=py999Rm29 room2=py999Rm30 isWall=(blockGrid[ 4][1]);
py999CnE30: Py999Cn room1=py999Rm30 room2=py999Rm31 isWall=(blockGrid[ 4][2]);
py999CnE32: Py999Cn room1=py999Rm32 room2=py999Rm33 isWall=(blockGrid[ 6][1]);
py999CnE33: Py999Cn room1=py999Rm33 room2=py999Rm34 isWall=(blockGrid[ 6][2]);

py999CnU07: Py999Cn room1=py999Rm07 room2=py999Rm26 isWall=(blockGrid[ 7][1]);
py999CnU08: Py999Cn room1=py999Rm08 room2=py999Rm27 isWall=(blockGrid[ 7][2]);
py999CnU09: Py999Cn room1=py999Rm09 room2=py999Rm28 isWall=(blockGrid[ 7][3]);
py999CnU12: Py999Cn room1=py999Rm12 room2=py999Rm29 isWall=(blockGrid[ 8][1]);
py999CnU13: Py999Cn room1=py999Rm13 room2=py999Rm30 isWall=(blockGrid[ 8][2]);
py999CnU14: Py999Cn room1=py999Rm14 room2=py999Rm31 isWall=(blockGrid[ 8][3]);
py999CnU17: Py999Cn room1=py999Rm17 room2=py999Rm32 isWall=(blockGrid[ 9][1]);
py999CnU18: Py999Cn room1=py999Rm18 room2=py999Rm33 isWall=(blockGrid[ 9][2]);
py999CnU19: Py999Cn room1=py999Rm19 room2=py999Rm34 isWall=(blockGrid[ 9][3]);
py999CnU30: Py999Cn room1=py999Rm30 room2=py999Rm35 isWall=(blockGrid[ 1][1]);

*/

//imgPyr01_01a: object;
//imgPyr01_01b: object;
//imgPyr01_01c: object;
//imgPyr01_01d: object;
//imgPyr01_01e: object;
//imgPyr01_01f: object;
//imgPyr01_01g: object;

// -------------------------------------------------------------------
// PYRAMID MAP FUNCTIONS
// -------------------------------------------------------------------

modify glob
  hackPyramidMap = nil
;

class BaseImgPyrCls: object
  room123(rm) {
    if (glob.hackPyramidMap == true)
      return '1';
    else if (gPlayerChar.location == rm)
      return '2';
    else if (rm.beenThere)
      return '1';
    else
      return '0';
  }
  conn123V(cn) {
    if (cn.isWall)
      return 'w';
    else
      return 'v';
  }
  //conn123F(cn) {
  //  if (cn.isWall)
  //    return 'e';
  //  else
  //    return 'f';
  //}
  conn123O(cn) {
    if (cn.isWall)
      return 'n';
    else
      return 'o';
  }
;

class ImgPyrCls    : BaseImgPyrCls
  rtnStr() { return 'pyr' + codeStr; }
;
class ImgPyrCls0   : BaseImgPyrCls
  rtnStr() { 
    return 'pyr' + codeStr + '-' + room123(room1);
  }
;
class ImgPyrClsV0  : BaseImgPyrCls
  rtnStr() {
    return 'pyr' + codeStr + '-' + conn123V(conn1) + room123(room1);
  }
;
class ImgPyrClsO00 : BaseImgPyrCls
  rtnStr() {
    return 'pyr' + codeStr + '-' + conn123O(conn1) + room123(room1) + room123(room2);
  }
;
class ImgPyrClsVV0 : BaseImgPyrCls
  rtnStr() {
    return 'pyr' + codeStr + '-' + conn123V(conn1) + conn123V(conn2) + room123(room1);
  }
;
class ImgPyrClsO0  : BaseImgPyrCls
  rtnStr() {
    return 'pyr' + codeStr + '-' + conn123O(conn1) + room123(room1);
  }
;
class ImgPyrClsF0  : BaseImgPyrCls
  rtnStr() {
    local str;
    str = 'pyr' + codeStr + '-';
    if (conn1.isWall)
      return str + 'e';
    else
      return str + 'f' + room123(room1);
  }
;
class ImgPyrClsFF00: BaseImgPyrCls
  rtnStr() {
    if (self == imgPyr02_07c) {
      local a = 1; 
      a = a / a;
    }
    
    
    local str;
    str = 'pyr' + codeStr + '-';
    if (!conn1.isWall && !conn2.isWall) {
      if (conn3.isWall || conn4.isWall)
        return str + 'gg' + room123(room1) + room123(room2);
      else
        return str + 'ff' + room123(room1) + room123(room2);
    }
    else if (!conn2.isWall) { // conn1 is blocked
      return str + 'ef' + room123(room2);
    }
    else if (!conn1.isWall) { // conn2 is blocked
      return str + 'fe' + room123(room1);
    }
    else { //both conn1 & conn2 are blocked
      return str + 'ee';
    }
  }
;

imgPyr01_01a: ImgPyrCls     codeStr='01a';
imgPyr01_01b: ImgPyrCls     codeStr='01b';
imgPyr01_01c: ImgPyrCls0    codeStr='01c' room1=py1Rm35;
imgPyr01_01d: ImgPyrClsV0   codeStr='01d' room1=py1Rm35               conn1=py1CnU30;
imgPyr01_01e: ImgPyrCls0    codeStr='01e' room1=py1Rm35;
imgPyr01_01f: ImgPyrCls     codeStr='01f';
imgPyr01_01g: ImgPyrCls     codeStr='01g';
imgPyr01_02a: ImgPyrCls0    codeStr='02a' room1=py1Rm26;
imgPyr01_02b: ImgPyrClsV0   codeStr='02b' room1=py1Rm26               conn1=py1CnU07;
imgPyr01_02c: ImgPyrClsO00  codeStr='02c' room1=py1Rm26 room2=py1Rm27 conn1=py1CnE26;
imgPyr01_02d: ImgPyrClsV0   codeStr='02d' room1=py1Rm27               conn1=py1CnU08;
imgPyr01_02e: ImgPyrClsO00  codeStr='02e' room1=py1Rm27 room2=py1Rm28 conn1=py1CnE27;
imgPyr01_02f: ImgPyrClsV0   codeStr='02f' room1=py1Rm28               conn1=py1CnU09;
imgPyr01_02g: ImgPyrCls0    codeStr='02g' room1=py1Rm28;
imgPyr01_03a: ImgPyrClsF0   codeStr='03a' room1=py1Rm26               conn1=py1CnN26;
imgPyr01_03b: ImgPyrClsF0   codeStr='03b' room1=py1Rm26               conn1=py1CnN26;
imgPyr01_03c: ImgPyrClsFF00 codeStr='03c' room1=py1Rm26 room2=py1Rm27 conn1=py1CnN26 conn2=py1CnN27 conn3=py1CnE26 conn4=py1CnE29;
imgPyr01_03d: ImgPyrClsF0   codeStr='03d' room1=py1Rm27               conn1=py1CnN27;
imgPyr01_03e: ImgPyrClsFF00 codeStr='03e' room1=py1Rm27 room2=py1Rm28 conn1=py1CnN27 conn2=py1CnN28 conn3=py1CnE27 conn4=py1CnE30;
imgPyr01_03f: ImgPyrClsF0   codeStr='03f' room1=py1Rm28               conn1=py1CnN28;
imgPyr01_03g: ImgPyrClsF0   codeStr='03g' room1=py1Rm28               conn1=py1CnN28;
imgPyr01_04a: ImgPyrClsF0   codeStr='04a' room1=py1Rm29               conn1=py1CnN26;
imgPyr01_04b: ImgPyrClsF0   codeStr='04b' room1=py1Rm29               conn1=py1CnN26;
imgPyr01_04c: ImgPyrClsFF00 codeStr='04c' room1=py1Rm29 room2=py1Rm30 conn1=py1CnN26 conn2=py1CnN27 conn3=py1CnE26 conn4=py1CnE29;
imgPyr01_04d: ImgPyrClsF0   codeStr='04d' room1=py1Rm30               conn1=py1CnN27;
imgPyr01_04e: ImgPyrClsFF00 codeStr='04e' room1=py1Rm30 room2=py1Rm31 conn1=py1CnN27 conn2=py1CnN28 conn3=py1CnE27 conn4=py1CnE30;
imgPyr01_04f: ImgPyrClsF0   codeStr='04f' room1=py1Rm31               conn1=py1CnN28;
imgPyr01_04g: ImgPyrClsF0   codeStr='04g' room1=py1Rm31               conn1=py1CnN28;
imgPyr01_05a: ImgPyrCls0    codeStr='05a' room1=py1Rm29;
imgPyr01_05b: ImgPyrClsV0   codeStr='05b' room1=py1Rm29               conn1=py1CnU12;
imgPyr01_05c: ImgPyrClsO00  codeStr='05c' room1=py1Rm29 room2=py1Rm30 conn1=py1CnE29;
imgPyr01_05d: ImgPyrClsVV0  codeStr='05d' room1=py1Rm30               conn1=py1CnU13 conn2=py1CnU30;
imgPyr01_05e: ImgPyrClsO00  codeStr='05e' room1=py1Rm30 room2=py1Rm31 conn1=py1CnE30;
imgPyr01_05f: ImgPyrClsV0   codeStr='05f' room1=py1Rm31               conn1=py1CnU14;
imgPyr01_05g: ImgPyrCls0    codeStr='05g' room1=py1Rm31;
imgPyr01_06a: ImgPyrClsF0   codeStr='06a' room1=py1Rm29               conn1=py1CnN29;
imgPyr01_06b: ImgPyrClsF0   codeStr='06b' room1=py1Rm29               conn1=py1CnN29;
imgPyr01_06c: ImgPyrClsFF00 codeStr='06c' room1=py1Rm29 room2=py1Rm30 conn1=py1CnN29 conn2=py1CnN30 conn3=py1CnE29 conn4=py1CnE32;
imgPyr01_06d: ImgPyrClsF0   codeStr='06d' room1=py1Rm30               conn1=py1CnN30;
imgPyr01_06e: ImgPyrClsFF00 codeStr='06e' room1=py1Rm30 room2=py1Rm31 conn1=py1CnN30 conn2=py1CnN31 conn3=py1CnE30 conn4=py1CnE33;
imgPyr01_06f: ImgPyrClsF0   codeStr='06f' room1=py1Rm31               conn1=py1CnN31;
imgPyr01_06g: ImgPyrClsF0   codeStr='06g' room1=py1Rm31               conn1=py1CnN31;
imgPyr01_07a: ImgPyrClsF0   codeStr='07a' room1=py1Rm32               conn1=py1CnN29;
imgPyr01_07b: ImgPyrClsF0   codeStr='07b' room1=py1Rm32               conn1=py1CnN29;
imgPyr01_07c: ImgPyrClsFF00 codeStr='07c' room1=py1Rm32 room2=py1Rm33 conn1=py1CnN29 conn2=py1CnN30 conn3=py1CnE29 conn4=py1CnE32;
imgPyr01_07d: ImgPyrClsF0   codeStr='07d' room1=py1Rm33               conn1=py1CnN30;
imgPyr01_07e: ImgPyrClsFF00 codeStr='07e' room1=py1Rm33 room2=py1Rm34 conn1=py1CnN30 conn2=py1CnN31 conn3=py1CnE30 conn4=py1CnE33;
imgPyr01_07f: ImgPyrClsF0   codeStr='07f' room1=py1Rm34               conn1=py1CnN31;
imgPyr01_07g: ImgPyrClsF0   codeStr='07g' room1=py1Rm34               conn1=py1CnN31;
imgPyr01_08a: ImgPyrCls0    codeStr='08a' room1=py1Rm32;
imgPyr01_08b: ImgPyrClsV0   codeStr='08b' room1=py1Rm32               conn1=py1CnU17;
imgPyr01_08c: ImgPyrClsO00  codeStr='08c' room1=py1Rm32 room2=py1Rm33 conn1=py1CnE32;
imgPyr01_08d: ImgPyrClsV0   codeStr='08d' room1=py1Rm33               conn1=py1CnU18;
imgPyr01_08e: ImgPyrClsO00  codeStr='08e' room1=py1Rm33 room2=py1Rm34 conn1=py1CnE33;
imgPyr01_08f: ImgPyrClsV0   codeStr='08f' room1=py1Rm34               conn1=py1CnU19;
imgPyr01_08g: ImgPyrCls0    codeStr='08g' room1=py1Rm34;
imgPyr01_09a: ImgPyrClsO00  codeStr='09a' room1=py1Rm01 room2=py1Rm02 conn1=py1CnE01;
imgPyr01_09b: ImgPyrClsO0   codeStr='09b' room1=py1Rm02               conn1=py1CnE01;
imgPyr01_09c: ImgPyrClsO00  codeStr='09c' room1=py1Rm02 room2=py1Rm03 conn1=py1CnE02;
imgPyr01_09d: ImgPyrCls0    codeStr='09d' room1=py1Rm03;
imgPyr01_09e: ImgPyrClsO00  codeStr='09e' room1=py1Rm03 room2=py1Rm04 conn1=py1CnE03;
imgPyr01_09f: ImgPyrClsO0   codeStr='09f' room1=py1Rm04               conn1=py1CnE04;
imgPyr01_09g: ImgPyrClsO00  codeStr='09g' room1=py1Rm04 room2=py1Rm05 conn1=py1CnE04;
imgPyr01_10a: ImgPyrClsFF00 codeStr='10a' room1=py1Rm01 room2=py1Rm02 conn1=py1CnN01 conn2=py1CnN02 conn3=py1CnE01 conn4=py1CnE06;
imgPyr01_10b: ImgPyrClsF0   codeStr='10b' room1=py1Rm02               conn1=py1CnN02;
imgPyr01_10c: ImgPyrClsFF00 codeStr='10c' room1=py1Rm02 room2=py1Rm03 conn1=py1CnN02 conn2=py1CnN03 conn3=py1CnE02 conn4=py1CnE07;
imgPyr01_10d: ImgPyrClsF0   codeStr='10d' room1=py1Rm03               conn1=py1CnN03;
imgPyr01_10e: ImgPyrClsFF00 codeStr='10e' room1=py1Rm03 room2=py1Rm04 conn1=py1CnN03 conn2=py1CnN04 conn3=py1CnE03 conn4=py1CnE08;
imgPyr01_10f: ImgPyrClsF0   codeStr='10f' room1=py1Rm04               conn1=py1CnN04;
imgPyr01_10g: ImgPyrClsFF00 codeStr='10g' room1=py1Rm04 room2=py1Rm05 conn1=py1CnN04 conn2=py1CnN05 conn3=py1CnE04 conn4=py1CnE09;
imgPyr01_11a: ImgPyrClsFF00 codeStr='11a' room1=py1Rm06 room2=py1Rm07 conn1=py1CnN01 conn2=py1CnN02 conn3=py1CnE01 conn4=py1CnE06;
imgPyr01_11b: ImgPyrClsF0   codeStr='11b' room1=py1Rm07               conn1=py1CnN02;
imgPyr01_11c: ImgPyrClsFF00 codeStr='11c' room1=py1Rm07 room2=py1Rm08 conn1=py1CnN02 conn2=py1CnN03 conn3=py1CnE02 conn4=py1CnE07;
imgPyr01_11d: ImgPyrClsF0   codeStr='11d' room1=py1Rm08               conn1=py1CnN03;
imgPyr01_11e: ImgPyrClsFF00 codeStr='11e' room1=py1Rm08 room2=py1Rm09 conn1=py1CnN03 conn2=py1CnN04 conn3=py1CnE03 conn4=py1CnE08;
imgPyr01_11f: ImgPyrClsF0   codeStr='11f' room1=py1Rm09               conn1=py1CnN04;
imgPyr01_11g: ImgPyrClsFF00 codeStr='11g' room1=py1Rm09 room2=py1Rm10 conn1=py1CnN04 conn2=py1CnN05 conn3=py1CnE04 conn4=py1CnE09;
imgPyr01_12a: ImgPyrClsO00  codeStr='12a' room1=py1Rm06 room2=py1Rm07 conn1=py1CnE06;
imgPyr01_12b: ImgPyrClsV0   codeStr='12b' room1=py1Rm07               conn1=py1CnU07;
imgPyr01_12c: ImgPyrClsO00  codeStr='12c' room1=py1Rm07 room2=py1Rm08 conn1=py1CnE07;
imgPyr01_12d: ImgPyrClsV0   codeStr='12d' room1=py1Rm08               conn1=py1CnU08;
imgPyr01_12e: ImgPyrClsO00  codeStr='12e' room1=py1Rm08 room2=py1Rm09 conn1=py1CnE08;
imgPyr01_12f: ImgPyrClsV0   codeStr='12f' room1=py1Rm09               conn1=py1CnU09;
imgPyr01_12g: ImgPyrClsO00  codeStr='12g' room1=py1Rm09 room2=py1Rm10 conn1=py1CnE09;
imgPyr01_13a: ImgPyrClsFF00 codeStr='13a' room1=py1Rm06 room2=py1Rm07 conn1=py1CnN06 conn2=py1CnN07 conn3=py1CnE06 conn4=py1CnE11;
imgPyr01_13b: ImgPyrClsF0   codeStr='13b' room1=py1Rm07               conn1=py1CnN07;
imgPyr01_13c: ImgPyrClsFF00 codeStr='13c' room1=py1Rm07 room2=py1Rm08 conn1=py1CnN07 conn2=py1CnN08 conn3=py1CnE07 conn4=py1CnE12;
imgPyr01_13d: ImgPyrClsF0   codeStr='13d' room1=py1Rm08               conn1=py1CnN08;
imgPyr01_13e: ImgPyrClsFF00 codeStr='13e' room1=py1Rm08 room2=py1Rm09 conn1=py1CnN08 conn2=py1CnN09 conn3=py1CnE08 conn4=py1CnE13;
imgPyr01_13f: ImgPyrClsF0   codeStr='13f' room1=py1Rm09               conn1=py1CnN09;
imgPyr01_13g: ImgPyrClsFF00 codeStr='13g' room1=py1Rm09 room2=py1Rm10 conn1=py1CnN09 conn2=py1CnN10 conn3=py1CnE09 conn4=py1CnE14;
imgPyr01_14a: ImgPyrClsFF00 codeStr='14a' room1=py1Rm11 room2=py1Rm12 conn1=py1CnN06 conn2=py1CnN07 conn3=py1CnE06 conn4=py1CnE11;
imgPyr01_14b: ImgPyrClsF0   codeStr='14b' room1=py1Rm12               conn1=py1CnN07;
imgPyr01_14c: ImgPyrClsFF00 codeStr='14c' room1=py1Rm12 room2=py1Rm13 conn1=py1CnN07 conn2=py1CnN08 conn3=py1CnE07 conn4=py1CnE12;
imgPyr01_14d: ImgPyrClsF0   codeStr='14d' room1=py1Rm13               conn1=py1CnN08;
imgPyr01_14e: ImgPyrClsFF00 codeStr='14e' room1=py1Rm13 room2=py1Rm14 conn1=py1CnN08 conn2=py1CnN09 conn3=py1CnE08 conn4=py1CnE13;
imgPyr01_14f: ImgPyrClsF0   codeStr='14f' room1=py1Rm14               conn1=py1CnN09;
imgPyr01_14g: ImgPyrClsFF00 codeStr='14g' room1=py1Rm14 room2=py1Rm15 conn1=py1CnN09 conn2=py1CnN10 conn3=py1CnE09 conn4=py1CnE14;
imgPyr01_15a: ImgPyrClsO00  codeStr='15a' room1=py1Rm11 room2=py1Rm12 conn1=py1CnE11;
imgPyr01_15b: ImgPyrClsV0   codeStr='15b' room1=py1Rm12               conn1=py1CnU12;
imgPyr01_15c: ImgPyrClsO00  codeStr='15c' room1=py1Rm12 room2=py1Rm13 conn1=py1CnE12;
imgPyr01_15d: ImgPyrClsV0   codeStr='15d' room1=py1Rm13               conn1=py1CnU13;
imgPyr01_15e: ImgPyrClsO00  codeStr='15e' room1=py1Rm13 room2=py1Rm14 conn1=py1CnE13;
imgPyr01_15f: ImgPyrClsV0   codeStr='15f' room1=py1Rm14               conn1=py1CnU14;
imgPyr01_15g: ImgPyrClsO00  codeStr='15g' room1=py1Rm14 room2=py1Rm15 conn1=py1CnE14;
imgPyr01_16a: ImgPyrClsFF00 codeStr='16a' room1=py1Rm11 room2=py1Rm12 conn1=py1CnN11 conn2=py1CnN12 conn3=py1CnE11 conn4=py1CnE16;
imgPyr01_16b: ImgPyrClsF0   codeStr='16b' room1=py1Rm12               conn1=py1CnN12;
imgPyr01_16c: ImgPyrClsFF00 codeStr='16c' room1=py1Rm12 room2=py1Rm13 conn1=py1CnN12 conn2=py1CnN13 conn3=py1CnE12 conn4=py1CnE17;
imgPyr01_16d: ImgPyrClsF0   codeStr='16d' room1=py1Rm13               conn1=py1CnN13;
imgPyr01_16e: ImgPyrClsFF00 codeStr='16e' room1=py1Rm13 room2=py1Rm14 conn1=py1CnN13 conn2=py1CnN14 conn3=py1CnE13 conn4=py1CnE18;
imgPyr01_16f: ImgPyrClsF0   codeStr='16f' room1=py1Rm14               conn1=py1CnN14;
imgPyr01_16g: ImgPyrClsFF00 codeStr='16g' room1=py1Rm14 room2=py1Rm15 conn1=py1CnN14 conn2=py1CnN15 conn3=py1CnE14 conn4=py1CnE19;
imgPyr01_17a: ImgPyrClsFF00 codeStr='17a' room1=py1Rm16 room2=py1Rm17 conn1=py1CnN11 conn2=py1CnN12 conn3=py1CnE11 conn4=py1CnE16;
imgPyr01_17b: ImgPyrClsF0   codeStr='17b' room1=py1Rm17               conn1=py1CnN12;
imgPyr01_17c: ImgPyrClsFF00 codeStr='17c' room1=py1Rm17 room2=py1Rm18 conn1=py1CnN12 conn2=py1CnN13 conn3=py1CnE12 conn4=py1CnE17;
imgPyr01_17d: ImgPyrClsF0   codeStr='17d' room1=py1Rm18               conn1=py1CnN13;
imgPyr01_17e: ImgPyrClsFF00 codeStr='17e' room1=py1Rm18 room2=py1Rm19 conn1=py1CnN13 conn2=py1CnN14 conn3=py1CnE13 conn4=py1CnE18;
imgPyr01_17f: ImgPyrClsF0   codeStr='17f' room1=py1Rm19               conn1=py1CnN14;
imgPyr01_17g: ImgPyrClsFF00 codeStr='17g' room1=py1Rm19 room2=py1Rm20 conn1=py1CnN14 conn2=py1CnN15 conn3=py1CnE14 conn4=py1CnE19;
imgPyr01_18a: ImgPyrClsO00  codeStr='18a' room1=py1Rm16 room2=py1Rm17 conn1=py1CnE16;
imgPyr01_18b: ImgPyrClsV0   codeStr='18b' room1=py1Rm17               conn1=py1CnU17;
imgPyr01_18c: ImgPyrClsO00  codeStr='18c' room1=py1Rm17 room2=py1Rm18 conn1=py1CnE17;
imgPyr01_18d: ImgPyrClsV0   codeStr='18d' room1=py1Rm18               conn1=py1CnU18;
imgPyr01_18e: ImgPyrClsO00  codeStr='18e' room1=py1Rm18 room2=py1Rm19 conn1=py1CnE18;
imgPyr01_18f: ImgPyrClsV0   codeStr='18f' room1=py1Rm19               conn1=py1CnU19;
imgPyr01_18g: ImgPyrClsO00  codeStr='18g' room1=py1Rm19 room2=py1Rm20 conn1=py1CnE19;
imgPyr01_19a: ImgPyrClsFF00 codeStr='19a' room1=py1Rm16 room2=py1Rm17 conn1=py1CnN16 conn2=py1CnN17 conn3=py1CnE16 conn4=py1CnE21;
imgPyr01_19b: ImgPyrClsF0   codeStr='19b' room1=py1Rm17               conn1=py1CnN17;
imgPyr01_19c: ImgPyrClsFF00 codeStr='19c' room1=py1Rm17 room2=py1Rm18 conn1=py1CnN17 conn2=py1CnN18 conn3=py1CnE17 conn4=py1CnE22;
imgPyr01_19d: ImgPyrClsF0   codeStr='19d' room1=py1Rm18               conn1=py1CnN18;
imgPyr01_19e: ImgPyrClsFF00 codeStr='19e' room1=py1Rm18 room2=py1Rm19 conn1=py1CnN18 conn2=py1CnN19 conn3=py1CnE18 conn4=py1CnE23;
imgPyr01_19f: ImgPyrClsF0   codeStr='19f' room1=py1Rm19               conn1=py1CnN19;
imgPyr01_19g: ImgPyrClsFF00 codeStr='19g' room1=py1Rm19 room2=py1Rm20 conn1=py1CnN19 conn2=py1CnN20 conn3=py1CnE19 conn4=py1CnE24;
imgPyr01_20a: ImgPyrClsFF00 codeStr='20a' room1=py1Rm21 room2=py1Rm22 conn1=py1CnN16 conn2=py1CnN17 conn3=py1CnE16 conn4=py1CnE21;
imgPyr01_20b: ImgPyrClsF0   codeStr='20b' room1=py1Rm22               conn1=py1CnN17;
imgPyr01_20c: ImgPyrClsFF00 codeStr='20c' room1=py1Rm22 room2=py1Rm23 conn1=py1CnN17 conn2=py1CnN18 conn3=py1CnE17 conn4=py1CnE22;
imgPyr01_20d: ImgPyrClsF0   codeStr='20d' room1=py1Rm23               conn1=py1CnN18;
imgPyr01_20e: ImgPyrClsFF00 codeStr='20e' room1=py1Rm23 room2=py1Rm24 conn1=py1CnN18 conn2=py1CnN19 conn3=py1CnE18 conn4=py1CnE23;
imgPyr01_20f: ImgPyrClsF0   codeStr='20f' room1=py1Rm24               conn1=py1CnN19;
imgPyr01_20g: ImgPyrClsFF00 codeStr='20g' room1=py1Rm24 room2=py1Rm25 conn1=py1CnN19 conn2=py1CnN20 conn3=py1CnE19 conn4=py1CnE24;
imgPyr01_21a: ImgPyrClsO00  codeStr='21a' room1=py1Rm21 room2=py1Rm22 conn1=py1CnE21;
imgPyr01_21b: ImgPyrCls0    codeStr='21b' room1=py1Rm22;
imgPyr01_21c: ImgPyrClsO00  codeStr='21c' room1=py1Rm22 room2=py1Rm23 conn1=py1CnE22;
imgPyr01_21d: ImgPyrCls0    codeStr='21d' room1=py1Rm23;
imgPyr01_21e: ImgPyrClsO00  codeStr='21e' room1=py1Rm23 room2=py1Rm24 conn1=py1CnE23;
imgPyr01_21f: ImgPyrCls0    codeStr='21f' room1=py1Rm24;
imgPyr01_21g: ImgPyrClsO00  codeStr='21g' room1=py1Rm24 room2=py1Rm25 conn1=py1CnE24;

imgPyr02_01a: ImgPyrCls     codeStr='01a';
imgPyr02_01b: ImgPyrCls     codeStr='01b';
imgPyr02_01c: ImgPyrCls0    codeStr='01c' room1=py2Rm35;
imgPyr02_01d: ImgPyrClsV0   codeStr='01d' room1=py2Rm35               conn1=py2CnU30;
imgPyr02_01e: ImgPyrCls0    codeStr='01e' room1=py2Rm35;
imgPyr02_01f: ImgPyrCls     codeStr='01f';
imgPyr02_01g: ImgPyrCls     codeStr='01g';
imgPyr02_02a: ImgPyrCls0    codeStr='02a' room1=py2Rm26;
imgPyr02_02b: ImgPyrClsV0   codeStr='02b' room1=py2Rm26               conn1=py2CnU07;
imgPyr02_02c: ImgPyrClsO00  codeStr='02c' room1=py2Rm26 room2=py2Rm27 conn1=py2CnE26;
imgPyr02_02d: ImgPyrClsV0   codeStr='02d' room1=py2Rm27               conn1=py2CnU08;
imgPyr02_02e: ImgPyrClsO00  codeStr='02e' room1=py2Rm27 room2=py2Rm28 conn1=py2CnE27;
imgPyr02_02f: ImgPyrClsV0   codeStr='02f' room1=py2Rm28               conn1=py2CnU09;
imgPyr02_02g: ImgPyrCls0    codeStr='02g' room1=py2Rm28;
imgPyr02_03a: ImgPyrClsF0   codeStr='03a' room1=py2Rm26               conn1=py2CnN26;
imgPyr02_03b: ImgPyrClsF0   codeStr='03b' room1=py2Rm26               conn1=py2CnN26;
imgPyr02_03c: ImgPyrClsFF00 codeStr='03c' room1=py2Rm26 room2=py2Rm27 conn1=py2CnN26 conn2=py2CnN27 conn3=py2CnE26 conn4=py2CnE29;
imgPyr02_03d: ImgPyrClsF0   codeStr='03d' room1=py2Rm27               conn1=py2CnN27;
imgPyr02_03e: ImgPyrClsFF00 codeStr='03e' room1=py2Rm27 room2=py2Rm28 conn1=py2CnN27 conn2=py2CnN28 conn3=py2CnE27 conn4=py2CnE30;
imgPyr02_03f: ImgPyrClsF0   codeStr='03f' room1=py2Rm28               conn1=py2CnN28;
imgPyr02_03g: ImgPyrClsF0   codeStr='03g' room1=py2Rm28               conn1=py2CnN28;
imgPyr02_04a: ImgPyrClsF0   codeStr='04a' room1=py2Rm29               conn1=py2CnN26;
imgPyr02_04b: ImgPyrClsF0   codeStr='04b' room1=py2Rm29               conn1=py2CnN26;
imgPyr02_04c: ImgPyrClsFF00 codeStr='04c' room1=py2Rm29 room2=py2Rm30 conn1=py2CnN26 conn2=py2CnN27 conn3=py2CnE26 conn4=py2CnE29;
imgPyr02_04d: ImgPyrClsF0   codeStr='04d' room1=py2Rm30               conn1=py2CnN27;
imgPyr02_04e: ImgPyrClsFF00 codeStr='04e' room1=py2Rm30 room2=py2Rm31 conn1=py2CnN27 conn2=py2CnN28 conn3=py2CnE27 conn4=py2CnE30;
imgPyr02_04f: ImgPyrClsF0   codeStr='04f' room1=py2Rm31               conn1=py2CnN28;
imgPyr02_04g: ImgPyrClsF0   codeStr='04g' room1=py2Rm31               conn1=py2CnN28;
imgPyr02_05a: ImgPyrCls0    codeStr='05a' room1=py2Rm29;
imgPyr02_05b: ImgPyrClsV0   codeStr='05b' room1=py2Rm29               conn1=py2CnU12;
imgPyr02_05c: ImgPyrClsO00  codeStr='05c' room1=py2Rm29 room2=py2Rm30 conn1=py2CnE29;
imgPyr02_05d: ImgPyrClsVV0  codeStr='05d' room1=py2Rm30               conn1=py2CnU13 conn2=py2CnU30;
imgPyr02_05e: ImgPyrClsO00  codeStr='05e' room1=py2Rm30 room2=py2Rm31 conn1=py2CnE30;
imgPyr02_05f: ImgPyrClsV0   codeStr='05f' room1=py2Rm31               conn1=py2CnU14;
imgPyr02_05g: ImgPyrCls0    codeStr='05g' room1=py2Rm31;
imgPyr02_06a: ImgPyrClsF0   codeStr='06a' room1=py2Rm29               conn1=py2CnN29;
imgPyr02_06b: ImgPyrClsF0   codeStr='06b' room1=py2Rm29               conn1=py2CnN29;
imgPyr02_06c: ImgPyrClsFF00 codeStr='06c' room1=py2Rm29 room2=py2Rm30 conn1=py2CnN29 conn2=py2CnN30 conn3=py2CnE29 conn4=py2CnE32;
imgPyr02_06d: ImgPyrClsF0   codeStr='06d' room1=py2Rm30               conn1=py2CnN30;
imgPyr02_06e: ImgPyrClsFF00 codeStr='06e' room1=py2Rm30 room2=py2Rm31 conn1=py2CnN30 conn2=py2CnN31 conn3=py2CnE30 conn4=py2CnE33;
imgPyr02_06f: ImgPyrClsF0   codeStr='06f' room1=py2Rm31               conn1=py2CnN31;
imgPyr02_06g: ImgPyrClsF0   codeStr='06g' room1=py2Rm31               conn1=py2CnN31;
imgPyr02_07a: ImgPyrClsF0   codeStr='07a' room1=py2Rm32               conn1=py2CnN29;
imgPyr02_07b: ImgPyrClsF0   codeStr='07b' room1=py2Rm32               conn1=py2CnN29;
imgPyr02_07c: ImgPyrClsFF00 codeStr='07c' room1=py2Rm32 room2=py2Rm33 conn1=py2CnN29 conn2=py2CnN30 conn3=py2CnE29 conn4=py2CnE32;
imgPyr02_07d: ImgPyrClsF0   codeStr='07d' room1=py2Rm33               conn1=py2CnN30;
imgPyr02_07e: ImgPyrClsFF00 codeStr='07e' room1=py2Rm33 room2=py2Rm34 conn1=py2CnN30 conn2=py2CnN31 conn3=py2CnE30 conn4=py2CnE33;
imgPyr02_07f: ImgPyrClsF0   codeStr='07f' room1=py2Rm34               conn1=py2CnN31;
imgPyr02_07g: ImgPyrClsF0   codeStr='07g' room1=py2Rm34               conn1=py2CnN31;
imgPyr02_08a: ImgPyrCls0    codeStr='08a' room1=py2Rm32;
imgPyr02_08b: ImgPyrClsV0   codeStr='08b' room1=py2Rm32               conn1=py2CnU17;
imgPyr02_08c: ImgPyrClsO00  codeStr='08c' room1=py2Rm32 room2=py2Rm33 conn1=py2CnE32;
imgPyr02_08d: ImgPyrClsV0   codeStr='08d' room1=py2Rm33               conn1=py2CnU18;
imgPyr02_08e: ImgPyrClsO00  codeStr='08e' room1=py2Rm33 room2=py2Rm34 conn1=py2CnE33;
imgPyr02_08f: ImgPyrClsV0   codeStr='08f' room1=py2Rm34               conn1=py2CnU19;
imgPyr02_08g: ImgPyrCls0    codeStr='08g' room1=py2Rm34;
imgPyr02_09a: ImgPyrClsO00  codeStr='09a' room1=py2Rm01 room2=py2Rm02 conn1=py2CnE01;
imgPyr02_09b: ImgPyrClsO0   codeStr='09b' room1=py2Rm02               conn1=py2CnE01;
imgPyr02_09c: ImgPyrClsO00  codeStr='09c' room1=py2Rm02 room2=py2Rm03 conn1=py2CnE02;
imgPyr02_09d: ImgPyrCls0    codeStr='09d' room1=py2Rm03;
imgPyr02_09e: ImgPyrClsO00  codeStr='09e' room1=py2Rm03 room2=py2Rm04 conn1=py2CnE03;
imgPyr02_09f: ImgPyrClsO0   codeStr='09f' room1=py2Rm04               conn1=py2CnE04;
imgPyr02_09g: ImgPyrClsO00  codeStr='09g' room1=py2Rm04 room2=py2Rm05 conn1=py2CnE04;
imgPyr02_10a: ImgPyrClsFF00 codeStr='10a' room1=py2Rm01 room2=py2Rm02 conn1=py2CnN01 conn2=py2CnN02 conn3=py2CnE01 conn4=py2CnE06;
imgPyr02_10b: ImgPyrClsF0   codeStr='10b' room1=py2Rm02               conn1=py2CnN02;
imgPyr02_10c: ImgPyrClsFF00 codeStr='10c' room1=py2Rm02 room2=py2Rm03 conn1=py2CnN02 conn2=py2CnN03 conn3=py2CnE02 conn4=py2CnE07;
imgPyr02_10d: ImgPyrClsF0   codeStr='10d' room1=py2Rm03               conn1=py2CnN03;
imgPyr02_10e: ImgPyrClsFF00 codeStr='10e' room1=py2Rm03 room2=py2Rm04 conn1=py2CnN03 conn2=py2CnN04 conn3=py2CnE03 conn4=py2CnE08;
imgPyr02_10f: ImgPyrClsF0   codeStr='10f' room1=py2Rm04               conn1=py2CnN04;
imgPyr02_10g: ImgPyrClsFF00 codeStr='10g' room1=py2Rm04 room2=py2Rm05 conn1=py2CnN04 conn2=py2CnN05 conn3=py2CnE04 conn4=py2CnE09;
imgPyr02_11a: ImgPyrClsFF00 codeStr='11a' room1=py2Rm06 room2=py2Rm07 conn1=py2CnN01 conn2=py2CnN02 conn3=py2CnE01 conn4=py2CnE06;
imgPyr02_11b: ImgPyrClsF0   codeStr='11b' room1=py2Rm07               conn1=py2CnN02;
imgPyr02_11c: ImgPyrClsFF00 codeStr='11c' room1=py2Rm07 room2=py2Rm08 conn1=py2CnN02 conn2=py2CnN03 conn3=py2CnE02 conn4=py2CnE07;
imgPyr02_11d: ImgPyrClsF0   codeStr='11d' room1=py2Rm08               conn1=py2CnN03;
imgPyr02_11e: ImgPyrClsFF00 codeStr='11e' room1=py2Rm08 room2=py2Rm09 conn1=py2CnN03 conn2=py2CnN04 conn3=py2CnE03 conn4=py2CnE08;
imgPyr02_11f: ImgPyrClsF0   codeStr='11f' room1=py2Rm09               conn1=py2CnN04;
imgPyr02_11g: ImgPyrClsFF00 codeStr='11g' room1=py2Rm09 room2=py2Rm10 conn1=py2CnN04 conn2=py2CnN05 conn3=py2CnE04 conn4=py2CnE09;
imgPyr02_12a: ImgPyrClsO00  codeStr='12a' room1=py2Rm06 room2=py2Rm07 conn1=py2CnE06;
imgPyr02_12b: ImgPyrClsV0   codeStr='12b' room1=py2Rm07               conn1=py2CnU07;
imgPyr02_12c: ImgPyrClsO00  codeStr='12c' room1=py2Rm07 room2=py2Rm08 conn1=py2CnE07;
imgPyr02_12d: ImgPyrClsV0   codeStr='12d' room1=py2Rm08               conn1=py2CnU08;
imgPyr02_12e: ImgPyrClsO00  codeStr='12e' room1=py2Rm08 room2=py2Rm09 conn1=py2CnE08;
imgPyr02_12f: ImgPyrClsV0   codeStr='12f' room1=py2Rm09               conn1=py2CnU09;
imgPyr02_12g: ImgPyrClsO00  codeStr='12g' room1=py2Rm09 room2=py2Rm10 conn1=py2CnE09;
imgPyr02_13a: ImgPyrClsFF00 codeStr='13a' room1=py2Rm06 room2=py2Rm07 conn1=py2CnN06 conn2=py2CnN07 conn3=py2CnE06 conn4=py2CnE11;
imgPyr02_13b: ImgPyrClsF0   codeStr='13b' room1=py2Rm07               conn1=py2CnN07;
imgPyr02_13c: ImgPyrClsFF00 codeStr='13c' room1=py2Rm07 room2=py2Rm08 conn1=py2CnN07 conn2=py2CnN08 conn3=py2CnE07 conn4=py2CnE12;
imgPyr02_13d: ImgPyrClsF0   codeStr='13d' room1=py2Rm08               conn1=py2CnN08;
imgPyr02_13e: ImgPyrClsFF00 codeStr='13e' room1=py2Rm08 room2=py2Rm09 conn1=py2CnN08 conn2=py2CnN09 conn3=py2CnE08 conn4=py2CnE13;
imgPyr02_13f: ImgPyrClsF0   codeStr='13f' room1=py2Rm09               conn1=py2CnN09;
imgPyr02_13g: ImgPyrClsFF00 codeStr='13g' room1=py2Rm09 room2=py2Rm10 conn1=py2CnN09 conn2=py2CnN10 conn3=py2CnE09 conn4=py2CnE14;
imgPyr02_14a: ImgPyrClsFF00 codeStr='14a' room1=py2Rm11 room2=py2Rm12 conn1=py2CnN06 conn2=py2CnN07 conn3=py2CnE06 conn4=py2CnE11;
imgPyr02_14b: ImgPyrClsF0   codeStr='14b' room1=py2Rm12               conn1=py2CnN07;
imgPyr02_14c: ImgPyrClsFF00 codeStr='14c' room1=py2Rm12 room2=py2Rm13 conn1=py2CnN07 conn2=py2CnN08 conn3=py2CnE07 conn4=py2CnE12;
imgPyr02_14d: ImgPyrClsF0   codeStr='14d' room1=py2Rm13               conn1=py2CnN08;
imgPyr02_14e: ImgPyrClsFF00 codeStr='14e' room1=py2Rm13 room2=py2Rm14 conn1=py2CnN08 conn2=py2CnN09 conn3=py2CnE08 conn4=py2CnE13;
imgPyr02_14f: ImgPyrClsF0   codeStr='14f' room1=py2Rm14               conn1=py2CnN09;
imgPyr02_14g: ImgPyrClsFF00 codeStr='14g' room1=py2Rm14 room2=py2Rm15 conn1=py2CnN09 conn2=py2CnN10 conn3=py2CnE09 conn4=py2CnE14;
imgPyr02_15a: ImgPyrClsO00  codeStr='15a' room1=py2Rm11 room2=py2Rm12 conn1=py2CnE11;
imgPyr02_15b: ImgPyrClsV0   codeStr='15b' room1=py2Rm12               conn1=py2CnU12;
imgPyr02_15c: ImgPyrClsO00  codeStr='15c' room1=py2Rm12 room2=py2Rm13 conn1=py2CnE12;
imgPyr02_15d: ImgPyrClsV0   codeStr='15d' room1=py2Rm13               conn1=py2CnU13;
imgPyr02_15e: ImgPyrClsO00  codeStr='15e' room1=py2Rm13 room2=py2Rm14 conn1=py2CnE13;
imgPyr02_15f: ImgPyrClsV0   codeStr='15f' room1=py2Rm14               conn1=py2CnU14;
imgPyr02_15g: ImgPyrClsO00  codeStr='15g' room1=py2Rm14 room2=py2Rm15 conn1=py2CnE14;
imgPyr02_16a: ImgPyrClsFF00 codeStr='16a' room1=py2Rm11 room2=py2Rm12 conn1=py2CnN11 conn2=py2CnN12 conn3=py2CnE11 conn4=py2CnE16;
imgPyr02_16b: ImgPyrClsF0   codeStr='16b' room1=py2Rm12               conn1=py2CnN12;
imgPyr02_16c: ImgPyrClsFF00 codeStr='16c' room1=py2Rm12 room2=py2Rm13 conn1=py2CnN12 conn2=py2CnN13 conn3=py2CnE12 conn4=py2CnE17;
imgPyr02_16d: ImgPyrClsF0   codeStr='16d' room1=py2Rm13               conn1=py2CnN13;
imgPyr02_16e: ImgPyrClsFF00 codeStr='16e' room1=py2Rm13 room2=py2Rm14 conn1=py2CnN13 conn2=py2CnN14 conn3=py2CnE13 conn4=py2CnE18;
imgPyr02_16f: ImgPyrClsF0   codeStr='16f' room1=py2Rm14               conn1=py2CnN14;
imgPyr02_16g: ImgPyrClsFF00 codeStr='16g' room1=py2Rm14 room2=py2Rm15 conn1=py2CnN14 conn2=py2CnN15 conn3=py2CnE14 conn4=py2CnE19;
imgPyr02_17a: ImgPyrClsFF00 codeStr='17a' room1=py2Rm16 room2=py2Rm17 conn1=py2CnN11 conn2=py2CnN12 conn3=py2CnE11 conn4=py2CnE16;
imgPyr02_17b: ImgPyrClsF0   codeStr='17b' room1=py2Rm17               conn1=py2CnN12;
imgPyr02_17c: ImgPyrClsFF00 codeStr='17c' room1=py2Rm17 room2=py2Rm18 conn1=py2CnN12 conn2=py2CnN13 conn3=py2CnE12 conn4=py2CnE17;
imgPyr02_17d: ImgPyrClsF0   codeStr='17d' room1=py2Rm18               conn1=py2CnN13;
imgPyr02_17e: ImgPyrClsFF00 codeStr='17e' room1=py2Rm18 room2=py2Rm19 conn1=py2CnN13 conn2=py2CnN14 conn3=py2CnE13 conn4=py2CnE18;
imgPyr02_17f: ImgPyrClsF0   codeStr='17f' room1=py2Rm19               conn1=py2CnN14;
imgPyr02_17g: ImgPyrClsFF00 codeStr='17g' room1=py2Rm19 room2=py2Rm20 conn1=py2CnN14 conn2=py2CnN15 conn3=py2CnE14 conn4=py2CnE19;
imgPyr02_18a: ImgPyrClsO00  codeStr='18a' room1=py2Rm16 room2=py2Rm17 conn1=py2CnE16;
imgPyr02_18b: ImgPyrClsV0   codeStr='18b' room1=py2Rm17               conn1=py2CnU17;
imgPyr02_18c: ImgPyrClsO00  codeStr='18c' room1=py2Rm17 room2=py2Rm18 conn1=py2CnE17;
imgPyr02_18d: ImgPyrClsV0   codeStr='18d' room1=py2Rm18               conn1=py2CnU18;
imgPyr02_18e: ImgPyrClsO00  codeStr='18e' room1=py2Rm18 room2=py2Rm19 conn1=py2CnE18;
imgPyr02_18f: ImgPyrClsV0   codeStr='18f' room1=py2Rm19               conn1=py2CnU19;
imgPyr02_18g: ImgPyrClsO00  codeStr='18g' room1=py2Rm19 room2=py2Rm20 conn1=py2CnE19;
imgPyr02_19a: ImgPyrClsFF00 codeStr='19a' room1=py2Rm16 room2=py2Rm17 conn1=py2CnN16 conn2=py2CnN17 conn3=py2CnE16 conn4=py2CnE21;
imgPyr02_19b: ImgPyrClsF0   codeStr='19b' room1=py2Rm17               conn1=py2CnN17;
imgPyr02_19c: ImgPyrClsFF00 codeStr='19c' room1=py2Rm17 room2=py2Rm18 conn1=py2CnN17 conn2=py2CnN18 conn3=py2CnE17 conn4=py2CnE22;
imgPyr02_19d: ImgPyrClsF0   codeStr='19d' room1=py2Rm18               conn1=py2CnN18;
imgPyr02_19e: ImgPyrClsFF00 codeStr='19e' room1=py2Rm18 room2=py2Rm19 conn1=py2CnN18 conn2=py2CnN19 conn3=py2CnE18 conn4=py2CnE23;
imgPyr02_19f: ImgPyrClsF0   codeStr='19f' room1=py2Rm19               conn1=py2CnN19;
imgPyr02_19g: ImgPyrClsFF00 codeStr='19g' room1=py2Rm19 room2=py2Rm20 conn1=py2CnN19 conn2=py2CnN20 conn3=py2CnE19 conn4=py2CnE24;
imgPyr02_20a: ImgPyrClsFF00 codeStr='20a' room1=py2Rm21 room2=py2Rm22 conn1=py2CnN16 conn2=py2CnN17 conn3=py2CnE16 conn4=py2CnE21;
imgPyr02_20b: ImgPyrClsF0   codeStr='20b' room1=py2Rm22               conn1=py2CnN17;
imgPyr02_20c: ImgPyrClsFF00 codeStr='20c' room1=py2Rm22 room2=py2Rm23 conn1=py2CnN17 conn2=py2CnN18 conn3=py2CnE17 conn4=py2CnE22;
imgPyr02_20d: ImgPyrClsF0   codeStr='20d' room1=py2Rm23               conn1=py2CnN18;
imgPyr02_20e: ImgPyrClsFF00 codeStr='20e' room1=py2Rm23 room2=py2Rm24 conn1=py2CnN18 conn2=py2CnN19 conn3=py2CnE18 conn4=py2CnE23;
imgPyr02_20f: ImgPyrClsF0   codeStr='20f' room1=py2Rm24               conn1=py2CnN19;
imgPyr02_20g: ImgPyrClsFF00 codeStr='20g' room1=py2Rm24 room2=py2Rm25 conn1=py2CnN19 conn2=py2CnN20 conn3=py2CnE19 conn4=py2CnE24;
imgPyr02_21a: ImgPyrClsO00  codeStr='21a' room1=py2Rm21 room2=py2Rm22 conn1=py2CnE21;
imgPyr02_21b: ImgPyrCls0    codeStr='21b' room1=py2Rm22;
imgPyr02_21c: ImgPyrClsO00  codeStr='21c' room1=py2Rm22 room2=py2Rm23 conn1=py2CnE22;
imgPyr02_21d: ImgPyrCls0    codeStr='21d' room1=py2Rm23;
imgPyr02_21e: ImgPyrClsO00  codeStr='21e' room1=py2Rm23 room2=py2Rm24 conn1=py2CnE23;
imgPyr02_21f: ImgPyrCls0    codeStr='21f' room1=py2Rm24;
imgPyr02_21g: ImgPyrClsO00  codeStr='21g' room1=py2Rm24 room2=py2Rm25 conn1=py2CnE24;

function pyramid1MapPrint() {
return '<table border="0" cellspacing="0" cellpadding="0">
  <tr>
    <td><img src="pyr1/' + imgPyr01_01a.rtnStr + '.png" width="100" height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_01b.rtnStr + '.png" width="24"  height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_01c.rtnStr + '.png" width="24"  height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_01d.rtnStr + '.png" width="29"  height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_01e.rtnStr + '.png" width="24"  height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_01f.rtnStr + '.png" width="24"  height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_01g.rtnStr + '.png" width="100" height="61" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr01_02a.rtnStr + '.png" width="100" height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_02b.rtnStr + '.png" width="24"  height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_02c.rtnStr + '.png" width="24"  height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_02d.rtnStr + '.png" width="29"  height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_02e.rtnStr + '.png" width="24"  height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_02f.rtnStr + '.png" width="24"  height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_02g.rtnStr + '.png" width="100" height="16" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr01_03a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_03b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_03c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_03d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_03e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_03f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_03g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr01_04a.rtnStr + '.png" width="100" height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_04b.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_04c.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_04d.rtnStr + '.png" width="29"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_04e.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_04f.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_04g.rtnStr + '.png" width="100" height="2"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr01_05a.rtnStr + '.png" width="100" height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_05b.rtnStr + '.png" width="24"  height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_05c.rtnStr + '.png" width="24"  height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_05d.rtnStr + '.png" width="29"  height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_05e.rtnStr + '.png" width="24"  height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_05f.rtnStr + '.png" width="24"  height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_05g.rtnStr + '.png" width="100" height="10" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr01_06a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_06b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_06c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_06d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_06e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_06f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_06g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr01_07a.rtnStr + '.png" width="100" height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_07b.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_07c.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_07d.rtnStr + '.png" width="29"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_07e.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_07f.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr01_07g.rtnStr + '.png" width="100" height="2"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr01_08a.rtnStr + '.png" width="100" height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_08b.rtnStr + '.png" width="24"  height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_08c.rtnStr + '.png" width="24"  height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_08d.rtnStr + '.png" width="29"  height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_08e.rtnStr + '.png" width="24"  height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_08f.rtnStr + '.png" width="24"  height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr01_08g.rtnStr + '.png" width="100" height="62" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr2/' + imgPyr01_09a.rtnStr + '.png" width="100" height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_09b.rtnStr + '.png" width="24"  height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_09c.rtnStr + '.png" width="24"  height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_09d.rtnStr + '.png" width="29"  height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_09e.rtnStr + '.png" width="24"  height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_09f.rtnStr + '.png" width="24"  height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_09g.rtnStr + '.png" width="100" height="25" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr2/' + imgPyr01_10a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_10b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_10c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_10d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_10e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_10f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_10g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr2/' + imgPyr01_11a.rtnStr + '.png" width="100" height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_11b.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_11c.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_11d.rtnStr + '.png" width="29"  height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_11e.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_11f.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_11g.rtnStr + '.png" width="100" height="2"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr2/' + imgPyr01_12a.rtnStr + '.png" width="100" height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_12b.rtnStr + '.png" width="24"  height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_12c.rtnStr + '.png" width="24"  height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_12d.rtnStr + '.png" width="29"  height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_12e.rtnStr + '.png" width="24"  height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_12f.rtnStr + '.png" width="24"  height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr01_12g.rtnStr + '.png" width="100" height="21" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr2/' + imgPyr01_13a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_13b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_13c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_13d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_13e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_13f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr01_13g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr3/' + imgPyr01_14a.rtnStr + '.png" width="100" height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_14b.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_14c.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_14d.rtnStr + '.png" width="29"  height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_14e.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_14f.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_14g.rtnStr + '.png" width="100" height="2"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr3/' + imgPyr01_15a.rtnStr + '.png" width="100" height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr01_15b.rtnStr + '.png" width="24"  height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr01_15c.rtnStr + '.png" width="24"  height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr01_15d.rtnStr + '.png" width="29"  height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr01_15e.rtnStr + '.png" width="24"  height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr01_15f.rtnStr + '.png" width="24"  height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr01_15g.rtnStr + '.png" width="100" height="24" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr3/' + imgPyr01_16a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_16b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_16c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_16d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_16e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_16f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr01_16g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr4/' + imgPyr01_17a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_17b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_17c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_17d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_17e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_17f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_17g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr4/' + imgPyr01_18a.rtnStr + '.png" width="100" height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_18b.rtnStr + '.png" width="24"  height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_18c.rtnStr + '.png" width="24"  height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_18d.rtnStr + '.png" width="29"  height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_18e.rtnStr + '.png" width="24"  height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_18f.rtnStr + '.png" width="24"  height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_18g.rtnStr + '.png" width="100" height="27" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr4/' + imgPyr01_19a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_19b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_19c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_19d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_19e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_19f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_19g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr4/' + imgPyr01_20a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_20b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_20c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_20d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_20e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_20f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr01_20g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr4/' + imgPyr01_21a.rtnStr + '.png" width="100" height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_21b.rtnStr + '.png" width="24"  height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_21c.rtnStr + '.png" width="24"  height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_21d.rtnStr + '.png" width="29"  height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_21e.rtnStr + '.png" width="24"  height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_21f.rtnStr + '.png" width="24"  height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr01_21g.rtnStr + '.png" width="100" height="86" border="0"></td>
  </tr>
</table>'
+ doCompass(upDown);
}

function pyramid2MapPrint() {
return '<table border="0" cellspacing="0" cellpadding="0">
  <tr>
    <td><img src="pyr1/' + imgPyr02_01a.rtnStr + '.png" width="100" height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_01b.rtnStr + '.png" width="24"  height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_01c.rtnStr + '.png" width="24"  height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_01d.rtnStr + '.png" width="29"  height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_01e.rtnStr + '.png" width="24"  height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_01f.rtnStr + '.png" width="24"  height="61" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_01g.rtnStr + '.png" width="100" height="61" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr02_02a.rtnStr + '.png" width="100" height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_02b.rtnStr + '.png" width="24"  height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_02c.rtnStr + '.png" width="24"  height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_02d.rtnStr + '.png" width="29"  height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_02e.rtnStr + '.png" width="24"  height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_02f.rtnStr + '.png" width="24"  height="16" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_02g.rtnStr + '.png" width="100" height="16" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr02_03a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_03b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_03c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_03d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_03e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_03f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_03g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr02_04a.rtnStr + '.png" width="100" height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_04b.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_04c.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_04d.rtnStr + '.png" width="29"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_04e.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_04f.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_04g.rtnStr + '.png" width="100" height="2"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr02_05a.rtnStr + '.png" width="100" height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_05b.rtnStr + '.png" width="24"  height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_05c.rtnStr + '.png" width="24"  height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_05d.rtnStr + '.png" width="29"  height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_05e.rtnStr + '.png" width="24"  height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_05f.rtnStr + '.png" width="24"  height="10" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_05g.rtnStr + '.png" width="100" height="10" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr02_06a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_06b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_06c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_06d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_06e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_06f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_06g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr02_07a.rtnStr + '.png" width="100" height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_07b.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_07c.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_07d.rtnStr + '.png" width="29"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_07e.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_07f.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr1/' + imgPyr02_07g.rtnStr + '.png" width="100" height="2"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr1/' + imgPyr02_08a.rtnStr + '.png" width="100" height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_08b.rtnStr + '.png" width="24"  height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_08c.rtnStr + '.png" width="24"  height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_08d.rtnStr + '.png" width="29"  height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_08e.rtnStr + '.png" width="24"  height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_08f.rtnStr + '.png" width="24"  height="62" border="0"></td>
    <td><img src="pyr1/' + imgPyr02_08g.rtnStr + '.png" width="100" height="62" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr2/' + imgPyr02_09a.rtnStr + '.png" width="100" height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_09b.rtnStr + '.png" width="24"  height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_09c.rtnStr + '.png" width="24"  height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_09d.rtnStr + '.png" width="29"  height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_09e.rtnStr + '.png" width="24"  height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_09f.rtnStr + '.png" width="24"  height="25" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_09g.rtnStr + '.png" width="100" height="25" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr2/' + imgPyr02_10a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_10b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_10c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_10d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_10e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_10f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_10g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr2/' + imgPyr02_11a.rtnStr + '.png" width="100" height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_11b.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_11c.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_11d.rtnStr + '.png" width="29"  height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_11e.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_11f.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_11g.rtnStr + '.png" width="100" height="2"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr2/' + imgPyr02_12a.rtnStr + '.png" width="100" height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_12b.rtnStr + '.png" width="24"  height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_12c.rtnStr + '.png" width="24"  height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_12d.rtnStr + '.png" width="29"  height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_12e.rtnStr + '.png" width="24"  height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_12f.rtnStr + '.png" width="24"  height="21" border="0"></td>
    <td><img src="pyr2/' + imgPyr02_12g.rtnStr + '.png" width="100" height="21" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr2/' + imgPyr02_13a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_13b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_13c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_13d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_13e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_13f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr2/' + imgPyr02_13g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr3/' + imgPyr02_14a.rtnStr + '.png" width="100" height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_14b.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_14c.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_14d.rtnStr + '.png" width="29"  height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_14e.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_14f.rtnStr + '.png" width="24"  height="2"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_14g.rtnStr + '.png" width="100" height="2"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr3/' + imgPyr02_15a.rtnStr + '.png" width="100" height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr02_15b.rtnStr + '.png" width="24"  height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr02_15c.rtnStr + '.png" width="24"  height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr02_15d.rtnStr + '.png" width="29"  height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr02_15e.rtnStr + '.png" width="24"  height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr02_15f.rtnStr + '.png" width="24"  height="24" border="0"></td>
    <td><img src="pyr3/' + imgPyr02_15g.rtnStr + '.png" width="100" height="24" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr3/' + imgPyr02_16a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_16b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_16c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_16d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_16e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_16f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr3/' + imgPyr02_16g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr4/' + imgPyr02_17a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_17b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_17c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_17d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_17e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_17f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_17g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr4/' + imgPyr02_18a.rtnStr + '.png" width="100" height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_18b.rtnStr + '.png" width="24"  height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_18c.rtnStr + '.png" width="24"  height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_18d.rtnStr + '.png" width="29"  height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_18e.rtnStr + '.png" width="24"  height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_18f.rtnStr + '.png" width="24"  height="27" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_18g.rtnStr + '.png" width="100" height="27" border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr4/' + imgPyr02_19a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_19b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_19c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_19d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_19e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_19f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_19g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr4/' + imgPyr02_20a.rtnStr + '.png" width="100" height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_20b.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_20c.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_20d.rtnStr + '.png" width="29"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_20e.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_20f.rtnStr + '.png" width="24"  height="3"  border="0"></td>
    <td><img src="pyr4/' + imgPyr02_20g.rtnStr + '.png" width="100" height="3"  border="0"></td>
  </tr>
  <tr>
    <td><img src="pyr4/' + imgPyr02_21a.rtnStr + '.png" width="100" height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_21b.rtnStr + '.png" width="24"  height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_21c.rtnStr + '.png" width="24"  height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_21d.rtnStr + '.png" width="29"  height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_21e.rtnStr + '.png" width="24"  height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_21f.rtnStr + '.png" width="24"  height="86" border="0"></td>
    <td><img src="pyr4/' + imgPyr02_21g.rtnStr + '.png" width="100" height="86" border="0"></td>
  </tr>
</table>'
+ doCompass(upDown);
}

