////////////////////////////////////////////////////////////////////////
//  
//  Inform.t Library File: Class 991214
//
//  Copyright (c) 1999 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the Inform.t library extension for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#include <declare.t>

#pragma C+

/*
 *  preinitObj: object
 *
 *  The preinit object is designed to allow for multiple occurrences
 *  of preinit(). Each instance of a preinit object should perform a
 *  unique function within your source. The piomethod() should not
 *	return any particular value,
 */
class preinitObj: object
	ispreinitObj = true
	piomethod( cmd ) =
	{
		return;
	}
;

/*
 *  preparseObj: object
 *
 *  The preparse object is designed to allow for multiple occurrences
 *  of preparse(). Each instance of a preparse object should perform a
 *  unique function within your source. The ppomethod() should return
 *  either true, nil, or a new command string, as required by
 *  preparse().
 */
class preparseObj: object
	ispreparseObj = true
	ppomethod( cmd ) =
	{
		return( true );
	}
;

/*
 *  listobj: object
 *
 *  A dynamically created object used for list_together purposes. This
 *  is a temporary object that only exists during the listing process.
 */
class listobj: object
    contents = []
;

/*
 *	wysiwyg: readable
 *
 *	The wysiwyg is meant to facilitate the creation of maps and other
 *	signs that the player encounters within a game. The layout of the map
 *	can be entered in the headerText, bodyText, and footerText lists as
 *	(single-quote) string elements, each string being a separate line of 
 *	text in the map. 
 *
 *	Formatting for HTML can be modified through the various Fmt methods.
 *	The size and colour of the box will need to be adjusted to the size 
 *	of your map. For best results Courier New should be used, as it 
 *	provides relatively little distortion.
 */
class wysiwyg: readable

#ifdef USE_HTML_STATUS

    ldesc =
    {
        if (systemInfo(__SYSINFO_SYSINFO) == true
        && systemInfo(__SYSINFO_HTML) == 1)
        {
        	self.tableFmtBegin;
        	self.tableBGCOLOR;
        	if (length(self.headerText) > 0)
        		self.displayHeader;
        	if (length(self.bodyText) > 0)
        		self.displayBody;
        	if (length(self.footerText) > 0)
        		self.displayFooter;
        	self.tableFmtEnd;
        }
		else
		{
        	if (length(self.headerText) > 0)
        		printWysiwyg(self.headerText);
        	if (length(self.bodyText) > 0)
        		printWysiwyg(self.bodyText);
        	if (length(self.footerText) > 0)
        		printWysiwyg(self.footerText);
		}
    }
    
#else 	/* USE_HTML_STATUS */
    
    ldesc =
    {
    	if (length(self.headerText) > 0)
      		printwysiwyg(self.headerText);
   		if (length(self.bodyText) > 0)
       		printwysiwyg(self.bodyText);
     	if (length(self.footerText) > 0)
     		printwysiwyg(self.footerText);
    }
    
#endif	/* USE_HTML_STATUS */
    
    displayHeader =
    {
        self.headerFont;
        self.headerFmtBegin;
        printWysiwyg(self.headerText);
        self.headerFmtEnd;
    }
    displayBody =
    {
        self.bodyFont;
        self.bodyFmtBegin;
        printWysiwyg(self.bodyText);
        self.bodyFmtEnd;        
    }
    displayFooter =
    {
        self.footerFont;
        self.footerFmtBegin;
        printWysiwyg(self.footerText);
        self.footerFmtEnd;        
    }
    tableFmtBegin =
    {
        "<BR>
        <P>
        <CENTER>
        <TABLE BORDER=1 CELLPADDING=15>";
    }
    tableBGCOLOR =
    {
        "<TR BGCOLOR=\"#AAAAAA\">
        <TD BGCOLOR=\"#AAAAAA\">";       
    }
    tableFmtEnd = 
    {
       	"</TD>
        </TR>
        </TABLE>
        </CENTER>
        </P>";       
    }
    headerFont =
    {
        "<FONT COLOR=YELLOW SIZE=\"+1\" FACE=\"Courier New,Courier New\">";
    }
    headerFmtBegin =
    {
        "<P>
        <B>
        <CENTER>";
    }
    headerFmtEnd=
    {
        "</CENTER>
        </B>		
        </P>
        </FONT>";
    }
    bodyFont =
    {
        "<FONT COLOR=WHITE FACE=\"Courier New,Courier New\">";  
    }
    bodyFmtBegin =
    {
        "<P>";
    }
    bodyFmtEnd =
    {
        "</P>
        </FONT>";
    }
    footerFont =
    {
        "<FONT COLOR=YELLOW SIZE=\"-2\" FACE=\"Courier New,Courier New\">";
    }
    footerFmtBegin =
    {   
        "<P>
        <B>
        <CENTER>";
    }
    footerFmtEnd =
    {
        "</CENTER>
        </B>		
        </P>
        </FONT>";
    }
    headerText = []
    bodyText = []
    footerText = []
;

/*
 *	wysiwygTable: wysiwyg
 *
 *	The wysiwygTable is a wysiwyg which formats the heading, body, and footer
 *	areas into table rows (three in total) which gives each section a separate 
 *	border.
 *
 *	Additionally, each section of the table can then have its own background
 *	color as well as font style.
 */
class wysiwygTable: wysiwyg
#ifdef USE_HTML_STATUS

    ldesc =
    {
        if (systemInfo(__SYSINFO_SYSINFO) == true
        && systemInfo(__SYSINFO_HTML) == 1)
        {
        	self.tableFmtBegin;
        	if (length(self.headerText) > 0)
        		self.displayHeader;
        	if (length(self.bodyText) > 0)
        		self.displayBody;
        	if (length(self.footerText) > 0)
        		self.displayFooter;
        	self.tableFmtEnd;
        }
		else
		{
        	if (length(self.headerText) > 0)
        		printWysiwyg(self.headerText);
        	if (length(self.bodyText) > 0)
        		printWysiwyg(self.bodyText);
        	if (length(self.footerText) > 0)
        		printWysiwyg(self.footerText);
		}
    }
    
#else 	/* USE_HTML_STATUS */
    
    ldesc =
    {
    	if (length(self.headerText) > 0)
      		printWysiwyg(self.headerText);
   		if (length(self.bodyText) > 0)
       		printWysiwyg(self.bodyText);
     	if (length(self.footerText) > 0)
     		printWysiwyg(self.footerText);
    }
    
#endif	/* USE_HTML_STATUS */
    displayHeader =
    {
        self.headerBGCOLOR;
        self.headerFont;
        self.headerFmtBegin;
        printWysiwyg(self.headerText);
        self.headerFmtEnd;
    }
    displayBody =
    {
        self.bodyFmtBegin;
        self.bodyBGCOLOR;
        self.bodyFont;
        printWysiwyg(self.bodyText);
        self.bodyFmtEnd;        
    }
    displayFooter =
    {
        self.footerBGCOLOR;
        self.footerFont;
        self.footerFmtBegin;
        printWysiwyg(self.footerText);
        self.footerFmtEnd;        
    }
    tableFmtBegin =
    {
        "<BR>
        <P>
        <CENTER>
        <TABLE BORDER=1 CELLPADDING=15>";
    }
    tableFmtEnd = 
    {
        "</TABLE>
        </CENTER>
        </P>";        
    }
    headerBGCOLOR = 
    {
        "<TR BGCOLOR=\"#AAAAAA\">
        <TD BGCOLOR=\"#AAAAAA\">";        
    }
    headerFont =
    {
        "<FONT COLOR=YELLOW SIZE=\"+1\" FACE=\"Courier New,Courier New\">";
    }
    headerFmtBegin =
    {
        "<P>
        <B>
        <CENTER>";
    }
    headerFmtEnd=
    {
        "</CENTER>
        </B>
        </P>
        </FONT>
        </TD>
        </TR>";
    }
    bodyBGCOLOR = { self.headerBGCOLOR; }
    bodyFont = 
    {
        "<FONT COLOR=WHITE FACE=\"Courier New,Courier New\">";  
    }
    bodyFmtBegin =
    {
        "<P>";
    }
    bodyFmtEnd =
    {
        "</P>
        </FONT>
        </TD>
        </TR>";
    }
    footerBGCOLOR = { self.headerBGCOLOR; }
    footerFont =
    {
        "<FONT COLOR=YELLOW SIZE=\"-2\" FACE=\"Courier New,Courier New\">";
    }
    footerFmtBegin =
    {   
        "<P>
        <B>
        <CENTER>";
    }
    footerFmtEnd =
    {
        "</CENTER>
        </B>		
        </P>
        </FONT>
        </TD>
        </TR>";
    }
;

/*
 *  door: floatingItem, fixeditem, obstacle
 *
 *  A door is an obstacle that impedes progress when it is closed.
 *  Unlike the doorway, a door is a floatingItem and should not have its
 *  location attribute coded. Instead, use the foundin list to indicate
 *  the rooms in which the door is to be found. 
 *
 *  To make a simple two-sided door you need only code both room
 *  objects in the foundin list of the door: 
 *
 *      foundin = [ hallway, kitchen ]
 *  
 *  To make a one-sided door, code the foundin list with the room
 *  object in which the door is to be found, and the doorto list
 *  with the room object to which the door leads:
 *
 *      foundin = hallway 
 *      doorto = kitchen 
 *
 *  If you want a door that returns rooms different from those it is
 *  found in, simply code both the foundin list and doorto list 
 *  accordingly, each element of the doorto list must have a 
 *  corresponding foundin element:
 *
 *      foundin = [ hallway, kitchen ]
 *      doorto = [ foyer, hallway ]
 *
 *  From the hallway the door leads to the foyer, but from the kitchen
 *  it leads to the hallway.
 */
class door: doorway, floatingItem
    foundin = []
    doorto = nil
    location =
    {
        local f, actor, loc, dloc, foundList;
        
        if (self.foundin == nil)
            return nil;
            
        actor = parserGetObj(PO_ACTOR);
        if (actor == nil)
            actor = parserGetMe();

        loc = actor;
        while(loc)
        {
            dloc = loc;
            loc = loc.location;
        }
        
        //
        //  convert self.foundin into a list
        //
        switch(proptype(self, &foundin))
        {
            case DTY_NIL:
                foundList = [];
                break;
            case DTY_OBJECT:
                foundList = [ self.foundin ];
                break;
            case DTY_LIST:
                foundList = self.foundin;
                break;
            default:
                return nil;
        }
        
        if (find(foundList, dloc))
            return dloc;
        else
            return nil;        
    }
    findDest(actor) = 
    {
        local f, loc, foundList, destList;       
            
        loc = self.location;
        
        //
        //  convert self.foundin into a list
        //
        switch(proptype(self, &foundin))
        {
            case DTY_NIL:
                foundList = [];
                break;
            case DTY_OBJECT:
                foundList = [ self.foundin ];
                break;
            case DTY_LIST:
                foundList = self.foundin;
                break;
            default:
                return nil;
        }
        
        //
        //  convert self.doorto into a list
        //
        switch(proptype(self, &doorto))
        {
            case DTY_NIL:
                destList = [];
                break;
            case DTY_OBJECT:
                destList = [ self.doorto ];
                break;
            case DTY_LIST:
                destList = self.doorto;
                break;
            default:
                return nil;
        }
        
        //
        //  door is linked to room directions, but no foundin
        //  list exists. Simply return the noexit message.
        //
        if (length(foundList) == 0)
        {
            return self.noexit;
        }
          
        //
        //  We don't have a doorto list. Use the foundin list,
        //  which must have 2 elements.
        //
        if (length(destList) == 0)
        {  
            if (length(foundList) != 2)
            {
                "\n[TADS-DOORS-001] When omitting the           
                doorto list, the foundin list should 
                contain 2 elements.\n";
                return nil;
            }
            if (loc == foundList[1])
                return foundList[2];
            else
                return foundList[1];
        }
        
        //
        //  We have a doorto list, each element should correspond to
        //  an element in the foundin list.
        //
        if (length(foundList) != length(destList))
        {
            "\n[TADS-DOORS-002] <<self.thedesc>> requires a
            corresponding doorto element for each foundin 
            element.\n";
            return nil;
        }
        
        //
        //  Search for the actor location in the foundin list.
        //
        f = find(foundList, loc);
        
        //
        //  Door isn't in the actor location. return noexit.
        //
        if (f == nil)
        {
            return self.noexit;
        }
       
        //
        //  return the corresponding doorto element
        //
        return destList[f];
    }
    destination =
    {
        local actor = parserGetObj(PO_ACTOR);
        
        if (actor == nil)
            actor = parserGetMe();
            
        //
        //  The door location is nil or the actor's location doesn't
        //  match any object in the foundin list
        //
        if (self.location == nil)
            return self.noexit;
            
        if (self.isopen)
            return self.findDest(actor);
        else if (!self.islocked
            && !self.noAutoOpen)
        {
            self.setIsopen(true);
            "(Opening << self.thedesc >>)\n";
            return self.findDest(actor);
        }
        else
        {
            "%You%'ll have to open << self.thedesc >> first. ";
            setit(self);
            return nil;
        }
    }    
    setIsopen(setting) =
    {
        /* update my status */
        self.isopen = setting;
    }
    setIslocked(setting) =
    {
        /* update my status */
        self.islocked = setting;
    }
    noexit = { "%You% can't go that way. "; return nil; }
; 

/*
 *  lockableDoor: door
 *
 *  This is just a normal door with the islockable and
 *  islocked properties set to true.  Fill in the other
 *  properties (foundin and doorto) as usual.  If
 *  the door has a key, set property mykey to the key object.
 */
class lockableDoor: door
    islockable = true
    islocked = true
;

/*
 *	sackItem: openable, clothingItem
 *
 *  Defines the sackItem class and modified class thing that
 *  allows for the implementation of a 'rucksack', into which the game 
 *	automatically tidies away things for the player. The sackItem class
 * 	will automatically move items from the player's possession to the 
 *	designated sackItem once the player attempts to take more than the
 *	maximum bulk allocated for them.  
 *
 *	This class allows an object to act like a 'rucksack', into which 
 *  the game automatically tidies away things for the player.
 */
class sackItem: containment, clothingItem
    isopenable = true
	maxbulk = 20
;

////////////////////////////////////////////////////////////////////////
//
//  containment: thing
//
//  The containment class is POLYMORPHIC. It combines the behaviours 
//  of the following classes: surface, qsurface, container, qcontainer, 
//  openable, lockable, keyedLockable, transparentItem, room, darkroom, 
//  nestedroom, vehicle, chairitem, and beditem.
//
////////////////////////////////////////////////////////////////////////
class containment: thing

    ////////////////////////////////////////////////////////////////////
    //
    //  IDENTITY ATTRIBUTES
    //
    ////////////////////////////////////////////////////////////////////

    isenterable = nil
    isvehicle = nil
    issurface = {       // simulates class inheritance
        if (self.isqsurface)
            return true;
        else
            return nil;
    }
    isqsurface = nil
    iscontainer = {     // simulates class inheritance
        if (self.isqcontainer 
        || self.isopenable 
        || self.islockable 
        || self.istransparent)
            return true;
        else 
            return nil;
    }
    isqcontainer = nil
    isopenable = {
        if (self.islockable
        || self.istransparent)
            return true;
        else
            return nil;
    }
    islockable = nil
    istransparent = nil

    ////////////////////////////////////////////////////////////////////
    //
    //  STATE ATTRIBUTES & METHODS
    //
    ////////////////////////////////////////////////////////////////////

    isopen = true  
    islocked = nil 
    islit = 
    {
        local i, o;

        if (self.lightsOn)
            return true;
        
        for (i = 1; i <= length(global.lamplist); ++i)
        {
            o = global.lamplist[i];
            if (o.isIn(self) && o.islit)
                return true;
        }

        if (self.location && self.contentsVisible)
            return self.location.islit;
        return nil;
    }
    contentsVisible = { 
        if(self.istransparent) 
            return true;
        else
            return self.isopen; 
    }
    contentsAudible = { return self.isopen; }
    contentsOlfactory = { return self.isopen; }
    contentsReachable = { return self.isopen; }

    maxbulk = 10            // maximum bulk the container can contain
    isseen = nil            // room has not been seen yet
    mykey = nil     // set 'mykey' to the key which locks/unlocks me

    ////////////////////////////////////////////////////////////////////
    //
    //  LISTING ATTRIBUTES
    //
    ////////////////////////////////////////////////////////////////////

    isListed = true
    isListedAudible = nil
    isListedOlfactory = nil

    isqVisible = nil
    isqAudible = nil
    isqOlfactory = nil
   
    ////////////////////////////////////////////////////////////////////
    //
    //  DESCRIPTION METHODS
    //
    ////////////////////////////////////////////////////////////////////

    statusPrep = { if (self.issurface) "on"; else "in"; }
    outOfPrep = { if (self.issurface) "off of"; else "out of"; }
    ldesc =
    {	    
        if (self.isopenable)
        {
            "\^<<self.thedesc>> <<self.isdesc>>";
            if (self.isopen)
            {
                " open. ";
            }
            else
            {
                " closed. ";
            
                /* if it's transparent, list its contents anyway */
                if (!self.contentsVisible)
                    return; 
            }
        }
        if (self.contentsVisible && itemcnt(self.contents) != 0)
        {
            "\^<<self.statusPrep>> <<self.thedesc>> %you% see%s% "; 
            listcont(self); ". ";
        }
        else
        {
            "There's nothing <<self.statusPrep>> <<self.thedesc>>. ";
        }
    }

    ////////////////////////////////////////////////////////////////////
    //
    //  VEHICLE XOBJGEN METHODS
    //
    ////////////////////////////////////////////////////////////////////

    dobjGen(a, v, i, p) =
    {
        if (!self.isvehicle) return;
        
        if (a.isIn(self) 
        && v != inspectVerb 
        && v != getOutVerb
        && v != getOffVerb
        && v != disembarkVerb
        && v != outVerb)
        {
            "%You%'ll have to get <<self.outOfPrep>> <<
                self.thedesc>> first. ";
            exit;
        }
    }
    iobjGen(a, v, d, p) =
    {
        if (!self.isvehicle) return;
        
        if (a.isIn(self) && v != putVerb)
        {
            "%You%'ll have to get <<self.outOfPrep>> <<
                self.thedesc>> first. ";
            exit;
        }
    }
    
    ////////////////////////////////////////////////////////////////////
    //
    //  SURFACE / CONTAINER DIRECT OBJECT METHODS
    //
    ////////////////////////////////////////////////////////////////////

    verDoBoard(actor) =
    {
        if (!self.isenterable && (self.issurface || self.iscontainer))
        {
            "That's not something you can enter. ";
        }
        else if (actor.location == self)
        {
            "%You're% already <<self.statusPrep>> <<self.thedesc>>! ";
        }
        else if (actor.isCarrying(self))
        {
            "%You%'ll have to drop <<self.thedesc>> first!";
        }
        else if (self.isopenable && !self.isopen)
    	{
        	"%You%'ll have to open <<self.thedesc>> first!";
    	}    	    
    } 	
    doBoard(actor) =
    {
        if (find(actor.clist, self))
        {   
            local ret;
            "(getting <<self.outOfPrep>> <<self.thedesc>>)\n";
            ret = execCommand(actor, disembarkVerb, actor.location, EC_HIDE_SUCCESS);
            if (ret != EC_SUCCESS || actor.location != self)
                exitobj;
        }
        else if (actor.location != self.location)
        {
            local ret;
            "(getting <<self.location.issurface ? "onto" : "into">> 
            <<self.location.thedesc>>)\n";
            ret = execCommand(actor, boardVerb, self.location, EC_HIDE_SUCCESS);
            if (ret != EC_SUCCESS || !actor.isIn(self.location)) 
                exitobj;
        }
        if (addbulk(self.contents) + actor.bulk > self.maxbulk)
        {
            "%You% can't fit <<self.statusPrep>> <<self.thedesc>>. ";
        }
        else
        {
            "Okay, %you're% now <<self.statusPrep>> <<self.thedesc>>.\b";
            actor.travelTo(self);
        }
    }
    verDoClose(actor) =
    {
        if (!self.isopenable)
        {
            "I don't know how to close <<self.thedesc>>. ";
        }
        else if (!self.isopen)
        {
            caps(); self.thedesc; " <<self.isdesc>> already closed! ";
        }
    }
    doClose(actor) =
    {
        "Closed. ";
        self.isopen = nil;
    }
    verDoDrop(actor) =
    {
        if (actor.isIn(self))
        {
            "%You% can't drop <<self.thedesc>>. ";
        }
    }
    verDoGetin(actor) = 
    {
        self.verDoBoard(actor);
        if (!self.iscontainer && !self.isvehicle)
            "That's not something you can get in. ";
    }
    doGetin(actor) = { self.doBoard(actor); }
    doSynonym('Getin') = 'Enter'
    verDoGeton(actor) =
    {
        self.verDoBoard(actor);
        if (!self.issurface && !self.isvehicle)
            "That's not something you can get on. ";
    }
    doGeton(actor) = { self.doBoard(actor); }
    verDoGetoff(actor) =
    {
        self.verDoUnoard(actor);
        if (!self.issurface && !self.isvehicle)
            "That's not something you can get off of. ";
    }
    doGetoff(actor) = { self.doUnboard(actor); }
    verDoGetout(actor) =
    {
        self.verDoUnoard(actor);
        if (!self.iscontainer && !self.isvehicle)
            "That's not something you can get out of. ";
    }
    doGetout(actor) = { self.doUnboard(actor); }
    doSynonym('Getout') = 'Exit'
    doInspect(actor) =
    {
	    if (actor.isIn( self ) 
	    && proptype(self, &insideDesc) != DTY_NIL)
	        self.insideDesc;
	    else
	        self.ldesc;
    }
    verDoLieon(actor) =
    {
        if (!self.issurface)
        {
            "I don't know how to lie <<self.statusPrep>> 
            <<self.thedesc>>. ";
        }
        else if (actor.location == self)
        {
            "%You're% already <<self.statusPrep>> 
            <<self.thedesc>>! ";
        }
    }
    doLieon(actor) =
    {
        "Okay, %you're% now lying <<self.statusPrep>> 
        <<self.thedesc>>.\n";
        actor.travelTo(self);
    }
    verDoLock(actor) =
    {
        if (!self.islockable)
        {
            "I don't know how to lock <<self.thedesc>>. ";
        }
        else if (self.islocked)
        {
            "\^<<self.itisdesc>> already locked! ";
        }
    }
    doLock(actor) =
    {
        if (self.mykey != nil)
        {
            askio(withPrep);
        }
        else if (self.isopen)
        {
            "%You%'ll have to close <<self.thedesc>> first. ";
        }
        else
        {
            "Locked. ";
            self.islocked = true;
        }
    }
    verDoLockWith(actor, io) =
    {
        if (!self.islockable)
        {
            "I don't know how to lock <<self.thedesc>>. ";
        }
        else if (self.islocked)
        {
            "\^<<self.itisdesc>> already locked. ";
        }
    }
    doLockWith(actor, io) =
    {
        if (self.mykey == nil)
        {
            "A key is not required to lock <<self.thedesc>>. ";
        }
        else if (self.isopen)
        {
            "%You% can't lock << self.thedesc >> when <<
            self.itisdesc>> open. ";
        }
        else if (io == self.mykey)
        {
            "Locked. ";
            self.islocked = true;
        }
        else
            "\^<<io.itnomdesc>> <<io.doesdesc>>n't fit the lock. ";
    }
    verDoLookin(actor) =
    {
        if (!self.iscontainer)
            "There's nothing in <<self.thedesc>>. ";
        /* we can look in it if either it's open or it's transparent */
        else if (!self.contentsVisible)
           "\^<<self.itisdesc>> closed. ";
    }
    doLookin(actor) =
    {
        self.doSearch(actor);
    }
    verDoMove(actor) =
    {
        if (actor.isIn(self))
        {
            "%You% can't move <<self.thedesc>>. ";
        }
    }
    verDoOpen(actor) =
    {
        if (!self.isopenable)
        {
            "I don't know how to open <<self.thedesc>>. ";
        }
        else if (self.islocked)
        {
            "\^<<self.itisdesc>> locked. ";
        }
        else if (self.isopen)
        {
            "\^<<self.thedesc>> <<self.isdesc>> already open! ";
        }
    }
    doOpen(actor) =
    {
        if (!self.contentsVisible
        && !actor.isIn(self)
        && itemcnt(self.contents))
        {
            "Opening <<self.thedesc>> reveals <<listcont(self)>>. ";
        }
        else
            "Opened. ";
        self.isopen = true;
    }
    verDoPutIn(actor, io) =
    {
        if (actor.isIn(self))
        {
            "%You% can't put <<self.thedesc>> anywhere. ";
        }
    }
    verDoPutOn(actor, io) =
    {
        if (actor.isIn(self))
        {
            "%You% can't put <<self.thedesc>> anywhere. ";
        }
    }
    verDoSearch(actor) = { self.verDoLookin(actor); }
    doSearch(actor) =
    {
        if (self.contentsVisible && itemcnt(self.contents) != 0)
            "\^<<self.statusPrep>> <<self.thedesc>> %you% see%s% 
            <<listcont(self)>>. ";
        else
            "There's nothing <<self.statusPrep>> <<self.thedesc>>. ";
    }
    verDoSiton(actor) =
    {
        if (!self.issurface)
        {
            "I don't know how to sit <<self.statusPrep>> 
            <<self.thedesc>>. ";
        }
        else if (actor.location == self)
        {
            "%You're% already <<self.statusPrep>> 
            <<self.thedesc>>! ";
        }
    }
    doSiton(actor) =
    {
        "Okay, %you're% now sitting <<self.statusPrep>> 
        <<self.thedesc>>.\n";
        actor.travelTo(self);
    }
    verDoTake(actor) =
    {
        if (actor.isIn(self))
        {
            "%You% can't have <<self.thedesc>>! ";
        }
        else if (actor.isCarrying(self))
        {
        	"%You% already %have% <<self.thedesc>>! ";
        }
    }
    verDoTakeOff(actor, io) =
    {
        self.verDoTake(actor);
    }
    verDoTakeOut(actor, io) =
    {
        self.verDoTake(actor);
    }
    verDoThrowAt(actor, iobj) =
    {
        if (actor.isIn(self))
        {
            "%You% can't throw <<self.thedesc>>.";
        }
    }
    verDoUnboard(actor) =
    {
        if (find(actor.clist, self) == nil)
        {
            "%You're% not <<self.statusPrep>> <<self.thedesc>>! ";
        }
        else if (self.location == nil)
        {
            "%You% can't leave <<self.thedesc>>! ";
        }
    	else if (!actor.location.isopen)
    	{
    		"%You%'ll have to open <<self.thedesc>> first!";
    	}
    }
    doUnboard(actor) =
    {
        if (actor.location != self)
        {
            local ret;
            "(getting <<actor.location.outOfPrep>> <<actor.location.thedesc>>)\n";
            self.leaveRoom(actor);
            actor.moveInto(actor.location.location);
            ret = execCommand(actor, disembarkVerb, 
                actor.location, EC_HIDE_SUCCESS);
            if (ret != EC_SUCCESS || actor.location == self)
                exitobj;
        }
        else
        {
            self.leaveRoom(actor);
            actor.moveInto(actor.location.location);
        }
        "Okay, %you're% no longer <<self.statusPrep>> <<self.thedesc>>. ";
    }
    verDoUnlock(actor) =
    {
        if (!self.islockable)
        {
            "I don't know how to unlock <<self.thedesc>>. ";
        }
        else if (!self.islocked)
        {
            "\^<<self.itisdesc>> not locked! ";
        }
    }
    doUnlock(actor) =
    {
        if (self.mykey != nil)
        {
            askio(withPrep);
        }
        else
        {
            "Unlocked. ";
            self.islocked = nil;
        }
    }
    verDoUnlockWith(actor, io) =
    {
        if (!self.islockable)
        {
            "I don't know how to unlock <<self.thedesc>>. ";
        }
        else if (!self.islocked)
        {
            "\^<<self.itisdesc>> not locked! ";
        }
        else if (self.mykey == nil)
        {
            "%You% %do%n't need anything to unlock << self.itobjdesc >>.";
        }
    }
    doUnlockWith(actor, io) =
    {
        if (self.mykey == nil)
        {
            "A key is not required to unlock <<self.thedesc>>. ";
        }
        else if (io == self.mykey)
        {
            "Unlocked. ";
            self.islocked = nil;
        }
        else
            "\^<<io.itnomdesc>> <<io.doesdesc>>n't fit the lock. ";
    }

    ////////////////////////////////////////////////////////////////////
    //
    //  SURFACE / CONTAINER INDIRECT OBJECT METHODS
    //
    ////////////////////////////////////////////////////////////////////

    verIoPutIn(actor) =
    {
        if (!self.iscontainer)
        {
            "%You% can't put anything into <<self.thedesc>>. ";
        }
        else if (!self.isopen)
        {
            "\^<<self.thedesc>> <<self.isdesc>> closed. ";
        }
    }
    ioPutIn(actor, dobj) =
    {
        if (addbulk(self.contents) + dobj.bulk > self.maxbulk)
        {
            "%You% can't fit <<dobj.thatdesc>> in "; self.thedesc; ". ";
        }
        else
        {
            dobj.doPutIn(actor, self);
        }
    }
    verIoPutOn(actor) = 
    {
        if (!self.issurface)
            "There's no good surface on <<self.thedesc>>. ";
    }
    ioPutOn(actor, dobj) =
    {
        dobj.doPutOn(actor, self);
    }

    ////////////////////////////////////////////////////////////////////
    //
    //  ACCESSIBILITY METHODS
    //
    ////////////////////////////////////////////////////////////////////

#ifdef USE_COMMON_SENSE

    passToContents(plist, filter) = (true)
    passToLocation(plist, filter) = {
        local i, ptr, list;
                
        switch(filter)
        {
	        case &scope_visible:
                ptr = &seeInto;
                break;
            case &scope_audible:
                ptr = &hearInto;
                break;
            case &scope_olfactory:
                ptr = &smellInto;
                break;
            case &scope_reachable:
                ptr = &reachInto;
                break;
            default:
                return nil;
        }
        
        list = self.retrieveLocs(self, ptr);
        
        if (list == true || list == nil)
            return list;
        
        for (i = 1; i <= length(list); ++i)
        {
            if (inPlist(plist, list[i], PTH_TLOC))
                return true;
        }
        return nil;
    }
    retrieveLocs(o, ptr) = {
        local list;
        
        switch(proptype(o, ptr))
        {
            case DTY_OBJECT:
                return [ self.(ptr) ];
                
            case DTY_LIST:
                return (self.(ptr));
                
            case DTY_CODE:
                list = self.(ptr);
                break;

            case DTY_TRUE:
                return true;
                
            case DTY_NIL:
                return nil;
                
            default:
                return nil;
        }
        switch(datatype(list))
        {
            case DTY_OBJECT:
                return [ list ];
                        
            case DTY_LIST:
                return (self.(ptr));
                
            case DTY_TRUE:
                return true;
                
            default:
                return nil;
        }
    }
    
    seeInto = {
        return self.location;
    }
    smellInto = {
        return self.location;
    }
    hearInto = {
        return self.location;
    }
    reachInto = {
        return self.location; 
    }
    
#else /* USE_COMMON_SENSE */

    passToContents(plist, filter) = (nil)
    passToLocation(plist, filter) = (nil)

#endif /* USE_COMMON_SENSE */

    
    cantSee(actor) = {
        local i;
        
        "%You% can't see any ";
        for (i = 1; i <= length(command.cantReachWords); ++i)
        {
            say(command.cantReachWords[i]);
            " ";
        }
        if (self.location && contentsVisible)
            "from <<self.thedesc>>. ";
        else
            "here. ";
    }
    cantHear(actor) =
    {
        local failloc = blocksPath(command.plist, &scope_visible);
        
        if (failloc)
            failloc.cantSee(actor);
        else if (contentsVisible)
            "%You%'ll have to open <<self.thedesc>> first. ";
        else if (self.location && contentsAudible)
            "%You% can't hear that from <<self.thedesc>>. ";
        else
            "%You can't hear that from here. ";
    }    
    cantSmell(actor) =
    {
        local failloc = blocksPath(command.plist, &scope_visible);
        
        if (failloc)
            failloc.cantSee(actor);
        else if (contentsVisible)
            "%You%'ll have to open <<self.thedesc>> first. ";
        else if (self.location && self.contentsOlfactory)
            "%You% can't smell that from <<self.thedesc>>. ";
        else
            "%You% can't smell that from here. ";
    }    
    cantReach(actor) =
    {
        local failloc = blocksPath(command.plist, &scope_visible);
        
        if (failloc)
            failloc.cantSee(actor);
        else if (self.contentsVisible)
            "%You%'ll have to open <<self.thedesc>> first. ";
        else if (self.location && self.contentsReachable)
            "%You% can't reach that from <<self.thedesc>>. ";
        else
            "%You% can't reach that from here. ";
    }
    
    verGrab(obj) =
    {
        if (self.contentsVisible
        && !self.contentsReachable)
            "%You% will have to open << self.thedesc >> first. ";
    }

    ////////////////////////////////////////////////////////////////////
    //
    //  "ROOM & NESTEDROOM" RELATED METHODS
    //
    ////////////////////////////////////////////////////////////////////
        
    /*
     *   roomCheck is true if the verb is valid in the room.  This
     *   is a first pass; generally, its only function is to disallow
     *   certain commands in a dark room.
     */
    roomCheck(v) =
    {
        if (v == openVerb 
        || v == getOutVerb
        || v == getOffVerb
        || v == disembarkVerb)
            return true;
        else if (self.islit || v.isDarkVerb)
            return true;
        else
        {
            "It's pitch black.\n";
            return nil;
        }
    }
    /*
     *   If canReachContents is nil, enforce the limitation 
     */
    roomAction(actor, v, dobj, prep, io) =
    {
        if (v == openVerb && find(actor.clist, dobj))
            return;
        if (v.isTravelVerb 
        || v == getOutVerb 
        || v == getOffVerb 
        || v == disembarkVerb)
            return;
            
        if (!self.islit && !v.isDarkVerb)
        {
            "%You% can't see a thing. ";
            exit;
        }
        else if (self.location && self.contentsVisible)
        {                
            self.location.roomAction(actor, v, dobj, prep, io);
        }
    }

    /*
     *   Whenever a normal object (i.e., one that does not override the
     *   default doDrop provided by 'thing') is dropped, the actor's
     *   location is sent roomDrop(object being dropped).  By default, 
     *   we move the object into this room.
     */
    roomDrop(obj) =
    {
        "Dropped. ";
        if (self.location && self.isopen && !self.isdroploc)
            obj.moveInto(self.location);
        else
            obj.moveInto(self);
    }
    enterRoom(actor) =      // sent to room as actor is entering it
    {
        if (self.issurface || self.iscontainer || self.isvehicle)
            return;

        if (self.islit)
        {
            if (! self.isseen)
            {
                self.initial;
                "\b";
            }
        }            
        self.senseAround(actor, (!self.isseen) || global.verbose);
        if (self.islit)
        {
            if (!self.isseen)
                self.firstseen;
            self.isseen = true;
        }
    }

    leaveRoom(actor) = {}

    /*
     *   dispParagraph - display the paragraph separator.  By default we
     *   display a \n\t sequence; games can change this (via 'modify') to
     *   customize the display format 
     */
    dispParagraph = "\n\t"

    /*
     *   dispBeginSdesc and dispEndSdesc - display begin and end of
     *   location short description sequence.  We'll call these just
     *   before and after showing a room's description.  By default these
     *   do nothing; games that want to customize the display format can
     *   change these (via 'modify').  
     */
    dispBeginSdesc = ""
    dispEndSdesc = ""

    /*
     *   dispBeginLdesc and dispEndLdesc - display begin and end of
     *   location long description sequence.  We'll call these just before
     *   and after showing a room's long description.  By default, we
     *   start a new paragraph before the ldesc; games that want to
     *   customize the display format can change these (via 'modify').  
     */
    dispBeginLdesc = { self.dispParagraph; }
    dispEndLdesc = ""

    senseAround(actor, verbosity) = {
        self.dispBeginSdesc;
        self.statusRoot;
        self.dispEndSdesc;
        
        self.lookAround(actor, verbosity, nil);
        self.listenAround(actor, verbosity, nil);
        self.smellAround(actor, verbosity, nil);
    }
    lookAround(actor, verbosity, sroot) =
    {
        if (self.islit)
        {
            if (sroot)
            {
                self.dispBeginSdesc;
                self.statusRoot;
                self.dispEndSdesc;
            }
            if (self.location && self.contentsVisible)
            {
                if (self.isopen)
                    self.location.nrmLkAround(actor, verbosity);
                else
                    self.location.xtndLkAround(actor, verbosity);
                if (self.isopen)
                    self.xtndLkAround(actor, verbosity);
                else
                    self.nrmLkAround(actor, verbosity);
            }
            else
            {
                self.nrmLkAround(actor, verbosity);
            }
        }
        else
        {
            self.dispParagraph;
            "It's pitch black. ";
        }
    }

    /*
     *   lookAround describes the room.  If verbosity is true, the full
     *   description is given, otherwise an abbreviated description (without
     *   the room's ldesc) is displayed.
     */
    nrmLkAround(actor, verbosity) =        // lookAround without location status
    {
        local i, o, len, ret, alsoMess, locList = [];
        local lookList  = self.contents;

#ifdef USE_COMMON_SENSE
        
        locList = self.retrieveLocs(self, &seeInto);
        if (locList == true)
            locList = global.tlist - self;
        else if (locList == nil)
            locList = [];

#else /* USE_COMMON_SENSE */

        locList = global.tlist - self;

#endif /* USE_COMMON_SENSE */

        locList -= actor.clist;
        
        ////////////////////////////////////////////////////////////////
        //
        //  Add floating items to the contents list
        //
        ////////////////////////////////////////////////////////////////
        
        for (i = 1; i <= length(global.floatingList); ++i)
        {
            o = global.floatingList[i];
            if (o.location == self
            && find(self.contents, o) == nil)
                lookList += o;
        }
        
        ////////////////////////////////////////////////////////////////
        //
        // Remove the actor from the contents list
        //
        ////////////////////////////////////////////////////////////////

        lookList -= [ actor, actor.location ];
                
        if (proptype(self, &describe) != DTY_NIL)
        {
            self.dispBeginLdesc;
            self.describe;
            self.dispEndLdesc;
        }
        else if (verbosity)
        {
            if (proptype(self, &insideDesc) != DTY_NIL)
            {
                self.dispBeginLdesc;
                self.insideDesc;
                self.dispEndLdesc;
            }
            if (self.location == nil)
            {
                self.dispBeginLdesc;
                self.ldesc;
                self.dispEndLdesc;
            }
        }
        
        ////////////////////////////////////////////////////////////////
        //
        //  The describe takes precedence over the others.
        //
        ////////////////////////////////////////////////////////////////
        
        len = length(lookList);
        for (i = len; i >= 1; --i)
        {
            o = lookList[i];
            if (proptype(o, &describe) != DTY_NIL)
            {
                local ret;
                ret = o.describe;
                if (ret)
                {
                    lookList -= o;
                    continue;
                }
            }
            
            if (o.hasmoved)
            {
                if (o.iscontainer
                && !o.isopen
                && proptype(o, &whenClosed) != DTY_NIL)
                {
                    self.dispBeginLdesc;
                    o.whenClosed;
                    self.dispEndLdesc;
                    lookList -= o;
                }
                else if (o.isswitchitem 
                && !o.isActive
                && proptype(o, &whenOff) != DTY_NIL)
                {
                    self.dispBeginLdesc;
                    o.whenOff;
                    self.dispEndLdesc;
                    lookList -= o;
                }
                else if (o.isobstacle 
                && !o.isopen
                && proptype(o, &whenClosed) != DTY_NIL)
                {
                    self.dispBeginLdesc;
                    o.whenClosed;
                    self.dispEndLdesc;
                    lookList -= o;
                }
            }
            else
            {
                if (o.iscontainer 
                && o.isopen
                && proptype(o, &whenOpen) != DTY_NIL)
                {
                    self.dispBeginLdesc;
                    o.whenOpen;
                    self.dispEndLdesc;
                    lookList -= o;
                }
                else if (o.iscontainer 
                && !o.isopen
                && proptype(o, &whenClosed) != DTY_NIL)
                {
                    self.dispBeginLdesc;
                    o.whenClosed;
                    self.dispEndLdesc;
                    lookList -= o;
                }
                else if (o.isswitchitem 
                && o.isActive
                && proptype(o, &whenOn) != DTY_NIL)
                {
                    self.dispBeginLdesc;
                    o.whenOn;
                    self.dispEndLdesc;
                    lookList -= o;
                }
                else if (o.isswitchitem 
                && !o.isActive
                && proptype(o, &whenOff) != DTY_NIL)
                {
                    self.dispBeginLdesc;
                    o.whenOff;
                    self.dispEndLdesc;
                    lookList -= o;
                }
                else if (o.isobstacle 
                && o.isopen
                && proptype(o, &whenOpen) != DTY_NIL)
                {
                    self.dispBeginLdesc;
                    o.whenOpen;
                    self.dispEndLdesc;
                    lookList -= o;
                }
                else if (o.isobstacle 
                && !o.isopen
                && proptype(o, &whenClosed) != DTY_NIL)
                {
                    self.dispBeginLdesc;
                    o.whenClosed;
                    self.dispEndLdesc;
                    lookList -= o;
                }
                else if (proptype(o, &initial) != DTY_NIL)
                {
                    self.dispBeginLdesc;
                    o.initial;
                    self.dispEndLdesc;
                    lookList -= o;                    
                }
            }
        }      
        
        for (i = 1; i <= length(lookList); ++i)
        {
            o = lookList[i];
            if (o.isactor)
            {
                self.dispParagraph;
                o.actorDesc;
            }
        }

        alsoMess = len - length(lookList);
                  
        if (itemcnt(lookList))
        {
            self.dispParagraph;
            if (alsoMess)
                "%You% can also see ";
            else
                "%You% see%s% "; 
            listcont(lookList); " here. ";
        }
        listcontcont(lookList); "\n";

        "\n";
           
        //
        //  Display xtndLkAround() for each location which forms a valid 
        //
        for (i = 1; i <= length(locList); ++i)
        {
            command.plist = path(actor, locList[i]);
            if (blocksPath(command.plist, &scope_visible) == nil)
                locList[i].xtndLkAround(actor, verbosity);
        }
    }
    /*
     *   xtndLkAround describes the room.  If verbosity is true, the full
     *   description is given, otherwise an abbreviated description (without
     *   the room's ldesc) is displayed.
     */
    xtndLkAround(actor, verbosity) = {
        local i;
        local lookList = self.contents + parserGetMe() - actor - actor.location;
        
        if (verbosity 
        && self.isListed)
        {
            if (proptype(self, &insideDesc) != DTY_NIL)
            {
                self.dispBeginLdesc;
                self.insideDesc;
                self.dispEndLdesc;
            }
            else if (self.location == nil)
            {
                self.dispBeginLdesc;
                self.ldesc;
                self.dispEndLdesc;
            }
        }
        
        if (self.isqVisible) return;
        
        if (verbosity)
        {
            for (i = 1; i <= length(self.contents); ++i)
            {
                if (self.contents[i].isfixed)
                    self.contents[i].heredesc;
            }
        }
        
        if (itemcnt(lookList))
        {
            self.dispParagraph;        
            "\^<<self.statusPrep>> <<self.thedesc>> %you% can see "; 
            listcont(lookList); ". ";
        }
        listcontcont(lookList); "\n";

        //
        //  list all ACTORS in the room except the actor
        //
        for (i = 1; i <= length(lookList); ++i)
        {
            if (lookList[i].isactor)
            {   
                self.dispParagraph;
                lookList[i].actorDesc;
            }
        }

        "\n";
    }
    listenAround(actor, verbosity, sroot) = {        
        if (sroot)
        {
            self.dispBeginSdesc;
            self.statusRoot;
            self.dispEndSdesc;
        }        
        self.nrmLnAround(actor, verbosity);   
    }
    /*
     *  nrmLnAround describes the room.  
     *  If verbosity is true, the full description is given, 
     *  otherwise an abbreviated description (without
     *  the room's ldesc) is displayed.
     */
    nrmLnAround(actor, verbosity) = {
        local i, locList = [], disptot = 0, ret;
        local verb          = command.verbPtr;

#ifdef USE_COMMON_SENSE

        locList = self.retrieveLocs(self, &hearInto);
        if (locList == true)
            locList = global.tlist - self;
        else if (locList == nil)
            locList = [];

#else /* USE_COMMON_SENSE */

        locList       = global.tlist - self;

#endif /* USE_COMMON_SENSE */
        
        if (verbose 
        && self.isListedAudible) 
        {
            self.dispBeginLdesc;
            self.listendesc;
            self.dispEndLdesc;
            ++disptot;
        }      

        if (self.isqAudible) return;

        self.dispParagraph;

#ifdef USE_COMMON_SENSE
        
        disptot += self.dispcont(actor, &scope_audible, 
            &isListedAudible, &proxDoListen);

#else /* USE_COMMON_SENSE */

        disptot += self.dispcont(actor, &scope_audible, 
            &isListedAudible, &listendesc);
            
#endif /* USE_COMMON_SENSE */
            
        "\n";
       
        for (i = 1; i <= length(locList); ++i)
        {
            command.plist = path(actor, locList[i]);
            if (blocksPath(command.plist, &scope_audible) == nil)
                disptot += locList[i].xtndLnAround(actor, verbosity);
        }
        
        if (disptot == 0
        && verb == listenVerb)
        {
            self.dispBeginLdesc;
            self.listendesc;
            self.dispEndLdesc;
        }
    }
    /*
     *  xtndLnAround describes the room.  
     *  If verbosity is true, the full description is given, 
     *  otherwise an abbreviated description (without
     *  the room's ldesc) is displayed.
     */
    xtndLnAround(actor, verbosity) = {
        local disptot = 0;

        if (verbosity 
        && self.isListedAudible)
        {
            self.dispBeginLdesc;
            self.listendesc;
            self.dispEndLdesc;
            ++disptot;
        }
        
        if (self.isqAudible) return disptot;
        
        self.dispParagraph;

#ifdef USE_COMMON_SENSE
        
        disptot += self.dispcont(actor, &scope_audible,
            &isListedAudible, &proxDoListen);

#else /* USE_COMMON_SENSE */

        disptot += self.dispcont(actor, &scope_audible,
            &isListedAudible, &listendesc);
            
#endif /* USE_COMMON_SENSE */           
            
        "\n";
        return disptot;
    }
    smellAround(actor, verbosity, sroot) = {        
        if (sroot)
        {
            self.dispBeginSdesc;
            self.statusRoot;
            self.dispEndSdesc;
        }        
        self.nrmSmAround(actor, verbosity);   
    }
    /*
     *   nrmSmAround describes the room.  If verbosity is true, the full
     *   description is given, otherwise an abbreviated description (without
     *   the room's ldesc) is displayed.
     */
    nrmSmAround(actor, verbosity) = {
        local i, locList = [], disptot = 0, ret;
        local verb      = command.verbPtr;

#ifdef USE_COMMON_SENSE

        locList = self.retrieveLocs(self, &smellInto);
        if (locList == true)
            locList = global.tlist - self;
        else if (locList == nil)
            locList = [];

#else /* USE_COMMON_SENSE */

        locList   = global.tlist - self;

#endif /* USE_COMMON_SENSE */
        
        if (verbose 
        && self.isListedOlfactory)
        {
            self.dispBeginLdesc;
            self.smelldesc;
            self.dispEndLdesc;
            ++disptot;
        }      

        if (self.isqOlfactory) return;

        self.dispParagraph;
        
#ifdef USE_COMMON_SENSE

        disptot += self.dispcont(actor, &scope_olfactory,
            &isListedOlfactory, &proxDoSmell);

#else /* USE_COMMON_SENSE */

        disptot += self.dispcont(actor, &scope_olfactory,
            &isListedOlfactory, &smelldesc);            
            
#endif /* USE_COMMON_SENSE */
            
        "\n";
       
        for (i = 1; i <= length(locList); ++i)
        {
            command.plist = path(actor, locList[i]);
            if (blocksPath(command.plist, &scope_olfactory) == nil)
                disptot += locList[i].xtndSmAround(actor, verbosity);
        }
        
        if (disptot == 0
        && verb == smellVerb)
        {
            self.dispBeginLdesc;
            self.smelldesc;
            self.dispEndLdesc;
        }
    }
    /*
     *  xtndSmAround describes objects outside of the room.  
     *  If verbosity is true, the full description is given, 
     *  otherwise an abbreviated description (without
     *  the room's ldesc) is displayed.
     */
    xtndSmAround(actor, verbosity) = {
        local disptot = 0;

        if (verbosity 
        && self.isListedOlfactory)
        {
            self.dispBeginLdesc;
            self.smelldesc;
            self.dispEndLdesc;
            ++disptot;
        }
        
        if (self.isqOlfactory) return disptot;
        
        self.dispParagraph;
        
#ifdef USE_COMMON_SENSE

        disptot += self.dispcont(actor, &scope_olfactory,
            &isListedOlfactory, &proxDoSmell);

#else /* USE_COMMON_SENSE */

        disptot += self.dispcont(actor, &scope_olfactory,
            &isListedOlfactory, &smelldesc);

#endif /* USE_COMMON_SENSE */
            
        "\n";
        return disptot;
    }
    
#ifdef USE_HTML_STATUS

    /* 
     *   Generate HTML formatting that provides for the TimeSys status 
     *   line appearance and allows us to toggle between a time-oriented
     *   display and the traditional status line appearance, using the 
     *   traditional status line computation mechanism.
     */
    statusLine =
    {
        /*
         *  Check the system to see if this is an HTML-enabled 
         *  run-time. If so, generate an HTML-style banner; otherwise, 
         *  generate an old-style banner. 
         */
        if (systemInfo(__SYSINFO_SYSINFO) == true
            && systemInfo(__SYSINFO_HTML) == 1)
        {
            self.htmlStatusLine;
        }
        else
        {
            /*
             *  Centering the time display around column 40. This
             *  won't be perfect, but it's close enough to the 
             *  center for a comfortable display.
             */
            local i, stat, sr, tsr, len;
            stat = outcapture(true);
            self.statusRoot;
            sr = outcapture(stat);
            stat = outcapture(true);
            self.timesysStatusRoot;
            tsr = outcapture(stat);
            if (tsr)
            {
                len = 40 - length(sr) - length(tsr)/2;
            }
            say(sr);
            for (i = 1; i <= len; ++i)
                "\ ";
            say(tsr);
            self.dispParagraph;
        }
    }
    htmlStatusLine =
    {
 		"<banner id=StatusLine height=previous border>
		<body bgcolor=statusbg text=statustext><b>";

		self.statusRoot;

		"</b>";
		
		if ( global.timeStatus == nil || global.timeStatus == 1 )
		{
			"<tab align=center><i>";	
			
            self.timesysStatusRoot;
            
			"</i>";
		}

		"<tab align=right><i>";
		
		if ( global.timeStatus == nil || global.timeStatus == 1 )
		{
			"Score: ";
			say( global.score );
		}
		else
		{
			say( global.score ); "/";
			say( global.turnsofar ); 
		}

		"</i></banner>";
    }
    
#else /* USE_HTML_STATUS */

    /* use the standard HTML status line system */
    statusLine =
    {
        self.statusRoot;
        self.dispParagraph;
    }

#endif /* USE_HTML_STATUS */

    timesysStatusRoot =
    {
		if ( global.timeStatus == nil )
		{
			say( gettimesys( TS_DAY, timesys.day, nil ) ); " ";
			say( gettimesys( TS_TIME, timesys.time, nil ) );
		}
		else if ( global.timeStatus == 1 )
		{
			say( gettimesys( TS_DAY, timesys.day, nil ) ); ", ";
			say( gettimesys( TS_DATE, timesys.date, nil ) ); " ";
			say( gettimesys( TS_TIME, timesys.time, nil ) );
		}        
    }

    statusRoot =
    {
        caps();
        if (!self.islit)
            "In the dark.";
        else if (self.location && self.contentsVisible)
            "<<self.location.sdesc>>, <<self.statusPrep>> <<self.thedesc>>";  
        else
            self.sdesc;
        if (global.srDisplayPlayer)
            " (as <<parserGetMe().thedesc>>)";
    }
    
    /*
     *   Direction handlers.  The directions are all set up to
     *   the default, which is no travel allowed.  To make travel
     *   possible in a direction, just assign a room to the direction
     *   property.
     */
    north = { return self.noexit; }
    south = { return self.noexit; }
    east  = { return self.noexit; }
    west  = { return self.noexit; }
    up    = { return self.noexit; }
    down  = { return self.noexit; }
    ne    = { return self.noexit; }
    nw    = { return self.noexit; }
    se    = { return self.noexit; }
    sw    = { return self.noexit; }
    in    = { return self.noexit; }
    out   = { 
        if (self.isvehicle)
            return self.location;
        else
            return self.noexit;
    }
    
    /*
     *   noexit displays a message when the player attempts to travel
     *   in a direction in which travel is not possible.
     */
    noexit =
    {
        if (self.islit)
        {
            if (self.issurface || self.iscontainer)
                "%You're% not going anywhere until %you%
                get%s% <<outOfPrep>> <<thedesc>>. ";
            else
                "%You% can't go that way. "; 
        }
        else
        {
            darkTravel();
        }
    }
;

/*
 *	class dvehicle: vehicle
 *
 *  Defines the dvehicle (directed-vehicle) class which allows for 
 *	the implentation of vehicles responsive to travelverb commands, 
 *  such as 'north' or 'up' or 'in'.
 *
 *	Unlike the TADS normal vehicle class, which responds with 'You 
 *  must get out of the vehicle first.', the directed vehicle will 
 *  attempt to move in the direction indicated.
 *
 *	This vehicle responds to travelVerb commands, such as 'north', 
 *  'south', 'up', 'out'. The vehicle will first check to see that it 
 *  is active. If not active then the inactiveMessage is printed. 
 *
 *	Next it checks to see if the direction indicated is a valid room 
 *  direction, indicated by the return of an object in the room's 
 *  direction property. The object is then checked to determine 
 *  whether it is a passable obstacle and if so, whether the obstacle 
 *  returns an object. If it does not then the cantgoMessage is 
 *  displayed.
 *
 *	Finally a check is made to determine if the direction is valid for 
 *  the vehicle. Authors may wish to customize this code to limit the 
 *  vehicle's movement to specific classes of rooms, lit rooms, or 
 *  directions. If the direction is not valid for the vehicle then the 
 *  cantgoMessage is displayed.
 *
 *	If these checks are passed then the vehicle is moved to its 
 *  travelloc and the goMessage is displayed. If parserGetMe() is in 
 *  the vehicle then we note the new room location.
 */
class dvehicle: fixeditem, containment
	isdvehicle = true
	iscontainer = true
	isenterable = true
	isActive = nil
	traveldir = nil
	travelloc = nil
	/*
	 *	Only travelVerbs are handled. We do all our travel validation 
	 *  from here.
	 */
	roomAction( a, v, d, p, i ) =
	{
		if ( isclass( v, travelVerb ) )
		{
			if ( self.isActive )
			{
				if ( ! ( self.validRoomDir( v ) 
				&& self.validVehDir( v, self.travelloc ) ) )
				{
					self.cantgoMessage( self.traveldir, self.travelloc );
					exit;
				}
			}
			else
			{
				self.inactiveMessage;
				exit;
			}
		}
	}
	/*
	 *	This method checks whether the room the vehicle is in allows 
	 *  movement in this direction. If the proptype of the room 
	 *  direction is an object we then check to see whether it is an 
	 *  obstacle and whether the obstacle will let us pass. If a 
	 *  valid room direction the method returns true; otherwise it 
	 *  returns nil.
	 */
	validRoomDir( v ) =
	{
		switch( v )
		{
			case nVerb:
				self.traveldir = 'north';
				if ( proptype( self.location, &north ) == 2 )
				{
					return self.checkPassable( self.location.north );
				}
				break;
			case sVerb:
				self.traveldir = 'south';
				if ( proptype( self.location, &south ) == 2 )
				{
					return self.checkPassable( self.location.south );
				}
				break;
			case eVerb:
				self.traveldir = 'east';
				if ( proptype( self.location, &east ) == 2 )
				{
					return self.checkPassable( self.location.east );
				}
				break;
			case wVerb:
				self.traveldir = 'west';
				if ( proptype( self.location, &west ) == 2 )
				{
					return self.checkPassable( self.location.west );
				}
				break;
			case neVerb:
				self.traveldir = 'northeast';
				if ( proptype( self.location, &ne ) == 2 )
				{
					return self.checkPassable( self.location.ne );
				}
				break;
			case nwVerb:
				self.traveldir = 'northwest';
				if ( proptype( self.location, &nw ) == 2 )
				{
					return self.checkPassable( self.location.nw );
				}
				break;
			case seVerb:
				self.traveldir = 'southeast';
				if ( proptype( self.location, &se ) == 2 )
				{
					return self.checkPassable( self.location.se );
				}
				break;
			case swVerb:
				self.traveldir = 'southwest';
				if ( proptype( self.location, &sw ) == 2 )
				{
					return self.checkPassable( self.location.sw );
				}
				break;
			case uVerb:
				self.traveldir = 'up';
				if ( proptype( self.location, &up ) == 2 )
				{
					return self.checkPassable( self.location.up );
				}
				break;
			case dVerb:
				self.traveldir = 'down';
				if ( proptype( self.location, &down ) == 2 )
				{
					return self.checkPassable( self.location.down );
				}
				break;
			case inVerb:
				self.traveldir = 'in';
				if ( proptype( self.location, &in ) == 2 )
				{
					return self.checkPassable( self.location.in );
				}
				break;
			case outVerb:
				self.traveldir = 'out';
				if ( proptype( self.location, &out ) == 2 )
				{
					return self.checkPassable( self.location.out );
				}
				break;
		}
		return nil;
	}
	/*
	 *	This method checks to see if the object pointed to by the room 
	 *  direction is an obstacle. In this case we really want the 
	 *  doordest, not the door. We also check to see if the 'door' is 
	 *  open, as most vehicles aren't driven through closed doors. 
	 *  This method returns true if there is no obstacle, or if the 
	 *  obstacle is open and returns an object as its doordest; 
	 *  otherwise it returns nil.
	 */
	checkPassable( o ) =
	{
		if ( isclass( o, obstacle ) )
		{	
			if ( o.isopen && proptype( o, &doordest ) == 2 )
			{
				self.travelloc = o.doordest;
				return true;
			}
			else
			{
				self.travelloc = nil;
				return nil;
			}
		}
		else
		{
			self.travelloc = o;
			return true;
		}
	}
	/*
	 *	This method is used to limit the movement of the vehicle 
	 *  itself. It is passed the travelVerb and the travelloc. By 
	 *  default it returns true. The author should override this 
	 *  method to limit vehicle movement. For instance, the author 
	 *  may want to limit a car so that it may not move up or down. 
	 *  If the direction is allowed then the method should return 
	 *  true; otherwise it should return nil.
	 */
	validVehDir( v, l ) = { return true; }
	inactiveMessage =
	{
		caps(); "<<self.thedesc>> is not active. ";
	}
	goMessage( d, l ) = 
	{
		caps(); "<<self.thedesc>> heads "; say( d ); ". ";
	}
	cantgoMessage( d, l ) =
	{
		caps(); "<<self.thedesc>> can't go "; say( d ); ". ";
	}
	/* 
	 *	This method displays the goMessage, and moves the vehicle to 
	 *  the new location. It returns nil, because the movement is 
	 *  handled here and not in the direction properties, which 
	 *  expect either a location or nil.
	 */
	moveVehicle =
	{
		self.goMessage( self.traveldir, self.travelloc ); "\b";
		self.moveInto( self.travelloc );
		if ( parserGetMe().isIn( self ) )
		{
			parserGetMe().location.lookAround(parserGetMe(), true, true );
			parserGetMe().location.isseen = true;
		}
		return nil;
	}
    north = { self.moveVehicle; }
    south = { self.moveVehicle; }
    east = 	{ self.moveVehicle; }
    west = 	{ self.moveVehicle; }
    ne = 	{ self.moveVehicle; }
    nw = 	{ self.moveVehicle; }
    se = 	{ self.moveVehicle; }
    sw = 	{ self.moveVehicle; }
    up = 	{ self.moveVehicle; }
    down = 	{ self.moveVehicle; }
    in = 	{ self.moveVehicle; }
    out = 	{ self.moveVehicle; }
;

/*
 *	driver: actor
 *
 *	The Cab Concept: Cabs consist of two distinct objects, a driver 
 *  and a vehicle, which are used to convey actors from one location 
 *  to another through topical references to story locations.
 *
 *	This differentiates them from other vehicles, such as   
 *	trackVehicles, which run a pre-arranged circuit or randomized
 *	route, and dVehicles (directed-vehicles, see dVehicle.zip on
 *	ftp.gmd.de/if-archive/programming/tads/examples), which rely
 *	on travelVerb commands (i.e. 'north', 'up', 'in').
 *
 *	The distinguishing feature is the use of parseUnknownVerb(),
 *	"driver,<location>" and "<location>", where there is no verb    
 *	present in the command.
 *
 *	ACTOR GRAMMAR is restricted in driver to the evaluation and     
 *	processing of a single valid travel destination noun phrase. 
 *
 *	The mechanism used to determine travel destination may vary, from
 *	simple travelDest objects, which act as pointers to locations, or
 *	more complex schemes that take into consideration location 
 *	connectivity, relative travel distance/time, etc. Cabs.t does not 
 *	involve itself with this aspect of the mechanism, but provides the
 *	validDest() method for the author to customize. This method is 
 *	provided with a list of valid objects or nil as provided by 
 *	parseNounList().
 */
class driver: Actor
	/*
	 *	grammar( tokenlist, typelist, errnum )
	 *
	 *	Method executes parseNP, passing all its arguments. If a non-	 
	 *	nil value is returned from parseNP then it returns true to 	 	 
	 *	parseUnknownVerb() to indicate that it has handled the error 	 
	 *	itself; otherwise it will pass to the inherited grammar.
	 */
	grammar( tokenlist, typelist, errnum ) =
	{
		local ret = self.parseNP( tokenlist, typelist, errnum );
		if ( ret ) return true;
		pass grammar;
	}
	/*
	 *	parseNP( tokenlist, typelist, errnum )
	 *
	 *	Method handles the processing of the unknown verb from 	 	 
	 *	tokenlist to destination. 
	 *
	 *	Method is passed the tokenlist, typelist, and errnum. If 	 	 
	 *	successful it will return true to indicate that actor grammar 	 
	 *	has handled the unknown verb, either by moving the cab to its 	 
	 *	new destination or through an appropriate error message.
	 */
	parseNP( tokenlist, typelist, errnum ) =
	{
		local ret, nplist, objlist, dest;
		
		ret = self.validPassenger;
		
		if ( ! ret ) return self.errorValidPassenger;
	
		nplist = parseNounList(tokenlist, typelist, 1, nil, true, 			nil);
                           
   		if ( nplist == nil ) return self.errorNPsyntax;
   		
   		if ( length( nplist ) == 1 ) 
   			return self.errorNPnomatch( tokenlist );

   		objlist = self.parseNplist( nplist );
   		
   		if ( length( objlist ) == 0 ) 
   			return self.errorParseNplist( tokenlist );	
		
		dest = self.parseObjlist( objlist );

		if ( dest == nil )
		{
			dest = self.matchVocab( tokenlist, objlist );

			if ( length(dest) != 1 ) 
				return self.errorParseObjlist( dest );
			else
				dest = car( dest );
		}
		
		dest = self.parseDest( dest );
	
		if ( dest == nil ) return self.errorParseDest;

		self.goDest( dest );
		
		return true;
	}
	/*
	 *	errorNPsyntax
	 *
	 *	Method displays an appropriate message when parseNounList()
	 *	has returned nil. This indicates that the noun phrase had a
	 *	syntax error.
	 *
	 *	Method returns nil, indicating that we wish the parser to 
 	 *	handle the error.
	 */
	errorNPsyntax = { return nil; }
 	/* 
	 *	errorNPnomatch( tokenlist )
	 *
	 *	Method displays an appropriate message when parseNounList()
	 *	has returned a list consisting solely of a number. This 
	 *	value indicates that parseNounList() has parsed a valid
 	 *	noun phrase, but no matching objects exist. 
	 *
	 *	This method is passed the parseUnknownVerb() tokenlist. It
	 *	returns true to indicate that actor.grammar has handled the
	 *	error itself.	 
 	 */
	errorNPnomatch( tokenlist ) = 
	{
		caps(); "<<self.thedesc>> says, <q>I've never heard of
		\'<<self.displayToklist( tokenlist )>>\'.</q> ";
		return true;
	}
	/*
	 *	validPassenger
	 *	
	 *	Method used to determine whether the player is in the cab when 	 
	 *	an actor grammar order is being given. This method is useful 	 
	 *	for cases such as ">driver, trafalgar square" where the 	 	 
	 *	command is explicitly being directed toward the driver.
	 *
	 *	For cases of command redirection, such as ">trafalgar square" 	 
	 *	where the actor grammar is redirected from the parserGetMe() 	 
	 *	the modification of the Me.grammar does a similar check.
	 *
	 *	Method returns true if the driver location is the same as the 	 
	 *	player location; otherwise it returns nil.
	 */
	validPassenger = 
	{ 
		if ( self.location == parserGetMe().location ) return true; 
	}
	/*
	 *	errorValidPassenger
	 *
	 *	Method displays an appropriate message when an actor grammar 	 
	 *	command fails the validPassenger test, indicating that the 	 	 
	 *	player is not in the cab with the driver.
	 *
	 *	Method receives no pararmeters.
	 */
	errorValidPassenger =
	{
		"Going somewhere guv'nor? 'ave to get in
		<<self.location.thedesc>> first. ";
		return true;
	}
 	/*
 	 *	parseNplist( nplist )
	 *
	 *	Method builds an object list from the noun-phrase sublists
	 *	passed by parseNounList(). Only objects with '0' flags are
	 *	added to the list, avoiding 'all except foo' lists.
 	 */
	parseNplist( nplist ) =
	{
		local i, objlist = [];
	
      	for (i = 2 ; i <= length(nplist) ; ++i)
      	{
        		local j, sub;

        		/* get the current sublist */
        		sub = nplist[i];

        		/* 
        		 *  scan the objects in the list - each object takes 
        		 *  two elements 
        		 */
        		for (j = 3 ; j <= length(sub) ; j += 2)
        		{
        			if (sub[j] && sub[j+1] == 0)
        			{
          				/* build the object list */
					    objlist += sub[j];
        			}
        		}
      	}
        return objlist;
	}
	/*
	 *	errorParseNplist( tokenlist )
	 *
	 *	Method displays an appropriate message when parseNplist()
	 *	method returns a nil list. This will be the case for 
	 *	noun phrases such as 'all except foo'.
	 *
	 *	Method returns true, indicating that actor.grammar has
	 *	handled the error.
	 */
	errorParseNplist( tokenlist ) = 
	{
		caps(); "<<self.thedesc>> says, <q>I don't know the word";
		if ( length( tokenlist ) > 1 ) "s";
		": \'<<self.displayToklist( tokenlist )>>\'.</q>";
		return true;
	}
	parseObjlist( objlist ) =
	{		
		if ( objlist && length( objlist ) == 1 )
			return car(objlist);
		else
			return nil;
	}
	/*
	 *	matchVocab( tokenlist, objlist )
	 *
	 *	Method is used only in cases where our object list has more 	 
	 *	than one object. This method uses the built-in getwords() 	 	 
	 *	function to match the tokenlist to each object's vocabulary. 	 
	 *	We do this simply by comparing the length of the tokenlist to 	 
	 *	the length of the vocabulary list. 
	 *
	 *	Two lists are created. If an object's vocabulary list is the 	 
	 *	same size as the tokenlist then it is added to matchlist; 	 	 
	 *	otherwise it is added to othlist. The method will return 	 	 
	 *	matchlist if it is not nil; otherwise it returns othlist.
	 *
	 *	Method takes tokenlist and objlist as parameters.
	 */
	matchVocab( tokenlist, objlist ) =
	{
		local i, matchlist = [], othlist = [];

		for ( i = 1; i <= length( objlist ); i++ )
		{
			local wlist = getwords( objlist[i], &adjective )
					+ getwords( objlist[i], &noun );

			if ( length( wlist ) == length( tokenlist ) )
				matchlist += objlist[i];
			else
				othlist += objlist[i];	
		}
		if ( length( matchlist ) )
			return matchlist;
		else
			return othlist;	
	}
	/*
	 *	errorParseObjlist( objlist )
	 *
	 *	Method displays an appropriate message when both 	 	 
	 *	parseObjlist() and matchVocab() have failed to resolve the 	 	 
	 *	objlist into a single object. This would be the case when the 	 
	 *	command contains tokens identifying more than one object.
	 *
	 *	Method is passed the object list. It returns true to indicate 	 
	 *	that grammar has handled the unknown verb itself.
	 */
	errorParseObjlist( objlist ) = 
	{
		"Sorry, guv. Could you be a little more specific?\b";
		return true;
	}
	/*
	 *	parseDest( obj )
	 *
	 *	Method is meant to allow for flexibility in the method of 	 	 
	 *	identification of the cab destination. While it is possible to 	 
	 *	add vocabulary to the room object, this allows for a pointer 	 
	 *	system, which can be more flexible when referencing a location 	 
	 *	using multiple names. 
	 *
	 *	For example, suppose we want to visit "The Arch", which is 	 	 
	 *	located in Hyde Park Corner. We can request ">hyde park 	 	 
	 *	corner" which points to itself, or we can request ">the arch" 	 
	 *	which points to Hyde Park Corner
	 *	through its destPtr. This should help reduce the number of 	 	 
	 *	objects required in a game.
	 *
	 *	Method takes the single object parsed from the object list and 	 
	 *	returns the property pointer &destPtr.
	 */
	parseDest( obj ) =
	{
		local propPtr = &destPtr;
		if ( obj.(propPtr) ) 
			return obj.(propPtr);
		else
			return obj;
	}
	/*
	 *	errorParseDest
	 *
	 *	Method displays an appropriate message when the parseDest() 	 
	 *	returns nil.
	 *	
	 *	Method takes no arguments. It returns true to indicate that 	 
	 *	actor grammar has handled the error itself.
	 */
	errorParseDest = 
	{ 
		"Sorry, I can't take <<self.location.thedesc>> there. ";
		return true; 
	}
	/*
	 *	goDest( dest )
	 *
	 *	Method moves the cab, driver, and passengers to their new 	 	 
	 *	location. The destination is first checked to determine that 	 
	 *	we aren't already at the location we've requested. If so then 	 
	 *	we display the errorGoMessage; otherwise we display a message, 	 
	 *	move the cab, and use the travelTo() method to display the new 	 
	 *	location.
	 *
	 *	Method takes the destination returned from parseDest().
	 */
	goDest( dest )=
	{
	    local actor = parserGetObj(PO_ACTOR);
	    if (actor == nil) actor = parserGetMe();
	    
		if ( dest == self.location.location )
			self.errorGoMessage;
		else
		{
			self.goMessage;
			self.location.moveInto( dest );
    		self.location.lookAround(actor, true, true);
		}
	}
	/*
	 *	goMessage
	 *	
	 *	Method displays an appropriate message to indicate that we are 	 
	 *	moving from one location to another.
	 *
	 *	Method is passed no parameters.
	 */
	goMessage = 
	{
		caps(); "<<self.location.thedesc>> speeds through the
		streets, whizzing past famous landmarks. Only a few minutes
		later, you arrive at your destination.\b";
	}
	/*
	 *	errorGoMessage
	 *
	 *	Method displays an appropriate message when we have requested 	 
	 *	the location we are at currently.
	 *
	 *	Method is passed no parameters.
	 */
	errorGoMessage =
	{
		caps(); "<<self.thedesc>> says, <q>We're already here,
		guv.</q> ";
	}
	/*
	 *	displayToklist( tokenlist )
	 *
	 *	Method is used to diplay the tokenlist, providing spacing 	 	 
	 *	between each token.
	 *
	 *	Method is passed the tokenlist.
	 */
	displayToklist( tokenlist ) =
	{
        	local w;
        	
			w = car( tokenlist );
			while( w )
        	{
          		say( w );
          		tokenlist = cdr( tokenlist );
          		w = car( tokenlist );
			if ( w ) say(' ');
       		}
	}
;

/*
 *  eventObj: object
 *
 *  The eventObj stores all the information for the event. This object
 *  is dynamically-created and re-used as needed.
 */
class eventObj: object
	isscheduled     = nil
	objPtr          = nil
	processPtr      = nil
	validatePtr     = nil
	stage           = nil
	retain          = nil
	frequency       = 0
	frequencyCnt    = 0
	commArg         = nil
	actorPtr        = nil
	verbPtr         = nil
	dobjListPtr     = []
	prepPtr         = nil
	iobjPtr         = nil
	eventCnt        = 0
	pendingReason   = nil
;

//**********************************************************************
//  
//  MODIFICATIONS TO ADV.T / STD.T
//
//**********************************************************************

modify thing
    clist = []
    contentsAudible = (true)
    contentsOlfactory = (true)
    
    //
    //  The isqXxxx methods should return true or nil depending on
    //  whether the object is "quiet" about its contents or not.
    //
    isqVisible = (nil)
    isqAudible = (nil)
    isqOlfactory = (nil)
    
    passToContents(plist, filter) = (true)
    passAcrossContents(plist, filter) = (true)
    passToLocation(plist, filter) = (true)
    
    //
    //  Additional isListed attributes for the sense involved.
    //
    isListedAudible = (nil)
    isListedOlfactory = (nil)
    
	/*
 	 *	Modifying verifyRemove() to 'bubble up' verGrab() check, but only as
 	 *	high as thescope ceiling. This allows us to grab things when we're
 	 *	inside closed locations, such as enterables.
 	 */
    replace verifyRemove(actor) = {
        /*
         *   Check with each container to make sure that the container
         *   doesn't object to the object's removal.
         */
        local loc, ceiling;

		ceiling = scopeCeiling( actor, nil, &scope_reachable );

        loc = self.location;
        while (loc)
        {
        	if (loc == ceiling)
      			break;
            if (loc != actor)
                loc.verGrab(self);
            loc = loc.location;
        }
    }
    /*
     *	isVisible() now returns true for all objects. This means that
     *  the cantReach() mechanism will handle the case where both the
     *  valid list produced by validXoList/validXo and the logical list
     *  produced by verXoVerb during Disambiguation are both empty.
     */
    isVisible(vantage) = (true)
    
    /*
     *  isSensible(vantage, scope_filter)
     *
     *  isSensible() will return either nil or the PRX_XXXX flag as
     *  determined by the proximity() function for the vantage and 
     *  target (i.e. self). The method returns a PRX_XXXX flag only if 
     *  the path between the vantage and target is accessible to at
     *  least one scope_filter.
     *
     *  If a scope filter is passed the method returns the PRX_XXXX flag
     *  only if the path is accessiblefor that scope_filter.
     */
    isSensible(vantage, ...) = {
        local p, filter, prx_flag, vsList;
        
        if (argcount > 1) filter = getarg(2);
        
        p = proximity(vantage, self);
        prx_flag    = p[1];
        vsList      = p[2];
        
        if (length(vsList) == 0)
            return nil;
        else if (filter == nil)
            return prx_flag;
        else if (find(vsList, filter))
            return prx_flag;
        else
            return nil;
    }
    /*	add CONSULT and LOOKUP verification and action methods */
	verIoConsultAbout( actor, dobj ) = {}
	ioConsultAbout( actor, dobj ) =
	{
		dobj.doConsultAbout( actor, self );
	}
	ioSynonym( 'ConsultAbout' ) = 'ConsultOn'
	verDoLookupIn( actor, dobj ) = {}	
    /* 
     *	We modify only the totbulk logic of thing (the idea being
     *	that a player can only carry so much weight regardless of 
     *	whether they have a sackItem or not.
     */
    doTake(actor) =
    {
        local totbulk, totweight;

        totbulk = addbulk(actor.contents) + self.bulk;
        totweight = addweight(actor.contents);
        if (! actor.isCarrying(self))
            totweight = totweight + self.weight + addweight(self.contents);

        if (totweight > actor.maxweight)
            "%Your% load is too heavy. ";
        else if (totbulk > actor.maxbulk)
        {
        	if ( self.moveToSackItem( actor ) )
        	{
        		self.moveInto( actor );
        		"Taken. ";
        	}
        	else
            	"%You've% already got %your% hands full. ";
        }
        else
        {
            self.moveInto(actor);
            "Taken. ";
        }
    }
    /*
     *	This is the heart of the sackItem logic. The logic does not 
     *	check for multiple sackItems, it takes the first one it finds
     *	Also it does not attempt to optimise the placement of items
     *	in the sackItem, except that items that are lit or that are
     *	being worn are not chosen.
     */
    moveToSackItem( actor ) =
    {
    	local c, n = [], o, s, totbulk;
    	
	    /* 
         *	determine if the actor is carrying a sackItem 
         */  	
    	c = actor.contents;
    	o = car( c );
 		while( o )
		{

			if ( isclass( o, sackItem ) ) 
			{ 
				break;
			}
			c = cdr( c );
			o = car( c );
		};
		s = o;			// sackItem value
		
    	if ( s == nil ) return nil;
	
		c = actor.contents - [ s ];	// exclude the sackItem
 	    /*
 	     *	Select items that aren't being worn or that aren't lit 
 	     *	as candidates for the sackItem.
 	     */		
		o = car( c );
		while( o )
		{
			if ( ! o.isworn && ! o.islit )
			{
				n = n + [ o ];
			}
			c = cdr( c );
			o = car( c );
		}
        /*
         *	Determine if an object will fit in the sack and move it
         *	if it does.
         */	
     	o = car( n );
    	while( o )
		{
			totbulk = addbulk( s.contents ) + o.bulk;
				
			if ( totbulk <= s.maxbulk )
			{
				o.moveInto( s );
				"(putting <<o.thedesc>> into <<s.thedesc>> to make 
				    room)\n";
				return true;
			};
			
			n = cdr( n );
			o = car( n );
    	}
    	return nil;
    }
    //
    //  modified to give object the attribute hasmoved = true
    //  for use in initial description processing.
    //
    moveInto(obj) =
    {
        local actor = parserGetObj(PO_ACTOR);
        local verb = parserGetObj(PO_VERB);
        local loc;

        /*
         *   For the object containing me, and its container, and so forth,
         *   tell it via a Grab message that I'm going away.
         */
        loc = self.location;
        while (loc)
        {
            loc.Grab(self);
            loc = loc.location;
        }
        
        if (self.location)
            self.location.contents = self.location.contents - self;
        self.location = obj;
        if (obj)
            obj.contents = obj.contents + self;
        
        if (actor == parserGetMe() 
        && verb != nil
        && verb != dropVerb)
            self.hasmoved = true;

        maintainScopelist(self);
    }
    
#ifdef USE_COMMAND_MESSAGING
    
    /*
     *  Starts capture of output that would result from the parser call
     *  for multiple object multisdesc / sdesc handling. Capture is
     *  turned off at the very beginning of verbAction() which merely 
     *  suppresses the print of obj.sdesc; ":"; messaging used for 
     *  multiple object commands.
     */
    multisdesc =
    {
        global.stat = outcapture(true);        
    }

#endif  /* USE_COMMAND_MESSAGING */
    
    //
    //  Display contents for self, using a given actor,
    //  scope filter, isListedXxxx attribute and sense 
    //  description.
    //
    dispcont(actor, filter, listedAttr, descAttr) = {
        local i, list, list_filter;
        local disptot = 0, stat, str;             
        
        switch(filter)
        {
            case &scope_visible:
                list_filter = &sccope_qVisible;
                break;
            case &scope_audible:
                list_filter = &scope_qAudible;
                break;
            case &scope_olfactory:
                list_filter = &scope_qOlfactory;
                break;
            default:
                return nil;
        }
        
        list  = scope(self, self, list_filter) - self;
        
        for (i = 1; i <= length(list); ++i)
        { 
            command.plist = path(actor, list[i]);
            
            if (blocksPath(command.plist, filter) == nil
            && list[i].(listedAttr))
            {
                stat = outcapture(true);

#ifdef USE_COMMON_SENSE
                
                list[i].(descAttr)(actor);
                
#else /* USE_COMMON_SENSE */
                
                list[i].(descAttr);
                
#endif /* USE_COMMON_SENSE */
                
                str = outcapture(stat);
                if (length(str))
                {
                    say(str);
                    ++disptot;
                }   
            }
        }
        return disptot;
    }
    
    /*
     *  cantSense methods are at the object level. They are passed the
     *  actor and the word list. 
     */
    cantSee(actor) =
    {
        local i;
        
        "%You% can't see any ";
        for (i = 1; i <= length(command.cantReachWords); ++i)
        {
            say(command.cantReachWords[i]);
        }
        " here.";
    }        
    cantHear(actor) = {
        local failloc = blocksPath(command.plist, &scope_visible);
        
        if (failloc)
            failloc.cantSee(actor);
        else
            "%You% can't hear that here.";
    }        
    cantSmell(actor) = {
        local failloc = blocksPath(command.plist, &scope_visible);
        
        if (failloc)
            failloc.cantSee(actor);
        else
            "%You% can't smell that from here.";
    }        
    replace cantReach(actor) = {
        local failloc = blocksPath(command.plist, &scope_visible);
        
        if (failloc)
            failloc.cantSee(actor);
        else    
            "%You% can't reach that from here. ";
    }
        
    /*
     *  Sense descriptions
     */
    ldesc = { "%You% see%s% nothing unusual about 
        <<self.thedesc>>. "; }
    listendesc = { "%You% hear%s% nothing unusual from 
        <<self.thedesc>>. "; }
    smelldesc = { "%You% smell%s% nothing unusual about 
        <<self.thedesc>>. "; }
    touchdesc = { "%You% feel%s% nothing unusual about 
        <<self.thedesc>>. "; }
    tastedesc = { "%You% taste%s% nothing unusual about 
        <<self.thedesc>>. "; }

#ifdef USE_COMMON_SENSE

    listdesc(descPtr) = {        
        local s = self.isSensible(command.actorPtr, nil);
        
        switch( descPtr )
        {
            case &adesc:
                switch( s )
                {
                    case PRX_REFLEXIVE:
                        self.adesc_reflexive;
                        break;
                    case PRX_POSSESSIVE:
                        self.adesc_possessive;
                        break;
                    case PRX_CONTAINED:
                        self.adesc_contained;
                        break;
                    case PRX_IMMEDIATE:
                        self.adesc_immediate;
                        break;
                    case PRX_LOCAL:
                        self.adesc;
                        break;
                    case PRX_REMOTE:
                        self.adesc_remote;
                        break;
                    default:
                        self.adesc;
                }
                
                /* show any additional information about the item */
                if (self.isworn)
                    " (being worn)";
                if (self.islamp and self.islit)
                    " (providing light)";        
                break;

            case &thedesc:
                switch( s )
                {
                    case PRX_REFLEXIVE:
                        self.thedesc_reflexive;
                        break;
                    case PRX_POSSESSIVE:
                        self.thedesc_possessive;
                        break;
                    case PRX_CONTAINED:
                        self.thedesc_contained;
                        break;
                    case PRX_IMMEDIATE:
                        self.thedesc_immediate;
                        break;
                    case PRX_LOCAL:
                        self.thedesc;
                        break;
                    case PRX_REMOTE:
                        self.thedesc_remote;
                        break;
                    default:
                        self.thedesc;
                }
                break;
                
            case &pluraldesc:
                switch( s )
                {
                    case PRX_REFLEXIVE:
                        self.pluraldesc_reflexive;
                        break;
                    case PRX_POSSESSIVE:
                        self.pluraldesc_possessive;
                        break;
                    case PRX_CONTAINED:
                        self.pluraldesc_contained;
                        break;
                    case PRX_IMMEDIATE:
                        self.pluraldesc_immediate;
                        break;
                    case PRX_LOCAL:
                        self.pluraldesc;
                        break;
                    case PRX_REMOTE:
                        self.pluraldesc_remote;
                        break;
                    default:
                        self.pluraldesc;
                }
                break;
        }
    }
    
    sdesc_reflexive         = { self.sdesc; }
    sdesc_possessive        = { self.sdesc; }
    sdesc_contained         = { self.sdesc; }
    sdesc_immediate         = { self.sdesc; }
    sdesc_remote            = { self.sdesc; }
    
    adesc_reflexive         = { self.adesc; }
    adesc_possessive        = { self.adesc; }
    adesc_contained         = { self.adesc; }
    adesc_immediate         = { self.adesc; }
    adesc_remote            = { self.adesc; }
    
    thedesc_reflexive       = { self.thedesc; }
    thedesc_possessive      = { self.thedesc; }
    thedesc_contained       = { self.thedesc; }
    thedesc_immediate       = { self.thedesc; }
    thedesc_remote          = { self.thedesc; }
    
    pluraldesc_reflexive    = { self.pluraldesc; }
    pluraldesc_possessive   = { self.pluraldesc; }
    pluraldesc_contained    = { self.pluraldesc; }
    pluraldesc_immediate    = { self.pluraldesc; }
    pluraldesc_remote       = { self.pluraldesc; }

    ldesc_reflexive         = { self.ldesc; }
    ldesc_possessive        = { self.ldesc; }
    ldesc_contained         = { self.ldesc; }
    ldesc_immediate         = { self.ldesc; }
    ldesc_remote            = { self.ldesc; }

    listendesc_reflexive    = { self.listendesc; }
    listendesc_possessive   = { self.listendesc; }
    listendesc_contained    = { self.listendesc; }
    listendesc_immediate    = { self.listendesc; }
    listendesc_remote       = { self.listendesc; }

    smelldesc_reflexive     = { self.smelldesc; }
    smelldesc_possessive    = { self.smelldesc; }
    smelldesc_contained     = { self.smelldesc; }
    smelldesc_immediate     = { self.smelldesc; }
    smelldesc_remote        = { self.smelldesc; }

#endif /* USE_COMMON_SENSE */
       
    /*
     *  Sense verb methods
     */

#ifdef USE_COMMON_SENSE

    replace doInspect(actor) =
    {
        self.proxDoInspect(actor);
    }
    proxDoInspect(actor) =
    {
        local s = self.isSensible(actor, nil);
        switch( s )
        {
            case PRX_REFLEXIVE:
                self.ldesc_reflexive;
                break;
            case PRX_POSSESSIVE:
                self.ldesc_possessive;
                break;
            case PRX_CONTAINED:
                self.ldesc_contained;
                break;
            case PRX_IMMEDIATE:
                self.ldesc_immediate;
                break;
            case PRX_LOCAL:
                self.ldesc;
                break;
            case PRX_REMOTE:
                self.ldesc_remote;
                break;
            default:
                self.ldesc;
        }
    }
    verDoListento(actor) = {}
    doListento(actor) = 
    {
        self.proxDoListen(actor);
    }
    proxDoListen(actor) =
    { 
        local s = self.isSensible(actor, nil);
        switch( s )
        {
            case PRX_REFLEXIVE:
                self.listendesc_reflexive;
                break;
            case PRX_POSSESSIVE:
                self.listendesc_possessive;
                break;
            case PRX_CONTAINED:
                self.listendesc_contained;
                break;
            case PRX_IMMEDIATE:
                self.listendesc_immediate;
                break;
            case PRX_LOCAL:
                self.listendesc;
                break;
            case PRX_REMOTE:
                self.listendesc_remote;
                break;
            default:
                self.listendesc;
        }
    }
    verDoSmell(actor) = {}
    doSmell(actor) = 
    {
        self.proxDoSmell(actor);
    }
    proxDoSmell(actor) =
    { 
        local s = self.isSensible(actor, nil);
        switch( s )
        {
            case PRX_REFLEXIVE:
                self.smelldesc_reflexive;
                break;
            case PRX_POSSESSIVE:
                self.smelldesc_possessive;
                break;
            case PRX_CONTAINED:
                self.smelldesc_contained;
                break;
            case PRX_IMMEDIATE:
                self.smelldesc_immediate;
                break;
            case PRX_LOCAL:
                self.smelldesc;
                break;
            case PRX_REMOTE:
                self.smelldesc_remote;
                break;
            default:
                self.smelldesc;
        }
    }

#else /* USE_COMMON_SENSE */
     
     verDoListento(actor) = {}
     doListento(actor) = { self.listendesc; }
     verDoSmell(actor) = {}
     doSmell(actor) = { self.smelldesc; }

#endif /* USE_COMMON_SENSE */

     verDoTouch(actor) = {}
     doTouch(actor) = { self.touchdesc; }
     verDoTaste(actor) = {}
     doTaste(actor) = { self.tastedesc; }

    //
    //  modified to use the isSensible() method
    //
     validActor = {
        if (self.isSensible(parserGetMe(), &scope_reachable))
            return true;
        else
            return nil;
     }
;

modify switchItem
    isswitchItem = true
;

///////////////////////////////////////////////////////////////////////
//
//  REPLACING THE ADV.T CONTAINMENT CLASSES
//
///////////////////////////////////////////////////////////////////////

replace surface: containment
    issurface = true
;
replace qsurface: containment
    isqsurface = true
    isqVisible = true
    isqAudible = true
    isqOlfactory = true
;
replace container: containment
    iscontainer = true
;
replace qcontainer: containment
    isqcontainer = true
    isqVisible = true
    isqAudible = true
    isqOlfactory = true
;
replace openable: containment
    isopenable = true
;
replace transparentItem: containment
    istransparent = true
;
replace lockable: containment
    islockable = true
;    
replace room: containment
    lightsOn = true
;
replace darkroom: containment
;

replace nestedroom: containment
;
replace vehicle: containment
    isvehicle = true
    iscontainer = true
;
replace chairitem: containment
    issurface = true
    isenterable = true
;
replace beditem: containment
    issurface = true
    isenterable = true
;
modify theFloor
    isListed = nil
;

modify movableActor
    passToLocation(plist, filter) = true
    isListedAudible = nil
    isListedOlfactory = nil
    
    isqVisible = true
    isqAudible = (nil)
    isqOlfactory = (nil)
   /*
    *	Modification of movableActor creating a default ACTOR
    *	GRAMMAR method that returns nil.
    */
	grammar( wordlist, typelist, errnum ) =
	{
		return nil;
	}
;

/*
 *  basicMe: Actor
 *
 *  MODIFIED TO USE maintainScopelist()
 *
 *  A default implementation of the Me object, which is the
 *  player character.  adv.t defines basicMe instead of
 *  Me to allow your game to override parts of the default
 *  implementation while still using the rest, and without changing
 *  adv.t itself.  To use basicMe unchanged as your player
 *  character, include this in your game:  "Me: basicMe;".
 *  
 *  The basicMe object defines all of the methods and properties
 *  required for an actor, with appropriate values for the player
 *  character.  The nouns "me" and "myself" are defined ("I"
 *  is not defined, because it conflicts with the "inventory"
 *  command's minimal abbreviation of "i" in certain circumstances,
 *  and is generally not compatible with the syntax of most player
 *  commands anyway).  The sdesc is "you"; the thedesc
 *  and adesc are "yourself," which is appropriate for most
 *  contexts.  The maxbulk and maxweight properties are
 *  set to 10 each; a more sophisticated Me might include the
 *  player's state of health in determining the maxweight and
 *  maxbulk properties.
 */
modify basicMe
    isqVisible = (true)
    isqAudible = (true)
    isqOlfactory = (true)
    
    replace travelTo(room) =
    {
        /* 
         *   if I'm not the current player character, inherit the default
         *   handling 
         */
        if (parserGetMe() != self)
        {
            /* 
             *   I'm not the current player character, so I'm just an
             *   ordinary actor - use the inherited version that ordinary
             *   actors use 
             */
            inherited.travelTo(room);
            return;
        }

        /* 
         *   if the room is not nil, travel to the new location; ignore
         *   attempts to travel to "nil", because direction properties and
         *   obstacles return nil after displaying a message explaining
         *   why the travel is impossible, hence we need do nothing more
         *   in these cases 
         */
        if (room != nil)
        {
            if (room.isobstacle)
            {
                /* it's an obstacle - travel to the obstacle's destination */
                self.travelTo(room.destination);
            }
            else if (!(self.location.islit || room.islit))
            {
                /* neither location is lit - travel in the dark */
                darkTravel();
            }
            else
            {
                /* notify the old location that we're leaving */
                if (self.location != nil)
                    self.location.leaveRoom(self);

                /* move to the new location */
                self.location = room;
                maintainScopelist(self);

                /* notify the new location that we're entering */
                room.enterRoom(self);
            }
        }
    }
    replace moveInto(room) =
    {
        /*
         *   If I"m not the current player character, inherit the default
         *   handling; otherwise, we must use special handling, because
         *   the current player character never appears in its location's
         *   contents list. 
         */
        if (parserGetMe() == self)
        {
            /* 
             *   we're the player character - move directly to the new
             *   room; do not update the contents lists for my old or new
             *   containers, because we do not appear in the old list and
             *   do not want to appear in the new list 
             */
            self.location = room;
            maintainScopelist(self);
        }
        else
        {
            /* 
             *   we're an ordinary actor at the moment - inherit the
             *   default handling 
             */
            inherited.moveInto(room);
        }
    }
    thedesc = {
        if (self == parserGetMe())
            "yourself";
        else
            "your former self";
    }
    adesc = {
        if (self == parserGetMe())
            "yourself";
        else
            "your former self";
    }
    ldesc = "You look just like <<self.thedesc>>. "
;

#pragma C-
