//
//  XTTextView.m
//  XTads
//
//  Created by Rune Berg on 28/01/2017.
//  Copyright © 2017 Rune Berg. All rights reserved.
//

#import "XTTextView.h"
#import "XTTabStopModelEntry.h"
#import "XTStringUtils.h"
#import "XTLogger.h"


@interface XTTextView ()

@property BOOL hasDoneCustomInit;
@property BOOL isObservingPrefs;

@end


@implementation XTTextView

static XTLogger* logger;

+ (void)initialize
{
	logger = [XTLogger loggerForClass:[XTTextView class]];
}

- (void)customInit
{
	if (_hasDoneCustomInit) {
		return;
	}

	_leftRightInset = 8.0;
	_topBottomInset = 8.0;

	_prefs = [XTPrefs prefs];
	[_prefs startObservingChangesToAll:self];
	_isObservingPrefs = YES;
	
	NSSize insetOutput = NSMakeSize(_leftRightInset, _topBottomInset);
	[self setTextContainerInset:insetOutput];
	
	_hasDoneCustomInit = YES;
}

- (void)teardown
{
	if (_isObservingPrefs) {
		[_prefs stopObservingChangesToAll:self];
		_isObservingPrefs = NO;
	}
	
	NSAttributedString *noText = [NSAttributedString new];
	[self.textStorage setAttributedString:noText];
	
	self.delegate = nil;
}

//TODO do in handler?
- (void)observeValueForKeyPath:(NSString *)keyPath
					  ofObject:(id)object
						change:(NSDictionary *)change
					   context:(void *)context
{
	XT_DEF_SELNAME;
	XT_TRACE_1(@"keyPath=\"%@\"", keyPath);
	
	if (object == self.prefs) {
		[self syncWithPrefs];
	} else {
		XT_TRACE_0(@"%@ OTHER");
	}
}

- (CGFloat)findTotalWidthAdjustedForInset
{
	CGFloat res = self.frame.size.width - (2 * self.leftRightInset);
	return res;
}

- (void)drawRect:(NSRect)dirtyRect {
    [super drawRect:dirtyRect];
    
    // Drawing code here.
}

- (void)viewDidEndLiveResize
{
	//XT_DEF_SELNAME;
	//XT_TRACE_0(@"");

	[super viewDidEndLiveResize];
	
	[self.outputFormatter recalcAllTabStops];
}

//TODO mv
- (BOOL)tabStopIsAtRhsOfView:(NSTextTab *)textTab
{
	return [self tabStop:textTab isOfType:XT_TAB_TYPE_RHS_OF_VIEW];
}

- (BOOL)tabStopIsAtHalfwayToRhsOfView:(NSTextTab *)textTab
{
	return [self tabStop:textTab isOfType:XT_TAB_TYPE_HALFWAY_TO_RHS_OF_VIEW];
}
- (BOOL)tabStopIsToId:(NSTextTab *)textTab
{
	return [self tabStop:textTab isOfType:XT_TAB_TYPE_TO_ID];
}

- (BOOL)tabStopIsAtMultiple:(NSTextTab *)textTab
{
	return [self tabStop:textTab isOfType:XT_TAB_TYPE_MULTIPLE];
}

- (BOOL)tabStopIsAtFixedPos:(NSTextTab *)textTab
{
	return [self tabStop:textTab isOfType:XT_TAB_TYPE_FIXED_POS];
}

//TODO mv
- (BOOL)tabStop:(NSTextTab *)textTab isOfType:(NSString *)type
{
	NSDictionary<NSString *,id> *options = textTab.options;
	NSString *actualType = [options objectForKey:XT_TAB_TYPE_KEY];
	BOOL res = [actualType isEqualToString:type];
	return res;
}

- (CGFloat)findCoordOfTabAtRhsOfView
{
	CGFloat res = [self findTotalWidthAdjustedForInset];
	CGFloat lfPadding = self.textContainer.lineFragmentPadding;
	res -= (2.0 * lfPadding);
	return res;
}

- (CGFloat)findCoordOfTabAtHalfwayToRhsOfViewFromLoc:(CGFloat)fromLoc
{
	//XT_DEF_SELNAME;
	
	CGFloat coordRhsOfView = [self findCoordOfTabAtRhsOfView];
	CGFloat pointsToHalfway = (coordRhsOfView - fromLoc) / 2.0;
	if (pointsToHalfway < 0.0) {
		pointsToHalfway = 0.0;
	}
	CGFloat res = fromLoc + pointsToHalfway;
	return res;
}

//
// Find lower end of insertion point, on "flipped" y-axis (0 being top of view)
//
- (CGFloat)findYCoordOfInsertionPoint
{
	XT_DEF_SELNAME;
	
	// http://stackoverflow.com/questions/7554472/gettting-window-coordinates-of-the-insertion-point-cocoa
	
	// -firstRectForCharacterRange:actualRange returns a frame rectangle
	// containing the insertion point or, in case there's a selection,
	// the first line of that selection. The origin of the frame rectangle is
	// in screen coordinates
	NSRange ipRange = [self selectedRange];
	NSRect ipRectScreen = [self firstRectForCharacterRange:ipRange actualRange:NULL];
	
	CGFloat res;
	
	if (ipRectScreen.origin.y == 0.0) {
		
		// Sometimes, after certain keystrokes at the more prompt (space, arrow-*, page-*), we get
		// an empty ipRectScreen. I haven't managed to understand why, but this is a way to compensate for that.
		// adhoc test game:
		//		only seems to happen at end of output, when called via resetForGameHasEndedMsg
		//		seems ins pos is somehow affected by those keys...
		//			visually, they behave different too...
		// dearbrian startup
		//		for ditto keys
		
		XT_WARN_0(@"got empty rectangle from firstRectForCharacterRange, using frame height - 2");
		CGFloat selfHeight = self.frame.size.height;
		res = selfHeight - 2;
		// Note: with pagination based on ins pt y coord being recalc'd after every text insertion,
		// we got this a lot more often, no matter what key is pressed :-(
		
	} else {
		
		// Normal case.
		
		NSPoint ipPoint = [self getInsertionPointNormalCase:ipRectScreen];
		res = ipPoint.y;
	}
	
	res = ceil(res);

	XT_TRACE_1(@"--> %f", res);
	return res;
}

- (CGFloat)findXCoordOfInsertionPointRaw
{
	XT_DEF_SELNAME;
	
	CGFloat res = [self findXCoordOfInsertionPointInternal];
	// *** No rounding of res ***
	
	XT_TRACE_1(@"--> %f", res);
	return res;
}

- (CGFloat)findXCoordOfInsertionPointInternal
{
	//XT_DEF_SELNAME;

	[self ensureLayoutForTextContainer];
	
	NSRange ipRange = [self selectedRange];
		// which we assume is at the very end of our textview's text
	
	//TODO use this approach in y coord case too:
	
	CGFloat res = [self findStartXCoordOfTextInRange:ipRange];
	return res;
}

- (CGFloat)findStartXCoordOfTextInRange:(NSRange)range
{
	XT_DEF_SELNAME;

	/* [self firstRectForCharacterRange:...] is unstable when text scrolls, so use approach suggested by:
	 http://webcache.googleusercontent.com/search?q=cache:E7qpW4SPJZAJ:www.cocoabuilder.com/archive/cocoa/158114-nstextview-firstrectforcharacterrange-returns-empty-rect.html+&cd=3&hl=en&ct=clnk&gl=no
	 */
	NSRect ipRectAlt = [self.layoutManager boundingRectForGlyphRange:range inTextContainer:self.textContainer];
	if (ipRectAlt.origin.x == 0) {
		XT_WARN_0(@"got empty rectangle from boundingRectForGlyphRange");
	}
	NSSize tCInset = self.textContainerInset;
	CGFloat resAlt = ipRectAlt.origin.x + tCInset.width;
	
	XT_TRACE_1(@"--> %f", resAlt);
	return resAlt; //was: res;
}

//TODO refactor wrt findStartXCoordOfTextInRange
- (CGFloat)findEndXCoordOfTextInRange:(NSRange)range
{
	XT_DEF_SELNAME;
	
	NSRect ipRectAlt = [self.layoutManager boundingRectForGlyphRange:range inTextContainer:self.textContainer];
	if (ipRectAlt.origin.x == 0) {
		XT_WARN_0(@"got empty rectangle from boundingRectForGlyphRange");
	}
	CGFloat resAlt = ipRectAlt.origin.x;
	resAlt += ipRectAlt.size.width;  // !
	NSSize tCInset = self.textContainerInset;
	resAlt += tCInset.width;
	
	XT_TRACE_1(@"--> %f", resAlt);
	return resAlt;
}

- (NSPoint)getInsertionPointNormalCase:(NSRect)ipRectScreen
{
	// -convertRectFromScreen: converts from screen coordinates to window coordinates
	NSRect ipRectWindow = [self.window convertRectFromScreen:ipRectScreen];
	
	// The origin is the lower left corner of the frame rectangle containing the insertion point
	NSPoint ipPointWindow = ipRectWindow.origin;
	
	NSPoint ipPointSelf = [self convertPoint:ipPointWindow fromView:nil];
	// Note: NSTextView uses a flipped y-axis, so .y is wrt top of frame
	
	return ipPointSelf;
}

- (void)setInsertionPointAtEndOfText
{
	NSUInteger index = self.textStorage.length;
	[self setSelectedRange:NSMakeRange(index, 0)];
}

- (void)ensureLayoutForTextContainer
{
	NSTextContainer* textContainer = [self textContainer];
	NSLayoutManager* layoutManager = [self layoutManager];
	[layoutManager ensureLayoutForTextContainer:textContainer];
}

- (CGFloat)findTotalHeight
{
	CGFloat res = self.frame.size.height;
	return res;
}

- (CGFloat)findVisibleHeight
{
	[self ensureLayoutForTextContainer];
	
	CGFloat res = self.enclosingScrollView.documentVisibleRect.size.height;
	return res;
}

@end
