#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Assert.t
 *
 *  Provides a mechanism for asserting the conditions of methods at
 *  runtime.
 */

#include "proteus.h"

/*
 *  Assert enforces the datatype of a variable, or a list of variables.
 *  This function can be used to enforce an interface or inheritance
 *  relationship by passing the interface or object to be checked
 *  against in the 'type' argument.
 *
 *  Function throws an exception if 'args' are not of datatype 'type' or
 *  implements interface 'type' or is ofKindOrUU(type).
 */
assert(type, [args])
{
    if (dataType(type) == TypeObject)
    {
        local sc = type.getSuperclassList.car();
        
        if (sc == AnonFuncPtr)
        {
            foreach (local arg in args)
                if (!type(arg))
                    throw new InvalidValueAssertException(type, arg);
        }
        else 
        {
            foreach (local arg in args)
                if (!arg.implementsIF(type)
                && !arg.ofKindOrUU(type))
                    throw new InvalidTypeAssertException(type, arg);
        }
    }
    else
    {
        foreach(local arg in args)
            if (dataType(arg) != type)
                throw new InvalidTypeAssertException(type, arg);
    }
}

/*
 *  A base class for Assert Exceptions.
 */
class AssertException: ProteusException
{
    construct(msg) { inherited(); }
    
    getTypeDesc(type)
    {    
        local typeName;
        
        switch(type)
        {
            case 1:
                typeName = 'TypeNil';
                break;
               
            case 2:
                typeName = 'TypeTrue';
                break;
                
            case 5:
                typeName = 'TypeObject';
                break;
                
            case 6:
                typeName = 'TypeProp';
                break;
                
            case 7:
                typeName = 'TypeInt';
                break;
                
            case 8:
                typeName = 'TypeSString';
                break;
                
            case 10:
                typeName = 'TypeList';
                break;
                
            case 12:
                typeName = 'TypeFuncPtr';
                break;
                
            case 15:
                typeName = 'TypeEnum';
                break;
                
            default:
                typeName = 'UNKNOWN';
        }
        
        return typeName;
    }
}

class InvalidValueAssertException: AssertException
{
    construct(type, arg)
    {
        local typeName, argDT, argName, argVal;
        
        typeName = String.toSString(type);
            
        argDT = dataType(arg);
        
        if (dataType(type) == TypeObject)
        {
            typeName = String.toSString(type);
            
            if (dataType(type) == TypeObject)
                argName = String.toSString(arg);
            else
                argName = getTypeDesc(argDT);
        }
        else 
        {
            typeName    = getTypeDesc(type);
            argName     = getTypeDesc(argDT);
        }
        
        argVal = String.toSString(arg);
            
        exceptionMessage = '[<b>Assertion failed:</b> 
            datatype ' + typeName + ' required for argument, 
            but argument has datatype ' + argName + ' with value \"' +
            argVal + '\".]';  
            
        inherited();
    }
}

class InvalidTypeAssertException: AssertException
{
    construct(type, arg)
    {
        local typeName, argDT, argName, argVal;
        
        argDT = dataType(arg);
        
        if (dataType(type) == TypeObject)
        {
            typeName = String.toSString(type);
            
            if (dataType(type) == TypeObject)
                argName = String.toSString(arg);
            else
                argName = getTypeDesc(argDT);
        }
        else 
        {
            typeName    = getTypeDesc(type);
            argName     = getTypeDesc(argDT);
        }
        
        argVal = String.toSString(arg);
            
        exceptionMessage = '[<b>Assertion failed:</b> 
            datatype ' + typeName + ' required for argument, 
            but argument has datatype ' + argName + ' with value \"' +
            argVal + '\".]';
            
        inherited();
    }
}