#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Dijkstra.t
 *
 *      Djikstra's algorithm (named after its discover, 
 *      E.W. Dijkstra) solves the problem of finding the 
 *      shortest path from a point in a graph (the source) 
 *      to a destination. It turns out that one can find 
 *      the shortest paths from a given source to all points 
 *      in a graph in the same time, hence this problem is 
 *      sometimes called the single-source shortest paths 
 *      problem. 
 *      
 *      This problem is related to the spanning tree one. 
 *      The graph representing all the paths from one 
 *      vertex to all the others must be a spanning tree - 
 *      it must include all vertices. There will also be 
 *      no cycles as a cycle would define more than one 
 *      path from the selected vertex to at least one 
 *      other vertex.
 */

/* include the TADS and T3 system headers */
#include "tads.h"
#include "t3.h"
#include "vector.h"

#define INFINITY	0x7fffffff

class Entry: object
{
    vertex              = nil
    known               = nil
    predecessor         = nil
    value       	    = nil

    destInfoList    	= []
    knownDestInfoList 	= []    
    adjacentEntryList   = []
    travelEdgeList      = []

    /*
     *  Method for comparing another entry value to this one
     *  for the purpose of sorting entries.
     */
    compareValueOf(e) {return value - e.value;}

    construct( v, p, n ) 
    {
        vertex		    = v;
        predecessor 	= p;
        value       	= n;

    }
}

class TravelLink: object
{
    currTravelNode          = nil
    nextTravelNode          = nil

    travelEdgeList          = []

    construct( c, n, l ) 
    {
        currTravelNode  = c;
        nextTravelNode  = n;
        travelEdgeList  = l;
    }
}

/*
 *  Builds a dijkstra shortest path from the starting vertex to the target
 *  vertex for the specified actor. Note that the travel graph is not static,
 *  but is dependent upon actor's knowledge of locations and its ability to 
 *  perceive connectors.
 */
class DijkstraPath: object
{
    /* the vertex of the initial entry object */
    startingVertex  	  	= nil
    targetVertex           	= nil

    entryList       	  	= []
    reachableEntryList     	= []
    shortestPathEntryList   = []
    shortestPathVertexList 	= []
    travelLinkList          = []
    
    /*
     *  Creates the initial entry, performs the dijkstra algorithm, then
     *  separates the results into various useful lists. 
     */        
    doMain()
    {
        local e;

        /*
         *  The addToEntryList() method will create a new
         *  default entry and add it to the path entry list.
         */
        e = addToEntryList( startingVertex );

        /* Indicate that this is the path starting entry */
        e.value = 0;

        /*
         *  We loop through the dijkstra algorithm until there are no more
         *  unknown vertices. This approach attempts to avoid stack
         *  overflow due to recursion.
         */
        while( dijkstra ) {}

        /* remove any entries not reachable by s */
        reachableEntryList = getNonMaxValues();

        /* get the entry for v using our modified list */
        e = getEntryByVertex( targetVertex, reachableEntryList );

        /* build the list in order from s to v */
        while( e != nil )
        {
            buildTravelEdgeListOnPredecessor(e);
            shortestPathEntryList = shortestPathEntryList.prepend(e);
            e = e.predecessor;
        }

        /*
         *  Build the vertex list from the shortest
         *  path entry list.
         */
        foreach (local entry in shortestPathEntryList)
        {
            shortestPathVertexList += entry.vertex;
        }

        /*
         *  Build the travel link list from the shortest
         *  path entry list.
         */
        for (local i = 1; i < shortestPathEntryList.length(); ++i)
            travelLinkList += new TravelLink(
                shortestPathEntryList[i].vertex,
                shortestPathEntryList[i + 1].vertex,
                shortestPathEntryList[i].travelEdgeList);
    }

    dijkstra()
    {
        local v, adjEntryList, edgeValue;

        /*
         *  From the set of vertices for which (e.known == nil), select
         *  the vertex having the smallest tentative value.
         */
         v = getMinUnknownEntry();

        /*
         *  If we have no unknown vertices we are done.
         */
        if ( v == nil )
            return nil;

        /*
         *  Set vertex v.known to true.
         */
        v.known = true;

        /*
         *  We are building the path entry list with each iteration
         *  of dijkstra(). Build the entry's adjacent entry list for
         *  this vertex.
         */
        adjEntryList = buildAdjacentEntryList( v );

        /*
         *  For each vertex w adjacent to v for which (w.known == nil),
         *  test whether the tentative value w.value > v.value + c(v,w).
         *  If it is, set w.value = v.value + c(v,w) and set
         *  w.predecessor = v.
         */
        foreach (local w in adjEntryList)
        {
            if (w.known == nil)
            {
                local tot;

                edgeValue = computeEdgeValue( v, w );

                tot = v.value + edgeValue;

                /*
                 *  If tot is smaller than either the 
                 *  v.value or the edgeValue then we
                 *  really want tot to be INFINITY.
                 */
                if ( tot < v.value && tot < edgeValue )
                    tot = INFINITY;

                if ( w.value > tot )
                {
                    w.value = tot;
                    w.predecessor = v;
                }
            }
        }

        /* continue with the next iteration of the algorithm */
        return true;
    }

    /*
     *  Retrieve the entry with known == nil that has
     *  the minimum value.
     */
    getMinUnknownEntry()
    {
        local unknownList;

        /* make a list of all unknown entries */
        unknownList = entryList.subset({e: e.known == nil});

        /* if the list is empty we signal that we are done */
        if (unknownList.length() == 0)
            return nil;

        /*
         *  Retrieve the first entry from the unknown list that
         *  has the smallest value.
         */
        return unknownList.sort(nil, {a, b: a.compareValueOf(b)})[1];
    }

    /*
     *  Builds the entry's adjacent entry list.
     */
    buildAdjacentEntryList( e ) 
    {
        local destInfoList;

        /* 
         *  We call getExitsList() on the actor to retrieve 
         *  the destination information for this entry based
         *  on actor knowledge and perception. The destInfoList
         *  is the same one created by exitLister, consisting of
         *  DestInfo class objects.
         */
        destInfoList = actor_.getExitsList(e.vertex, true);

        e.destInfoList = destInfoList;

        /* get only those elements destinations are known to this actor */
        destInfoList = destInfoList.subset({x: x.dest_ != nil});

        /*
         *  Assign the list to our entry. Each entry, except the last,
         *  will keep track of its own destination information.
         */
        e.knownDestInfoList = destInfoList;

        foreach (local destInfo in destInfoList)
        {
            /*
             *  We get the corresponding entry for each destination of
             *  the destInfoList. If the entry does not already
             *  exist in the path entry list then addToEntryList()
             *  will create a new entry and add it to the path
             *  entry list.
             */
            e.adjacentEntryList += addToEntryList( destInfo.dest_ );
        }

	    /* return the adjacent entry list for this entry */
	    return e.adjacentEntryList;
    }

    /*
     *  Loads entries to the path entry list. First the
     *  list is searched by vertex. If a match is found then we
     *  simply return the entry. If no match is found we instantiate
     *  an entry, add it to the list, then return the new entry.
     */
    addToEntryList( v )
    {
        local e;

        /*
         *  If we don't find an entry for this vertex,
         *  instantiate an entry for it and add the entry
         *  to the path entry list.
         */
        e = getEntryByVertex( v );
        if ( e == nil )
        {
            e = new Entry( v, nil, INFINITY );
            entryList += e;
        }

        /* return the entry */
        return e;
    }

    /*
     *  Search the specified entry list for the entry matching
     *  this vertex. If no entry list is specified as a second 
     *  argument, the method searches the path entryList.
     *  If we don't find an entry for this vertex
     *  return nil.
     */
    getEntryByVertex( v, [args] )
    {
        local eList;

        if ( args.length() == 1 ) 
            eList = args[1];
        else eList = entryList;

        return eList.subset({e : e.vertex == v}).car();
    }

    /*
     *  Provide a value for the edge between vertices v and w.
     *  The default is to assume that the edge is unweighted.
     */
    computeEdgeValue( v, w )
    {
        return 1;
    }

    /*
     *  Retrieve only the entries whose values are less than
     *  INFINITY. When called after doMain() this produces
     *  a list of entries for which the starting vertex has
     *  a path.
     */
    getNonMaxValues()
    {
        return entryList.subset({e: e.value != INFINITY});
    }

    /*
     *  For each entry's predecessor we produce a list of directions that
     *  lead from the predecessor to the entry. We then assign the list to
     *  the predecessor so that for a given entry along our path we can know
     *  what travel directions lead to the next entry along the path.
     */
    buildTravelEdgeListOnPredecessor(entry)
    {
        local pred = entry.predecessor;

        /* if the entry has no predecessor we're done */
        if (pred == nil)
            return;

        foreach (local destInfo in pred.knownDestInfoList)
        {
            if (destInfo.dest_ == entry.vertex)
            {
                pred.travelEdgeList += destInfo.dir_;
                pred.travelEdgeList += destInfo.others_;
            }
        }
    }

    /*
     * A path requires an actor, a starting location, and a target location
     */
    construct( actor, start, target )
    {
        /* set the actor who is going to traverse this path */
        actor_ = actor;

        /* set the path starting and target vertices */
        startingVertex   = start;
        targetVertex     = target;

        /* build the shortest path from start to target */
        doMain();
    }
}

modify BasicLocation
{
    /* let the actor handle its own familiarity */
    actorIsFamiliar(actor) { return actor.locationIsFamiliar(self); }
}

modify Actor
{
    locationMemory = nil

    /* 
     *  Determines whether the location is familiar to the actor. An
     *  actor may be familiar with its own home, for instance. This 
     *  method should return true if the actor is familar with the 
     *  location; nil otherwise. By default no location is familiar 
     *  to the actor until the locationMemory table is loaded.
     */
    locationIsFamiliar(loc) { return locationMemory[loc]; }

    /*
     *  Sets the value for this location in the location memory table.
     *  The values should be true, the actor knows the location, or nil, 
     *  the actor doesn't know the location.
     */
    setLocationIsFamiliar(loc, val) { locationMemory[loc] = val; }

    travelBlocked(loc, dir) { return nil; }

    /*
     *  Returns a list of DestInfo objects that represent the information
     *  that this actor has regarding the destinations of the travel directions
     *  available from loc.
     */
    getExitsList(loc, [args])
    {
        local locIsLit;
        local destList;
        local showDest = true;
        local computingPath = args.car();

        locIsLit = loc.wouldBeLitFor(self);

        /* run through all of the directions used in the game */
        destList = new Vector(Direction.allDirections.length());
        foreach (local dir in Direction.allDirections)
        {
            local conn;

            if (travelBlocked(loc, dir))
                continue;

            /* 
             *   If the actor's location has a connector in this
             *   direction, and the connector is apparent, add it to the
             *   list.
             *   
             *   If the actor is in the dark, we can only see the
             *   connector if the connector is visible in the dark.  If
             *   the actor isn't in the dark, we can show all of the
             *   connectors.  
             */
            if ((conn = loc.getTravelConnector(dir, self)) != nil
                && (computingPath || (conn.isConnectorApparent(location, self)
                && (locIsLit || conn.isConnectorVisibleInDark(loc, self)))))
            {
                local dest;
                local destName = nil;
                local destIsBack;

                /* 
                 *   We have an apparent connection in this direction, so
                 *   add it to our list.  First, check to see if we know
                 *   the destination. 
                 */
                dest = conn.getApparentDestination(loc, self);

                /* note if this is the "back to" connector for the actor */
                destIsBack = (conn == lastTravelBack);

                /* 
                 *   If we know the destination, and they want to include
                 *   destination names where possible, get the name.  If
                 *   there's a name to show, include the name.  
                 */
                if (dest != nil
                    && showDest
                    && (destName = dest.getDestName(self, loc)) != nil)
                {
                    local orig;
                    
                    /* 
                     *   If this destination name already appears in the
                     *   list, don't include this one in the list.
                     *   Instead, add this direction to the 'others' list
                     *   for the existing entry, so that we will show each
                     *   known destination only once. 
                     */
                    orig = destList.valWhich({x: x.destName_ == destName});
                    if (orig != nil)
                    {
                        /* 
                         *   this same destination name is already present
                         *   - add this direction to the existing entry's
                         *   list of other directions going to the same
                         *   place 
                         */
                        orig.others_ += dir;

                        /* 
                         *   if this is the back-to connector, note it in
                         *   the original destination item 
                         */
                        if (destIsBack)
                            orig.destIsBack_ = true;

                        /* 
                         *   don't add this direction to the main list,
                         *   since we don't want to list the destination
                         *   redundantly 
                         */
                        continue;
                    }
                }

                /* add it to our list */
                destList.append(new DestInfo(dir, 
                    dest, destName, destIsBack));
            }
        }

        /* return the list */
        return destList.toList();
    }

    /*
     *  Build a location memory table for this actor
     */
    initializeActor()
    {
        inherited();

        /* create a location memory lookup table for this actor */
        locationMemory = new LookupTable();
    }
}