#charset "us-ascii"

/* NOTE:
 *
 * #define __KNOWLEDGE to test knowledge-checking. (This requires the
 * t3knowledge.t module, included in the t3RAP archive.)
 *
 */

/*

TADS-3 port and revision/extension
Copyright (c) 2002-2004 by Steve Breslin
(email: versim@hotmail.com)

originally based on:

Tads-2 RAP 1.0 Kernel
Copyright (c) by Nate Cull

Thanks to Mike Roberts and to George Karatheodoris for advice on
optimizing the main processor, Rapper.rFind().


License:

You can use this material however you want, but if you decide to
publicize anything that uses, #include's or otherwise borrows from
this material, you have to make your improvements publicly available,
and advertise them to the IF community.

That way, you will help keep this up to date for everybody else, and
everybody else will help keep it up to date for you.

You may redistribute this verbatim or in modified form only if you
keep the copyrights and license intact.

Feel encouraged to release your source code along with your game,
though this isn't a requirement.


Kernel information:
-------------------

  Main decision engine, planbase definition, and support.

Defines

   rDo, rBe: enumerators (we call them "opcodes").
   Rapper: the main "mix-in" class, to be inherited by RAP actors.
   rStackEntry: service object for the RAP processor
   rAnimateDaemon: for easy animation of RAP actors.
                   instances are created by Rapper.rAnimate().
   rRTAnimateDaemon: like rAnimateDaemon, but for realtime
   rPseudoRTAnimateDaemon: like rRTAnimateDaemon, but executes actions
                           only when a player turn is taken.
   rProxyFuse: service fuse for rPseudoRTAnimateDaemon.

   libGlobal modification: adds properties for tracking RAP information
   rStep: the base class for plan objects
   rToStr: function
   debugging verb and relevant modifications.
   time switching verbs.
   setAnimateDaemons: a service object for setting/resetting all RAP
                      daemons of all RAP actors.

*/

#include <adv3.h>
#include <en_us.h>

/* These "opcodes" are used to indicate step type in plans.
 *
 * The two opcodes are rBe and rDo. rBe indicates a plan (i.e.,
 * a condition to be met); rDo indicates an action to be executed.
 */

enum rBe, rDo;

/* Class: Rapper
 *
 *  This mix-in class defines the core RAP methods.
 *
 *  Use multiple inheritance to use it - Rapper doesn't define any
 *  Actor bahaviors, so you can use it with whatever actor class you
 *  have. In games using the standard adv3 library, you would define
 *  your NPCs as Rapper, Person.
 *
 *  (To minimise namespace conflicts, RAP methods and properties
 *  are prefixed with either r or rap.)
 *
 *   methods:
 *
 *   rAnimate           the user-interface. Automates RAP behavior.
 *   rapAct             asks rFind to supply an action, & executes it.
 *   rFind              the core search algorithm. returns an action.
 *   rAddPlans          rFind utility, adds plans to the rStack.
 *   rStampEntry        rFind utility, finalizes an rStack entry.
 *   rIdleCode          what the actor does when no action works.
 *   rToggleTrace       toggles decision tracing, for debug only.
 *
 *   properties:
 *
 *   myPlan             current plan (for use with callAnimateDaemon())
 *   myParam            current parameter (for use with ")
 *   rPlanPathCache     records the most recently resolved plan path.
 *   rSwitchableAnimateDaemon realtime/pseudo-turn-based/turn-based.
 *   rCurAnimateDaemon  simply records the current animation daemon.
 *   rTrace             flag for decision tracing, for debug only.
 *
 */

class Rapper: object

    /* A service method for calling rAnimate based on user-defined
     * (and easily switchable) plans and parameters. This is called
     * on each RAP actor by setAnimateDaemons().
     */
    callAnimateDaemon() {
        if (myPlan && myParam)
            rAnimate(myPlan, myParam);
    }
    myPlan = nil
    myParam = nil

    /* rAnimate: This is the user interface. You can call this method
     * with a goal and parameter to animate your Rapper. The Rapper
     * will pursue the goal until it is accomplished. When the goal is
     * accomplished, the daemon automatically removes itself.
     *
     * (Animation daemons are defined below, after Rapper and
     * rStackEntry.)
     */
    rAnimate(rstep, param) {
        if (curAnimateDaemon)
            curAnimateDaemon.removeEvent();
        if (rSwitchableAnimateDaemon == rRTAnimateDaemon)
            curAnimateDaemon = new rRTAnimateDaemon(self,rstep,param);
        else if (rSwitchableAnimateDaemon == rPseudoRTAnimateDaemon)
            curAnimateDaemon =
                       new rPseudoRTAnimateDaemon(self, rstep, param);
        else
            curAnimateDaemon = new rAnimateDaemon(self, rstep, param);
    }

    /* This property can be set to rPseudoRTAnimateDaemon or
     * rRTAnimateDaemon. This switches which animation daemon the RAP
     * actor will use. Note that you need to call rAnimate (or
     * callAnimateDaemon) after switching this value, for the change to
     * take effect.
     */
    rSwitchableAnimateDaemon = rAnimateDaemon

    /* curAnimateDaemon keeps track of the daemon currently animating
     * this Rapper.
     */
    curAnimateDaemon = nil

    /* rapAct: this function animates a RAP actor for one complete
     * action cycle (one turn).
     *
     * This calls rFind(). If an action is returned, it is executed.
     * Otherwise, rFailedAction.rAction() is executed.
     *
     * This is a lower-level function; rAnimate should be used in
     * most cases.
     */

    rapAct(rstep, param) {
        local x, retRStep, retParam;

        /* First, we call the main Rap plansearch method (rFind) to
         * find an appropriate action.
         */
        x = rFind(rstep, param);

        /* Now we run the action that was returned. If we have an
         * action, x will be a list: [rStep, parameter].
         *
         * A nil x value means we're calling the rFailedAction rStep.
         */
        if (dataType(x) == TypeList) {
            retRStep = x[1];
            retParam = x[2];
        }
        else {
            retRStep = rFailedAction;
            retParam = [rstep, param];
        }
        retRStep.rAction(self, retParam);
    }

    /* rPlanPathCache records the entire plan-path resolved by the most
     * recent call to rFind, if libGlobal.rCachePlanPath is true.
     */
    rPlanPathCache = nil

    /* rFind: the main processor. We backwards-chain from the master
     * plan to an accomplishable sub-goal (i.e., precondition). The
     * main processor is a plans loop containing a steps loop. The
     * inner (steps) loop iterates through each step in each given
     * plan, adding non-redundant and not-currently-satisfied steps
     * to the list of plans to iterate through. This goes on until
     * 1) we're out of options or 2) a (sub-)plan is immediately
     * actionable (i.e., until the current step is an rDo step).
     */

    rFind (cond, param) {
        local x, // a multi-use service variable
        rStack,  // local vector: stack of rStackEntries (plan entries)
        rStLookupTab; // local LookupTable: this contains information
                      // redundant with rStack, but it optimizes the
                      // speed of duplicate plan detection.

        libGlobal.enableSenseCache();

        try {

#ifdef __DEBUG
        if (rTrace)
            "\bRAP decision trace for <<name>>:
            master plan = (<<rToStr(cond)>>, <<rToStr(param)>>)\n";
#endif // __DEBUG

        /* First we test if we're currently fulfilling the rStep (i.e.,
         * condition, plan) for the given param.
         *
         * If we're currently fulfilling this condition, there's
         * nothing left to do. We can simply return nil.
         */
        if (cond.rIsTrue(self, param)) {

#ifdef __DEBUG
            if (rTrace)
                "<<name>> has fulfilled master-plan: <<rToStr(cond)>>(
                <<name>>, <<rToStr(param)>>)\n";
#endif // __DEBUG

            return nil;
        }

        /* First we try executing the next step in the rPlanPathCache,
         * but only if this cache has our master plan as its last
         * entry -- otherwise we're being asked to resolve a different
         * goal, in which case we'll have to start afresh.
         */
        if (rPlanPathCache && rPlanPathCache.length &&
            rPlanPathCache[rPlanPathCache.length] == [cond, param]) {

            local curPlan, curParam;

            /* On the off chance that the first element(s) of the
             * rPlanPathCache are currently satisfied, we want
             * to remove them first.
             * 
             * NOTE: THAT MAY BE RARE ENOUGH THAT THIS ACTUALLY WASTES
             *       TIME ON AVERAGE!
             *
             * WARNING: IN ADDITION TO BEING DUBIOUS, THIS BLOCK IS
             *          BROKEN/BUGGY. CURRENTLY COMMENTED OUT.
             *
            while (rPlanPathCache[1][1].rIsTrue(self,
                                             rPlanPathCache[1][2])) {
                rPlanPathCache.removeElementAt(1);
                if (!rPlanPathCache.length())
                    break;
            }
             */

            curPlan = rPlanPathCache[1][1];
            curParam = rPlanPathCache[1][2];

#ifdef __DEBUG
            if (rTrace)
                "Current rPlanPathCache is consistent with current
                master goal.\n
                Checking if [<<rToStr(curPlan)>>, <<rToStr(curParam)>>]
                is currently actionable....\n";
#endif // __DEBUG

            /* Now we cycle through the plans defined by the first
             * entry in the rPlanPathCache, i.e., the "current action."
             * 
             * Our goal here is to determine if we can reach the
             * action step (i.e., the rDo step) of any plans defined
             * by the current action. If we can, we return the action;
             * otherwise, we continue processing as usual.
             */
            local plans = curPlan.rGetPlans(self, curParam);

            /* There can be more than one plan defined for a given
             * action, so we iterate through each of them.
             */
            for (local i = 1 ; i <= plans.length ; i++) {
                local plan = plans[i];

                for (local j = 1 ; j <= plan.length ; j += 3) {

                    if (plan[j] == rDo) { // this plan works.

#ifdef __DEBUG
                        if (rTrace)
                            "Success on previously cached action step:
                            [<<rToStr(plan[j+1])>>,
                            <<rToStr(plan[j+2])>>].
                            Returning this value.\n";
#endif // __DEBUG

                        /* We're going to execute this action, so
                         * remove it from the rPlanPathCache.
                         */
                        rPlanPathCache.removeElementAt(1);

                        /* Return the resolved action. */
                        return ([plan[j+1],plan[j+2]]);
                    }

                    /* check if the current step is now satisfied */
                    if (!plan[j+1].rIsTrue(self, plan[j+2])) {

                        /* It's not satisfied, so this isn't a
                         * currently actionable plan. We'll break
                         * out of the j loop, continuing with the
                         * i loop.
                         */
                        break;
                    }
                    /* good so far: continue the j loop */
                }
            }
#ifdef __DEBUG
            if (rTrace)
                "Failure: previously cached action no longer
                actionable. Now continuing RAP action resolution
                as normal.\n";
#endif // __DEBUG
        }
            
        /* If we made it this far, plan cacheing didn't go for whatever
         * reason. So, next we check if there's at least one plan
         * defined for this rStep, given the current actor and
         * parameter.
         *
         * If not, we return. (But if it's a list with more than zero
         * elements, we assume it's a viable plan.)
         */
        x = cond.rGetPlans(self, param);
        if (dataType(x) != TypeList || !x.length()) {

#ifdef __DEBUG
            if (rTrace)
                "No plans found for <<name>>'s master-plan:
                <<rToStr(cond)>>(<<name>>, <<rToStr(param)>>)\n";
#endif // __DEBUG
            return nil;
        }

        /* If we made it this far, we initiate standard RAP processing:
         *
         * We'll do some initialization, then enter a loop-within-loop
         * action resolution algorithm.
         *
         * First, initialize the rStack, placing the master plan at
         * index 1.
         *
         * Because this is the master plan, and therefore the first
         * entry in the rStack, it has no parent index or parent-plan
         * index. (I.e., The third and fourth arguments for the
         * rStackEntry construct() are both nil.)
         *
         * An explanation of the rStack entries is given below, in the
         * header-comment for the rStackEntry object.
         *
         * We maintain a parallel rStLookupTab (a lookup table which
         * mirrors some data from the rStack), to make more efficient
         * the detection of redundant plans.
         */
        rStack = new Vector(30, [new rStackEntry(cond,param,nil,nil)]);
        rStack[1].planSet = x;
        rStLookupTab = new LookupTable(20,20);
        rStLookupTab[[cond, param]] = true;

        /* Add a slot to the rStack for each plan-group of the master
         * plan.
         *
         * In this case, the third argument, 'parent,' is 1, since the
         * parent for the now newly added plans is the master plan (at
         * index 1 of the rStack).
         *
         * We pass 'rStack' as an argument so that the rAddPlans method
         * can directly alter this local vector.
         */
        rAddPlans(rStack, x.length(), 1);

        /* Now that we've made our preliminary checks and set up the
         * rStack, we can enter the main process, which iterates
         * through all the steps of all the plans, expanding the
         * rStack when we find a step unsatisfied, until we reach an
         * rDo opcode, or we run out of options. (When a step is
         * unsatisfied, it registers as a new plan in the rStack, so
         * its satisfaction is a new plan. Thus we backwards-chain from
         * the master plan to a currently satisfiable sub-plan.)
         *
         * Thus begins...
         ----------------
         * the plans loop. 'i' is an index of the rStack. We begin
         * iteration at 2, skipping the master plan, which we have
         * already checked and expanded.
         *
         * Note that rStack will increase in length as new entries are
         * added to the stack. (This happens inside the steps loop, via
         * the rAddPlans() method call there.)
         *
         * The outer plans loop simply sets some local variables which
         * don't need to be altered during the steps loop; but this is
         * really a strict and bare loop-within-loop structure.
         */
        for (local i = 2 ; i <= rStack.length() ; i++) {
            local
              thisplan,    // indexes which of its parent's plans it is
              parentcond,  // this plan's parent's cond
              parentparam; // this plan's parent's parameter

            /* Read next plan off stack...
             */
            x = rStack[i];

            /* ...and set local variables accordingly.
             */
            thisplan = x.plan;
            parentcond = x.parent.cond;
            parentparam = x.parent.param;

            /* Thus begins...
             ----------------
             * the steps loop, conveniently marked, for use with
             * continue and break commands from within the loop.
             *
             * We go through the number of iterations necessary to go
             * through all the steps in a given plan.
             *
             * If we reach an rDo step, it's the end of the line.
             * Otherwise it's an rBe step: we check if we should add
             * it to the rStack, to be iterated in the plans loop.
             *
             * To find the number of necessary iterations, we divide
             * the number of elements in the plan by 3, as there are
             * three elements per plan-step: [opcode, rStep, p].
             */

            x = x.parent.planSet[thisplan];

            stepsLoop:

            for (local j = 1, local y = x.length/3 ; j <= y ; j++) {
                local
                    thisopcode,  // the first element in this step
                    thisstep,    // the second element in this step
                    thisparam;   // the third element in this step

                /* We set our local variables. x is the plan, j is the
                 * step in the plan. The following isolates elements in
                 * the j-step of the plan.
                 */
                thisopcode = x[(j-1)*3+1]; // first element in the step
                thisstep = x[(j-1)*3+2];   // second element in step
                thisparam = x[(j-1)*3+3];  // third element in the step

                /* this allows function pointers as param elements in
                 * plan lists.
                 */
                if (dataType(thisparam) == TypeObject &&
                    thisparam.ofKind(AnonFuncPtr))
                    thisparam = (thisparam)();

#ifdef __DEBUG
                if (rTrace)
                    "\ntrying <<rToStr(thisopcode)>> <<
                    rToStr(thisstep)>> <<rToStr(thisparam)
                    >> for <<rToStr(parentcond)>> <<
                    rToStr(parentparam)>>";
#endif // __DEBUG
                if (thisopcode == rDo) { // Got an rDo opcode.
#ifdef __DEBUG
                    if (rTrace)
                        "...firing\n";
                    if (rTrace) {

                        "Action resolved:
                        [<<rToStr(rStack[i].parent.cond)>>,
                        <<rToStr(rStack[i].parent.param)>>] \n
                        Full backwards path: [";

                        for (local par = rStack[i].parent ;
                             par ; par = par.parent) {
                            "[<<rToStr(par.cond)>>,
                            <<rToStr(par.param)>>]";
                            if (par.parent) ", ";
                            else "].\n";
                        }
                    }
#endif // __DEBUG

                    /* if libglobal.rCachePlanPath is true, we want to
                     * record the entire path, so that we can try to 
                     * use this pre-recorded path next time rFind is
                     * called, rather than calculating the entire path
                     * again "from scratch". The algorithm is the same
                     * one we use immediately above for debug reports.
                     */
                    if (libGlobal.rCachePlanPath) {

                        /* if we've reached this code block, our
                         * current rPlanPathCache is either not defined
                         * or didn't work out. So we reset the cache.
                         */
                        rPlanPathCache = [];

                        /* Trace the resolved action's parents through
                         * the stack, and record each of them in the
                         * rPlanPathCache.
                         */
                        for (local par = rStack[i].parent ;
                             par ; par = par.parent)
                            rPlanPathCache += [[par.cond, par.param]];
                    }

                    return([thisstep,thisparam]); // The end!
                }

                else { // Got an rBe opcode.

                    /* First we find out if the step is already being
                     * satisfied. If it is, then we don't need to add
                     * it to the rStack, but we will remove any other
                     * plans in the rStack aimed at making it true.
                     */
                    if (thisstep.rIsTrue(self, thisparam)) {
#ifdef __DEBUG
                        if (rTrace)
                            "...satisfied\n";
#endif // __DEBUG

                        /* scan the rstack for entries which have the
                         * same parent as the satisfied entry.
                         * for each one of these siblings found, remove
                         * all child entries of that sibling.
                         * 
                         * This process is a.k.a. nephew killing:
                         */
                        foreach (local ent in rStack)
                            if (ent.parent == rStack[i].parent)
                                foreach (local entr in rStack)
                                    if (entr.parent == ent)
                                        rStack.removeElement(entr);

                        /* since this step is already satisfied, we
                         * don't need to process it. So we now just
                         * begin another iteration of stepsloop.
                         */
                        continue stepsLoop;
                    }

#ifdef __KNOWLEDGE
                    /* If knowledge checking is enabled, we make sure
                     * that the parameter or parameters are known to
                     * the actor. If not, this is not a viable plan.
                     *
                     * Note that this requires the included knowledge
                     * module.
                     */
                    if (
                         (
                             dataType(thisparam) == TypeObject &&
                             !scopeList.indexOf(thisparam)
                         )
                         ||
                         (
                             dataType(thisparam) == TypeList &&
                             (
                              !scopeList.indexOf(thisparam[1]) ||
                              !scopeList.indexOf(thisparam[2])
                             )
                         )
                       )
                        break stepsLoop;
#endif // __KNOWLEDGE

                    /* As this step isn't satisfied presently, its plan
                     * or plans are now candidates for being added to
                     * the rStack, and so to become sub-goal(s).
                     *
                     * Before we decide to add it to the rStack, we
                     * make sure this rStep is not already present in
                     * the rStack. This culls out redundance. We check
                     * the LookupTable that we maintain in parallel
                     * to the rStack for maximum efficiency.
                     */
                    if (rStLookupTab[[thisstep, thisparam]]) {
                        /* this rStep(a,p) is already present in
                         * the rStack.
                         */
#ifdef __DEBUG
                        if (rTrace)
                            "...dupe\n";
#endif // __DEBUG
                        break stepsLoop;
                    }

                    /* At this point, we can make the computation-heavy
                     * call to the planbase:
                     */
                    x = thisstep.rGetPlans(self,thisparam);

                    /* Now we make sure that there's at least one plan
                     * defined by this rStep, given the a and p
                     * arguments. (And make sure it's not an empty
                     * plan.)
                     */
                    if(dataType(x) != TypeList || !x.length()) {
#ifdef __DEBUG
                        if (rTrace)
                            "...empty plan\n";
#endif // __DEBUG
                        rStampEntry(rStack[i], thisstep, thisparam, x);
                        rStLookupTab[[thisstep, thisparam]] = true;
                        break stepsLoop;
                    }

                    /* It's passed our tests, so we add it to the
                     * rStack, to be iterated in the plans loop.
                     * (And we add it to the parallel lookup
                     * table.)
                     */
                    rAddPlans(rStack, x.length(), i);
                    rStampEntry(rStack[i], thisstep, thisparam, x);
                    rStLookupTab[[thisstep, thisparam]] = true;
                    break stepsLoop; // break back to plans loop
                    
                } // end rBe
            } // end of steps loop
        } // end of plans loop
        return nil; // if we reach this, no viable path was found
        } // try
        finally {
        libGlobal.disableSenseCache();
        }
    }

    /* rAddPlans: add plan entries for a specified parent plan.
     */

    rAddPlans(rStack, numPlans, parent) {
        local i;

        for (i = 1 ; i <= numPlans ; i++) {
            local x = new rStackEntry(nil, nil, rStack[parent], i);
            rStack.append(x);
        }
    }

    /* rStampEntry: records condition and param for this rStackEntry
     * object, and records the plan list (planSet), which is the
     * same as cond.rGetPlans(self, param). This is a time-consuming
     * calculation, so we store it in the entry.
     */

    rStampEntry(entry, cond, param, x) {
        entry.cond = cond;
        entry.param = param;
        entry.planSet = x;
    }

    /* rIdleCode is called by rNoAction, when the RAP search finds no
     * plans. You might want to remove the unworkable animation daemon,
     * or switch to a different plan here.
     */
    rIdleCode() {
        if (!location && curAnimateDaemon) {
            curAnimateDaemon.removeEvent();
            curAnimateDaemon = nil;
            return;
        }
#ifdef __DEBUG
        "[\^<<theName>> in <<location.theName>> idles.]\n";
#endif // __DEBUG
        rIdleMessage();
        return;
    }

    rIdleMessage() { }

#ifdef __DEBUG

    /* rTrace: debug parameter, for tracing the kernel's process.
     */
    rTrace = nil

    /* rToggleTrace toggles tracing on or off. */
    rToggleTrace() {
        /* This toggles decision tracing mode.
         */
        rTrace = !rTrace;
    }

#endif // __DEBUG

; /* class Rapper */

/* rStackEntry encapsulates a data entry in the rStack.
 *
 * There are five pieces of information:
 *
 * - cond: simply the rStep for this entry.
 *
 * - parameter: this is the second value passed to the rStep, the 'p'
 *              of rHave(a, p) for example. (The first value, 'a'
 *              (actor) is simply the Rapper object which is handling
 *              the rStack at the moment; we don't need to record
 *              that.)
 *
 * - parent: the rStackEntry that is this entry's parent, i.e., the
 *           plan that placed this sub-plan.
 *
 * - plan: parent can have multiple plans. this indexes which one.
 *
 * - planSet: the plan-list returned by cond.rGetPlans(a, param). We
 *            record this because recalculating it is time consuming.
 */
class rStackEntry: object
    construct (c, prm, par, pl) {
        cond = c;
        param = prm;
        parent = par;
        plan = pl;
    }
    cond = nil
    param = nil
    parent = nil
    plan = nil
    planSet = nil
;

/* In order to carry out the RAP action, we will normally need to call
 * rapAct over more than one turn. We automate this with a special
 * daemon given the actor, rStep (the master goal), and its parameter.
 *
 * This daemon is instantiated by the Rapper.rAnimate() user-interface
 * method.
 */
class rAnimateDaemon: Daemon
    actor_ = nil
    step_ = nil
    param_ = nil
    construct(actor, step, param) {
        actor_ = actor;
        step_ = step;
        param_ = param;
        eventManager.addEvent(self);
        nextRunTime = Schedulable.gameClockTime + interval - 1;
    }

    callMethod() {

        if (!actor_.location) {
            removeEvent();

#ifdef DEBUG__

            /* report a message, if, for whatever reason, a
             * RAP daemon is running on an object which has been moved
             * to nil.
             */
            "DEBUG MESSAGE: (<<actor_.theName>>'s location is nil;
            this Rapper's animation daemon has been removed.\n";
#endif // __DEBUG
            return;
        }

        /* set our globals */
        libGlobal.rCurAnimateDaemon = self;
        libGlobal.rIsRapping = true;

        /* we use the try/finally structure so that our globals will
         * always be reset in the finally stage, no matter how the
         * try stage is exited.
         */
        try {

            /* execute the daemon. We find and execute the next RAP
             * action, using the lower-level method, rapAct. And we
             * wrap the call in a sense context and action environment.
             */
            withActionEnv(EventAction, gPlayerChar,
                {: callWithSenseContext(source_, sense_,
                    {: actor_.rapAct(step_, param_) }) });

        }
        finally {

            /* We reset the globals back to nil. */
            libGlobal.rIsRapping = nil;
            libGlobal.rCurAnimateDaemon = nil;

            /* if we've fulfilled the master goal, we're done: remove
             * this animation daemon.
             */
            if (step_.rIsTrue(actor_, param_))
                removeEvent();
        }
    }

    /* The interval should be one move per turn (unless we perfer
     * realtime effects -- see rRTAnimateDaemon for realtime RAP
     * effects).
     */
    interval = 1
;

/* rRTAnimateDaemon: for handling realtime RAP action.
 */
rRTAnimateDaemon: RealTimeDaemon
    actor_ = nil
    step_ = nil
    param_ = nil
    construct(actor, step, param) {
        actor_ = actor;
        step_ = step;
        param_ = param;
        realTimeManager.addEvent(self);
        eventTime = realTimeManager.getElapsedTime() + interval_;
    }
    callMethod() {
        if (!actor_.location) {
            removeEvent();

#ifdef DEBUG__

            /* report a message, if, for whatever reason, a
             * RAP daemon is running on an object which has been moved
             * to nil.
             */
            "DEBUG MESSAGE: (<<actor_.theName>>'s location is nil;
            this Rapper's animation daemon has been removed.\n";
#endif // __DEBUG
            return;
        }

        /* set our globals */
        libGlobal.rCurAnimateDaemon = self;
        libGlobal.rIsRapping = true;

        /* we use the try/finally structure so that our globals will
         * always be reset in the finally stage, no matter how the
         * try stage is exited.
         */

        /* We don't want realtime action getting too gummed up, so we
         * introduce a time-control so that not too many daemons can
         * fire concurrently or immediately consecutively.
         */
        realTimeManager.setElapsedTime(realTimeManager.getElapsedTime -
                                                                 4000);
        try {

            /* execute the daemon. We find and execute the next RAP
             * action, using the lower-level method, rapAct. And we
             * wrap the call in a sense context and action environment.
             */
            withActionEnv(EventAction, gPlayerChar,
                {: callWithSenseContext(source_, sense_,
                    {: actor_.rapAct(step_, param_) }) });
        }
        finally {

            /* We reset the globals back to nil. */
            libGlobal.rIsRapping = nil;
            libGlobal.rCurAnimateDaemon = nil;

            /* if we've fulfilled the master goal, we're done: remove
             * this animation daemon.
             */
            if (step_.rIsTrue(actor_, param_))
                removeEvent();            
        }
    }

    /* The interval is randomized, so that there's some variation
     * between execution delays. The specific randomization is
     * (rand(interval) + interval/2), so if interval is 5000, we
     * get a random delay between 2500 and 7499.
     */
    interval_() {
        return (rand(interval) + interval/2);
    }
    interval = 7000
;

rPseudoRTAnimateDaemon: rRTAnimateDaemon

    callMethod() {

        /* we let the proxy fuse re-initialize this daemon if it
         * decides that it wants to.
         */
        removeEvent();
        if (!actor_.location) {

#ifdef DEBUG__

            /* report a message, if, for whatever reason, a RAP
             * daemon is running on an object which has been moved to
             * nil.
             */
            "DEBUG MESSAGE: (<<actor_.theName>>'s location is nil;
            this Rapper's animation daemon has been removed.\n";
#endif // __DEBUG
            return;
        }

        /* we know we're going to perform a RAP process, which means
         * we want to slow down all the other pseudo RT animation
         * daemons, so as not to lock the game for too long. The easy
         * way to do this is to rewind the clock four seconds. On
         * systems where a single RAP process can exceed four seconds,
         * you might want to stretch this number. Of if you're doing
         * really processing-heavy plan computation. Four is a good
         * advisory for our immediate purposes.
         */
        realTimeManager.setElapsedTime(realTimeManager.getElapsedTime -
                                                               lagAdj);
        libGlobal.rCurAnimateDaemon = self;
        libGlobal.rIsRapping = true;
        try {
            local retRStep, retParam, x = actor_.rFind(step_, param_);
            if (dataType(x) == TypeList) {
                retRStep = x[1];
                retParam = x[2];
            }
            else {
                retRStep = rFailedAction;
                retParam = [step_, param_];
            }
            new rProxyFuse(actor_, retRStep, retParam, self);
        }
        finally {
            libGlobal.rIsRapping = nil;
            libGlobal.rCurAnimateDaemon = nil;
        }
    }

    lagAdj = 4000

    executeEvent() {
        callMethod();
    }
;

rProxyFuse: Fuse
    actor_ = nil
    step_ = nil
    param_ = nil
    callerDaemon_ = nil
    construct(actor, step, param, callerDaemon) {
        actor_ = actor;
        step_ = step;
        param_ = param;
        callerDaemon_ = callerDaemon;
        eventManager.addEvent(self);
        nextRunTime = Schedulable.gameClockTime;
    }
    callMethod() {
        if (!actor_.location) {
            removeEvent();

#ifdef DEBUG__

            /* report a message, if, for whatever reason, a
             * RAP daemon is running on an object which has been moved
             * to nil.
             */
            "DEBUG MESSAGE: (<<actor_.theName>>'s location is nil;
            this Rapper's animation daemon has been removed, as has its
            proxy fuse.\n";
#endif // __DEBUG
            return;
        }

        libGlobal.rCurAnimateDaemon = self;
        libGlobal.rIsRapping = true;
        try {
            /* only try an action if it's not true. if it is, someone
             * beat the actor to his goal, and the actor loses his
             * turn.
             */
            if (!step_.rIsTrue(actor_, param_))
                withActionEnv(EventAction, gPlayerChar,
                    {: callWithSenseContext(source_, sense_,
                        {: step_.rAction(actor_, param_) }) });
        }
        finally {
            libGlobal.rIsRapping = nil;
            libGlobal.rCurAnimateDaemon = nil;
            if (!(callerDaemon_.step_).rIsTrue((callerDaemon_.actor_),
                                             (callerDaemon_.param_))) {

                /* if the caller's step still isn't satisfied,
                 * re-initialize the daemon.
                 */
                callerDaemon_.eventTime =
                    realTimeManager.getElapsedTime() +
                    callerDaemon_.interval_();
                    realTimeManager.addEvent(callerDaemon_);
            }
        }
    }
;


/* We add three properties to libGlobal: rCurAnimateDaemon, rIsRapping,
 * which are used by RAP-animation daemons, and rCachePlanPath, a
 * boolean which indicates whether we want to use previously resolved
 * action paths where plausible, or instead recalculate the action path
 * each turn.
 *
 * If you are calling Rapper actions outside of the provided animation
 * daemons, the first two globals will not necessarily be available.
 */
modify libGlobal

    /* rCurAnimateDaemon is set to the current daemon during an
     * automated (rAnimate/rAnimateDaemon) action process.
     *
     * It is currently not used by RAP and extensions, but is
     * possibly useful. If, for example, you want to remove an
     * animation process during an action step, you will need to
     * know what is the current AnimateDaemon.
     */
    rCurAnimateDaemon = nil

    /* rIsRapping is set to true at the beginning of automated RAP
     * action processes, and reset to nil when the process finishes.
     *
     * This is used by the t3rapOrder module, so that our modifications
     * to doActionOnce() can check if the current action is resulting
     * from an automated RAP process. You may find other uses for this
     * boolean flag.
     */
    rIsRapping = nil // we use this in t3rapOrder.

    /* rCachePlanPath is nil by default. At present this is an
     * experimental option. If you set it to true, it may speed up
     * RAP processing considerably, on average. In the worst case,
     * it may slow processing slightly.
     */
    rCachePlanPath = nil
;

/*
 *    rStep class is the base class for planbase objects.
 *    Note that we override everything when writing an rStep.
 *
 *   methods
 *
 *      rTrue     condition's truth method; user defined
 *      rPlans    main planbase method, returns plan list for a given
 *                actor and parameter; user defined.
 *      rAction   the actual code for doing an action; user defined.
 *      rIsTrue   returns rTrue, but accepts lists, vectors, Classes,
 *                or anonymous functions as the parameter.
 *      rGetPlans returns rPlans, but accepts lists, vectors, Classes,
 *                or anonymous functions as the parameter.
 *
 *   properties
 *
 *      name      condition or action name (for debugging messages)
 */

class rStep: object

    name = 'UNNAMED rStep' // override for nicer debugging messages

    /* rTrue(actor, param) returns true or nil only, based on whether
     * or not the condition (goal) represented by this rStep is
     * currently being fulfilled.
     */

    rTrue (actor, param) {
#ifdef __DEBUG
        "ERROR. No rTrue method defined for <<name>>.";
#endif // __DEBUG
        return (nil);
    }

    /* rPlans(actor, param) returns a list of plans in this format:
     *
     * [
     *     [ rOpcode, rStep, param,    // three elements per entry
     *       rOpcode, rStep, param...],
     *     [ rOpcode... ]              // optional additional plans
     * ]
     *
     * This should be defined by any rStep in the planbase and
     * extensions, if the rStep wants to define necessary
     * sub-plans.
     *
     * This is only necessary to define when this rStep is going to
     * be referred to with an rBe opcode.
     */

    rPlans(actor, param) { }

    /* rAction(actor, parameter) must be defined if this rStep is
     * referenced by a rDo opcode, but otherwise it will not be called.
     *
     * It should make the actor perform the action that will satisfy
     * the condition (goal) represented by the rStep. So for example,
     * the rGet rAction(a,p) method executes:
     *
     *     nestedActorAction(a, Take, p);
     *
     * This makes the (a)ctor 'Take' the (p)arameter (or target
     * object).
     */
    rAction(a,p) {
#ifdef __DEBUG
        "\bError: <<a.name>> has no defined RAP action (<<name
        >> <<rToStr(p)>>).\b";
#endif // __DEBUG
    }

    /* rIsTrue(actor, parameter) accepts lists, vectors, Classes,
     * or anonymous functions as the parameter.
     * This generally won't need to be overridden by specific rSteps.
     */
    rIsTrue(actor, parameter) {
        if (!parameter)
            return rTrue(actor, parameter);

        if (dataType(parameter) == TypeObject &&
                                         parameter.ofKind(AnonFuncPtr))
            parameter = (parameter)();
        if ((parameter.ofKind(Vector) || parameter.ofKind(List)) &&
            (parameter.length() > 1)) {
            for(local i=1, local j=parameter.length() ; i <= j ; i++) {
                if (rTrue(actor,parameter[i]))
                    return true;
            }
            return nil;
        }
        if (parameter.isClass) {
            for(local obj = firstObj(parameter) ; obj ;
                          obj = nextObj(obj, parameter)) {
                if (rTrue(actor, obj))
                    return true;
            }
            return nil;
        }
        // else we assume parameter is simply an object
        return rTrue(actor, parameter);
    }

    /* rGetPlans(actor, parameter) accepts lists, vectors, Classes, or
     * anonymous functions as the parameter.
     * This generally won't need to be overridden by specific rSteps.
     */
    rGetPlans(actor, parameter) {
        if (!parameter)
            return rPlans(actor, parameter);

        if (dataType(parameter) == TypeObject &&
                                         parameter.ofKind(AnonFuncPtr))
            parameter = (parameter)();
        if (parameter.ofKind(Vector) || parameter.ofKind(List)) {
            local ret = [];
            for(local i=1, local j=parameter.length() ; i <= j ; i++) {
                ret += rPlans(actor,parameter[i]);
            }
            return ret;
        }
        if (parameter.isClass) {
            local ret = [];
            for(local obj = firstObj(parameter) ; obj ;
                          obj = nextObj(obj, parameter)) {
                ret += rPlans(actor, obj);
            }
            return ret;
        }
        // else we assume parameter is simply an object
        return rPlans(actor, parameter);
    }
;

/* rStep2P, or "rStep with two-element parameters" is useful for rSteps
 * which involve a relation between two objects: for example, the rStep
 * rObjOn, a plan for an object to be placed upon another object.
 */
rStep2P: rStep
    rIsTrue(actor, parameter) {
        if (!parameter)
            return rTrue(actor, parameter);

        /* first resolve either or both anonymous functions, as
         * necessary.
         */
        if (dataType(parameter[1]) == TypeObject &&
                                      parameter[1].ofKind(AnonFuncPtr))
            parameter[1] = (parameter[1])();
        if (dataType(parameter[2]) == TypeObject &&
                                      parameter[2].ofKind(AnonFuncPtr))
            parameter[2] = (parameter[2])();

        /* If parameter[1] is a vector or a list, iterate through it
         * and apply rIsTrue(a, p) to each instance of p[1].
         */
        if ((parameter[1].ofKind(Vector) ||
             parameter[1].ofKind(List))) {
            for(local i=1, local j=parameter[1].length();i<=j;i++) {
                if (rIsTrue(actor, [parameter[1][i], parameter[2]]))
                    return true;
            }
            return nil;
        }

        /* Do the same with parameter[2]. */
        if (parameter[2].ofKind(Vector) ||
             parameter[2].ofKind(List)) {
            for(local i=1, local j=parameter[2].length();i<=j;i++) {
                if (rIsTrue(actor, [parameter[1], parameter[2][i]]))
                    return true;
            }
            return nil;
        }

        /* If parameter[1] is a class, iterate through each instance of
         * the class, and apply rIsTrue to it.
         */
        if (parameter[1].isClass) {
            for(local obj = firstObj(parameter[1]) ; obj ;
                          obj = nextObj(obj, parameter[1])) {
                if (rIsTrue(actor, [obj, parameter[2]]))
                    return true;
            }
            return nil;
        }

        /* Do the same with parameter[2]. */
        if (parameter[2].isClass) {
            for(local obj = firstObj(parameter[2]) ; obj ;
                          obj = nextObj(obj, parameter[2])) {
                if (rIsTrue(actor, [parameter[1], obj]))
                    return true;
            }
            return nil;
        }

        // else we assume both parameters are simple objects
        return rTrue(actor, parameter);        
    }

    rGetPlans(actor, parameter) {

        if (!parameter)
            return (rPlans(actor, parameter));

        /* Resolve anonymous functions as needed. */
        if (dataType(parameter[1]) == TypeObject &&
                                      parameter[1].ofKind(AnonFuncPtr))
            parameter[1] = (parameter[1])();
        if (dataType(parameter[2]) == TypeObject &&
                                      parameter[2].ofKind(AnonFuncPtr))
            parameter[2] = (parameter[2])();

        /* If parameter[1] is a list or vector, apply rGetPlans to each
         * element.
         */
        if (parameter[1].ofKind(Vector) || parameter[1].ofKind(List)) {
            local ret = [];
            for(local i=1, local j=parameter[1].length();i<=j;i++) {
                ret += rGetPlans(actor,[parameter[1][i],parameter[2]]);
            }
            return ret;
        }

        /* Do the same for parameter[2]. */
        if (parameter[2].ofKind(Vector) || parameter[2].ofKind(List)) {
            local ret = [];
            for(local i=1, local j=parameter[2].length();i<=j;i++) {
                ret += rGetPlans(actor,[parameter[1],parameter[2][i]]);
            }
            return ret;
        }        

        /* If parameter[1] is a class, apply rGetPlans to each instance
         * of that class.
         */
        if (parameter[1].isClass) {
            local ret = [];
            for(local obj = firstObj(parameter[1]) ; obj ;
                          obj = nextObj(obj, parameter[1])) {
                ret += rGetPlans(actor, [obj, parameter[2]]);
            }
            return ret;
        }

        /* Do the same for parameter[2]. */
        if (parameter[2].isClass) {
            local ret = [];
            for(local obj = firstObj(parameter[2]) ; obj ;
                          obj = nextObj(obj, parameter[2])) {
                ret += rGetPlans(actor, [parameter[1], obj]);
            }
            return ret;
        }

        // else we assume parameter is simply an object
        return rPlans(actor, parameter);
    }
;

#ifdef __DEBUG

/* an easy service function, slightly easier than typing the call to
 * reflectionServices every time. This is only used in debugging
 * messages.
 */
rToStr(value) {
    "<<reflectionServices.valToSymbol(value)>>";
}
;

/* RapTrace: Debugging verb for RAP actors.
 * a convenient way of activating rToggleTrace().
 *
 * Rapper.rToggleTrace switches RAP trace mode on or off for the given
 * Rapper.
 */

DefineTAction(RapTrace);
VerbRule(RapTrace)
    'raptrace' dobjList
    : RapTraceAction
    verbPhrase = 'raptrace/raptracing (what)'
;

modify Thing
    dobjFor(RapTrace) {
        verify() {
            illogical('{That dobj/he} {is}n\'t something you can
                trace. Try tracing a RAP actor. ');
        }
    }
;

modify Rapper
    verDoTrace(actor) {}
    dobjFor(RapTrace) {
        preCond = [];
        verify() { }
        action() { rToggleTrace(); }
    }
    rToggleTrace() {
        rTrace = !rTrace;
        "\^<<theName>>'s rTrace is now <<(rTrace ? 'on' : 'off')>>. ";
    }
    rTrace = nil // default trace mode is off
;

#endif // __DEBUG

#ifdef __KNOWLEDGE

modify Actor

    /* note that this requires the included knowledge module. */
    scopeList() {
        if (ofKind(Rapper))
            return (knownObjs.keysToList +
                    seenObjs.keysToList +
                    inherited());
        return inherited();
    }
;

#endif // __KNOWLEDGE

/* This calls the animation daemon on all RAP-actors who are currently
 * in the game. You can use this to reset the animation daemons of all
 * RAP-actors, or to initialize the animation of all RAP actors.
 *
 * You can make this a preinit object if you want to start the game
 * with RAP-actors animated.
 */
setAnimateDaemons: object
    execute() {
        for (local obj = firstObj(Rapper) ; obj ;
             obj = nextObj(obj, Rapper)) {
            if (obj.location)
                obj.callAnimateDaemon();
        }
    }
;

/* The following verbs allow the player some control over realtime
 * processing. If you aren't using realtime capabilities, simply delete
 * the following line, #define __REALTIME
 */

#define __REALTIME

#ifdef __REALTIME

DefineSystemAction(ToggleRealtime)
  execSystemAction
  {
    if (Rapper.rSwitchableAnimateDaemon == rRTAnimateDaemon)
        Rapper.rSwitchableAnimateDaemon = rPseudoRTAnimateDaemon;
    else
        Rapper.rSwitchableAnimateDaemon = rRTAnimateDaemon;

    for (local obj = firstObj(rProxyFuse) ; obj ;
         obj = nextObj(obj, rProxyFuse)) {
        obj.removeEvent();
    }

    // reset all daemons.
    setAnimateDaemons.execute();

    "Okay, realtime is now
    <<(Rapper.rSwitchableAnimateDaemon == rRTAnimateDaemon) ?
    'on' : 'off'>>. ";
  }
;

VerbRule(ToggleRealtime)
  ('toggle' |'t' | ) ('realtime' | 'time' | 'rt')
  : ToggleRealtimeAction
  verbPhrase = 'toggle/toggling realtime'
;

DefineSystemAction(Slow)
  execSystemAction
  {
    "Now in \"slower\" mode: processing demands are less
    frequent. [You can reverse this effect with the \">FASTER\"
    command.]\n";
    rRTAnimateDaemon.interval = 11000;
    for (local obj = firstObj(Rapper) ; obj ;
         obj = nextObj(obj, Rapper)) {
        obj.lagAdj = 7000;
    }
    setAnimateDaemons.execute();
  }
;

VerbRule(Slow)
  ('slow' | 'slower') ('mode' | )
  : SlowAction
  verbPhrase = 'slow/slowing time'
;

DefineSystemAction(Fast)
  execSystemAction
  {
    "Now in \"faster\" mode: processing demands are of normal
    frequency. [You can slow down processing demand frequency with the
    \">SLOWER\" command.]\n";
    rRTAnimateDaemon.interval = 7000;
    for (local obj = firstObj(Rapper) ; obj ;
         obj = nextObj(obj, Rapper)) {
        obj.lagAdj = 4000;
    }
    setAnimateDaemons.execute();
  }
;

VerbRule(Fast)
  ('fast' | 'faster') ('mode' | )
  : FastAction
  verbPhrase = 'quicken/quickening time'
;

#endif // __REALTIME

RAPModuleID: ModuleID
    name = 'TADS 3 Reactive Agent Planner'
    byline = 'Copyright (C) 1998 by Nate Cull\n
             Tads-3 port and revision/extension (t3RAP)
             Copyright (c) 2002-2004 by Steve Breslin\n'
    htmlByline = 'Copyright (C) 1998
                 by <a href="mailto:nate@natecull.org">Nate Cull\n
                 Tads-3 port and revision/extension (t3RAP)
                 Copyright (c) 2002-2004 by
                 <a href="mailto:versim@hotmail.com">Steve
                 Breslin</a>\n'
    version = '1.3'
;

