"""
t3dataholder.py

Abstraction classes for the T3 Data Holder structure.
Also provides a get_data_holder(bytes) function that
returns a data holder of the appropriate class.
"""

import struct, cStringIO
from t3Function import T3Function

class T3DataHolder:
	"""
	Base class for all Data Holder value types. Any data holder can be
	initialised by passing it the five bytes of which it is made.
	Typical usage, however, will involve calling the utility function
	get_data_holder, also defined in this module.
	"""
	
	def __init__(self, s):
		self.typecode = struct.unpack("<B", s[0])[0]
		self.intvalue = int(struct.unpack("<I", s[1:])[0])
		self.stringvalue = ""
		
	def to_pyval(self):
		"""
		Returns the value of the class as the python equivalent.
		"""
		return self.intvalue
		
	def get_reference(self, image, visited=[]):
		return self.to_pyval()
		
	def __str__(self):
		return self.stringvalue
		
	def __repr__(self):
		return self.__str__()

class VM_NIL(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = "Nil"

	def to_pyval(self):
		return False
		
class VM_TRUE(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = "True"

	def to_pyval(self):
		return True
		
class VM_OBJ(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = "(Object ID: " + str(self.intvalue) + ")"
		
	def get_reference(self, image, visited=[]):
		return image.objectDict[self.intvalue]

class VM_PROP(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.intvalue = struct.unpack("<H", s[1:3])[0]
		self.stringvalue = "(Property ID: " + str(self.intvalue) + ")"
		
class VM_INT(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = str(self.intvalue)

class VM_SSTRING(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = "(sstring ID: " + str(self.intvalue) + ")"
		
	def get_reference(self, image, visited=[]):
		sizeofpage = image.constantPool[0].data["o_cpdf"].data["i_sizeOfPage"]
		p = self.intvalue / sizeofpage
		offset = self.intvalue % sizeofpage
		pagestream = cStringIO.StringIO(
							image.constantPool[p].data["s_pageData"])
		pagestream.seek(offset)
		length = struct.unpack("<H", pagestream.read(2))[0]
		string = pagestream.read(length)
		pagestream.close()
		return string

class VM_DSTRING(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = "(dstring ID: " + str(self.intvalue) + ")"
		
	def get_reference(self, image, visted=[]):
		sizeofpage = image.constantPool[0].data["o_cpdf"].data["i_sizeOfPage"]
		p = self.intvalue / sizeofpage
		offset = self.intvalue % sizeofpage
		pagestream = cStringIO.StringIO(
							image.constantPool[p].data["s_pageData"])
		pagestream.seek(offset)
		length = struct.unpack("<H", pagestream.read(2))[0]
		string = pagestream.read(length)
		pagestream.close()
		return string
		
class VM_LIST(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = "(list ID: " + str(self.intvalue) + ")"
		
	def get_reference(self, image, visited=[]):
#		if self in visited:
#			return [self.stringvalue]
		visited.append(self)
		sizeofpage = image.constantPool[0].data["o_cpdf"].data["i_sizeOfPage"]
		p = self.intvalue / sizeofpage
		offset = self.intvalue % sizeofpage
		pagestream = cStringIO.StringIO(
							image.constantPool[p].data["s_pageData"])
		pagestream.seek(offset)
		length = struct.unpack("<H", pagestream.read(2))[0]
		objects = []
		for i in xrange(length):
			obj = get_data_holder(pagestream.read(5))
			ref = obj.get_reference(image, visited)
			objects.append(obj.get_reference(image, visited))
		pagestream.close()
		return objects
		
class VM_CODEOFS(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = "(code offset: " + str(self.intvalue) + ")"
		
	def get_reference(self, image, visited=[]):
		sizeofpage = image.codePool[0].data["o_cpdf"].data["i_sizeOfPage"]
		p = int(self.intvalue / sizeofpage)
		offset = self.intvalue % sizeofpage
		if p >= len(image.codePool):
			return "[invalid code offset]"
		pagestream = cStringIO.StringIO(
							image.codePool[p].data["s_pageData"])
		pagestream.seek(offset)
		func = T3Function(pagestream.read(), len(image.functionList))
		pagestream.close()
		image.functionList.append(func)
		return func

class VM_FUNCPTR(VM_CODEOFS):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = "(function pointer: " + str(self.intvalue) + ")"	
		
class VM_EMPTY(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = "(empty)"
		
	def to_pyval(self):
		return None
		
class VM_ENUM(T3DataHolder):
	def __init__(self, s):
		T3DataHolder.__init__(self, s)
		self.stringvalue = "(enumerated constant: " + str(self.intvalue) + ")"
		
DataHolderTypedict = {1: VM_NIL,
						2: VM_TRUE,
						5: VM_OBJ,
						6: VM_PROP,
						7: VM_INT,
						8: VM_SSTRING,
						9: VM_DSTRING,
						10: VM_LIST,
						11: VM_CODEOFS,
						12: VM_FUNCPTR,
						13: VM_EMPTY,
						15: VM_ENUM}
		
			
def get_data_holder(bytes):
	return DataHolderTypedict[struct.unpack("<B", bytes[0])[0]](bytes)