"""
t3MACR.py

Provides a class representing a macro symbol table block in a t3 image file.
"""

import struct, cStringIO
from t3Block import *


class MACR_Block(Block):
	"""
	A preprocessor macro symbol table block. This is normally only included if
	the image file was compiled for debugging.
	"""
	name = "Preprocessor Macro Symbol Table Block"
	debug = True

	def __init__(self, datastream, image, attribs):
		Block.__init__(self, datastream, image, attribs)
		self.read_block()
		
	def read_block(self):
		"""
		Reads the data included in the block.
		"""
		n = struct.unpack("<I", self.datastream.read(4))[0]
		entries = []
		for i in xrange(n):
			entrydict = {}
			lensymn = struct.unpack("<H", self.datastream.read(2))[0]
			entrydict["s_symbolName"] = self.datastream.read(lensymn)
			f = struct.unpack("<H", self.datastream.read(2))[0]
			flags = [bool(f & (2**i)) for i in xrange(16)]
			entrydict["b_isFunctionLike"] = (flags[0] == True)
			entrydict["b_takesVaryingNumberOfArgs"] = (flags[1] == True)
			nparams = struct.unpack("<H", self.datastream.read(2))[0]
			entrydict["i_numberOfParameters"] = nparams
			paramlist = []
			for j in xrange(nparams):
				lenstr = struct.unpack("<H", self.datastream.read(2))[0]
				paramlist.append(repr(self.datastream.read(lenstr)))
			entrydict["l_parameters"] = paramlist
			lenexp = struct.unpack("<I", self.datastream.read(4))[0]
			entrydict["s_expansiontext"] = self.decode_expansion_text(lenexp)
			entries.append(entrydict)
		self.data["l_entries"] = entries

		
	def decode_expansion_text(self, n):
		"""
		The expansion text in each entry of this block can include special byte
		values, which mean different things. This function reads the expansion
		text (using the n argument as the length to read) and returns a 
		decoded string of the expansion text.
		"""
		decodeddata = cStringIO.StringIO()
		mode = "normal"
		delimiter = None
		substr = []
		for i in xrange(n):
			c = self.datastream.read(1)
			cintval = struct.unpack("<B", c)[0]
			if mode == "formalid":
				# this byte is the index of the parameter
				decodeddata.write(str(cintval) + "}")
				mode = "normal"
			elif mode in ["foreach", "ifempty", "ifnempty"]:
				# check if we have a delimiter
				if delimiter == None:
					# we don't, so this must be it
					delimiter = c
				else:
					# we do, so we're in the middle of the delimited text
					if c <> delimiter:
						substr.append(c)
					else:
						# the end of the text
						decodeddata.write("".join(substr) + "}")
						delimiter = None
						substr = []
						mode = "normal"
			elif mode == "normal":
				if cintval == 1:
					mode = "formalid"
					decodeddata.write("{arg: ")
				elif cintval == 5:
					mode = "foreach"
					decodeddata.write("{#foreach: ")
				elif cintval == 6:
					decodeddata.write("{#argcount}")
				elif cintval == 7:
					mode = "ifempty"
					decodeddata.write("{#ifempty:")
				elif cintval == 8:
					mode = "ifnempty"
					decodeddata.write("{#ifnempty:")
				else:
					decodeddata.write(c)
		dd = decodeddata.getvalue()
		decodeddata.close()
		return dd

	def report_data(self):
		sl = []
		for entry in self.data["l_entries"]:
			sl.append("\n")
			sl.append("symbol name: " + entry["s_symbolName"] + "\n")
			sl.append("is function-like: ") 
			sl.append(str(entry["b_isFunctionLike"]) + "\n")
			sl.append("takes varying number of arguments: ")
			sl.append(str(entry["b_takesVaryingNumberOfArgs"]) + "\n")
			sl.append("number of parameters: ")
			sl.append(str(entry["i_numberOfParameters"]) + "\n")
			sl.append("parameters: ")
			sl.append(str(entry["l_parameters"]) + "\n")
			sl.append("expansion text: ")
			sl.append(repr(entry["s_expansiontext"]) + "\n")
		return "".join(sl)