/*
 * stream.c
 *
 * IO stream implementation
 *
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "frotz.h"

#ifndef SEEK_SET
#define SEEK_SET 0
#endif

#ifndef SEEK_CUR
#define SEEK_CUR 1
#endif

#ifndef SEEK_END
#define SEEK_END 2
#endif

static int saved_buffering1 = 0;
static int saved_buffering2 = 0;

zword script_width = 0;

zword redirect_size = 0;
zword redirect_addr = 0;

int stream_message = 0;
int stream_display = 1;
int stream_script = 0;
int stream_memory = 0;
int stream_record = 0;

int replaying = 0;

static FILE *sfp = NULL;
static FILE *rfp = NULL;
static FILE *pfp = NULL;

char script_name[MAX_FILE_NAME + 1] = DEFAULT_SCRIPT_NAME;
char command_name[MAX_FILE_NAME + 1] = DEFAULT_COMMAND_NAME;

char *euro_substitute =
"\
aeoeueAeOeUess>><<e \
i y E I a e i o u y \
A E I O U Y a e i o \
u A E I O U a e i o \
u A E I O U a A o O \
a n o A N O aeAEc C \
ththThThL oeOE! ? \
";

/*
 * message_start
 *
 */

void message_start (void)
{

    os_message_start ();

    saved_buffering1 = enable_buffering;
    enable_buffering = 0;

    stream_message = 1;

}/* message_start */

/*
 * message_end
 *
 */

void message_end (void)
{

    os_message_end ();

    enable_buffering = saved_buffering1;

    stream_message = 0;

}/* message_end */

/*
 * open_playback
 *
 */

static void open_playback (void)
{
    char new_name[MAX_FILE_NAME + 1];

    if (os_get_file_name (new_name, command_name, FILE_PLAYBACK) != 0)

	if ((pfp = fopen (new_name, "rt")) != 0) {

	    strcpy (command_name, new_name);
	    replaying = 1;

	} else print_string ("Cannot open command file\n");

}/* open_playback */

/*
 * close_playback
 *
 */

static void close_playback (void)
{

    fclose (pfp);
    replaying = 0;

}/* close_playback */

/*
 * playback_char
 *
 */

static int playback_char (int force_encoding)
{
    int code;
    int c;

    c = fgetc (pfp);

    if (c == '[' || force_encoding != 0) {

	if (c != '[')
	    return -1;

	code = 0;

	do {
	    c = fgetc (pfp);
	    if (isdigit (c))
		code = 10 * code + c - '0';
	} while (isdigit (c));

	if (c != ']')
	    return -1;

	return code;

    } else return c;

}/* playback_char */

/*
 * playback_key
 *
 */

static int playback_key (void)
{
    int key;

    key = playback_char (0);

    if (key == 253 || key == 254) {
	mouse_x = playback_char (1);
	mouse_y = playback_char (1);
    }

    if (fgetc (pfp) != '\n')
	key = -1;

    if (key == -1)
	close_playback ();

    return key;

}/* playback_key */

/*
 * playback_line
 *
 */

static int playback_line (char *buffer)
{
    int terminator;
    int i;
    int c;

    i = 0;

    for (;;) {

	c = playback_char (0);

	if (c == -1) {
	    terminator = -1;
	    break;
	}

	if (c == '\n') {
	    ungetc ('\n', pfp);
	    c = 13;
	}

	if (is_terminator (c) != 0) {

	    terminator = c;

	    if (terminator == 253 || terminator == 254) {
		mouse_x = playback_char (1);
		mouse_y = playback_char (1);
	    }

	    break;
	}

	buffer[i++] = c;
    }

    buffer[i] = 0;

    if (fgetc (pfp) != '\n')
	terminator = -1;

    if (terminator == -1)
	close_playback ();

    display_string (buffer);

    return terminator;

}/* playback_line */

/*
 * open_record
 *
 */

static void open_record (void)
{
    char new_name[MAX_FILE_NAME + 1];

    if (os_get_file_name (new_name, command_name, FILE_RECORD) != 0)

	if ((rfp = fopen (new_name, "wt")) != NULL) {

	    strcpy (command_name, new_name);
	    stream_record = 1;

	} else print_string ("Cannot open command file\n");

}/* open_record */

/*
 * close_record
 *
 */

static void close_record (void)
{

    fclose (rfp);
    stream_record = 0;

}/* close_record */

/*
 * record_char
 *
 */

static void record_char (int c, int force_encoding)
{
    int i;

    if (!isprint (c) || c == '[' || force_encoding != 0) {

	fputc ('[', rfp);

	for (i = 10000; i != 0; i /= 10)
	    if (c >= i || i == 1)
		record_char ('0' + (c / i) % 10, 0);

	fputc (']', rfp);

    } else fputc (c, rfp);

}/* record_char */

/*
 * record_key
 *
 */

static void record_key (int key)
{

    record_char (key, 0);

    if (key == 253 || key == 254) {
	record_char (mouse_x, 1);
	record_char (mouse_y, 1);
    }

    if (fputc ('\n', rfp) == EOF)
	close_record ();

}/* record_key */

/*
 * record_line
 *
 */

static void record_line (const char *buffer, int terminator)
{
    int i;

    if (terminator != 0)
	for (i = 0; buffer[i] != 0; i++)
	    record_char ((unsigned char) buffer[i], 0);

    if (terminator != 13) {

	record_char (terminator, 0);

	if (terminator == 253 || terminator == 254) {
	    record_char (mouse_x, 1);
	    record_char (mouse_y, 1);
	}
    }

    if (fputc ('\n', rfp) == EOF)
	close_record ();

}/* record_line */

/*
 * open_script
 *
 * Open the transscript file. 'AMFV' makes this more complicated as it
 * turns transscription on/off several times to exclude some text from
 * the transscription file. This wasn't a problem for the original V4
 * interpreters which always sent transscription to the printer, but it
 * means a problem to modern interpreters that open a new file every
 * time transscription is turned on. Our solution is to append to the
 * old transscription file in V1 to V4, and to open a new file for V5+.
 *
 */

static void open_script (void)
{
    static int script_file_valid = 0;
    char new_name[MAX_FILE_NAME + 1];

    if (h_version >= V5 || script_file_valid == 0) {

	if (os_get_file_name (new_name, script_name, FILE_SCRIPT) == 0)
	    goto done;

	if ((sfp = fopen (new_name, "wt")) != NULL)
	    strcpy (script_name, new_name);

    } else sfp = fopen (script_name, "at");

    if (sfp != NULL) {

	script_file_valid = 1;
	stream_script = 1;

    } else print_string ("Cannot open transscript file\n");

done:

    if (stream_script)
	h_flags |= SCRIPTING_FLAG;
    else
	h_flags &= ~SCRIPTING_FLAG;

    SET_WORD (H_FLAGS, h_flags)

    script_width = 0;

}/* open_script */

/*
 * close_script
 *
 */

static void close_script (void)
{

    fclose (sfp);
    stream_script = 0;

    h_flags &= ~SCRIPTING_FLAG;
    SET_WORD (H_FLAGS, h_flags)

}/* close_script */

/*
 * script_char
 *
 */

static void script_char (int c)
{
    int c1, c2;

    /* Handle European characters */

    if (c >= EURO_MIN && c <= EURO_MAX) {

	c1 = euro_substitute[2 * (c - EURO_MIN)];
	c2 = euro_substitute[2 * (c - EURO_MIN) + 1];

	script_char (c1);

	if (c2 != ' ')
	    c = c2;
	else
	    return;
    }

    /* Handle special indentations */

    if (c == 9) {
	script_char (' ');
	c = ' ';
    }
    if (c == 11) {
	script_char (' ');
	script_char (' ');
	c = ' ';
    }

    script_width += os_char_width (c);

    fputc (c, sfp);

}/* script_char */

/*
 * script_new_line
 *
 */

static void script_new_line (void)
{

    script_width = 0;

    if (fputc ('\n', sfp) == EOF)
	close_script ();

}/* script_new_line */

/*
 * script_line
 *
 */

static void script_line (const char *s, int terminator)
{
    int i;

    for (i = 0; s[i] != 0; i++)
	script_char ((unsigned char) s[i]);

    if (terminator == 13)
	script_new_line ();

}/* script_line */

/*
 * script_rewind
 *
 */

static void script_rewind (const char *s)
{

    if (s[0] == 0)
	return;

    fseek (sfp, -strlen (s), SEEK_CUR);

    script_width -= os_string_width (s);

}/* script_rewind */

/*
 * open_memory
 *
 */

static void open_memory (zword table, zword width)
{

    redirect_addr = table;
    z_storew (redirect_addr, 0, 0);

    if ((short) width <= 0)
	redirect_size = get_line_width (-width);
    else
	redirect_size = width;

    h_line_width = 0;

    saved_buffering2 = enable_buffering;
    enable_buffering = (width != 9999);

    stream_memory = 1;

}/* open_memory */

/*
 * close_memory
 *
 */

static void close_memory (void)
{
    zword size;

    if (redirect_size != 9999) {

	LOW_WORD (redirect_addr, size)

	redirect_addr += size + 2;
	z_storew (redirect_addr, 0, 0);
    }

    if (h_version == V6)
	SET_WORD (H_LINE_WIDTH, h_line_width)

    enable_buffering = saved_buffering2;

    stream_memory = 0;

}/* close_memory */

/*
 * memory_char
 *
 */

static void memory_char (int c)
{
    zword size;
    zword addr;

    LOW_WORD (redirect_addr, size)

    addr = redirect_addr + size + 2;
    SET_BYTE (addr, c)

    z_storew (redirect_addr, 0, size + 1);

    h_line_width += os_char_width (c);

}/* memory_char */

/*
 * memory_new_line
 *
 */

static void memory_new_line (void)
{
    zword size;

    if (redirect_size != 9999) {

	LOW_WORD (redirect_addr, size)

	redirect_addr += size + 2;
	z_storew (redirect_addr, 0, 0);

    } else memory_char (13);

    h_line_width = 0;

}/* memory_new_line */

/*
 * new_line
 *
 * Copy newline to the output streams.
 *
 */

void new_line (void)
{

    if (stream_message)

	/* We are currently printing a debugging message so just
	   pass a newline character to the interface. */

	os_display_char ('\n');

    else if (stream_memory) {

	/* Redirection to memory. All other output streams are
	   disabled as long as redirection lasts. */

	memory_new_line ();

    } else {

	/* No output redirection, pass new_line to the screen
	   or to the transscript file. */

	if (stream_display)
	    display_new_line ();
	if (stream_script && enable_scripting)
	    script_new_line ();
    }

}/* new_line */

/*
 * print_char
 *
 * Copy a character to the output streams.
 *
 */

void print_char (int c)
{

    if (stream_message)

	/* We are currently printing a debugging message so just
	   pass the character to the interface. */

	os_display_char (c);

    else if (stream_memory)

	/* Redirection to memory. All other output streams are
	   disabled as long as redirection lasts. */

	memory_char (c);

    else {

	/* No output redirection, pass the character to the screen
	   or to the transscript file. */

	if (stream_display)
	    display_char (c);
	if (stream_script && enable_scripting)
	    script_char (c);
    }

}/* print_char */

/*
 * read_line
 *
 * Read a line of input from the current input stream.
 *
 */

int read_line (int max_size, char *buffer, int timeout, int continued)
{
    int terminator;

    flush_buffer ();

    /* Remove initial input from the transscript file or screen */

    if (continued == 0) {

	if (stream_script && enable_scripting && h_version != V6)
	    script_rewind (buffer);
	if (replaying)
	    display_rewind (buffer);
    }

    /* Loop until input succeeds */

    if (!replaying)
	shift_cursor (-os_string_width (buffer));

    do {

	if (replaying)
	    terminator = playback_line (buffer);
	else
	    terminator = os_read (max_size, buffer, timeout, get_units_left ());

    } while (terminator == -1);

    if (!replaying)
	shift_cursor (os_string_width (buffer));

    /* Empty buffer when a hot key was pressed */

    if (terminator >= HOT_KEY_MIN && terminator <= HOT_KEY_MAX)
	buffer[0] = 0;

    /* Copy input line to the transscript or command files */

    if (stream_script && enable_scripting && h_version != V6)
	script_line (buffer, terminator);
    if (stream_record && !replaying)
	record_line (buffer, terminator);

    /* Return terminating key */

    return terminator;

}/* read_line */

/*
 * read_char
 *
 * Read a single keystroke from the current input stream.
 *
 */

int read_char (int timeout)
{
    int key;

    flush_buffer ();

    /* Loop until input succeeds */

    do {

	if (replaying)
	    key = playback_key ();
	else
	    key = os_read_char (timeout);

    } while (key == -1);

    /* Copy key to command file */

    if (stream_record && !replaying)
	record_key (key);

    /* Return key */

    return key;

}/* read_char */

/*
 * z_input_stream
 *
 * Choose the input stream which can be either 0 (keyboard) or 1 (playback).
 *
 */

void z_input_stream (zword number)
{

    /* Turn on keyboard input */

    if (number == 0 && replaying)
	close_playback ();

    /* Turn on playback from a file */

    if (number == 1 && !replaying)
	open_playback ();

}/* z_input_stream */

/*
 * z_output_stream
 *
 * Open and close various output streams. These can be:
 *
 *    1 = the screen
 *    2 = transscript file
 *    3 = redirection to memory
 *    4 = command line recording
 *
 * A positive value turns the stream on, a negative value turns it off.
 *
 * For redirection to memory an additional parameter points to a table
 * within the memory of the Z-machine. The first word of the table must
 * be written with the number of characters printed. The text itself
 * follows after this first word. When the optional third parameter is
 * also given, then the text must be word-wrapped to fit into width
 * screen units; or, if this optional parameter is less or equal zero,
 * word-wrapped to fit into window |width|. When the optional parameter
 * is present, each line of text written to the memory has the format of
 * a table; a zero word marks the last line.
 *
 * In V6 only, the length of the redirected text (in screen units) is
 * written to the story file header.
 *
 */

void z_output_stream (int argc, zword number, zword table, zword width)
{

    flush_buffer ();

    /* Supply default arguments */

    if (argc < 3)
	width = 9999;

    /* Turn output stream on or off */

    if ((short) number == 1)
	stream_display = 1;
    if ((short) number == -1)
	stream_display = 0;
    if ((short) number == 2 && !stream_script)
	open_script ();
    if ((short) number == -2 && stream_script)
	close_script ();
    if ((short) number == 3 && !stream_memory)
	open_memory (table, width);
    if ((short) number == -3 && stream_memory)
	close_memory ();
    if ((short) number == 4 && !stream_record)
	open_record ();
    if ((short) number == -4 && stream_record)
	close_record ();

}/* z_output_stream */
