/*
 * variable.c
 *
 * Variable and stack related opcodes
 *
 */

#include "frotz.h"

/*
 * z_dec
 *
 * Decrement a variable.
 *
 */

void z_dec (zword variable)
{
    zword current;
    zword addr;

    if (variable == 0)
	stack[sp]--;
    else if (variable < 16)
	stack[fp - variable]--;
    else {
	addr = h_globals + 2 * (variable - 16);
	LOW_WORD (addr, current)
	current--;
	SET_WORD (addr, current)
    }

}/* z_dec */

/*
 * z_dec_chk
 *
 * Decrement a variable and then check its value against a target.
 *
 */

void z_dec_chk (zword variable, zword value)
{
    zword current;
    zword addr;

    if (variable == 0)
	current = --stack[sp];
    else if (variable < 16)
	current = --stack[fp - variable];
    else {
	addr = h_globals + 2 * (variable - 16);
	LOW_WORD (addr, current)
	current--;
	SET_WORD (addr, current)
    }

    branch ((short) current < (short) value);

}/* z_dec_chk */

/*
 * z_inc
 *
 * Increment a variable.
 *
 */

void z_inc (zword variable)
{
    zword current;
    zword addr;

    if (variable == 0)
	stack[sp]++;
    else if (variable < 16)
	stack[fp - variable]++;
    else {
	addr = h_globals + 2 * (variable - 16);
	LOW_WORD (addr, current)
	current++;
	SET_WORD (addr, current)
    }

}/* z_inc */

/*
 * z_inc_chk
 *
 * Increment a variable and then check its value against a target.
 *
 */

void z_inc_chk (zword variable, zword value)
{
    zword current;
    zword addr;

    if (variable == 0)
	current = ++stack[sp];
    else if (variable < 16)
	current = ++stack[fp - variable];
    else {
	addr = h_globals + 2 * (variable - 16);
	LOW_WORD (addr, current)
	current++;
	SET_WORD (addr, current)
    }

    branch ((short) current > (short) value);

}/* z_inc_chk */

/*
 * z_load
 *
 * Store the value of a variable.
 *
 */

void z_load (zword variable)
{
    zword current;
    zword addr;

    if (variable == 0)
	current = stack[sp];
    else if (variable < 16)
	current = stack[fp - variable];
    else {
	addr = h_globals + 2 * (variable - 16);
	LOW_WORD (addr, current)
    }

    store (current);

}/* z_load */

/*
 * z_pop
 *
 * Pop a value off the game stack and throw it away.
 *
 */

void z_pop (void)
{

    sp++;

}/* z_pop */

/*
 * z_pop_stack
 *
 * Pop several values off a stack and throw them away. By default the
 * game stack is used, otherwise a user stack.
 *
 */

void z_pop_stack (int argc, zword items, zword stack)
{
    zword size;

    if (argc == 2) {

	LOW_WORD (stack, size)
	z_storew (stack, 0, size + items);

    } else sp += items;

}/* z_pop_stack */

/*
 * z_pull
 *
 * Before V6: Pop a value off the game stack and store it in a
 * variable which is given as an argument. V6: Pop a value off
 * the game stack and store it unless an optional user stack is
 * given.
 *
 */

void z_pull (int argc, zword arg)
{
    zword size;
    zword current;

    if (h_version != V6)

	z_store (arg, stack[sp++]);

    else {

	/* Pop value off the game or user stack */

	if (argc == 1) {

	    LOW_WORD (arg, size)

	    size++;
	    z_storew (arg, 0, size);

	    arg += 2 * size;
	    LOW_WORD (arg, current)

	} else current = stack[sp++];

	/* Return value to Z-machine */

	store (current);
    }

}/* z_pull */

/*
 * z_push
 *
 * Push a value onto the game stack.
 *
 */

void z_push (zword value)
{

    stack[--sp] = value;

}/* z_push */

/*
 * z_push_stack
 *
 * Push a value onto a user stack and branch if successful.
 *
 */

void z_push_stack (zword value, zword stack)
{
    zword size;

    LOW_WORD (stack, size)

    if (size != 0) {

	z_storew (stack, size, value);
	z_storew (stack, 0, size - 1);

	branch (1);

    } else branch (0);

}/* z_push_stack */

/*
 * z_store
 *
 * Set a variable to the given value.
 *
 */

void z_store (zword variable, zword value)
{
    zword addr;

    if (variable == 0)
	stack[sp] = value;
    else if (variable < 16)
	stack[fp - variable] = value;
    else {
	addr = h_globals + 2 * (variable - 16);
	SET_WORD (addr, value)
    }

}/* z_store */
