#ifdef RCSID
static char RCSid[] =
"$Header: d:/tads/tads2/RCS/tdd.c 1.9 96/10/14 16:10:47 mroberts Exp $";
#endif

/* Copyright (c) 1992 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  tdd.c - TADS 2 debugger main
Function
  Main entrypoint for TADS 2 debugger
Notes
  None
Modified
  04/12/92 MJRoberts     - creation
*/
#include <stdio.h>
#include "os.h"
#include "std.h"
#include "err.h"
#include "mch.h"
#include "obj.h"
#include "run.h"
#include "voc.h"
#include "bif.h"
#include "dbg.h"
#include "sup.h"
#include "cmd.h"
#include "fio.h"
#include "prs.h"
#include "tok.h"
#include "emt.h"
#include "oem.h"


static void tddlogerr();

/* use os_printf rather than printf */
#define tddptf os_printf

void tddusage(ec)
errcxdef *ec;
{
    int  i;
    char buf[128];
    
    for (i = ERR_TDBUS1 ; i <= ERR_TDBUSL ; ++i)
    {
        errmsg(ec, buf, (uint)sizeof(buf), i);
        tddptf("%s\n", buf);
    }

    errsig(ec, ERR_USAGE);
}

/*
 *   Default memory sizes, if previously defined 
 */
#ifndef TDD_SETTINGS_DEFINED
# define TDD_HEAPSIZ  4096
# define TDD_STKSIZ   200
# define TDD_UNDOSIZ  (16 * 1024)
# define TDD_POOLSIZ  (2 * 1024)
# define TDD_LCLSIZ   0
#endif


static void tddmain1(ec, argc, argv)
errcxdef *ec;
int       argc;
char     *argv[];
{
    int        t;
    osfildef  *swapfp = (osfildef *)0;
    runcxdef   runctx;
    bifcxdef   bifctx;
    voccxdef   vocctx;
    void     (*bif[75])(/*_ dvoid *ctx _*/);
    mcmcxdef  *mctx;
    mcmcx1def *globalctx;
    dbgcxdef   dbg;
    supcxdef   supctx;
    int        err;
    char      *swapname = 0;
    char       swapbuf[OSFNMAX];
    char     **argp;
    char      *arg;
    char      *infile;
    ulong      swapsize = 0xffffffffL;        /* allow unlimited swap space */
#ifdef UNIX    
    int        swapena = FALSE;               /* TRUE if swapping is enabled */
#else
    int        swapena = TRUE;               /* TRUE if swapping is enabled */
#endif    
    int        i;
    int        pause = FALSE;                 /* pause after finishing game */
    fiolcxdef  fiolctx;
    noreg int  loadopen = FALSE;
    char       inbuf[128];
    ulong      cachelimit = 0xffffffff;
    ushort     undosiz = TDD_UNDOSIZ;      /* default undo context size 16k */
    objucxdef *undoptr;
    uint       flags;                          /* flags read from game file */
    objnum     preinit;                   /* preinit function object number */
    prscxdef  *pctx;
    tokcxdef  *tc;
    emtcxdef  *ectx;
    uint       heapsiz = TDD_HEAPSIZ;
    uint       stksiz = TDD_STKSIZ;
    runsdef   *mystack;
    uchar     *myheap;
    uint       lclsiz = TDD_LCLSIZ;
    uint       poolsiz = TDD_POOLSIZ;

    extern FILE *cmdfile;         /* hacky v1 qa interface - command log fp */
    extern FILE *logfp;            /* hacky v1 qa interface - output log fp */
    
    NOREG((&loadopen))
        
    /* initialize lexical analysis context */
    tc = tokcxini(ec, (mcmcxdef *)0, supsctab);
    tc->tokcxdbg = &dbg;
    tc->tokcxsst = (ushort (*)(/*_ dvoid * _*/))prsxsst;
    tc->tokcxsad = (void (*)(/*_ dvoid *, char *, ushort _)*/))prsxsad;
    tc->tokcxsend = (void (*)(/*_ dvoid * _*/))prsxsend;
    
    /* add current directory as first entry in search path */
    tokaddinc(tc, "", 0);

    /* parse arguments */
    for (i = 1, argp = argv + 1 ; i < argc ; ++argp, ++i)
    {
        arg = *argp;
        if (*arg == '-')
        {
            switch(*(arg+1))
            {
            case 'm':
                switch(*(arg + 2))
                {
                case 's':
                    stksiz = atoi(cmdarg(ec, &argp, &i, argc, 2, tddusage));
                    break;
                    
                case 'h':
                    heapsiz = atoi(cmdarg(ec, &argp, &i, argc, 2, tddusage));
                    break;
                    
                case 'p':
                    poolsiz = atoi(cmdarg(ec, &argp, &i, argc, 2, tddusage));
                    break;
                    
                default:
                    cachelimit = atol(cmdarg(ec, &argp, &i, argc, 1,
                                             tddusage));
                    break;
                }
                break;
                
            case 't':
                /* swap file options:  -tf file, -ts size, -t- (no swap) */
                switch(*(arg+2))
                {
                case 'f':
                    swapname = cmdarg(ec, &argp, &i, argc, 2, tddusage);
                    break;
                    
                case 's':
                    swapsize = atol(cmdarg(ec, &argp, &i, argc, 2, tddusage));
                    break;
                    
                default:
                    swapena = cmdtog(ec, swapena, arg, 1, tddusage);
                    break;
                }
                break;
                
            case 'u':
                undosiz = atoi(cmdarg(ec, &argp, &i, argc, 1, tddusage));
                break;
                
            case 'i':
                {
                    char *path = cmdarg(ec, &argp, &i, argc, 1, tddusage);
                    tokaddinc(tc, path, (int)strlen(path));
                    break;
                }
                
            default:
                tddusage(ec);
            }
        }
        else break;
    }

    /* get input name argument, and make sure it's the last argument */
    infile = *argp;
    if (i + 1 != argc) tddusage(ec);

    /* add default .GAM extension to input file */
    strcpy(inbuf, infile);
    os_defext(inbuf, "gam");

    /*
     *   if the name with the .GAM extension doesn't exist, use the
     *   original name 
     */
    if (osfacc(inbuf)) strcpy(inbuf, infile);

    /* use the buffer's current contents as the input filename */
    infile = inbuf;
    
    /* open up the swap file */
    if (swapena && swapsize)
    {
        swapfp = os_create_tempfile(swapname, swapbuf);
        if (swapname == 0) swapname = swapbuf;
        if (swapfp == 0) errsig(ec, ERR_OPSWAP);
    }

    ERRBEGIN(ec)

    /* initialize cache manager context */
    globalctx = mcmini(cachelimit, 128, swapsize, swapfp, ec);
    mctx = mcmcini(globalctx, 128, fioldobj, &fiolctx,
                   objrevert, (dvoid *)0);
    mctx->mcmcxrvc = mctx;

    /* allocate and initialize parsing context */
    pctx = (prscxdef *)mchalo(ec, (ushort)(sizeof(prscxdef) + poolsiz +
                                           lclsiz), "tcdmain");
    pctx->prscxerr = ec;
    pctx->prscxtok = tc;
    pctx->prscxmem = mctx;
    pctx->prscxprp = 0;
    pctx->prscxext = 0;
    pctx->prscxnsiz = pctx->prscxrrst = poolsiz;
    pctx->prscxnrst = pctx->prscxpool;
    pctx->prscxplcl = &pctx->prscxpool[poolsiz];
    pctx->prscxslcl = lclsiz;
    pctx->prscxflg = 0;
    pctx->prscxnode = &pctx->prscxpool[0];
    pctx->prscxgtab = (toktdef *)0;
    pctx->prscxvoc = &vocctx;
    pctx->prscxcpp = (uchar *)0;
    pctx->prscxcps = 0;
    pctx->prscxcpf = 0;
    pctx->prscxfsp = (uchar *)0;
    pctx->prscxfsf = 0;
    pctx->prscxfss = 0;
    pctx->prscxstab = (toktdef *)0;

    tc->tokcxscx = (dvoid *)pctx;

    /* allocate a code generator context */
    ectx = (emtcxdef *)mchalo(ec, (ushort)(sizeof(emtcxdef) +
                                           511*sizeof(emtldef)), "tcdmain");
    ectx->emtcxerr = ec;
    ectx->emtcxmem = pctx->prscxmem;
    ectx->emtcxptr = 0;   /* mcmalo(pctx->prscxmem, 1024, &ectx->emtcxobj); */
    ectx->emtcxlcnt = 512;
    ectx->emtcxfrob = MCMONINV;
    emtlini(ectx);
    
    pctx->prscxemt = ectx;

    /* set up an undo context */
    if (undosiz)
        undoptr = objuini(mctx, undosiz, vocdundo, vocdusz, &vocctx);
    else
        undoptr = (objucxdef *)0;

    /* set up vocabulary context */
    vocini(&vocctx, ec, mctx, &runctx, undoptr, 50, 50, 100);    
    
    /* allocate stack and heap */
    mystack = (runsdef *)mchalo(ec, (ushort)(stksiz * sizeof(runsdef)),
                                             "runtime stack");
    myheap = mchalo(ec, (ushort)heapsiz, "runtime heap");

    /* set up execution context */
    runctx.runcxerr = ec;
    runctx.runcxmem = mctx;
    runctx.runcxstk = mystack;
    runctx.runcxstop = &mystack[stksiz];
    runctx.runcxsp = mystack;
    runctx.runcxbp = mystack;
    runctx.runcxheap = myheap;
    runctx.runcxhp = myheap;
    runctx.runcxhtop = &myheap[heapsiz];
    runctx.runcxundo = undoptr;
    runctx.runcxbcx = &bifctx;
    runctx.runcxbi = bif;
    runctx.runcxtio = (tiocxdef *)0;
    runctx.runcxdbg = &dbg;
    runctx.runcxvoc = &vocctx;
    runctx.runcxdmd = supcont;
    runctx.runcxdmc = &supctx;
    runctx.runcxext = 0;

    /* set up setup context */
    supctx.supcxerr = ec;
    supctx.supcxmem = mctx;
    supctx.supcxtab = (tokthdef *)0;
    supctx.supcxbuf = (uchar *)0;
    supctx.supcxlen = 0;
    supctx.supcxvoc = &vocctx;
    supctx.supcxrun = &runctx;
    
    /* set up debug context */
    dbg.dbgcxtio = (tiocxdef *)0;
    dbg.dbgcxmem = mctx;
    dbg.dbgcxerr = ec;
    dbg.dbgcxtab = (tokthdef *)0;
    dbg.dbgcxfcn = 0;
    dbg.dbgcxdep = 0;
    dbg.dbgcxflg = DBGCXFSS + DBGCXFOK;    /* start off in single-step mode */
    dbg.dbgcxprs = pctx;
    dbg.dbgcxrun = &runctx;
    dbg.dbgcxlin = (lindef *)0;
    dbg.dbgcxnams = 2048;
    dbg.dbgcxnam = mchalo(ec, (ushort)dbg.dbgcxnams, "bp names");
    dbg.dbgcxnamf = 0;
    dbg.dbgcxhstl = 4096;
    dbg.dbgcxhstf = 0;
    dbg.dbgcxhstp = mchalo(ec, (ushort)dbg.dbgcxhstl, "history buffer");
    
    memset(dbg.dbgcxbp, 0, sizeof(dbg.dbgcxbp));       /* clear breakpoints */
    memset(dbg.dbgcxwx, 0, sizeof(dbg.dbgcxwx));           /* clear watches */
    
    /* set up built-in function context */
    CLRSTRUCT(bifctx);
    bifctx.bifcxerr = ec;
    bifctx.bifcxrun = &runctx;
    bifctx.bifcxtio = (tiocxdef *)0;
    bifctx.bifcxrnd = 0;
    bifctx.bifcxrndset = FALSE;
    
    ec->errcxlog = tddlogerr;
    ec->errcxlgc = &dbg;

    /* initialize debugger user interface */
    dbguini(&dbg);
    
    /* read the game from the binary file */
    fiord(mctx, &vocctx, (struct tokcxdef *)0,
          infile, (char *)0, &fiolctx, &preinit, &flags,
          tc->tokcxinc, (uchar **)0, (uint *)0, (uint *)0, FALSE);
    loadopen = TRUE;

    /* use same case sensitivity as original compilation */
    if (flags & FIOFCASE)
        tc->tokcxflg |= TOKCXCASEFOLD;
    else
        tc->tokcxflg &= ~TOKCXCASEFOLD;

    /* make sure the game was compiled for debugging */
    if ((flags & (FIOFSYM + FIOFLIN + FIOFPRE)) !=
        (FIOFSYM + FIOFLIN + FIOFPRE))
        errsig(ec, ERR_NODBG);
    
    /* add the built-in functions, keywords, etc */
    suprsrv(&supctx, bif, dbg.dbgcxtab, (int)(sizeof(bif)/sizeof(bif[0])),
            FALSE, (char *)0, tc->tokcxflg & TOKCXCASEFOLD);
    tc->tokcxstab = (toktdef *)dbg.dbgcxtab;
    
    /* set up status line hack */
    runistat(&vocctx, &runctx, (tiocxdef *)0);
    
    /* play the game, running preinit if necessary */
    plygo(&runctx, &vocctx, (tiocxdef *)0, preinit, FALSE);
    
    /* close load file */
    fiorcls(&fiolctx);
    
    if (pause)
    {
        tddptf("[strike a key to exit]");
        os_waitc();
        tddptf("\n");
    }
    
    /* close and delete swapfile, if one was opened */
    if (swapfp)
    {
        osfcls(swapfp);
        swapfp = (osfildef *)0;
        osfdel(swapname);
    }

    ERRCLEAN(ec)
        /* close and delete swapfile, if one was opened */
        if (swapfp)
        {
            osfcls(swapfp);
            swapfp = (osfildef *)0;
            osfdel(swapname);
        }
        
        /* close the load file if one was opened */
        if (loadopen) fiorcls(&fiolctx);
    ERRENDCLN(ec)
}

/* log an error */
static void tddlogerr(ctx, fac, err, argc, argv)
dvoid    *ctx;
char     *fac;
int       err;
int       argc;
erradef  *argv;
{
    int   i;
    char  buf[128];
    char  msg[128];

    errmsg(((dbgcxdef *)ctx)->dbgcxerr, msg, (uint)sizeof(msg), err);
    errfmt(buf, (int)sizeof(buf), msg, argc, argv);
    dbguerr((dbgcxdef *)ctx, err, buf);
}

/* log an error - used before debugger context is set up */
static void tddlogerr1(ctx, fac, err, argc, argv)
errcxdef *ctx;
char     *fac;
int       err;
int       argc;
erradef  *argv;
{
    int   i;
    char  buf[128];
    char  msg[128];

    errmsg(ctx, msg, (uint)sizeof(msg), err);
    errfmt(buf, (int)sizeof(buf), msg, argc, argv);
    tddptf("%s-%d: %s\n", fac, err, buf);
}

/* main - called by os main after setting up arguments */
int tddmain(argc, argv)
int   argc;
char *argv[];
{
    errcxdef  errctx;
    int       err;
    osfildef *fp;
    
    errctx.errcxlog = tddlogerr1;
    errctx.errcxlgc = &errctx;
    errctx.errcxfp  = (osfildef *)0;
    errctx.errcxofs = 0;
    fp = oserrop(argv[0]);
    errini(&errctx, fp);
    
    tddptf("tdb: the TADS Debugger v2.2.1.%d (%s)\n",
           TADS_OEM_VERSION, TADS_OEM_NAME);
    tddptf("Copyright (c) 1993, 1996 Michael J. Roberts\n");

#ifdef	UNIX
    tddptf("Unix Patchlevel %s\n", UNIXPATCHLEVEL);
    tddptf("Compiled for %s (%s) [Patchlevel %s].\n", SYSNAME, OS_SYSTEM_NAME, SYSPL);
    tddptf("%s maintains this port.\n", SYSMAINTAINER);
    tddptf("NOTE: This program must be run in a 80x50 character window.\n");
#endif
    
    ERRBEGIN(&errctx)
        tddmain1(&errctx, argc, argv);
    ERRCATCH(&errctx, err)
        if (err != ERR_USAGE && err != ERR_RUNQUIT)
            errclog(&errctx);
        if (errctx.errcxfp) osfcls(errctx.errcxfp);
        return(OSEXFAIL);
    ERREND(&errctx)
        
    if (errctx.errcxfp) osfcls(errctx.errcxfp);
    return(OSEXSUCC);
}
