#ifdef RCSID
static char RCSid[] =
"$Header: d:/tads/tads2/RCS/tok.c 1.9 96/10/14 16:12:07 mroberts Exp $";
#endif

/*  Copyright (c) 1987, 1988 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  tok  - tokenizer
Function
  Tokenizes an input stream
Notes
  None
Modified
  08/14/91 MJRoberts     - created
*/

#include "os.h"
#include "std.h"
#include "mch.h"
#include "mcm.h"
#include "tok.h"
#include "lin.h"
#include "linf.h"
#include "dbg.h"
#include <ctype.h>
#include <string.h>
#include <stdlib.h>


/* special temporary buffers for <<expr>> macro expansion */
static char tokmac1[] = ",say((";
static char tokmac2[] = "),nil),\"";
static char tokmac3[] = "),nil)";


/* find a #define symbol */
tokdfdef *tok_find_define(ctx, sym, len)
tokcxdef *ctx;
char     *sym;
int       len;
{
    int       hsh;
    tokdfdef *df;

    /* find the appropriate chain the hash table */
    hsh = tokdfhsh(sym, len);

    /* search the chain for this symbol */
    for (df = ctx->tokcxdf[hsh] ; df ; df = df->nxt)
    {
        /* if this one matches, return it */
        if (df->len == len && !memcmp(df->nm, sym, (size_t)len))
        {
            /* fix it up if it's the special __FILE__ or __LINE__ symbol */
            if (len == 8)
            {
                char *p;
                long  l;
                
                if (!memcmp(sym, "__FILE__", (size_t)8))
                {
                    linppos(ctx->tokcxlin, df->expan+1, 128);
                    df->expan[0] = '\'';
                    for (p = df->expan ; *p && *p != '(' ; ++p) ;
                    *p++ = '\'';
                    df->explen = p - df->expan;
                }
                else if (!memcmp(sym, "__LINE__", (size_t)8))
                {
                    linppos(ctx->tokcxlin, df->expan, 128);
                    for (p = df->expan ; *p && *p != '(' ; ++p) ;
                    if (*p) ++p;
                    l = atol(p);
                    sprintf(df->expan, "%lu", l);
                    df->explen = strlen(df->expan);
                }
            }
            
            /* return it */
            return df;
        }
    }

    /* didn't find anything */
    return 0;
}

/*
 *   Write preprocessor state to a file 
 */
void tok_write_defines(ctx, fp, ec)
tokcxdef *ctx;
osfildef *fp;
errcxdef *ec;
{
    int        i;
    tokdfdef **dfp;
    tokdfdef  *df;
    char       buf[4];

    /* write each element of the hash chains */
    for (i = TOKDFHSHSIZ, dfp = ctx->tokcxdf ; i ; ++dfp, --i)
    {
        /* write each entry in this hash chain */
        for (df = *dfp ; df ; df = df->nxt)
        {
            /* write this entry */
            oswp2(buf, df->len);
            oswp2(buf + 2, df->explen);
            if (osfwb(fp, buf, 4)
                || osfwb(fp, df->nm, df->len)
                || osfwb(fp, df->expan, df->explen))
                errsig(ec, ERR_WRTGAM);
        }

        /* write a zero-length entry to indicate the end of this chain */
        oswp2(buf, 0);
        if (osfwb(fp, buf, 4)) errsig(ec, ERR_WRTGAM);
    }
}

/*
 *   Read preprocessor state from a file 
 */
void tok_read_defines(ctx, fp, ec)
tokcxdef *ctx;
osfildef *fp;
errcxdef *ec;
{
    int        i;
    tokdfdef **dfp;
    tokdfdef  *df;
    char       buf[4];

    /* write each element of the hash chains */
    for (i = TOKDFHSHSIZ, dfp = ctx->tokcxdf ; i ; ++dfp, --i)
    {
        /* read this hash chain */
        for (;;)
        {
            /* read the next entry's header, and stop if this is the end */
            if (osfrb(fp, buf, 4)) errsig(ec, ERR_RDGAM);
            if (osrp2(buf) == 0) break;

            /* set up a new symbol of the appropriate size */
            df = (tokdfdef *)mchalo(ec,
                                    (ushort)(sizeof(tokdfdef) + osrp2(buf)
                                                    + osrp2(buf+2) - 1),
                                             "tok_read_defines");
            df->explen = osrp2(buf+2);
            df->nm = df->expan + df->explen;
            df->len = osrp2(buf);

            /* read the rest of the symbol */
            if (osfrb(fp, df->nm, df->len) || osfrb(fp, df->expan, df->explen))
                errsig(ec, ERR_RDGAM);

            /*
             *   If a symbol with this name already exists in the table,
             *   discard the new one -- the symbols defined by -D and the
             *   current set of built-in symbols takes precedence over the
             *   set loaded from the file.  
             */
            if (tok_find_define(ctx, df->nm, df->len))
            {
                /* simply discard this symbol */
                mchfre(df);
            }
            else
            {
                /* link it into this hash chain */
                df->nxt = *dfp;
                *dfp = df;
            }
        }
    }
}



/* compute a #define symbol's hash value */
static int tokdfhsh(char *sym, int len)
{
    uint hsh;

    for (hsh = 0 ; len ; ++sym, --len)
        hsh = (hsh + *sym) & TOKDFHSHMASK;
    return hsh;
}

/* add a symbol to the #define symbol table */
void tok_add_define(ctx, sym, len, expan, explen)
tokcxdef *ctx;
char     *sym;
int       len;
char     *expan;
int       explen;
{
    int       hsh;
    tokdfdef *df;
    
    /* if it's already defined, ignore it */
    if (tok_find_define(ctx, sym, len))
        return;

    /* find the appropriate entry in the hash table */
    hsh = tokdfhsh(sym, len);

    /* allocate space for the symbol */
    df = (tokdfdef *)mchalo(ctx->tokcxerr,
                            (ushort)(sizeof(tokdfdef) + len + explen - 1),
                            "tok_add_define");

    /* set up the new symbol */
    df->nm = df->expan + explen;
    df->len = len;
    df->explen = explen;
    memcpy(df->expan, expan, explen);
    memcpy(df->nm, sym, len);

    /* link it into the hash chain */
    df->nxt = ctx->tokcxdf[hsh];
    ctx->tokcxdf[hsh] = df;
}

/* undefine a #define symbol */
void tok_del_define(ctx, sym, len)
tokcxdef *ctx;
char     *sym;
int       len;
{
    int       hsh;
    tokdfdef *df;
    tokdfdef *prv;
    
    /* find the appropriate chain the hash table */
    hsh = tokdfhsh(sym, len);

    /* search the chain for this symbol */
    for (prv = 0, df = ctx->tokcxdf[hsh] ; df ; prv = df, df = df->nxt)
    {
        /* if this one matches, delete it */
        if (df->len == len && !memcmp(df->nm, sym, (size_t)len))
        {
            /* unlink it from the chain */
            if (prv)
                prv->nxt = df->nxt;
            else
                ctx->tokcxdf[hsh] = df->nxt;

            /* delete this symbol, and we're done */
            mchfre(df);
            break;
        }
    }
}

/* scan a #define symbol to see how long it is */
static int tok_scan_defsym(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    int symlen;

    /* make sure it's a valid symbol */
    if (!(isalpha(*p) || *p == '_' || *p == '$'))
    {
        errlog(ctx->tokcxerr, ERR_REQSYM);
        return 0;
    }

    /* count characters as long as we have valid symbol characters */
    for (symlen = 0 ; len && TOKISSYM(*p) ; ++p, --len, ++symlen) ;
    return symlen;
}

/* process a #define */
static void tokdefine(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    char *sym;
    int   symlen;
    char *expan;
    
    /* get the symbol */
    sym = p;
    if (!(symlen = tok_scan_defsym(ctx, p, len)))
        return;

    /* if it's already in the table, log an error */
    if (tok_find_define(ctx, sym, symlen))
    {
        errlog(ctx->tokcxerr, ERR_DEFREDEF);
        return;
    }

    /* skip whitespace following the symbol */
    expan = sym + symlen;
    len -= symlen;
    while (len && isspace(*expan)) --len, ++expan;

    /* define the symbol */
    tok_add_define(ctx, sym, symlen, expan, len);
}

/* process an #ifdef or a #ifndef */
static void tok_ifdef_ifndef(ctx, p, len, is_ifdef)
tokcxdef *ctx;
char     *p;
int       len;
int       is_ifdef;
{
    int   symlen;
    char *sym;
    int   stat;
    int   found;

    /* get the symbol */
    sym = p;
    if (!(symlen = tok_scan_defsym(ctx, p, len)))
        return;

    /* see if we can find it in the table, and set the status accordingly */
    found = (tok_find_define(ctx, sym, symlen) != 0);

    /* invert the test if this is an ifndef */
    if (!is_ifdef) found = !found;

    /* set the #if status accordingly */
    if (found)
        stat = TOKIF_IF_YES;
    else
        stat = TOKIF_IF_NO;

    ctx->tokcxif[ctx->tokcxifcnt] = stat;

    /* allocate a new #if level (making sure we have room) */
    if (ctx->tokcxifcnt >= TOKIFNEST)
    {
        errlog(ctx->tokcxerr, ERR_MANYPIF);
        return;
    }
    ctx->tokcxifcnt++;
}

/* process a #error */
static void tok_p_error(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    errlog1(ctx->tokcxerr, ERR_P_ERROR,
            ERRTSTR, errstr(ctx->tokcxerr, p, len));
}

/* process a #ifdef */
static void tokifdef(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    tok_ifdef_ifndef(ctx, p, len, TRUE);
}

/* process a #ifndef */
static void tokifndef(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    tok_ifdef_ifndef(ctx, p, len, FALSE);
}

/* process a #if */
static void tokif(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    errsig(ctx->tokcxerr, ERR_PIF_NA);
}

/* process a #elif */
static void tokelif(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    errsig(ctx->tokcxerr, ERR_PELIF_NA);
}

/* process a #else */
static void tokelse(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    int cnt;
    
    /* if we're not expecting #else, it's an error */
    cnt = ctx->tokcxifcnt;
    if (cnt == 0 || ctx->tokcxif[cnt-1] == TOKIF_ELSE_YES
        || ctx->tokcxif[cnt-1] == TOKIF_ELSE_NO)
    {
        errlog(ctx->tokcxerr, ERR_BADPELSE);
        return;
    }

    /* switch to the appropriate #else state (opposite the #if state) */
    if (ctx->tokcxif[cnt-1] == TOKIF_IF_YES)
        ctx->tokcxif[cnt-1] = TOKIF_ELSE_NO;
    else
        ctx->tokcxif[cnt-1] = TOKIF_ELSE_YES;
}

/* process a #endif */
static void tokendif(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    /* if we're not expecting #endif, it's an error */
    if (ctx->tokcxifcnt == 0)
    {
        errlog(ctx->tokcxerr, ERR_BADENDIF);
        return;
    }

    /* remove the #if level */
    ctx->tokcxifcnt--;
}

/* process a #undef */
static void tokundef(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    char *sym;
    int   symlen;
    char *expan;
    
    /* get the symbol */
    sym = p;
    if (!(symlen = tok_scan_defsym(ctx, p, len)))
        return;

    /* if it's not defined, log a warning */
    if (!tok_find_define(ctx, sym, symlen))
    {
        errlog(ctx->tokcxerr, ERR_PUNDEF);
        return;
    }

    /* undefine the symbol */
    tok_del_define(ctx, sym, symlen);
}

/* process a #pragma directive */
static void tokpragma(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    /* ignore empty pragmas */
    if (len == 0)
    {
        errlog(ctx->tokcxerr, ERR_PRAGMA);
        return;
    }

    /* see what we have */
    if (len > 1 && (*p == 'c' || *p == 'C')
        && (*(p+1) == '+' || *(p+1) == '-') || isspace(*(p+1)))
    {
        /* skip spaces after the 'C', if any */
        for (++p, --len ; len && isspace(*p) ; ++p, --len) ;

        /* look for the + or - flag */
        if (len && *p == '+')
            ctx->tokcxflg |= TOKCXFCMODE;
        else if (len && *p == '-')
            ctx->tokcxflg &= ~TOKCXFCMODE;
        else
        {
            errlog(ctx->tokcxerr, ERR_PRAGMA);
            return;
        }
    }
    else
    {
        errlog(ctx->tokcxerr, ERR_PRAGMA);
    }
}

/* process a #include directive */
static void tokinclude(ctx, p, len)
tokcxdef *ctx;
char     *p;
int       len;
{
    linfdef *child;
    tokpdef *path;
    char    *fname;
    int      match;
    int      flen;
    linfdef *lin;
    char    *q;
    size_t   flen2;

    /* find the filename portion */
    fname = p + 1;                            /* remember start of filename */
    path = ctx->tokcxinc;                    /* start with first path entry */

    if (!len)
    {
        errlog(ctx->tokcxerr, ERR_INCNOFN);
        return;
    }
    
    switch(*p)
    {
    case '<':
        match = '>';
        if (path && path->tokpnxt) path = path->tokpnxt;   /* skip 1st path */
        goto find_matching_delim;

    case '"':
        match = '"';

    find_matching_delim:
        for (++p, --len ; len && *p != match ; --len, ++p);
        if (len == 0 || *p != match) errlog(ctx->tokcxerr, ERR_INCMTCH);
        break;
        
    default:
        errlog(ctx->tokcxerr, ERR_INCSYN);
        return;
    }
    
    flen = p - fname;                         /* compute length of filename */
    for (q = p, flen2 = 0 ;
         q > fname && *(q-1) != OSPATHCHAR && !strchr(OSPATHALT, *(q-1)) ;
         --q, ++flen2) ;
    
    /* check to see if this file has already been included */
    for (lin = ctx->tokcxhdr ; lin ; lin = (linfdef *)lin->linflin.linnxt)
    {
        char *p = lin->linfnam;
        
        p += strlen(p);
        
        while (p > lin->linfnam && *(p-1) != OSPATHCHAR
               && !strchr(OSPATHALT, *(p-1)))
            --p;
        if (strlen(p) == flen2
            && !memicmp(p, q, flen2))
        {
            errlog1(ctx->tokcxerr, ERR_INCRPT, ERRTSTR,
                    errstr(ctx->tokcxerr, fname, flen));
            return;
        }
    }
    
    /* initialize the line source */
    child = linfini(ctx->tokcxmem, ctx->tokcxerr, fname, flen, path);
    
    /* if not found, signal an error */
    if (!child) errsig1(ctx->tokcxerr, ERR_INCSEAR,
                        ERRTSTR, errstr(ctx->tokcxerr, fname, flen));
    
    /* link into tokenizer list of line records */
    child->linflin.linnxt = (lindef *)ctx->tokcxhdr;
    ctx->tokcxhdr = child;

    /* if we're tracking sources for debugging, add into the chain */
    if (ctx->tokcxdbg)
    {
        ctx->tokcxdbg->dbgcxlin = &child->linflin;
        child->linflin.linid = ctx->tokcxdbg->dbgcxfid++;
    }

    /* remember my C-mode setting */
    if (ctx->tokcxflg & TOKCXFCMODE)
        ctx->tokcxlin->linflg |= LINFCMODE;
    else
        ctx->tokcxlin->linflg &= ~LINFCMODE;
    
    child->linflin.linpar = ctx->tokcxlin;   /* remember parent line source */
    ctx->tokcxlin = &child->linflin;   /* make the child the current source */
}

/* get a new line from line source, processing '#' directives */
int tokgetlin(ctx, dopound)
tokcxdef *ctx;
int       dopound;    /* handle '#' directives (don't if in comment/string) */
{
    for (;;)
    {
        if (linget(ctx->tokcxlin))
        {
            /* at eof in current source; resume parent if there is one */
            if (ctx->tokcxlin->linpar)
            {
                lindef *parent;
                
                parent = ctx->tokcxlin->linpar;          /* remember parent */
                lincls(ctx->tokcxlin);               /* close included file */
                if (!ctx->tokcxdbg)               /* if no debug context... */
                    mchfre(ctx->tokcxlin);              /* free line source */
                ctx->tokcxlin = parent;      /* reset to parent line source */
                if (parent->linflg & LINFCMODE)
                    ctx->tokcxflg |= TOKCXFCMODE;
                else
                    ctx->tokcxflg &= ~TOKCXFCMODE;
                continue;                       /* back for another attempt */
            }
            else
            {
                /* check for outstanding #if/#ifdef */
                if (ctx->tokcxifcnt)
                    errlog(ctx->tokcxerr, ERR_NOENDIF);

                /* return end-of-file indication */
                return TRUE;
            }
        }
        
        /* if this is a multi-segment line, copy it into our own buffer */
        if (ctx->tokcxlin->linflg & LINFMORE)
        {
            char *p;
            uint  rem;
            int   done;
            
            if (!ctx->tokcxbuf)
            {
                /* allocate 1k as a default buffer */
                ctx->tokcxbuf = mchalo(ctx->tokcxerr, (ushort)1024, "tok");
                ctx->tokcxbsz = 1024;
            }
            ctx->tokcxlen = 0;
            
            for (done = FALSE, p = ctx->tokcxbuf, rem = ctx->tokcxbsz ;
                 !done ; )
            {
                size_t len = ctx->tokcxlin->linlen;

                /* add the current segment's length into line length */
                ctx->tokcxlen += len;
                
                /* we're done after this piece if the last fetch was all */
                done = !(ctx->tokcxlin->linflg & LINFMORE);
                if (len + 1 > rem)
                {
                    char *newp;

                    /* increase the size of the buffer */
                    if (ctx->tokcxbsz > (unsigned)0x8000)
                        errsig(ctx->tokcxerr, ERR_LONGLIN);
                    rem += 4096;
                    ctx->tokcxbsz += 4096;
                    
                    /* allocate a new buffer and copy line into it */
                    newp = mchalo(ctx->tokcxerr, ctx->tokcxbsz, "tok");
                    memcpy(newp, ctx->tokcxbuf, (size_t)(p - ctx->tokcxbuf));
                    
                    /* free the original buffer, and use the new one */
                    p = (p - ctx->tokcxbuf) + newp;
                    mchfre(ctx->tokcxbuf);
                    ctx->tokcxbuf = newp;
                }
                
                /* add the line to the buffer */
                memcpy(p, ctx->tokcxlin->linbuf, len);
                p += len;
                rem -= len;
                
                /* get the next piece of the line if there is one */
                if (!done)
                {
                    if (linget(ctx->tokcxlin)) break;
                }
            }
            
            /* null-terminate the buffer, and use it for input */
            *p = '\0';
            ctx->tokcxptr = ctx->tokcxbuf;
        }
        else
        {
            ctx->tokcxptr = ctx->tokcxlin->linbuf;
            ctx->tokcxlen = ctx->tokcxlin->linlen;
        }
        
        /* check for preprocessor directives */
        if (dopound && ctx->tokcxlen != 0 && ctx->tokcxptr[0] == '#'
            && !(ctx->tokcxlin->linflg & LINFNOINC))
        {
            char   *p;
            int     len;
            static  struct
            {
                char  *nm;
                int    len;
                int    ok_in_if;
                void (*fn)(/*_ tokcxdef *, char *, int _*/);
            }
            *dirp, dir[] =
            {
                { "include", 7, FALSE, tokinclude },
                { "pragma",  6, FALSE, tokpragma },
                { "define",  6, FALSE, tokdefine },
                { "ifdef",   5, TRUE, tokifdef },
                { "ifndef",  6, TRUE, tokifndef },
                { "if",      2, TRUE, tokif },
                { "else",    4, TRUE, tokelse },
                { "elif",    4, TRUE, tokelif },
                { "endif",   5, TRUE, tokendif },
                { "undef",   5, FALSE, tokundef },
                { "error",   5, FALSE, tok_p_error }
            };
            int  i;

            /* scan off spaces between '#' and directive */
            for (len = ctx->tokcxlen, p = &ctx->tokcxptr[1] ;
                 len && isspace(*p) ; --len, ++p);

            /* find and process the directive */
            for (dirp = dir, i = sizeof(dir)/sizeof(dir[0]) ; i ; --i, ++dirp)
            {
                /* compare this directive; if it wins, call its function */
                if (len >= dirp->len && !memcmp(p, dirp->nm, (size_t)dirp->len)
                    && (len == dirp->len || isspace(*(p + dirp->len))))
                {
                    int cnt;
                    int stat;
                    
                    /*
                     *   if we're not in a #if's false part, or if the
                     *   directive is processed even in #if false parts,
                     *   process the line, otherwise skip it 
                     */
                    cnt = ctx->tokcxifcnt;
                    if (dirp->ok_in_if || cnt == 0
                        || ((stat = ctx->tokcxif[cnt - 1]) == TOKIF_IF_YES
                            || stat == TOKIF_ELSE_YES))
                    {
                        /* skip whitespace following the directive */
                        for (p += dirp->len, len -= dirp->len ;
                             len && isspace(*p) ;
                             --len, ++p);

                        /* invoke the function to process this directive */
                        (*dirp->fn)(ctx, p, len);
                    }

                    /* there's no need to look at more directives */
                    break;
                }
            }

            /* if we didn't find anything, flag the error */
            if (i == 0)
                errlog(ctx->tokcxerr, ERR_PRPDIR);

            /* ignore this line */
            continue;
        }
        else
        {
            /*
             *   Check the #if level.  If we're in an #if, and we're to
             *   ignore lines (because of a false condition or an #else
             *   part for a true condition), skip this line. 
             */
            if (ctx->tokcxifcnt)
            {
                switch(ctx->tokcxif[ctx->tokcxifcnt - 1])
                {
                case TOKIF_IF_NO:
                case TOKIF_ELSE_NO:
                    /* ignore this line */
                    continue;

                default:
                    /* we're in a true part - keep the line */
                    break;
                }
            }
            
            ctx->tokcxlin->linflg &= ~LINFDBG;       /* no debug record yet */
            return(FALSE);                      /* return the line we found */
        }
    }
}

/* get the next token, removing it from the input stream */
int toknext(ctx)
tokcxdef *ctx;                                  /* lexical analysis context */
{
    char   *p;
    tokdef *tok = &ctx->tokcxcur;
    int     len;
    
    p = ctx->tokcxptr;
    len = ctx->tokcxlen;

    /* scan off whitespace and comments until we find something */
    do
    {
    skipblanks:
        /* if there's nothing on this line, get the next one */
        if (len == 0)
        {
            /* if we're in a macro expansion, continue after it */
            if (ctx->tokcxmlvl)
            {
                ctx->tokcxmlvl--;
                p = ctx->tokcxmsav[ctx->tokcxmlvl];
                len = ctx->tokcxmsvl[ctx->tokcxmlvl];
            }
            else
            {
                if (tokgetlin(ctx, TRUE))
                {
                    tok->toktyp = TOKTEOF;
                    goto done;
                }
                p = ctx->tokcxptr;
                len = ctx->tokcxlen;
            }
        }
        while (len && isspace(*p)) ++p, --len;       /* scan off whitespace */
        
        /* check for comments, and remove if present */
        if (len >= 2 && *p == '/' && *(p+1) == '/')
            len = 0;
        else if (len >= 2 && *p == '/' && *(p+1) == '*')
        {
            while (len < 2 || *p != '*' || *(p+1) != '/')
            {
                if (len) ++p, --len;
                if (len == 0)
                {
                    if (ctx->tokcxmlvl)
                    {
                        ctx->tokcxmlvl--;
                        p = ctx->tokcxmsav[ctx->tokcxmlvl];
                        len = ctx->tokcxmsvl[ctx->tokcxmlvl];
                    }
                    else
                    {
                        if (tokgetlin(ctx, FALSE))
                        {
                            ctx->tokcxptr = p;
                            tok->toktyp = TOKTEOF;
                            goto done;
                        }
                        p = ctx->tokcxptr;
                        len = ctx->tokcxlen;
                    }
                }
            }
            p += 2;
            len -= 2;
            goto skipblanks;
        }
    } while (len == 0);
    
nexttoken:
    if (isalpha(*p) || *p == '_' || *p == '$')
    {
        int       l;
        int       hash;
        char     *q;
        toktdef  *tab;
        int       found = FALSE;
        uchar     thischar;
        tokdfdef *df;
        
        for (hash = 0, l = 0, q = tok->toknam ;
             len != 0 && TOKISSYM(*p) && l < TOKNAMMAX ;
             (thischar = ((isupper(*p) && (ctx->tokcxflg & TOKCXCASEFOLD))
                         ? tolower(*p) : *p)),
             (hash = ((hash + thischar) & (TOKHASHSIZE - 1))),
             (*q++ = thischar), ++p, --len, ++l);
        *q = '\0';
        if (len != 0 && TOKISSYM(*p))
        {
            while (len != 0 && TOKISSYM(*p)) ++p, --len;
            errlog1(ctx->tokcxerr, ERR_TRUNC, ERRTSTR,
                    errstr(ctx->tokcxerr, tok->toknam, tok->toklen));
        }
        tok->toklen = l;
        tok->tokhash = hash;

        /*
         *   check for the special defined() preprocessor operator 
         */
        if (l == 9 && !memcmp(tok->toknam, "__DEFINED", (size_t)9)
            && len > 2 && *p == '(' && TOKISSYM(*(p+1)) && !isdigit(*(p+1)))
        {
            int symlen;
            
            /* find the matching ')', allowing only symbolic characters */
            ++p, --len;
            for (symlen = 0, q = p ; len && *p != ')' && TOKISSYM(*p) ;
                 ++p, --len, ++symlen) ;

            /* make sure we found the closing paren */
            if (!len || *p != ')')
                errsig(ctx->tokcxerr, ERR_BADISDEF);
            ++p, --len;

            /* check to see if it's defined */
            tok->toktyp = TOKTNUMBER;
            tok->tokval = (tok_find_define(ctx, q, symlen) != 0);
            goto done;
        }

        /* substitute the preprocessor #define, if any */
        if ((df = tok_find_define(ctx, tok->toknam, l)) != 0)
        {
            /* save the current parsing position */
            if (ctx->tokcxmlvl >= TOKMACNEST)
                errsig(ctx->tokcxerr, ERR_MACNEST);
            ctx->tokcxmsav[ctx->tokcxmlvl] = p;
            ctx->tokcxmsvl[ctx->tokcxmlvl] = len;
            ctx->tokcxmlvl++;

            /* point to the token's expansion and keep going */
            p = df->expan;
            len = df->explen;
            goto nexttoken;
        }
        
        /* look up in symbol table(s), if any */
        for (tab = ctx->tokcxstab ; tab ; tab = tab->toktnxt)
        {
            if (found = (*tab->toktfsea)(tab, tok->toknam, l, hash,
                                         &tok->toksym))
                break;
        }
        
        if (found && tok->toksym.tokstyp == TOKSTKW)
            tok->toktyp = tok->toksym.toksval;
        else
        {
            tok->toktyp = TOKTSYMBOL;
            if (!found) tok->toksym.tokstyp = TOKSTUNK;
        }
        goto done;
    }
    else if (isdigit(*p))
    {
        long acc = 0;
        
        /* check for octal/hex */
        if (*p == '0')
        {
            ++p, --len;
            if (len && (*p == 'x' || *p == 'X'))
            {
                /* hex */
                ++p, --len;
                while (len && TOKISHEX(*p))
                {
                    acc = (acc << 4) + TOKHEX2INT(*p);
                    ++p, --len;
                }
            }
            else
            {
                /* octal */
                while (len && TOKISOCT(*p))
                {
                    acc = (acc << 3) + TOKOCT2INT(*p);
                    ++p, --len;
                }
            }
        }
        else
        {
            /* decimal */
            while (len && isdigit(*p))
            {
                acc = (acc << 1) + (acc << 3) + TOKDEC2INT(*p);
                ++p, --len;
            }
        }
        tok->tokval = acc;
        tok->toktyp = TOKTNUMBER;
        goto done;
    }
    else if (*p == '"' || *p == '\'')
    {
        char  delim;                 /* closing delimiter we're looking for */
        char *strstart;                       /* pointer to start of string */
        int   warned;
        
        delim = *p;
        --len;
        strstart = ++p;

        if (delim == '"' && len >= 2 && *p == '<' && *(p+1) == '<')
        {
            /* save the current parsing position */
            if (ctx->tokcxmlvl >= TOKMACNEST)
                errsig(ctx->tokcxerr, ERR_MACNEST);
            ctx->tokcxmsav[ctx->tokcxmlvl] = p + 2;
            ctx->tokcxmsvl[ctx->tokcxmlvl] = len - 2;
            ctx->tokcxmlvl++;

            /* read from the special "<<" expansion string */
            p = &tokmac1[1];
            len = strlen(p);
            ctx->tokcxflg |= TOKCXFINMAC;
            goto nexttoken;
        }
        tok->toktyp = (delim == '"' ? TOKTDSTRING : TOKTSSTRING);
        
        tok->tokofs = (*ctx->tokcxsst)(ctx->tokcxscx);  /* start the string */
        for (warned = FALSE ;; )
        {
            if (len >= 2 && *p == '\\')
            {
                if (*(p+1) == '"' || *(p+1) == '\'')
                {
                    (*ctx->tokcxsad)(ctx->tokcxscx, strstart,
                                     (ushort)(p - strstart));
                    strstart = p + 1;
                }
                p += 2;
                len -= 2;
            }
            else if (len == 0 || *p == delim ||
                     (delim == '"' && len >= 2 && *p == '<' && *(p+1) == '<'
                      && !(ctx->tokcxflg & TOKCXFINMAC)))
            {
                (*ctx->tokcxsad)(ctx->tokcxscx, strstart,
                                 (ushort)(p - strstart));
                if (len == 0)
                {
                    if (ctx->tokcxmlvl)
                    {
                        ctx->tokcxmlvl--;
                        p = ctx->tokcxmsav[ctx->tokcxmlvl];
                        len = ctx->tokcxmsvl[ctx->tokcxmlvl];
                    }
                    else
                        (*ctx->tokcxsad)(ctx->tokcxscx, " ", (ushort)1);
                    
                    while (len == 0)
                    {
                        if (tokgetlin(ctx, FALSE))
                            errsig(ctx->tokcxerr, ERR_STREOF);
                        p = ctx->tokcxptr;
                        len = ctx->tokcxlen;

                        /* warn if it looks like the end of an object */
                        if (!warned && len && (*p == ';' || *p == '}'))
                        {
                            errlog(ctx->tokcxerr, ERR_STREND);
                            warned = TRUE;     /* warn only once per string */
                        }

                        /* scan past whitespace at start of line */
                        while (len && isspace(*p)) ++p, --len;
                    }
                    strstart = p;
                }
                else break;
            }
            else
                ++p, --len;
        }
        (*ctx->tokcxsend)(ctx->tokcxscx);                 /* end the string */
        if (len != 0 && *p == delim) ++p, --len;     /* move past delimiter */
        if (len != 0 && *p == '<')
        {
            /* save the current parsing position */
            if (ctx->tokcxmlvl >= TOKMACNEST)
                errsig(ctx->tokcxerr, ERR_MACNEST);
            ctx->tokcxmsav[ctx->tokcxmlvl] = p + 2;
            ctx->tokcxmsvl[ctx->tokcxmlvl] = len - 2;
            ctx->tokcxmlvl++;

            /* read from the "<<" expansion */
            p = tokmac1;
            len = strlen(p);
            ctx->tokcxflg |= TOKCXFINMAC;
        }
        goto done;
    }
    else if (len >= 2 && *p == '>' && *(p+1) == '>'
             && (ctx->tokcxflg & TOKCXFINMAC))
    {
        /* skip the ">>" */
        ctx->tokcxflg &= ~TOKCXFINMAC;
        p += 2;
        len -= 2;

        /* save the current parsing position */
        if (ctx->tokcxmlvl >= TOKMACNEST)
            errsig(ctx->tokcxerr, ERR_MACNEST);
        ctx->tokcxmsav[ctx->tokcxmlvl] = p;
        ctx->tokcxmsvl[ctx->tokcxmlvl] = len;
        ctx->tokcxmlvl++;

        if (*p == '"')
        {
            ++(ctx->tokcxmsav[ctx->tokcxmlvl - 1]);
            --(ctx->tokcxmsvl[ctx->tokcxmlvl - 1]);
            p = tokmac3;
            len = strlen(p);
        }
        else
        {
            p = tokmac2;
            len = strlen(p);
        }

        goto nexttoken;
    }
    else
    {
        tokscdef *sc;
        
        for (sc = ctx->tokcxsc[ctx->tokcxinx[*p]] ; sc ; sc = sc->tokscnxt)
        {
            if (toksceq(sc->tokscstr, p, sc->toksclen, len))
            {
                tok->toktyp = sc->toksctyp;
                p += sc->toksclen;
                len -= sc->toksclen;
                goto done;
            }
        }
        errsig(ctx->tokcxerr, ERR_INVTOK);
    }
    
done:
    ctx->tokcxptr = p;
    ctx->tokcxlen = len;
    return(tok->toktyp);
}

/* initialize a linear symbol table */
void toktlini(errctx, toktab, mem, siz)
errcxdef *errctx;
toktldef *toktab;                          /* table structure to initialize */
uchar    *mem;              /* memory block to use for symbols in the table */
uint      siz;                                      /* size of memory block */
{
    CLRSTRUCT(*toktab);
    
    /* initialize superclass data */
    toktab->toktlsc.toktfadd = toktladd;           /* set add-symbol method */
    toktab->toktlsc.toktfsea = toktlsea;         /* set search-table method */
    toktab->toktlsc.toktfeach = toktleach;             /* set 'each' method */
    toktab->toktlsc.toktfset = toktlset;             /* set 'update' method */
    toktab->toktlsc.tokterr = errctx;         /* set error handling context */
    
    /* initialize class data */
    toktab->toktlptr = mem;
    toktab->toktlnxt = mem;
    toktab->toktlsiz = siz;
}

/* add a symbol to a linear symbol table */
void toktladd(toktab1, name, namel, typ, val, hash)
toktdef  *toktab1;
char     *name;
int       typ;
int       val;
int       hash;
{
    int       siz = sizeof(toks1def) + namel;
    toksdef  *sym;
    toktldef *toktab = (toktldef *)toktab1;
    
    VARUSED(hash);
    
    if (toktab->toktlsiz < siz)
        errsig(toktab->toktlsc.tokterr, ERR_NOLCLSY);
    
    sym = (toksdef *)toktab->toktlnxt;
    siz = osrndsz(siz);
    toktab->toktlnxt += siz;
    if (siz > toktab->toktlsiz) toktab->toktlsiz = 0;
    else toktab->toktlsiz -= siz;

    /* set up symbol */
    sym->toksval = val;
    sym->tokslen = namel;
    sym->tokstyp = typ;
    sym->toksfr  = 0;
    memcpy(sym->toksnam, name, (size_t)(namel + 1));
    
    /* indicate there's one more symbol in the table */
    ++(toktab->toktlcnt);
}

/* delete all symbols from a linear symbol table */
void toktldel(tab)
toktldef *tab;
{
    tab->toktlcnt = 0;
    tab->toktlsiz += tab->toktlnxt - tab->toktlptr;
    tab->toktlnxt = tab->toktlptr;
}

/* call a function for every symbol in a linear symbol table */
void toktleach(tab1, cb, ctx)
toktdef   *tab1;
int      (*cb)(/*_ dvoid *ctx, toksdef *sym _*/);
dvoid     *ctx;                            /* context for callback function */
{
    toksdef  *p;
    uint      cnt;
    toktldef *tab = (toktldef *)tab1;
    
    for (p = (toksdef *)tab->toktlptr, cnt = tab->toktlcnt ; cnt ; --cnt )
    {
        (*cb)(ctx, p);
        p = (toksdef *)(((uchar *)p)
                        + osrndsz(p->tokslen + sizeof(toks1def)));
    }
}

/* search a linear symbol table */
int toktlsea(tab1, name, namel, hash, ret)
toktdef  *tab1;
char     *name;
int       namel;
int       hash;
toksdef  *ret;
{
    toksdef  *p;
    uint      cnt;
    toktldef *tab = (toktldef *)tab1;
    
    VARUSED(hash);
    
    for (p = (toksdef *)tab->toktlptr, cnt = tab->toktlcnt ; cnt ; --cnt )
    {
        if (p->tokslen == namel && !memcmp(p->toksnam, name, (size_t)namel))
        {
            memcpy(ret, p, (size_t)(sizeof(toks1def) + namel));
            return(TRUE);
        }
        
        p = (toksdef *)(((uchar *)p)
                        + osrndsz(p->tokslen + sizeof(toks1def)));
    }

    /* nothing found - indicate by returning FALSE */
    return(FALSE);
}

/* update a symbol in a linear symbol table */
void toktlset(tab1, new)
toktdef  *tab1;
toksdef  *new;
{
    toksdef  *p;
    uint      cnt;
    toktldef *tab = (toktldef *)tab1;
    
    VARUSED(hash);
    
    for (p = (toksdef *)tab->toktlptr, cnt = tab->toktlcnt ; cnt ; --cnt )
    {
        if (p->tokslen == new->tokslen
            && !memcmp(p->toksnam, new->toksnam, (size_t)new->tokslen))
        {
            p->toksval = new->toksval;
            p->tokstyp = new->tokstyp;
            return;
        }
        
        p = (toksdef *)(((uchar *)p)
                        + osrndsz(p->tokslen + sizeof(toks1def)));
    }
}

tokcxdef *tokcxini(errctx, mcmctx, sctab)
errcxdef *errctx;
mcmcxdef *mcmctx;
tokldef  *sctab;
{
    int       i;
    int       cnt;
    tokldef  *p;
    uchar     c;
    uchar     index[256];
    tokcxdef *ret;
    tokscdef *sc;
    ushort    siz;
    
    /* set up index table: finds tokcxsc entry from character value */
    memset(index, 0, (size_t)sizeof(index));
    for (i = cnt = 0, p = sctab ; c = p->toklstr[0] ; ++cnt, ++p)
        if (!index[c]) index[c] = ++i;
    
    /* allocate memory for table plus the tokscdef's */
    siz = sizeof(tokcxdef) + (i * sizeof(tokscdef *))
          + ((cnt + 1) * sizeof(tokscdef));
    ret = (tokcxdef *)mchalo(errctx, siz, "tokcxini");
    memset(ret, 0, (size_t)siz);
    
    /* copy the index, set up fixed part */
    memcpy(ret->tokcxinx, index, sizeof(ret->tokcxinx));
    ret->tokcxerr = errctx;
    ret->tokcxmem = mcmctx;
    
    /* force the first toknext() to read a line */
    ret->tokcxptr = "\000";
    
    /* figure where the tokscdef's go (right after sc pointer array) */
    sc = (tokscdef *)&ret->tokcxsc[i+1];
    
    /* set up the individual tokscdef entries, and link into lists */
    for (p = sctab ; c = p->toklstr[0] ; ++p, ++sc)
    {
        size_t len;
        
        sc->toksctyp = p->tokltyp;
        len = sc->toksclen = strlen(p->toklstr);
        memcpy(sc->tokscstr, p->toklstr, len);
        sc->tokscnxt = ret->tokcxsc[index[c]];
        ret->tokcxsc[index[c]] = sc;
    }
    
    return(ret);
}

/* add an include path to a tokdef */
void tokaddinc(ctx, path, pathlen)
tokcxdef *ctx;
char     *path;
int       pathlen;
{
    tokpdef *new;
    tokpdef *last;
    
    /* find the tail of the include path list, if any */
    for (last = ctx->tokcxinc ; last && last->tokpnxt ; last = last->tokpnxt);
    
    /* allocate storage for and set up a new path structure */
    new = (tokpdef *)mchalo(ctx->tokcxerr,
                            (ushort)(sizeof(tokpdef) + pathlen - 1),
                            "tokaddinc");
    new->tokplen = pathlen;
    new->tokpnxt = (tokpdef *)0;
    memcpy(new->tokpdir, path, (size_t)pathlen);
    
    /* link in at end of list (if no list yet, new becomes first entry) */
    if (last)
        last->tokpnxt = new;
    else
        ctx->tokcxinc = new;
}

