/*
 * x_init.c
 *
 * X interface, initialisation
 *
 */
/*
 * Copyright (c) 1998-2000 Daniel Schepler
 * Permission is granted to distribute, copy, and modify this source and
 * resulting binaries under the turns of the Artistic License.  This
 * should have been included in the COPYING file along with this
 * distribution.
 *
 * xfrotz comes with NO WARRANTY.  See the Artistic License for more
 * information.
 */

#include "../common/frotz.h"
#include "x_frotz.h"

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <X11/Intrinsic.h>

f_setup_t f_setup;
u_setup_t u_setup;

/* Variables to save from os_process_arguments for use in os_init_screen */
static int saved_argc;
static char **saved_argv;
static char *user_bg, *user_fg, *user_title, *user_geometry;
static char user_sizex[5], user_sizey[5]; /* for custom window size */
static int user_tandy_bit;
static int user_random_seed = -1;
char *x_class;
char *x_name;

static int *alt_prompt=FALSE;

void os_init_setup(void)
{

	f_setup.attribute_assignment = 0;
	f_setup.attribute_testing = 0;
	f_setup.context_lines = 0;
	f_setup.object_locating = 0;
	f_setup.object_movement = 0;
	f_setup.left_margin = 0;
	f_setup.right_margin = 0;
	f_setup.ignore_errors = 0;
	f_setup.piracy = 0;		/* enable the piracy opcode */
	f_setup.undo_slots = MAX_UNDO_SLOTS;
	f_setup.expand_abbreviations = 0;
	f_setup.script_cols = 80;
	f_setup.save_quetzal = 1;
	f_setup.sound = 1;
	f_setup.err_report_mode = ERR_DEFAULT_REPORT_MODE;

	u_setup.disable_color = 0;
	u_setup.force_color = 0;
	u_setup.foreground_color = -1;
	u_setup.background_color = -1;
	u_setup.screen_width = -1;
	u_setup.screen_height = -1;
	u_setup.random_seed = -1;
	u_setup.random_seed = -1;
	u_setup.tandy_bit = 0;
	u_setup.current_text_style = 0;
			/* Since I can't use attr_get, which
			would make things easier, I need
			to use the same hack the MS-DOS port
			does...keep the current style in a
			global variable. */
	u_setup.plain_ascii = 0; /* true if user wants to disable Latin-1 */
	u_setup.curses_active = 0;      /* true if os_init_screen has run */
	/* u_setup.interpreter = INTERP_DEFAULT; */
	u_setup.current_color = 0;
	u_setup.color_enabled = FALSE;

}

/*
 * os_fatal
 *
 * Display error message and exit program.
 *
 */

void os_fatal (const char *s)
{
  fprintf(stderr, "%s\n", s);
  exit(1);
}/* os_fatal */

/*
 * os_process_arguments
 *
 * Handle command line switches. Some variables may be set to activate
 * special features of Frotz:
 *
 *     option_attribute_assignment
 *     option_attribute_testing
 *     option_context_lines
 *     option_object_locating
 *     option_object_movement
 *     option_left_margin
 *     option_right_margin
 *     option_ignore_errors
 *     option_piracy
 *     option_undo_slots
 *     option_expand_abbreviations
 *     option_script_cols
 *
 * The global pointer "story_name" is set to the story file name.
 *
 */

static void parse_boolean(char *value, void *parsed) {
  int *parsed_var = (int *) parsed;

  if (! strcasecmp(value, "on") || ! strcasecmp(value, "true") ||
      ! strcasecmp(value, "yes"))
    *parsed_var = 1;
  else if (! strcasecmp(value, "off") || ! strcasecmp(value, "false") ||
           ! strcasecmp(value, "no"))
    *parsed_var = 0;
  else
    fprintf(stderr, "Warning: invalid boolean resource `%s'\n", value);
}

static void parse_int(char *value, void *parsed) {
  int *parsed_var = (int *) parsed;
  char *parse_end;

  int result = (int) strtol(value, &parse_end, 10);
  if (*parse_end || !(*value))
    fprintf(stderr, "Warning: invalid integer resource `%s'\n", value);
  else
    *parsed_var = result;
}

static void parse_zstrict(char *value, void *parsed) {
  int *parsed_var = (int *) parsed;

  if (! strcasecmp(value, "on") || ! strcasecmp(value, "true") ||
      ! strcasecmp(value, "yes") || ! strcasecmp(value, "always"))
    *parsed_var = ERR_REPORT_ALWAYS;
  else if (! strcasecmp(value, "off") || ! strcasecmp(value, "false") ||
           ! strcasecmp(value, "no") || ! strcasecmp(value, "never"))
    *parsed_var = ERR_REPORT_NEVER;
  else if (! strcasecmp(value, "once"))
    *parsed_var = ERR_REPORT_ONCE;
  else if (! strcasecmp(value, "fatal"))
    *parsed_var = ERR_REPORT_FATAL;
  else {
    char *parse_end;
    int result = (int) strtol(value, &parse_end, 10);

    if (*parse_end || !(*value) || result < ERR_REPORT_NEVER ||
        result > ERR_REPORT_FATAL)
      fprintf(stderr, "Warning: invalid zstrict level resource `%s'\n",
              value);
    else
      *parsed_var = result;
  }
}

static void parse_string(char *value, void *parsed) {
  *((char **) parsed) = value;
}

void os_process_arguments (int argc, char *argv[])
{
  static XrmOptionDescRec options[] = {
    { "-aa", ".watchAttrAssign", XrmoptionNoArg, (caddr_t) "true" },
    { "+aa", ".watchAttrAssign", XrmoptionNoArg, (caddr_t) "false" },
    { "-at", ".watchAttrTest", XrmoptionNoArg, (caddr_t) "true" },
    { "+at", ".watchAttrTest", XrmoptionNoArg, (caddr_t) "false" },
    { "-c", ".contextLines", XrmoptionSepArg, (caddr_t) NULL },
    { "-ol", ".watchObjLocating", XrmoptionNoArg, (caddr_t) "true" },
    { "+ol", ".watchObjLocating", XrmoptionNoArg, (caddr_t) "false" },
    { "-om", ".watchObjMovement", XrmoptionNoArg, (caddr_t) "true" },
    { "+om", ".watchObjMovement", XrmoptionNoArg, (caddr_t) "false" },
    { "-lm", ".leftMargin", XrmoptionSepArg, (caddr_t) NULL },
    { "-rm", ".rightMargin", XrmoptionSepArg, (caddr_t) NULL },
    { "-e", ".ignoreErrors", XrmoptionNoArg, (caddr_t) "true" },
    { "+e", ".ignoreErrors", XrmoptionNoArg, (caddr_t) "false" },
    { "-p", ".piracy", XrmoptionNoArg, (caddr_t) "true" },
    { "+p", ".piracy", XrmoptionNoArg, (caddr_t) "false" },
    { "-t", ".tandy", XrmoptionNoArg, (caddr_t) "true" },
    { "+t", ".tandy", XrmoptionNoArg, (caddr_t) "false" },
    { "-u", ".undoSlots", XrmoptionSepArg, (caddr_t) NULL },
    { "-x", ".expandAbbrevs", XrmoptionNoArg, (caddr_t) "true" },
    { "+x", ".expandAbbrevs", XrmoptionNoArg, (caddr_t) "false" },
    { "-sc", ".scriptColumns", XrmoptionSepArg, (caddr_t) NULL },
    { "-rs", ".randomSeed", XrmoptionSepArg, (caddr_t) NULL },
    /* I can never remember whether it's zstrict or strictz */
    /* { "-zs", ".zStrict", XrmoptionSepArg, (caddr_t) NULL }, */
    /* It's "StrictZ" --- val */
    { "-sz", ".StrictZ", XrmoptionSepArg, (caddr_t) NULL },
    { "-fn-b", ".fontB", XrmoptionSepArg, (caddr_t) NULL },
    { "-fn-i", ".fontI", XrmoptionSepArg, (caddr_t) NULL },
    { "-fn-bi", ".fontBI", XrmoptionSepArg, (caddr_t) NULL },
    { "-fn-f", ".fontF", XrmoptionSepArg, (caddr_t) NULL },
    { "-fn-fb", ".fontFB", XrmoptionSepArg, (caddr_t) NULL },
    { "-fn-fi", ".fontFI", XrmoptionSepArg, (caddr_t) NULL },
    { "-fn-fbi", ".fontFBI", XrmoptionSepArg, (caddr_t) NULL },
    { "-fn-z", ".fontZ", XrmoptionSepArg, (caddr_t) NULL },
    { "-geometry", ".geometry", XrmoptionSepArg, (caddr_t) NULL },
    { "-m", ".altprompt", XrmoptionNoArg, (caddr_t) "true" },
    { "+m", ".altprompt", XrmoptionNoArg, (caddr_t) "false" },
    { "-title", ".title", XrmoptionSepArg, (caddr_t) NULL }
  };
  static struct {
    char *class;
    char *name;
    void (*parser)(char *, void *);
    void *ptr;
  } vars[] = {
    { ".WatchAttribute", ".watchAttrAssign", parse_boolean,
      &option_attribute_assignment },
    { ".WatchAttribute", ".watchAttrTest", parse_boolean,
      &option_attribute_testing },
    { ".ContextLines", ".contextLines", parse_int,
      &option_context_lines },
    { ".WatchObject", ".watchObjLocating", parse_boolean,
      &option_object_locating },
    { ".WatchObject", ".watchObjMovement", parse_boolean,
      &option_object_movement },
    { ".Margin", ".leftMargin", parse_int,
      &option_left_margin },
    { ".Margin", ".rightMargin", parse_int,
      &option_right_margin },
    { ".IgnoreErrors", ".ignoreErrors", parse_boolean,
      &option_ignore_errors },
    { ".Piracy", ".piracy", parse_boolean,
      &option_piracy },
    { ".Tandy", ".tandy", parse_boolean,
      &user_tandy_bit },
    { ".UndoSlots", ".undoSlots", parse_int,
      &option_undo_slots },
    { ".ExpandAbbrevs", ".expandAbbrevs", parse_boolean,
      &option_expand_abbreviations },
    { ".ScriptColumns", ".scriptColumns", parse_int,
      &option_script_cols },
    { ".RandomSeed", ".randomSeed", parse_int,
      &user_random_seed },
    { ".ZStrict", ".zStrict", parse_zstrict,
      &f_setup.err_report_mode },
    { ".Background", ".background", parse_string,
      &user_bg },
    { ".Foreground", ".foreground", parse_string,
      &user_fg },
    { ".Font", ".font", parse_string,
      &font_names[0] },
    { ".Font", ".fontB", parse_string,
      &font_names[1] },
    { ".Font", ".fontI", parse_string,
      &font_names[2] },
    { ".Font", ".fontBI", parse_string,
      &font_names[3] },
    { ".Font", ".fontF", parse_string,
      &font_names[4] },
    { ".Font", ".fontFB", parse_string,
      &font_names[5] },
    { ".Font", ".fontFI", parse_string,
      &font_names[6] },
    { ".Font", ".fontFBI", parse_string,
      &font_names[7] },
    { ".Font", ".fontZ", parse_string,
      &font_names[8] },
    { ".Geometry", ".geometry", parse_string,
      &user_geometry },
    { ".AltPrompt", ".altprompt", parse_boolean,
      &alt_prompt },
    { ".Title", ".title", parse_string,
      &user_title }
  };
  XtAppContext app_context;
  char *str_type_return;
  char *class_buf, *name_buf, *class_append, *name_append;
  int i;
  XrmValue value;

  saved_argv = malloc(sizeof(char *) * argc);
  memcpy(saved_argv, argv, sizeof(char *) * argc);
  saved_argc = argc;
  app_context = XtCreateApplicationContext();
  dpy = XtOpenDisplay(app_context, NULL, NULL, "XFrotz",
                      options, XtNumber(options), &argc, argv);
  if (dpy == NULL)
    os_fatal("Could not open display.");
  if (argc != 2)
    /*    os_fatal("Usage: xfrotz [options] storyfile");*/
    os_fatal(USAGE);
  story_name = argv[1];

  XtGetApplicationNameAndClass(dpy, &x_name, &x_class);

  class_buf = malloc(strlen(x_class) + 20);
  strcpy(class_buf, x_class);
  class_append = strchr(class_buf, 0);
  name_buf = malloc(strlen(x_name) + 20);
  strcpy(name_buf, x_name);
  name_append = strchr(name_buf, 0);

  for (i = 0; i < XtNumber(vars); i++) {
    strcpy(class_append, vars[i].class);
    strcpy(name_append, vars[i].name);
    if (XrmGetResource(XtDatabase(dpy), name_buf, class_buf,
                       &str_type_return, &value))
      vars[i].parser((char *) value.addr, vars[i].ptr);
  }
}/* os_process_arguments */

/*
 * os_init_screen
 *
 * Initialise the IO interface. Prepare the screen and other devices
 * (mouse, sound board). Set various OS depending story file header
 * entries:
 *
 *     h_config (aka flags 1)
 *     h_flags (aka flags 2)
 *     h_screen_cols (aka screen width in characters)
 *     h_screen_rows (aka screen height in lines)
 *     h_screen_width
 *     h_screen_height
 *     h_font_height (defaults to 1)
 *     h_font_width (defaults to 1)
 *     h_default_foreground
 *     h_default_background
 *     h_interpreter_number
 *     h_interpreter_version
 *     h_user_name (optional; not used by any game)
 *
 * Finally, set reserve_mem to the amount of memory (in bytes) that
 * should not be used for multiple undo and reserved for later use.
 *
 * (Unix has a non brain-damaged memory model which dosen't require such
 *  ugly hacks, neener neener neener. --GH :)
 *
 */

Display *dpy;
Window main_window = 0;
const XFontStruct *current_font_info;
GC normal_gc, reversed_gc, bw_gc, cursor_gc, current_gc;

void os_init_screen (void)
{
  XSetWindowAttributes window_attrs;
  XSizeHints *size_hints;
  XWMHints *wm_hints;
  XClassHint *class_hint;
  const char *story_basename;
  char *window_title;
  int font_width, font_height;
  unsigned int window_height, window_width, i=0, i2=0;
  char x=0;

  if (h_version == V6)  /* default for ZCode VM6: scale graphics to 2x */ 
    {                   /* see also hard-coded calculations in x_pic.c */
      window_width=640;
      window_height=400;
    }
  else
    if(user_geometry=='\0')
      {
	window_width=600;
	window_height=700;
      }
    else
      {
	for(; i<strlen(user_geometry) && user_geometry!='+'; i++)
	  if(x==0 && user_geometry[i]!='x')
	    user_sizex[i]=user_geometry[i];
	  else if(x==1)
	    {
	      user_sizey[i2]=user_geometry[i];
	      i2++;
	    }
	  else 
	    {
	      x=1;
	      continue;
	    }
	window_height=atoi(user_sizey); /* size in pixels */
	window_width=atoi(user_sizex); /* size in pixels */
      }

  /*
  fprintf(stderr,"%d x %d\n",window_width,window_height);
  */

  XGCValues gc_setup;

  /* First, configuration parameters get set up */
  if (h_version == V3 && user_tandy_bit != 0)
    h_config |= CONFIG_TANDY;
  if (h_version == V3 && alt_prompt == TRUE)
    h_config |= CONFIG_NOSTATUSLINE;
  if (h_version == V3)
    h_config |= CONFIG_SPLITSCREEN | CONFIG_PROPORTIONAL;
  if (h_version >= V4)
    h_config |= CONFIG_BOLDFACE | CONFIG_EMPHASIS | CONFIG_FIXED;
  if (h_version >= V5) 
    {
      h_flags |= GRAPHICS_FLAG | UNDO_FLAG | MOUSE_FLAG | COLOUR_FLAG;
#ifdef NO_SOUND
      h_flags &= ~SOUND_FLAG;
#else
      h_flags |= SOUND_FLAG;
#endif
    }
  h_config |= CONFIG_TIMEDINPUT;
  if (h_version >= V6) 
    {
      h_config |= CONFIG_PICTURES;
      h_flags &= ~MENU_FLAG;
    }

  if (h_version >= V5 && (h_flags & UNDO_FLAG) && option_undo_slots == 0)
    h_flags &= ~UNDO_FLAG;

  h_interpreter_number = INTERP_DEC_20;
  h_interpreter_version = 'F';

  x_init_colour(user_bg, user_fg);

  /*  window_attrs.background_pixel = def_bg_pixel;*/
  window_attrs.backing_store = NotUseful; /* Always */
  window_attrs.save_under = FALSE;
  /*  window_attrs.event_mask = ExposureMask | KeyPressMask | ButtonPressMask; */
  window_attrs.event_mask = ExposureMask | KeyPressMask | KeyReleaseMask |
                            ButtonPressMask | ButtonReleaseMask | FocusChangeMask;
  window_attrs.background_pixel = None; 
  window_attrs.override_redirect = FALSE;
  main_window = XCreateWindow(dpy, DefaultRootWindow(dpy),
			      0, 0, window_height, window_width, 0, CopyFromParent,
			      InputOutput, CopyFromParent,
			      CWBackPixel | CWBackingStore |
			      CWOverrideRedirect | CWSaveUnder | CWEventMask,
			      &window_attrs);
                                /*CWEventMask | CWBackPixel,*/

  size_hints = XAllocSizeHints();
  size_hints->min_width = size_hints->max_width = size_hints->base_width = window_width;
  size_hints->min_height = size_hints->max_height =
    size_hints->base_height = window_height;

  size_hints->flags = PMinSize | PMaxSize | PBaseSize;

  wm_hints = XAllocWMHints();
  wm_hints->input = True;
  wm_hints->initial_state = NormalState;
  wm_hints->flags = InputHint | StateHint;

  class_hint = XAllocClassHint();
  class_hint->res_name = x_name;
  class_hint->res_class = x_class;

  story_basename = strrchr(story_name, '/');
  if (story_basename == NULL)
    story_basename = story_name;
  else
    story_basename++;
  if (user_title == NULL) 
    {
      window_title = malloc(strlen(story_basename) + 14);
      sprintf(window_title, "xfrotz --- %s", story_basename);
    }
  else
    {
      window_title = malloc(strlen(user_title) + 4);
      sprintf(window_title, "%s", user_title);
    }
  

  XmbSetWMProperties(dpy, main_window, window_title, story_basename,
                     saved_argv, saved_argc, size_hints, wm_hints, class_hint);

  XMapWindow(dpy, main_window);
  free(window_title);

  current_font_info = get_font(TEXT_FONT, 0);
  if (! current_font_info)
    os_fatal("Could not open default font");

  h_screen_width = window_width;
  h_screen_height = window_height;
  os_font_data(FIXED_WIDTH_FONT, &font_height, &font_width);
  h_font_height = font_height; h_font_width = font_width;
  h_screen_cols = window_width / h_font_width;
  h_screen_rows = window_height / h_font_height;

  fg_pixel = def_fg_pixel;
  bg_pixel = def_bg_pixel;

  gc_setup.function = GXcopy;
  gc_setup.foreground = fg_pixel;
  gc_setup.background = bg_pixel;
  gc_setup.fill_style = FillSolid;
  gc_setup.font = current_font_info->fid;
  normal_gc = XCreateGC(dpy, main_window,
                        GCFunction | GCForeground | GCBackground |
                        GCFillStyle | GCFont, &gc_setup);
  gc_setup.foreground = bg_pixel;
  gc_setup.background = fg_pixel;
  reversed_gc = XCreateGC(dpy, main_window,
                          GCFunction | GCForeground | GCBackground |
                          GCFillStyle | GCFont, &gc_setup);
  gc_setup.foreground = ~0UL;
  gc_setup.background = 0UL;
  bw_gc = XCreateGC(dpy, main_window,
                    GCFunction | GCForeground | GCBackground |
                    GCFillStyle | GCFont, &gc_setup);
  gc_setup.background = 0UL;
  gc_setup.foreground = bg_pixel ^ fg_pixel;
  gc_setup.function = GXxor;
  cursor_gc = XCreateGC(dpy, main_window,
                        GCFunction | GCForeground | GCBackground |
                        GCFillStyle, &gc_setup);
}/* os_init_screen */

/*
 * os_reset_screen
 *
 * Reset the screen before the program stops.
 *
 */

void os_reset_screen (void)
{
  os_set_text_style(0);
  print_string("[Hit any key to exit.]");
  flush_buffer();
  os_set_text_style(0);
  os_read_key(0, FALSE);
}/* os_reset_screen */

/*
 * os_restart_game
 *
 * This routine allows the interface to interfere with the process of
 * restarting a game at various stages:
 *
 *     RESTART_BEGIN - restart has just begun
 *     RESTART_WPROP_SET - window properties have been initialised
 *     RESTART_END - restart is complete
 *
 */

void os_restart_game (int stage)
{
}

/*
 * os_random_seed
 *
 * Return an appropriate random seed value in the range from 0 to
 * 32767, possibly by using the current system time.
 *
 */

int os_random_seed (void)
{
  if (user_random_seed == -1)
    return time(0) & 0x7fff;
  else
    return user_random_seed;
}/* os_random_seed */


#define __UNIX_PORT_FILE

#include <string.h>
#include <unistd.h>
#include <ctype.h>


/*
 * pathopen
 *
 * Given a standard Unix-style path and a filename, search the path for
 * that file.  If found, return a pointer to that file and put the full
 * path where the file was found in fullname.
 *
 */

FILE *pathopen(const char *name, const char *p, const char *mode, char *fullname)
{
  FILE *fp;
  char buf[FILENAME_MAX + 1];
  char *bp, lastch;
  
  lastch = 'a';	/* makes compiler shut up */
  
  while (*p) 
    {
      bp = buf;
      while (*p && *p != PATHSEP)
	lastch = *bp++ = *p++;
      if (lastch != DIRSEP)
	*bp++ = DIRSEP;
      strcpy(bp, name);
      if ((fp = fopen(buf, mode)) != NULL) 
	{
	  strncpy(fullname, buf, FILENAME_MAX);
	  return fp;
	}
      if (*p)
	p++;
    }
  return NULL;
} /* FILE *pathopen() */

/*
 * os_path_open
 *
 * Open a file in the current directory.  If this fails, then search the
 * directories in the ZCODE_PATH environmental variable.  If that's not
 * defined, search INFOCOM_PATH.
 *
 */

FILE *os_path_open(const char *name, const char *mode)
{
  FILE *fp;
  char buf[FILENAME_MAX + 1];
  char *p;
  
  /* Let's see if the file is in the currect directory */
  /* or if the user gave us a full path. */
  if ((fp = fopen(name, mode))) 
    {
      return fp;
    }
  
  /* If zcodepath is defined in a config file, check that path. */
  /* If we find the file a match in that path, great. */
  /* Otherwise, check some environmental variables. */
  if (option_zcode_path != NULL) 
    {
      if ((fp = pathopen(name, option_zcode_path, mode, buf)) != NULL) 
	{
	  strncpy(story_name, buf, FILENAME_MAX);
	  return fp;
	}
    }
  
  if ( (p = getenv(PATH1) ) == NULL)
    if ( (p = getenv(PATH2) ) == NULL)
      p = ZCODEPATH_DEF;
  
  if (p != NULL) 
    {
      fp = pathopen(name, p, mode, buf);
      strncpy(story_name, buf, FILENAME_MAX);
      return fp;
    }
  return NULL;	/* give up */
} /* os_path_open() */

