
/* advint.c - an interpreter for adventure games */
/*
	Copyright (c) 1986, by David Michael Betz
	All rights reserved
*/

#ifdef CURSES
#undef DUMB
#endif
#ifdef DUMB
#undef CURSES
#endif

#pragma lint -1

#if STACKDEBUG
#pragma debug 24
#endif

#if OPTIMIZE
#pragma optimize 9
#endif

/*#include "intproto.h"*/ /* TUU */
#include <stdlib.h> /* TUU */
#include "interpreter.h"
#include "../shared/dbaccess.h"
#include <setjmp.h>

/* global variables */
jmp_buf restart;
int moreprompts=TRUE;

/* external variables */
extern int h_init;
extern int h_update;
extern int h_before;
extern int h_after;
extern int h_error;

/* interpreter identification */


extern char f_reverse; /* from term_curses.c */

/* main - the main routine */
int main(int argc,char *argv[])
{
    char *fname,*lname;
    char storyfile[256];
    int rows,cols,i;
    char* envrows=getenv("LINES");
    char* envcols=getenv("COLUMNS");
    FILE *ftest;

    fname = NULL;
    lname = NULL;
    rows = 24;
    cols = 80;

    if(envrows != NULL) rows=atoi(envrows);
    if(envcols != NULL) cols=atoi(envcols);

    /* parse the command line */
    for (i = 1; i < argc; i++)
	if (argv[i][0] == '-')
	    switch (argv[i][1]) {
	    case 't':
		    lname = &argv[i][2];
	    	    break;
#ifdef CURSES
	    case 'r':
	            f_reverse=TRUE;
	            break;
#endif
#ifdef DUMB
	    case 'l':
		    rows = atoi(&argv[i][2]);
		    break;
	    case 'c':
		    cols = atoi(&argv[i][2]);
		    break;
	    case 'd':
	            moreprompts=FALSE;
	    	    break;
#endif
	    }
	else
	    fname = argv[i];

    if (fname == NULL) 
      {
	printf(MSG_VERSION);
	printf(MSG_USAGE);
	exit(1);
    }

    /* test if the game file exists; be more flexible with file suffices:
       hanlde "foo" as well as "foo.dat" or "foo.<whatever>" and append 
       .dat only if just "foo" is given (more flexible with menu or 
       helper frontends)
    */
    if(strrchr(fname,'.')==NULL)
      strcat(fname,".dat");

    strcpy(storyfile,fname);

    /* try to open the data file */

    if ((ftest = fopen(storyfile,"r"))==NULL)
      {
	sprintf(storyfile, "%s/%s",STORYDIR,fname);
	if ((ftest = fopen(storyfile,"r"))==NULL)
	  {
	    fprintf(stderr,"There is no '%s'.\n", fname);
	    exit(1);
	  }
      }
    else
      fclose(ftest);

    /* game file exists, let's run it */

    /* initialize the database */
    db_init(storyfile);

    /* initialize terminal i/o */
    trm_init(rows,cols,lname);

    /* play the game */
    play();
}


/* handler loop states */
enum _handler_loop {
    HL_ERROR = 0,   /* everthing has gone wrong */
    HL_INIT,
    HL_UPDATE,
    HL_PARSE,
    HL_BEFORE,
    HL_ACTION,
    HL_AFTER,
    HL_PARSE_ERROR
};

/* play - the main loop */
void play()
{
    int hl_state, hdl_status;

                     /* error     FINISH    CHAIN     ABORT   */
    int hl_states[] = { HL_ERROR, HL_AFTER, HL_ERROR, HL_UPDATE };

    /* establish the restart point */
    setjmp(restart);

    hl_state = HL_INIT;

    for(;;) {
#ifdef DEBUG
       printf("handler loop : %d\n", hl_state);
#endif
       switch(hl_state) {
           case HL_INIT: if( (hdl_status = execute(h_init)) != CHAIN ) {
                             hl_state = hl_states[hdl_status];
                             break;
                         }
                         /* CHAIN => continue with next handler UPDATE */

           case HL_UPDATE: if( (hdl_status = execute(h_update)) != CHAIN ) {
                             hl_state = hl_states[hdl_status];
                             break;
                           }
                         /* CHAIN => continue with next handler Parse */

           case HL_PARSE:
                         if( !next() )
                              if( !parse() ) {
                                 hl_state = HL_PARSE_ERROR;
                                 break;
                              }
                   /* next & Parse ok => continue with next handler BEFORE */

           case HL_BEFORE: if( (hdl_status = execute(h_before)) != CHAIN ) {
                             hl_state = hl_states[hdl_status];
                             break;
                           }
                         /* CHAIN => continue with next handler Action */

           case HL_ACTION: if((hdl_status = execute(getafield(getvalue(V_ACTION),A_CODE))) != CHAIN ) {
                             hl_state = hl_states[hdl_status];
                             break;
                         }
                         /* CHAIN => continue with next handler AFTER */

           case HL_AFTER: if( (hdl_status = execute(h_after)) != CHAIN )
                             hl_state = hl_states[hdl_status];
                           else
                             hl_state = HL_UPDATE;
                          break;

           case HL_ERROR: trm_str("All in the life boats!!\n");
                          trm_str("This should have never happend.\n");
                          trm_str("Please report this error to RISC!\n");
                          hl_state = HL_UPDATE;
                          break;

           case HL_PARSE_ERROR: if((hdl_status = execute(h_error)) != CHAIN)
                                    hl_state = hl_states[hdl_status];
                                else
                                    hl_state = HL_UPDATE;
                                break;

           default : { char buf[64];

                     trm_str("DESASTER has begun.\n");
                     sprintf(buf,"Handler-loop with unknown HL (%d).\n",hl_state);
                     trm_str(buf);
                     trm_str("Please report this error!\n");
                     exit(1);
                     break;
                     }

       } /* switch(hl_state) */
    }
}

#ifdef OLDHANDLER
void play(void)
{
    /* establish the restart point */
    setjmp(restart);

    /* execute the initialization code */
    execute(h_init);

    /* turn handling loop */
    for (;;) {

	/* execute the update code */
	execute(h_update);

	/* parse the next input command */
	if (parse()) {
	    if (single())
		while (next() && single())
		    ;
	}

	/* parse error, call the error handling code */
	else
	    execute(h_error);
    }
}
#endif

/* single - handle a single action */
int single(void)
{
    /* execute the before code */
    switch (execute(h_before)) {
    case ABORT:	/* before handler aborted sequence */
	return (FALSE);
    case CHAIN:	/* execute the action handler */
	if (execute(getafield(getvalue(V_ACTION),A_CODE)) == ABORT)
	    return (FALSE);
    case FINISH:/* execute the after code */
	if (execute(h_after) == ABORT)
	    return (FALSE);
	break;
    }
    return (TRUE);
}

/* error - print an error message and exit */
void error(char *msg)
{
    trm_str(msg);
    trm_chr('\n');
    exit(1);
}


int advsave(char *hdr,int hlen,char *save,int slen)
{
    char fname[50];
    FILE *fd;

    trm_str("File name? ");
    trm_get(fname);

    /* add the extension */
    strcat(fname,".sav");

    /* create the data file */
    if ((fd = fopen(fname,"w")) == 0)
	return (0);

    /* write the header */
    if (fwrite(hdr,1,hlen,fd) != hlen) {
	fclose(fd);
	return (0);
    }

    /* write the data */
    if (fwrite(save,1,slen,fd) != slen) {
	fclose(fd);
	return (0);
    }

    /* close the file and return successfully */
    fclose(fd);
    return (1);
}

int advrestore(char *hdr,int hlen,char *save,int slen)
{
    char fname[50],hbuf[50],*p;
    FILE *fd;

    if (hlen > 50)
	error("save file header buffer too small");

    trm_str("File name? ");
    trm_get(fname);

    /* add the extension */
    strcat(fname,".sav");

    /* create the data file */
    if ((fd = fopen(fname,"r")) == 0)
	return (0);

    /* read the header */
    if (fread(hbuf,1,hlen,fd) != hlen) {
	fclose(fd);
	return (0);
    }

    /* compare the headers */
    for (p = hbuf; hlen--; )
	if (*hdr++ != *p++) {
	    trm_str("This save file does not match the adventure!\n");
	    return (0);
	}

    /* read the data */
    if (fread(save,1,slen,fd) != slen) {
	fclose(fd);
	return (0);
    }

    /* close the file and return successfully */
    fclose(fd);
    return (1);
}
