/* NS32000 Assembler
 * Inst.c
 * Code to assemble the various instruction types.
 */
#include <stdio.h>
#ifdef MSDOS
#  include "a_out.h"
#else
#  include "a.out.h"
#endif
#include "glob.h"
#include "conv.h"

/* Special instructions for generating operands.  If type is positive, it is
 * the size of the integer operand.
 */
#define FLOAT_OP	-1		/* floating point operand */
#define FLOAT_OP10	-2		/* fp operand, size in bit 10, not 8 */
#define DEF_INT		0		/* default int, size in opcode */
#define IS_FLOAT_OP(x)	(x < 0)		/* op is some kind of float */
#define IS_DEF_INT(x)	(x == 0)	/* op is int w/ size in opcode */
#define IS_SPEC_INT(x)  (x > 0)		/* op is int w/ fixed size */

/* Information needed to output an instruction during phase1.  Outp is
 * the current position in the output string, notwritten is the number
 * of additional bytes which will be written during backpatching.
 */
typedef struct  {
  U8 outstr [38];
  U8 *outp;
  int notwritten;
} outrec;
typedef outrec *outrecp;

/* Handler for one byte no operand instructions.
 */
no_opnd1 (p)
opptr p;
{
  check_tseg (p);
  update_lnmap (1);
  myputc ((int)(p->inst & 0xff), tmpt);
}

/* Assemble one byte opcode instructions which take a displacement
 * operand, e.g. RET.
 */
disp1 (p)
opptr p;
{
  generic_disp1 (p, pNEED_IMM);
}

/* Assemble one byte opcode instructions which take a pc relative displacement
 * operand, e.g. BSR.
 */
pcdisp1 (p)
opptr p;
{
  generic_disp1 (p, pPCREL | pNEED_TEXT);
}

/* One byte instruction with displacement operand.
 */
generic_disp1 (p, flags)
opptr p;
int flags;
{
  exptyp exp;
  struct patch pch;
  int len;
  outrec out;

  check_tseg (p);
  *out.outstr = p->inst;
  out.outp = out.outstr + 1;
  out.notwritten = 0;
  compile_exp (exp, &len);
  pch.exp = exp;
  pch.flags = flags;
  pch.pcoff = 1;
  proc_disp_ext (&pch, len, &out);
  write_rec (&out);
}

/* Handler for save: one byte immediate operand.
 */
imm1 (p)
opptr p;
{
  generic_imm1 (p, 0);
}

/* Handler for exit, restore: one byte immediate operand with bits reversed
 * so you can use same constant used in enter/save.
 */
revimm1 (p)
opptr p;
{
  generic_imm1 (p, pBYTEREV);
}

/* Does the work for imm1() and revimm1().  Generates a one byte instruction
 * with a one byte immediate extension.
 */
generic_imm1 (p, flags)
int flags;
opptr p;
{
  exptyp exp;
  struct patch pch;
  int len;
  outrec out;

  check_tseg (p);
  compile_exp (exp, &len);
  pch.exp = exp;
  pch.flags = pNEED_IMM | flags;
  pch.pcoff = 1;
  pch.size = 1;
  *out.outstr = p->inst;
  out.outp = out.outstr + 1;
  out.notwritten = 0;
  proc_imm_ext (&pch, len, &out);
  write_rec (&out);
}

/* Handler for enter: one byte immediate operand, optional displacement.
 */
immdisp1 (p)
opptr p;
{
  exptyp exp;
  struct patch pch;
  outrec out;
  int len;

  check_tseg (p);
  compile_exp (exp, &len);
  pch.exp = exp;
  pch.flags = pNEED_IMM;
  pch.pcoff = 1;
  pch.size = 1;
  *out.outstr = p->inst;
  out.outp = out.outstr + 1;
  out.notwritten = 0;
  proc_imm_ext (&pch, len, &out);
  if (curtok == ',') {                 /* get displacement */
    scan();
    compile_exp (exp, &len);
    pch.exp = exp;
    pch.flags = pNEED_IMM;
    pch.pcoff = out.outp - out.outstr + out.notwritten;
    proc_disp_ext (&pch, len, &out);
  } else *out.outp++ = 0;              /* default: disp = 0 */
  write_rec (&out);
}

/* Process instruction with two byte basic opcode and one general operand.
 */
gen2 (p)
opptr p;
{
  check_tseg (p);
  generic_gen2 (p->inst);
}

/* Handler for LPRi, SPRi.  CPU register plus gen operand.
 */
cpureggen2(p)
opptr p;
{
  check_tseg (p);
  generic_gen2 ((U32)(p->inst | (U32)getreg(rUS, rMOD) << 7));
}

/* Does the work for gen2() and cpureggen2(), shortgen2().
 */
generic_gen2 (opcode)
U32 opcode;
{
  struct genrec gen;
  outrec out;

  gen.flt = FALSE;
  parse_gen (&gen);
  out.outp = out.outstr;
  opcode |= ((long)(gen.type2? gen.type2: gen.type1)) << 11;
  *out.outp++ = opcode;
  *out.outp++ = opcode >> 8;
  if (gen.type2)
    *out.outp++ = gen.type1 << 3 | gen.inxreg;
  out.notwritten = 0;
  proc_gen_ext (&gen, (int)((opcode & 3) + 1), &out);
  write_rec (&out);
}

/* gen shifted 14 bits for SFSR.
 */
gen3a (p)
opptr p;
{
  check_tseg (p);
  generic_gen3 (p->inst, 14);  
}

/* gen shifted 19 bits for LFSR, RDVAL, WRVAL
 */
gen3b (p)
opptr p;
{
  check_tseg (p);
  generic_gen3 (p->inst, 19);
}

/* Handler for LMR, SMR.  MMU register plus gen operand.
 */
mmureggen3(p)
opptr p;
{
  check_tseg (p);
  generic_gen3 ((U32)(p->inst | (U32)getreg(rBPR0, rEIA) << 15), 19);
}

/* Does the work for gen3a(), gen3b() and mmureggen3().
 */
generic_gen3 (opcode, shift)
int shift;                             /* where addressing mode goes */
U32 opcode;
{
  struct genrec gen;
  outrec out;

  gen.flt = FALSE;
  parse_gen (&gen);
  out.outp = out.outstr;
  opcode |= ((long)(gen.type2? gen.type2: gen.type1)) << shift;
  *out.outp++ = opcode;
  *out.outp++ = opcode >> 8;
  *out.outp++ = opcode >> 16;
  if (gen.type2)
    *out.outp++ = gen.type1 << 3 | gen.inxreg;
  out.notwritten = 0;
  proc_gen_ext (&gen, 4, &out);
  write_rec (&out);
}

/* Used by cpureggen() and mmureggen() to parse "<reg>," and check that
 * reg is in some legal range (first to last).
 */
int
getreg (first, last)
int first, last;
{
  int regnum;

  if (curtok == tREG) regnum = curval;
  else if (curtok == tMOD) regnum = rMOD;
  else {
    error ("#expected CPU or MMU register");
    return 0;
  }
  if (regnum < first || regnum > last)
    error ("#register not compatible with instruction");
  scan();
  if (curtok != ',') error ("#expected comma");
  else scan();
  return regnum & 0xf;
}

/* Handler for quick instructions -- quick operand, general operand.
 */
shortgen2(p)
opptr p;
{
  U32 opcode;

  check_tseg (p);
  opcode = p->inst;
  proc_short (&opcode, 2);
  if (curtok != ',')
    error ("#expected comma");
  else scan();
  generic_gen2 (opcode);
}

/* Handler for ACBi -- short operand, general operand, displacement.
 */
shortgendisp2(p)
opptr p;
{
  U32 opcode;
  struct genrec gen;
  outrec out;
  int len;
  exptyp exp;
  struct patch pch;

  check_tseg (p);
  opcode = p->inst;
  proc_short (&opcode, 2);
  if (curtok != ',') {
    error ("#expected comma");
    return;
  }
  scan();
  gen.flt = FALSE;
  parse_gen (&gen);
  out.outp = out.outstr;
  opcode |= ((long)(gen.type2? gen.type2: gen.type1)) << 11;
  *out.outp++ = opcode;
  *out.outp++ = opcode >> 8;
  if (gen.type2)
    *out.outp++ = gen.type1 << 3 | gen.inxreg;
  out.notwritten = 0;
  proc_gen_ext (&gen, (int)((opcode & 3) + 1), &out);
  if (curtok != ',') {
    error ("#expected comma");
    return;
  }
  scan();
  compile_exp (exp, &len);
  pch.exp = exp;
  pch.flags = pPCREL;
  pch.pcoff = out.outp - out.outstr + out.notwritten;
  proc_disp_ext (&pch, len, &out);
  write_rec (&out);
}

/* Handler for setcfg -- one short operand.
 */
short3(p)
opptr p;
{
  U8 outstr [4], *outp;
  U32 opcode;

  check_tseg (p);
  opcode = p->inst;
  proc_short (&opcode, 3);
  outp = outstr;
  *outp++ = opcode;
  *outp++ = opcode >> 8;
  *outp++ = opcode >> 16;
  myfwrite ((char *)outstr, 3, tmpt);
  update_lnmap (3);
}

/* Process instruction with two byte basic opcode and two general operands.
 */
gengen2 (p)
opptr p;
{
  outrec out;

  check_tseg (p);
  generic_gengen (2, &out, p->inst, DEF_INT, DEF_INT);
  write_rec (&out);
}

/* Process instruction with three byte basic opcode and two non-float
 * general operands.
 */
gengen3 (p)
opptr p;
{
  outrec out;

  check_tseg (p);
  generic_gengen (3, &out, p->inst, DEF_INT, DEF_INT);
  write_rec (&out);
}

/* Process instruction with three byte basic opcode and two non-float
 * general operands, the first of which has size 1 byte, e.g. ash, lsh,
 * rot.
 */
bgengen3 (p)
opptr p;
{
  outrec out;

  check_tseg (p);
  generic_gengen (3, &out, p->inst, 1, DEF_INT);
  write_rec (&out);
}

/* Process instruction with three byte basic opcode and two non-float
 * general operands, the second of which has size 1 byte, e.g. ffs.
 */
genbgen3 (p)
opptr p;
{
  outrec out;

  check_tseg (p);
  generic_gengen (3, &out, p->inst, DEF_INT, 1);
  write_rec (&out);
}

/* Process instruction with three byte basic opcode, one float general
 * operand and one non-float general operand.
 */
fgengen3 (p)
opptr p;
{
  outrec out;

  check_tseg (p);
  generic_gengen (3, &out, p->inst, FLOAT_OP10, DEF_INT);
  write_rec (&out);
}

/* Process instruction with three byte basic opcode, one non-float general
 * operand and one float general operand.
 */
genfgen3 (p)
opptr p;
{
  outrec out;

  check_tseg (p);
  generic_gengen (3, &out, p->inst, DEF_INT, FLOAT_OP);
  write_rec (&out);
}

/* Process instruction with three byte basic opcode and two float
 * general operands.
 */
fgenfgen3 (p)
opptr p;
{
  outrec out;

  check_tseg (p);
  generic_gengen (3, &out, p->inst, FLOAT_OP, FLOAT_OP);
  write_rec (&out);
}

/* Handler for CVTP, CHECKi and INDEXi -- register, gen, gen.
 */
reggengen3 (p)
opptr p;
{
  outrec out;

  check_tseg (p);
  generic_gengen (3, &out, p->inst | (U32)getreg(rR0, rR0+7)<<11,
    DEF_INT, DEF_INT);
  write_rec (&out);
}
  
/* Handler for EXTi, INSi -- register, gen, gen, disp.
 */
reggengendisp3 (p)
opptr p;
{
  int len;
  exptyp exp;
  struct patch pch;
  outrec out;

  check_tseg (p);
  generic_gengen (3, &out, p->inst | (U32)getreg(rR0, rR0+7)<<11,
    DEF_INT, DEF_INT);
  if (curtok != ',') {
    error ("#expected comma");
    return;
  }
  scan();
  compile_exp (exp, &len);
  pch.exp = exp;
  pch.flags = pNEED_IMM;
  pch.pcoff = out.outp - out.outstr + out.notwritten;
  proc_disp_ext (&pch, len, &out);
  write_rec (&out);
}
  
/* Handler for MOVM, CMPM -- gen, gen, disp.
 */
gengendisp3 (p)
opptr p;
{
  int len, wordsize;
  exptyp exp;
  struct patch pch;
  outrec out;

  check_tseg (p);
  generic_gengen (3, &out, p->inst, DEF_INT, DEF_INT);
  if (curtok != ',') {
    error ("#expected comma");
    return;
  }
  scan();
  compile_exp (exp, &len);
  pch.exp = exp;
  pch.flags = pNEED_IMM;
  wordsize = (p->inst >> 8) & 3;
  switch (wordsize) {
    case 0: pch.flags |= pBYTE; break;
    case 1: pch.flags |= pWORD; break;
    case 3: pch.flags |= pDOUBLE; break;
  }
  pch.pcoff = out.outp - out.outstr + out.notwritten;
  proc_disp_ext (&pch, len, &out);
  write_rec (&out);
}
  
/* Handler for string instructions -- gen, gen, immediate.
 */
gengenimm3 (p)
opptr p;
{
  outrec out;
  U8 imm;
  U32 tmp;

  check_tseg (p);
  generic_gengen (3, &out, p->inst, DEF_INT, DEF_INT);
  if (curtok != ',') {
    error ("#missing comma");
    return;
  }
  scan();
  tmp = get_imm();
  if (tmp > 7) error ("#offset greater than 7");
  imm = tmp << 5;
  if (curtok != ',') {
    error ("#missing comma");
    return;
  }
  scan();
  tmp = get_imm();
  if (tmp > 32 || tmp == 0) error ("#length zero or greater than 32");
  imm |= tmp - 1;
  *out.outp++ = imm;
  write_rec (&out);
}
  
/* Process string instructions; e.g. CMPSi, CMPST.
 * Operands: [{B|U|W}[,{B|U|W}]*]
 */
buw3 (p)
opptr p;
{
  outrec out;
  U32 opcode;

  check_tseg (p);
  opcode = p->inst;
  if (curtok == tLBL)
    for (;;) {
      if (curtok != tLBL) {
        error ("#expected B, U, or W");
        return;
      }
      if (*curstr == 'b' || *curstr == 'w' || *curstr == 'u')
        *curstr += 'A' - 'a';
      if (!strcmp (curstr, "B")) opcode |= 0x10000;
      else if (!strcmp (curstr, "W")) opcode |= 0x20000;
      else if (!strcmp (curstr, "U")) opcode |= 0x60000;
      else {
        error ("#expected B, U, or W");
        return;
      }
      scan();
      if (curtok != ',') break;
      scan();
    }
  out.notwritten = 0;
  out.outp = out.outstr;
  *out.outp++ = opcode;
  *out.outp++ = opcode >> 8;
  *out.outp++ = opcode >> 16;
  write_rec (&out);
}

/* Process CINV instruction */
cinv (p)
opptr p;
{
  U32 opcode;

  check_tseg (p);
  opcode = p->inst;
  if (curtok == tLBL)
    for (;;) {
      if (curtok != tLBL) break;
      if (*curstr >= 'a' && *curstr <= 'z')
        *curstr += 'A' - 'a';
      if (!strcmp (curstr, "A")) opcode |= 0x20000;
      else if (!strcmp (curstr, "D")) opcode |= 0x8000;
      else if (!strcmp (curstr, "I")) opcode |= 0x10000;
      else {
        error ("#expected A, D, or I");
        return;
      }
      scan();
      if (curtok != ',') {
        error ("#missing comma");
        return;
      }
      scan();
    }
  generic_gen3 (opcode, 19);
}

generic_gengen (instlen, out, opcode, spec1, spec2)
U32 opcode;
outrecp out;
int instlen,
    spec1, spec2;		/* special instructions for operands */
{
  struct genrec gen1, gen2;
  int immlen1, immlen2;		/* immediate length */

  gen1.flt = IS_FLOAT_OP (spec1);
  gen2.flt = IS_FLOAT_OP (spec2);
  parse_gengen (&gen1, &gen2);
  opcode |= ((long)(gen1.type2? gen1.type2: gen1.type1))
              << (instlen==3? 19: 11) |
            ((long)(gen2.type2? gen2.type2: gen2.type1))
              << (instlen==3? 14: 6);
  out->notwritten = 0;
  out->outp = out->outstr;
  *out->outp++ = opcode;
  *out->outp++ = opcode >> 8;
  if (instlen == 3) *out->outp++ = opcode >> 16;
  if (gen1.type2) *out->outp++ = gen1.type1 << 3 | gen1.inxreg;
  if (gen2.type2) *out->outp++ = gen2.type1 << 3 | gen2.inxreg;

  if (IS_DEF_INT(spec1)) immlen1 = (opcode>>(instlen==3? 8: 0) & 3) + 1;
  else if (IS_FLOAT_OP(spec1)) {
    if (opcode & (spec1 == FLOAT_OP10? 0x400: 0x100)) immlen1 = 4;
    else immlen1 = 8;
  } else immlen1 = spec1;
  if (IS_DEF_INT(spec2)) immlen2 = (opcode>>(instlen==3? 8: 0) & 3) + 1;
  else if (IS_FLOAT_OP(spec2)) {
    if (opcode & (spec2 == FLOAT_OP10? 0x400: 0x100)) immlen2 = 4;
    else immlen2 = 8;
  } else immlen2 = spec2;
  proc_gen_ext (&gen1, immlen1, out);
  proc_gen_ext (&gen2, immlen2, out);
}

/* Write outrec to temp file and update line map.
 */
write_rec (p)
register outrecp p;
{
  /* (int) casts fix bug in Mac version */
  myfwrite ((char *)(p->outstr), (int)(p->outp - p->outstr), tmpt);
  update_lnmap ((int)(p->outp - p->outstr + p->notwritten));
}

/* Handler for unimplemented instructions and directives.  Prints
 * exactly one error message per assemble.
 */
not_implemented (p)
opptr p;
{
  if (p->inst != 0xffffffff) {
    fprintf (stderr, "as: line %d, not implemented: %s\n", lnnum, p->id);
    p->inst = 0xffffffff;
  }
  scannl();
}

/* Report an error if not in text segment, change segment to text.
 */
check_tseg(p)
opptr p;
{
  if (curseg != T_TEXT) {
    error ("#%s may only appear in text segment", p->id);
    curseg = T_TEXT;
  }
}

/* Insert patch into text or data patch list.  Fill in link, lnnum, offset.
 * If exp is not NULL, allocate memory and copy it.
 */
insert_patch (pch, len)
register patchptr pch;
int len;
{
  register patchptr *tail, *head;

  pch->offset = *curlocptr;
  pch->lnnum = lnnum;
  pch->link = NULL;
  if (pch->exp != NULL)
    copy_exp (&pch->exp, len);
  if (curseg == T_TEXT) {
    head = &tpatch_head;
    tail = &tpatch_tail;
  } else {
    head = &dpatch_head;
    tail = &dpatch_tail;
  }
  if (*head == NULL)                   /* insert into queue */
    *head = *tail = pch;
  else {
    (*tail)->link = pch;
    *tail = pch;
  }
}

/* Passed info for a gen operand, writes the appropriate immediate or
 * displacement(s) if mode requires them and their values can be resolved.
 * Else, increments notwritten by the number of bytes that would be written
 * were the value known.  If the value is unresolved or the value needs
 * relocation, a patch record is created.
 */
proc_gen_ext (gen, size, out)
register genptr gen;                   /* struct with operand info */
int size;                              /* # of bytes if immediate */
outrecp out;
{
  struct patch pch;

  if (gen->type1 >= mR0 &&             /* if reg or TOS then no extension */
  gen->type1 <= mR0 + 7 ||
  gen->type1 == mTOS)
    return;
  pch.exp = gen->exp1;
  pch.pcoff = out->outp - out->outstr + out->notwritten;
  pch.flags = 0;
  if (gen->type1 == mIMM) {            /* immediate mode */
    if (gen->flt)
      proc_fltimm_ext (gen, size, out);
    else {
      pch.size = size;
      proc_imm_ext (&pch, gen->len1, out);
    }
  } else {                             /* rest of modes need a disp */
    if (gen->type1 == mPCIND)
      pch.flags = pPCREL;
    proc_disp_ext (&pch, gen->len1, out);
    if (gen->type1 >= mFPDBLIND &&     /* double indirect needs another disp */
    gen->type1 <= mFPDBLIND + 2 ||     /* another disp if external */
    gen->type1 == mEXT) {
      pch.flags = 0;
      pch.exp = gen->exp2;
      pch.pcoff = out->outp - out->outstr + out->notwritten;
      proc_disp_ext (&pch, gen->len2, out);
    }
  }
}

/* Write a floating point immediate extension during phase 1.  No patching
 * allowed so this is easy.  Remember that immediates have most
 * significant bytes first.
 */
proc_fltimm_ext (gen, size, out)
register genptr gen;                   /* struct with operand info */
int size;                              /* # of bytes if immediate */
outrecp out;
{
  unsigned char *p, *endp;
  float flt;

# ifdef BIG_ENDIAN
  if (size == 4) {
    flt = gen->fltimm;
    p = (unsigned char *)&flt;
  } else p = (unsigned char *)&gen->fltimm;
  endp = p + size;
  for (; p < endp; ++p) *(out->outp)++ = *p;
# else
  if (size == 4) {
    flt = gen->fltimm;
    endp = (unsigned char *)&flt;
  } else endp = (unsigned char *)&gen->fltimm;
  p = endp + size - 1;
  for (; p >= endp; --p) *(out->outp)++ = *p;
# endif
}

/* Called in phase1 when an instruction needs a displacement extension.
 * There are several cases.  If the expression is resolved, the displacement
 * can be written in phase1.  Otherwise a patch record is created so that
 * the displacement can be written in phase3.  A patch record is also
 * needed if the displacement will need relocating.
 *
 * Input fields in pch: exp, flags, pcoff.
 */
proc_disp_ext (pch, len, out)
patchptr pch;                          /* patch template */
outrecp out;
int len;                               /* expression length */
{
  patchptr pchp;

  if (pch->flags & pPCREL)             /* defer pc relative - set unresolved */
    pch->typ = 0;
  else eval_exp (pch->exp, &pch->typ, &pch->val, &pch->undf);
  pch->flags |= pDISP;
  if (pch->typ & T_RESOLVED) {         /* write the displacement now */
    pch->size = disp_size (pch->val);
    fmt_patch (pch, 0L, out->outp);
    out->outp += pch->size;
    pch->flags |= pWRITTEN;
  } else out->notwritten += (pch->size = 4);
  if (!(pch->typ & T_RESOLVED) ||      /* create a patch record */
  (pch->typ & T_LBL) != T_IMM) {
    pchp = (patchptr) myalloc (sizeof (struct patch));
    *pchp = *pch;
    insert_patch (pchp, len);
  }
}

/* Called in phase1 when an instruction needs an immediate extension.
 * There are several cases.  If the expression is resolved, the immediate
 * can be written in phase1.  Otherwise a patch record is created so that
 * the immediate can be written in phase3.  A patch record is also
 * needed if the immediate will need relocating.
 *
 * Input fields in pch: exp, flags, pcoff, size.
 */
proc_imm_ext (pch, len, out)
patchptr pch;                          /* patch template */
outrecp out;
int len;                               /* expression length */
{
  patchptr pchp;

  eval_exp (pch->exp, &pch->typ, &pch->val, &pch->undf);
  pch->flags |= pGENIMM;
  if (pch->typ & T_RESOLVED) {         /* write the displacement */
    fmt_patch (pch, 0L, out->outp);
    out->outp += pch->size;
    pch->flags |= pWRITTEN;
  } else out->notwritten += pch->size;
  if (!(pch->typ & T_RESOLVED) ||      /* create a patch record */
  (pch->typ & T_LBL) != T_IMM) {
    pchp = (patchptr) myalloc (sizeof (struct patch));
    *pchp = *pch;
    insert_patch (pchp, len);
  }
}

/* Called in phase1 when an instruction needs a short operand.
 * There are several cases.  If the expression is resolved, the short
 * can be inserted into the instruction in phase1.  Otherwise a patch
 * record is created so that the short can be written in phase3.
 */
proc_short (opcode, size)
U32 *opcode;
int size;
{
  struct patch pch;
  patchptr pchp;
  exptyp exp;
  int len;

  pch.size = size;
  compile_exp (exp, &len);
  eval_exp (exp, &pch.typ, &pch.val, &pch.undf);
  if (pch.typ & T_RESOLVED)            /* insert short */
    insert_short (&pch, opcode);
  else {                               /* create a patch record */
    pchp = (patchptr) myalloc (sizeof (struct patch));
    pch.exp = exp;
    pch.pcoff = 0;
    pch.flags = pSHORT | pNEED_IMM;
    *pchp = pch;
    insert_patch (pchp, len);
  }
}

/* Passed a value, returns the number of bytes of the smallest displacement
 * that would be adequate to hold the value.
 */
int
disp_size (val)
register U32 val;
{
  if (val <= 0x3f        || val >= 0xffffffc0) return 1;
  if (val <= 0x1fff      || val >= 0xffffe000) return 2;
  if (val <= 0x1fffffff  || val >= 0xe0000000) return 4;
  error ("#value too large for displacement");
  return 4;
}
