/* Ar.c -- File archiver
 *
 * Bruce Culbertson  9 April 1987
 *
 * Options:
 *   -a: append files to end of archive
 *   -d: delete files from archive
 *   -p: prepend files to beginning of archive
 *   -t: print table of contents of archive
 *   -x: extract file from archive
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "ar.h"
#include "magic.h"
#include "conv.h"

#define TRUE 1
#define FALSE 0
#define ARSZ  (sizeof (struct ar_header))
#define BUFSZ 0x400

int append(), delete(), prepend(), table(), extract();
struct option *getopt();

char usage[] = "Usage: ar -adptx archive files...",
     buf [BUFSZ],
     **argv,
     tmpname[20] = "",
     *ctime();

int  argc,
     infile;
     
struct option {
  char option;
  int  (*f)();
  short need_ar;
} options [] = {
  {'a', append, FALSE},
  {'d', delete, TRUE},
  {'p', prepend, FALSE},
  {'t', table, TRUE},
  {'x', extract, TRUE},
};

main (argc_real, argv_real)
int argc_real;
char **argv_real;
{
  struct option *p;

  argv = argv_real;
  argc = argc_real;
  p = getopt ();                   /* lookup option */
  open_ck (p->need_ar);            /* open archive, check magic # */
  (*p->f) ();                      /* do operation on archive */
  exit (0);
}

/* Return pointer to option record.
 */
struct option *
getopt ()
{
  struct option *p;
  char c;

  if (argc >= 3 && *(argv[1]++) == '-') {
    c = *(argv[1]);
    for (p = options; p < options + sizeof(options)/sizeof(struct option); ++p)
      if (p->option == c) return p;
  }
  puts (usage);
  exit (-1);
}

/* Try to open existing archive.  If need_ar and archive does not exist,
 * report error.  If not need_ar and archive does not exist, create it.
 */
open_ck (need_ar)
int need_ar;
{
  unsigned char magic [sizeof (MAGIC_TYPE)];

#ifdef MSDOS
  if (-1 == (infile = open (argv[2], O_RDWR | O_BINARY))) {
#else
  if (-1 == (infile = open (argv[2], O_RDWR))) {
#endif
    if (need_ar) error ("could not open %s", argv[2]);
#ifdef MSDOS
    if (-1 == (infile = open (argv[2], O_RDWR | O_CREAT | O_BINARY,
    S_IREAD | S_IWRITE)))
#else
    if (-1 == (infile = open (argv[2], O_RDWR | O_CREAT, 0660)))
#endif
      error ("could not open %s", argv[2]);
    WL2M (*magic, AR_MAGIC);
    if (sizeof (MAGIC_TYPE) != write (infile, (char *)magic,
    sizeof (MAGIC_TYPE)))
      error ("could not write %s", argv[2]);
    return;
  }
  if (sizeof (MAGIC_TYPE) != read (infile, buf, sizeof (MAGIC_TYPE)))
    error ("could not read %s", argv[2]);
  if (AR_MAGIC != WM2L(*((long *)buf)))
    error ("%s is not an archive", argv[2]);
}

/* Create a temporary file for writing.  Put its name in
 * tmpname.  Write magic number.
 */
create_tmp(f)
int *f;
{
  unsigned char magic [sizeof (MAGIC_TYPE)];

  sprintf (tmpname, "ar.%d", getpid());
#ifdef MSDOS
  if (-1 == (*f = open (tmpname, O_WRONLY | O_CREAT | O_BINARY,
  S_IREAD | S_IWRITE)))
#else
  if (-1 == (*f = open (tmpname, O_WRONLY | O_CREAT, 0660)))
#endif
    error ("could not open temporary file");
  WL2M (*magic, AR_MAGIC);
  if (sizeof (MAGIC_TYPE) != write (*f, (char *)magic, sizeof (MAGIC_TYPE)))
    error ("could not write temporary file");
}

/* Move the temporary file to the old archive.
 */
relink(tmp)
int tmp;
{
  unlink (argv[2]);
  close (tmp);
#ifdef MSDOS
  rename (tmpname, argv[2]);
#else
  link (tmpname, argv[2]);
  unlink (tmpname);
#endif
}

/* Perform append option.
 */
append()
{
  char **p;

  lseek (infile, 0L, 2);
  for (p = argv + 3; p < argv + argc; ++p)
    mk_head_cp_file (*p, infile); 
}

/* Create and write ar_header to f, write file to f.
 */
mk_head_cp_file (p, f)
int f;
char *p;
{
  int auxfile;
  char *q;
  struct ar_header ar;
  struct stat s;

#ifdef MSDOS
  if (-1 == (auxfile = open (p, O_RDONLY | O_BINARY)))
#else
  if (-1 == (auxfile = open (p, O_RDONLY)))
#endif
    error ("could not open %s", p);
  stat (p, &s);
  WL2M (ar.ar_mtime, s.st_mtime);
  WL2M (ar.ar_len, s.st_size);
  q = p + strlen (p);
  do {                                /* strip path from name */
    --q;
    if (*q == '/') {
      ++q;
      break;
    }
  } while (q > p);
  strcpy (ar.ar_name, q);
  copy_head_file (auxfile, f, &ar, (long)s.st_size);
  close (auxfile);
}

/* Copy archive and len bytes of in to out.
 */
copy_head_file (in, out, ar, len)
int in, out;
struct ar_header *ar;
long len;
{
  if (ARSZ != write (out, ar, ARSZ))   /* write header */
    error ("write error");
  copy (in, out, len);
}

/* Copy len bytes of file in to file out.
 */
copy (in, out, len)
int in, out;
long len;
{
  unsigned buflen;
  extern int errno;

  do {
    buflen = (len > BUFSZ)? BUFSZ: len;
    if (buflen != read (in, buf, buflen)) error ("read error");
    len -= buflen;
    if (buflen != write (out, buf, buflen)) error ("write error");
  } while (len);
}
      
/* Perform delete option.
 */
delete()
{
  long offset, len;
  struct ar_header ar;
  char **p;
  int tmp, found;

  create_tmp (&tmp);
  offset = sizeof (MAGIC_TYPE);
  for (;;) {
    if (ARSZ != read (infile, &ar, ARSZ)) break;
    len = WM2L (ar.ar_len);
    offset += ARSZ + len;
    found = FALSE;
    for (p = argv + 3; p < argv + argc; ++p)
      if (0 == strcmp (*p, ar.ar_name)) {
        found = TRUE;
        break;
      }
    if (found) lseek (infile, offset, 0);
    else copy_head_file (infile, tmp, &ar, len);
  }
  relink(tmp);
}

/* Perform prepend option.
 */
prepend()
{
  int tmp;
  char **p;
  struct stat s;

  create_tmp (&tmp);
  for (p = argv + 3; p < argv + argc; ++p)
    mk_head_cp_file (*p, tmp);
  stat (argv[2], &s);
  copy (infile, tmp, (long)(s.st_size - sizeof (MAGIC_TYPE)));
  relink(tmp);
}

table()
{
  long offset;
  struct ar_header ar;
  char **p;

  offset = sizeof (MAGIC_TYPE);
  for (;;) {
    if (ARSZ != read (infile, &ar, ARSZ)) break;
    CM2L (ar.ar_mtime);
    CM2L (ar.ar_len);
    offset += ARSZ + ar.ar_len;
    if (argc == 3) printhead (&ar);
    else for (p = argv + 3; p < argv + argc; ++p)
      if (0 == strcmp (*p, ar.ar_name)) {
        printhead (&ar);
        break;
      }
    lseek (infile, offset, 0);
  }
}

/* Print table of contents entry for single file.
 */
printhead(ar)
struct ar_header *ar;
{
  char *p, *q;

  p = ctime (&ar->ar_mtime);
  for (q = p; *q; ++q) if (*q == '\n') *q = '\0';
  printf ("%8ld %s %s\n", ar->ar_len, p, ar->ar_name);
}

/* Perform extract option.
 */
extract()
{
  long offset;
  struct ar_header ar;
  char **p;

  offset = sizeof (MAGIC_TYPE);
  for (;;) {
    if (ARSZ != read (infile, &ar, ARSZ)) break;
    CM2L (ar.ar_len);
    offset += ARSZ + ar.ar_len;
    if (argc == 3) extractfile (&ar);
    else for (p = argv + 3; p < argv + argc; ++p)
      if (0 == strcmp (*p, ar.ar_name)) {
        extractfile (&ar);
        break;
      }
    lseek (infile, offset, 0);
  }
}

/* Extract a single file from archive.  File pointer points to first
 * byte of file (header already read).
 */
extractfile (ar)
struct ar_header *ar;
{
  int outfile;

#ifdef MSDOS
  if (-1 == (outfile = open (ar->ar_name, O_WRONLY | O_CREAT | O_BINARY,
  S_IWRITE | S_IREAD)))
#else
  if (-1 == (outfile = open (ar->ar_name, O_WRONLY | O_CREAT, 0660)))
#endif
    error ("could not open %s", ar->ar_name);
  copy (infile, outfile, ar->ar_len);
  close (outfile);
}

error (format, arg)
char *format, *arg;
{
  fputs ("ar: ", stderr);
  fprintf (stderr, format, arg);
  putc ('\n', stderr);
  if (*tmpname) unlink (tmpname);
  exit (-1);
}
