/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Aug27-GiuseppeG: code write
*/

namespace XSP
{

class Window; // forward def needed by GetParentWindow

class View  
{
private:
	uint32 refcount;
public:
	void addref();
	void release();

public:
	typedef refc<View> owner;
	typedef std::list<owner> Children;
	const Children& GetChildren() const { return children; }
private:
	Children children;

private:
	Rect2D bounds; // bounds relative to the parent view
	bool   hidden; 
	// set by the parent when Add is called
	View::owner parent; 
	// keep the skin here (may be NULL)
	ViewSkin::owner skin;
    TextFont font; // this may be NULL too
    
public:
	enum ViewState
	{ 
		StateHovered    =0x0001
	,	StateCapturing  =0x0002
	,	StateFocusable  =0x0004
	,	StateFocused    =0x0008 // the one receiving keys when activated
	,	StateOldFocus   =0x0010 // the one that used to have focus ( for reactivation )
	,	StateActivated  =0x0020 // all children of the window with the focus are active
	,	StateDisabled   =0x0040 

	,   ViewStateMask =  0x007F
	};
	uint32 GetState() const { return viewState; }
	uint32 SetState(uint32 s);
	uint32 StateSetReset(uint32 sSet, uint32 sReset) { return SetState((viewState|sSet)&~sReset); }
	uint32 StateAssign(uint32 state, bool on) { return SetState(on ? (viewState|state) : (viewState&~state)); }
	bool   IsStateSet(uint32 s) { return (0 != (GetState() & s)); }
	bool   IsStateMatch(uint32 s, uint32 m) { return (s == (GetState() & m)); }
private:	
	uint32 viewState;

public: // notifications
	class _ViewChange
	{
	public: 
		virtual void ViewClosed(View* view);
		virtual void ChildAddRemove(bool added, const owner& child, View* view);
		virtual void ViewShowHide(bool hide, View* view);
		virtual void ViewStateChange(uint32 oldState, uint32 newState, View* view);
		virtual void ViewBoundsChange(const Rect2D& oldBounds, const Rect2D& newBounds, View* view);
		virtual void ViewScroll(const Size2D& scrollSize, const Rect2D& bounds, View* scrollView);
	}; // class _ViewChange

	void NotifyChange_ViewClosed();
	void NotifyChange_ChildAddRemove(bool added, const View::owner& child);
	void NotifyChange_ViewShowHide();
	void NotifyChange_ViewStateChange(uint32 oldState, uint32 newState);
	void NotifyChange_ViewBoundsChange(const Rect2D& oldBounds, const Rect2D& newBounds);
	void NotifyChange_ViewScroll(const Size2D& scrollSize, const Rect2D& bounds);

	// allow 2 views to be bound together by a set of scroll notifications
	// the ListenToScrollNotificationsFrom must be called by both sides of the link
	virtual void ListenToScrollNotificationsFrom(const View::owner& view);
	virtual void CancelListenToScrollNotifications();

	typedef ListenerOf<_ViewChange> ViewListener;
	typedef ViewListener::Container ViewListeners;
	void AddListener(const ViewListener::owner& l);
	bool IsWithinNotify() { return viewListeners.withinNotify; }
private:
	ViewListeners viewListeners;

public:
	View();
protected:
	virtual ~View();
	void Close();
	// paint this view's background into the specified graphics
	// do not touch the children
	// the clip rect of the graphics contains the area of interest
	// the NW corner of the view should be drawn at 
	// coordinates 0,0 in the graphics port
	virtual void Draw(Graphics& g, const Rect2D& area);

public:
	// paint this view and all it's children into the graphics
	virtual void DrawDeep(Graphics& g, const Rect2D& area);
	// override and do something when the bounds are changed
	virtual void SetBounds(const Rect2D& bounds);  
	// override to return the size you would like to have
	virtual Size2D GetPreferredSize();
	// override and do something when the skin is set
	virtual void SetSkin(const ViewSkin::owner& skin);
	virtual void SetFont(const TextFont& f);

public:
	// children add/remove
	virtual void AddView(const owner& view);
	virtual void RemoveView(const owner& view);
	void RemoveAllChildren();
	virtual Window GetParentWindow();
	static View::owner DetermineCommonAncestor(const View::owner& view1, 
											   const View::owner& view2);
	View::owner GetRootView();
	const View::owner& GetParent() const { return parent; }

	// determine relative based on state
	View::owner GetChild(unsigned viewState);
	View::owner GetParent(unsigned vState);
	// locate child
	virtual bool ContainsPoint(const Point2D& where) const;
	virtual bool FindChildAt(Point2D& where, View::owner& view); 
	bool FindDeepChildAt(Point2D& where, View::owner& view);
	bool HasChildren() const { return !children.empty(); }
	// bounds control
	Rect2D GetLocalBounds() const;
	Rect2D GetSkinContentBounds();
	const Rect2D& GetBounds() const { return bounds; }
	virtual Rect2D GetVisibleBoundsOnScreen();

	// hide the view
	void Hide(bool on = true);
	inline bool IsHidden() const { return hidden; }
    inline bool IsDisabled() const { return (0!= (viewState & StateDisabled)); }

	// refresh the rect (in view local coordinates)
	virtual void Refresh(const Rect2D& r);
	void Refresh(); // entire view rect
	// skin
	ViewSkin::owner& GetSkin() { return skin; }
	ViewSkin::owner GetSkinPartOrDefault(const String& s);
	// font
	const TextFont& GetFont() const { return font; }

public:	
	// focus & keyboard
    inline bool IsFocusable() const { return (0 != (viewState & StateFocusable)); }
    inline bool IsFocused() const { return (0 != (viewState & StateFocused)); }
	virtual void SetCaret(const Rect2D& r); // set it to size=0 to hide, override in Window::_Data
    void SetFocusable(bool on);
    void SetFocused();		    // only one and it's parents can be focused
    void SetActivated(bool on); // all children of the window with the focus are active
    virtual void HandleFocused(bool on);
    virtual void HandleActivated(bool on);
	virtual bool DispatchKeyEvent( const UIEvent& ev ); // any key event
	virtual bool HandleKeyEvent( const UIEvent& ev );  // table of events for this view
protected:
	bool _DispatchKeyEventToFocus( const UIEvent& ev );
	bool _DispatchKeyEventAsHotKey( const UIEvent& ev );
public:	
	// keyboard triggered actions
	bool ActionFocusLastChild();
	bool ActionFocusPrevChild();
	bool ActionFocusFirstChild();
	bool ActionFocusNextChild();
	 
public:	
	// mouse & capturing
	virtual void StartMouseCapture(const owner& view, const Point2D& origin);
	virtual void EndMouseCapture(const owner& view);
	bool IsMouseCapturing() const { return ((viewState & StateCapturing) != 0); }
	// notification received when OS force breaks the capture 
	virtual void MouseCaptureBroken(); 
	// handle events from the user 
	virtual bool HandleMouseEvent( const UIEvent& ev );
	bool IsHovered() const { return ((viewState & StateHovered) != 0); }
	virtual void MouseEnter(bool enter); // false for leave
	virtual void MouseCursorSetup(); // override to set the appropriate mouse cursor on entry

	// drag&drop
	// by default the handler will set the eid to 0 (effect = data not used)
	// this is a callback (event) called by the drag&drop mechanism to tell the view
	// something has been dropped on it
	virtual void HandleDrop(DragDropType::TypeSet& availableTypes, 
							DragDropType::Effect&  effect,
							const Point2D& 		   mousePos,
							bool 				   onlySimulate,
							DragDropType::FatData& theFatData );
	// copy&paste
	// this is not a callback, the view calls this and the params get filled with the 
	// appropriate information, returns true if something can be pasted
	bool PasteFromClipboard(DragDropType::TypeSet& availableTypes, 
							bool 				   onlySimulate,
							DragDropType::FatData& theFatData );
	bool CopyToClipboard(DragDropType::TypeSet& availableTypes, 
						   DragDropType::FatData& theFatData );
}; // class View::Interface


} // namespace XSP
